/*
 * @file  service.c
 * @brief The function  of l7vsd is managed. 
 * @brief Two or more service takes charge of the 
 * @brief function respectively. 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs_config.h"
#include "l7vs_service.h"
#include "l7vs_module.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_lsock.h"
#include "l7vs_sched.h"


//! service list pointer
static GList * l7vs_service_list;

//! service handle
uint32_t service_handle_base = 0;

//! remove_conn list pointer
static GList * removeconn_list;

/*!
 * l7vs_servcie_get_service_arg
 * service arg pointer lookup.
 * @param[in]	srv	l7vs_service pointer.
 */
static struct l7vs_service_arg_multi *
l7vs_service_get_service_arg(struct l7vs_service *srv)
{
        struct l7vs_service_arg_multi *sa;
	int ret = 0;

	if( NULL == srv ){
                VANESSA_LOGGER_ERR("Service get service arg/ argument srv is NULL");
		return NULL;
	}
	if( NULL == srv->pm || NULL == srv->lsock || NULL == srv->scheduler ){
		VANESSA_LOGGER_ERR("Service / Invalid argument");
		return NULL;
	}

        sa = (struct l7vs_service_arg_multi *) calloc(1, sizeof(struct l7vs_service_arg_multi));
        if (sa == NULL) {
                VANESSA_LOGGER_ERR("Service get service arg/ Could not allocate memory");
                return NULL;
        }

	ret = srv->pm->service_arg(sa, srv->handle);
	if (ret == -1) {
		VANESSA_LOGGER_ERR("Service / service not found");
		return NULL;
	}

	sa->srv_arg.len = sizeof(struct l7vs_service_arg_multi);
        sa->srv_arg.addr = srv->lsock->addr;
        sa->srv_arg.proto = srv->lsock->proto;
        sa->srv_arg.persist = 0 /* XXX not yet */;
        sa->srv_arg.backlog = 5 /* XXX not yet */;
        strcpy(sa->srv_arg.protomod, srv->pm->modname);
        strcpy(sa->srv_arg.schedmod, srv->scheduler->modname);

	sa->srv_arg.sorry_cc	= srv->sorry_cc;
	sa->srv_arg.sorry_flag	= srv->sorry_flag;
	memcpy( &sa->srv_arg.sorry_addr, &srv->sorry_dest->addr, sizeof(struct sockaddr_in) );

        return sa;
}

/*!
 * service_arg destroy function.
 * @param[in] sa	l7vs_service_arg pointer
 *
 */
static void
l7vs_service_put_service_arg(struct l7vs_service_arg_multi *sa)
{
        free(sa);
}


/*!
 * Target real servers of sets of real servers in service are retrieved. 
 * @param[in]	*srv	service pointer
 * @param[in]	*sin	IP and port
 * @return	l7vs_dest* real server struct pointer
 */
struct l7vs_dest *
l7vs_service_lookup_dest(struct l7vs_service *srv,
                         struct sockaddr_in *sin)
{
        GList *l;
        struct l7vs_dest *d, *found;

	if( NULL == srv ){
                VANESSA_LOGGER_ERR("Service / lookup dest : argument \"srv\" is NULL");
		return NULL;
	}
	if( NULL == sin ){
                VANESSA_LOGGER_ERR("Service / lookup dest : argument \"sin\" is NULL");
		return NULL;
	}

        found = NULL;
        for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
                d = (struct l7vs_dest *)l->data;
                if ((d->addr.sin_addr.s_addr == sin->sin_addr.s_addr) && (d->addr.sin_port == sin->sin_port)) {
                        found = d;
                        break;
                }
        }

        return found;
}

/*!
 *
 * @param[out] **sas	l7vs_service_arg pointer list
 * @param[out] *num	pointer list count
 * @return		pointer list count
 */
int
l7vs_service_list_service_arg(struct l7vs_service_arg_multi **sas, int *num)
{
        GList *l;
        struct l7vs_service *srv;
        struct l7vs_service_arg_multi *r, *s;
        struct l7vs_service_arg_multi **sa;
        int i, n, len;

	if( NULL == sas ){
		VANESSA_LOGGER_ERR("Service / list service_arg : argument \"sas\" is NULL");
		return -1;
	}
	if( NULL == num ){
		VANESSA_LOGGER_ERR("Service / list service_arg : argument \"num\" is NULL");
		return -1;
	}

        n = g_list_length(l7vs_service_list);
        if (n == 0) {
                *sas = NULL;
                *num = 0;
                return 0;
        }

        sa = (struct l7vs_service_arg_multi **)calloc(n, sizeof(*sa));
	if (sa == NULL) {
		VANESSA_LOGGER_ERR("Service / list service_arg : Could not allocate memory");
		return -1;
	}

        len = 0;
        l = g_list_first(l7vs_service_list);
        for (i = 0; i < n; i++) {
                srv = (struct l7vs_service *)l->data;
                sa[i] = l7vs_service_get_service_arg(srv);
                if (sa[i] == NULL) {
                        for (i-- ; i >= 0; i--) {
                                l7vs_service_put_service_arg(sa[i]);
                        }
                        free(sa);
                        return -1;
                }
                len += sa[i]->srv_arg.len;
                l = g_list_next(l);
        }

        r = (struct l7vs_service_arg_multi *)malloc(len);
        if (r == NULL) {
                VANESSA_LOGGER_ERR("Service / list service_arg : Could not allocate memory");
                return -1;
        }

        s = r;
        for (i = 0; i < n; i++) {
                memcpy(s, sa[i], sa[i]->srv_arg.len);
                s = (struct l7vs_service_arg_multi *)((uint8_t *)s + sa[i]->srv_arg.len);
                l7vs_service_put_service_arg(sa[i]);
        }
        free(sa);
        *sas = r;
        *num = n;

        return len;
}

/*!
 * get l7vs_dest_arg lists
 * @param[in] *srv	service pointer
 * @param[out] **das	l7vs_dest_arg pointer list
 * @return		pointer list count
 */
int
l7vs_service_list_dest_arg(struct l7vs_service *srv,
                           struct l7vs_dest_arg **das)
{
        GList *l;
        struct l7vs_dest *d;
        struct l7vs_dest_arg *darg;
        int i, num;

	if( NULL == srv ){
                VANESSA_LOGGER_ERR("Service / list dest_arg : argument \"srv\" is NULL");
                return -1;
	}
	if( NULL == das ){
                VANESSA_LOGGER_ERR("Service / list dest_arg : argument \"das\" is NULL");
                return -1;
	}

        num = g_list_length(srv->dest_list);
        if (num == 0) {
                *das = NULL;
                return num;
        }

        darg = (struct l7vs_dest_arg *)malloc(num * sizeof(*darg));
        if (darg == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

        i = 0;
        for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
                d = (struct l7vs_dest *)l->data;
                l7vs_dest_to_arg(d, &darg[i]);
                i++;
        }

        *das = darg;
        return num;
}

/*!
 * create service instance. and set service value.
 * @param[in]	*arg	l7vs_service_arg pointer
 * @param[out]	*err	error code
 * @return		l7vs_service pointer
 *
 */
struct l7vs_service *
l7vs_service_create(struct l7vs_service_arg_multi *arg, int *err)
{
        struct l7vs_protomod *pmod;
        struct l7vs_scheduler *sched;
        struct l7vs_service *srv, *sref;
        struct l7vs_lsock *lsock;
	struct l7vs_dest *sorry_dest;	//! sorry-server destination
	int ret = 0;
        GList *l;

	if( NULL == arg ){
                VANESSA_LOGGER_ERR("Service / Could not create service : argument \"arg\" is NULL");
		return NULL;
	}
	if( NULL == err ){
                VANESSA_LOGGER_ERR("Service / Could not create service : argument \"err\" is NULL");
		return NULL;
	}

        *err = 0;
        lsock = l7vs_lsock_get(&arg->srv_arg.addr, arg->srv_arg.proto, arg->srv_arg.backlog);
        if (lsock == NULL) {
                VANESSA_LOGGER_ERR("Could not create listen socket");
                *err = L7VS_CONFIG_ERR_NOSOCK;
                return NULL;
        }

        pmod = l7vs_protomod_get(arg->srv_arg.protomod);
        if (pmod == NULL) {
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOMEM;
                return NULL;
        }

        sched = l7vs_sched_get(arg->srv_arg.schedmod);
        if (sched == NULL) {
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOSCHED;
                return NULL;
        }

	// create new destination for sorry-server (weight not set)
	sorry_dest = (struct l7vs_dest *)l7vs_dest_create((struct sockaddr_in *)&arg->srv_arg.sorry_addr, 0);
	if (sorry_dest == NULL) {
		l7vs_sched_put(sched);
		l7vs_protomod_put(pmod);
		l7vs_lsock_put(lsock);
		 *err = L7VS_CONFIG_ERR_NOMEM;
		return NULL;
	}

	srv = (struct l7vs_service *) calloc(1, sizeof(struct l7vs_service));
        if (srv == NULL) {
		l7vs_dest_destroy(sorry_dest);
                l7vs_sched_put(sched);
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOMEM;
                return NULL;
        }

	srv->handle = ++service_handle_base;
	if( TEMP_SERVICEHANDLE == srv->handle ){
		srv->handle = ++service_handle_base;
	}
        ret = pmod->create(&arg->protomod_arg, srv->handle);
        if (ret != 0) {
		free(srv);
		l7vs_dest_destroy(sorry_dest);
                l7vs_sched_put(sched);
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                *err = L7VS_CONFIG_ERR_NOMEM;
                return NULL;
        }

        for (l = g_list_first(l7vs_service_list); l != NULL;
             l = g_list_next(l)) {
                sref = (struct l7vs_service *)l->data;

                if (lsock != sref->lsock) {
                        continue;
                }

                if (pmod != sref->pm) {
                        continue;
                }

                if (pmod->compare(srv->handle, sref->handle) != 0) {
                        continue;
                }

                VANESSA_LOGGER_ERR("Virtual service already exists");
		l7vs_dest_destroy(sorry_dest);
                l7vs_sched_put(sched);
                l7vs_protomod_put(pmod);
                l7vs_lsock_put(lsock);
                pmod->destroy(srv->handle);
		free(srv);
                *err = L7VS_CONFIG_ERR_VS_EXISTS;
                return NULL;
        }

        srv->lsock = lsock;
        srv->pm = pmod;
        l7vs_sched_bind(sched, srv);

	// set sorry data
	srv->sorry_cc = arg->srv_arg.sorry_cc;
	srv->sorry_dest = sorry_dest;
	srv->sorry_flag = arg->srv_arg.sorry_flag;
	// set QoS value
	srv->QoS_threshold_service	= arg->srv_arg.qos_s;
	srv->QoS_threshold_conn		= arg->srv_arg.qos_c;

        l7vs_lsock_add_service(lsock, srv);
        l7vs_service_list = g_list_append(l7vs_service_list, srv);

	// create g_hash_table
	srv->conn_hash = g_hash_table_new( NULL, NULL );
        return srv;
}


/*!
 * append to conn remove list
 * @param[in]	key	not use
 * @param[in]	value	connection pointer
 * @param[in]	userdata not use
 */
static void		rmvlistappend( gpointer key, gpointer value, gpointer userdata ){
	removeconn_list = g_list_append( removeconn_list, value );
}


/*!
 * service destroy function
 * @param[in] *srv	service pointer
 * @return    void
 */
int
l7vs_service_destroy(struct l7vs_service *srv)
{
        struct l7vs_scheduler *sched;
	struct l7vs_dest * rmv_dest;
	struct l7vs_conn * rmvconn;

	if( NULL == srv ){
		VANESSA_LOGGER_ERR( "Service Destroy failure: argument is NULL" );
		return -1;
	}
	
	//find service from glist(l7vs_service_list)
	if( 0 == g_list_length( l7vs_service_list ) ){
		VANESSA_LOGGER_ERR( "Service Destroy failure: service not created" );
		return -1;
	}
	if( NULL == g_list_find( l7vs_service_list, srv ) ){
		VANESSA_LOGGER_ERR( "Service Destroy failure: cannot find in list" );
		return -1;
	}
	
	// remove all hash-table members.
	g_hash_table_foreach( srv->conn_hash, rmvlistappend, NULL );
	
	for( removeconn_list = g_list_first( removeconn_list );
	     removeconn_list != NULL; ){
		rmvconn = (struct l7vs_conn*)removeconn_list->data;
		l7vs_conn_destroy( rmvconn );
		removeconn_list = g_list_remove( removeconn_list, rmvconn );
	}
	

	// destroy g_hash_table
	g_hash_table_destroy( srv->conn_hash );

	// remove all dest(exclude sorry_dest)
	for( srv->dest_list = g_list_first( srv->dest_list ); srv->dest_list != NULL; ){
		rmv_dest = (struct l7vs_dest*)srv->dest_list->data;
		srv->dest_list = g_list_remove( srv->dest_list, rmv_dest );
		l7vs_dest_destroy( rmv_dest );
	}

	sched = srv->scheduler;
	l7vs_service_list = g_list_remove(l7vs_service_list, srv);
	l7vs_sched_unbind(sched, srv);
	l7vs_dest_destroy(srv->sorry_dest);
	l7vs_sched_put(sched);
        l7vs_lsock_remove_service(srv->lsock, srv);
        l7vs_lsock_put(srv->lsock);
        srv->pm->destroy(srv->handle);
	l7vs_protomod_put( srv->pm );
	free(srv);

	return 0;
}

/*!
 * serch from service list.
 * @param[in]	*arg service arg
 * @return	service pointer ( NULL is no lookup )
 *
 */
struct l7vs_service *
l7vs_service_lookup(struct l7vs_service_arg_multi *arg) //checks if the virtual service to be added already exists--Anshu
{
        GList *l;
        struct l7vs_lsock *lsock;
        struct l7vs_protomod *pmod;
        struct l7vs_service *s, *sref;
	int ret = 0;

	if( NULL == arg ){
		VANESSA_LOGGER_ERR( "Service lookup failure: agreement is NULL" );
		return NULL;
	}

        lsock = l7vs_lsock_table_lookup(&arg->srv_arg.addr, arg->srv_arg.proto);
        if (lsock == NULL) {
                return NULL;
        }

        pmod = l7vs_protomod_lookup(arg->srv_arg.protomod);
        if (pmod == NULL) {
                return NULL;
        }

	s = (struct l7vs_service *) calloc(1, sizeof(struct l7vs_service));
	if (s == NULL) {
		return NULL;
	}

	s->handle = TEMP_SERVICEHANDLE;
        ret = pmod->create(&arg->protomod_arg, s->handle);
	if (ret != 0) {
		free(s);
		return NULL;
	}

        for (l = g_list_first(l7vs_service_list);
             l != NULL; l = g_list_next(l)) {
                sref = (struct l7vs_service *)l->data;
                if (lsock != sref->lsock) {
                        continue;
                }

                if (pmod != sref->pm) {
                        continue;
                }

                if (pmod->compare(s->handle, sref->handle) != 0) {
                        continue;
                }

                pmod->destroy(s->handle);
		free(s);
                return sref;
        }
        if (s != NULL) {
                pmod->destroy(s->handle);
		free(s);
        } 
        return NULL;
}

/*!
 * real server add on service
 * @param[out]	*srv	service pointer
 * @param[in]	*darg	l7vs_dest_arg pointer
 * @return 		success = 0 fail = -1
 */
int
l7vs_service_add_dest(struct l7vs_service *srv,
                      struct l7vs_dest_arg *darg)
{
        struct l7vs_dest *d;

	if( NULL == srv || NULL == darg ){
                VANESSA_LOGGER_ERR("Service / add dest : Invalid argument");
		return -1;
	}

        d = l7vs_service_lookup_dest(srv, &darg->addr);
        if (d != NULL) {
                VANESSA_LOGGER_INFO("Cannot add duplicate real service");
                return -1;
        }

        d = (struct l7vs_dest*) l7vs_dest_create(&darg->addr, darg->weight);
        if (d == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
	}
		else {
			VANESSA_LOGGER_INFO("ADD_RS: IFRM004: added real server");
		}
        
        srv->dest_list = g_list_append(srv->dest_list, d);

        return 0;
}

/* remove real server pointer function
 * @param[out]	*srv	l7vs_service_pointer
 * @param[in]   *darg	l7vs_dest pointer
 * @return		success = 0 false = -1
 */
int
l7vs_service_remove_dest(struct l7vs_service *srv,
                         struct l7vs_dest_arg *darg)
{
	struct l7vs_dest *d;
	struct l7vs_conn * tmp_conn;


	if( NULL == srv || NULL == darg ){
                VANESSA_LOGGER_ERR("Service / remove dest : Invalid argument");
		return -1;
	}

	d = l7vs_service_lookup_dest(srv, &darg->addr);
        if (d == NULL) {
                VANESSA_LOGGER_ERR("No such real service");
                return -1;
        }
	else {
		VANESSA_LOGGER_INFO("DEL_RS: IFRM005: removed real server");
	}

	g_hash_table_foreach( srv->conn_hash, rmvlistappend, NULL );
	for( removeconn_list = g_list_first( removeconn_list ); removeconn_list != NULL; ){
		tmp_conn = (struct l7vs_conn*)removeconn_list->data;
		if( d == tmp_conn->dest ){
			l7vs_conn_destroy( tmp_conn );
		}
		removeconn_list = g_list_remove( removeconn_list, tmp_conn );
	} 

	srv->dest_list = g_list_remove(srv->dest_list, d);
	l7vs_dest_destroy(d);

        return 0;
}

/*!
 * schedule dest on con.
 * @param[in] *srv	service pointer
 * @param[in] *conn	l7vs_conn pointer
 * @return		success = 0 false = -1
 */
int
l7vs_service_schedule(struct l7vs_service *srv, 
                      struct l7vs_conn *conn)
{
        struct l7vs_dest *dest;

	if( NULL == srv || NULL == conn || NULL == srv->scheduler ){
                VANESSA_LOGGER_ERR("Service / lookup schedule-module : Invalid argument");
		return -1;
	}

        dest = srv->scheduler->schedule(srv, conn);
        if (dest == NULL) {
                VANESSA_LOGGER_ERR("no real server defined");
                return -1;
        }

        return l7vs_conn_connect_rs(conn, dest);
}


/*!
 * 
 *
 *
 */
int
l7vs_service_establish(struct l7vs_service *srv, 
                       struct l7vs_conn *conn)
{
        return 0;
}

/*!
 * connection list append conn in service
 * @param[in]	*srv	service pointer
 * @param[in]	*conn	l7vs_conn pointer
 * @return	void
 */
int
l7vs_service_register_conn(struct l7vs_service *srv,
                           struct l7vs_conn *conn)
{
	if( NULL == srv || NULL == conn ){
                VANESSA_LOGGER_ERR("Service / register connection : Invalid argument");
		return -1;
	}

	if( NULL != g_hash_table_lookup( srv->conn_hash, &conn->ciom->fd ) ){
                VANESSA_LOGGER_ERR("Service / register connection : conflict connection.");
		return -1;
//		g_hash_table_remove( srv->conn_hash, &(conn->ciom->fd) );
	}

	g_hash_table_insert( srv->conn_hash, &conn->ciom->fd, conn );
	return 0;
}

/*!
 * connection list delete con in service
 * @param[in]	*srv	service pointer
 * @param[in]	*conn	l7vs_conn pointer
 * @return		void
 */
int
l7vs_service_remove_conn(struct l7vs_service *srv,
                              struct l7vs_conn *conn)
{
	if( NULL == srv || NULL == conn ){
                VANESSA_LOGGER_ERR("Service / remove connection : Invalid argument");
		return -1;
	}

	g_hash_table_remove( srv->conn_hash, &conn->ciom->fd );
	return 0;
}

/*!
 * destroy all member in all service.
 *
 */
void
l7vs_service_flush_all(void)
{
	GList *l;
	struct l7vs_service *srv;

	while ((l = g_list_first(l7vs_service_list)) != NULL) {
		srv = (struct l7vs_service *)l->data;
		l7vs_service_destroy( srv );
	}
}

/*!
 * set QoS(TraficControl) Threashold value
 * @param[in]	*srv	service pointer
 * @param[in]	*arg	l7vs_service_arg_multi pointer
 * return		0 = success / -1 = failur
 */
int
l7vs_service_set_QoS_Threashold( struct l7vs_service * srv, struct l7vs_service_arg_multi * arg )
{
	if( NULL == srv ){
                VANESSA_LOGGER_ERR("Service set QoS Threashold / argument srv is NULL");
		return -1;
	}
	if( NULL == arg ){
                VANESSA_LOGGER_ERR("Service set QoS Threashold / argument arg is NULL");
		return -1;
	}
	srv->QoS_threshold_service	= arg->srv_arg.qos_s;
	srv->QoS_threshold_conn		= arg->srv_arg.qos_c;
	return 0;
}

/*!
 * set Sorry-Server values
 * @param[in]   *srv    service pointer
 * @param[in]   *arg    l7vs_service_arg_multi pointer
 * return               0 = success / -1 = failure
 */
int
l7vs_service_set_SorryServer_Values( struct l7vs_service * srv, struct l7vs_service_arg_multi * arg )
{
	struct sockaddr_in * saddr;
	if( NULL == srv ){
		VANESSA_LOGGER_ERR("Service set SorryServer Values / argument srv is NULL");
		return -1;
	}
	if( NULL == arg ){
		VANESSA_LOGGER_ERR("Service set QoS Threashold / argument arg is NULL");
		return -1;
	}
	// set sorry data
	saddr = (struct sockaddr_in *)&arg->srv_arg.sorry_addr;
	srv->sorry_dest->addr   = *saddr;
	srv->sorry_cc   = arg->srv_arg.sorry_cc;
	srv->sorry_flag = arg->srv_arg.sorry_flag;
	return 0;
}

