/*
 * @file  l7vsadm.c
 * @brief the main module of l7vsadm 
 * @brief it interprets command line option 
 * @brief and communucate with l7vsd 
 *
 * L7VSADM: Virtual Server Administration Program for L7vsd
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE     /* for getopt_long() */
#endif /* _GNU_SOURCE */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/un.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <getopt.h>
#include <vanessa_logger.h>
#include <vanessa_adt.h>
#include <net/if.h>
#include "glib.h"
#include "l7vs.h"
#include "l7vs_config.h"
#include "l7vs_service.h"
#include "l7vs_module.h"
#include "l7vs_sync.h"

#define L7VSADM_NAME            "l7vsadm"
#define L7VSADM_LIST_VS_BUFSIZE 1024
#define L7VSADM_LIST_RS_RSNUM   16
#define L7VSADM_DEFAULT_BACKLOG 127
#define L7VSADM_DEFAULT_SCHEDULER "rr"

#define L7VSADM_RECV_FAILED             (-1)
#define L7VSADM_RECV_TOO_LARGE          (-2)

#define L7VSADM_CONFIG_SOCK_WAIT 10

struct l7vsadm_options {
        /* XXX not yet */
#define L7VSADM_OP_LIST         0       /* run as default */
#define L7VSADM_OP_ADD_VS       1
#define L7VSADM_OP_ADD_RS       2
#define L7VSADM_OP_DEL_VS       3
#define L7VSADM_OP_DEL_RS       4
#define L7VSADM_OP_EDIT_VS      5
#define L7VSADM_OP_EDIT_RS      6
#define L7VSADM_OP_FLUSH_VS     7
#define L7VSADM_OP_SYNC_INIT	8
#define L7VSADM_OP_SYNC_FINI	9
#define L7VSADM_OP_VERBOSE	10
#define L7VSADM_OP_KEY		11
        int operation;
        struct sockaddr_in vaddr;
        struct sockaddr_in raddr;
        char *protomodname;
        char *schedmod;
        struct l7vs_protomod *protomod;
        int persist;
        int backlog;
        int weight;
        int isnumeric;
        vanessa_dynamic_array_t *pm_args;
	int sorry_cc;					//! sorry connection count limit
	struct sockaddr_storage sorry_addr;		//! sorry-server address
	int sorry_flag;					//! sorry flag
	unsigned long long qos_s;			//! QoS(Trafic Control) Threashold for Service
	unsigned long long qos_c;			//! QoS(Trafic Control) Threashold for Clients
	int sync_mode;					//! sync mode(send/recv)
	char *sync_ipaddr;				//! sync-server ip address
	char *sync_port;				//! sync-server port number
	char *sync_nic;					//! sync-server nic name
};

struct l7vsadm_operations {
        int code;
        int (*func)(int, struct l7vsadm_options *);
};

static int signal_flg = 0;   //! signal recve flug
static int received_sig = 0;

extern void *l7vs_module_load(char *modname, char *type);

static void usage(FILE *fp);
static int parse_args(struct l7vsadm_options *opt, int argc, char *argv[]);
static int parse_service(struct sockaddr_in *addr, const char *srvstr);
static int parse_sync_address(const char *addr_str, int mode, char **ipaddr, char **port);
static char *get_servicename(struct sockaddr_in *addr, int isnumeric);
static int create_socket(void);
static void destroy_socket(void);
static int send_request(int s, void *req, size_t len);
static int sendv_request(int s, struct iovec *iov, int iovlen);
static int recv_response(int s, struct iovec *iov, size_t iovlen);
static const char *config_strerror(int err);
static int list_vs(int s, struct l7vsadm_options *opt);
static int list_rs(int s, struct l7vs_service_arg *sarg, int isnumeric);
void list_ss(struct sockaddr_in);
static int operate_vs(int s, struct l7vsadm_options *opt);
static int operate_rs(int s, struct l7vsadm_options *opt);
static int set_signal(void);

static const char l7vs_config_sockname[] = L7VS_CONFIG_SOCK_PATH "/l7vs";
static const char *l7vs_version_string = L7VS_VERSION_STRING;
static char local_sockname[sizeof(L7VS_CONFIG_SOCK_PATH) + sizeof(L7VSADM_NAME) + 10];
static char service_name[NI_MAXHOST + NI_MAXSERV + 2];

static void
usage(FILE *fp)
{
	if (!fp) {
		VANESSA_LOGGER_ERR("fp is NULL.");
		return;
	}
        fprintf(fp,
                "Usage: \n"
                "  l7vsadm -A|E -t service-address -m proto-module [module-args]\n"
		"          [-s scheduler] [-u connection-count] [-b sorry-server]\n"
		"          [-f sorry-flag] [-Q QoSval-service] [-q QoSval-clients]\n"
                "  l7vsadm -D -t service-address -m proto-module [module-args]\n"
                "  l7vsadm -C\n"
#if 0
                "  l7vsadm -R\n"
                "  l7vsadm -S [-n]\n"
#endif
                "  l7vsadm -a|e -t service-address -m proto-module [module-args]\n"
                "          -r server-address [-w weight]\n"
                "  l7vsadm -d -t service-address -m proto-module [module-args]\n"
                "          -r server-address\n"
		"  l7vsadm -I -o sync-send-server-address -N nic-name\n"
		"  l7vsadm -I -i sync-recv-server-address -N nic-name\n"
		"  l7vsadm -F\n"
                "  l7vsadm -l [-n]\n"
                "  l7vsadm -V\n"
                "  l7vsadm -K\n"
                "  l7vsadm -h\n"
		"\n");
        fprintf(fp,
                "Commands:\n"
                "  --add-service     -A        add virtual service with options\n"
                "  --edit-service    -E        edit virtual service with options\n"
                "  --delete-service  -D        delete virtual service with options\n"
                "  --flush           -C        flush virtual service\n"
#if 0
                "  --restore         -R        restore rules from stdin\n"
                "  --save            -S        save rules to stdout\n"
#endif
                "  --add-server      -a        add real server with options\n"
                "  --edit-server     -e        edit real server with options\n"
                "  --delete-server   -d        delete real server with options\n"
		"  --init-sync       -I        initialize sync server with options\n"
		"  --fini-sync       -F        finalize sync server with options\n"
                "  --list            -l        list the table\n"
                "  --verbose         -V        list the table in verbose format\n"
                "  --key             -K        list the table in key setting format\n"
                "  --help            -h        show usage\n"
                "\n");
        fprintf(fp,
                "Options:\n"
                "  --tcp-service  -t service-address            service-address is host:port\n"
                "  --proto-module -m proto-module [module-args] protocol module name and module argment\n"
                "  --scheduler    -s scheduler                  one of rr,lc,wrr\n"
		"  --upper        -u connection-count           maximum number of connections\n"
		"  --bypass       -b sorry-server               sorry server address is host:port\n"
		"  --flag         -f sorry-flag                 sorry status set to virtual service\n"
		"  --qos-service  -Q QoSval-service             QoS Threashold(bps) set to virtual service\n"
		"  --qos-clients  -q QoSval-clients             QoS Threashold(bps) set to client connections\n"
                "  --real-server  -r server-address             server-address is host:port\n"
                "  --weight       -w weight                     scheduling weight set to real server\n"
		"  --out-sync     -o sync-send-server-address   sync send server address is host:port\n"
		"  --in-sync      -i sync-recv-server-address   sync recieve server address is host(don't care):port\n"
		"  --nic          -N nic-name                   sync server nic name\n"
		"  --numeric      -n                            list the table in numeric\n"
		"\n");
}

/*
 * XXX Comment here.
 */
static int
parse_args(struct l7vsadm_options *opt, int argc, char *argv[])
{
#define OPTION_STRING   "-lVKADECadeIFnhr:m:t:s:w:u:b:f:Q:q:o:i:N:"

        static struct option options[] = {
                /* operation options (toplevel) */
                {"list",                no_argument,            NULL, 'l'},
                {"verbose",		no_argument,            NULL, 'V'},
		{"key",			no_argument,		NULL, 'K'},
                {"add-service",         no_argument,            NULL, 'A'},
                {"delete-service",      no_argument,            NULL, 'D'},
                {"edit-service",        no_argument,            NULL, 'E'},
                {"flush",               no_argument,            NULL, 'C'},
                {"add-server",          no_argument,            NULL, 'a'},
                {"delete-server",       no_argument,            NULL, 'd'},
                {"edit-server",         no_argument,            NULL, 'e'},
		{"sync-init",           no_argument,            NULL, 'I'},
		{"sync-fini",           no_argument,            NULL, 'F'},

                {"real-server",         required_argument,      NULL, 'r'},
                {"proto-module",        required_argument,      NULL, 'm'},

                {"tcp-service",         required_argument,      NULL, 't'},
                {"scheduler",           required_argument,      NULL, 's'},

                {"weight",              required_argument,      NULL, 'w'},
		{"upper",		required_argument,	NULL, 'u'},
		{"bypass",		required_argument,	NULL, 'b'},
		{"flag",		required_argument,	NULL, 'f'},

		{"qos-service",		required_argument,	NULL, 'Q'},
		{"qos-clients",		required_argument,	NULL, 'q'},

		{"out-sync",		required_argument,	NULL, 'o'},
		{"in-sync",		required_argument,	NULL, 'i'},
		{"nic",			required_argument,	NULL, 'N'},

                {"numeric",             no_argument,            NULL, 'n'},

                {"help",                no_argument,            NULL, 'h'},

                {NULL,                  0,                      NULL, 0}
        };

        int c;
        int ret;
	int cmd_flag = 0;
	int cmd_vs_flag = 0, cmd_rs_flag = 0, cmd_sync_flag = 0;
	int opt_t_flag = 0, opt_m_flag = 0, opt_r_flag = 0, opt_o_flag = 0, opt_i_flag = 0, opt_N_flag = 0, opt_w_flag = 0;
	char qosstr[33];


	if(opt == NULL)
	{
		return(-1);
	}

#define SET_OP(OPTP, OP) \
do {                                                                    \
        if ((OPTP)->operation) {                                        \
                VANESSA_LOGGER_ERR("Operation option conflicts");       \
                exit(1);                                                \
        }                                                               \
        (OPTP)->operation = (OP);                                       \
        cmd_flag = 1;                                                   \
} while (0)

        opterr = 0;
        while ((c = getopt_long(argc, argv, OPTION_STRING, options, NULL)) != -1) {
                switch (c) {
                case 'l':
                        SET_OP(opt, L7VSADM_OP_LIST);
                        break;
                case 'V':
                        SET_OP(opt, L7VSADM_OP_VERBOSE);
                        break;
		case 'K':
			SET_OP(opt, L7VSADM_OP_KEY);
			break;
                case 'A':
                        SET_OP(opt, L7VSADM_OP_ADD_VS);
			cmd_vs_flag = 1;
                        break;
                case 'D':
                        SET_OP(opt, L7VSADM_OP_DEL_VS);
			cmd_vs_flag = 1;
                        break;
                case 'E':
                        SET_OP(opt, L7VSADM_OP_EDIT_VS);
			cmd_vs_flag = 1;
                        break;
                case 'C':
                        SET_OP(opt, L7VSADM_OP_FLUSH_VS);
                        break;
                case 'a':
                        SET_OP(opt, L7VSADM_OP_ADD_RS);
			cmd_rs_flag = 1;
                        break;
                case 'd':
                        SET_OP(opt, L7VSADM_OP_DEL_RS);
			cmd_rs_flag = 1;
                        break;
                case 'e':
                        SET_OP(opt, L7VSADM_OP_EDIT_RS);
			cmd_rs_flag = 1;
                        break;
		case 'I':
			SET_OP(opt, L7VSADM_OP_SYNC_INIT);
			cmd_sync_flag = 1;
			break;
		case 'F':
			SET_OP(opt, L7VSADM_OP_SYNC_FINI);
			break;
                case 't':
                        ret = parse_service(&opt->vaddr, optarg);
                        if (ret < 0) {
                                VANESSA_LOGGER_ERR_UNSAFE("Could not parse"
                                                " service string: %s",
                                                optarg);
                                exit(1);
                        }
			opt_t_flag = 1;
                        break;
                case 'r':
                        ret = parse_service(&opt->raddr, optarg);
                        if (ret < 0) {
                                VANESSA_LOGGER_ERR_UNSAFE("Could not parse"
                                                " service string: %s",
                                                optarg);
                                exit(1);
                        }
			opt_r_flag = 1;
                        break;

                case 's':
                        if (opt->schedmod != NULL) {
                                VANESSA_LOGGER_ERR("scheduler module option"
                                                   "conflicts");
                                exit(1);
                        }
                        if (strlen(optarg) >= L7VS_MODNAME_LEN - 1) {
                                VANESSA_LOGGER_ERR_UNSAFE("%s: name too long",
                                                          optarg);
                                exit(1);
                        }
                        opt->schedmod = strdup(optarg);
                        if (opt->schedmod == NULL) {
                                VANESSA_LOGGER_ERR("Could not allocate memory");
                                exit(1);
                        }
                        break;

                case 'm':
                        if (opt->protomodname != NULL) {
                                VANESSA_LOGGER_ERR("protocol module option"
                                                   "conflicts");
                                exit(1);
                        }
                        if (strlen(optarg) >= L7VS_MODNAME_LEN - 1) {
                                VANESSA_LOGGER_ERR_UNSAFE("%s: name too long",
                                                          optarg);
                                exit(1);
                        }
                        opt->protomodname = strdup(optarg);
                        if (opt->protomodname == NULL) {
                                VANESSA_LOGGER_ERR("Could not allocate memory");
                                exit(1);
                        }
                        opt->protomod = (struct l7vs_protomod *)
                                l7vs_module_load(opt->protomodname,
                                                 "protomod");
                        if (opt->protomod == NULL) {
                                exit(1);
                        }
			opt_m_flag = 1;

                        break;

                case 'w':
                        opt->weight = atoi(optarg);
			opt_w_flag = 1;
                        break;
                
                case 'n':
                        opt->isnumeric = 1;
                        break;
		case 'u':
			// set sorry connection limit
			opt->sorry_cc = atoi(optarg);
			if (opt->sorry_cc <= 0) {
				VANESSA_LOGGER_ERR_UNSAFE("Invalid sorry-connection value %d", 
							opt->sorry_cc);
				exit(1);
			}
			break;
		case 'b':
			// set sorry-server address
			ret = parse_service((struct sockaddr_in *)&opt->sorry_addr, optarg);
			if (ret < 0) {
				VANESSA_LOGGER_ERR_UNSAFE("Could not parse service string: %s",
							optarg);
				exit(1);
			}
                        break;
		case 'f':
			// set sorry flag
			opt->sorry_flag = atoi(optarg);
			if (opt->sorry_flag != 0 && opt->sorry_flag != 1) {
				VANESSA_LOGGER_ERR("Sorry-flag is 0 or 1 only");
				exit(1);
			}
			break;
		case 'Q':
			// set QoS Threashold(service)
			if( strlen(optarg) < 2 ){
				opt->qos_s = atoi(optarg);
			}
			else{
				if( optarg[strlen(optarg)-1] == 'K' || optarg[strlen(optarg)-1] == 'k' ){
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_s = ( (unsigned long long)atol(optarg) / 8ULL );
					sscanf(qosstr,"%llu",&(opt->qos_s));
					if( 10485760000ULL < opt->qos_s ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10485760000K");
						exit(1);
					}
					opt->qos_s = (opt->qos_s * 1024ULL) / 8ULL;
				}else
				if( optarg[strlen(optarg)-1] == 'M' || optarg[strlen(optarg)-1] == 'm' ){
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_s = ( (unsigned long long)atoi(optarg) / 8ULL );
					sscanf(qosstr,"%llu",&(opt->qos_s));
					if( 10240000ULL < opt->qos_s ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10240000M");
						exit(1);
					}
					opt->qos_s = (opt->qos_s * 1024ULL * 1024ULL)/8ULL;
				}else
				if( optarg[strlen(optarg)-1] == 'G' || optarg[strlen(optarg)-1] == 'g' ){
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_s = ( (unsigned long long)atoi(optarg) / 8ULL );
					sscanf(qosstr,"%llu",&(opt->qos_s));
					if( 10000ULL < opt->qos_s ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10000G");
						exit(1);
					}
					opt->qos_s = (opt->qos_s * 1024ULL * 1024ULL * 1024ULL) / 8ULL;
				}else{
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_s = (unsigned long long)atoi(optarg);
					sscanf(optarg,"%llu",&(opt->qos_s));
					if( 1342177280000ULL < opt->qos_s ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10737418240000");
						exit(1);
					}
				}
			}
			break;
		case 'q':
			// set QoS Threashold(clients)
			if( strlen(optarg) < 2 ){
				opt->qos_c = atoi(optarg);
			}
			else{
				if( optarg[strlen(optarg)-1] == 'K' || optarg[strlen(optarg)-1] == 'k' ){
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_c = ( (unsigned long long)atoi(optarg) / 8ULL );
					sscanf(qosstr,"%llu",&(opt->qos_c));
					if( 10485760000ULL < opt->qos_c ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10485760000K");
						exit(1);
					}
					opt->qos_c = (opt->qos_c * 1024ULL) / 8ULL;
				}else
				if( optarg[strlen(optarg)-1] == 'M' || optarg[strlen(optarg)-1] == 'm' ){
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_c = ( (unsigned long long)atoi(optarg) / 8ULL );
					sscanf(qosstr,"%llu",&(opt->qos_c));
					if( 10240000ULL < opt->qos_c ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10240000M");
						exit(1);
					}
					opt->qos_c = (opt->qos_c * 1024ULL * 1024ULL) / 8ULL;
				}else
				if( optarg[strlen(optarg)-1] == 'G' || optarg[strlen(optarg)-1] == 'g' ){
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_c = ( (unsigned long long)atoi(optarg) / 8ULL );
					sscanf(qosstr,"%llu",&(opt->qos_c));
					if( 10000ULL < opt->qos_c ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10000G");
						exit(1);
					}
					opt->qos_c = (opt->qos_c * 1024ULL * 1024ULL * 1024ULL) / 8ULL;
				}else{
					strncpy( qosstr, optarg, strlen(optarg)-1 );
					//opt->qos_c = (unsigned long long)atoi(optarg);
					sscanf(optarg,"%llu",&(opt->qos_c));
					if( 1342177280000ULL < opt->qos_c ){
						VANESSA_LOGGER_ERR("QoS-Threashold too large. value less than 10737418240000");
						exit(1);
					}
				}
			}
			break;
		case 'o':
			// set sync mode(SYNC_SEND)
			opt->sync_mode = SYNC_SEND;
			// set sync-send-server address
			ret = parse_sync_address(optarg, SYNC_SEND, &opt->sync_ipaddr, &opt->sync_port);
			if (ret < 0) {
				VANESSA_LOGGER_ERR_UNSAFE("Could not parse sync address string: %s",
							optarg);
				exit(1);
			}
			opt_o_flag = 1;
			break;
		case 'i':
			// set sync mode(SYNC_RECV)
			opt->sync_mode = SYNC_RECV;
			// set sync-recv-server address
			ret = parse_sync_address(optarg, SYNC_RECV, &opt->sync_ipaddr, &opt->sync_port);
			if (ret < 0) {
				VANESSA_LOGGER_ERR_UNSAFE("Could not parse sync address string: %s",
							optarg);
				exit(1);
			}
			opt_i_flag = 1;
			break;
		case 'N':
			// nic name length check
			if (strlen(optarg) >= IFNAMSIZ - 1) {
				VANESSA_LOGGER_ERR_UNSAFE("%s: name too long",
							optarg);
				exit(1);
			}
			// set nic name
			opt->sync_nic = strdup(optarg);
			if (opt->sync_nic == NULL) {
				VANESSA_LOGGER_ERR("Could not allocate memory");
				exit(1);
			}
			opt_N_flag = 1;
			break;

                case 'h':
                        usage(stdout);
                        exit(0);
                        break;

                case '1':
                default:
//                      VANESSA_LOGGER_DEBUG_UNSAFE("added %s", argv[optind-1]);
                        if (vanessa_dynamic_array_add_element(
                                opt->pm_args, argv[optind - 1]) == NULL) {
                                VANESSA_LOGGER_ERR("Could not allocate memory");
                                exit(1);
                                
                        }
                        break;
                }
        }

	// Check command option
	// When the command is only "l7vsadm", it becomes a default command. (-l)
	if (!cmd_flag && argc != 1) {
		VANESSA_LOGGER_ERR("Command option nothing.\n");
		usage(stderr);
		exit(1);
	}
	// Check indispensable option for virtual service command
	if (cmd_vs_flag) {
		if (!opt_t_flag || !opt_m_flag) {
			VANESSA_LOGGER_ERR("Need -t and -m option for virtual service command.\n");
			exit(1);
		}
	}
	// Check indispensable option for real server command
	if (cmd_rs_flag) {
		if (!opt_t_flag || !opt_m_flag || !opt_r_flag) {
			VANESSA_LOGGER_ERR("Need -t and -m and -r option for real server command.\n");
			exit(1);
		}
	}
	// Check indispensable option for sync server command
	if (cmd_sync_flag) {
		if ((!opt_o_flag && !opt_i_flag) || (opt_o_flag && opt_i_flag)) {
			VANESSA_LOGGER_ERR("Need -o or -i option for sync server command.\n");
			exit(1);
		}
		if (!opt_N_flag) {
			VANESSA_LOGGER_ERR("Need -N option for sync server command.\n");
			exit(1);
		}
	}
	// set default weight
	if (!opt_w_flag) {
		opt->weight = 1;
	}

#undef SET_OP

        return 0;
}

static int
parse_service(struct sockaddr_in *addr, const char *srvstr)
{
        struct addrinfo *res, hints;
        struct sockaddr_in *sin;
        char *s, *t;
        int ret;

	// Argment check
	if (!srvstr) {
		VANESSA_LOGGER_ERR("srvstr is NULL");
		return -1;
	}

        s = strdup(srvstr);
        t = strrchr(s, ':');
        if (t == NULL) {
                free(s);
                return -1;
        }
        *t++ = '\0';

        memset(&hints, 0, sizeof(hints));
        hints.ai_family = PF_INET; /* inet only. no support for inet6 (yet) */
        hints.ai_socktype = SOCK_STREAM;
        ret = getaddrinfo(s, t, &hints, &res);
        if (ret != 0) {
                VANESSA_LOGGER_ERR_UNSAFE("%s:%s: %s",
                                          s, t, gai_strerror(ret));
                free(s);
                return -1;
        }
        free(s);

        /*
         * We always use the first entry, because we can't distinguish
         * which entry you want to specify if we have multiple entries.
         */
        sin = (struct sockaddr_in *)res->ai_addr;
        if (sin->sin_addr.s_addr == htonl(INADDR_ANY)) {
                VANESSA_LOGGER_ERR("You can't specify INADDR_ANY"
                                   " for virtual service");
                freeaddrinfo(res);
                return -1;
        }
        
        if (sin->sin_port == htons(0)) {
                VANESSA_LOGGER_ERR("You can't specify port number 0");
                freeaddrinfo(res);
                return -1;
        }

        *addr = *sin;
        freeaddrinfo(res);

        return 0;
}

/*!
 * Parse and checck sync server adress data.
 * @param[in]	*addr_str	sync server adress string
 * @param[in]	mode	sync mode(send/recv)
 * @param[out]	*ipaddr	result of sync server ip adress
 * @param[out]	*port	result of sync server port
 * @return	int	parse and check result OK=0, NG=-1
 */
static int
parse_sync_address(const char *addr_str, int mode, char **ipaddr, char **port)
{
        struct addrinfo *res, hints;
        struct sockaddr_in *sin = NULL;
        char *s, *t;
        int ret;

	// Argment check
	if (!addr_str) {
		VANESSA_LOGGER_ERR("addr_str is NULL");
		return -1;
	}
	if (mode != SYNC_SEND && mode != SYNC_RECV) {
		VANESSA_LOGGER_ERR("Invalid mode");
		return -1;
	}

        s = strdup(addr_str);
        t = strrchr(s, ':');
        if (t == NULL) {
                free(s);
                return -1;
        }
        *t++ = '\0';

	// sync server address check by getaddrinfo()
	memset(&hints, 0, sizeof(struct addrinfo));
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	if (mode == SYNC_SEND) {
		ret = getaddrinfo(s, t, &hints, &res);
		if (ret != 0) {
			free(s);
			return -1;
		}
		// ip address check
		sin = (struct sockaddr_in *)res->ai_addr;
		if (sin->sin_addr.s_addr == htonl(INADDR_ANY)) {
			VANESSA_LOGGER_ERR("You can't specify INADDR_ANY"
					" for virtual service");
			freeaddrinfo(res);
			free(s);
			return -1;
		}
		*ipaddr = s;
	} else if (mode == SYNC_RECV) {
		hints.ai_flags = AI_PASSIVE;
		ret = getaddrinfo(NULL, t, &hints, &res);
		if (ret != 0) {
			free(s);
			return -1;
		}
		sin = (struct sockaddr_in *)res->ai_addr;
		*ipaddr = NULL;
	}
	// port check
	if (sin->sin_port == htons(0)) {
		VANESSA_LOGGER_ERR("You can't specify port number 0");
		freeaddrinfo(res);
		free(s);
		return -1;
	}
	*port = t;

	freeaddrinfo(res);
	return 0;
}

static char *
get_servicename(struct sockaddr_in *addr, int isnumeric)
{
	char hostname[NI_MAXHOST], portname[NI_MAXSERV];
        int flags;
        int ret;

	// Argment check
	if (!addr) {
		VANESSA_LOGGER_ERR("addr is NULL.\n");
		return NULL;
	}
	if (isnumeric != 0 && isnumeric != 1) {
		VANESSA_LOGGER_ERR("invalid isnumeric.\n");
		return NULL;
	}

        flags = 0;
        if (isnumeric) {
                flags = NI_NUMERICHOST | NI_NUMERICSERV;
        }

        ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
                          hostname, sizeof(hostname),
                          portname, sizeof(portname), flags);
	if (ret == EAI_AGAIN) {
		flags = flags | NI_NUMERICHOST;
		ret = getnameinfo((struct sockaddr *)addr, sizeof(*addr),
				hostname, sizeof(hostname),
				portname, sizeof(portname), flags);
		if (ret != 0) {
			VANESSA_LOGGER_ERR_UNSAFE("getnameinfo: %s",
						gai_strerror(ret));
			return NULL;
		}
	} else if (ret != 0) {
                VANESSA_LOGGER_ERR_UNSAFE("getnameinfo: %s",
                                          gai_strerror(ret));
                return NULL;
        }
	sprintf(service_name, "%s:%s", hostname, portname);
	return service_name;
}

static int
create_socket(void)
{
        struct sockaddr_un addr;
	char sockname[sizeof(L7VS_CONFIG_SOCK_PATH) + sizeof(L7VSADM_NAME) + 10];
        int opt;
        int ret;
        int s;
	int retry_count = 0;

	sprintf(sockname, "%s/%s-%d", L7VS_CONFIG_SOCK_PATH, L7VSADM_NAME, getpid());

        unlink(sockname);
	if (sizeof(sockname) > sizeof(addr.sun_path)) {
                VANESSA_LOGGER_ERR("Internal error. socket name too long.");
                return -1;
        }

        memset(&addr, 0, sizeof(addr));
	memcpy(addr.sun_path, sockname, sizeof(sockname));
        addr.sun_family = AF_LOCAL;

        s = socket(PF_LOCAL, SOCK_DGRAM, 0);
        if (s < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("socket: %s", strerror(errno));
                return s;
        }

        opt = 1;
        ret = setsockopt(s, SOL_SOCKET, SO_PASSCRED, &opt, sizeof(opt));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("setsockopt SO_PASSCRED: %s",
                                          strerror(errno));
                close(s);
                return ret;
        }

        ret = bind(s, (struct sockaddr *)&addr, SUN_LEN(&addr));
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("bind on %s: %s",
                                          sockname, strerror(errno));
                close(s);
                return ret;
        }

	strcpy(local_sockname, sockname);

        memset(&addr, 0, sizeof(addr));
        addr.sun_family = AF_LOCAL;
        strcpy(addr.sun_path, l7vs_config_sockname);

	while (1) {
		// try connect to config socket
		ret = connect(s, (struct sockaddr *)&addr, SUN_LEN(&addr));
		if (ret == 0) {
			// connect OK
			break;
		}
		retry_count++;
		if (retry_count > L7VSADM_CONFIG_SOCK_WAIT) {
			VANESSA_LOGGER_ERR_UNSAFE("connect to daemon: %s",
						strerror(errno));
			close(s);
			return ret;
		}
		// connect retrying.
		sleep(1);
	}

        return s;
}

static void
destroy_socket(void)
{
       	unlink(local_sockname);
}

static int
send_request(int s, void *req, size_t len)
{
        int ret;

	if(signal_flg > 0)
	{
		return -1;
	}

        ret = send(s, req, len, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("send to daemon: %s",
                                          strerror(errno));
        }

        return ret;
}

static int
sendv_request(int s, struct iovec *iov, int iovlen)
{
        int ret;

	if(signal_flg > 0)
	{
		return -1;
	}

        ret = writev(s, iov, iovlen);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("send to daemon: %s",
                                          strerror(errno));
        }

        return ret;
}

static int
recv_response(int s, struct iovec *iov, size_t iovlen)
{
        struct msghdr msg;
        struct ucred *cred;
        struct cmsghdr *cmsg;
        unsigned char cbuf[CMSG_LEN(sizeof(struct ucred))];
        int ret;

	if(signal_flg > 0)
	{
		return -1;
	}

        memset(&msg, 0, sizeof(msg));
        msg.msg_control = cbuf;
        msg.msg_controllen = sizeof(cbuf);
        msg.msg_iov = iov;
        msg.msg_iovlen = iovlen;
        cmsg = (struct cmsghdr *)cbuf;

        ret = recvmsg(s, &msg, 0);
        if (ret < 0) {
                VANESSA_LOGGER_ERR_UNSAFE("recvmsg: %s", strerror(errno));
                return L7VSADM_RECV_FAILED;
        }

        if (msg.msg_flags & MSG_CTRUNC) {
                VANESSA_LOGGER_ERR("Invalid response from l7vsd");
                return L7VSADM_RECV_FAILED;
        }

        if (msg.msg_flags & MSG_TRUNC) {
                return L7VSADM_RECV_TOO_LARGE;
        }

        if (! (cmsg->cmsg_level == SOL_SOCKET &&
               cmsg->cmsg_type == SCM_CREDENTIALS)) {
                VANESSA_LOGGER_ERR("Could not receive a remote credential");
                return L7VSADM_RECV_FAILED;
        }

        cred = (struct ucred *)CMSG_DATA(cmsg);
        if (cred->uid != 0) {
                VANESSA_LOGGER_ERR("Response from unprivileged user");
                return L7VSADM_RECV_FAILED;
        }

        return ret;
}

static const char *
config_strerror(int err)
{
        switch (err) {
        case 0:
                return "No error";
        case L7VS_CONFIG_ERR_INVALID_COMMAND:
                return "Invalid command";
        case L7VS_CONFIG_ERR_NOMEM:
                return "Could not allocate memory";
        case L7VS_CONFIG_ERR_VS_EXISTS:
                return "Virtual service already exists";
        case L7VS_CONFIG_ERR_RS_EXISTS:
                return "Real server already exists";
        case L7VS_CONFIG_ERR_NOVS:
                return "No such virtual service";
        case L7VS_CONFIG_ERR_NORS:
                return "No such real server";
        case L7VS_CONFIG_ERR_NOSCHED:
                return "No such scheduler";
        case L7VS_CONFIG_ERR_NOSOCK:
                return "Could not create a service socket";
	case L7VS_CONFIG_ERR_QOSSET:
		return "QoS threashold setting failure";
	case L7VS_CONFIG_ERR_SORRYSET:
		return "Sorry-Server modification failure";
        default:
                return "Unknown error";
        }
}

static int
list_vs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_config_req_list_vs req;
        struct l7vs_config_rsp_list_vs rsp;
        struct l7vs_service_arg *sarg;
        struct iovec iov[2];
        void *buf;
        char *srvname;
        int bufsize;
        int i, off;
        int ret;

        struct sockaddr_in vaddr_0;
	memset(&vaddr_0, 0, sizeof(struct sockaddr_in));	// for vaddr check

	struct sockaddr_storage sorry_addr_0;	// for sorry_addr check
	memset(&sorry_addr_0, 0, sizeof(struct sockaddr_storage));

	if(opt == NULL)
	{
		return -1;
	}

	if(signal_flg > 0)
	{
		return -1;
	}


        bufsize = L7VSADM_LIST_VS_BUFSIZE;

retry:
        buf = malloc(bufsize);
        if (buf == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                return -1;
        }

	if (bufsize == L7VSADM_LIST_VS_BUFSIZE) {
		printf("Layer-7 Virtual Server version %s\n", l7vs_version_string);
		if (opt->operation == L7VSADM_OP_VERBOSE) {
			printf("Prot LocalAddress:Port ProtoMod Scheduler Reschedule Protomod_opt_string\n");
			printf("     SorryAddress:Port Sorry_cc Sorry_flag\n");
		} else if (opt->operation == L7VSADM_OP_KEY) {
			printf("Prot LocalAddress:Port ProtoMod Scheduler Reschedule Protomod_key_string\n");
			printf("     SorryAddress:Port Sorry_cc Sorry_flag\n");
		} else {
			printf("Prot LocalAddress:Port ProtoMod Scheduler\n");
		}
		printf("  -> RemoteAddress:Port           Forward Weight ActiveConn InactConn\n");
	}

        req.cmd = L7VS_CONFIG_LIST_VS;
        ret = send_request(s, &req, sizeof(req));
        if (ret < 0) {
                free(buf);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        iov[1].iov_base = buf;
        iov[1].iov_len = bufsize;

        ret = recv_response(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                free(buf);
                if (ret == L7VSADM_RECV_TOO_LARGE) {
                        bufsize += L7VSADM_LIST_VS_BUFSIZE;
                        goto retry;
                }
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                free(buf);
                return -1;
        }

        /* now display it */
        sarg = (struct l7vs_service_arg *)buf;
        off = 0;
        for (i = 0; i < rsp.num; i++) {
                /* XXX should be replaced with pretty-print routine. */
		// Check vs address
		if (memcmp(&sarg->addr, &vaddr_0, sizeof(struct sockaddr_in)) == 0) {
			// sarg->addr not set
			srvname = NULL;
		} else {
			srvname = get_servicename(&sarg->addr, opt->isnumeric);
		}
                if (srvname == NULL) {
                        break;
                }

		/* If it ie verbose, display more details else simple details */
		if (opt->operation == L7VSADM_OP_VERBOSE) {
                	printf("TCP %s %s %s %d %s\n",
                       srvname, sarg->protomod, sarg->schedmod, sarg->reschedule, sarg->protomod_opt_string);
		} else if (opt->operation == L7VSADM_OP_KEY) {
                	printf("TCP %s %s %s %d %s\n",
                       srvname, sarg->protomod, sarg->schedmod, sarg->reschedule, sarg->protomod_key_string);
		} else {
                	printf("TCP %s %s %s\n",
                       srvname, sarg->protomod, sarg->schedmod);
		}

		// print sorry information when L7VSADM_OP_VERBOSE or L7VSADM_OP_KEY
		if (opt->operation == L7VSADM_OP_VERBOSE) {
			// Check sorry address
			if (memcmp(&sarg->sorry_addr, &sorry_addr_0, sizeof(struct sockaddr_storage)) == 0) {
				// sarg->sorry_addr not set
				// if srvname is NULL then string is (null)
				srvname = NULL;
			} else {
				srvname = get_servicename((struct sockaddr_in *)&sarg->sorry_addr, opt->isnumeric);
			}
			printf("    %s %d %d\n", srvname, sarg->sorry_cc, sarg->sorry_flag);
		}

		ret = list_rs(s, sarg, opt->isnumeric);
		if(ret < 0)
		{
        		free(buf);
			return -1;
		}
                off += sarg->len;
                if (off >= bufsize) {
                        VANESSA_LOGGER_INFO("Message size did not match");
                        VANESSA_LOGGER_INFO("version mismatch between"
                                            " l7vsd and l7vsadm?");
                        break;
                }
                sarg = (struct l7vs_service_arg *)((char *)buf + off);
        }
        free(buf);

        return 0;
}

static int
list_rs(int s, struct l7vs_service_arg *sarg, int isnumeric)
{
        struct l7vs_config_req_list_rs req;
		struct l7vs_config_rsp_list_vs rsp;
		struct l7vs_dest_arg *darg;
		struct iovec iov[2];
		char *srvname;
		int ret;
		int i, num;
		int weight,nactive,ninact;

		struct sockaddr_in raddr_0;
		memset(&raddr_0, 0, sizeof(struct sockaddr_in));	// for raddr check

		num = L7VSADM_LIST_RS_RSNUM;

		if(signal_flg > 0)
		{
			return -1;
		}

	retry:

		if(sarg == NULL)
		{
			return -1;
		}

		//darg = (struct l7vs_dest_arg *)malloc(num * sizeof(*darg));
		darg = (struct l7vs_dest_arg *)malloc(num * sizeof(struct l7vs_dest_arg));
		if (darg == NULL) {
			VANESSA_LOGGER_ERR("Could not allocate memory");
			return -1;
		}

		req.cmd = L7VS_CONFIG_LIST_RS;
		iov[0].iov_base = &req;
		iov[0].iov_len = sizeof(req);
		iov[1].iov_base = sarg;
		iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);
		ret = sendv_request(s, iov, sizeof(iov) / sizeof(iov[0]));
		if (ret < 0) {
			free(darg);
			return -1;
		}

		iov[0].iov_base = &rsp;
		iov[0].iov_len = sizeof(rsp);
		iov[1].iov_base = darg;
		iov[1].iov_len = num * sizeof(*darg);

		ret = recv_response(s, iov, sizeof(iov) / sizeof(iov[0]));
		if (ret < 0) {
			free(darg);
			if (ret == L7VSADM_RECV_TOO_LARGE) {
				num = rsp.num;
				goto retry;
			}
			return -1;
		}

		if (rsp.code != 0) {
			VANESSA_LOGGER_ERR(config_strerror(rsp.code));
			return -1;
		}
		/* now display it */
		num = rsp.num;
		for (i = 0; i < num; i++) {
			weight = darg[i].weight;
			nactive = darg[i].nactive;
			ninact =darg[i].ninact;
			// Check rs address
			if (memcmp(&darg[i].addr, &raddr_0, sizeof(struct sockaddr_in)) == 0) {
				// darg[i].addr not set
				srvname = NULL;
			} else {
				srvname = get_servicename(&darg[i].addr, isnumeric);
			}
			if (srvname == NULL) {
				break;
			}
			printf("  -> %-28s %-7s %-6d %-10d %-10d\n",
			       srvname,
			       "Masq",
			       weight,nactive,ninact);
		}

		free(darg);
		return 0;
	}

	static int
	operate_vs(int s, struct l7vsadm_options *opt)
	{
		struct l7vs_service_arg_multi sarg;
		struct l7vs_config_req_operate_vs req;
		struct l7vs_config_rsp_operate_vs rsp;
		struct iovec iov[2];
		void *protomod_service_arg;
		char *opstr = NULL;
		int ret;

		if(signal_flg > 0)
		{
			return -1;
		}

		// Argment check
		if (!opt) {
			return -1;
		}
		struct l7vs_protomod *pm = opt->protomod;

		assert(opt->operation == L7VSADM_OP_ADD_VS ||
		       opt->operation == L7VSADM_OP_DEL_VS ||
		       opt->operation == L7VSADM_OP_EDIT_VS);

		switch (opt->operation) {
		case L7VSADM_OP_ADD_VS:
			req.cmd = L7VS_CONFIG_ADD_VS;
			opstr = "add";
			break;
		case L7VSADM_OP_DEL_VS:
			req.cmd = L7VS_CONFIG_DEL_VS;
			opstr = "delete";
			break;
		case L7VSADM_OP_EDIT_VS:
			req.cmd = L7VS_CONFIG_EDIT_VS;
			opstr = "edit";
			break;
		default:
			return -1;
		}

		if (pm == NULL) {
			VANESSA_LOGGER_ERR_UNSAFE("Need -m option"
					   " to %s a virtual service", opstr);
			return -1;
		}
		// pm function pointer check
		if (!(pm->create_sa)) {
			VANESSA_LOGGER_ERR("create_sa() function pointer is NULL");
			return -1;
		}
		if (!(pm->destroy_sa)) {
			VANESSA_LOGGER_ERR("destroy_sa() function pointer is NULL");
			return -1;
		}

		if (!(pm->parse)) {
			VANESSA_LOGGER_ERR("parse() function pointer is NULL");
			return -1;
		}

		memset(&sarg, 0, sizeof(struct l7vs_service_arg_multi));
		sarg.srv_arg.addr = opt->vaddr;
		sarg.srv_arg.proto = IPPROTO_TCP;
		sarg.srv_arg.persist = opt->persist;
		sarg.srv_arg.backlog = opt->backlog;

        if (opt->schedmod == NULL) {
                opt->schedmod = strdup(L7VSADM_DEFAULT_SCHEDULER);
                if (opt->schedmod == NULL) {
                        VANESSA_LOGGER_ERR("Could not allocate memory");
			return -1;
                }
        }
        strcpy(sarg.srv_arg.schedmod, opt->schedmod);

	// set sorry data
	sarg.srv_arg.sorry_cc = opt->sorry_cc;
	sarg.srv_arg.sorry_addr = opt->sorry_addr;
	sarg.srv_arg.sorry_flag = opt->sorry_flag;
        
	// QoS Threashold data
	sarg.srv_arg.qos_s	= opt->qos_s;
	sarg.srv_arg.qos_c	= opt->qos_c;

	protomod_service_arg = pm->create_sa(&sarg.srv_arg);
	if (protomod_service_arg == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return -1;
	}
	memcpy(&sarg.protomod_arg, protomod_service_arg, sizeof(sarg.protomod_arg));
	pm->destroy_sa(&protomod_service_arg);

        ret = pm->parse(sarg.protomod_arg, vanessa_dynamic_array_get_count(opt->pm_args),
                        (char **)
                            vanessa_dynamic_array_get_vector(opt->pm_args));
        if (ret < 0) {
                VANESSA_LOGGER_ERR("protocol-module parameter error!");
                l7vs_module_unload(pm->handle);
                return -1;
        }

        iov[0].iov_base = &req;
        iov[0].iov_len = sizeof(req);
        iov[1].iov_base = &sarg;
        iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);

        ret = sendv_request(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        ret = recv_response(s, iov, 1);
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                l7vs_module_unload(pm->handle);
                return -1;
        }

        l7vs_module_unload(pm->handle);

        return 0;
}

static int
operate_rs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_protomod *pm;
        struct l7vs_service_arg_multi sarg;
        struct l7vs_config_req_operate_rs req;
        struct l7vs_config_rsp_operate_rs rsp;
        struct iovec iov[2];
        char *opstr = NULL;
        int ret;
	void *protomod_service_arg;

	if(signal_flg > 0)
	{
		return -1;
	}

	// Argment check
	if (!opt) {
		return -1;
	}

        assert(opt->operation == L7VSADM_OP_ADD_RS ||
               opt->operation == L7VSADM_OP_DEL_RS ||
               opt->operation == L7VSADM_OP_EDIT_RS);

        switch (opt->operation) {
        case L7VSADM_OP_ADD_RS:
                req.cmd = L7VS_CONFIG_ADD_RS;
                opstr = "add";
                break;
        case L7VSADM_OP_DEL_RS:
                req.cmd = L7VS_CONFIG_DEL_RS;
                opstr = "delete";
                break;
        case L7VSADM_OP_EDIT_RS:
                req.cmd = L7VS_CONFIG_EDIT_RS;
                opstr = "edit";
                break;
        default:
		return -1;
        }

        req.darg.addr = opt->raddr;
        req.darg.weight = opt->weight;

        if (opt->protomodname == NULL) {
                VANESSA_LOGGER_ERR_UNSAFE("Need -m option"
                                   " to %s a virtual service", opstr);
                return -1;
        }

        pm = (struct l7vs_protomod *)
                l7vs_module_load(opt->protomodname, "protomod");
        if (pm == NULL) {
                return -1;
        }
	// pm function pointer check
	if (!(pm->create_sa)) {
		VANESSA_LOGGER_ERR("create_sa() function pointer is NULL");
		return -1;
	}
	if (!(pm->destroy_sa)) {
		VANESSA_LOGGER_ERR("destroy_sa() function pointer is NULL");
		return -1;
	}
	if (!(pm->parse)) {
		VANESSA_LOGGER_ERR("parse() function pointer is NULL");
		return -1;
	}

	memset(&sarg, 0, sizeof(struct l7vs_service_arg_multi));
        sarg.srv_arg.addr = opt->vaddr;
        sarg.srv_arg.proto = IPPROTO_TCP;
        
        optind = 0;

	protomod_service_arg = pm->create_sa(&sarg.srv_arg);
	if (protomod_service_arg == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return -1;
	}
	memcpy(&sarg.protomod_arg, protomod_service_arg, sizeof(sarg.protomod_arg));
	pm->destroy_sa(&protomod_service_arg);

        ret = pm->parse(&sarg.protomod_arg, vanessa_dynamic_array_get_count(opt->pm_args),
                        (char **)
                            vanessa_dynamic_array_get_vector(opt->pm_args));
        if (ret < 0) {
                VANESSA_LOGGER_ERR("protocol-module parameter error!");
                l7vs_module_unload(pm->handle);
                return -1;
        }

        iov[0].iov_base = &req;
        iov[0].iov_len = sizeof(req);
        iov[1].iov_base = &sarg;
        iov[1].iov_len = sizeof(struct l7vs_service_arg_multi);

        ret = sendv_request(s, iov, sizeof(iov) / sizeof(iov[0]));
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                return -1;
        }

        iov[0].iov_base = &rsp;
        iov[0].iov_len = sizeof(rsp);
        ret = recv_response(s, iov, 1);
        if (ret < 0) {
                l7vs_module_unload(pm->handle);
                return -1;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                l7vs_module_unload(pm->handle);
                return -1;
        }

        l7vs_module_unload(pm->handle);

        return 0;
}

static int
flush_vs(int s, struct l7vsadm_options *opt)
{
        struct l7vs_config_req_flush_vs req;
        struct l7vs_config_rsp_flush_vs rsp;
        struct iovec iov;
        int ret;

	if(signal_flg > 0)
	{
		return -1;
	}

        req.cmd = L7VS_CONFIG_FLUSH_VS;
        ret = send_request(s, &req, sizeof(req));
        if (ret < 0) {
                return ret;
        }

        iov.iov_base = &rsp;
        iov.iov_len = sizeof(rsp);

        ret = recv_response(s, &iov, 1);
        if (ret < 0) {
                return ret;
        }

        if (rsp.code != 0) {
                VANESSA_LOGGER_ERR(config_strerror(rsp.code));
                return -1;
        }

        return 0;
}

/*!
 * Operate sync initialize command.
 * @param[in]	s	socket file discripter
 * @param[in]	*opt	command option pointter
 * @return	int	operate result OK=0, NG=-1
 */
static int
operate_sync_init(int s, struct l7vsadm_options *opt)
{
	struct l7vs_sync_init_data init_data;
	struct l7vs_config_req_sync_init req;
	struct l7vs_config_rsp_sync_init rsp;
	struct iovec iov;
	int ret;

	if(signal_flg > 0)
	{
		return -1;
	}

	// Argment check
	if (!opt) {
		return -1;
	}

	// set request command
	req.cmd = L7VS_CONFIG_SYNC_INIT;

	// set command request data
	init_data.ip_addr = opt->sync_ipaddr;
	init_data.servname = opt->sync_port;
	init_data.nic = opt->sync_nic;
	init_data.mode = (enum sync_operation)opt->sync_mode;
	req.data = init_data;

	//set iov data for request
	iov.iov_base = &req;
	iov.iov_len = sizeof(req);

	// send request
	ret = sendv_request(s, &iov, 1);
	if (ret < 0) {
		return -1;
	}

	//set iov data for response
	iov.iov_base = &rsp;
	iov.iov_len = sizeof(rsp);

	// recieve response
	ret = recv_response(s, &iov, 1);
	if (ret < 0) {
		return -1;
	}

	if (rsp.code != 0) {
		VANESSA_LOGGER_ERR(config_strerror(rsp.code));
		return -1;
	}

	return 0;
}

/*!
 * Operate sync finalize command.
 * @param[in]	s	socket file discripter
 * @param[in]	*opt	command option pointter
 * @return	int	operate result OK=0, NG=-1
 */
static int
operate_sync_fini(int s, struct l7vsadm_options *opt)
{
	struct l7vs_config_req_sync_fini req;
	struct l7vs_config_rsp_sync_fini rsp;
	struct iovec iov;
	int ret;

	if(signal_flg > 0)
	{
		return -1;
	}

	// Argment check
	if (!opt) {
		return -1;
	}

	// set request command
	req.cmd = L7VS_CONFIG_SYNC_FINI;

	//set iov data for request
	iov.iov_base = &req;
	iov.iov_len = sizeof(req);

	// send request
	ret = sendv_request(s, &iov, 1);

	//set iov data for response
	iov.iov_base = &rsp;
	iov.iov_len = sizeof(rsp);
	if (ret < 0) {
		return -1;
	}

	// recieve response
	ret = recv_response(s, &iov, 1);
	if (ret < 0) {
		return -1;
	}

	if (rsp.code != 0) {
		VANESSA_LOGGER_ERR(config_strerror(rsp.code));
		return -1;
	}

	return 0;
}

static void
sig_exit_handler(int sig)
{
	received_sig = sig;
	++signal_flg;
}

struct l7vsadm_sig {
        int sig;
        void (*func)(int sig);
};

static int
set_signal(void)
{
        static struct l7vsadm_sig sigs[] = {
                {SIGHUP,        sig_exit_handler},
                {SIGINT,        sig_exit_handler},
                {SIGQUIT,       sig_exit_handler},
                {SIGPIPE,       sig_exit_handler},
                {SIGTERM,       sig_exit_handler},
                {SIGUSR1,       sig_exit_handler},
                {SIGUSR2,       sig_exit_handler},
                {0,             NULL}
        };
        struct l7vsadm_sig *s;
	struct sigaction act, oact;
        int ret;

	memset(&act,0x00,sizeof(struct sigaction));
	memset(&oact,0x00,sizeof(struct sigaction));

        for (s = sigs; s->sig != 0; s++)
	{
                ret = sigaction(s->sig, NULL, &oact);
                if (ret < 0) {
                        VANESSA_LOGGER_ERR_UNSAFE("sigaction: %s",
                                                  strerror(errno));
                        return ret;
                }

                act = oact;
                act.sa_handler = s->func;
                ret = sigaction(s->sig, &act, NULL);
                if (ret < 0) {
                        VANESSA_LOGGER_ERR_UNSAFE("sigaction: %s",
                                                  strerror(errno));
                        return ret;
                }
        }

        return 0;
}

int
main(int argc, char *argv[])
{
        /* XXX test version */
        struct l7vsadm_options options;
        vanessa_logger_t *vl;
        int s;
        int ret;
        struct l7vsadm_operations *op;

        static struct l7vsadm_operations oplist[] = {
                {L7VSADM_OP_LIST,       list_vs},
                {L7VSADM_OP_ADD_VS,     operate_vs},
                {L7VSADM_OP_DEL_VS,     operate_vs},
                {L7VSADM_OP_EDIT_VS,    operate_vs},
                {L7VSADM_OP_ADD_RS,     operate_rs},
                {L7VSADM_OP_DEL_RS,     operate_rs},
                {L7VSADM_OP_EDIT_RS,    operate_rs},
                {L7VSADM_OP_FLUSH_VS,   flush_vs},
		{L7VSADM_OP_SYNC_INIT,	operate_sync_init},
		{L7VSADM_OP_SYNC_FINI,	operate_sync_fini},
                {L7VSADM_OP_VERBOSE,	list_vs},
                {L7VSADM_OP_KEY,	list_vs},
                {-1, NULL}
        };

        vl = vanessa_logger_openlog_filehandle(stderr, "l7vsadm",
                                               LOG_DEBUG, 0);
        vanessa_logger_set(vl);
        if (set_signal() < 0) {
                exit(1);
        }

        memset(&options, 0, sizeof(options));
        options.backlog = L7VSADM_DEFAULT_BACKLOG;
        options.pm_args = vanessa_dynamic_array_create(0,
                                                       VANESSA_DESS,
                                                       VANESSA_DUPS,
                                                       VANESSA_DISS,
                                                       VANESSA_LENS);
        if (options.pm_args == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                exit(1);
        }

        /* Add argv[0] first because getopt_long() skip argv[0]... */
        if (vanessa_dynamic_array_add_element(options.pm_args, argv[0]) == NULL) {
                VANESSA_LOGGER_ERR("Could not allocate memory");
                exit(1);
        }

        l7vs_module_init(NULL);
        ret = parse_args(&options, argc, argv);
	if (ret < 0) {
		usage(stderr);
	}

	if(signal_flg > 0)
	{
		goto SIG_END;
	}

        s = create_socket();
        if (s < 0) {
                exit(1);
        }

	if(signal_flg > 0)
	{
		goto SIG_END;
	}

        for (op = oplist; op->code != -1; op++) {
                if (op->code == options.operation) {
                        break;
                }
        }

        if (op->code == -1) {
                destroy_socket();
                usage(stderr);
                exit(1);
        }

	if(signal_flg > 0)
	{
		goto SIG_END;
	}

        ret = op->func(s, &options);

SIG_END:
        destroy_socket();

	/* free for strdup() */
	if (options.schedmod) {
		free(options.schedmod);
	}
	if (options.protomodname) {
		free(options.protomodname);
	}
	if (options.sync_ipaddr) {
		free(options.sync_ipaddr);
	}
	if (options.sync_port) {
		free(options.sync_port);
	}
	if (options.sync_nic) {
		free(options.sync_nic);
	}

        l7vs_module_fini();

	/* free for vannesa_dynamic_array_create() */
	vanessa_dynamic_array_destroy(options.pm_args);

        if (ret < 0) {
                exit(1);
        }

        return 0;
}
