#include <stdlib.h>
#include <time.h>
#include <getopt.h>
#include "vanessa_logger.h"
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_dest.h"
#include "l7vs_module.h"
#include "module_http.h"
#include <fnmatch.h>


#define SERVICE_ARG_MAXSIZE    (512)
#define PATTERN_MATCH_MAXSIZE  (128)
#define URL_SERVICE_NUMBER     (16)

struct l7vs_url_service {
	uint32_t service_handle;
	char pattern_match[PATTERN_MATCH_MAXSIZE];
	int reschedule;
};

struct  l7vs_url_service_arg {
	char pattern_match[PATTERN_MATCH_MAXSIZE];
	int reschedule;
};

static void  fini(void);
static int   create(void *url_arg, uint32_t service_handle);
static void* create_sa(struct l7vs_service_arg *srv_arg);
static int   compare(uint32_t srv_handle1, uint32_t srv_handle2);
static int   match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
                        char *buf, size_t *len, struct l7vs_dest **dest, int *tcps);
static int   analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
                        char *response, size_t *response_length);
static int   destroy(uint32_t srv_handle);
static void  destroy_sa(void **url_arg);
static int   service_arg(struct l7vs_service_arg_multi *srv_arg_mt, uint32_t srv_handle);
static int   parse(void *url_arg, int argc, char *argv[]);

static struct l7vs_url_service *l7vs_protomod_url_search_service(uint32_t service_handle);
static struct l7vs_url_service *l7vs_protomod_url_create_service();
static struct l7vs_url_service *l7vs_protomod_url_create_temp_service();

struct l7vs_url_service *url_service_list[URL_SERVICE_NUMBER];

static struct l7vs_protomod url_protomod = {
	NULL,           /* handle */
	"url",          /* modname */
	0,              /* refcnt */
	create,         /* create function */
	compare,        /* compare function */
	match_cldata,   /* match_cldata function */
	analyze_rsdata, /* analyze_rsdata function */
	destroy,        /* destroy function */
	fini,           /* fini function */
	create_sa,      /* create_sa function */
	service_arg,    /* service_arg function */
	parse,          /* parse function */
	destroy_sa,     /* destroy_sa function */
};

/*!
 * Protocol module initialize function. This function run when dlopen and dlsym at first time.
 * @param  handle dlopen's handle
 * @return l7vs_protomod struct
 */
struct l7vs_protomod *
init(void *handle)
{
	/* initialize url service list */
	memset(url_service_list, 0, sizeof(struct l7vs_url_service *) * URL_SERVICE_NUMBER);
	/* set dlopen's handle */

	url_protomod.handle = handle;
	return &url_protomod;
}

/*!
 * Protocol module finalize function. free all url service list just in case.
 * @param   void
 * @return  void
 */
static void
fini(void)
{
	/* url service list counter */
	int service_number = 0;

	/* check all url service list */
	for (service_number = 0; service_number < URL_SERVICE_NUMBER; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (url_service_list[service_number] != NULL) {
			/* free and points NULL */
			free(url_service_list[service_number]);
			url_service_list[service_number] = NULL;
		}
	}
}

/*!
 * Create url service struct.
 * @param  url_arg    url service argument struct
 * @param  service_handle a unique service ID
 * @retval 0  successfully create url service.
 * @retval -1 some errors occur.
 */
static int
create(void *url_arg, uint32_t service_handle)
{
	struct l7vs_url_service *url_service;
	struct l7vs_url_service_arg *url_service_arg;

	/* check null */
	if (url_arg == NULL) {
		VANESSA_LOGGER_ERR("url_arg is null");
		return -1;
	}

	if (service_handle != TEMP_SERVICEHANDLE) {
		/* search empty url service list and create url service */
		url_service = l7vs_protomod_url_create_service();
	} else {
		/* create temporary url service */
		url_service = l7vs_protomod_url_create_temp_service();
	}
	if (url_service == NULL) {
		VANESSA_LOGGER_ERR("Could not make temporary url service");
		return -1;
	}

	url_service_arg = (struct l7vs_url_service_arg *) url_arg;

	/* set service handle, pattern match and reschedule flag */
	url_service->service_handle = service_handle;
	strncpy(url_service->pattern_match, url_service_arg->pattern_match, PATTERN_MATCH_MAXSIZE);
	url_service->reschedule = url_service_arg->reschedule;

	return 0;
}

/*!
 * Create url service argument struct.
 * @param  srv_arg service argument struct
 * @return void*   url service argument struct
 */
static void *
create_sa(struct l7vs_service_arg *srv_arg)
{
	struct l7vs_url_service_arg *url_service_arg;

	if (srv_arg == NULL) {
		return NULL;
	}

	/* create url service argument struct */
	url_service_arg = (struct l7vs_url_service_arg *) calloc(1, sizeof(struct l7vs_url_service_arg));
	if (url_service_arg == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return (void *) url_service_arg;
	}

	/* set url service argument size and protomod name "url" */
	srv_arg->len = sizeof(struct l7vs_url_service_arg);
	strcpy(srv_arg->protomod, url_protomod.modname);

	return (void *) url_service_arg;
}

/*!
 * Compare two service.
 * @param  srv_handle1 one of a unique service ID
 * @param  srv_handle2 one of a unique service ID
 * @retval 0  they matched perfectly.
 * @retval -1 they are different.
 */
static int
compare(uint32_t srv_handle1, uint32_t srv_handle2)
{
	struct l7vs_url_service *url_srv1, *url_srv2;

	/* search service that has such a service ID(1) */
	url_srv1 = l7vs_protomod_url_search_service(srv_handle1);
	if (url_srv1 == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's url service");
		return -1;
	}

	/* search service that has such a service ID(2) */
	url_srv2 = l7vs_protomod_url_search_service(srv_handle2);
	if (url_srv2 == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's url service");
		return -1;
	}

	/* compare two pattern match */
	if (strcmp(url_srv1->pattern_match, url_srv2->pattern_match) != 0) {
		return -1;
	}

	return 0;
}

/*!
 * Check the client packet and determine a real server.
 * @param  srv service struct include service handle, protocol module and schedule module.
 * @param  conn connection data.
 * @param  request packet data from client
 * @param  len length of packet data
 * @param  dest destination (real server) list
 * @param  tcps TCP Splicer flag
 * @retval 0  successfully check packet data
 * @retval -1 some errors occur.
 */
static int
match_cldata(struct l7vs_service *srv, struct l7vs_conn *conn,
      char *buf, size_t *len, struct l7vs_dest **dest, int *tcps)
{
	struct l7vs_url_service *url_service;
	int ret;
	size_t klen;
	char *uri,*host,pattern[PATTERN_MATCH_MAXSIZE+2];

	/* check null */
	if (srv == NULL) {
		VANESSA_LOGGER_ERR("srv is null");
		return -1;
	}
	if (conn == NULL) {
		VANESSA_LOGGER_ERR("conn is null");
		return -1;
	}
	if (buf == NULL) {
		VANESSA_LOGGER_ERR("buf is null");
		return -1;
	}
	if (len == NULL) {
		VANESSA_LOGGER_ERR("len is null");
		return -1;
	}
	if (dest == NULL) {
		VANESSA_LOGGER_ERR("dest is null");
		return -1;
	}
	if (tcps == NULL) {
		VANESSA_LOGGER_ERR("tcps is null");
		return -1;
	}
	/* check tcps flag */
	if (*tcps != 0 && *tcps != 1) {
		VANESSA_LOGGER_ERR("tcps is invalid");
		return -1;
	}

	/* search service that has such a service ID */
	url_service = l7vs_protomod_url_search_service(srv->handle);
	if (url_service == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's url service");
		return -1;
	}

	/* initialize protocol module ... clear destination list */
	ret = srv->pm->initialize(srv, conn, buf, *len, dest);
	if (ret != 0) {
		VANESSA_LOGGER_ERR("Could not initialize protomod");
		return -1;
	}

	/* check pattern_match != 0 */
	if (url_service->pattern_match[0] == '\0') {
		VANESSA_LOGGER_ERR("pattern match is NULL");
		return -1;
	}

	/* check the size of request data */
	klen = strnlen(url_service->pattern_match,PATTERN_MATCH_MAXSIZE);
	if (*len < (15 + klen) ) {
		VANESSA_LOGGER_ERR("request data is too short");
		return -1;
	}

	/* check the HTTP method in HTTP request header */
	uri = http_check_request_method(buf, len);
	if (uri == NULL) {
		VANESSA_LOGGER_ERR("inavalid method");
		return -1;
	}
	
	/* check keyword (URI)*/
	sprintf(pattern, "*%s*", url_service->pattern_match);
	ret = fnmatch(pattern, uri, 0);
	if (ret != 0) {

		/* search Host field in HTTP request header */
		host = http_search_header_field(buf,"Host");
		if (host == NULL) {
			VANESSA_LOGGER_ERR("Could not find a Host field");
			return 1;
		}else{	
			/* check keyword (HOST) */
			ret = fnmatch(pattern, host, 0);
			if ( ret != 0 ) {
				VANESSA_LOGGER_ERR("Could not find a pattern match")
				return 1;
			}
		}
	}

	/* finalize */
	ret = srv->pm->finalize(srv, conn, buf, *len, dest, url_service->reschedule);
	if (ret != 0){
		VANESSA_LOGGER_ERR("Could not finalize protomod");
		return -1;
	}

	return 0;
}

/*!
 * Check the real server packet and insert a Set-Cookie field.
 * @param  srv service struct include service handle, protocol module and schedule module.
 * @param  conn connection data.
 * @param  response packet data from real server
 * @param  len length of packet data. it will be lengthened.
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
analyze_rsdata(struct l7vs_service *srv, struct l7vs_conn *conn,
	char *response, size_t *response_length)
{
	return 0;
}

/*!
 * Destroy url service
 * @param  srv_handle a unique service ID
 * @retval 0  successfully check packet data.
 * @retval -1 some errors occur.
 */
static int
destroy(uint32_t srv_handle)
{
	/* url service list counter */
	int service_number = 0;
	int free_flag = 0;

	/* check all url service list */
	for (service_number = 0; service_number < URL_SERVICE_NUMBER; ++service_number) {
		/* found url service that has srv_handle */
		if (url_service_list[service_number] != NULL && 
		    url_service_list[service_number]->service_handle == srv_handle) {

			/* free and NULL */
			free(url_service_list[service_number]);
			url_service_list[service_number] = NULL;

			free_flag = 1;
			break;
		}
	}
	
	/* url service was not found */
	if (free_flag == 0) {
		VANESSA_LOGGER_ERR("Could not find such service handle's url service");
		return -1;
	}

	return 0;
}

/*!
 * Destroy url service argument
 * @param  url_arg url service argument
 * @return void
 */
static void
destroy_sa(void **url_arg)
{
	/* check null */
	if (url_arg != NULL) {
		/* free and NULL */
		free((struct l7vs_url_service_arg *) *url_arg);
		*url_arg = NULL;
	}
}

/*!
 * Create strings for service list of l7vsadm
 * @param  srv_arg service argument struct
 * @param  srv_handle a unique service ID
 * @retval 0  successfully create strings
 * @retval -1 some errors occur.
 */
static int
service_arg(struct l7vs_service_arg_multi *srv_arg_mt, uint32_t srv_handle)
{
	struct l7vs_url_service *url_service;
	struct l7vs_url_service_arg c_sarg;
	char url_argument[SERVICE_ARG_MAXSIZE];

	/* check null */
	if (srv_arg_mt == NULL) {
		VANESSA_LOGGER_ERR("srv_arg_mt is null");
		return -1;
	}

	/* search service that has such a service ID */
	url_service = l7vs_protomod_url_search_service(srv_handle);
	if (url_service == NULL) {
		VANESSA_LOGGER_ERR("Could not find such service handle's url service");
		return -1;
	}

	/* initialize argument strings */
	memset(url_argument, 0, SERVICE_ARG_MAXSIZE);

	/* set url args to service argument struct */
	srv_arg_mt->srv_arg.reschedule = url_service->reschedule;

	/* create long argument (l7vsadm option -L/-l) */
	sprintf(url_argument, "--pattern-match %s", url_service->pattern_match);
	strcpy(srv_arg_mt->srv_arg.protomod_key_string, url_argument);

	/* create verbose argument (l7vsadm option -V/-v) */
	strcpy(srv_arg_mt->srv_arg.protomod_opt_string, url_argument);

	strncpy(c_sarg.pattern_match, url_service->pattern_match, PATTERN_MATCH_MAXSIZE);
	c_sarg.reschedule = url_service->reschedule;

	memcpy(srv_arg_mt->protomod_arg, &c_sarg, sizeof(struct l7vs_url_service_arg));

	return 0;
}

/*!
 * Parse l7vsadm options to url argument
 * @param  url_arg url service argument struct
 * @param  argc number of l7vsadm argument
 * @param  argv l7vsadm argument list
 * @retval 0  successfully parse argument
 * @retval -1 some errors occur.
 */
static int
parse(void *url_arg, int argc, char *argv[])
{
	struct l7vs_url_service_arg *url_service_arg;
	static struct option opt[] = {
		{"pattern-match",   required_argument, NULL, 'P'},
		{"reschedule",      no_argument,       NULL, 'F'},
		{"no-reschedule",   no_argument,       NULL, 'N'},
		{NULL,              0,                 NULL, 0  }
	};
	int c;
	int pattern_match_flag = 0;
	int reschedule_flag = 0;

	/* check null */
	if (url_arg == NULL) {
		VANESSA_LOGGER_ERR("url_arg is null");
		return -1;
	}

	url_service_arg = (struct l7vs_url_service_arg *) url_arg;
	optind = 0;

	/* check all argument */
	while ((c = getopt_long(argc, argv, "P:FN", opt, NULL)) != -1) {
		switch (c) {
		/* --pattern-match / -P */
		case 'P':
			/* check maximum length */
			if (strnlen(optarg,PATTERN_MATCH_MAXSIZE) >= PATTERN_MATCH_MAXSIZE) {
				VANESSA_LOGGER_ERR_UNSAFE("%s: path too long", optarg);
				return -1;
			}
			/* check minimum length */
			if (strnlen(optarg,PATTERN_MATCH_MAXSIZE) <= 0 ) {
				VANESSA_LOGGER_ERR_UNSAFE("%s: no pattern match string", optarg);
				return -1;
			}
			strncpy(url_service_arg->pattern_match, optarg, PATTERN_MATCH_MAXSIZE);
			pattern_match_flag = 1;
			break;

		/* --reschedule / -F */
		case 'F':
			/* reschedule on */
			url_service_arg->reschedule = 1;
			reschedule_flag++;
			break;

		/* --no-reschedule / -N */
		case 'N':
			/* reschedule off */
			url_service_arg->reschedule = 0;
			reschedule_flag++;
			break;

		/* else error */
		default:
			return -1;
		}
	}

	/* when set both -F and -N at the same time */
	if (reschedule_flag == 2) {
		VANESSA_LOGGER_ERR("You should choose either of reschdule or no-reschedule");
		return -1;
	}

	/* set default no reschedule */
	if (reschedule_flag == 0) {
		url_service_arg->reschedule = 0;
	}

	/* */
	if (pattern_match_flag == 0) {
		VANESSA_LOGGER_ERR("error");
		return -1;
	}

	return 0;
}

/*!
 * Search url service from url service list using service handle
 * @param  service_handle a unique service ID
 * @return url service struct when service was found. NULL when service was not found.
 */
static struct l7vs_url_service *
l7vs_protomod_url_search_service(uint32_t service_handle)
{
	/* url service list counter */
	int service_number = 0;

	/* check all url service list */
	for (service_number = 0; service_number < URL_SERVICE_NUMBER; ++service_number) {
		/* found the service has same service handle */
		if (url_service_list[service_number] != NULL && 
		    url_service_list[service_number]->service_handle == service_handle) {
			return url_service_list[service_number];
		}
	}
	
	/* not found */
	return NULL;
}

/*!
 * Create url service.
 * @param  void
 * @return url service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_url_service *
l7vs_protomod_url_create_service()
{
	/* url service list counter */
	int service_number = 0;

	/* check all url service list */
	for (service_number = 0; service_number < URL_SERVICE_NUMBER - 1; ++service_number) {
		/* if pointer that does not point NULL exists ... */
		if (url_service_list[service_number] == NULL) {
			/* create a service at empty pointer */
			url_service_list[service_number] = (struct l7vs_url_service *) calloc(1, sizeof(struct l7vs_url_service));
			if (url_service_list[service_number] == NULL) {
				VANESSA_LOGGER_ERR("Could not allocate memory");
				break;
			}
			return url_service_list[service_number];
		}
	}
	
	/* all service list is full */
	return NULL;
}

/*!
 * Create temporary url service.
 * @param  void
 * @return url service struct when create a service. NULL when cannot create service.
 */
static struct l7vs_url_service *
l7vs_protomod_url_create_temp_service()
{
	/* if pointer that does not point NULL exists ... */
	if (url_service_list[URL_SERVICE_NUMBER - 1] != NULL) {
		VANESSA_LOGGER_ERR("temporary url service is being used by other process");
		return NULL;
	}
	url_service_list[URL_SERVICE_NUMBER - 1] = (struct l7vs_url_service *) calloc(1, sizeof(struct l7vs_url_service));
	if (url_service_list[URL_SERVICE_NUMBER - 1] == NULL) {
		VANESSA_LOGGER_ERR("Could not allocate memory");
		return NULL;
	}

	return url_service_list[URL_SERVICE_NUMBER - 1];
}
