/*
 * @file        sched_rr.c
 * @brief       Round-Robin Scheduling Module for UltraMonkey-L7
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs_dest.h"
#include "l7vs_service.h"
#include "l7vs_conn.h"
#include "l7vs_sched.h"

struct servicedest_contenor{
	handle_t	handle;
	struct in_addr	addr;
	u_short		port;
};

GList*	servicedest_list;

static void fini(void);
static struct l7vs_dest *l7vs_sched_rr_schedule(struct l7vs_service *srv, 
                                                struct l7vs_conn *conn);

static struct l7vs_scheduler sched_rr_scheduler = {
        NULL,
        "rr",
        0,
        l7vs_sched_rr_schedule,         /* schedule function */
        NULL,                           /* bind function */
        NULL,                           /* unbind function */
        fini                            /* fini function */
};

struct l7vs_scheduler *
init(void *handle)
{
	servicedest_list = NULL;
        sched_rr_scheduler.handle = handle;
        return &sched_rr_scheduler;
}

static void
fini(void)
{
	GList*	ptr = NULL;
	for( ptr = g_list_first( servicedest_list ); ptr ; ptr = g_list_next(ptr) ){
		free( ptr->data );
		ptr->data = NULL;
	}
	g_list_free(servicedest_list);
	servicedest_list = NULL;
}

static struct l7vs_dest *
l7vs_sched_rr_schedule(struct l7vs_service *srv, struct l7vs_conn *conn)
{
	struct l7vs_dest* ret = NULL;
	GList* active_dest_list = NULL;
	GList* ptr = NULL;
	struct servicedest_contenor* contenor = NULL;

	if( !srv ) return NULL;

	//create active list
	for( ptr = g_list_first( srv->dest_list ); ptr ; ptr = g_list_next(ptr) ){
		if( ((struct l7vs_dest*) ( ptr->data ))->weight > 0 ) active_dest_list = g_list_append( active_dest_list, ptr->data );
	}

	//active dest don't have 
	if( !active_dest_list ) return NULL;

	//first use on service
	if( !srv->sched_data ){
		ptr = g_list_first( active_dest_list );
		ret = (struct l7vs_dest*) ptr->data;
		contenor = malloc( sizeof(struct servicedest_contenor) );
		servicedest_list = g_list_append( servicedest_list, contenor );
		srv->sched_data = contenor;
		goto L7VS_SCHED_RR_SCHEDULE_END;
	}
	contenor = (struct servicedest_contenor*) srv->sched_data;
	for( ptr = g_list_first( active_dest_list ); ptr ; ptr = g_list_next( ptr ) ){
		struct l7vs_dest* dest = ptr->data;
		if( memcmp( &(dest->addr.sin_addr), &(contenor->addr), sizeof( struct in_addr ) )
		    &&
		    dest->addr.sin_port == contenor->port ){
		    ret = ptr->data;
		    break;
		}
	}
	if( ! ret ){//not have dest list
		ptr = g_list_first( active_dest_list );
		ret = ( struct l7vs_dest* ) ptr->data;
		goto L7VS_SCHED_RR_SCHEDULE_END;
	}
	ptr = g_list_next( active_dest_list );
	if( !ptr ){
		ptr = g_list_first( active_dest_list );
		ret = ( struct l7vs_dest* ) ptr->data;
		goto L7VS_SCHED_RR_SCHEDULE_END;
	}
	ret = ( struct l7vs_dest* ) ptr->data; 

L7VS_SCHED_RR_SCHEDULE_END:
	contenor->handle = srv->handle;
	contenor->addr = ret->addr.sin_addr;
	contenor->port = ret->addr.sin_port;
	g_list_free( active_dest_list );
	return ret;
}
