 /*
 * @file	sched_wrr.c
 * @brief	Weighted Round-Robin Scheduling Module for UltraMonkey-L7
 * @auther	nakai norihisa.
 * August 2007
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 **********************************************************************/

#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs.h"
#include "l7vs_conn.h"
#include "l7vs_service.h"
#include "l7vs_sched.h"
#include "l7vs_dest.h"

#define MAX_VS 128
#define VS_INITIALIZED 1
#define VS_NOTINUSE 0

struct wrr_weights {
  int currentWeight;	/* Current Weight */
  int maxWeight;	/* Max of Weight */
  int gcd;		/* GCD of dest_list */
};

struct servicedest_contenor{
  handle_t		handle;
  struct in_addr 	addr;
  u_short		port;
};

GList*	servicedest_list;

static int vs_initialized_flags[MAX_VS];
static struct wrr_weights *vs_weights[MAX_VS];

static void fini(void);
struct l7vs_dest* l7vs_sched_wrr_schedule(struct l7vs_service*, struct l7vs_conn*);
static int l7vs_sched_wrr_service_init(struct l7vs_service*);
static int l7vs_sched_wrr_recalc(struct l7vs_service*);
static int l7vs_sched_wrr_getMaxWeight(struct l7vs_service*);
static int l7vs_sched_wrr_gcd(int, int);
static int l7vs_sched_wrr_getGCD(struct l7vs_service*);
static GList* l7vs_sched_wrr_search_suitableRS(GList*, int);


static struct l7vs_scheduler sched_wrr_scheduler = {
  NULL,
  "wrr",
  0,
  l7vs_sched_wrr_schedule,
  NULL,
  NULL,
  fini
};


struct l7vs_scheduler* init(void *handle)
{
  servicedest_list = NULL;
  memset(vs_initialized_flags, VS_NOTINUSE, sizeof (vs_initialized_flags));
  memset(vs_weights, 0, sizeof (vs_weights));
  sched_wrr_scheduler.handle = handle;
  return &sched_wrr_scheduler;
}


void fini(void) 
{
  int i;
  GList* ptr = NULL;

  for (i = 0; i < MAX_VS; ++i) {

    if (vs_initialized_flags[i]) {
	if(vs_weights[i] != NULL)
	{
      		free(vs_weights[i]);
      		vs_weights[i] = NULL;
	}
      	vs_initialized_flags[i] = VS_NOTINUSE;
    }
  }

  for( ptr = g_list_first(servicedest_list); ptr; ptr = g_list_next(ptr) ){
	free( ptr->data );
	ptr->data = NULL;
  }
  g_list_free( servicedest_list );
  servicedest_list = NULL;
}


struct l7vs_dest* l7vs_sched_wrr_schedule(struct l7vs_service *srv, struct l7vs_conn *conn)
{
  struct l7vs_dest *dest;
  GList* active_dest_list = NULL;
  GList* ptr;
  GList *now;

	if(srv == NULL || srv->handle >= MAX_VS || srv->dest_list == NULL || srv->dest_list->data == NULL)
	{
		dest = NULL;
		goto OUT;
	}

	//create active dest
	for( ptr = g_list_first( srv->dest_list ); ptr; ptr = g_list_next(ptr) ){
		active_dest_list = g_list_append( active_dest_list, ptr->data );
	}
	if( !active_dest_list ) goto OUT;

	if (vs_initialized_flags[srv->handle] == VS_NOTINUSE) {

		if (l7vs_sched_wrr_service_init(srv) == -1) {
			dest = NULL;
			goto OUT;
		}

		vs_weights[srv->handle]->currentWeight = vs_weights[srv->handle]->maxWeight;
		struct servicedest_contenor* contenor = malloc( sizeof( struct servicedest_contenor ) );
		struct l7vs_dest* tdest = ( struct l7vs_dest* ) g_list_first( active_dest_list );
		contenor->handle = srv->handle;
		contenor->addr = tdest->addr.sin_addr;//C90 stuct copy ok!
		contenor->port = tdest->addr.sin_port;
		servicedest_list = g_list_append( servicedest_list, contenor );
		srv->sched_data = contenor;
	} else {
		if (l7vs_sched_wrr_recalc(srv) != 0) {
			dest = NULL;
			goto OUT;
		}
		if (srv->sched_data == NULL) {
			struct servicedest_contenor* contenor = malloc( sizeof( struct servicedest_contenor ) );
			struct l7vs_dest* tdest = ( struct l7vs_dest* ) g_list_first( active_dest_list );
			contenor->handle = srv->handle;
			contenor->addr = tdest->addr.sin_addr;
			contenor->port = tdest->addr.sin_port;
			servicedest_list = g_list_append( servicedest_list, contenor );
			srv->sched_data = contenor;
		}
	}

	for( ptr = g_list_first( active_dest_list ); ptr; ptr = g_list_next(ptr) ){
		struct l7vs_dest* tdest = ptr->data;
		struct servicedest_contenor* contenor = (struct servicedest_contenor*) srv->sched_data;
		if( memcmp( &(tdest->addr.sin_addr), &(contenor->addr), sizeof( struct in_addr ) )
		    &&
		    tdest->addr.sin_port == contenor->port ){
			now = (GList*) ptr;
			break;
		}
	}
//	now = (GList*)srv->sched_data;

	while (1)
	{
		if (((struct l7vs_dest*)now->data)->weight >= vs_weights[srv->handle]->currentWeight)
		{
			dest = (struct l7vs_dest*)now->data;
			
			ptr = (void*)l7vs_sched_wrr_search_suitableRS(now, vs_weights[srv->handle]->currentWeight);
			if (ptr == NULL)
			{
				ptr = g_list_first(active_dest_list);
				vs_weights[srv->handle]->currentWeight -= vs_weights[srv->handle]->gcd;

				if (vs_weights[srv->handle]->currentWeight <= 0)
				{
  					vs_weights[srv->handle]->currentWeight = vs_weights[srv->handle]->maxWeight;
				}
			}
			struct servicedest_contenor* contenor = ( struct servicedest_contenor* )srv->sched_data;
			struct l7vs_dest* tdest = ptr->data;
			contenor->handle = srv->handle;
			contenor->addr = tdest->addr.sin_addr;
			contenor->port = tdest->addr.sin_port;
			goto OUT;
		}
		else
		{
			now = g_list_next(now);
			if (now == NULL) {
				now = g_list_first(srv->dest_list);
			}
		}
	}

 OUT:
  g_list_free( active_dest_list );
  return  dest;

}


static int l7vs_sched_wrr_service_init(struct l7vs_service *srv)
{


  if(srv == NULL || srv->handle >= MAX_VS)
  {
	return -1;
  }

  vs_weights[srv->handle] = (struct wrr_weights*)malloc(sizeof(struct wrr_weights));

  if (vs_weights[srv->handle] == NULL) 
  {
    return -1;
  }

  memset(vs_weights[srv->handle], 0, sizeof(struct wrr_weights));
  vs_weights[srv->handle]->gcd = l7vs_sched_wrr_getGCD(srv);
  if(vs_weights[srv->handle]->gcd < 0)
  {
	return -1;
  }
  vs_weights[srv->handle]->maxWeight = l7vs_sched_wrr_getMaxWeight(srv);
  if(vs_weights[srv->handle]->maxWeight < 0)
  {
	return -1;
  }
  vs_initialized_flags[srv->handle] = VS_INITIALIZED;

  return 0;

}

static int l7vs_sched_wrr_recalc(struct l7vs_service *srv)
{

  if(srv == NULL || srv->handle >= MAX_VS)
  {
	return -1;
  }

  if (vs_weights[srv->handle] == NULL) 
  {
    return -1;
  }

  vs_weights[srv->handle]->gcd = l7vs_sched_wrr_getGCD(srv);
  if(vs_weights[srv->handle]->gcd < 0)
  {
	return -1;
  }
  vs_weights[srv->handle]->maxWeight = l7vs_sched_wrr_getMaxWeight(srv);
  if(vs_weights[srv->handle]->maxWeight < 0)
  {
	return -1;
  }
  if (vs_weights[srv->handle]->currentWeight > vs_weights[srv->handle]->maxWeight)
  {
    vs_weights[srv->handle]->currentWeight = vs_weights[srv->handle]->maxWeight;
  }

  return 0;

}


static int l7vs_sched_wrr_getMaxWeight(struct l7vs_service *srv)
{
  GList *l, *now;
  struct l7vs_dest *dest;
  int weight = 0;

  if(srv == NULL || srv->dest_list == NULL)
  {
	return -1;
  }
  l = srv->dest_list;

  for (now = g_list_first(l); now != NULL; now = g_list_next(now)) {
    dest = (struct l7vs_dest*)now->data;

    if (dest->weight > weight) {
      weight = dest->weight;
    }
  }

  return weight;
}


static int l7vs_sched_wrr_gcd(int a, int b)
{

  if (a == b) {
    return a;

  }
  else if (a > b) {
    return l7vs_sched_wrr_gcd (a - b, b);

  }
  else {
    return l7vs_sched_wrr_gcd (b, a);
    
  }
}


static int l7vs_sched_wrr_getGCD(struct l7vs_service *srv)
{
  GList *now;
  int currentGCD = 1;

  if(srv == NULL || srv->dest_list == NULL || srv->dest_list->data == NULL)
  {
	return -1;
  }

  now = g_list_first(srv->dest_list);
  currentGCD = ((struct l7vs_dest*)now->data)->weight;
  now = g_list_next(now);
  
  while (now != NULL) {
    currentGCD = l7vs_sched_wrr_gcd(currentGCD, ((struct l7vs_dest*)now->data)->weight);
    now = g_list_next(now);
  }

  return currentGCD;
  
}

static GList* l7vs_sched_wrr_search_suitableRS(GList *list, int weight)
{
  GList *now, *candidate;

  now = g_list_next(list);
  candidate = NULL;

  while (now != NULL) {
    if (((struct l7vs_dest*)now->data)->weight >= weight) {
      candidate = now;
      goto OUT;

    }

    now = g_list_next(now);
  }

 OUT:
  return candidate;

}
