/*
 * @file  l7vsadm.h
 * @brief l7vsadm header file.
 *
 * L7VSADM: Virtual Server Administration Program for L7vsd
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef __L7VSADM_H__
#define __L7VSADM_H__

#include <glib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include "l7vs_module.h"
#include "l7vs_service.h"

#include "logger_wrapper.h"
#include "parameter_wrapper.h"

#define L7VSADM_NAME "l7vsadm"			//!< L7vsadm name string
#define L7VSADM_COMPONENT PARAM_COMP_L7VSADM	//!< L7vsadm component
#define L7VSADM_DEFAULT_LIST_VS_BUFSIZE (4096)	//!< Default vs buffer size
#define L7VSADM_DEFAULT_LIST_RS_RSNUM   (16)	//!< Default number of rs
#define L7VSADM_DEFAULT_BACKLOG (127)		//!< Default backlog size
#define L7VSADM_DEFAULT_SCHEDULER "rr"		//!< Default scheduler name
#define L7VSADM_DEFAULT_WAIT_INTERVAL (1)	//!< Default wait interval
#define L7VSADM_DEFAULT_WAIT_COUNT (10)		//!< Default wait count
#define L7VSADM_MAX_WAIT (60)			//!< Max wait value
#define L7VSADM_INIT_ENDPOINT_STR "0.0.0.0:0"	//!< Endpoint string for initialize

#define L7VSADM_RECV_FAILED (-1)		//!< Receive faild
#define L7VSADM_RECV_TOO_LARGE (-2)		//!< Receive size too large

//! L7vsadm command code.
enum L7VSADM_COMMAND_CODE_TAG {
	CMD_LIST,		//!< List command(-l)
	CMD_LIST_VERBOSE,	//!< Verbose list command(-V)
	CMD_LIST_KEY,		//!< Key list command(-K)
	CMD_ADD_VS,		//!< Add VirtualService command(-A)
	CMD_DEL_VS,		//!< Delete VirtualService command(-D)
	CMD_EDIT_VS,		//!< Edit VirtualService command(-E)
	CMD_FLUSH_VS,		//!< Flush VirtualService command(-C)
	CMD_ADD_RS,		//!< Add RealServer command(-a)
	CMD_DEL_RS,		//!< Delete RealServer command(-d)
	CMD_EDIT_RS,		//!< Edit RealServer command(-e)
	CMD_REPLICATION,	//!< Replication command(-R)
	CMD_LOG,		//!< Logger command(-L)
	CMD_SNMP,		//!< SNMPAgent command(-S)
	CMD_PARAMETER,		//!< Parameter command(-P)
	CMD_HELP,		//!< Help command(-h)
	CMD_NONE		//!< No command
};

//! L7vsadm option code.
enum L7VSADM_OPTION_CODE_TAG {
	OPT_TCP_SERVICE,	//!< VirtualService option(-t)
	OPT_PROTO_MODULE,	//!< ProtocolModule option(-m)
	OPT_SCHEDULER,		//!< ScheduleModule option(-s)
	OPT_UPPER,		//!< Sorry max connection option(-u)
	OPT_BYPASS,		//!< SorryServer option(-b)
	OPT_FLAG,		//!< Sorry flag option(-f)
	OPT_QOS_UP,		//!< QoS-UP option(-Q)
	OPT_QOS_DOWN,	//!< QoS-DOWN option(-q)
	OPT_REAL_SERVER,	//!< RealServer option(-r)
	OPT_WEIGHT,		//!< RealServer weight option(-w)
	OPT_SWITCH,		//!< Replication switching option(-s)
	OPT_FORCE,		//!< Force Replication option(-f)
	OPT_DUMP,		//!< Replication data dump option(-d)
	OPT_CATEGORY,		//!< Log-Category option(-c)
	OPT_LEVEL,		//!< Log-Level option(-l)
	OPT_RELOAD,		//!< Reload Parameter option(-r)
	OPT_NUMERIC,		//!< Numeric list option(-n)
	OPT_NONE		//!< No option
};

//! L7vsadm parameter code.
enum L7VSADM_PARAMETER_TAG {
	PARA_CMD_INTERVAL,	//!< Command wait interval parameter
	PARA_CMD_COUNT,		//!< Command wait count parameter
	PARA_CON_INTERVAL,	//!< Connect wait interval parameter
	PARA_CON_COUNT,		//!< Connect wait count parameter
	PARA_NONE		//!< No parameter
};

//! L7vsadm parsed option data.
struct l7vsadm_option_data {
	enum L7VSADM_COMMAND_CODE_TAG command_code;	//!< Command code
	int (*parse_func)(int argc, char *argv[]);	//!< Parse function pointer
	int (*operate_func)(void);			//!< Operate function pointer
	struct sockaddr_in vs_addr;			//!< VirtualService endpoint
	struct sockaddr_in rs_addr;			//!< RealServer endpoint
	char protomod_name[L7VS_MODNAME_LEN];		//!< ProtocolModule name
	char schedmod_name[L7VS_MODNAME_LEN];		//!< ScheduleModule name
	struct l7vs_protomod *protomod;			//!< ProtocolModule data
	int persist;					//!< Persist
	int backlog;					//!< Backlog
	int weight;					//!< RealServer weight
	int isnumeric;					//!< Numeric list
	GArray *protomod_options;			//!< ProtocolModule option
	struct l7vs_service_arg_multi protomod_sarg;	//!< Service_arg_multi
	int max_connection;				//!< Sorry max connection
	struct sockaddr_storage sorry_addr;		//!< SorryServer endpoint
	int sorry_flag;					//!< Sorry flag
	unsigned long long qos_threshold_up;			//!< QoS threshold to real server (UP Direction)
	unsigned long long qos_threshold_down;			//!< QoS threshold to client (DOWN Direction)
	enum L7VSADM_OPTION_CODE_TAG control_code;	//!< Control code(Sub command)
	enum LOG_CATEGORY_TAG log_category;		//!< Log-Category
	enum LOG_LEVEL_TAG log_level;			//!< Log-Level
	int category_all_flag;				//!< All Log-Category flag
	int replication_start_flag;			//!< Start Replication flag
	enum PARAMETER_COMPONENT_TAG reload_param;	//!< Parameter Reload Componet
};

//! L7vsadm command code list.
struct l7vsadm_command_list {
	enum L7VSADM_COMMAND_CODE_TAG command_code;	//!< Command code
	const char *command_name;			//!< Command name
	const char *command_name_long;			//!< Command long name
	int (*parse_func)(int argc, char *argv[]);	//!< Parse function pointer
	int (*operate_func)(void);			//!< Operate function pointer
};

//! L7vsadm option code list.
struct l7vsadm_option_list {
	enum L7VSADM_OPTION_CODE_TAG option_code;	//!< Option code
	const char *option_name;			//!< Option name
	const char *option_name_long;			//!< Option long name
	int check_flag;					//!< Conflict check flag
};

//! L7vsadm parameter code list.
struct l7vsadm_parameter_list {
	enum L7VSADM_PARAMETER_TAG parameter_code;	//!< Parameter code
	const char *parameter_name;			//!< Parameter name
};

//! L7vsadm log-category code list.
struct l7vsadm_category_name_list {
	enum LOG_CATEGORY_TAG category_code;		//!< Log-Category code
	const char *category_name;			//!< Log-Category name
	const char *category_name_short;		//!< Log-Category short name
};

//! L7vsadm log-level code list.
struct l7vsadm_level_name_list {
	enum LOG_LEVEL_TAG level_code;			//!< Log-Level code
	const char *level_name;				//!< Log-Level name
};

//! L7vsadm component code list.
struct l7vsadm_component_name_list {
	enum PARAMETER_COMPONENT_TAG component_code;	//!< Component code
	const char *component_name;			//!< Component name
};

//! L7vsadm io vector data.
struct l7vsadm_iov_data {
	struct iovec req_iov[2];			//!< Request io-vector
	int req_cnt;					//!< Request io-vector count
	struct iovec rsp_iov[2];			//!< Response io-vector
	int rsp_cnt;					//!< Response io-vector count
};

//! L7vsadm signal list.
struct l7vsadm_sig_list {
	int sig;					//!< Signal number
	void (*sig_func)(int sig);			//!< Signal hander pointer
};

//! Debug functions.
extern void endpoint_c_str(char* ep_str, void *addr);
extern void argc_argv_c_str(char* arg_str, int argc, char *argv[]);
extern void l7vsadm_option_data_c_str(char* parse_str);

#endif //__L7VSADM_H__
