/*
 * Copyright (c) 2007 Mockito contributors
 * This program is made available under the terms of the MIT License.
 */
package org.mockito.stubbing;

import org.mockito.Mockito;
import org.mockito.internal.progress.IOngoingStubbing;

/**
 * Simply put: "<b>When</b> the x method is called <b>then</b> return y". E.g:
 *
 * <pre>
 * <b>when</b>(mock.someMethod()).<b>thenReturn</b>(10);
 *
 * //you can use flexible argument matchers, e.g:
 * when(mock.someMethod(<b>anyString()</b>)).thenReturn(10);
 *
 * //setting exception to be thrown:
 * when(mock.someMethod("some arg")).thenThrow(new RuntimeException());
 *
 * //you can set different behavior for consecutive method calls.
 * //Last stubbing (e.g: thenReturn("foo")) determines the behavior of further consecutive calls.
 * when(mock.someMethod("some arg"))
 *  .thenThrow(new RuntimeException())
 *  .thenReturn("foo");
 *
 * //There is a shorter way of consecutive stubbing:
 * when(mock.someMethod()).thenReturn(1,2,3);
 * when(mock.otherMethod()).thenThrow(exc1, exc2);
 * </pre>
 *
 * See examples in javadoc for {@link Mockito#when}
 */
public interface OngoingStubbing<T> extends IOngoingStubbing {

    /**
     * Sets a return value to be returned when the method is called. E.g:
     * <pre>
     * when(mock.someMethod()).thenReturn(10);
     * </pre>
     *
     * See examples in javadoc for {@link Mockito#when}
     *
     * @param value return value
     *
     * @return iOngoingStubbing object that allows stubbing consecutive calls
     */
    OngoingStubbing<T> thenReturn(T value);
    OngoingStubbing<T> thenReturn(byte value);
    OngoingStubbing<T> thenReturn(short value);
    OngoingStubbing<T> thenReturn(int value);
    OngoingStubbing<T> thenReturn(long value);
    OngoingStubbing<T> thenReturn(float value);
    OngoingStubbing<T> thenReturn(double value);
    OngoingStubbing<T> thenReturn(char value);
    OngoingStubbing<T> thenReturn(boolean value);

    /**
     * Sets consecutive return values to be returned when the method is called. E.g:
     * <pre>
     * when(mock.someMethod()).thenReturn(1, 2, 3);
     * </pre>
     *
     * Last return value in the sequence (in example: 3) determines the behavior of further consecutive calls.
     * <p>
     * See examples in javadoc for {@link Mockito#when}
     *
     * @param value first return value
     * @param values next return values
     *
     * @return iOngoingStubbing object that allows stubbing consecutive calls
     */
    OngoingStubbing<T> thenReturn(T value, T... values);
    OngoingStubbing<T> thenReturn(T value0, T value1);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9,
    		T value10);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9,
    		T value10, T value11);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9,
    		T value10, T value11, T value12);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9,
    		T value10, T value11, T value12, T value13);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9,
    		T value10, T value11, T value12, T value13, T value14);
    OngoingStubbing<T> thenReturn(T value0, T value1, T value2, T value3, T value4, T value5, T value6, T value7, T value8, T value9,
    		T value10, T value11, T value12, T value13, T value14, T value15);
    OngoingStubbing<T> thenReturn(byte value0, byte value1);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9,
    		byte value10);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9,
    		byte value10, byte value11);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9,
    		byte value10, byte value11, byte value12);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9,
    		byte value10, byte value11, byte value12, byte value13);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9,
    		byte value10, byte value11, byte value12, byte value13, byte value14);
    OngoingStubbing<T> thenReturn(byte value0, byte value1, byte value2, byte value3, byte value4, byte value5, byte value6, byte value7, byte value8, byte value9,
    		byte value10, byte value11, byte value12, byte value13, byte value14, byte value15);
    OngoingStubbing<T> thenReturn(short value0, short value1);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9,
    		short value10);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9,
    		short value10, short value11);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9,
    		short value10, short value11, short value12);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9,
    		short value10, short value11, short value12, short value13);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9,
    		short value10, short value11, short value12, short value13, short value14);
    OngoingStubbing<T> thenReturn(short value0, short value1, short value2, short value3, short value4, short value5, short value6, short value7, short value8, short value9,
    		short value10, short value11, short value12, short value13, short value14, short value15);
    OngoingStubbing<T> thenReturn(int value0, int value1);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9,
    		int value10);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9,
    		int value10, int value11);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9,
    		int value10, int value11, int value12);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9,
    		int value10, int value11, int value12, int value13);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9,
    		int value10, int value11, int value12, int value13, int value14);
    OngoingStubbing<T> thenReturn(int value0, int value1, int value2, int value3, int value4, int value5, int value6, int value7, int value8, int value9,
    		int value10, int value11, int value12, int value13, int value14, int value15);
    OngoingStubbing<T> thenReturn(long value0, long value1);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9,
    		long value10);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9,
    		long value10, long value11);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9,
    		long value10, long value11, long value12);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9,
    		long value10, long value11, long value12, long value13);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9,
    		long value10, long value11, long value12, long value13, long value14);
    OngoingStubbing<T> thenReturn(long value0, long value1, long value2, long value3, long value4, long value5, long value6, long value7, long value8, long value9,
    		long value10, long value11, long value12, long value13, long value14, long value15);
    OngoingStubbing<T> thenReturn(float value0, float value1);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9,
    		float value10);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9,
    		float value10, float value11);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9,
    		float value10, float value11, float value12);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9,
    		float value10, float value11, float value12, float value13);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9,
    		float value10, float value11, float value12, float value13, float value14);
    OngoingStubbing<T> thenReturn(float value0, float value1, float value2, float value3, float value4, float value5, float value6, float value7, float value8, float value9,
    		float value10, float value11, float value12, float value13, float value14, float value15);
    OngoingStubbing<T> thenReturn(double value0, double value1);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9,
    		double value10);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9,
    		double value10, double value11);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9,
    		double value10, double value11, double value12);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9,
    		double value10, double value11, double value12, double value13);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9,
    		double value10, double value11, double value12, double value13, double value14);
    OngoingStubbing<T> thenReturn(double value0, double value1, double value2, double value3, double value4, double value5, double value6, double value7, double value8, double value9,
    		double value10, double value11, double value12, double value13, double value14, double value15);
    OngoingStubbing<T> thenReturn(char value0, char value1);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9,
    		char value10);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9,
    		char value10, char value11);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9,
    		char value10, char value11, char value12);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9,
    		char value10, char value11, char value12, char value13);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9,
    		char value10, char value11, char value12, char value13, char value14);
    OngoingStubbing<T> thenReturn(char value0, char value1, char value2, char value3, char value4, char value5, char value6, char value7, char value8, char value9,
    		char value10, char value11, char value12, char value13, char value14, char value15);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9,
    		boolean value10);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9,
    		boolean value10, boolean value11);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9,
    		boolean value10, boolean value11, boolean value12);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9,
    		boolean value10, boolean value11, boolean value12, boolean value13);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9,
    		boolean value10, boolean value11, boolean value12, boolean value13, boolean value14);
    OngoingStubbing<T> thenReturn(boolean value0, boolean value1, boolean value2, boolean value3, boolean value4, boolean value5, boolean value6, boolean value7, boolean value8, boolean value9,
    		boolean value10, boolean value11, boolean value12, boolean value13, boolean value14, boolean value15);


    /**
     * Sets Throwable objects to be thrown when the method is called. E.g:
     * <pre>
     * when(mock.someMethod()).thenThrow(new RuntimeException());
     * </pre>
     *
     * If throwables contain a checked exception then it has to
     * match one of the checked exceptions of method signature.
     * <p>
     * You can specify throwables to be thrown for consecutive calls.
     * In that case the last throwable determines the behavior of further consecutive calls.
     * <p>
     * if throwable is null then exception will be thrown.
     * <p>
     * See examples in javadoc for {@link Mockito#when}
     *
     * @param throwables to be thrown on method invocation
     *
     * @return iOngoingStubbing object that allows stubbing consecutive calls
     */
    OngoingStubbing<T> thenThrow(Throwable... throwables);
    OngoingStubbing<T> thenThrow(Throwable throwable0);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9,
    		Throwable throwable10);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9,
    		Throwable throwable10, Throwable throwable11);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9,
    		Throwable throwable10, Throwable throwable11, Throwable throwable12);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9,
    		Throwable throwable10, Throwable throwable11, Throwable throwable12, Throwable throwable13);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9,
    		Throwable throwable10, Throwable throwable11, Throwable throwable12, Throwable throwable13, Throwable throwable14);
    OngoingStubbing<T> thenThrow(Throwable throwable0, Throwable throwable1, Throwable throwable2, Throwable throwable3, Throwable throwable4,
    		Throwable throwable5, Throwable throwable6, Throwable throwable7, Throwable throwable8, Throwable throwable9,
    		Throwable throwable10, Throwable throwable11, Throwable throwable12, Throwable throwable13, Throwable throwable14, Throwable throwable15);

    /**
     * Sets the real implementation to be called when the method is called on a mock object.
     * <p>
     * As usual you are going to read <b>the partial mock warning</b>:
     * Object oriented programming is more less tackling complexity by dividing the complexity into separate, specific, SRPy objects.
     * How does partial mock fit into this paradigm? Well, it just doesn't...
     * Partial mock usually means that the complexity has been moved to a different method on the same object.
     * In most cases, this is not the way you want to design your application.
     * <p>
     * However, there are rare cases when partial mocks come handy:
     * dealing with code you cannot change easily (3rd party interfaces, interim refactoring of legacy code etc.)
     * However, I wouldn't use partial mocks for new, test-driven & well-designed code.
     * <pre>
     *   // someMethod() must be safe (e.g. doesn't throw, doesn't have dependencies to the object state, etc.)
     *   // if it isn't safe then you will have trouble stubbing it using this api. Use Mockito.doCallRealMethod() instead.
     *   when(mock.someMethod()).thenCallRealMethod();
     *
     *   // calls real method:
     *   mock.someMethod();
     *
     * </pre>
     * See also javadoc {@link Mockito#spy(Object)} to find out more about partial mocks.
     * <b>Mockito.spy() is a recommended way of creating partial mocks.</b>
     * The reason is it guarantees real methods are called against correctly constructed object because you're responsible for constructing the object passed to spy() method.
     * <p>
     * See examples in javadoc for {@link Mockito#when}
     *
     * @return iOngoingStubbing object that allows stubbing consecutive calls
     */
    OngoingStubbing<T> thenCallRealMethod();

    /**
     * Sets a generic Answer for the method. E.g:
     * <pre>
     * when(mock.someMethod(10)).thenAnswer(new Answer&lt;Integer&gt;() {
     *     public Integer answer(InvocationOnMock invocation) throws Throwable {
     *         return (Integer) invocation.getArguments()[0];
     *     }
     * }
     * </pre>
     *
     * @param answer the custom answer to execute.
     *
     * @return iOngoingStubbing object that allows stubbing consecutive calls
     */
    OngoingStubbing<T> thenAnswer(Answer<?> answer);
}
