#!/bin/sh
# the next line restarts using wish \
exec wish "$0" "$@"

# mac
# - (追加) 録音／再生ボタンを追加
# - (追加) 各種ショートカット
# - (追加) 単独音/連続音の自動パラメータ設定機能
# - (追加) 詳細設定で、F0ターゲット値に合わせて他の設定値を自動設定するボタンを追加した(autoF0Settings)
# - (変更) 読み込んだ音名リストや発声タイプリストに合わせて一覧表の横幅を増やすようにした
# - (変更) ustファイルから発声リストを作る場合，ustをnkf -wに通すようにした(makeRecListFromUst)
# - (修正) 録音中の再生キー使用を無効にした。
# - (変更) F0最高値のデフォルト値を400から800に引き上げた
# - (変更) メトロノームや音叉再生中に録音開始した場合はそれらを再生停止するようにした
# - (変更) Win版ではControl+ホイールで波形横幅や音名リスト横幅を変更していたが、
#          Mac版ではCommand+ホイールに割り当てた。
# - (削除) オーディオI/Oの設定メニューを削除(Macでは無効そうだったので)
#
#todo,koko:
# チュートリアル作成
# オンラインマニュアルmac版
# UTAUでのチェック。ファイルの漢字コードはUTF-8?
# menu->file->ustからリスト生成をチェック。ustの漢字コードとか。
# Windowsへ移植。recNowなど
# 録音後終了後に保存しますかと聞かれない→M1-qをbindできなかったので無理そう
# alias=* あ...母音結合用の原音設定らしい。

# 2.0-b110624
# - (追加) ustファイルから発声リストを作成できるようにした(makeRecListFromUst)
# - (削除) setParam用のサブルーチンを削除した
# - (修正) オーディオI/O設定窓でエラーが出る問題を修正した(ioSettings)
# - (追加) オーディオI/O設定窓下部に警告文を追加した。
# - (追加) ESCで各種設定窓を閉じるようにした(bgmGuide, pitchGuide, tempoGuide, ioSettings, settings)
# - (追加) 音叉窓最下部にショートカットキー一覧を表示した

# 2.0-b100509
# - (変更) 描画コードをsetParamで作ったものベースにした(Redraw)
# - (修正) オーディオデバイスの文字化け周りのバグ修正(setIODevice)

# 2.0-b100204
# - (追加) 波形の表示/非表示を切り替えられるようにした(toggleWaveなど)
# - (追加) 読み込み済みのパラメータに別の原音パラメータファイルをマージする(mergeParamFile)
# - (追加) 選択中の範囲の値を一括変更する機能(changeCell)
# - (追加) オーディオデバイスのレイテンシを変更する機能(ioSettings)
# - (修正) オーディオドライバ名の文字化けを若干解消(ioSettings)
# - (変更) 原音パラメータを読む際にwavが存在しないエントリは削除するようにした(readParamFile)

# 2.0-b091205
# - (追加) メイン窓にD&Dされたときの処理を追加(procDnd)
# - (追加) 発声タイミング補正切替を追加(timingAdjMode)
# - (修正) 細かいバグの修正。
# - (追加) プログレスバー表示
# - (追加) oto.ini読み込み高速化用のキャッシュ機能
# - (追加) F3やAlt-F3で他パラメータを連動してうごかせるようにした。
# - (追加) エイリアス一括変換機能を追加(changeAlias)
# - (修正) 連続音パラメータ生成直後にspaceで再生すると、表示中の波形でない波形が再生されるバグを修正。

# 2.0-b091120
# - (変更) 全メッセージを外部ファイル化。
# - (追加) wav両端の無音をカットする機能を追加 (cutWav)
# - (修正) パラメータ一覧表の数値を削除すると"0"と表示されるバグ?を修正。

# 2.0-b091104
# - (追加) 読み込み時にパラメータ生成(単独/連続音)を選択実行できるようにした。
# - (追加) 先行発声チェック用の試聴機能および設定窓を追加
# - (追加) 自動収録した連続音のパラメータ自動生成(genParam)
# - (変更) 初期化ファイル保存の保存対象を変更(saveSettings)
# - (修正) 行を複製する際にパラメータに空欄がある場合のコピーのバグを修正。

# 2.0-b091007
# - (修正) 以前作った左右ブランク自動推定を最新バージョンで動くように修正。

# 2.0-b090903
# - (追加) 左ブランク値を変更した際に、同wavファイルの他の音の左ブランク値も
#          連動して変更できるようにした。
# - (変更) 各種窓を開いたときにフォーカスするように変更。
# - (追加) パラメータ検索を実装。
# - (変更) パラメータ一覧表のタイトルが長すぎる場合は切り詰めるように変更。
# - (変更) マウス+F1～F5で各パラメータをドラッグ可能。
# - (修正) setParamでF0が表示されないバグの修正。
# - (修正) setParamでマウスドラッグによるセル複数選択ができないバグの修正。

# 2.0-b090822
# - (修正) setParamの一覧表タイトルのファイル名表示が更新されないバグを修正。
# - (修正) setParamの一覧表の値に挿入・削除したときカーソルが末尾に行くバグを修正。
# - (変更) 全パラメータをマイクロsec精度にした。
# - (追加) 右ブランクの負の値に対応。
# - (追加) オプションで右ブランクの正負を切り替えられるようにした。
# - (追加) オプションで左ブランクの変更時の他パラメータのふるまいを
#          切り替えられるようにした
# (2.0-b090813)
# - (変更) リストスクロールで２つ前後の音が見えるようにした。
# - (追加) 初期化ファイルを生成できるようにした。

# 2.0-b090803
# - (修正) readParamFile。oto.iniにエントリが足りない場合のバグを修正。
# -（追加) ツールメニューにDC成分一括除去を追加
# -（追加) ツールメニューにwavファイル名変更（冒頭に"_"を付ける)を追加
# - (追加) リストボックスの横幅をctrl+wheelで変更可能にした。

# 2.0-b090727
# - (変更) setParamで波形窓にエイリアスを表示。
# - (変更) 一覧表タイトルにファイル名を表示。
# - (変更) 一覧表の上下矢印移動で表の上端・下端でワープしないようにした。
# - (変更) ガイドBGM設定窓で、BGM試聴、録音イメージ音試聴ボタンを追加。
# - (変更) オーディオI/O設定窓に説明文を表示。
# - (修正) 自動録音(loop)で、音名リスト末尾までいったら終了するようにした。
# - (変更) Redrawの演算回数を少し削減。
# - (修正) makeRecListFromDirでのファイル名登録のバグを修正。

# 2.0-b090724
# - (修正) val2sampで実数値を返すことがあるバグを修正。

# 2.0-b090719
# - (修正) メトロノーム再生を停止できなかったバグを修正。

# 2.0-b090715
# - saveParamFile 高速化(paramUの内容を直接書き出すようにした)

# 2.0-b090706
# - oremo.tcl 本体のサブルーチン集を別ファイルに移行。

#---------------------------------------------------
# サブルーチン

#---------------------------------------------------
# メイン画面表示をリセットする？
#
proc resetDisplay {} {
  global v t rec type

  set v(recSeq)  0
  set v(typeSeq) 0
  set v(listSeq) 1
  set v(recLab)  [lindex $v(recList)  $v(recSeq)]
  set v(typeLab) [lindex $v(typeList) $v(typeSeq)]
  readWavFile
  Redraw all
}

#---------------------------------------------------
# 保存フォルダにあるwavファイルを読み、リストに記憶する
#
proc makeRecListFromDir {{overWriteRecList 1}} {
  global v t

  set recList {}
  foreach filename [glob -nocomplain [format "%s/*.wav" $v(saveDir)]] {
    set filename [file rootname [file tail $filename]]
    if {$filename == ""} continue
    ;# フォルダおよび拡張子を取り除いたファイル名をリストに格納
    ;# 音名と発声タイプは分けない
    lappend recList $filename
  }
  if $overWriteRecList {
    set v(recList) $recList
    set v(typeList) {""}
  }
  initParamS
  initParamU 0 $recList
}

#---------------------------------------------------
# reclist.txtを保存する
#
proc saveRecList {} {
  global v t

  set fn [my_getSaveFile -initialfile $v(recListFile) \
            -title $t(saveRecList,title) -defaultextension "txt" ]
  if {$fn == ""} return

  set v(recListFile) $fn
  if [catch {open $v(recListFile) w} out] { 
    tk_messageBox -message [eval format $t(saveRecList,errMsg)] \
      -title $t(.confm.fioErr) -icon warning
  } else { 
    foreach sn $v(recList) {
      if [catch {set data [puts $out $sn]}] {
        tk_messageBox -message [eval format $t(saveRecList,errMsg2)] \
          -title $t(.confm.fioErr) -icon warning
      }
    }
    close $out
  }
  set v(msg) [eval format $t(saveRecList,doneMsg)]
}

#---------------------------------------------------
# USTファイルからリストを生成する
#
proc makeRecListFromUst {args} {
  global v t

  if {[llength $args] == 0 || ! [file exists $v(recListFile)]} {
    set fn [my_getOpenFile -initialfile $v(recListFile) \
            -title $t(makeRecListFromUst,title1) -defaultextension "ust" \
            -filetypes { {{reclist file} {.ust}} {{All Files} {*}} }]
  } else {
    set fn [lindex $args 0]
  }
  if {$fn == ""} return
  set v(recListFile) [file rootname $fn].txt

  if {$::tcl_platform(os) == "Darwin"} {
    global nkf nkfResult
    exec -- $nkf -w8 $fn > $nkfResult     ;# 漢字コードをutf-8に変換
    set fn $nkfResult
  }

  if [catch {open $fn r} in] { 
    tk_messageBox -message [eval format $t(makeRecListFromUst,errMsg)]
      -title $t(.confm.fioErr) -icon warning
  } else { 
    set v(recList) {}
    while {![eof $in]} {
      set data [split [gets $in] "="]
      if {[llength $data] > 1} {
        set item [lindex $data 0]       ;# 項目名
        set val  [lindex $data 1]       ;# データ内容
        if {$item == "Lyric"} {
          ;# 重複がなければリストに追加
          if {[lsearch -exact $v(recList) $val] < 0} {
            lappend v(recList) $val
          }
        }
      }
    }
    close $in
    if {$::tcl_platform(os) == "Darwin"} {
      file delete $nkfResult
    }
  }
  set v(recSeq) 0
  set v(recLab) [lindex $v(recList) $v(recSeq)]
  set v(msg) [eval format $t(makeRecListFromUst,doneMsg)]
  set v(typeList) {""}
}

#---------------------------------------------------
# 音名リストファイルを読み、リストに記憶する
#
proc readRecList {args} {
  global v t rec

  if {[llength $args] == 0 || ! [file exists $v(recListFile)]} {
    set fn [my_getOpenFile -initialfile $v(recListFile) \
            -title $t(readRecList,title1) -defaultextension "txt" \
            -filetypes { {{reclist file} {.txt}} {{All Files} {*}} }]
  } else {
    set fn [lindex $args 0]
  }
  if {$fn == ""} return
  set v(recListFile) $fn

  if [catch {open $v(recListFile) r} in] { 
    tk_messageBox -message [eval format $t(readRecList,errMsg)]
      -title $t(.confm.fioErr) -icon warning
  } else { 
    if [catch {set data [read -nonewline $in]}] {
      tk_messageBox -message [eval format $t(readRecList,errMsg2)]
        -title $t(.confm.fioErr) -icon warning
    }
    regsub -all {[[:space:]]} $data " " data
    set v(recList) {}
    set maxStrlen 0
    foreach line [split $data " "] {
      if {$line != ""} {
        lappend v(recList) $line
	if {$maxStrlen < [string bytelength $line]} {
          set maxStrlen [string bytelength $line]
	}
      }
    }
    close $in
    if {$maxStrlen > 20} {
      set maxStrlen 20
    }
    if [info exists rec] {
      $rec configure -width [expr $maxStrlen -1]
    }
    set v(recSeq) 0
    set v(recLab) [lindex $v(recList) $v(recSeq)]
    set v(msg) [eval format $t(readRecList,doneMsg)]
    #initParamU
  }
}

#---------------------------------------------------
# 発声タイプのリストファイルを読み、リストに記憶する
# リストの最初の要素は "" を入れておく
#
proc readTypeList {args} {
  global v t type

  if {[llength $args] == 0 || ! [file exists $v(typeListFile)]} {
    set fn [my_getOpenFile -initialfile $v(typeListFile) \
            -title $t(readTypeList,title) -defaultextension "txt" \
            -filetypes { {{typelist file} {.txt}} {{All Files} {*}} }]
  } else {
    set fn [lindex $args 0]
  }
  if {$fn == ""} return
  set v(typeListFile) $fn

  set v(typeList) {""}
  if [catch {open $v(typeListFile) r} in] { 
    tk_messageBox -message [eval format $t(readTypeList,errMsg) \
      -title $t(.confm.fioErr) -icon warning
  } else { 
    if [catch {set data [read -nonewline $in]}] {
      tk_messageBox -message [eval format $t(readTypeList,errMsg2) \
        -title $t(.confm.fioErr) -icon warning
    } else {
      regsub -all {[[:space:]]} $data " " data
      set maxStrlen 0
      foreach line [split $data " "] {
        if {$line != ""} {
          lappend v(typeList) $line
	  if {$maxStrlen < [string bytelength $line]} {
            set maxStrlen [string bytelength $line]
	  }
        }
      }
    }
    close $in
    set v(typeSeq) 0
    if {$maxStrlen > 20} {
      set maxStrlen 20
    }
    if [info exists type] {
      $type configure -width [expr $maxStrlen -1]
    }
    set v(typeLab) [lindex $v(typeList) $v(typeSeq)]
    set v(msg) [eval format $t(readTypeList,doneMsg)]
    #initParamU
  }
}

#---------------------------------------------------
# 日本語フォントを設定する
#
proc fontSetting {} {
  global v t

  switch $::tcl_platform(platform) {
    unix {
      font create bigkfont -family gothic -size $v(bigFontSize) \
        -weight normal -slant roman
      font create kfont -family gothic -size $v(fontSize) \
        -weight normal -slant roman
      font create smallkfont -family gothic -size $v(smallFontSize) \
        -weight normal -slant roman
    }
    windows {
      font create bigkfont -family $t(fontName) -size $v(bigFontSize) \
        -weight normal -slant roman
      font create kfont    -family $t(fontName) -size $v(fontSize) \
        -weight normal -slant roman
      font create smallkfont -family $t(fontName) -size $v(smallFontSize) \
        -weight normal -slant roman
    }
  }
}

#---------------------------------------------------
# オンラインマニュアル
#
proc onlineHelp {url} {
    global v t

    if {$::tcl_platform(platform) == "windows"} {
        if {[string match $::tcl_platform(os) "Windows NT"]} {
            exec $::env(COMSPEC) /c start $url &
        } {
            exec start $url &
        }
    } elseif {$::tcl_platform(os) == "Darwin"} {
      exec open "$url"
    } else {
        # atode, ここはせめてfirefoxにしないと。。
        if [catch {exec sh -c "netscape -remote 'openURL($url)' -raise"} res] {
            if [string match *netscape* $res] {
                exec sh -c "netscape $url" &
            }
        }
    }
}

#---------------------------------------------------
# 保存ディレクトリを指定する
# 変更したら1、キャンセルしたら0を返す
#
proc choosesaveDir {{readParam 0}} {
  global v t

  set d [my_chooseDirectory -initialdir $v(saveDir) -title $t(choosesaveDir,title)]
  if {$d != ""} {
    set v(saveDir) $d
    set v(msg) [eval format $t(choosesaveDir,doneMsg)]

    #if {$readParam != 0} {
    #  set act [tk_dialog .confm $t(.confm) $t(choosesaveDir,q) \
    #    question 0 $t(.confm.r) $t(.confm.nr)]
    #  set v(paramFile) "$v(saveDir)/oto.ini"
    #  if {$act == 0} readParamFile
    #}
    return 1  ;# 変更あり
  }
  return 0    ;# 変更なし
}

#---------------------------------------------------
# 未保存であれば波形をファイルに保存する
#
proc saveWavFile {} {
  global v snd t

  if $v(recStatus) {
    if {[snd length] > 0} {
      if {[file exists $v(saveDir)] == 0} {
        file mkdir $v(saveDir)
      }
      snd write $v(saveDir)/$v(recLab)$v(typeLab).wav 
      set v(msg) [eval format $t(saveWavFile,doneMsg)]
      set v(recStatus) 0
    }
  }
}

#---------------------------------------------------
# ファイルから波形を読む
#
proc readWavFile {} {
  global v snd t

  if {[snd length] > 0} { snd length 0 }
  if {[file readable $v(saveDir)/$v(recLab)$v(typeLab).wav]} {
    snd read $v(saveDir)/$v(recLab)$v(typeLab).wav
  }
}

#---------------------------------------------------
# 平均律の各音階の周波数を求める
#
proc setSinScale {} {
  global v t
  set v(sinScale) {}
  set v(sinNote) {}
  for {set oct $v(sinScaleMin)} {$oct <= $v(sinScaleMax)} {incr oct} {
    for {set i 0} {$i < 12} {incr i} {
      lappend v(sinScale) [expr int(27.5 * pow(2, $oct + ($i - 9.0)/12.0) + 0.5)]
      lappend v(sinNote) "[lindex $v(toneList) $i]$oct"
    }
  }
}

#---------------------------------------------------
# F0計算中にキーボード、マウス入力を制限させるための窓
# うまく窓を表示できていないが、F0計算中の入力を制限できるので
# とりあえずOK(F0計算中に入力すると落ちることがあるため)
#
proc waitWindow {message fraction} {
  global t
  set w .waitw
  if {$fraction >= 1.0 && [winfo exists $w]} {
    grab release $w
    destroy $w
    return
  }
  if [winfo exists $w] return

  toplevel $w
  grab set $w
  wm title $w $t(waitWindow,title)
  label $w.l -text "calc.."
  pack $w.l
  wm transient $w .
  wm geom $w +100+100
}

#---------------------------------------------------
#
proc changeTone {chg} {
  global f0 v t
  
  set next [expr [lsearch $v(toneList) $f0(guideTone)] + $chg]

  if {$next < 0} {
    if {$f0(guideOctave) > $v(sinScaleMin)} {
      set next [expr ($next + [llength $v(toneList)])]
      incr f0(guideOctave) -1
    } else {
      set next [lsearch $v(toneList) $f0(guideTone)]
    }
  } elseif {$next >= [llength $v(toneList)]} {
    if {$f0(guideOctave) < $v(sinScaleMax)} {
      set next [expr $next % [llength $v(toneList)]]
      incr f0(guideOctave)
    } else {
      set next [lsearch $v(toneList) $f0(guideTone)]
    }
  }

  set f0(guideTone) [lindex $v(toneList) $next]
  if {$v(playOnsaStatus)} {  ;# もしループ再生中だったなら、
    toggleOnsaPlay           ;# 停止して、
    toggleOnsaPlay           ;# もう一度再生。
  }
}

#---------------------------------------------------
# 収録BGMの窓
#
proc bgmGuide {} {
  global v bgm t
  if [isExist .bgmg] return ;# 二重起動を防止
  toplevel .bgmg
  wm title .bgmg $t(bgmGuide,title)
  bind .bgmg <Escape> {destroy .bgmg}

  label .bgmg.lMode -text $t(bgmGuide,mode)
  radiobutton .bgmg.r1 -variable v(rec) -value 1 -command {bgm stop} \
    -text $t(bgmGuide,r1)
  radiobutton .bgmg.r2 -variable v(rec) -value 2 -command {bgm stop} \
    -text $t(bgmGuide,r2)
  radiobutton .bgmg.r3 -variable v(rec) -value 3 -command {bgm stop} \
    -text $t(bgmGuide,r3)
  radiobutton .bgmg.r4 -variable v(rec) -value 0 -command {bgm stop} \
    -text $t(bgmGuide,r4)

  label .bgmg.lWav -text $t(bgmGuide,bgm)
  frame .bgmg.fWav
  button .bgmg.fWav.b1 -textvar v(bgmFile) -relief solid -command {
    set fn [my_getOpenFile -initialfile $v(bgmFile) \
            -title $t(bgmGuide,bTitle) -defaultextension "wav" \
            -filetypes { {{wav file} {.wav}} {{All Files} {*}} }]
    if {$fn != ""} {
      set v(bgmFile) $fn
      bgm stop
;#      set v(playMetroStatus) 0
    }
  }
  button .bgmg.fWav.b2 -image snackOpen -highlightthickness 0 -bg $v(bg) -command {
    set fn [my_getOpenFile -initialfile $v(bgmFile) \
            -title $t(bgmGuide,bTitle) -defaultextension "wav" \
            -filetypes { {{wav file} {.wav}} {{All Files} {*}} }]
    if {$fn != ""} {
      set v(bgmFile) $fn
      bgm stop
;#      set v(playMetroStatus) 0
    }
  }
  button .bgmg.fWav.bp -text $t(bgmGuide,play) -bitmap snackPlay -command {
    testPlayBGM $v(bgmFile)
  }
  button .bgmg.fWav.bs -text $t(bgmGuide,stop) -bitmap snackStop -command {
    testStopBGM 
  }
  pack .bgmg.fWav.b1 -side left -fill x -expand 1
  pack .bgmg.fWav.b2 .bgmg.fWav.bp .bgmg.fWav.bs -side left

  frame .bgmg.fImg
  label .bgmg.fImg.l -text $t(bgmGuide,tplay)
  button .bgmg.fImg.bp -text $t(bgmGuide,play) -bitmap snackPlay -command {
    set ext [file extension $v(bgmFile)]
    testPlayBGM [file rootname $v(bgmFile)]-sample$ext
  }
  button .bgmg.fImg.bs -text $t(bgmGuide,stop) -bitmap snackStop -command {
    testStopBGM 
  }
  pack .bgmg.fImg.l .bgmg.fImg.bp .bgmg.fImg.bs -side left

  grid .bgmg.lMode   -row 0 -column 0 -sticky e
  grid .bgmg.r1      -row 0 -column 1 -sticky w
  grid .bgmg.r2      -row 1 -column 1 -sticky w
  grid .bgmg.r3      -row 2 -column 1 -sticky w
  grid .bgmg.r4      -row 3 -column 1 -sticky w

  grid .bgmg.lWav    -row 4 -column 0 -sticky e
  grid .bgmg.fWav    -row 4 -column 1 -sticky ewsn

  grid .bgmg.fImg    -row 5 -column 1 -sticky ewsn
}

#---------------------------------------------------
# 指定したファイルを読み込んで再生する
#
proc testPlayBGM {fname} {
  global bgm t
  if [snack::audio active] return
  if ![file exists $fname] {
    tk_messageBox -message "[eval format $t(testPlayBGM,errMsg)] (fname=$fname)" \
      -title $t(testPlayBGM,errTitle) -icon warning -parent .bgmg
    return
  }
  bgm read $fname
  bgm play
}

#---------------------------------------------------
# BGMを停止する
#
proc testStopBGM {} {
  global bgm t
  bgm stop
}


#---------------------------------------------------
# メトロノームの窓
#
proc tempoGuide {} {
  global v metro t
  if [isExist .tg] return ;# 二重起動を防止
  toplevel .tg
  wm title .tg $t(tempoGuide,title)

  bind .tg <KeyPress-m>     toggleMetroPlay
  bind .tg <KeyPress-M>     toggleMetroPlay
  bind .tg <Escape>         {destroy .tg}

  label .tg.lwav -text $t(tempoGuide,click)
  button .tg.bwav -textvar v(clickWav) -relief solid -command {
    set fn [my_getOpenFile -initialfile $v(clickWav) \
            -title $t(tempoGuide,clickTitle) -defaultextension "wav" \
            -filetypes { {{wav file} {.wav}} {{All Files} {*}} }]
    if {$fn != ""} {
      set v(clickWav) $fn
      metro stop
      set v(playMetroStatus) 0
    }
  }

  label .tg.l -text $t(tempoGuide,tempo)
  entry .tg.ebpm -textvar v(tempo) -validate key -validatecommand {
          if {![string is integer %P]} {return 0}
          if {%P <= 0} {return 0}
          set v(tempoMSec) [expr 60000.0 / double(%P)]
          metro stop
          set v(playMetroStatus) 0
          return 1
        }
  label .tg.lbpm -text $t(tempoGuide,bpm)
  label .tg.lbpmSec1 -textvar v(tempoMSec) -fg red
  label .tg.lbpmSec2 -text $t(tempoGuide,bpmUnit)
  label .tg.mes -text $t(tempoGuide,comment)

  grid .tg.lwav -row 0 -column 0 -sticky e
  grid .tg.bwav -row 0 -column 1 -columnspan 4 -sticky nesw

  grid .tg.l        -row 1 -column 0 -sticky e
  grid .tg.ebpm     -row 1 -column 1 -sticky nesw
  grid .tg.lbpm     -row 1 -column 2 -sticky w
  grid .tg.lbpmSec1 -row 1 -column 3 -sticky e
  grid .tg.lbpmSec2 -row 1 -column 4 -sticky w

  grid .tg.mes -row 2 -columnspan 5
}

#---------------------------------------------------
# 周波数を指定してsin波を再生する
#
proc pitchGuide {} {
  global v f0 t
  if [isExist .pg] return ;# 二重起動を防止
  toplevel .pg
  wm title .pg $t(pitchGuide,title)

  bind .pg <KeyPress-Up>    {changeTone 1}
  bind .pg <KeyPress-8>     {changeTone 1}
  bind .pg <KeyPress-Down>  {changeTone -1}
  bind .pg <KeyPress-2>     {changeTone -1}
  bind .pg <KeyPress-Left>  {changeTone -12}
  bind .pg <KeyPress-4>     {changeTone -12}
  bind .pg <KeyPress-Right> {changeTone 12}
  bind .pg <KeyPress-6>     {changeTone 12}
  bind .pg <KeyPress-o>     toggleOnsaPlay
  bind .pg <KeyPress-O>     toggleOnsaPlay
  bind .pg <Escape>         {destroy .pg}

  packToneList .pg.tl $t(pitchGuide,sel) guideTone guideOctave guideFreqTmp 10 guideVol

  pack [frame .pg.vl] -fill x
  label .pg.vl.l -text $t(pitchGuide,vol)
  scale .pg.vl.s -from 0 -to 32768 -show no -var f0(guideVol) -orient horiz
  pack .pg.vl.l -side left -anchor nw
  pack .pg.vl.s -side left -anchor nw -fill x -expand 1
  label .pg.mes -text $t(pitchGuide,comment)
  pack .pg.mes -side left -anchor nw

  ;# 各音名に対応する周波数を自動計算して表示する(非常に汚いやり方)
  ;# 音名orオクターブに変化があれば周波数計算を行う
  ;# 周波数をf0(guideFreq)でなくf0(guideFreqTmp)に入れるのは、
  ;#「OK」or「適用」ボタンを押すまで値変更を反映させないため。
  set f0(guideFreqTmp) [tone2freq "$f0(guideTone)$f0(guideOctave)"]
  trace variable f0 w calcGuideFreq
  proc calcGuideFreq {var elm mode} {
    global f0 t
    switch $elm {
      "guideTone" -
      "guideOctave" {
        set f0(guideFreqTmp) [tone2freq "$f0(guideTone)$f0(guideOctave)"]
      }
    }
  }
  bind .pg <Destroy> { trace vdelete f0 w calcGuideFreq }
}

#---------------------------------------------------
# 指定した周波数[Hz]のsin波を再生する
#
proc playSin {freq vol length} {
  global v onsa t
  if [snack::audio active] return
  if $::debug {puts $freq}
  if {$freq > 10 && $vol > 0} { 
#    set f [snack::filter generator $freq $vol 0.0 sine $v(sampleRate)]
    set g  [snack::filter generator $freq $vol 0.01 triangle $length]
    set f1 [snack::filter formant 500 50]
    set f2 [snack::filter formant 1500 75]
    set f3 [snack::filter formant 2500 100]
    set f4 [snack::filter formant 3500 150]
    set f  [snack::filter compose $g $f1 $f2 $f3 $f4]
    onsa play -filter $f -command "$f destroy" 
  }
}

#---------------------------------------------------
# 前の音の収録に移動
#
proc prevRec {args} { 
  global v rec t

  # 引数処理
  array set a [list -save 1]
  array set a $args

  set v(msg) ""
  if $a(-save) { saveWavFile }
  $rec selection clear $v(recSeq)
  if {$v(recSeq) > 0} {
    incr v(recSeq) -1
    $rec see [expr $v(recSeq) - 2]
  } else {
    set v(recSeq) [expr [llength $v(recList)] - 1]
    $rec see $v(recSeq)
  }
  $rec selection set $v(recSeq)
  set v(recLab) [lindex $v(recList) $v(recSeq)] 
  set v(recStatus) 0
  readWavFile
  Redraw all
  if $::debug {puts "前の音へ, recseq=$v(recSeq), v(recLab)=$v(recLab)"}
}

#---------------------------------------------------
# 次の音の収録に移動
#
proc nextRec {args} { 
  global v rec t

  # 引数処理
  array set a [list -save 1]
  array set a $args

  set v(msg) ""
  if $a(-save) { saveWavFile }
  $rec selection clear $v(recSeq)
  if {$v(recSeq) < [expr [llength $v(recList)] - 1]} {
    incr v(recSeq)
    $rec see [expr $v(recSeq) + 2]
  } else {
    set v(recSeq) 0
    $rec see 0
  }
  $rec selection set $v(recSeq)
  set v(recLab) [lindex $v(recList) $v(recSeq)] 
  set v(recStatus) 0
  readWavFile
  Redraw all
  if $::debug { puts "次の音へ, recseq=$v(recSeq), v(recLab)=$v(recLab)" }
}

#---------------------------------------------------
# 指定した番号の音の収録に移動
#
proc jumpRec {index args} { 
  global v rec t

  # 引数処理
  array set a [list -save 1]
  array set a $args

  if {$v(recSeq) == $index} return
  set v(msg) ""
  if $a(-save) { saveWavFile }
  $rec selection clear $v(recSeq)
  set v(recSeq) $index
  $rec see $v(recSeq)
  $rec selection set $v(recSeq)
  set v(recLab) [lindex $v(recList) $v(recSeq)] 
  set v(recStatus) 0
  readWavFile
  Redraw all
  if $::debug { puts "次の音へ, recseq=$v(recSeq), v(recLab)=$v(recLab)" }
}

#---------------------------------------------------
# 前の発話タイプの収録に移動
#
proc prevType {args} { 
  global v type t

  # 引数処理
  array set a [list -save 1]
  array set a $args

  set v(msg) ""
  if $a(-save) { saveWavFile }
  $type selection clear $v(typeSeq)
  if {$v(typeSeq) > 0} {
    incr v(typeSeq) -1
  } else {
    set v(typeSeq) [expr [llength $v(typeList)] - 1]
  }
  $type see $v(typeSeq)
  $type selection set $v(typeSeq)
  set v(typeLab) [lindex $v(typeList) $v(typeSeq)] 
  set v(recStatus) 0
  readWavFile
  Redraw all
  if $::debug { puts "前のタイプへ, typeseq=$v(typeSeq), v(typeLab)=$v(typeLab)" }
}

#---------------------------------------------------
# 次の発話タイプの収録に移動
#
proc nextType {args} { 
  global v type t

  # 引数処理
  array set a [list -save 1]
  array set a $args

  set v(msg) ""
  if $a(-save) { saveWavFile }
  $type selection clear $v(typeSeq)
  if {$v(typeSeq) < [expr [llength $v(typeList)] - 1]} {
    incr v(typeSeq)
  } else {
    set v(typeSeq) 0
  }
  $type see $v(typeSeq)
  $type selection set $v(typeSeq)
  set v(typeLab) [lindex $v(typeList) $v(typeSeq)] 
  set v(recStatus) 0
  readWavFile
  Redraw all
  if $::debug { puts "次のタイプへ, typeseq=$v(typeSeq), v(typeLab)=$v(typeLab)" }
}

#---------------------------------------------------
# 指定した番号の発声タイプの収録に移動
#
proc jumpType {index args} { 
  global v type t

  # 引数処理
  array set a [list -save 1]
  array set a $args

  if {$v(typeSeq) == $index} return
  set v(msg) ""
  if $a(-save) { saveWavFile }
  $type selection clear $v(typeSeq)
  set v(typeSeq) $index
  $type see $v(typeSeq)
  $type selection set $v(typeSeq)
  set v(typeLab) [lindex $v(typeList) $v(typeSeq)] 
  set v(recStatus) 0
  readWavFile
  Redraw all
  if $::debug { puts "次の音へ, typeseq=$v(typeSeq), v(typeLab)=$v(typeLab)" }
}

#---------------------------------------------------
# DC成分除去
#
proc removeDC {} {
  global snd t
  set flt [snack::filter iir -numerator "0.99 -0.99" -denominator "1 -0.99"] 
  snd filter $flt -continuedrain 0
}

#---------------------------------------------------
# 波形を横方向に拡大縮小(ctrl+マウスホイール)
# mode=1...拡大, mode=0...縮小
#
proc changeWidth {mode} {
  global v t
  if $mode {
    incr v(cWidth) +40
  } elseif {$v(cWidth) <= $v(cWidthMin)} {
    set v(cWidth) $v(cWidthMin)
  } else {
    incr v(cWidth) -40
  }
  Redraw scale
}

#---------------------------------------------------
# 音名リストボックスを横方向に拡大縮小(ctrl+マウスホイール)
# mode=1...拡大, mode=0...縮小
#
proc changeRecListWidth {mode} {
  global rec t
  set width [$rec cget -width]
  if $mode {
    $rec configure -width [expr $width +1]   ;# 拡大
  } elseif {$width > 5} {
    $rec configure -width [expr $width -1]   ;# 縮小
  }
}

#---------------------------------------------------
# 発声タイプリストボックスを横方向に拡大縮小(ctrl+マウスホイール)
# mode=1...拡大, mode=0...縮小
#
proc changeTypeListWidth {mode} {
  global type t
  set width [$type cget -width]
  if $mode {
    $type configure -width [expr $width +1]   ;# 拡大
  } elseif {$width > 5} {
    $type configure -width [expr $width -1]   ;# 縮小
  }
}

#---------------------------------------------------
# 自動録音開始(BGMつき)
#
proc autoRecStart {} {
  global bgm bgmParam snd v t

  if [snack::audio active] return
  if {$v(rec) == 0} return   ;# 収録モードでないなら終了

  ;# BGMファイルを読み込む
  if ![file exists $v(bgmFile)] {
    tk_messageBox -title $t(.confm.fioErr) -icon error \
      -message [eval format $t(autoRecStart,errMsg)]
    return
  }
  bgm read $v(bgmFile)

  ;# BGM設定ファイルを読み込む
  set v(bgmParamFile) [file rootname $v(bgmFile)].txt
  if [catch {open $v(bgmParamFile) r} fp] {
    tk_messageBox -title $t(.confm.fioErr) -icon error \
      -message [eval format $t(autoRecStart,errMsg2)]
    return
  }

  set unit [regsub -all -- {,} [string trim [gets $fp]] ""]   ;# 時刻単位を取得
  if ![regexp {^(sec|SEC|msec|MSEC|sample)$} $unit] {
    tk_messageBox -message "[eval format $t(autoRecStart,errMsg3)] ($t(autoRecStart,unit)=$unit)" -icon warning
    return
  }
  array unset bgmParam
  set bgmParam(autoRecStatus) 0  ;# 以下の解析中にエラーが生じた場合は0でreturn
  set sr [bgm cget -rate]
  while {![eof $fp]} {
    set l [gets $fp]
    if {[regexp {^[[:space:]]*#} $l]} continue
    set p [split $l ","] ;# 行,時刻,録音開始,録音停止,次の収録音へ移動,リピート先
    if {[llength $p] >= 6} {
      set seq [string trim [lindex $p 0]]
      set bgmParam($seq,pStart)  [val2samp [string trim [lindex $p 1]] $unit $sr]
      set bgmParam($seq,rStart)  [string trim [lindex $p 2]]
      set bgmParam($seq,rStop)   [string trim [lindex $p 3]]
      set bgmParam($seq,nextRec) [string trim [lindex $p 4]]
      set bgmParam($seq,repeat)  [string trim [lindex $p 5]]
      if {[llength $p] >= 7} {
        set bgmParam($seq,msg)   [string trim [lindex $p 6]]
      } else {
        set bgmParam($seq,msg)   ""
      }
      set bgmParam($seq,repeat)  [string trim [lindex $p 5]]
      if {$seq > 1} {
        if {$bgmParam([expr $seq - 1],repeat) != 0} {
          set bgmParam([expr $seq - 1],pStop) $bgmParam([expr $seq - 1],pStart)
        } else {
          set bgmParam([expr $seq - 1],pStop) [expr $bgmParam($seq,pStart) - 1]
        }
      }
    }
  }
  close $fp
  if {$bgmParam($seq,repeat) != 0} {
    set bgmParam($seq,pStop) $bgmParam($seq,pStart)
  } else {
    ;# スパゲティな設定ファイルの場合、このエラーチェックをすりぬける可能性がある
    tk_messageBox -message [eval format $t(autoRecStart,errMsg4)] -icon error
    return
    ;# set bgmParam($seq,pStop) [bgm length]  ;# BGM末尾
  }

  set bgmParam(autoRecStatus) 1
  set v(recStatus) 1
  set v(recNow) 1
  .msg.msg configure -fg blue
  if {$::tcl_platform(os) == "Darwin"} {
    ;# 起動後に録音を一度も行わず自動収録を開始すると落ちる問題があったため
    ;# ここで一瞬録音させる
    snd record
    snd stop
  }
  autoRec 1 ;# BGM再生・録音開始
}

#---------------------------------------------------
# 自動録音開始(ガイドBGMを再生,録音開始/停止。再帰的に呼ばれる)
#
proc autoRec {seq} {
  global bgm bgmParam v t
  if $bgmParam(autoRecStatus) {
    set com ""
    if {$bgmParam($seq,repeat) != 0} {
      if {$v(rec) == 3} {
        set com "autoRec $bgmParam($seq,repeat)"  ;#リピート
      } else {
        autoRecStop                               ;#リピートせず終了
        return
      }
    } else {
      set com "autoRec [expr $seq + 1]"
    }
    if {$bgmParam($seq,rStart) != 0} {
      .msg.msg configure -fg red
      aRecStart
    }
    if {$bgmParam($seq,rStop)   != 0} {
      .msg.msg configure -fg blue
      aRecStop
    }
    if {$bgmParam($seq,nextRec) != 0} {
      if {$v(rec) == 3 && $v(recSeq) < [expr [llength $v(recList)] - 1]} {
        .msg.msg configure -fg blue
        nextRec                             ;# 次の音へ
      } else {
        autoRecStop                         ;# 次の音へ行かず終了
        return
      }
    }
    set v(msg) $bgmParam($seq,msg)
    bgm play -start $bgmParam($seq,pStart) -end $bgmParam($seq,pStop) -command $com
  }
}

#---------------------------------------------------
# 引数で指定した単位の値をサンプル単位に変換する
#
proc val2samp {val from sr} {
  switch $from {
    MSEC -
    msec { ;# msec → サンプル単位に変換
      return [expr int($val / 1000.0 * $sr)]
    }
    SEC  -
    sec  { ;# sec → サンプル単位に変換
      return [expr int($val * $sr)]
    }
    default { ;# そのまま返す(念のため整数化する)
      return int($val);
    }
  }
}

#---------------------------------------------------
# 録音開始
#
proc aRecStart {} {
  global snd v bgmParam t
  if {$v(rec) == 0 || $bgmParam(autoRecStatus) == 0} return   ;# 収録モードでないなら終了
  snd record
  set v(recStatus) 1
  set v(recNow) 1       ;# mac
}

#---------------------------------------------------
# 録音終了
#
proc aRecStop {} {
  global snd v t
  if {$v(rec) == 0} return   ;# 収録モードでないなら終了
  snd stop
  if $v(removeDC) removeDC
;#  Redraw all
  set v(recNow) 0
  .recinfo.b.rec configure -image snackRecord
}

proc autoRecStop {} {
  global bgm bgmParam v t
  set bgmParam(autoRecStatus) 0
  bgm stop   ;# ←本当はこれで止まって欲しいけど止まってくれない。。
  .msg.msg configure -fg black
  set v(msg) [eval format $t(autoRecStop,doneMsg)]
  aRecStop
}

#---------------------------------------------------
# メトロノーム再生/停止の切替
#
proc toggleMetroPlay {} {
  global v metro t

  if $v(playMetroStatus) {
    set v(msg) $t(toggleMetroPlay,stopMsg)
    metro stop  ;# 本当はここで止まって欲しいが止まらない
    set v(playMetroStatus) 0
  } else {
    ;# テンポのチェック
    if {$v(tempo) < 50 || $v(tempo) > 200} {
      tk_messageBox -title $t(toggleMetroPlay,errTitle) -icon error \
        -message [eval format $t(toggleMetroPlay,errMsg)]
      return
    }
    if ![file exists $v(clickWav)] {
      tk_messageBox -title $t(toggleMetroPlay,errTitle) -icon error \
        -message [eval format $t(toggleMetroPlay,errMsg2)]
      return
    }
    metro read $v(clickWav)
    set v(playMetroStatus) 1
    loopPlay metro [expr int($v(tempoMSec) / 1000.0 * [metro cget -rate])]
    set v(msg) $t(toggleMetroPlay,playMsg)
  }
}

#---------------------------------------------------
# wavをループ再生する
#
proc loopPlay {s end} {
  global v t
  if $v(playMetroStatus) {
    $s play -start 0 -end $end -command "loopPlay $s $end"
  }
}

#---------------------------------------------------
# 音叉再生/停止の切替
#
proc toggleOnsaPlay {} {
  global v f0 onsa t

  if $v(playOnsaStatus) {
    set v(msg) $t(toggleOnsaPlay,stopMsg)
    onsa stop
    set v(playOnsaStatus) 0
  } else {
    set v(msg) $t(toggleOnsaPlay,playMsg)
    set v(playOnsaStatus) 1
    playSin [tone2freq $f0(guideTone)$f0(guideOctave)] \
      $f0(guideVol) -1
  }
}

#---------------------------------------------------
# 再生/停止の切替
#
#mac
proc togglePlay {{start 0} {end -1}} {
  global v snd t

  if $v(recNow) return ;# 収録中なら

  if $v(playStatus) {
    snd stop
    set v(playStatus) 0
    set v(msg) $t(togglePlay,stopMsg)
    if {$::tcl_platform(os) == "Darwin"} {
      .recinfo.b.play configure -image snackPlay
    }
  } else {
    set v(msg) $t(togglePlay,playMsg)
    set v(playStatus) 1
    if {$::tcl_platform(os) == "Darwin"} {
      .recinfo.b.play configure -image snackStop
    }
    snd play -start $start -end $end -command {
      set v(playStatus) 0       ;# 再生終了したときの処理
      set v(msg) $t(togglePlay,stopMsg)
      if {$::tcl_platform(os) == "Darwin"} {
        .recinfo.b.play configure -image snackPlay
      }
    }
  }
}

#---------------------------------------------------
# 発声タイミング補正モードON/OFFの切替
#
proc timingAdjMode {} {
  global v t

  if $v(timingAdjMode) {
    tk_messageBox -message [eval format $t(timingAdjMode,startMsg)] \
      -icon info
    set v(msg)  $t(timingAdjMode,on)
  } else {
    tk_messageBox -message [eval format $t(timingAdjMode,doneMsg)] \
      -icon info
    set v(msg)  $t(timingAdjMode,off)
  }
}

#---------------------------------------------------
# 波形表示/非表示の切替
#
proc toggleWave {} {
  global v t

  ;#if [snack::audio active] return
  if $v(showWave) {
      set v(waveh) $v(wavehbackup)
  } else {
      set v(wavehbackup) $v(waveh)
      set v(waveh) 0
  }
  Redraw wave
}

#---------------------------------------------------
# スペクトル表示/非表示の切替
#
proc toggleSpec {} {
  global v t

  ;#if [snack::audio active] return
  if $v(showSpec) {
      set v(spech) $v(spechbackup)
  } else {
      set v(spechbackup) $v(spech)
      set v(spech) 0
  }
  Redraw spec
}

#---------------------------------------------------
# パワー表示/非表示の切替
#
proc togglePow {} {
  global v t

  ;#if [snack::audio active] return
  if $v(showpow) {
      set v(powh) $v(powhbackup)
  } else {
      set v(powhbackup) $v(powh)
      set v(powh) 0
  }
  Redraw pow
}

#---------------------------------------------------
# F0表示/非表示の切替
#
proc toggleF0 {} {
  global v t

  ;#if [snack::audio active] return
  if $v(showf0) {
      set v(f0h) $v(f0hbackup)
  } else {
      set v(f0hbackup) $v(f0h)
      set v(f0h) 0
  }
  Redraw f0
}

#---------------------------------------------------
# UTAU用原音パラメータ表示/非表示の切替
#
proc toggleParam {} {
  global v t

  Redraw param
}

#---------------------------------------------------
# パワーパネルの縦軸表示
# snack の unix/snack.tcl の frequencyAxis を改造
proc powerAxis {canvas x y width height args} {
  # 引数処理
  array set a [list \
    -tags snack_y_axis -font {Helvetica 8} -pwMax 100 \
    -fill black -draw0 0 -pwMin 0]
  array set a $args

  if {$height <= 0} return
  if {$a(-pwMax) <= $a(-pwMin)} return

  ;# ticklist...目盛りの間隔の候補
  set ticklist [list 1 2 5 10 20 50 100 200 500]
  set npt 1   ;# npt...目盛りの値の間隔
  ;# dy...目盛りを描画する間隔(y座標)
  set dy [expr {double($height * $npt) / ($a(-pwMax) - $a(-pwMin))}]

  while {$dy < [font metrics $a(-font) -linespace]} {
    foreach elem $ticklist {
      if {$elem <= $npt} {
        continue
      }
      set npt $elem
      break
    }
    set dy [expr {double($height * $npt) / ($a(-pwMax) - $a(-pwMin))}]
  }
  set hztext dB

  if $a(-draw0) {
    set i0 0
    set j0 0
  } else {
    set i0 $dy
    set j0 1
  }

  if {$a(-pwMin) != 0} {
    set j0 [expr int($a(-pwMin) / $npt) + 1]
  }

  for {set i $i0; set j $j0} {$i < $height} {set i [expr {$i+$dy}]; incr j} {
    set yc [expr {$height + $y - $i}]  ;# 描画するy座標

    if {$npt < 1000} {
      set tm [expr {$j * $npt}]
    } else {
      set tm [expr {$j * $npt / 1000}]
    }
    if {$yc > [expr {8 + $y}]} {
      if {[expr {$yc - [font metrics $a(-font) -ascent]}] > \
          [expr {$y + [font metrics $a(-font) -linespace]}] ||
          [font measure $a(-font) $hztext]  < \
          [expr {$width - 8 - [font measure $a(-font) $tm]}]} {
        $canvas create text [expr {$x +$width - 8}] [expr {$yc-2}]\
          -text $tm -fill $a(-fill)\
          -font $a(-font) -anchor e -tags $a(-tags)
      }
      $canvas create line [expr {$x + $width - 5}] $yc \
        [expr {$x + $width}]\
        $yc -tags $a(-tags) -fill $a(-fill)
    }
  }
  $canvas create text [expr {$x + 2}] [expr {$y + 1}] -text $hztext \
    -font $a(-font) -anchor nw -tags $a(-tags) -fill $a(-fill)

  return $npt
}

#---------------------------------------------------
# F0パネルの縦軸表示
# snack の unix/snack.tcl の frequencyAxis を改造
proc f0Axis {canvas x y width height args} {
  # 引数処理
  array set a [list \
    -tags snack_y_axis -font {Helvetica 8} -f0Max 100 \
    -fill black -draw0 0 -f0Min 0 -unit Hz]
  array set a $args

  if {$height <= 0} return
  if {$a(-f0Max) <= $a(-f0Min)} return

  ;# ticklist...目盛りの間隔の候補
  set ticklist [list 1 2 5 10 20 50 100 200 500]
  set npt 1   ;# npt...目盛りの値の間隔
  ;# dy...目盛りを描画する間隔(y座標)
  set dy [expr {double($height * $npt) / ($a(-f0Max) - $a(-f0Min))}]

  while {$dy < [font metrics $a(-font) -linespace]} {
    foreach elem $ticklist {
      if {$elem <= $npt} {
        continue
      }
      set npt $elem
      break
    }
    set dy [expr {double($height * $npt) / ($a(-f0Max) - $a(-f0Min))}]
  }
  set hztext $a(-unit)
  if {$hztext == "semitone"} {set hztext st} ;# 表示を短縮

  if $a(-draw0) {
    set i0 0
    set j0 0
  } else {
    set i0 $dy
    set j0 1
  }

  if {$a(-f0Min) != 0} {
    set j0 [expr int($a(-f0Min) / $npt) + 1]
  }

  ;# j=描画する目盛りの番号, i=描画座標に関する変数
  for {set i $i0; set j $j0} {$i < $height} {set i [expr {$i+$dy}]; incr j} {
    set yc [expr {$height + $y - $i}]  ;# 描画するy座標

    if {$npt < 1000} {
      set tm [expr {$j * $npt}]
    } else {
      set tm [expr {$j * $npt / 1000}]
    }
    if {$yc > [expr {8 + $y}]} {
      if {[expr {$yc - [font metrics $a(-font) -ascent]}] > \
          [expr {$y + [font metrics $a(-font) -linespace]}] ||
          [font measure $a(-font) $hztext]  < \
          [expr {$width - 8 - [font measure $a(-font) $tm]}]} {
        $canvas create text [expr {$x +$width - 8}] [expr {$yc-2}]\
          -text $tm -fill $a(-fill)\
          -font $a(-font) -anchor e -tags $a(-tags)
      }
      $canvas create line [expr {$x + $width - 5}] $yc \
        [expr {$x + $width}]\
        $yc -tags $a(-tags) -fill $a(-fill)
    }
  }
  $canvas create text [expr {$x + 2}] [expr {$y + 1}] -text $hztext \
    -font $a(-font) -anchor nw -tags $a(-tags) -fill $a(-fill)

  return $npt
}

#---------------------------------------------------
# 色選択
#
proc chooseColor {w key initcolor} {
  global v t
  set ctmp [tk_chooseColor -initialcolor $initcolor -title $t(chooseColor,title)]
  if {$ctmp != ""} {
    set v($key) $ctmp
    $w configure -bg $v($key)
  }
}

#---------------------------------------------------
# 音名に対応する周波数を返す
#
proc tone2freq {tone} {
  global v t
  for {set i 0} {$i < [llength $v(sinNote)]} {incr i} {
    if {$tone == [lindex $v(sinNote) $i]} break
  }
  return [lindex $v(sinScale) $i]
}

#---------------------------------------------------
# 波形色設定
#
proc setColor {w key msg} {
  global v t

  set ic $v($key)
  pack [frame $w.$key] -anchor nw
  label  $w.$key.l  -text $msg -width 20 -anchor nw
  label  $w.$key.l2 -textvar v($key) -width 7 -anchor nw -bg $v($key)
  button $w.$key.b  -text $t(setColor,selColor) -command "chooseColor $w.$key.l2 $key $ic"
  pack $w.$key.l $w.$key.l2 $w.$key.b -side left
}

#---------------------------------------------------
# 1Hzに対する周波数比をセミトーンにする
#
proc hz2semitone {hz} {
  return [expr log($hz) / log(2) * 12.0]
}


#---------------------------------------------------
# 項目名：[エントリー]のフレームを作って配置する(power用)
#
proc packEntryPower {wname text key} {
  global power t
  pack [frame $wname] -anchor w
  label $wname.lfl -text $text -width 20 -anchor w
  entry $wname.efl -textvar power($key) -wi 6
  pack $wname.lfl $wname.efl -side left
}

#---------------------------------------------------
# 項目名：[エントリー]のフレームを作って配置する(f0用)
#
proc packEntryF0 {wname text key} {
  global f0 t
  pack [frame $wname] -anchor w
  label $wname.lfl -text $text -width 20 -anchor w
  entry $wname.efl -textvar f0($key) -wi 6
  pack $wname.lfl $wname.efl -side left
}

#---------------------------------------------------
# 音名の選択メニューをpackしたフレームを生成
#
proc packToneList {w text toneKey octaveKey freqKey width vol} {
  global f0 v t
  pack [frame $w] -fill x
  # 項目名ラベル
  label $w.l -text $text -width $width -anchor w
  # 音名選択
  eval tk_optionMenu $w.t f0($toneKey) $v(toneList)
  # オクターブ選択
  set ss {}
  for {set i $v(sinScaleMin)} {$i <= $v(sinScaleMax)} {incr i} {
    lappend ss $i
  }
  eval tk_optionMenu $w.o f0($octaveKey) $ss
  # 試聴ボタン
  button $w.play -text $t(packToneList,play) -bitmap snackPlay -command \
    "playSin \[tone2freq \$f0($toneKey)\$f0($octaveKey)\] \$f0($vol) \$v(sampleRate)"
  button $w.togglePlay -text $t(packToneList,repeat) -command {
    toggleOnsaPlay
  }
  # 音に対応する周波数を表示するラベル
  label $w.$freqKey -textvar f0($freqKey) -width 3 -anchor e
  label $w.unit -text "Hz"
  pack $w.l $w.t $w.o $w.play $w.togglePlay $w.$freqKey $w.unit -side left
}

#---------------------------------------------------
#   入出力デバイスやバッファサイズを初期化する
#
proc audioSettings {} {
  global dev snd bgm t

  set dev(in)   [encoding convertfrom [lindex [snack::audio inputDevices]  0]]
  set dev(out)  [encoding convertfrom [lindex [snack::audio outputDevices] 0]]
  set dev(ingain)    [snack::audio record_gain]
  set dev(outgain)   [snack::audio play_gain]
  set dev(latency)   [snack::audio playLatency]
  set dev(sndBuffer) [snd cget -buffersize]
  set dev(bgmBuffer) [bgm cget -buffersize]
  # snack::audio selectInput $dev(in) ;# 漢字コード未対応
}

#---------------------------------------------------
#   入出力デバイスの設定窓の値をデバイスに反映させる
#
proc setIODevice {} {
  global dev snd bgm t
  ;# dev(in),dev(out)にはメニュー表示のため漢字コードをsjis→utf-8に
  ;# 変換した文字列を入れている。デバイス設定時には元の漢字コード文字列で
  ;# 指定しないとエラーになる様子なので以下のようなコードで対応している
  foreach dname [snack::audio inputDevices] {
    if {$dev(in) == [encoding convertfrom $dname]} {
      snack::audio selectInput  $dname
      break
    }
  }
  foreach dname [snack::audio outputDevices] {
    if {$dev(out) == [encoding convertfrom $dname]} {
      snack::audio selectOutput $dname
      break
    }
  }
  snack::audio record_gain  $dev(ingain)
  snack::audio play_gain    $dev(outgain)
  snack::audio playLatency  $dev(latency)
  snd configure -buffersize $dev(sndBuffer)
  bgm configure -buffersize $dev(bgmBuffer)
}

#---------------------------------------------------
#   現在の設定を保存する
#
proc saveSettings {} {
  global startup bgmParam v f0 power startup dev uttTiming genParam estimate t

  set fn [my_getSaveFile -initialfile $startup(initFile) \
            -title $t(saveSettings,title) -defaultextension "tcl" ]
  if {$fn == ""} return

  set aList {bgmParam v f0 power startup dev uttTiming genParam estimate}

  set fp [open $fn w]   ;# 保存ファイルを開く

  foreach aName $aList {
    set sList [array get $aName]
    foreach {key value} $sList {
      if {$aName == "v" && $key != "paramChanged" 
                        && $key != "msg"
                        && $key != "version"} {
        puts $fp [format "set %s(%s)\t\t{%s}" $aName $key $value]
      }
    }
  }
  close $fp
}

#---------------------------------------------------
#   指定した窓が起動済みかチェック。起動済みならフォーカスする。
#
proc isExist {w} {
  if [winfo exists $w] {
    raise $w
    focus $w
    return 1
  } else {
    return 0
  }
}

#---------------------------------------------------
#   入出力デバイスやバッファサイズを設定する窓
#
proc ioSettings {} {
  global ioswindow dev dev_bk snd bgm t

  if [isExist $ioswindow] return ;# 二重起動を防止
  toplevel $ioswindow
  wm title $ioswindow $t(ioSettings,title)
  bind $ioswindow <Escape> {destroy $ioswindow}

  ;# ゲイン、レイテンシの最新状況を取得
  set dev(ingain)  [snack::audio record_gain]
  set dev(outgain) [snack::audio play_gain]
  set dev(latency) [snack::audio playLatency]
  set dev(sndBuffer) [snd cget -buffersize]
  set dev(bgmBuffer) [bgm cget -buffersize]

  array set dev_bk [array get dev]     ;# パラメータバックアップ

  ;# 入力デバイスの選択
  set devList {}
  foreach d [snack::audio inputDevices] {
    set d [encoding convertfrom $d]
    lappend devList "$d"
    # if {[string length $d] == [string bytelength $d]} {  ;# 英語デバイスのみ登録
    #  lappend devList "$d"
    # }
  }
  set f1 [frame $ioswindow.f1]
  label $f1.l -text $t(ioSettings,inDev) -width 12 -anchor w
  eval tk_optionMenu $f1.in dev(in) $devList
  pack $f1.l $f1.in -side left
  pack $f1 -anchor w

  ;# 出力デバイスの選択
  set devList {}
  foreach d [snack::audio outputDevices] {
    set d [encoding convertfrom $d]
    lappend devList "$d"
    #if {[string length $d] == [string bytelength $d]} {  ;# 英語デバイスのみ登録
    #  lappend devList "$d"
    #}
  }
  set f2 [frame $ioswindow.f2]
  label $f2.l -text $t(ioSettings,outDev) -width 12 -anchor w
  eval tk_optionMenu $f2.out dev(out) $devList
  pack $f2.l $f2.out -side left
  pack $f2 -anchor w

  ;# 入力ゲインの指定
  set f3 [frame $ioswindow.f3]
  label $f3.l -text $t(ioSettings,inGain) -width 28 -anchor w
  entry $f3.e -textvar dev(ingain) -wi 6
  scale $f3.s -variable dev(ingain) -orient horiz \
    -from 0 -to 100 -res 1 -showvalue 0
  pack $f3.l $f3.e $f3.s -side left
  pack $f3 -anchor w

  ;# 出力ゲインの指定
  set f4 [frame $ioswindow.f4]
  label $f4.l -text $t(ioSettings,outGain) -width 28 -anchor w
  entry $f4.e -textvar dev(outgain) -wi 6
  scale $f4.s -variable dev(outgain) -orient horiz \
    -from 0 -to 100 -res 1 -showvalue 0
  pack $f4.l $f4.e $f4.s -side left
  pack $f4 -anchor w

  ;# レイテンシの指定
  set f5 [frame $ioswindow.f5]
  label $f5.l -text $t(ioSettings,latency) -width 28 -anchor w
  entry $f5.e -textvar dev(latency) -wi 6
  label $f5.u -text "(msec)"
  pack $f5.l $f5.e $f5.u -side left
  pack $f5 -anchor w

  ;# 収録音のバッファサイズの指定
  set f6 [frame $ioswindow.f6]
  label $f6.l -text $t(ioSettings,sndBuffer) -width 28 -anchor w
  entry $f6.e -textvar dev(sndBuffer) -wi 6
  label $f6.u -text "(sample)"
  pack $f6.l $f6.e $f6.u -side left
  pack $f6 -anchor w

  ;# ガイドBGMのバッファサイズの指定
  set f7 [frame $ioswindow.f7]
  label $f7.l -text $t(ioSettings,bgmBuffer) -width 28 -anchor w
  entry $f7.e -textvar dev(bgmBuffer) -wi 6
  label $f7.u -text "(sample)"
  pack $f7.l $f7.e $f7.u -side left
  pack $f7 -anchor w

  ;# 決定ボタン
  set fb [frame $ioswindow.fb]
  button $fb.ok -text $t(.confm.ok) -wi 6 -command {
    setIODevice
    destroy $ioswindow
  }
  button $fb.ap -text $t(.confm.apply) -wi 6 -command {
    setIODevice
    array set dev_bk [array get dev]     ;# パラメータバックアップ
  }
  button $fb.cn -text $t(.confm.c) -wi 6 -command {
    array set dev [array get dev_bk]     ;# パラメータを以前の状態に戻す
    setIODevice
    destroy $ioswindow
  }
  pack $fb.ok $fb.ap $fb.cn -side left
  pack $fb -anchor w

  ;# 説明文
  set fm [frame $ioswindow.fm]
  label $fm.lm0  -fg red -text $t(ioSettings,comment0)
  label $fm.lm0b -fg red -text $t(ioSettings,comment0b)
  label $fm.lm1  -fg red -text $t(ioSettings,comment1)
  label $fm.lm2  -fg red -text $t(ioSettings,comment2)
  pack $fm.lm0 $fm.lm0b $fm.lm1 $fm.lm2 -anchor w -side top
  pack $fm -anchor w

  raise $ioswindow
  focus $ioswindow
}

#---------------------------------------------------
# プログレスバーを初期化して表示する
#
proc initProgressWindow {{title "now processing..."}} {
  global prgWindow v
  if {$::tcl_platform(os) == "Darwin"} {
    if [isExist $prgWindow] {  ;# macではprogressbarを二回実行すると落ちたので
      set v(progress) 0 
      wm deiconify $prgWindow
      return
    }
  } else {
    if [isExist $prgWindow] return
  }

  toplevel $prgWindow
  wm title $prgWindow $title
  if {$::tcl_platform(os) != "Darwin"} {
    wm attributes $prgWindow -toolwindow 1   ;# mac
    wm attributes $prgWindow -topmost 1
  }
  set topg [split [wm geometry .] "x+"]
  set x [expr [lindex $topg 2] + [lindex $topg 0] / 2 - 100]
  set y [expr [lindex $topg 3] + [lindex $topg 1] / 2 - 5]
  wm geometry $prgWindow "+$x+$y"

  set v(progress) 0 
  ttk::progressbar $prgWindow.p -length 200 -variable v(progress) -mode determinate
  pack $prgWindow.p

  raise .progress
  focus .progress
}

#---------------------------------------------------
# プログレスバーを更新する。進捗状況は$progress(0～100)で指定する)
#
proc updateProgressWindow {progress} {
  global v prgWindow

  set v(progress) $progress
  focus $prgWindow
  raise $prgWindow
  update
}

#---------------------------------------------------
# プログレスバーを消去する
#
proc deleteProgressWindow {} {
  global prgWindow
  if {$::tcl_platform(os) == "Darwin"} {
    wm withdraw $prgWindow
  } else {
    destroy $prgWindow
  }
}

#---------------------------------------------------
# 原音パラメータを保存する
# return: 1=保存した。0=保存しなかった。
#
proc saveParamFile {} {
  global paramU paramUsize v t
 
  ;#if {$v(paramChanged) == 0} {
  ;#  set v(msg) "パラメータが変更されていないので保存しませんでした"
  ;#  tk_dialog .confm "Warning" \
  ;#    "パラメータが変更されていないので保存しませんでした" \
  ;#    warning 0 OK
  ;#  return 0
  ;#}
  set fn [my_getSaveFile -initialfile $v(paramFile) -initialdir $v(saveDir) \
            -title $t(saveParamFile,selFile) -defaultextension "ini" ]
  if {$fn == ""} {return 0}

  #koko, mac版はoto_ini.txtで、一行目が「#Charset:UTF8」、以降はutf-8で原音設定。
  #ここでは.ini、.txtのいずれかを指定したら他方も保存するように作ろうかと
  #if {$::tcl_platform(os) != "Darwin"} {
  #  if {[file extension $fn] == ".ini"} {
  #    set fnTxt [file rootname $fn]_ini.txt
  #  } elseif {[file extension $fn] == ".txt"} {
  #    set 
  #  }
  #}

  set v(msg) $t(saveParamFile,startMsg)

  ;# 保存ファイルを開く
  if [catch {open $fn w} fp] { 
    tk_messageBox -message "error: can not open $fn" \
      -title $t(.confm.fioErr) -icon warning
    return
  }
  if {$::tcl_platform(os) != "Darwin"} {
    set v(paramFile) $fn  ;# macだとパス解析に失敗して次回のmy_getSaveFileで不正確なフォルダが初期指定されたので
  }

  for {set i 1} {$i < $paramUsize} {incr i} {
    if {[array names paramU "$i,0"] != ""} {
      set name $paramU($i,0).wav
      set S 0; set O 0; set P 0; set C 0; set E 0; set A "";
      if {[array names paramU "$i,1"] != ""} { set S $paramU($i,1) }
      if {[array names paramU "$i,2"] != ""} { set O $paramU($i,2) }
      if {[array names paramU "$i,3"] != ""} { set P $paramU($i,3) }
      if {[array names paramU "$i,4"] != ""} { set C $paramU($i,4) }
      if {[array names paramU "$i,5"] != ""} { set E $paramU($i,5) }
      if {[array names paramU "$i,6"] != ""} { set A $paramU($i,6) }
      puts $fp $name=$A,$S,$C,$E,$P,$O    ;# ファイルへ書き出し
    }
  }
  close $fp        ;# ファイルを閉じる

  if {$::tcl_platform(os) == "Darwin"} {
    global nkf 
    exec -- $nkf -s --in-place $fn      ;# 漢字コードをsjisに変換
  }
  # saveCacheFile    ;# キャッシュ作成。これはoto.ini保存より先にやってはいけない。
  # set v(paramChanged) 0
  # setEPWTitle
  set v(msg) [eval format $t(saveParamFile,doneMsg)]
  return 1
}



#---------------------------------------------------
# 詳細設定
#
proc settings {} {
  global swindow v power f0 v_bk power_bk f0_bk snd t
    # ↑*_bkは大域変数にしないとキャンセル時にバックアップ復帰できなかった

  ;# 二重起動を防止
  if [isExist $swindow] {
    return
  }
  toplevel $swindow
  wm title $swindow $t(settings,title)
  wm resizable $swindow 0 0
  bind $swindow <Escape> {destroy $swindow}

  array set v_bk     [array get v]     ;# パラメータバックアップ
  array set power_bk [array get power] ;# パラメータバックアップ
  array set f0_bk    [array get f0]    ;# パラメータバックアップ

  ;# 1カラム目のフレーム
  set frame1 [frame $swindow.l]
  pack $frame1 -side left -anchor n -fill y -padx 2 -pady 2

  ;#---------------------------
  ;# 波形
  set lf1 [labelframe $frame1.lf1 -text $t(settings,wave) \
    -relief groove -padx 5 -pady 5]
  pack $lf1 -anchor w -fill x

  ;# 波形色の設定
  set cw [frame $lf1.f4w]
  setColor $cw "wavColor" $t(settings,waveColor)
  pack $cw -anchor nw

  ;# サンプリング周波数の設定
  pack [frame $lf1.f20] -anchor w
  label $lf1.f20.l -text $t(settings,sampleRate) -width 20 -anchor w
  entry $lf1.f20.e -textvar v(sampleRate) -wi 6
  pack $lf1.f20.l $lf1.f20.e  -side left

  ;#---------------------------
  ;# スペクトルパラメータ
  set lf2 [labelframe $frame1.lf2 -text $t(settings,spec) \
    -relief groove -padx 5 -pady 5]
  pack $lf2 -anchor w -fill x

  ;# スペクトルの配色
  pack [frame $lf2.f45] -anchor w
  label $lf2.f45.l -text $t(settings,specColor) -width 20 -anchor w
  tk_optionMenu $lf2.f45.cm v(cmap) grey color1 color2
  pack $lf2.f45.l $lf2.f45.cm -side left

  ;# スペクトル周波数の最高値
  pack [frame $lf2.f20] -anchor w
  label $lf2.f20.l -text $t(settings,maxFreq) -width 20 -anchor w
  entry $lf2.f20.e -textvar v(topfr) -wi 6
  scale $lf2.f20.s -variable v(topfr) -orient horiz \
    -from 0 -to [expr $v(sampleRate)/2] -showvalue 0
  pack $lf2.f20.l $lf2.f20.e $lf2.f20.s -side left

  ;# 明るさ
  pack [frame $lf2.f30] -anchor w
  label $lf2.f30.l -text $t(settings,brightness) -width 20 -anchor w
  entry $lf2.f30.e -textvar v(brightness) -wi 6
  scale $lf2.f30.s -variable v(brightness) -orient horiz \
    -from -100 -to 100 -res 0.1 -showvalue 0
  pack $lf2.f30.l $lf2.f30.e $lf2.f30.s -side left

  ;# コントラスト
  pack [frame $lf2.f31] -anchor w
  label $lf2.f31.l -text $t(settings,contrast) -width 20 -anchor w
  entry $lf2.f31.e -textvar v(contrast) -wi 6
  scale $lf2.f31.s -variable v(contrast) -orient horiz \
    -from -100 -to 100 -res 0.1 -showvalue 0
  pack $lf2.f31.l $lf2.f31.e $lf2.f31.s -side left

  ;# FFT長(必ず2のべき乗にすること)
  pack [frame $lf2.f32] -anchor w
  label $lf2.f32.l -text $t(settings,fftLength) -width 20 -anchor w
  tk_optionMenu $lf2.f32.om v(fftlen) 8 16 32 64 128 256 512 1024 2048 4096
  pack $lf2.f32.l $lf2.f32.om -side left

  ;# 窓長(必ずFFT長以下にすること)
  pack [frame $lf2.f33] -anchor w
  label $lf2.f33.l -text $t(settings,winLength) -width 20 -anchor w
  entry $lf2.f33.e -textvar v(winlen) -wi 6
  scale $lf2.f33.s -variable v(winlen) -orient horiz \
    -from 8 -to 4096 -showvalue 0
  pack $lf2.f33.l $lf2.f33.e $lf2.f33.s -side left

  ;# プリエンファシス
  pack [frame $lf2.f34] -anchor w
  label $lf2.f34.l -text $t(settings,fftPreemph) -width 20 -anchor w
  entry $lf2.f34.e -textvar v(preemph) -wi 6
  pack $lf2.f34.l $lf2.f34.e -side left

  ;# 窓の選択
  pack [frame $lf2.f35] -anchor w
  label $lf2.f35.lwn -text $t(settings,fftWinKind) -width 20 -anchor w
  tk_optionMenu $lf2.f35.mwn v(window) \
    Hamming Hanning Bartlett Blackman Rectangle
  pack $lf2.f35.lwn $lf2.f35.mwn -side left

  ;#---------------------------
  ;# パワーの設定
  set lf3 [labelframe $frame1.lf3 -text $t(settings,pow) \
    -relief groove -padx 5 -pady 5]
  pack $lf3 -anchor w -fill x

  ;# パワー色の設定
  set cp [frame $lf3.f4p]
  setColor $cp "powcolor" $t(settings,powColor)
  pack $cp -anchor nw

  ;# パワー抽出刻みの設定
  packEntryPower $lf3.ffl $t(settings,powLength) frameLength

  ;# プリエンファシスの設定
  packEntryPower $lf3.fem $t(settings,powPreemph) preemphasis

  ;# 窓長の設定
  packEntryPower $lf3.fwl $t(settings,winLength) windowLength

  ;# 窓の選択
  pack [frame $lf3.fwn] -anchor w
  label $lf3.fwn.lwn -text $t(settings,powWinKind) -width 20 -anchor w
  tk_optionMenu $lf3.fwn.mwn power(window) \
    Hamming Hanning Bartlett Blackman Rectangle
  pack $lf3.fwn.lwn $lf3.fwn.mwn -side left

  ;#---------------------------
  ;#---------------------------
  ;# 2カラム目のフレーム
  set frame2 [frame $swindow.r]
  pack $frame2 -side left -anchor n -fill both -expand true -padx 2 -pady 2

  ;#---------------------------
  ;# F0の設定
  set lf4 [labelframe $frame2.lf4 -text $t(settings,f0) \
    -relief groove -padx 5 -pady 5]
  pack $lf4 -anchor w -fill x

  ;# F0色の設定
  set cf [frame $lf4.f4f]
  setColor $cf "f0color" $t(settings,f0Color)
  pack $cf -anchor nw

  ;# 抽出アルゴリズムの選択
  pack [frame $lf4.p1] -anchor w
  label $lf4.p1.l -text $t(settings,f0Argo) -width 20 -anchor w
  tk_optionMenu $lf4.p1.mt f0(method) ESPS AMDF
  pack $lf4.p1.l $lf4.p1.mt -side left

  ;# entry型の設定いろいろ
  packEntryF0 $lf4.p2 $t(settings,f0Length)    frameLength
  packEntryF0 $lf4.p3 $t(settings,f0WinLength) windowLength
  packEntryF0 $lf4.p4 $t(settings,f0Max)       max
  packEntryF0 $lf4.p5 $t(settings,f0Min)       min

  ;# 表示単位の選択
  pack [frame $lf4.p6] -anchor w
  label $lf4.p6.l -text $t(settings,f0Unit) -width 20 -anchor w
  tk_optionMenu $lf4.p6.mt f0(unit) Hz semitone
  pack $lf4.p6.l $lf4.p6.mt -side left

  ;# グラフ範囲の設定
  checkbutton $lf4.p7cb -text $t(settings,f0FixRange) \
    -variable f0(fixShowRange) -onvalue 1 -offvalue 0 -anchor w
  pack [labelframe $lf4.p7 -labelwidget $lf4.p7cb \
    -relief ridge -padx 5 -pady 5] -anchor w -fill x
  packToneList $lf4.p7.tl1 $t(settings,f0FixRange,h) \
    showMaxTone showMaxOctave showMaxTmp 10 checkVol
  packToneList $lf4.p7.tl2 $t(settings,f0FixRange,l) \
    showMinTone showMinOctave showMinTmp 10 checkVol

  ;# 各音の線を表示
  checkbutton  $lf4.p8cb -text $t(settings,grid) \
    -variable f0(showToneLine) -onvalue 1 -offvalue 0 -anchor w
  pack [labelframe $lf4.p8 -labelwidget $lf4.p8cb \
    -relief ridge -padx 5 -pady 5] -anchor w -fill x
  setColor $lf4.p8 "toneLineColor" $t(settings,gridColor)

  ;# ターゲット音の線を表示
  checkbutton $lf4.p9cb -text $t(settings,target) \
    -variable f0(showTgtLine) -onvalue 1 -offvalue 0 -anchor w
  pack [labelframe $lf4.p9 -labelwidget $lf4.p9cb \
    -relief ridge -padx 5 -pady 5] -anchor w -fill x
  packToneList $lf4.p9.tl $t(settings,targetTone) \
    tgtTone tgtOctave tgtFreqTmp 10 checkVol
  setColor $lf4.p9 "tgtf0color" $t(settings,targetColor)
  label  $lf4.p9.al -text $t(settings,autoSetting) -anchor nw
  button $lf4.p9.ab -text $t(.confm.run) -command autoF0Settings
  pack $lf4.p9.al $lf4.p9.ab -side left

  ;# 各音名に対応する周波数を自動計算して表示する(非常に汚いやり方)
  ;# 音名orオクターブに変化があれば周波数計算を行う
  ;# 周波数をf0(tgtFreq)などでなくf0(tgtFreqTmp)などに入れるのは、
  ;#「OK」or「適用」ボタンを押すまで値変更を反映させないため。
  set f0(showMaxTmp) [tone2freq "$f0(showMaxTone)$f0(showMaxOctave)"]
  set f0(showMinTmp) [tone2freq "$f0(showMinTone)$f0(showMinOctave)"]
  set f0(tgtFreqTmp) [tone2freq "$f0(tgtTone)$f0(tgtOctave)"]
  trace variable f0 w calcFreq
  proc calcFreq {var elm mode} {
    global f0 t
    switch $elm {
      "showMaxTone" -
      "showMaxOctave" {
        set f0(showMaxTmp) [tone2freq "$f0(showMaxTone)$f0(showMaxOctave)"]
      }
      "showMinTone" -
      "showMinOctave" {
        set f0(showMinTmp) [tone2freq "$f0(showMinTone)$f0(showMinOctave)"]
      }
      "tgtTone" -
      "tgtOctave" {
        set f0(tgtFreqTmp) [tone2freq "$f0(tgtTone)$f0(tgtOctave)"]
      }
    }
  }
  bind $swindow <Destroy> { trace vdelete f0 w calcFreq }

  ;#---------------------------
  ;# OK, Apply, キャンセルボタン
  pack [frame $frame2.f] -anchor e -side bottom -padx 2 -pady 2
  button $frame2.f.exit -text $t(.confm.c) -command {
    array set v     [array get v_bk]     ;# パラメータを以前の状態に戻す
    array set power [array get power_bk] ;# パラメータを以前の状態に戻す
    array set f0    [array get f0_bk]    ;# パラメータを以前の状態に戻す
    Redraw all
    destroy $swindow
  }
  button $frame2.f.app -text $t(.confm.apply) -command {
    ;# サンプリング周波数の変更
    if {$v(sampleRate) != $v_bk(sampleRate)} {
      snd configure -rate $v(sampleRate)
    }
    ;# ターゲット音の周波数を求める
    set f0(tgtFreq) [tone2freq "$f0(tgtTone)$f0(tgtOctave)"]
    ;# F0表示範囲周波数を求める
    if $f0(fixShowRange) {
      set f0(showMin) [tone2freq "$f0(showMinTone)$f0(showMinOctave)"]
      set f0(showMax) [tone2freq "$f0(showMaxTone)$f0(showMaxOctave)"]
    }
    Redraw all
    ;# パラメータバックアップの更新
    array set v_bk     [array get v]     ;# パラメータバックアップ
    array set power_bk [array get power] ;# パラメータバックアップ
    array set f0_bk    [array get f0]    ;# パラメータバックアップ
  }
  button $frame2.f.ok -text $t(.confm.ok) -wi 6 -command {
    ;# サンプリング周波数の変更
    if {$v(sampleRate) != $v_bk(sampleRate)} {
      snd configure -rate $v(sampleRate)
    }
    ;# ターゲット音の周波数を求める
    set f0(tgtFreq) [tone2freq "$f0(tgtTone)$f0(tgtOctave)"]
    ;# F0表示範囲周波数を求める
    if $f0(fixShowRange) {
      set f0(showMin) [tone2freq "$f0(showMinTone)$f0(showMinOctave)"]
      set f0(showMax) [tone2freq "$f0(showMaxTone)$f0(showMaxOctave)"]
    }
    Redraw all
    destroy $swindow
  }
  pack $frame2.f.exit $frame2.f.app $frame2.f.ok -side right
}

#---------------------------------------------------
# F0ターゲットに合わせて他の設定値を自動設定する
#
proc autoF0Settings {} {
  global v f0

  set tgtFreq [tone2freq "$f0(tgtTone)$f0(tgtOctave)"]

  set f0(max) [expr int($tgtFreq + 200)]
  if {$tgtFreq >= 260} {
    set f0(min) [expr int($tgtFreq - 200)]
  } else {
    set f0(min) 60
  }

  set f0(fixShowRange) 1
  set ret [calcTone $f0(tgtTone) $f0(tgtOctave) 2]
  set f0(showMaxTone)   [lindex $ret 0]
  set f0(showMaxOctave) [lindex $ret 1]
  set ret [calcTone $f0(tgtTone) $f0(tgtOctave) -2]
  set f0(showMinTone)   [lindex $ret 0]
  set f0(showMinOctave) [lindex $ret 1]
}

#---------------------------------------------------
# tone-octaveをadd度？上げたときのトーンとオクターブのリストを返す
#
proc calcTone {tone octave add} {
  global v
  for {set i 0} {$i < [llength $v(toneList)]} {incr i} {
    if {$tone == [lindex $v(toneList) $i]} break
  }
  set seq [expr $i + $add]
  while {$seq >= [llength $v(toneList)]} {
    incr octave
    incr seq -12
  }
  while {$seq < 0} {
    incr octave -1
    incr seq +12
  }
  set ret {}
  lappend ret [lindex $v(toneList) $seq]
  lappend ret $octave
  return $ret
}


#---------------------------------------------------
# キャンバス再描画
#
proc Redraw {opt} {
  global v c cYaxis snd power f0 rec t

  # 描画中は他の操作ができないようにする
  # grab set $c
  # ↑これがあると窓の隅をドラッグしてサイズ変更できなくなるのでボツ

  ;# キャンバス上のものを削除して高さを再調整する
  set v(cHeight) [expr $v(waveh) + $v(spech) + $v(powh) + $v(f0h) + $v(timeh)]
  if {$v(cHeight) < [winfo height $rec]} {set v(cHeight) [winfo height $rec]}
  $c delete obj
  $c delete axis
  $c configure -height $v(cHeight) -width $v(cWidth)
  $c create line 0 0 $v(cWidth) 0 -tags axis -fill $v(fg)

  $cYaxis delete axis
  $cYaxis configure -height $v(cHeight)
  $cYaxis create line 0 2 $v(yaxisw) 2 -tags axis -fill $v(fg)
  $cYaxis create line $v(yaxisw) 0 $v(yaxisw) $v(cHeight) -tags axis -fill $v(fg)
  set sndLen [snd length -unit SECONDS]
  if {$sndLen > 0} {
    set v(wavepps) [expr double($v(cWidth)) / $sndLen]
  } else {
    set v(wavepps) [expr double(1.0 / $v(cWidth))]
  }

  ;# 波形表示
  if $v(showWave) {
    $c create waveform 0 0 -sound snd -height $v(waveh) -width $v(cWidth) \
      -tags [list obj wave] -debug $::debug -fill $v(wavColor)
    $c lower wave
    $cYaxis create text $v(yaxisw) 4 -text [snd max] \
      -font $v(sfont) -anchor ne -tags axis -fill $v(fg)
    $cYaxis create text $v(yaxisw) $v(waveh) -text [snd min] \
      -font $v(sfont) -anchor se -tags axis -fill $v(fg)
    set ylow $v(waveh)
    $c create line 0 $ylow $v(cWidth) $ylow -tags axis -fill $v(fg)
    set yAxisLow [expr $v(waveh) + 2]
    $cYaxis create line 0 $yAxisLow $v(yaxisw) $yAxisLow -tags axis -fill $v(fg)
#    $c create line $v(yaxisw) 0 $v(yaxisw) $v(waveh) -tags axis -fill $v(fg)
  }

  ;# スペクトル表示
  if $v(showSpec) {
    if {$v(winlen) > $v(fftlen)} {
      set v(winlen) $v(fftlen)
    }
    $c create spectrogram 0 $v(waveh) -sound snd -height $v(spech) \
      -width $v(cWidth) -tags [list obj spec] -debug $::debug \
      -fftlength $v(fftlen) -winlength $v(winlen) -windowtype $v(window) \
      -topfr $v(topfr) -contrast $v(contrast) -brightness $v(brightness) \
      -preemph $v(preemph) -colormap $v($v(cmap)) -topfrequency $v(topfr)
    $c lower spec
    snack::frequencyAxis $cYaxis 0 $v(waveh) $v(yaxisw) $v(spech) \
          -topfr $v(topfr) -tags axis -font $v(sfont)
    set ylow [expr $v(spech) + $v(waveh)]
    $c create line 0 $ylow $v(cWidth) $ylow -tags axis
    set yAxisLow [expr $ylow + 2]
    $cYaxis create line 0 $yAxisLow $v(yaxisw) $yAxisLow -tags axis -fill $v(fg)
#    $c create line $v(yaxisw) $v(waveh) $v(yaxisw) $ylow -tags axis -fill $v(fg)
  }

  ;# パワー表示
  if $v(showpow) {
    ;# パワーを抽出
    set ytop [expr $v(waveh) + $v(spech)]
    set ylow [expr $ytop     + $v(powh)]
    if {$opt == "all" || $opt == "pow"} {
      if {$power(recLab) != $v(recLab) || $power(typeLab) != $v(typeLab)} {
        set power(power) [snd power -framelength $power(frameLength) \
        -windowtype $power(window) -preemphasisfactor $power(preemphasis) \
        -windowlength [expr int($power(windowLength) * $v(sampleRate))] \
        -start 0 -end -1]
      }
      if {[llength $power(power)] > 0} {
        # パワーの最大値・最小値を求める
        if {$opt == "all" || $opt == "pow"} {
          set power(powerMax) [lindex $power(power) 0]
          set power(powerMin) [lindex $power(power) 0]
          for {set i 1} {$i < [llength $power(power)]} {incr i} {
            if {$power(powerMax) < [lindex $power(power) $i]} {
              set power(powerMax) [lindex $power(power) $i]
            }
            if {$power(powerMin) > [lindex $power(power) $i]} {
              set power(powerMin) [lindex $power(power) $i]
            }
          }
        }
      }
    }

    if {[llength $power(power)] > 0} {
      # ppd= 1dBあたりのピクセル数。4は上下各2ピクセルのマージン
      if {[expr $power(powerMax) - $power(powerMin)] > 0} {
        set ppd [expr double($v(powh)) / ($power(powerMax) - $power(powerMin))]
      } else {
        set ppd 0
      }

      set coord {} ;# パワー曲線を引く座標(x,y)列
      for {set i 0} {$i < [llength $power(power)]} {incr i} {
        lappend coord \
          [expr $i * $power(frameLength) * $v(wavepps)] \
          [expr $ylow - ([lindex $power(power) $i] - $power(powerMin)) * $ppd]
      }
      set pwtags {obj pow}
      eval {$c create line} $coord -tags {$pwtags} -fill $v(powcolor)
      powerAxis $cYaxis 0 $ytop $v(yaxisw) $v(powh) \
        -pwMax $power(powerMax) -tags axis -fill $v(fg) \
        -font $v(sfont) -pwMin $power(powerMin)
    }
    $c create line 0 $ylow $v(cWidth) $ylow -tags axis
#    $c create line $v(yaxisw) $ytop $v(yaxisw) $ylow -tags axis -fill $v(fg)
    set yAxisLow [expr $ylow + 2]
    $cYaxis create line 0 $yAxisLow $v(yaxisw) $yAxisLow -tags axis -fill $v(fg)

    ;# パワーを抽出したならそのFIDを記録する
    if {$power(fid) != $v(recLab)} {
      set power(fid) $v(recLab)
      set power(recLab)  $v(recLab) 
      set power(typeLab) $v(typeLab)
    }
  }

  ;# F0表示
  if $v(showf0) {
    set ytop [expr $v(waveh) + $v(spech) + $v(powh)]
    set ylow [expr $ytop + $v(f0h)]
    ;# F0を抽出
    if {$opt == "all" || $opt == "f0"} {
      if {$f0(recLab) != $v(recLab) || $f0(typeLab) != $v(typeLab)} {
        set seriestmp {}
        if {[catch {set seriestmp [snd pitch -method $f0(method) \
          -framelength $f0(frameLength) -windowlength $f0(windowLength) \
          -maxpitch $f0(max) -minpitch $f0(min) \
          -progress waitWindow] } ret]} {
          if {$ret != ""} {
            puts "error: $ret"
          }
          set seriestmp {}
        }
        set f0(f0) {}
        foreach s $seriestmp {
          set val [lindex [split $s " "] 0]
          if {$f0(unit) == "semitone" && $val > 0} {
            set val [hz2semitone $val]
          }
          lappend f0(f0) $val
        }

        if {[llength $f0(f0)] > 0} {
          # F0の最大値・最小値を求める
          if {$opt == "all" || $opt == "f0"} {
            if {$f0(recLab) != $v(recLab) || $f0(typeLab) != $v(typeLab)} {
              set f0(extractedMax) [lindex $f0(f0) 0]
              set f0(extractedMin) [lindex $f0(f0) 0]
              for {set i 1} {$i < [llength $f0(f0)]} {incr i} {
                if {$f0(extractedMax) < [lindex $f0(f0) $i]} {
                  set f0(extractedMax) [lindex $f0(f0) $i]
                }
                if {$f0(extractedMin) > [lindex $f0(f0) $i] && [lindex $f0(f0) $i] > 0 ||
                    $f0(extractedMin) <= 0} {
                  set f0(extractedMin) [lindex $f0(f0) $i]
                }
              }
            }
          }
          # 描画するスケールを決める
          if $f0(fixShowRange) {
            set f0(extractedMax) $f0(showMax)
            set f0(extractedMin) $f0(showMin)
            if {$f0(unit) == "semitone"} {
              if {$f0(extractedMax) > 0} { set f0(extractedMax) [hz2semitone $f0(extractedMax)] }
              if {$f0(extractedMin) > 0} { set f0(extractedMin) [hz2semitone $f0(extractedMin)] }
            }
    #      } else {
    #        set f0(extractedMin) $f0(min)  ;# あえてf0(min)にしている
    #        if {$f0(unit) == "semitone"} {
    #          if {$f0(extractedMin) > 0} { set f0(extractedMin) [hz2semitone $min] }
    #        }
          }
        }
      }
    }

    if {[llength $f0(f0)] > 0} {
      if {$f0(extractedMax) > $f0(extractedMin)} {
        # ppd= 1Hzあたりのピクセル数。4は上下各2ピクセルのマージン
        set ppd [expr double($v(f0h)) / ($f0(extractedMax) - $f0(extractedMin))]

        # 各音に対応する周波数で横線を引く
        if $f0(showToneLine) {
          for {set i 0} {$i < [llength $v(sinScale)]} {incr i} {
            if {$f0(unit) == "semitone"} {
              set tgt [hz2semitone [lindex $v(sinScale) $i]]
            } else {
              set tgt [lindex $v(sinScale) $i]
            }
            set y1 [expr $ylow - ($tgt - $f0(extractedMin)) * $ppd]
            if {$y1 <= [expr $ylow - $v(f0h)]} break
            if {$y1 < $ylow} {
              $c create line 0 $y1 $v(cWidth) $y1 -tags axis \
                -fill $v(toneLineColor)
            }
          }
        }

        # ターゲット線をひく
        if $f0(showTgtLine) {
          if {$f0(unit) == "semitone"} {
            set tgt [hz2semitone $f0(tgtFreq)]
          } else {
            set tgt $f0(tgtFreq)
          }
          set y1 [expr $ylow - ($tgt - $f0(extractedMin)) * $ppd ]
          if {$y1 <= $ylow && $y1 >= [expr $ylow - $v(f0h)]} {
            $c create text [expr 0 + 2] $y1 \
              -text "$f0(tgtTone)$f0(tgtOctave)" -fill $v(tgtf0color) \
              -font smallkfont -anchor w -tags {axis tgtName}
            $c create line [lindex [$c bbox tgtName] 2] $y1 $v(cWidth) $y1 -tags axis \
              -fill $v(tgtf0color)
          }
        }

        # F0データをプロットする
        # set coord {} ;# F0曲線を引く座標(x,y)列
        set f0tags {obj f0}
        for {set i 0} {$i < [llength $f0(f0)]} {incr i} {
          # lappend coord \
          #   [expr $i * $f0(frameLength) * $v(wavepps)] \
          #   [expr $ylow - ([lindex $f0(f0) $i] - $f0(f0Min)) * $ppd]
          if {[lindex $f0(f0) $i] > 0} {
            set x1 [expr $i * $f0(frameLength) * $v(wavepps) - 2]
            set y1 [expr $ylow - ([lindex $f0(f0) $i] - $f0(extractedMin)) * $ppd - 2]
            set x2 [expr $x1 + 3]
            set y2 [expr $y1 + 3]
            if {$y1 <= $ylow && $y1 >= [expr $ylow - $v(f0h)]} {
              $c create oval $x1 $y1 $x2 $y2 -tags $f0tags -fill $v(f0color)
            }
          }
        }
      }
#      eval {$c create line} $coord -tags {$f0tags} -fill $v(f0color)
      f0Axis $cYaxis 0 $ytop $v(yaxisw) $v(f0h) \
        -tags axis -fill $v(fg) -font $v(sfont) \
        -f0Max $f0(extractedMax) -f0Min $f0(extractedMin) -unit $f0(unit)
    }
    # 下線
    $c create line 0 $ylow $v(cWidth) $ylow -tags axis
#    $c create line $v(yaxisw) $ytop $v(yaxisw) $ylow -tags axis -fill $v(fg)
    set yAxisLow [expr $ylow + 2]
    $cYaxis create line 0 $yAxisLow $v(yaxisw) $yAxisLow -tags axis -fill $v(fg)

    ;# F0を抽出したならFIDを記録する
    if {$f0(fid) != $v(recLab)} {
      set f0(fid) $v(recLab)
      set f0(recLab) $v(recLab)
      set f0(typeLab) $v(typeLab)
    }
  }

  ;# 時間軸表示
  if {$sndLen > 0} {
    if {$v(showWave) || $v(showSpec) || $v(showpow) || $v(showf0)} {
      set ytop [expr $v(waveh) + $v(spech) + $v(powh) + $v(f0h)]
      set ylow [expr $ytop + $v(timeh)]
      snack::timeAxis $c 0 $ytop $v(cWidth) $v(timeh) $v(wavepps) \
        -tags axis -starttime 0 -fill $v(fg)
      $c create line 0 $ylow $v(cWidth) $ylow -tags axis
    }
  }

  ;# grabを解放
  ;#  grab release $c
}

#---------------------------------------------------
# 録音開始
#
#mac
proc recStart {} {
  global snd v bgmParam t
  if {$v(recNow) || $v(rec) == 0} return
  if $v(playOnsaStatus)  toggleOnsaPlay   ;# 音叉再生中なら停止させる
  if $v(playMetroStatus) toggleMetroPlay  ;# メトロノーム再生中なら停止させる
  .recinfo.b.rec configure -image snackRecordNow
  set v(recStatus) 1
  set v(recNow) 1
  if {$v(rec) >= 2} {
    ;# 自動収録の場合
    if {$bgmParam(autoRecStatus) == 0} {
      autoRecStart
    ;#} else {
    ;#  autoRecStop   <-- macだとrを長く押した場合に録音停止してしまうため
    }
  } else {
    ;# 手動収録(ver.1.0の方法)の場合
    set v(msg) $t(recStart,msg)
    snd record
  }
}

#---------------------------------------------------
# 録音終了
#
#mac
proc recStop {} {
  global snd v t
  if {$v(rec) == 0} return
  if {$v(rec) >= 2} {
    ;# 自動収録モードなら
    autoRecStop
  } elseif $v(recNow) {
    ;# 手動収録モードで録音中なら
    set v(msg) $t(recStop,msg)
    snd stop
    if $v(removeDC) removeDC
    Redraw all
    set v(recNow) 0
  }
  .recinfo.b.rec configure -image snackRecord
}

#---------------------------------------------------
# ファイルを保存して終了
#
proc Exit {} {
  global v t
#  if $v(paramChanged) {
#    set act [tk_dialog .confm $t(.confm) $t(Exit,q1) \
#      question 2 $t(Exit,a1) $t(Exit,a2) $t(Exit,a3)]
#    switch $act {
#      0 {                      ;# 保存して終了する場合
#          if ![saveParamFile] {
#            return  ;# もしここで保存しなかったら終了中止。
#          }
#        }
#      1 { }                    ;# 保存せず終了する場合
#      2 { return }             ;# 終了しない場合
#    }
#  }

  if $v(recStatus) {
    set act [tk_dialog .confm $t(.confm) $t(Exit,q2) \
      question 2 $t(Exit,a1) $t(Exit,a2) $t(Exit,a3)]
    if {$act == 2} {
      return
    } elseif {$act == 0} {
      saveWavFile
    }
  }

  exit
}

#---------------------------------------------------
# 右クリックメニュー
#
proc PopUpMenu {X Y x y} {
  global v rclickMenu t

  $rclickMenu delete 0 end
  $rclickMenu add checkbutton -variable v(showWave) \
    -label $t(PopUpMenu,showWave) -command toggleWave
  $rclickMenu add checkbutton -variable v(showSpec) \
    -label $t(PopUpMenu,showSpec) -command toggleSpec
  $rclickMenu add checkbutton -variable v(showpow) \
    -label $t(PopUpMenu,showPow) -command togglePow
  $rclickMenu add checkbutton -variable v(showf0) \
    -label $t(PopUpMenu,showF0) -command toggleF0
  $rclickMenu add command -label $t(PopUpMenu,pitchGuide) -command pitchGuide
  $rclickMenu add command -label $t(PopUpMenu,tempoGuide) -command tempoGuide
  $rclickMenu add command -label $t(PopUpMenu,settings)   -command settings

  catch {tk_popup $rclickMenu $X $Y}
}

#---------------------------------------------------
# バージョン情報表示
#
proc Version {} {
  global v t
  tk_messageBox -title $t(Version,msg) \
    -message "$v(appname) version $v(version)"
}

#---------------------------------------------------
# リストボックスウィジット w を d だけスクロールさせる(Windows版)
# +/-120 は Windows でホイールを1つ動かした際に%Dにセットされる値
#
proc listboxScroll {w d} {
  if {$w ne ""} {
    $w yview scroll [expr -$d / 120] units
  }
}

#---------------------------------------------------
# usage
#
proc usage {} {
  global argv0 t
  puts "usage: $argv0 \[-d saveDir|-f initScript\]"
}

#---------------------------------------------------
# 単独音のUTAU原音パラメータを推定する際の設定窓
#
proc estimateParam {} {
  global epwindow power estimate v t

  if [isExist $epwindow] return  ;# 二重起動を防止
  toplevel $epwindow
  wm title $epwindow $t(estimateParam,title)
  wm resizable $epwindow 0 0
  bind $epwindow <Escape> "destroy $epwindow"

  set w [frame $epwindow.al]
  pack $w
  set row 0

  ;# パワー抽出刻みの設定
  label $w.lfl -text $t(estimateParam,pFLen)
  entry $w.efl -textvar power(frameLength) \
    -validate all -validatecommand {
         if {[string is double %P]} {
           if {[string length %P] > 0 && %P > 0} {
             set tmp [sec2samp $power(uttLengthSec) %P]
             if {$tmp >= 0} {
               set power(uttLength) $tmp
             }
           }
           expr {1}
         } else {
           expr {0}
         }
    }
  label $w.lflu -text "(sec)"
  grid $w.lfl  -sticky w -row $row -column 0
  grid $w.efl  -sticky w -row $row -column 1
  grid $w.lflu -sticky w -row $row -column 2 -columnspan 2
  incr row

  ;# プリエンファシスの設定
  # frame $w.fem
  label $w.lem -text $t(estimateParam,preemph)
  entry $w.eem -textvar power(preemphasis)
  grid $w.lem  -sticky w -row $row -column 0
  grid $w.eem  -sticky w -row $row -column 1 -columnspan 3
  incr row

  ;# パワー抽出窓長の設定
  label $w.lwl -text $t(estimateParam,pWinLen)
  entry $w.ewlSec -textvar power(windowLength)
  label $w.lwlSec -text "(sec)"
  grid $w.lwl      -sticky w -row $row -column 0
  grid $w.ewlSec   -sticky w -row $row -column 1
  grid $w.lwlSec   -sticky w -row $row -column 2 -columnspan 2
  incr row

  ;# 窓の選択
  # frame $w.fwn
  label $w.lwn -text $t(estimateParam,pWinkind)
  tk_optionMenu $w.mwn power(window) \
    Hamming Hanning Bartlett Blackman Rectangle
  grid $w.lwn  -sticky w -row $row -column 0
  grid $w.mwn  -sticky w -row $row -column 1 -columnspan 3
  incr row

  ;# 発話中とみなされるパワーの設定
  # frame $w.fhi
  label $w.lhi -text $t(estimateParam,pUttMin)
  entry $w.ehi -textvar power(uttHigh)
  label $w.lhiu -text "(db)"
  grid $w.lhi  -sticky w -row $row -column 0
  grid $w.ehi  -sticky w -row $row -column 1
  grid $w.lhiu -sticky w -row $row -column 2 -columnspan 2
  incr row

  ;# 発話中とみなされる時間長の設定
  # frame $w.ful
  label $w.lul -text $t(estimateParam,pUttMinTime)
  entry $w.eulSec -textvar power(uttLengthSec) \
    -validate all -validatecommand {
         set tmp [sec2samp %P $power(frameLength)]
         if {$tmp >= 0} {
           set power(uttLength) $tmp
           expr {1}
         } else {
           expr {0}
         }
    }
  label $w.lulSec -text "(sec) = "
  label $w.lulSamp -textvar power(uttLength)
  label $w.lulSampu -text "(sample)"
  grid $w.lul      -sticky w -row $row -column 0
  grid $w.eulSec   -sticky w -row $row -column 1
  grid $w.lulSec   -sticky w -row $row -column 2
  grid $w.lulSamp  -sticky w -row $row -column 3
  grid $w.lulSampu -sticky w -row $row -column 4
  incr row

  ;# 発声中に生じるパワーの揺らぎの大きさ設定
  # frame $w.fhi
  label $w.lkp -text $t(estimateParam,uttLen)
  entry $w.ekp -textvar power(uttKeep)
  label $w.lkpu -text "(db)"
  grid $w.lkp  -sticky w -row $row -column 0
  grid $w.ekp  -sticky w -row $row -column 1
  grid $w.lkpu -sticky w -row $row -column 2 -columnspan 2
  incr row

  ;# ポーズ中とみなされるパワーの設定
  # frame $w.flw
  label $w.llw -text $t(estimateParam,silMax)
  entry $w.elw -textvar power(uttLow)
  label $w.llwu -text "(db)"
  grid $w.llw  -sticky w -row $row -column 0
  grid $w.elw  -sticky w -row $row -column 1
  grid $w.llwu -sticky w -row $row -column 2 -columnspan 2
  incr row

  ;# 母音パワー最小値の設定（この値未満は子音部とみなす）
  label $w.lmv -text $t(estimateParam,vLow)
  entry $w.emvSec -textvar power(vLow)
  label $w.lmvSec -text "(db)"
  grid $w.lmv      -sticky w -row $row -column 0
  grid $w.emvSec   -sticky w -row $row -column 1
  grid $w.lmvSec   -sticky w -row $row -column 2
  incr row

  ;# ポーズとみなされる時間長の設定
  # frame $w.fsl
  label $w.lsl -text $t(estimateParam,silMinTime)
  entry $w.eslSec -textvar power(silLengthSec) \
    -validate all -validatecommand {
         set tmp [sec2samp %P $power(frameLength)]
         if {$tmp >= 0} {
           set power(silLength) $tmp
           expr {1}
         } else {
           expr {0}
         }
    }
  label $w.lslSec -text "(sec) = "
  label $w.lslSamp -textvar power(silLength)
  label $w.lslSampu -text "(sample)"
  grid $w.lsl      -sticky w -row $row -column 0
  grid $w.eslSec   -sticky w -row $row -column 1
  grid $w.lslSec   -sticky w -row $row -column 2
  grid $w.lslSamp  -sticky w -row $row -column 3
  grid $w.lslSampu -sticky w -row $row -column 4
  incr row

  ;# 子音長最小値の設定（子音部=0でUTAUがエラーになるのを回避するため）
  label $w.lmc -text $t(estimateParam,minC)
  entry $w.emcSec -textvar estimate(minC)
  label $w.lmcSec -text "(sec)"
  grid $w.lmc      -sticky w -row $row -column 0
  grid $w.emcSec   -sticky w -row $row -column 1
  grid $w.lmcSec   -sticky w -row $row -column 2
  incr row

  ;# F0に関する説明文
  label $w.lf0 -text $t(estimateParam,f0)
  grid $w.lf0      -sticky w -row $row -column 0 -columnspan 3
  incr row

  ;# ボタンの設定
  # frame $w.fbt
  button $w.do -text $t(.confm.run) -command {
    ;# もしエントリ欄が空欄だったなら0を代入
    if {[string length $power(silLengthSec)] == 0} {
      set power(silLengthSec) 0
    }
    if {[string length $power(uttLengthSec)] == 0} {
      set power(uttLengthSec) 0
    }

#    grab set $epwindow     ;# 推定中は他の操作が出来ないようにする
    makeRecListFromDir 0
    doEstimateParam all
    deleteProgressWindow   ;# なぜかmacではdoEstimateParamの中で消せなかった
    saveParamFile
#    grab release $epwindow ;# 推定中は他の操作が出来ないようにする
    destroy $epwindow
  }
  button $w.cancel -text $t(.confm.c) -command {destroy $epwindow}
  grid $w.do     -sticky we -row $row -column 0
  grid $w.cancel -sticky we -row $row -column 1
  incr row
}

#---------------------------------------------------
# UTAU単独音パラメータの自動推定
#
proc doEstimateParam {{mode all}} {
  global power v paramS paramU paramUsize estimate t f0 ;# f0は他でも使っている

  set v(msg) $t(doEstimateParam,startMsg)

  snack::sound sWork -channels Mono -rate $v(sampleRate) -fileformat WAV

  initProgressWindow

  ;# 推定する行番号のリストを作る
  set targetList {}
  if {$mode == "all"} {
    for {set i 1} {$i < $paramUsize} {incr i} {
      lappend targetList $i
    }
  } else {
    foreach pos [.entpwindow.t curselection] {
      set r [lindex [split $pos ","] 0]
      if {[lindex $targetList end] != $r} {
        lappend targetList $r
      }
    }
  }

  foreach i $targetList {
    ;# wavファイルを読む
    set fid $paramU($i,0)
    set filename $v(saveDir)/$fid.wav
    sWork read $filename

    ;# パワーを抽出する
    set pw [sWork power -framelength $power(frameLength) \
      -windowtype $power(window) -preemphasisfactor $power(preemphasis) \
      -windowlength [expr int($power(windowLength) * $v(sampleRate))] \
      -start 0 -end -1]

    ;# 初期値設定
    if $estimate(S) { set paramS($i,S) 0 }
    if $estimate(C) { set paramS($i,C) $estimate(minC) }
    if $estimate(E) { set paramS($i,E) [sWork length -unit SECONDS] }
    set uttS 0           ;# 発声音量が十分な大きさになっているとみなされた位置
    set uttE [expr [llength $pw] - 1]     ;# 発声音量が減衰し始める位置

    if {[llength $pw] > 0} {
      ;# 左側の発話中確定点を探す
      set length 0   ;# 発話が連続しているサンプル数
      for {set j 0} {$j < [llength $pw]} {incr j} {
        if {[lindex $pw $j] >= $power(uttHigh)} {
          incr length
        } else {
          set length 0
        }
        if {$length >= [expr $power(uttLength) + 1]} {
          set uttS $j     ;# 現在位置を保存
          break
        }
      }
      ;# 左にたどって左ブランク位置（発話開始点）を探す
      if $estimate(S) {
        set length 0
        for {set k $j} {$k > 0} {incr k -1} {
          if {[lindex $pw $k] <= $power(uttLow)} {
            incr length
          } else {
            set length 0
          }
          if {$length >= [expr $power(silLength) + 1]} {
            ;# 現在位置を左ブランクにする
            set tm [expr $k * $power(frameLength)] ;# 時刻[sec]を計算
            set paramS($i,S) $tm
            break
          }
        }
      }

      ;# 発声音量が減衰し始める点を求める
      set length 0   ;# 発話が連続しているサンプル数
      for {set j [expr [llength $pw] - 1]} {$j > $uttS} {incr j -1} {
        if {[lindex $pw $j] >= $power(uttHigh)} {
          incr length
        } else {
          set length 0
        }
        if {$length >= [expr $power(uttLength) + 1]} {
          set uttE $j     ;# 現在位置を保存
          break
        }
      }

      ;# uttS〜uttE間の中央付近の平均パワーavePを求める
      set Nmax 30
      set N 0
      set aveP 0
      set center [expr int(($uttE + $uttS) / 2)]
      for {set j [expr $center + 1]} {$j <= $uttE && $N < [expr $Nmax / 2]} {incr j} {
        set aveP [expr $aveP + [lindex $pw $j]]
        incr N
      }
      for {set j $center} {$j >= $uttS && $N < $Nmax} {incr j -1} {
        set aveP [expr $aveP + [lindex $pw $j]]
        incr N
      }
      set aveP [expr $aveP / $N]

      ;# 右ブランク位置を探す
      if $estimate(E) {
        for {set j $center} {$j <= $uttE} {incr j} {
;#koko, パワーが平均値より大きい方に揺らいでいる分はOKとした。
;#koko, パワーが平均値より小さく揺らいだときに反応させるようにした
          if {[expr $aveP - [lindex $pw $j]] > [expr $power(uttKeep) / 2]} {
            break
          }
        }
        ;# 現在位置を右ブランクにする
        set tm [expr $j * $power(frameLength)] ;# 時刻[sec]を計算
        if {$paramS($i,S) < $tm} {
          set paramS($i,E) $tm
        } else {
          set paramS($i,E) $paramS($i,S)
        }
      }

      ;# 子音部位置を探す
      if $estimate(C) {
        for {set j $center} {$j >= $uttS} {incr j -1} {
;#koko, パワーが平均値より大きい方に揺らいでいる分はOKとした。
;#koko, パワーが平均値より小さく揺らいだときに反応させるようにした
          if {[expr $aveP - [lindex $pw $j]] > [expr $power(uttKeep) / 2]} {
            break
          }
        }
        ;# 現在位置を子音部にする
        set tm [expr $j * $power(frameLength)] ;# 時刻[sec]を計算
        if {$tm >= [expr $paramS($i,S) + $estimate(minC)]} {
          set paramS($i,C) $tm
        } else {
          set paramS($i,C) [expr $paramS($i,S) + $estimate(minC)]
        }
      }
    }

    ;# 先行発声位置を探す
    if $estimate(P) {
      set paramS($i,P) $paramS($i,S)

      ;# 有声開始位置を求める
      set seriestmp {}
      if {[catch {set seriestmp [sWork pitch -method $f0(method) \
        -framelength $f0(frameLength) -windowlength $f0(windowLength) \
        -maxpitch $f0(max) -minpitch $f0(min) \
        ] } ret]} {
        if {$ret != ""} {
          puts "error: $ret"
        }
        set seriestmp {}
      }
      for {set vot [expr int($paramS($i,P) / $f0(frameLength))]} \
          {$vot < [llength $seriestmp]} {incr vot} {
        if {[lindex [split [lindex $seriestmp $vot] " "] 0] > 0} break
      }
      set votSec [expr $vot * $f0(frameLength)]

      ;# votから右に行き、母音パワー最小値を超える所まで移動する
      for {set j [expr int($votSec / $power(frameLength))]} \
          {[expr $j * $power(frameLength)] < $paramS($i,C)} \
          {incr j} {
        if {[lindex $pw $j] >= $power(vLow)} break
      }

      ;# 現在位置を先行発声にする
      set tm [expr $j * $power(frameLength)] ;# 時刻[sec]を計算
      if {$tm <= $paramS($i,C)} {
        set paramS($i,P) $tm
      } else {
        set paramS($i,P) $paramS($i,S)
      }
    }

    ;# オーバーラップ位置を探す。現在未実装。
    if $estimate(O) {
      set paramS($i,O) $paramS($i,S)
    }

    ;# paramUを設定する
    foreach kind {S E C P O} {
      if $estimate($kind) {
        set paramU($i,[kind2c $kind]) [sec2u $i $kind $paramS($i,$kind)]
      }
    }
    updateProgressWindow [expr 100 * $i / $paramUsize]
    set v(msg) "$t(doEstimateParam,startMsg) ($i / $paramUsize)"
  }
  deleteProgressWindow
#  set v(paramChanged) 1
#  setEPWTitle
  set v(msg) [eval format $t(doEstimateParam,doneMsg)]
}

#---------------------------------------------------
# 自動収録した連続発声からoto.iniを生成
#
proc genParam {} {
  global v genWindow genParam t

  if [isExist $genWindow] return ;# 二重起動を防止
  toplevel $genWindow
  wm title $genWindow $t(genParam,title)
  bind $genWindow <Escape> "destroy $genWindow"

  set r 0

  # 初期設定
  set f($r) [labelframe $genWindow.f($r) -relief groove -padx 5 -pady 5]

  label $f($r).lB  -text $t(genParam,tempo)
  entry $f($r).eB  -textvar genParam(bpm) -wi 10
  label $f($r).lBU -text $t(genParam,bpm)

  label $f($r).lS  -text $t(genParam,S)
  entry $f($r).eS  -textvar genParam(S) -wi 10
  label $f($r).lSU -text $t(genParam,unit)
  tk_optionMenu $f($r).mSU genParam(SU) msec $t(genParam,haku)

  grid  $f($r).lB  -row 0 -column 0 -sticky nse
  grid  $f($r).eB  -row 0 -column 1 -sticky nse
  grid  $f($r).lBU -row 0 -column 2 -sticky nsw -columnspan 2
  grid  $f($r).lS  -row 1 -column 0 -sticky nse
  grid  $f($r).eS  -row 1 -column 1 -sticky nse
  grid  $f($r).lSU -row 1 -column 2 -sticky nsw
  grid  $f($r).mSU -row 1 -column 3 -sticky nsw
  incr r

  # ボタン
  set f($r) [frame $genWindow.f($r) -padx 5 -pady 0]
  label  $f($r).arrow1 -text $t(genParam,darrow)
  button $f($r).bInit  -text $t(genParam,bInit)  -command initGenParam
  label  $f($r).arrow2 -text $t(genParam,darrow)

  grid  $f($r).arrow1 -row 0 -column 0 -sticky nsew
  grid  $f($r).bInit  -row 1 -column 0 -sticky nsew
  grid  $f($r).arrow2 -row 2 -column 0 -sticky nsew
  incr r

  # msec単位での各設定
  set f($r) [labelframe $genWindow.f($r) -relief groove -padx 5 -pady 5]
  label $f($r).lO  -text $t(genParam,O)
  entry $f($r).eO  -textvar genParam(O) -wi 10
  label $f($r).lOU -text $t(genParam,msec)

  label $f($r).lP  -text $t(genParam,P)
  entry $f($r).eP  -textvar genParam(P) -wi 10
  label $f($r).lPU -text $t(genParam,msec)

  label $f($r).lC  -text $t(genParam,C)
  entry $f($r).eC  -textvar genParam(C) -wi 10
  label $f($r).lCU -text $t(genParam,msec)

  label $f($r).lE  -text $t(genParam,E)
  entry $f($r).eE  -textvar genParam(E) -wi 10
  label $f($r).lEU -text $t(genParam,msec)

  grid  $f($r).lO    -row 0 -column 0 -sticky nse
  grid  $f($r).eO    -row 0 -column 1 -sticky nse
  grid  $f($r).lOU   -row 0 -column 2 -sticky nse
  grid  $f($r).lP    -row 1 -column 0 -sticky nse
  grid  $f($r).eP    -row 1 -column 1 -sticky nse
  grid  $f($r).lPU   -row 1 -column 2 -sticky nse
  grid  $f($r).lC    -row 2 -column 0 -sticky nse
  grid  $f($r).eC    -row 2 -column 1 -sticky nse
  grid  $f($r).lCU   -row 2 -column 2 -sticky nse
  grid  $f($r).lE    -row 3 -column 0 -sticky nse
  grid  $f($r).eE    -row 3 -column 1 -sticky nse
  grid  $f($r).lEU   -row 3 -column 2 -sticky nse
  incr r

  # 先行発声位置の自動推定用の設定窓
  set f($r) [labelframe $genWindow.f($r) -relief groove -padx 5 -pady 5]
  checkbutton $f($r).cb -variable genParam(autoAdjustRen) -text $t(genParam,autoAdjustRen)
  label $f($r).lv  -text $t(genParam,vLow)
  entry $f($r).ev  -textvar genParam(vLow) -wi 10
  label $f($r).lvu -text $t(genParam,db)
  label $f($r).ls  -text $t(genParam,sRange)
  entry $f($r).es  -textvar genParam(sRange) -wi 10
  label $f($r).lsu -text $t(genParam,msec)
  label $f($r).lb -text $t(genParam,f0pow)

  grid  $f($r).cb  -row 0 -column 0 -sticky nsw -columnspan 3
  grid  $f($r).lv  -row 1 -column 0 -sticky nse
  grid  $f($r).ev  -row 1 -column 1 -sticky nse
  grid  $f($r).lvu -row 1 -column 2 -sticky nsw
  grid  $f($r).ls  -row 2 -column 0 -sticky nse
  grid  $f($r).es  -row 2 -column 1 -sticky nse
  grid  $f($r).lsu -row 2 -column 2 -sticky nsw
  grid  $f($r).lb  -row 3 -column 0 -sticky nsw -columnspan 3
  incr r

  # エイリアス重複に関する設定欄
  set f($r) [labelframe $genWindow.f($r) -relief groove -padx 5 -pady 5]

  set genParam(useAliasMax) 0  ;# 0=重複をそのままにする,1=通し番号を付ける
  set genParam(aliasMax) 0     ;# 重複番号の最大値
  label $f($r).lt -text $t(genParam,aliasMax)
  radiobutton $f($r).rm1 -variable genParam(useAliasMax) -value  0 -text $t(genParam,aliasMaxNo)
  radiobutton $f($r).rm2 -variable genParam(useAliasMax) -value  1 -text $t(genParam,aliasMaxYes)
  label $f($r).ll   -text $t(genParam,aliasMaxNum)
  entry $f($r).e    -textvar genParam(aliasMax) -wi 10

  grid $f($r).lt    -row 0 -column 0 -sticky nsw
  grid $f($r).rm1   -row 1 -column 0 -sticky nsw
  grid $f($r).rm2   -row 1 -column 1 -sticky nsw
  grid $f($r).ll    -row 2 -column 0 -sticky nse
  grid $f($r).e     -row 2 -column 1 -sticky nse
  incr r

  # 実行ボタン
  set f($r) [frame $genWindow.f($r) -padx 5 -pady 0]

  label  $f($r).arrow2 -text $t(genParam,darrow)
  button $f($r).bs -text $t(genParam,do) -command {
    doGenParam
    saveParamFile
    destroy $genWindow
  }
  button $f($r).bc -text $t(.confm.c) -command {destroy $genWindow}

  grid  $f($r).arrow2 -row 0 -column 0 -columnspan 2 -sticky nsew
  grid  $f($r).bs     -row 1 -column 0 -sticky nsew
  grid  $f($r).bc     -row 1 -column 1 -sticky nsew
  incr r

  for {set i 0} {$i < $r} {incr i} {
    pack $f($i) -anchor nw -padx 2 -pady 2 -expand 1 -fill x
  }

  raise $genWindow
  focus $genWindow
}

#---------------------------------------------------
# BPM、冒頭Sからパラメータの初期値を求める
#
proc initGenParam {} {
  global genParam t

  set mspb [expr 60000.0 / $genParam(bpm)]  ;# 1拍の長さ[msec]を求める

  set genParam(O) [cut3 [expr $mspb / 6.0]]
  set genParam(P) [cut3 [expr $mspb / 2.0]]
  set genParam(C) [cut3 [expr $mspb * 3.0 / 4.0]]
  set genParam(E) [cut3 [expr - ($mspb + $genParam(O)) ]]
  set genParam(sRange) $genParam(P)
  ;#set genParam(E) [cut3 [expr - $mspb ]]
}

#---------------------------------------------------
# 連続発声の先頭モーラの先行発声位置を推定し、
# 先行発声がその場所になるための補正量(sec)を返す
# Porg, rangeの単位：sec。Porg-range〜Porg+rangeの範囲を探索する
#
proc autoAdjustRen {fid Porg range} {
  global v t f0 power genParam

  snack::sound sWork
  if {[file exists "$v(saveDir)/$fid.wav"]} {
    sWork read "$v(saveDir)/$fid.wav"
  }

  if {$Porg > $range} {
    set Lsec [expr $Porg - $range]
  } else {
    set Lsec 0
  }

  ;# 有声開始位置を求める
  set seriestmp {}
  set start [expr int($Lsec * $v(sampleRate))]
  set end   [expr int(($Porg + $range) * $v(sampleRate))]
  if {[catch {set seriestmp [sWork pitch -method $f0(method) \
    -framelength $f0(frameLength) -windowlength $f0(windowLength) \
    -maxpitch $f0(max) -minpitch $f0(min) \
    -start $start -end $end \
    ] } ret]} {
    if {$ret != ""} {
      puts "error: $ret"
    }
    set seriestmp {}
  }
  set f0old 1
  for {set i 0} {$i < [llength $seriestmp]} {incr i} {
    set f0now [lindex [split [lindex $seriestmp $i] " "] 0]
    if {$f0old <= 0 && $f0now > 0} break  ;# 直前が無声、当該が有声ならbreak
    set f0old $f0now
  }
  if {$i < [llength $seriestmp]} {
    ;# 有声開始点であればそこを先行発声候補とし、更に次のパワーに基づく
    ;# 探索開始点にする
    set Pnew [expr $i * $f0(frameLength) + $Lsec]
    set Lsec $Pnew
    set start [expr int($Lsec * $v(sampleRate))]   ;# end はF0と同じ
  } else {
    set Pnew $Porg
  }

  ;# 固定範囲〜右ブランク間の平均パワーavePを求める。ただし値が30個以上なら
  ;# 30個で平均を求める
  set Nmax 30
  set N 0
  set aveP 0
  set avePstart [expr int(($Pnew + ($genParam(C) - $genParam(P)) / 1000.0) * $v(sampleRate))]
  set avePend   [expr int(($Pnew + (abs($genParam(E)) - $genParam(P)) / 1000.0) * $v(sampleRate))]
  set pw [sWork power -framelength $power(frameLength) \
    -windowtype $power(window) -preemphasisfactor $power(preemphasis) \
    -windowlength [expr int($power(windowLength) * $v(sampleRate))] \
    -start $start -end $end]
  for {set i 0} {$i < [llength $pw] && $i < $Nmax} {incr i} {
    set aveP [expr $aveP + [lindex $pw $i]]
    incr N
  }
  if {$N > 0} {
    set aveP [expr $aveP / $N]
  }

  ;# 探索区間のパワーを求める
  set pw [sWork power -framelength $power(frameLength) \
    -windowtype $power(window) -preemphasisfactor $power(preemphasis) \
    -windowlength [expr int($power(windowLength) * $v(sampleRate))] \
    -start $start -end $end]

  ;# 閾値に対してパワー曲線が右上がりにクロスする点を探す
  ;# votから右に行き、パワーが右上がりに閾値を超える所まで移動する
  ;# そのような場所が複数ある場合は凹みがより深いものにする
  set vLow [expr $aveP - $genParam(vLow)]
  set powOld $vLow
  set pn -1                         ;# 先行発声位置を入れる変数
  set pnMin     10001               ;# 先行発声前の凹みの深さ
  set powNowMin 10000
  for {set i 0} {$i < [llength $pw]} {incr i} {
    set powNow [lindex $pw $i]
    if {$powNow < $powNowMin} {
      set powNowMin $powNow         ;# 凹みの値を求めていく
    }
    if {$powOld < $vLow && $powNow >= $vLow && $powNowMin < $pnMin} {
      set pn $i
      set pnMin $powNowMin       ;# 凹みの値を保存
      set powNowMin 10000        ;# 再初期化
    }
    set powOld $powNow
  }

  #if {$pn < 0 && [expr $genParam(avePPrev) - $genParam(vLow) - $aveP] > 0} {
  #  ;# パワー凹みを見つけられず、かつ先行モーラより当該モーラの平均パワーがある程度小さい場合、
  #  ;# votから右に行き、パワー曲線が当該モーラの平均パワーにクロスする点を探す
  #  for {set i 0} {$i < [llength $pw]} {incr i} {
  #    set powNow [lindex $pw $i]
  #    if {$powNow <= $aveP} {
  #      set pn $i
  #      break
  #    }
  #  }
  #}

  if {$pn >= 0} {
    set Pnew [expr $pn * $power(frameLength) + $Lsec]
  }

#koko,もう一つ規則を作るなら、上記パワーの凹みが検出できない場合に
#現在のPnewの直近の凹み(x(i-1) >= x(i) < x(i+1) なi)を探すとか。

#koko,平均パワーが先行モーラより大きい場合の規則。

  set genParam(avePPrev) $aveP    ;# 次回の推定のために平均パワーを保存
 
  ;# 現在位置が先行発声になるための補正量(sec)を返す
  return [expr $Pnew - $Porg]
}

#---------------------------------------------------
# もしSが負なら他のパラメータ位置が変わらないようにS=0にする
# ※本ルーチンは連続音用(genParam用)
# ※本ルーチンではparamUのみ訂正し、paramSには反映させないことに注意
#
proc setSto0 {r} {
  global paramU

  if {$paramU($r,1) < 0} {     ;# S < 0 なら
    ;# E(負) の値を修正
    set paramU($r,5) [cut3 [expr $paramU($r,5) - $paramU($r,1)]]
    ;# C の値を修正
    set paramU($r,4) [cut3 [expr $paramU($r,4) + $paramU($r,1)]]
    ;# P の値を修正
    set paramU($r,3) [cut3 [expr $paramU($r,3) + $paramU($r,1)]]
    ;# O の値を修正
    set paramU($r,2) [cut3 [expr $paramU($r,2) + $paramU($r,1)]]
    if {$paramU($r,2) < 0} {
      set paramU($r,2) 0
    }
    ;# S を0にする
    set paramU($r,1) 0
  }
}

#---------------------------------------------------
# 連続発声のパラメータを自動生成する
#
proc doGenParam {} {
  global genParam v snd paramS paramU paramUsize t

  set recList {}
  foreach filename [glob -nocomplain [format "%s/*.wav" $v(saveDir)]] {
    set filename [file rootname [file tail $filename]]
    if {$filename == ""} continue
    lappend recList $filename
  }
  initParamS
  initParamU 1

  set mspb [expr 60000.0 / $genParam(bpm)]  ;# 1拍の長さ[msec]を求める

  # 1モーラ目の開始位置[ms]を求める
  if {$genParam(SU) == "msec"} {
    set Sstart $genParam(S)
  } else {
    set mspb [expr 60000.0 / $genParam(bpm)]  ;# 1拍の長さ[msec]を求める
    set Sstart [expr $genParam(S) * $mspb]
  }

  initProgressWindow

  array unset aliasChoufuku   ;# エイリアス重複数を入れる
  for {set recListSeq 0} {$recListSeq < [llength $recList]} {incr recListSeq} {
    set fid [lindex $recList $recListSeq]
    set S $Sstart
    set morae [getMorae [string trimleft $fid "_"]]
    set genParam(avePPrev) 0    ;# 平均パワーを初期化
    set fname ""
    set fnameOld ""
    for {set i 0} {$i < [llength $morae]} {incr i} {
      ;# エイリアスの決定
      set alias [getRenAlias $morae $i]
      if {$genParam(useAliasMax) && [array names aliasChoufuku $alias] != ""} {
        incr aliasChoufuku($alias)
        if {$genParam(aliasMax) <= 0 || $aliasChoufuku($alias) <= $genParam(aliasMax)} {
          set alias "$alias$aliasChoufuku($alias)"
        } else {
          set S [expr $S + $mspb]   ;# Sを次の位置に移動
          continue  ;# 重複が上限を超えたので登録せず次へ。
        }
      } else {
        set aliasChoufuku($alias) 1
      }
      # Sの位置補正
      if $genParam(autoAdjustRen) {
        ;# 先行発声位置を自動推定し、その差分をSに加える
        set Psec [expr ($S + $genParam(P)) / 1000.0]
        set range [expr $genParam(sRange) / 1000.0]
        set S [cut3 [expr $S + 1000.0 * [autoAdjustRen $fid $Psec $range]]]
      }
      set paramU($paramUsize,0) $fid           ;# fid
      set paramU($paramUsize,6) $alias         ;# A
      set paramU($paramUsize,1) $S             ;# S
      set paramU($paramUsize,4) $genParam(C)   ;# C
      set paramU($paramUsize,5) $genParam(E)   ;# E
      set paramU($paramUsize,3) $genParam(P)   ;# P
      set paramU($paramUsize,2) $genParam(O)   ;# O
      set paramU($paramUsize,R) $recListSeq    ;# recListの配列番号

      setSto0 $paramUsize    ;# Sが負の場合は他のパラメータ位置を動かさず0に修正

      set S [expr $S + $mspb]   ;# Sを次の位置に移動

      ;# paramSを求める
      set fname "$v(saveDir)/$fid.wav"
      if [file exists "$fname"] {
        if {$fname != $fnameOld} {
          snd read "$fname"
          set fname $fnameOld
        }
        paramU2paramS $paramUsize
      }

      incr paramUsize
    }
    updateProgressWindow [expr 100 * $recListSeq / [llength $recList]]
  }

  deleteProgressWindow
  set v(msg) [eval format $t(doGenParam,doneMsg)]
}

#---------------------------------------------------
# モーラ列moraeの指定した位置iから連続音用のエイリアス(「a い」など)を求めて返す
#
proc getRenAlias {morae i} {
  global genParam v snd paramS paramU paramUsize t

  set mora [lindex $morae $i]
  if {$i == 0} {
    set prev "-"
  } else {
    set prev [getVowel [lindex $morae [expr $i - 1]]]
  }
  return "$prev $mora"
}

#---------------------------------------------------
# 文字列を1モーラに分解して返す
#
proc getMorae {inMorae} {
  set morae {}
  for {set i 0} {$i < [string length $inMorae]} {incr i} {
    set char [string range $inMorae $i $i]
    if [isKana $char] {
      if [isMora $char] {
        ;# 現在の$charは一モーラなのでリストに追加
        lappend morae $char
      } else {
        set last [expr [llength $morae] -1]
        set mora "[lindex $morae $last]$char"
        set morae [lreplace $morae $last $last $mora]
      }
    }
  }
  return $morae
}

#---------------------------------------------------
# 一モーラの母音部の音素を返す
#
proc getVowel {mora} {
  set last [expr [string length $mora] -1]
  set char [string range $mora $last $last]

  set vA {あ か さ た な は ま や ら わ が ざ だ ば ぱ ゃ ぁ ゎ \
          ア カ サ タ ナ ハ マ ヤ ラ ワ ガ ザ ダ バ パ ャ ァ ヮ }
  set vI {い き し ち に ひ み    り    ぎ じ ぢ び ぴ    ぃ ゐ \
          イ キ シ チ ニ ヒ ミ    リ    ギ ジ ヂ ビ ピ    ィ ヰ }
  set vU {う く す つ ぬ ふ む ゆ る    ヴ ぐ ず づ ぶ ぷ ぅ ゅ っ \
          ウ ク ス ツ ヌ フ ム ユ ル       グ ズ ヅ ブ プ ゥ ュ ッ }
  set vE {え け せ て ね へ め    れ    げ ぜ で べ ぺ    ぇ ゑ \
          エ ケ セ テ ネ ヘ メ    レ    ゲ ゼ デ ベ ペ    ェ ヱ }
  set vO {お こ そ と の ほ も よ ろ を ご ぞ ど ぼ ぽ ょ ぉ    \
          オ コ ソ ト ノ ホ モ ヨ ロ ヲ ゴ ゾ ド ボ ポ ョ ォ    }
  set vN {ん ン}

  if {[lsearch $vA $char] >= 0} { return "a" }
  if {[lsearch $vI $char] >= 0} { return "i" }
  if {[lsearch $vU $char] >= 0} { return "u" }
  if {[lsearch $vE $char] >= 0} { return "e" }
  if {[lsearch $vO $char] >= 0} { return "o" }
  if {[lsearch $vN $char] >= 0} { return "n" }
}

#---------------------------------------------------
# charが平仮名または片仮名なら1を、それ以外なら0を返す
#
proc isKana {char} {
  set kanaList {あ か さ た な は ま や ら わ    が ざ だ ば ぱ ゃ ぁ ゎ \
                ア カ サ タ ナ ハ マ ヤ ラ ワ    ガ ザ ダ バ パ ャ ァ ヮ \
                い き し ち に ひ み    り       ぎ じ ぢ び ぴ    ぃ ゐ \
                イ キ シ チ ニ ヒ ミ    リ       ギ ジ ヂ ビ ピ    ィ ヰ \
                う く す つ ぬ ふ む ゆ る       ぐ ず づ ぶ ぷ ゅ ぅ っ \
                ウ ク ス ツ ヌ フ ム ユ ル    ヴ グ ズ ヅ ブ プ ュ ゥ ッ \
                え け せ て ね へ め    れ       げ ぜ で べ ぺ    ぇ ゑ \
                エ ケ セ テ ネ ヘ メ    レ       ゲ ゼ デ ベ ペ    ェ ヱ \
                お こ そ と の ほ も よ ろ を    ご ぞ ど ぼ ぽ ょ ぉ    \
                オ コ ソ ト ノ ホ モ ヨ ロ ヲ    ゴ ゾ ド ボ ポ ョ ォ    \
                ん ン ゛ ゜ °}
  if {[lsearch $kanaList $char] >= 0} {
    return 1
  } else {
    return 0
  }
}

#---------------------------------------------------
# charが一モーラなら1を、拗音などなら0を返す
#
proc isMora {char} {
  set notMora {ぁ ぃ ぅ ぇ ぉ ゃ ゅ ょ ゎ っ \
               ァ ィ ゥ ェ ォ ャ ュ ョ ヮ ッ ゛ ゜ °}
  if {[lsearch $notMora $char] >= 0} {
    return 0
  } else {
    return 1
  }
}

#---------------------------------------------------
# 指定した行のparamUをparamSに変換する
# snd に波形を読んでいることが前提
#
proc paramU2paramS {r} {
  global paramU paramS t

  for {set c 1} {$c < 6} {incr c} {
    set kindTmp [c2kind $c]
    if {[array names paramU "$r,$c"] != ""} {
      set paramS($r,$kindTmp) [u2sec $kindTmp $r $c]
    }
  }
}

#---------------------------------------------------
# 実数を小数点以下6桁で打ち切る
#
proc cut6 {val} {
  return [expr int($val * 1000000) / 1000000.0 ]
}

#---------------------------------------------------
# 実数を小数点以下3桁で打ち切る
#
proc cut3 {val} {
  return [expr int($val * 1000) / 1000.0 ]
}

#---------------------------------------------------
# パラメータ種類名をparamU列番号に変換
#
proc kind2c {k} {
  switch $k {
    ;# A       6
    ;# fid     0
    ;#         R
    S { return 1 }
    O { return 2 }
    P { return 3 }
    C { return 4 }
    E { return 5 }
    - { return ""}
  }
}

#---------------------------------------------------
# paramUの列番号をパラメータ種類名に変換
#
proc c2kind {c} {
  switch $c {
    1 { return S }
    2 { return O }
    3 { return P }
    4 { return C }
    5 { return E }
    - { return ""}
    ;# 0       fid
    ;# 6       A
    ;# R
  }
}

#---------------------------------------------------
# 単位変換
#
proc sec2samp {sec length} {
  if {[string length $sec] == 0 || [string length $length] == 0} {
    return 0
  }
  if {[string is double $sec] && [string is double $length]} {
    return [expr int(double($sec) / $length)]
  } else {
    return -1
  }
}

#---------------------------------------------------
# 単位変換(秒→UTAU原音パラメータ)
#
proc sec2u {r kind newVal} {  ;# r=-1:アクティブセルを変換。
  global paramS paramU snd v t

  if {$r < 0} {set r $v(listSeq)}  ;# 現在のアクティブセルの行番号

  if {[llength $newVal] == 0} {
    set newVal $paramS($r,$kind)   ;# 変換したい値(秒単位)を決定。
  }

  set fid $paramU($r,0)
  set fname $v(saveDir)/$fid.wav
  if {[snd cget -load] != $fname && [file readable $fname]} {
    snd read $fname
  }
  if {[array names paramS "$r,S"] != ""} {
    set S $paramS($r,S)
  } else {
    set S 0
  }
  switch $kind {
    S { return   [cut3 [expr double($newVal) * 1000.0] ] }
    E { if {$v(setE) < 0 } {
          return [cut3 [expr - ($newVal - $S) * 1000.0] ]
        } else {
          return [cut3 [expr [snd length -unit SECONDS] * 1000.0 - $newVal * 1000.0]] 
        }
      }
    C -
    P -
    O { return [cut3 [expr ($newVal - $S) * 1000] ]}
  }
}

#---------------------------------------------------
# 単位変換(UTAU原音パラメータ→秒)
# snd に波形を読んでいることが前提
#
proc u2sec {kind r c} {
  global paramS paramU snd t
  if {[array names paramS "$r,S"] != ""} {
    set S $paramS($r,S)
  } else {
    set S 0
  }
  if {$paramU($r,$c) != ""} {
    set u $paramU($r,$c)
  } else {
    set u 0
  }
  switch $kind {
    S { return [cut6 [expr $u / 1000.0]] }
    C -
    P -
    O { return [cut6 [expr $u / 1000.0 + $S]] }
    E { if {$u >= 0} {
          return [cut6 [expr [snd length -unit SECONDS] - $u / 1000.0]] 
        } else {
          return [cut6 [expr $u / -1000.0 + $S]]
        }
      }
  }
}

#---------------------------------------------------
# 原音パラメータ値の初期化
#
proc initParamS {} {
  global paramS v t

  array unset paramS
}

#---------------------------------------------------
# ParamUを初期化
#
proc initParamU {{clean 0} {recList {}}} {
  global paramU v paramUsize t

  if {[llength $recList] <= 0} {
    set recList $v(recList)
  }

  array unset paramU
  set paramU(0,0) $t(initParamU,0)
  set paramU(0,1) $t(initParamU,1)
  set paramU(0,2) $t(initParamU,2)
  set paramU(0,3) $t(initParamU,3)
  set paramU(0,4) $t(initParamU,4)
  set paramU(0,5) $t(initParamU,5)
  set paramU(0,6) $t(initParamU,6)
  set paramUsize 1
  if $clean return  ;# もし配列サイズを0にする初期化ならここで終了
  for {set i 0} {$i < [llength $recList]} {incr i} {
    set fid [lindex $recList $i]

    ;# 表に表示するデータを設定
    set paramU($paramUsize,0) $fid

    ;# 内部で参照するデータを設定
    set paramU($paramUsize,R) $i    ;# 行番号→recListの配列番号

    incr paramUsize
  }
  ;# 一覧表のサイズを更新する
  #if [winfo exists .entpwindow] {
  #  .entpwindow.t configure -rows $paramUsize
  #}
}

#---------------------------------------------------
# tk_getSaveFile 用のラッパー(主にmac対応のため)
# macでは-initialfileに絶対パスで存在しないファイルを指定するとエラーをおこした
#
proc my_getSaveFile {{args {}}} {
  global topdir
  array set a $args

  if {[array names a "-initialfile"] != "" && [regexp {/} $a(-initialfile)]} {
    set a(-initialdir)  [file dirname $a(-initialfile)]
    set a(-initialfile) [file tail    $a(-initialfile)]
  }

  if {[array names a "-initialdir"] == "" || ! [file exists $a(-initialdir)] } {
    set a(-initialdir) $topdir
  }
  set command [join "tk_getSaveFile [array get a]" " "]
  eval $command
}

#---------------------------------------------------
# tk_getOpenFile 用のラッパー(主にmac対応のため)
# macでは-initialfileに絶対パスで存在しないファイルを指定するとエラーをおこした
#
proc my_getOpenFile {{args {}}} {
  global topdir
  array set a $args

  if {[array names a "-initialfile"] != "" && [regexp {/} $a(-initialfile)]} {
    set a(-initialdir)  [file dirname $a(-initialfile)]
    set a(-initialfile) [file tail    $a(-initialfile)]
  }

  if {[array names a "-initialdir"] == "" || ! [file exists $a(-initialdir)] } {
    set a(-initialdir) $topdir
  }

  if {[array names a "-initialfile"] != "" && \
      ! [file exists "$a(-initialdir)$a(-initialfile)"]} {
    unset a(-initialfile)
  }

  if {$::tcl_platform(os) == "Darwin" && [array names a "-filetypes"] != ""} {
    unset a(-filetypes)   ;# macだとエラーが出たので
  }

  set command [join "tk_getOpenFile [array get a]" " "]
  eval $command
}

#---------------------------------------------------
# tk_chooseDirectory 用のラッパー(主にmac対応のため)
# macでは-initialdirに存在しないファイルを指定するとエラーをおこした
#
proc my_chooseDirectory {{args {}}} {
  global topdir
  array set a $args

  if {[array names a "-initialdir"] == "" || ! [file exists $a(-initialdir)] } {
    set a(-initialdir) $topdir
  }

  set command [join "tk_chooseDirectory [array get a]" " "]
  eval $command
}

