#ifndef POPT_HPP
#define POPT_HPP

#include <vector>
#include <map>
#include <string>
#include <stdexcept>

template <typename C> class popts {
public:
  /*
   * amap[short_name] = long_name 
   * cmap[long_name] = index
   * nmap[short_name] = index // short_name only
   * vmap[index] = value
   */
  std::map<std::basic_string<C>, int> cmap;
  std::map<std::basic_string<C>, int> nmap;
  std::map<std::basic_string<C>, 
   std::vector<std::basic_string<C> > > vmap;
  std::map<std::basic_string<C>, std::basic_string<C> > amap;
  std::vector<std::basic_string<C> > args;
private:
  typedef typename std::map<std::basic_string<C>,
   int>::iterator cmapit;
  typedef typename std::map<std::basic_string<C>,
   int>::iterator nmapit;
  typedef typename std::map<std::basic_string<C>, 
   std::vector<std::basic_string<C> > >::iterator vmapit;
  typedef typename std::map<std::basic_string<C>,
   std::basic_string<C> >::iterator amapit; 

  static bool checkspace(C c) {
    return ((c == ' ') || (c == '\t'));
  }
  static bool checkspace(const std::basic_string<C> &s) {
    for (size_t i = 0; i < s.size(); i++) {
      if (checkspace(s[i])) return true;
    }
    return false; 
  }
  static std::basic_string<C> escapequote(const std::basic_string<C> &s) {
    std::basic_string<C> d;
    size_t i = 0;
    while (i < s.size()) {
      if (s[i] == C('\\')) {
        size_t n = 0;
	while ((i + n) < s.size()) {
	  if (s[i + n] != '\\') break;
	  n++;
	}
	if ((i + n) == s.size()) { /* end of quote(next = '"') */
	  d += std::basic_string<C>(n * 2, C('\\'));
	} else if (s[i + n] == C('"')) {
	  d += std::basic_string<C>(n * 2, C('\\'));
	} else {
	  d += std::basic_string<C>(n, C('\\'));
	}
	i += n;
      } else if (s[i] == C('"')) {
	d += C('\\');
	d += C('"');
	i++;
      } else {
        d += s[i];
	i++;
      }
    }
    return d; 
  }
  int check_long(const std::basic_string<C> &s,
   cmapit &cit, vmapit &vit) {
    cit = cmap.find(s);
    if (cit == cmap.end()) return 0; /* not found */
    vit = vmap.find(s);
    if (vit == vmap.end()) return 1; /* found without value */
    return 2; /* found with value */
  }
  int check_short(const std::basic_string<C> &s,
   nmapit &nit, vmapit &vit) {
    nit = nmap.find(s);
    if (nit == nmap.end()) return 0; /* not found */
    vit = vmap.find(s);
    if (vit == vmap.end()) return 1; /* found without value */
    return 2; /* found with value */
  }
  void store_long(
   const std::basic_string<C> &s,
   const std::basic_string<C> &v,
   cmapit &cit, vmapit &vit) {
    if (cit != cmap.end()) {
      cit->second++;
      if (vit != vmap.end()) {
        if (v.size()) {
          vit->second.push_back(v);
        }
      }
    }
  }
  void store_short(
   const std::basic_string<C> &s,
   const std::basic_string<C> &v,
   cmapit &nit, vmapit &vit) {
    if (nit != nmap.end()) {
      nit->second++;
      if (vit != vmap.end()) {
        if (v.size()) {
          vit->second.push_back(v);
        }
      }
    }
  }
  int fetch(int argc, C *argv[], int pos) {
    std::basic_string<C> s = argv[pos];
    std::basic_string<C> v;
    cmapit cit;
    vmapit vit;
    if ((s.size() > 2) && (s[0] == C('-')) && (s[1] == C('-'))) {
      s = s.substr(2);
      int n = check_long(s, cit, vit);
      if (n == 0) return -1;
      size_t m = s.find(C('='));
      if (m != std::basic_string<C>::npos) {
        if ((m == 0) || (m == (s.size() - 1))) return -1; 
        v = s.substr(m + 1);
        s = s.substr(0, m);
	if (n != 2) return -1;
      }
      if (n == 1) {
        store_long(s, v, cit, vit);
	return +1; 
      } else if ((n == 2) && v.size()) {
        store_long(s, v, cit, vit);
	return +1; 
      } else if ((pos + 1) < argc) {
        store_long(s, argv[pos + 1], cit, vit);
	return +2;
      } else {
        return -1;
      }
    } else if ((s.size() > 1) && 
     ((s[0] == C('-')) || (s[0] == C('/')))) {
      s = s.substr(1);
      for (size_t i = 0; i < s.size(); i++) {
        auto ait = amap.find(s.substr(i, 1));
	if (ait != amap.end()) {
	  std::basic_string<C> l = ait->second;
	  int n = check_long(l, cit, vit);
	  if (n == 0) {
            return -1;
	  } else if (((i + 1) < s.size()) && (s[i + 1] == C('='))) {
            v = s.substr(i + 2);
	    if (n != 2) return -1;
	    if (!v.size()) return -1;
	    store_long(l, v, cit, vit);
	    return +1;
	  } else if (((i + 1) < s.size()) && (n == 2)) {
            v = s.substr(i + 1);
	    if (!v.size()) return -1;
	    store_long(l, v, cit, vit);
	    return +1;
	  } else if (((i + 1) == s.size()) && (n == 2)) {
	    if (!((pos + 1) < argc)) return -1;
	    store_long(l, argv[pos + 1], cit, vit);
	    return +2;
	  } else if (n == 2) {
            v = s.substr(1);
	    if (!v.size()) return -1;
	    store_long(l, s.substr(1), cit, vit);
	    return +1;
	  } else if (n == 1) {
	    v.clear();
	    store_long(s.substr(i, 1), v, cit, vit);
	    // not return
	  } else {
	    return -1;
	  }
	} else {
          auto nit = nmap.find(s.substr(i, 1));
          std::basic_string<C> l = s.substr(0, 1);
	  int n = check_short(l, nit, vit);
	  if (n == 0) {
            return -1;
	  } else if (((i + 1) < s.size()) && (s[i + 1] == C('='))) {
            v = s.substr(i + 2);
	    if (n != 2) return -1;
	    if (!v.size()) return -1;
	    store_short(l, v, nit, vit);
	    return +1;
	  } else if (((i + 1) < s.size()) && (n == 2)) {
            v = s.substr(i + 1);
	    if (!v.size()) return -1;
	    store_short(l, v, nit, vit);
	    return +1;
	  } else if (((i + 1) == s.size()) && (n == 2)) {
	    if (!((pos + 1) < argc)) return -1;
	    store_short(l, argv[pos + 1], nit, vit);
	    return +2;
	  } else if (n == 2) {
            v = s.substr(1);
	    if (!v.size()) return -1;
	    store_short(l, s.substr(1), nit, vit);
	    return +1;
	  } else if (n == 1) {
	    v.clear();
	    store_short(l, v, nit, vit);
	    // not return
	  } else {
	    return -1;
	  }
	  if (n == 0) {
            return -1;
          } 
	  return -1;
	}
      }
      return +1;
    }
    return 0;
  }
  int buildargs(int argc, C *argv[], int o = 1) {
    for (int i = o; i < argc; i++) {
      args.push_back(argv[i]);
    }
    return 0;
  }
public:
  static int split(std::vector<std::basic_string<C> > &v,
   const std::basic_string<C> &s) {
    std::basic_string<C> buff;
    bool quote = false;
    size_t i = 0;
    while (i < s.size()) {
      if (!quote && checkspace(s[i])) {
        if (i > 0) {
	  v.push_back(buff);
	  buff.clear();
	}
	while ((i < s.size()) && checkspace(s[i])) i++;
      } else if (s[i] == C('"')
       && (quote && (i < (s.size() - 1)) && (s[i + 1] == C('"')))) {
        buff += C('"');
	i += 2;
      } else if (s[i] == C('"')) {
        quote = !quote;
	i++;
      } else if (s[i] == C('\\')) {
	size_t n = 0;
	for (n = 0; (i + n) < s.size(); n++) {
	  if (s[i + n] != C('\\')) break;
	}
	if (((i + n) < s.size()) && (s[i + n] == C('"'))) {
	  if (n / 2) {
	    buff += std::basic_string<C>(n / 2, C('\\'));
	    i += n;
	  }
	  if (n & 1) {
	    buff += C('"');
	    i++;
	  }
	} else {
	  buff += std::basic_string<C>(n, C('\\'));
	  i += n;
	}
      } else {
        buff += s[i];
	i++;
      }
    }
    if (buff.size()) v.push_back(buff);
    return 0;
  }
  std::basic_string<C> join(int offset = 0) {
    std::basic_string<C> d;
    for (int i = offset; i < args.size(); i++) {
      if (d.size()) d += C(' ');
      if (checkspace(args[i])) {
        d += C('"');
	d += escapequote(args[i]);
        d += C('"');
      } else {
	d += args[i];
      }
    }
    return d;
  }
  void option(const std::basic_string<C> &name, bool value = false) {
    std::basic_string<C> l = name;
    std::basic_string<C> s;
    size_t pos = l.find(C(':'));
    if (pos != std::basic_string<C>::npos) {
      s = l.substr(pos + 1);
      l = l.substr(0, pos);
    }
    if (l.size()) {
      if (value) vmap[l].clear();
      cmap[l] = 0;
      for (size_t i = 0; i < s.size(); i++) { amap[s.substr(i, 1)] = l; }
    } else { // short only
      for (size_t i = 0; i < s.size(); i++) { 
        std::basic_string<C> key = s.substr(i, 1);
        if (value) vmap[key].clear();
        nmap[key] = 0;
      }
    }
  }
  bool check(const std::basic_string<C> &name) {
    std::basic_string<C> key = name;
    auto ait = amap.find(key);
    if (ait != amap.end()) key = ait->second;
    auto cit = cmap.find(key);
    if (cit != cmap.end()) return (cit->second > 0);
    auto nit = nmap.find(key);
    if (nit != nmap.end()) return (nit->second > 0);
    return false;
  }
  std::vector<std::basic_string<C> > &value(
   const std::basic_string<C> &name) {
    std::basic_string<C> key = name;
    auto ait = amap.find(key);
    if (ait != amap.end()) key = ait->second;
    if (check(key)) {
      auto vit = vmap.find(key);
      if (vit != vmap.end()) return vit->second;
    }
    throw std::runtime_error("bad option-value");
  }
  size_t count_values(const std::basic_string<C> &name) {
    if (check(name)) return value(name).size();
    return 0;
  }
  int parse(int argc, C *argv[], int o = 1) {
    for (int i = o, n = 0; i < argc; i += n) {
      std::basic_string<C> s = argv[i];
      if ((s.size() == 2) && (s[0] == C('-')) && (s[1] == C('-'))) {
        return buildargs(argc, argv, i + 1); // END
      } else if ((s.size() == 1) && (s[0] == C('-'))) {
        return buildargs(argc, argv, i); // END
      } else {
        n = fetch(argc, argv, i);
        if (n < 0) { // ERROR
          return i + 1;
        } else if (n == 0) {
          return buildargs(argc, argv, i); // END
	}
      }
    }
    return 0;
  }
  int parse(const std::basic_string<C> &cmd, int offset = 1) {
    std::vector<std::basic_string<C> > args;
    split(args, cmd);
    size_t m = 0;
    for (size_t i = 0; i < args.size(); i++) {
      m += (args[i].size() + 1);
    }
    std::vector<C> str(m);
    std::vector<C *> vec(args.size() + 1);
    size_t n = 0;
    for (size_t i = 0; i < args.size(); i++) {
      vec[i] = &(str[n]);
      memcpy(&(str[n]), args[i].c_str(), 
       sizeof (C) * (args[i].size() + 1));
      n += (args[i].size() + 1);
    }
    vec[vec.size() - 1] = NULL;
    /*
    for (size_t i = 0; vec[i]; i++) {
      wprintf(L"DEBUG:argv[%d]=\"%s\"\n", i, vec[i]);
    }
    */
    return parse(args.size(), &(vec[0]), offset);
  }
};

#if defined(WIN32) && defined(WINSYS_HPP)
#include "winsys.hpp"

class popt_tools {
public:
  static void exwild(std::vector<std::wstring> &args, size_t o = 0) {
    std::vector<std::wstring> exps;
    for (size_t i = o; i < args.size(); i++) {
      winsys::dirinf_w ent(args[i]);
      int n = 0;
      while (ent.step()) {
        if ((wcscmp(ent.wfd.cFileName, L"..") == 0)
         || (wcscmp(ent.wfd.cFileName, L".")) == 0) {
	  continue;
	}
	exps.push_back(ent.wfd.cFileName);
        n++;
      }
      if (n == 0) { exps.push_back(args[i]); } 
    }
    args = exps;
  }
  static void exwild(std::vector<std::string> &args, size_t o = 0) {
    std::vector<std::string> exps;
    for (size_t i = o; i < args.size(); i++) {
      winsys::dirinf_a ent(args[i]);
      int n = 0;
      while (ent.step()) {
        if ((strcmp(ent.wfd.cFileName, "..") == 0)
         || (strcmp(ent.wfd.cFileName, ".")) == 0) {
	  continue;
	}
	exps.push_back(ent.wfd.cFileName);
        n++;
      }
      if (n == 0) { exps.push_back(args[i]); } 
    }
    args = exps;
  }
};
#endif

#endif
