/*
 * NPlot - A charting library for .NET
 * 
 * PDF.PlotSurface2D.cs
 * Copyright (C) 2003-2006 Matt Howlett and others.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of NPlot nor the names of its contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * patch for writing to PDF using iTextSharp.
 * Copyright (C) 2006-2007 mocchi
 */

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections;

namespace NPlot {

	namespace PDF {

		/// <summary>
		/// Wrapper around NPlot.PlotSurface2D that provides extra functionality
		/// specific to drawing to Bitmaps.
		/// </summary>
		public class PlotSurface2D: IPlotSurface2D {

			/// <summary>
			/// Constructor.
			/// </summary>
			/// <param name="cb">PDFContextByte</param>
			/// <param name="bf">BaseFont</param>
			public PlotSurface2D( iTextSharp.text.pdf.PdfContentByte cb, iTextSharp.text.pdf.BaseFont bf ) {
				oc_ = cb;
				bf_ = bf;
				ps_ = new NPlot.PlotSurface2D(this);
			}

			/// <summary>
			/// Clears the plot.
			/// </summary>
			public void Clear() {
				ps_.Clear();
			}


			/// <summary>
			/// Adds a drawable object to the plot surface. If the object is an IPlot, 
			/// the PlotSurface2D axes will also be updated.
			/// </summary>
			/// <param name="p">The IDrawable object to add to the plot surface.</param>
			public void Add( IDrawable p ) {
				ps_.Add( p );
			}


			/// <summary>
			/// Adds a drawable object to the plot surface against the specified axes. If
			/// the object is an IPlot, the PlotSurface2D axes will also be updated.
			/// </summary>
			/// <param name="p">the IDrawable object to add to the plot surface</param>
			/// <param name="xp">the x-axis to add the plot against.</param>
			/// <param name="yp">the y-axis to add the plot against.</param>
			public void Add( IDrawable p, NPlot.PlotSurface2D.XAxisPosition xp, NPlot.PlotSurface2D.YAxisPosition yp ) {
				ps_.Add( p, xp, yp );
			}

			/// <summary>
			/// Adds a drawable object to the plot surface. If the object is an IPlot, 
			/// the PlotSurface2D axes will also be updated.
			/// </summary>
			/// <param name="p">The IDrawable object to add to the plot surface.</param>
			/// <param name="zOrder">The z-ordering when drawing (objects with lower numbers are drawn first)</param>
			public void Add( IDrawable p, int zOrder ) {
				ps_.Add( p, zOrder );
			}


			/// <summary>
			/// Adds a drawable object to the plot surface against the specified axes. If
			/// the object is an IPlot, the PlotSurface2D axes will also be updated.
			/// </summary>
			/// <param name="p">the IDrawable object to add to the plot surface</param>
			/// <param name="xp">the x-axis to add the plot against.</param>
			/// <param name="yp">the y-axis to add the plot against.</param>
			/// <param name="zOrder">The z-ordering when drawing (objects with lower numbers are drawn first)</param>
			public void Add( IDrawable p, NPlot.PlotSurface2D.XAxisPosition xp,
				NPlot.PlotSurface2D.YAxisPosition yp, int zOrder ) {
				ps_.Add( p, xp, yp , zOrder);
			}

			/// <summary>
			/// The plot surface title.
			/// </summary>
			public string Title {
				get {
					return ps_.Title;
				}
				set {
					ps_.Title = value;
				}
			}


			/// <summary>
			/// The plot title font.
			/// </summary>
			public Font TitleFont {
				get {
					return ps_.TitleFont;
				}
				set {
					ps_.TitleFont = value;
				}
			}


			/// <summary>
			/// The distance in pixels to leave between of the edge of the bounding rectangle
			/// supplied to the Draw method, and the markings that make up the plot.
			/// </summary>
			public int Padding {
				get {
					return ps_.Padding;
				}
				set {
					ps_.Padding = value;
				}
			}


			/// <summary>
			/// The bottom abscissa axis.
			/// </summary>
			public Axis XAxis1 {
				get {
					return ps_.XAxis1;
				}
				set {
					ps_.XAxis1 = value;
				}
			}


			/// <summary>
			/// The left ordinate axis.
			/// </summary>
			public Axis YAxis1 {
				get {
					return ps_.YAxis1;
				}
				set {
					ps_.YAxis1 = value;
				}
			}


			/// <summary>
			/// The top abscissa axis.
			/// </summary>
			public Axis XAxis2 {
				get {
					return ps_.XAxis2;
				}
				set {
					ps_.XAxis2 = value;
				}
			}


			/// <summary>
			/// The right ordinate axis.
			/// </summary>
			public Axis YAxis2 {
				get {
					return ps_.YAxis2;
				}
				set {
					ps_.YAxis2 = value;
				}
			}


			/// <summary>
			/// Gets or Sets the legend to use with this plot surface.
			/// </summary>
			public NPlot.Legend Legend {
				get {
					return ps_.Legend;
				}
				set {
					ps_.Legend = value;
				}
			}

			/// <summary>
			/// Gets or Sets the legend z-order.
			/// </summary>
			public int LegendZOrder {
				get {
					return ps_.LegendZOrder;
				}
				set {
					ps_.LegendZOrder = value;
				}
			}

			/// <summary>
			/// A color used to paint the plot background. Mutually exclusive with PlotBackImage and PlotBackBrush
			/// </summary>
			public System.Drawing.Color PlotBackColor {
				set {
					ps_.PlotBackColor = value;
				}
			}


			/// <summary>
			/// An imaged used to paint the plot background. Mutually exclusive with PlotBackColor and PlotBackBrush
			/// </summary>
			public System.Drawing.Bitmap PlotBackImage {
				set {
					ps_.PlotBackImage = value;
				}
			}


			/// <summary>
			/// A Rectangle brush used to paint the plot background. Mutually exclusive with PlotBackColor and PlotBackBrush
			/// </summary>
			public IRectangleBrush PlotBackBrush {
				set {
					ps_.PlotBackBrush = value;
				}
			}


			/// <summary>
			/// Smoothing mode to use when drawing plots.
			/// </summary>
			public System.Drawing.Drawing2D.SmoothingMode SmoothingMode { 
				get {
					return ps_.SmoothingMode;
				}
				set {
					ps_.SmoothingMode = value;
				}
			}

			/// <summary>
			/// The bitmap background color outside the bounds of the plot surface.
			/// </summary>
			public Color BackColor {
				set {
					backColor_ = value;
				}
			}
			object backColor_ = null;
			

			/// <summary>
			/// Refreshes (draws) the plot.
			/// </summary>
			public void Refresh() {
				GraphicsPdfContentByte gcb_ = new GraphicsPdfContentByte(oc_, bf_);
				float width = oc_.PdfDocument.PageSize.Width, height = oc_.PdfDocument.PageSize.Height;
				if (this.backColor_!=null) {
					gcb_.FillRectangle( (new Pen( (Color)this.backColor_)).Brush,0,0,width,height );
				}
				ps_.Draw( gcb_, new System.Drawing.Rectangle(0,0,(int)width,(int)height) );
				gcb_.Dispose();
			}


			private NPlot.PlotSurface2D ps_;
			private iTextSharp.text.pdf.PdfContentByte oc_;
			private iTextSharp.text.pdf.BaseFont bf_;

			
			/// <summary>
			/// Add an axis constraint to the plot surface. Axis constraints can
			/// specify relative world-pixel scalings, absolute axis positions etc.
			/// </summary>
			/// <param name="c">The axis constraint to add.</param>
			public void AddAxesConstraint( AxesConstraint c ) {
				ps_.AddAxesConstraint( c );
			}


			/// <summary>
			/// Whether or not the title will be scaled according to size of the plot 
			/// surface.
			/// </summary>
			public bool AutoScaleTitle {
				get {
					return ps_.AutoScaleTitle;
				}
				set {
					ps_.AutoScaleTitle = value;
				}
			}


			/// <summary>
			/// When plots are added to the plot surface, the axes they are attached to
			/// are immediately modified to reflect data of the plot. If 
			/// AutoScaleAutoGeneratedAxes is true when a plot is added, the axes will
			/// be turned in to auto scaling ones if they are not already [tick marks,
			/// tick text and label size scaled to size of plot surface]. If false,
			/// axes will not be autoscaling.
			/// </summary>
			public bool AutoScaleAutoGeneratedAxes {
				get {
					return ps_.AutoScaleAutoGeneratedAxes;
				}
				set {
					ps_.AutoScaleAutoGeneratedAxes = value;
				}
			}


			/// <summary>
			/// Sets the title to be drawn using a solid brush of this color.
			/// </summary>
			public Color TitleColor {
				set {
					ps_.TitleColor = value;
				}
			}


			/// <summary>
			/// The brush used for drawing the title.
			/// </summary>
			public Brush TitleBrush {
				get {
					return ps_.TitleBrush;
				}
				set {
					ps_.TitleBrush = value;
				}
			}

			/// <summary>
			/// Remove a drawable object from the plot surface.
			/// </summary>
			/// <param name="p">the drawable to remove</param>
			/// <param name="updateAxes">whether or not to update the axes after removing the idrawable.</param>
			public void Remove(IDrawable p, bool updateAxes) {
				ps_.Remove(p, updateAxes);
			}


			/// <summary>
			/// Gets an array list containing all drawables currently added to the PlotSurface2D.
			/// </summary>
			public ArrayList Drawables {
				get {
					return ps_.Drawables;
				}
			}

			/// <summary>
			/// This returns IGraphics of GraphicsPdfContentByte.
			/// </summary>
			public IGraphics Graphics{
				get {
					return new GraphicsPdfContentByte(oc_, bf_);
				}
			}

		}
	}
}
