/* File: input.c */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/ps2/dev.h>

#include "common.h"
#include "input.h"

unsigned int grp_key_buttons[GRP_GAMEDEV_MAX];
unsigned int grp_key_triggers[GRP_GAMEDEV_MAX];
int grp_key_analog1_v[GRP_GAMEDEV_MAX];
int grp_key_analog1_h[GRP_GAMEDEV_MAX];
int grp_key_analog2_v[GRP_GAMEDEV_MAX];
int grp_key_analog2_h[GRP_GAMEDEV_MAX];

static int _gps2_status[GRP_GAMEDEV_MAX];
static int _gps2_padid[GRP_GAMEDEV_MAX];
static int _gps2_padfd[GRP_GAMEDEV_MAX] = {-1, -1};
static int _gps2_padstatfd;
static unsigned int __grp_old_key_buttons[GRP_GAMEDEV_MAX] = {0U, 0U};

static void clear_port_info( void )
{
	int i;

	for (i = 0; i < GRP_GAMEDEV_MAX; i++) {
		grp_key_buttons[i]	= 0U;
		grp_key_analog1_v[i]	= 0;
		grp_key_analog1_h[i]	= 0;
		grp_key_analog2_v[i]	= 0;
		grp_key_analog2_h[i]	= 0;
	}
	return;
}

int __grp_gameport_init( void )
{
	int i;

	for (i = 0; i < GRP_GAMEDEV_MAX; i++)
		__grp_old_key_buttons[i] = 0;
	clear_port_info();
	if (_gps2_padstatfd < 0)
		_gps2_padstatfd = open( PS2_DEV_PADSTAT, O_RDWR | O_NONBLOCK );
	for (i = 0; i < GRP_GAMEDEV_MAX; i++) {
		if (_gps2_padfd[i] < 0)
			_gps2_padfd[i] = open( !i ? PS2_DEV_PAD0 : PS2_DEV_PAD1, 0 );
		_gps2_status[i] = 0;
		_gps2_padid[i]  = 0;
	}
	return TRUE;
}

int grp_gameport_read( void )
{
	int i, status, len;
	unsigned char pad_buf[PS2PAD_DATASIZE];
	struct ps2pad_modeinfo pad_modeinfo;
	struct ps2pad_mode padmode;

	clear_port_info();
	for (i = 0; i < GRP_GAMEDEV_MAX; i++) {
		if (_gps2_padfd[i] < 0)
			continue;
		if (ioctl( _gps2_padfd[i], PS2PAD_IOCGETSTAT, &status) < 0)
			continue;
		if (status == PS2PAD_STAT_NOTCON) {
			_gps2_status[i] = 0;
			continue;
		}
		switch (_gps2_status[i]) {
			case 0 :
				if (status != PS2PAD_STAT_READY)
					break;
				pad_modeinfo.term = PS2PAD_MODECURID;
				if (ioctl( _gps2_padfd[i], PS2PAD_IOCMODEINFO, &pad_modeinfo ) < 0)
					break;
				_gps2_padid[i] = pad_modeinfo.result;
				if (!_gps2_padid[i])
					break;
				pad_modeinfo.term = PS2PAD_MODECUREXID;
				if (ioctl( _gps2_padfd[i], PS2PAD_IOCMODEINFO, &pad_modeinfo ) < 0)
					break;
				if (pad_modeinfo.result <= 0)
					_gps2_status[i] = 9999;
				else {
					_gps2_padid[i] = pad_modeinfo.result;
					if (_gps2_padid[i] == 4)
						_gps2_status[i] = 100;
					else {
						if (_gps2_padid[i] == 7)
							_gps2_status[i] = 200;
						else
							_gps2_status[i] = 9999;
					}
				}
				break;
			case 100 : /* Normal controller or Dual shock normal mode */
				padmode.offs = 1;
				padmode.lock = 3;
				if (ioctl( _gps2_padfd[i], PS2PAD_IOCSETMODE,  &padmode ) < 0)
					break;
				_gps2_status[i] = 0;
				break;
			case 200 : /* Dual Shock or Dual Shock II */
				/* Act, press mode  */
				/* ioctl( _gps2_padfd[i], PS2PAD_IOCENTERPRESSMODE, NULL ); */
				_gps2_status[i] = 9999;
				break;
			default :
				if (_gps2_padid[i] != 4 && _gps2_padid[i] != 5 && _gps2_padid[i] != 7)
					break;

				/* Normal, Old analog, DualShock */
				len = read( _gps2_padfd[i], &pad_buf, PS2PAD_DATASIZE);
				if (len >= 4)
					grp_key_buttons[i] = ~((pad_buf[2] << 8) | pad_buf[3]);

				/* DualShock only */
				if (_gps2_padid[i] != 7 || len < 8)
					break;
				grp_key_analog1_h[i]	= pad_buf[6] - 127;
				grp_key_analog1_v[i]	= pad_buf[7] - 127;
				grp_key_analog2_h[i]	= pad_buf[4] - 127;
				grp_key_analog2_v[i]	= pad_buf[5] - 127;
				if ((grp_key_analog1_h[i] < GRP_GAMEDEV_ANALOG_ERROR_DIST) &&
						(-GRP_GAMEDEV_ANALOG_ERROR_DIST < grp_key_analog1_h[i]))
					grp_key_analog1_h[i] = 0;
				if ((grp_key_analog1_v[i] < GRP_GAMEDEV_ANALOG_ERROR_DIST) &&
						(-GRP_GAMEDEV_ANALOG_ERROR_DIST < grp_key_analog1_v[i]))
					grp_key_analog1_v[i] = 0;
				if ((grp_key_analog2_h[i] < GRP_GAMEDEV_ANALOG_ERROR_DIST) &&
						(-GRP_GAMEDEV_ANALOG_ERROR_DIST < grp_key_analog2_h[i]))
					grp_key_analog2_h[i] = 0;
				if ((grp_key_analog2_v[i] < GRP_GAMEDEV_ANALOG_ERROR_DIST) &&
						(-GRP_GAMEDEV_ANALOG_ERROR_DIST < grp_key_analog2_v[i]))
					grp_key_analog2_v[i] = 0;
				break;
		}
	}
	for (i = 0; i < GRP_GAMEDEV_MAX; i++) {
		grp_key_triggers[i] = grp_key_buttons[i] & (~__grp_old_key_buttons[i]);
		__grp_old_key_buttons[i] = grp_key_buttons[i];
	}
	return TRUE;
}
