/* File: font.c */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>

#include "common.h"
#include "draw.h"
#include "context.h"
#include "prim2d.h"
#include "tex.h"
#include "font.h"

#define GRD_FONT_BUFSIZE	4096

static struct grd_font_info {
	int attr;
	GRD_TEX_INFO texinfo;
	int size_w, size_h, w_chars;
	int buflen, bufct;
	unsigned char *buf, *cp;
} __grd_font[GRD_FONTENV_MAX];

char gri_default_fontpath[LNMAX] = "font.tm2";

int __grd_font_init( void )
{
	int i;

	for (i = 0; i < GRD_FONTENV_MAX; i++)
		__grd_font[i].attr = GRD_FONT_NONE;
	if (grd_font_loadadd( GRD_FONT_DYNAMIC, gri_default_fontpath, 8, 16, 16, -1 ) < 0) {
		ERRMSG("__grd_font_init() init error, font_loadadd()\n");
		EXIT(1);
	}
	return TRUE;
}

int grd_font_loadadd( int attr, char *path, int w, int h, int chars,
		int buflen )
{
	int i;

	/* allocation */
	for (i = 0; i < GRD_FONTENV_MAX; i++) {
		if (!__grd_font[i].attr)
			break;
	}
	if (i >= GRD_FONTENV_MAX)
		return -1;

	/* texture read */
	__grd_font[i].attr   = attr;
	__grd_font[i].size_w = w;
	__grd_font[i].size_h = h;
	__grd_font[i].w_chars = chars;
	if (!grd_tex_read( GRD_TRATTR_ERROREXIT, NULL, &__grd_font[i].texinfo, path )) {
		__grd_font[i].attr = GRD_FONT_NONE;
		return -1;
	}
	if (buflen <= 0)
		buflen = GRD_FONT_DEFAULT_BUFSIZE;
	__grd_font[i].buflen = buflen;
	__grd_font[i].buf = MALLOC( buflen );
	if (!__grd_font[i].buf) {
		ERRMSG("__grd_font_loadadd() memory allocation error\n");
		return -1;
	}
	grd_font_clear(i);
	return i;
}

void grd_font_clear( int fenvsel )
{
	__grd_font[fenvsel].cp    = __grd_font[fenvsel].buf;
	__grd_font[fenvsel].bufct = 0;
	return;
}

void grd_font_select( int fenvsel )
{
	if (__grd_font[fenvsel].bufct + 1 >= __grd_font[fenvsel].buflen - 1)
		return;
	__grd_font[fenvsel].bufct++;
	*(__grd_font[fenvsel].cp)++ = 0xf0;
	return;
}

void grd_font_setsize( int fenvsel, int fontsize )
{
	if (__grd_font[fenvsel].bufct + 3 >= __grd_font[fenvsel].buflen - 1)
		return;
	__grd_font[fenvsel].bufct += 3;
	*(__grd_font[fenvsel].cp)++ = 0xf1;
	*(__grd_font[fenvsel].cp)++ = fontsize & 0xff;
	*(__grd_font[fenvsel].cp)++ = fontsize >> 8;
	return;
}

void grd_font_setcolor( int fenvsel, int r, int g, int b, int a, int clut )
{
	if (__grd_font[fenvsel].bufct + 6 >= __grd_font[fenvsel].buflen - 1)
		return;
	__grd_font[fenvsel].bufct += 6;
	*(__grd_font[fenvsel].cp)++ = 0xf2;
	*(__grd_font[fenvsel].cp)++ = r / 2;
	*(__grd_font[fenvsel].cp)++ = g / 2;
	*(__grd_font[fenvsel].cp)++ = b / 2;
	*(__grd_font[fenvsel].cp)++ = a;
	*(__grd_font[fenvsel].cp)++ = clut;
	return;
}

void grd_font_locate( int fenvsel, int x, int y )
{
	if (__grd_font[fenvsel].bufct + 5 >= __grd_font[fenvsel].buflen - 1)
		return;
	__grd_font[fenvsel].bufct += 5;
	*(__grd_font[fenvsel].cp)++ = 0xf3;
	*(__grd_font[fenvsel].cp)++ = x & 0xff;
	*(__grd_font[fenvsel].cp)++ = x >> 8;
	*(__grd_font[fenvsel].cp)++ = y & 0xff;
	*(__grd_font[fenvsel].cp)++ = y >> 8;
	return;
}

int grd_font_puts( int fenvsel, char *s )
{
	while (*s) {
		if (__grd_font[fenvsel].bufct >= __grd_font[fenvsel].buflen - 1)
			break;
		*(__grd_font[fenvsel].cp)++ = *s++;
		__grd_font[fenvsel].bufct++;
	}
	return TRUE;
}

int grd_font_printf( int fenvsel, char *s, ... )
{
	int val;
	static char ___msg_buf[256];
	va_list arg_ptr;

	va_start( arg_ptr, s );
	val = vsprintf(___msg_buf, s, arg_ptr);
	va_end( arg_ptr );
	grd_font_puts( fenvsel, ___msg_buf );

	return val;
}

void __grd_font_framest( void )
{
	int fenvsel;

	for (fenvsel = 0; fenvsel < GRD_FONTENV_MAX; fenvsel++) {
		if (__grd_font[fenvsel].attr)
			grd_font_clear( fenvsel );
	}
	return;
}

void __grd_font_frameend( void )
{
	int fenvsel;
	int x, y, multiply, old_multiply, c;
	int r, g, b, a, clut;
	float u, v;
	unsigned char *p;

	for (fenvsel = 0; fenvsel < GRD_FONTENV_MAX; fenvsel++) {
		if (!__grd_font[fenvsel].attr)
			continue;

		/* initialize */
		x = 10;
		y = 10;
		clut = 0;
		multiply = 100;
		old_multiply = -1;
		r = g = b = a = 128;

		/* set buffer end */
		*(__grd_font[fenvsel].cp) = '\0';
		__grd_font[fenvsel].buf[__grd_font[fenvsel].buflen - 1] = '\0';

		/* display message */
		grd_set_currentdrawpath( GRDRAW_PATH1 );
		grd_set_currentcontext( GRDRAW_CONTEXT1 );
		grd_set_drawmode_anormal_znotest_znoupdate( grd_get_currentcontext() );
		grd_set_drawmode_texclamp( grd_get_currentcontext() );
		for (p = __grd_font[fenvsel].buf; (p != (__grd_font[fenvsel].cp)) && *p; ) {
			switch (c = (*p++)) {
				case 0xf0 :	/* font select */
					old_multiply = -1;
					/* no implemented */
					break;
				case 0xf1 :	/* font size */
					multiply = *p++;
					multiply = multiply | ((*p++) << 8);
					break;
				case 0xf2 :	/* color */
					r = *p++;
					g = *p++;
					b = *p++;
					a = *p++;
					clut = *p++;
					break;
				case 0xf3 :	/* cursor */
					x = *p++;
					x = x | ((*p++) << 8);
					y = *p++;
					y = y | ((*p++) << 8);
					break;
				case '\r' :
					break;
				case '\n' :
					y = y + (__grd_font[fenvsel].size_h*multiply/100);
					x = 10;
					break;
				default :
					u = ((c-' ') % __grd_font[fenvsel].w_chars) * (float)__grd_font[fenvsel].size_w;
					v = ((c-' ') / __grd_font[fenvsel].w_chars) * (float)__grd_font[fenvsel].size_h;
					if (multiply != old_multiply) {
						if (multiply == 100)
							grd_tex_use( &__grd_font[fenvsel].texinfo, GRTEX_OPT_POINT, 0, clut );
						else
							grd_tex_use( &__grd_font[fenvsel].texinfo, GRTEX_OPT_LINEAR_MAX, 0, clut );
					}
					old_multiply = multiply;
					grd_2d_usprite( x, y,
						x+(float)((__grd_font[fenvsel].size_w)*multiply)/100.0f,
						y+(float)((__grd_font[fenvsel].size_h)*multiply)/100.0f,
						r, g, b, a, u, v, u+(float)__grd_font[fenvsel].size_w, v+(float)__grd_font[fenvsel].size_h,
						GRDRAW_OPT_GOURAUD | GRDRAW_OPT_ALPHABLEND);
					x = x + __grd_font[fenvsel].size_w*multiply/100;
					break;
			}
		}
	}
	return;
}
