/* File: matrix.c */

#include "config.h"

#include <stdio.h>
#include <math.h>

#include "common.h"
#include "matrix.h"
#include "draw.h"

GRMatrix44 m_unitmatrix = {{
	{ 1.0f, 0.0f, 0.0f, 0.0f },
	{ 0.0f, 1.0f, 0.0f, 0.0f },
	{ 0.0f, 0.0f, 1.0f, 0.0f },
	{ 0.0f, 0.0f, 0.0f, 1.0f }
}};

#ifdef USE_TABLE_COSSIN
float __grm_sin_table[GRM_SINTABLE_MAX];
float __grm_cos_table[GRM_COSTABLE_MAX];
#endif

void __grm_init( void )
{
#ifdef USE_TABLE_COSSIN
	int i;

	for (i = 0; i < GRM_SINTABLE_MAX; i++)
		__grm_sin_table[i] = sinf( (((float)i-GRM_SINTABLE_MAX/2.0f)*GRMFPI*2.0f) / ((float)GRM_SINTABLE_MAX/2.0f) );
	for (i = 0; i < GRM_COSTABLE_MAX; i++)
		__grm_cos_table[i] = cosf( (((float)i-GRM_COSTABLE_MAX/2.0f)*GRMFPI*2.0f) / ((float)GRM_COSTABLE_MAX/2.0f) );
#endif
	return;
}

static void grm_rotmatrix44_vector3_common( GRMatrix44 *target_m, int mode, float xsin, float xcos, float ysin, float ycos, float zsin, float zcos )
{
	target_m->m[0].w = target_m->m[1].w = target_m->m[2].w
			 = target_m->m[3].x = target_m->m[3].y = target_m->m[3].z = 0.0f;
	target_m->m[3].w = 1.0f;

	switch (mode) {
		case GRM_XYZ :
			target_m->m[0].x = ycos*zcos;
			target_m->m[0].y = -ycos*zsin;
			target_m->m[0].z = -ysin;
			target_m->m[1].x = -xsin*ysin*zcos+xcos*zsin;
			target_m->m[1].y = xcos*zcos+xsin*ysin*zsin;
			target_m->m[1].z = -xsin*ycos;
			target_m->m[2].x = xcos*ysin*zcos+xsin*zsin;
			target_m->m[2].y = xsin*zcos-xcos*ysin*zsin;
			target_m->m[2].z = xcos*ycos;
			break;
		case GRM_ZYX :
			target_m->m[0].x = ycos*zcos;
			target_m->m[0].y = -xsin*ysin*zcos-xcos*zsin;
			target_m->m[0].z = -xcos*ysin*zcos+xsin*zsin;
			target_m->m[1].x = ycos*zsin;
			target_m->m[1].y = xcos*zcos-xsin*ysin*zsin;
			target_m->m[1].z = -xsin*zcos-xcos*ysin*zsin;
			target_m->m[2].x = ysin;
			target_m->m[2].y = xsin*ycos;
			target_m->m[2].z = xcos*ycos;
			break;
		default :
			ERRMSG("grm_rotmatrix44_vector3*() unsupported matrix rotation mode %d\n", mode);
	}
        return;
}

void grm_rotmatrix44_vector3( GRMatrix44 *target_m, GRVector3 *angles, int mode )
{
	float xsin, xcos, ysin, ycos, zsin, zcos;

	xsin = grm_tsinf( angles->x ); xcos = grm_tcosf( angles->x );
	ysin = grm_tsinf( angles->y ); ycos = grm_tcosf( angles->y );
	zsin = grm_tsinf( angles->z ); zcos = grm_tcosf( angles->z );
	grm_rotmatrix44_vector3_common( target_m, mode, xsin, xcos, ysin, ycos, zsin, zcos );
        return;
}

void grm_rotmatrix44_vector3_nt( GRMatrix44 *target_m, GRVector3 *angles, int mode )
{
	float xsin, xcos, ysin, ycos, zsin, zcos;

	xsin = sinf( angles->x ); xcos = cosf( angles->x );
	ysin = sinf( angles->y ); ycos = cosf( angles->y );
	zsin = sinf( angles->z ); zcos = cosf( angles->z );
	grm_rotmatrix44_vector3_common( target_m, mode, xsin, xcos, ysin, ycos, zsin, zcos );
        return;
}

void grm_translate_matrix44( GRMatrix44 *target_m, GRMatrix44 *mat, GRVector4 *vec )
{
	grm_vector4_a_vector4( &target_m->m[3], &mat->m[3], vec );
	if (target_m != mat)
		memcpy( &target_m->m, &mat->m, (sizeof( GRVector4 )*3) );
	return;
}

void grm_matrix44_m_matrix44( GRMatrix44 *target_m, GRMatrix44 *mat1, GRMatrix44 *mat2 )
{
	int i;

	for (i = 0; i < 4; i++) {
		target_m->m[i].x = mat1->m[i].x * mat2->m[0].x + mat1->m[i].y * mat2->m[1].x
				 + mat1->m[i].z * mat2->m[2].x + mat1->m[i].w * mat2->m[3].x;
		target_m->m[i].y = mat1->m[i].x * mat2->m[0].y + mat1->m[i].y * mat2->m[1].y
				 + mat1->m[i].z * mat2->m[2].y + mat1->m[i].w * mat2->m[3].y;
		target_m->m[i].z = mat1->m[i].x * mat2->m[0].z + mat1->m[i].y * mat2->m[1].z
				 + mat1->m[i].z * mat2->m[2].z + mat1->m[i].w * mat2->m[3].z;
		target_m->m[i].w = mat1->m[i].x * mat2->m[0].w + mat1->m[i].y * mat2->m[1].w
				 + mat1->m[i].z * mat2->m[2].w + mat1->m[i].w * mat2->m[3].w;
	}
	return;
}

void grm_vector4_m_matrix44( GRVector4 *target_v, GRVector4 *vec1, GRMatrix44 *mat1 )
{
	target_v->x = vec1->x * mat1->m[0].x + vec1->y * mat1->m[1].x
		    + vec1->z * mat1->m[2].x + vec1->w * mat1->m[3].x;
	target_v->y = vec1->x * mat1->m[0].y + vec1->y * mat1->m[1].y
		    + vec1->z * mat1->m[2].y + vec1->w * mat1->m[3].y;
	target_v->z = vec1->x * mat1->m[0].z + vec1->y * mat1->m[1].z
		    + vec1->z * mat1->m[2].z + vec1->w * mat1->m[3].z;
	target_v->w = vec1->x * mat1->m[0].w + vec1->y * mat1->m[1].w
		    + vec1->z * mat1->m[2].w + vec1->w * mat1->m[3].w;
        return;
}

void grm_matrix44_m_vector4( GRVector4 *target_v, GRMatrix44 *mat1, GRVector4 *vec1 )
{
	target_v->x = mat1->m[0].x * vec1->x + mat1->m[0].y * vec1->y
		    + mat1->m[0].z * vec1->z + mat1->m[0].w * vec1->w;
	target_v->y = mat1->m[1].x * vec1->x + mat1->m[1].y * vec1->y
		    + mat1->m[1].z * vec1->z + mat1->m[1].w * vec1->w;
	target_v->z = mat1->m[2].x * vec1->x + mat1->m[2].y * vec1->y
		    + mat1->m[2].z * vec1->z + mat1->m[2].w * vec1->w;
	target_v->w = mat1->m[3].x * vec1->x + mat1->m[3].y * vec1->y
		    + mat1->m[3].z * vec1->z + mat1->m[3].w * vec1->w;
        return;
}
