package org.dyndns.nuda.dynamic.compiler;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.SecureClassLoader;
import java.util.HashMap;
import java.util.Map;

import javax.tools.DiagnosticListener;
import javax.tools.FileObject;
import javax.tools.ForwardingJavaFileManager;
import javax.tools.JavaCompiler;
import javax.tools.JavaFileManager;
import javax.tools.JavaFileObject;
import javax.tools.JavaFileObject.Kind;

import org.dyndns.nuda.tools.util.StringUtil;

/**
 * 動的生成したJavaクラスを管理するためのファイルマネージャクラスです
 * 
 * @author nkoseki
 * 
 */
public class ClassFileManager extends
		ForwardingJavaFileManager<JavaFileManager> {
	private static Method		DEFINE_CLASS_METHOD				= null;
	private static String		DEFINE_CLASS_METHOD_NAME		= "defineClass";
	private static Class<?>[]	DEFINE_CLASS_METHOD_SIGNITURE	= new Class<?>[] {
			String.class, byte[].class, int.class, int.class	};
	
	public ClassFileManager(final JavaCompiler compiler,
			final DiagnosticListener<? super JavaFileObject> listener) {
		super(compiler.getStandardFileManager(listener, null, null));
	}
	
	/**
	 * クラス名/クラスオブジェクトマップ
	 */
	private static final Map<String, JavaClassObject>	map	= new HashMap<String, JavaClassObject>();
	
	@Override
	public JavaFileObject getJavaFileForOutput(final Location location,
			final String className, final Kind kind, final FileObject libling)
			throws IOException {
		JavaClassObject co = new JavaClassObject(className, kind);
		map.put(className, co);
		return co;
	}
	
	protected static ClassLoader	loader	= null;
	
	public static Class<?> loadClass(String className) throws ClassNotFoundException {
		if(loader == null) {
			String message = StringUtil.format("class[{}] is not defiend cause DynamicClassLoader is null.", className);
			throw new ClassNotFoundException(message);
		} else {
			return loader.loadClass(className);
		}
		
	}
	
	@Override
	public ClassLoader getClassLoader(final Location location) {
		if (loader == null) {
			// System.out.println("loader init");
			
			// コンテキストクラスローダを親として、動的生成クラスロード用のクラスローダを
			// 作成する
			loader = new Loader(Thread.currentThread().getContextClassLoader());
			Thread.currentThread().setContextClassLoader(loader);
		}
		return loader;
	}
	
	/**
	 * 動的生成されたクラスのためのクラスローダーです.<br />
	 * 
	 * @author nkoseki
	 * 
	 */
	private class Loader extends SecureClassLoader {
		
		public Loader(final ClassLoader loader) {
			super(loader);
			
			if (DEFINE_CLASS_METHOD == null) {
				try {
					Class<?> loaderClass = ClassLoader.class;
					DEFINE_CLASS_METHOD = loaderClass.getDeclaredMethod(
							DEFINE_CLASS_METHOD_NAME,
							DEFINE_CLASS_METHOD_SIGNITURE);
					DEFINE_CLASS_METHOD.setAccessible(true);
				} catch (SecurityException e) {
					e.printStackTrace();
				} catch (NoSuchMethodException e) {
					e.printStackTrace();
				}
			}
			
		}
		
		@Override
		public Class<?> findClass(final String name)
				throws ClassNotFoundException {
			System.out.println("findClass:" + name);
			JavaClassObject co = map.get(name);
			Class<?> c = null;
			if (co == null) {
				// System.out.println(name);
				if (exists(name)) {
					// クラスファイルがすでにファイルシステムに存在すれば
					// ファイルシステムからクラスロードを行う
					
					byte[] b = null;
					
					// クラスファイル読み込み処理
					String basePath = System
							.getProperty(CompileTask.APP_PROP_CLASS_PATH);
					basePath = basePath + "\\";
					String nativeClassPathName = name.replace(".", "\\");
					basePath = basePath + nativeClassPathName
							+ Kind.CLASS.extension;
					
					File classFile = new File(basePath);
					
					int len = (int) classFile.length();
					b = new byte[len];
					try {
						FileInputStream in = new FileInputStream(classFile);
						
						DataInputStream dis = new DataInputStream(in);
						
						dis.read(b);
						
					} catch (FileNotFoundException e) {
						throw new ClassNotFoundException(
								"指定されたクラス名が見つかりませんでした", e);
					} catch (IOException e) {
						throw new ClassNotFoundException("I/O処理に失敗しました", e);
					}
					
					try {
						c = (Class<?>) DEFINE_CLASS_METHOD.invoke(getParent(),
								new Object[] { name, b, 0, b.length });
						
					} catch (SecurityException e) {
						throw new ClassNotFoundException(
								"クラスロードエラー：セキュリティ違反が発生しました", e);
					} catch (IllegalArgumentException e) {
						throw new ClassNotFoundException(
								"クラスロードエラー：システムエラーが発生しました(SYS_ERR_C1001)", e);
					} catch (IllegalAccessException e) {
						throw new ClassNotFoundException(
								"クラスロードエラー：システムエラーが発生しました(SYS_ERR_C1002)", e);
					} catch (InvocationTargetException e) {
						throw new ClassNotFoundException(
								"クラスロードエラー：システムエラーが発生しました(SYS_ERR_C1003)", e);
					}
					
					co = new JavaClassObject(name, Kind.CLASS);
					map.put(name, co);
					
					co.setDefinedClass(c);
					
					return c;
				} else {
					return super.findClass(name);
				}
				
			}
			
			c = co.getDefinedClass();
			
			if (c == null) {
				byte[] b = co.getBytes();
				
				try {
					c = (Class<?>) DEFINE_CLASS_METHOD.invoke(getParent(),
							new Object[] { name, b, 0, b.length });
					
				} catch (SecurityException e) {
					throw new ClassNotFoundException(
							"クラスロードエラー：セキュリティ違反が発生しました", e);
				} catch (IllegalArgumentException e) {
					throw new ClassNotFoundException(
							"クラスロードエラー：システムエラーが発生しました(SYS_ERR_C1001)", e);
				} catch (IllegalAccessException e) {
					throw new ClassNotFoundException(
							"クラスロードエラー：システムエラーが発生しました(SYS_ERR_C1002)", e);
				} catch (InvocationTargetException e) {
					throw new ClassNotFoundException(
							"クラスロードエラー：システムエラーが発生しました(SYS_ERR_C1003)", e);
				}
				
				co.setDefinedClass(c);
				outputFileSystem(c, b);
			}
			
			return c;
		}
		
		private void outputFileSystem(final Class<?> cls,
				final byte[] classBinaly) {
			
			String rootPath = System
					.getProperty(CompileTask.APP_PROP_CLASS_PATH);
			
			String subPath = cls.getName().replace(cls.getSimpleName(), "")
					.replace(".", "\\");
			
			File rootDir = new File(rootPath + "\\" + subPath);
			boolean makeFlg = rootDir.mkdirs();
			if (makeFlg || rootDir.exists()) {
				// クラスファイルをファイルシステムに書き出す
				ByteArrayOutputStream baos = new ByteArrayOutputStream();
				// ファイルストリーム作成
				try {
					File outputClassFile = new File(rootPath + "\\" + subPath
							+ "\\" + cls.getSimpleName() + Kind.CLASS.extension);
					
					System.out.println("outputFileSystem:" + cls.getCanonicalName() + " -> " + rootPath + "\\" + subPath
							+ "\\" + cls.getSimpleName() + Kind.CLASS.extension);
					
					if (outputClassFile.exists()) {
						boolean deleteFlg = outputClassFile.delete();
						if (!deleteFlg) {
							System.out.println("古いクラスファイルを削除できませんでした");
						}
					}
					
					FileOutputStream fos = new FileOutputStream(outputClassFile);
					baos.write(classBinaly);
					
					baos.writeTo(fos);
					
					fos.close();// ファイルを閉じる
					baos.reset();// ストリームを消去
					baos.close();// ストリームを閉じる
					
				} catch (FileNotFoundException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}
			} else {
				System.out.println("ディレクトリを作成できませんでした：" + rootDir.exists());
			}
		}
	}
	
	/**
	 * 指定したクラス名のクラスがアプリケーションクラスパスに存在するかテストします
	 * 
	 * @param className
	 * @return 存在チェック結果
	 */
	public boolean exists(final String className) {
		// クラスファイル読み込み処理
		String basePath = System.getProperty(CompileTask.APP_PROP_CLASS_PATH);
		basePath = basePath + "\\";
		String nativeClassPathName = className.replace(".", "\\");
		basePath = basePath + nativeClassPathName + Kind.CLASS.extension;
		
		File classFile = new File(basePath);
		return classFile.exists();
	}
}
