package org.dyndns.nuda.dynamic.compiler;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.tools.DiagnosticListener;
import javax.tools.JavaCompiler;
import javax.tools.JavaCompiler.CompilationTask;
import javax.tools.JavaFileObject;
import javax.tools.ToolProvider;

/**
 * Javaソースを動的コンパイルするためのクラスです. <br />
 * 
 * このタスクを動作させるためには、以下のプロパティキーが必要です.<br />
 * 
 * <b>[java.class.path] システム本体のJavaクラスパス</b><br />
 * <b>[app.class.path] コンパイル対象クラスのためのクラスパス</b><br />
 * <b>[app.source.path] コンパイル対象ソースが保存されるパスのルート</b><br />
 * 
 * ※このクラスを用いて動的コンパイルを実施するには、tools.jarが必要です
 * 
 * @author nkoseki
 * 
 */
public class CompileTask {

	/**
	 * コンパイルエラーを検出した際の委譲先リスナ
	 */
	protected DiagnosticListener<? super JavaFileObject> listener = new ErrorListener();

	/**
	 * コンパイラ
	 */
	protected JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();

	/**
	 * コンパイラ引数：クラスパス
	 */
	public static final String ARGS_CLASSPATH = "-classpath";

	/**
	 * システムプロパティキー：クラスパス
	 */
	public static final String SYS_PROP_CLASS_PATH = "java.class.path";

	/**
	 * システムプロパティキー：動的生成クラスパス
	 */
	public static final String APP_PROP_CLASS_PATH = "app.class.path";

	/**
	 * システムプロパティキー：動的生成ソースパス
	 */
	public static final String APP_PROP_SOURCE_PATH = "app.source.path";

	/**
	 * システムプロパティキー：強制的に再コンパイルを行うかを表す値
	 */
	public static final String APP_REVERT = "app.revert";

	/**
	 * Javaソースのコンパイルを行います
	 * 
	 * @param <T>
	 * @param source
	 * @return コンパイル結果
	 */
	public <T> Class<T> compile(final SourceBean source) {
		JavaFileObject fo = new JavaSourceFromString(source);
		return this.compileInner(fo, source.getClassName(),
				source.getSourceCode());
	}

	/**
	 * Javaソースのコンパイルを行います
	 * 
	 * @param <T>
	 * @param resolver
	 * @return コンパイル結果
	 */
	public <T> Class<T> compile(final SourceResolver resolver) {
		SourceBean sourceBean = resolver.resolve();
		JavaFileObject fo = new JavaSourceFromString(sourceBean);

		return this.compileInner(fo, sourceBean.getClassName(),
				sourceBean.getSourceCode());
	}

	/**
	 * Javaソースのコンパイルを行います
	 * 
	 * @param <T>
	 * @param className
	 * @param sourceCode
	 * @return コンパイル結果
	 */
	public <T> Class<T> compile(final String className, final String sourceCode) {
		JavaFileObject fo = new JavaSourceFromString(className, sourceCode);

		if (this.compiler == null) {
			throw new RuntimeException("Javaコンパイラインスタンスが存在しません");
		}

		return this.compileInner(fo, className, sourceCode);
	}

	/**
	 * テキストソースからコンパイルを行うためのメソッド
	 * 
	 * @param <T>
	 *            対象クラス
	 * @param fo
	 *            JavaFileObject
	 * @param className
	 *            完全限定クラス名
	 * @param sourceCode
	 *            コンパイル対象ソースコード
	 * @return コンパイルされたクラスオブジェクト
	 */
	private <T> Class<T> compileInner(final JavaFileObject fo,
			final String className, final String sourceCode) {

		// コンパイル対象ユニット
		List<JavaFileObject> compilationUnits = Arrays.asList(fo);

		// クラスパスの決定
		List<String> options = Arrays.asList(ARGS_CLASSPATH,

		// プロパティ:システムクラスパス
				System.getProperty(SYS_PROP_CLASS_PATH) + ";"

				// プロパティ：アプリケーションクラスパス
						+ System.getProperty(APP_PROP_CLASS_PATH));

		// コンパイラとリスナを設定してクラスファイルマネージャを作成
		ClassFileManager manager = new ClassFileManager(this.compiler,
				this.listener);

		// クラスローダを取得
		ClassLoader cl = manager.getClassLoader(null);
		
		String isRevert = System.getProperty(APP_REVERT);
		if(isRevert == null) {
			isRevert = "";
		}

		// クラスローダ上に指定されたクラスが登録されているかどうかで
		// 処理を分ける
		if (manager.exists(className) && isRevert.isEmpty()) {
			// クラスローダ上にクラスが存在する
			try {

				// クラスローダからクラスをロード
				@SuppressWarnings("unchecked")
				Class<T> c = (Class<T>) cl.loadClass(className);
				return c;
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		} else {
			// クラスローダ上にクラスが存在しない場合は
			// 入力ソースをコンパイルする

			CompilationTask task = this.compiler.getTask(null, manager,
					this.listener, options, null, compilationUnits);

			// コンパイル実行
			boolean successCompile = task.call();

			if (!successCompile) {
				// コンパイル失敗の場合
				throw new RuntimeException("コンパイル失敗：" + className);
			} else {
				// System.out.println("コンパイル成功");
				// ソースフォルダにjavaファイルを出力
				// System.out.println(className);

				String sourceRoot = System.getProperty(APP_PROP_SOURCE_PATH);

				if (sourceRoot != null && !sourceRoot.isEmpty()) {
					// 入力ソース出力

					Pattern p = Pattern.compile(".+(\\..+)");
					Matcher m = p.matcher(className);
					if (m.find()) {
						String replacement = m.group(1);

						String packageName = className.replace(replacement, "");

						// System.out.println(packageName);

						File sourceRootDir = new File(sourceRoot + "\\"
								+ packageName.replace(".", "\\"));

						if (!sourceRootDir.exists()) {
							sourceRootDir.mkdirs();
						}
						String fileName = replacement.replace(".", "")
								+ ".java";

						File outputFile = new File(sourceRoot + "\\"
								+ packageName.replace(".", "\\") + "\\"
								+ fileName);
						FileOutputStream fos = null;
						try {
							fos = new FileOutputStream(outputFile);
							fos.write(sourceCode.getBytes("UTF-8"));
						} catch (FileNotFoundException e) {
							// TODO 自動生成された catch ブロック
							e.printStackTrace();
						} catch (UnsupportedEncodingException e) {
							// TODO 自動生成された catch ブロック
							e.printStackTrace();
						} catch (IOException e) {
							// TODO 自動生成された catch ブロック
							e.printStackTrace();
						} finally {
							if (fos != null) {
								try {
									fos.close();
								} catch (IOException e) {
									// TODO 自動生成された catch ブロック
									e.printStackTrace();
								}
							}
						}
					}
				}

			}

			try {
				@SuppressWarnings("unchecked")
				Class<T> c = (Class<T>) cl.loadClass(className);
				return c;
			} catch (ClassNotFoundException e) {
				throw new RuntimeException(e);
			}
		}

	}
}
