# Copyright (c) 2021 Jone Bontus.
# All rights reserved.
# This component and the accompanying materials are made available
# under the terms of "Eclipse Public License v1.0"
# which accompanies this distribution, and is available
# at the URL "http://www.eclipse.org/legal/epl-v10.html".
#
# Initial Contributors:
# Jone Bontus.
#
# Contributors:
#
# Description:
# epoc32\tools\elfexport.pl
# Export symbols from an object file
# Syntax:
# perl elfexport.pl [--dynamic] <deffile> <elffile>
#
#

use strict;
no strict 'subs';
use integer;
use Getopt::Long;

use constant {
	SHT_NULL => 0,
	SHT_PROGBITS => 1,
	SHT_SYMTAB => 2,
	SHT_STRTAB => 3,
	SHT_RELA => 4,
	SHT_HASH => 5,
	SHT_DYNAMIC => 6,
	SHT_NOTE => 7,
	SHT_NOBITS => 8,
	SHT_REL => 9,
	SHT_SHLIB => 10,
	SHT_DYNSYM => 11,
};

use constant {
	STB_LOCAL => 0,
	STB_GLOBAL => 1,
	STB_WEAK => 2,
	STB_LOOS => 10,
	STB_GNU_UNIQUE => 10,
	STB_HIOS => 12,
	STB_LOPROC => 13,
	STB_HIPROC => 15,
};

use constant {
	STV_DEFAULT => 0,
	STV_INTERNAL => 1,
	STV_HIDDEN => 2,
	STV_PROTECTED => 3,
};

sub processDefFile ($$);
sub bytesToInt ($);
sub intToBytes32 ($);
sub intToBytes16 ($);
sub intToBytes8 ($);
sub readElfHeader ($);
sub isValidElfHeader ($);
sub readSectionHeader ($);
sub readSectionHeaders ($$$$);
sub findSection ($$);
sub readSymbolTableEntry ($);
sub symbolTableEntryToBytes ($);
sub getStringFromStringTable ($$);
sub contains ($$);

my %Options;

unless (GetOptions(\%Options, 'dynamic')) {
	exit 1;
}

unless (@ARGV==2) {
	die "perl elfexport.pl [--dynamic] <deffile> <elffile>\n";
}
my ($deffile, $elffile) = @ARGV;

my @exports;
processDefFile($deffile, \@exports);
# nothing to process
if (@exports==0) {
	exit 0;
}

open ELFFILE, $elffile or die "Can't open $elffile for read\n";
binmode(ELFFILE);
my %elfHeader = readElfHeader(ELFFILE);
unless (%elfHeader) {
	close ELFFILE;
	die "Error reading $elffile\n";
}
unless (isValidElfHeader(\%elfHeader)) {
	close ELFFILE;
	die "Bad ELF Magic\n";
}

my $e_shoff = bytesToInt($elfHeader{"e_shoff"});
my $e_shentsize = bytesToInt($elfHeader{"e_shentsize"});
my $e_shnum = bytesToInt($elfHeader{"e_shnum"});
my @elfSections = readSectionHeaders(ELFFILE, $e_shoff, $e_shentsize, $e_shnum);
unless (@elfSections) {
	close ELFFILE;
	die "Error reading $elffile\n";
}

my ($symtabEntryNum, $strtabEntryNum) = findSection(\@elfSections, $Options{dynamic} ? SHT_DYNSYM : SHT_SYMTAB);
unless ($symtabEntryNum and $strtabEntryNum) {
	close ELFFILE;
	exit 0;
}
my $symtabEntry = $elfSections[$symtabEntryNum];
my $strtabEntry = $elfSections[$strtabEntryNum];
my $strtabData;
unless (seek(ELFFILE, bytesToInt($$strtabEntry{"sh_offset"}), 0) and read(ELFFILE, $strtabData, bytesToInt($$strtabEntry{"sh_size"}))) {
	close ELFFILE;
	die "Error reading $elffile\n";
}

my $symtabData;
my $symtabEntryCount = bytesToInt($$symtabEntry{"sh_size"}) / bytesToInt($$symtabEntry{"sh_entsize"});
unless (seek(ELFFILE, bytesToInt($$symtabEntry{"sh_offset"}), 0)) {
	close ELFFILE;
	die "Error reading $elffile\n";
}
# Process each symbol entry
#my $localSymbolCount = 0;
for (my $i = 0; $i < $symtabEntryCount; $i++) {
	my %symbolTableEntry = readSymbolTableEntry(ELFFILE);
	unless (%symbolTableEntry) {
		close ELFFILE;
		die "Error reading $elffile\n";
	}

	# Get and change the visibility of symbol
	my $symbolVisibility = bytesToInt($symbolTableEntry{"st_other"});
	my $symbolBind = bytesToInt($symbolTableEntry{"st_info"}) >> 4;
	if (contains(\@exports, getStringFromStringTable($strtabData, bytesToInt($symbolTableEntry{"st_name"})))) {
		if ($Options{dynamic}) {
			if ($symbolBind == STB_WEAK) {
				$symbolBind = STB_GLOBAL;
			}
		} else {
			$symbolVisibility = STV_DEFAULT;
		}
	}
	#if ($symbolBind == STB_LOCAL) {
		#$localSymbolCount++;
	#}
	if ($Options{dynamic}) {
		$symbolTableEntry{"st_info"} = intToBytes8($symbolBind << 4 | (bytesToInt($symbolTableEntry{"st_info"})) & 0xf);
	} else {
		$symbolTableEntry{"st_other"} = intToBytes8($symbolVisibility);
	}

	$symtabData .= symbolTableEntryToBytes(\%symbolTableEntry);
}

close ELFFILE;

open ELFFILE, "+<$elffile" or die "Can't open $elffile for write\n";
binmode(ELFFILE);
unless (seek(ELFFILE, bytesToInt($$symtabEntry{"sh_offset"}), 0)) {
	close ELFFILE;
	die "Error writing $elffile\n";
}
print ELFFILE $symtabData;
close ELFFILE;

sub processDefFile ($$)
{
	my ($defFile, $exportListRef) = @_;

	open DEFFILE, $defFile or die "Can't open $defFile for read\n";

	# Process exports section
	while (<DEFFILE>) {
		if (/EXPORTS/) {
			last;
		}
	}

	while (<DEFFILE>) {
		if ($_ =~ /^\s*(\S+)\s+@\s+\S+\s+NONAME(.*)/) {
			if ($2 !~ /\s+ABSENT/) {
				push @$exportListRef, $1;
			}
		}
	}
	close DEFFILE;
}

sub bytesToInt ($)
{
	my $bytes = shift;
	my $val;
	$val = ord(substr($bytes, 0));
	$val |= ord(substr($bytes, 1)) << 8;
	$val |= ord(substr($bytes, 2)) << 16;
	$val |= ord(substr($bytes, 3)) << 24;
	return $val;
}

sub intToBytes32 ($)
{
	my $int = shift;
	my $val;
	$val = chr($int & 0xff);
	$val .= chr($int >> 8 & 0xff);
	$val .= chr($int >>16 & 0xff);
	$val .= chr($int >> 24 & 0xff);
	return $val;
}

sub intToBytes16 ($)
{
	my $int = shift;
	my $val;
	$val = chr($int & 0xff);
	$val .= chr($int >> 8 & 0xff);
	return $val;
}

sub intToBytes8 ($)
{
	my $int = shift;
	my $val;
	$val = chr($int & 0xff);
	return $val;
}

sub readElfHeader ($)
{
	my $elfFileHandle = shift;
	my %elfHeader;
	my $buf;

	seek($elfFileHandle, 0, 0) or return ();
	read($elfFileHandle, $buf, 16) or return ();
	$elfHeader{"e_ident"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_type"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_machine"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfHeader{"e_version"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfHeader{"e_entry"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfHeader{"e_phoff"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfHeader{"e_shoff"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfHeader{"e_flags"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_ehsize"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_phentsize"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_phnum"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_shentsize"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_shnum"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$elfHeader{"e_shstrndx"} = $buf;

	return %elfHeader;
}

sub isValidElfHeader ($)
{
	my $elfHeaderRef = shift;
	my $e_ident = $$elfHeaderRef{"e_ident"};

	if (length($e_ident)!=16
		or substr($e_ident, 0, 1) ne "\x7f"
		or substr($e_ident, 1, 1) ne "E"
		or substr($e_ident, 2, 1) ne "L"
		or substr($e_ident, 3, 1) ne "F") {
		return;
	}
	return 1;
}

sub readSectionHeader ($)
{
	my $elfFileHandle = shift;
	my %elfSectionHeader;
	my $buf;

	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_name"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_type"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_flags"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_addr"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_offset"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_size"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_link"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_info"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_addralign"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$elfSectionHeader{"sh_entsize"} = $buf;

	return %elfSectionHeader;
}

sub readSectionHeaders ($$$$)
{
	my ($elfFileHandle, $sectionHeaderTableOffset, $entrySize, $entryCount) = @_;
	my @sectionEntries;

	for (my $i = 0; $i < $entryCount; $i++) {
		my %elfSectionHeader;
		seek($elfFileHandle, $sectionHeaderTableOffset + $i * $entrySize, 0) or return ();
		%elfSectionHeader = readSectionHeader($elfFileHandle);
		unless (%elfSectionHeader) {
			return ();
		}
		push @sectionEntries, \%elfSectionHeader;
	}
	return @sectionEntries;
}

sub findSection ($$)
{
	my ($sectionEntriesRef, $type) = @_;
	my $sectionEntryNum;
	my $linkedSectionEntryNum;

	for (my $i = 0; $i < scalar(@$sectionEntriesRef); $i++) {
		my $sectionEntryRef = @$sectionEntriesRef[$i];
		if (bytesToInt($$sectionEntryRef{"sh_type"}) == $type) {
			$sectionEntryNum = $i;
			$linkedSectionEntryNum = bytesToInt($$sectionEntryRef{"sh_link"});
			last;
		}
	}
	return ($sectionEntryNum, $linkedSectionEntryNum);
}

sub readSymbolTableEntry ($)
{
	my $elfFileHandle = shift;
	my %symbolTableEntry;
	my $buf;

	read($elfFileHandle, $buf, 4) or return ();
	$symbolTableEntry{"st_name"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$symbolTableEntry{"st_value"} = $buf;
	read($elfFileHandle, $buf, 4) or return ();
	$symbolTableEntry{"st_size"} = $buf;
	read($elfFileHandle, $buf, 1) or return ();
	$symbolTableEntry{"st_info"} = $buf;
	read($elfFileHandle, $buf, 1) or return ();
	$symbolTableEntry{"st_other"} = $buf;
	read($elfFileHandle, $buf, 2) or return ();
	$symbolTableEntry{"st_shndx"} = $buf;

	return %symbolTableEntry;
}

sub symbolTableEntryToBytes ($)
{
	my $symbolTableEntryRef = shift;
	my $buf;

	$buf = $$symbolTableEntryRef{"st_name"};
	$buf .= $$symbolTableEntryRef{"st_value"};
	$buf .= $$symbolTableEntryRef{"st_size"};
	$buf .= $$symbolTableEntryRef{"st_info"};
	$buf .= $$symbolTableEntryRef{"st_other"};
	$buf .= $$symbolTableEntryRef{"st_shndx"};

	return $buf;
}

sub getStringFromStringTable ($$)
{
	my ($strtabData, $offset) = @_;
	my $string;
	my $chr;

	while (ord($chr = substr($strtabData, $offset, 1)) != 0) {
		$string .= $chr;
		$offset++;
	}

	return $string;
}

sub contains ($$)
{
	my ($arrayRef, $element) = @_;

	foreach my $i (@$arrayRef) {
		if ($i eq $element) {
			return 1;
		}
	}
	return;
}
