// 
// Copyright (c) 2003-2010, MIST Project, Nagoya University
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice,
// this list of conditions and the following disclaimer in the documentation
// and/or other materials provided with the distribution.
// 
// 3. Neither the name of the Nagoya University nor the names of its contributors
// may be used to endorse or promote products derived from this software
// without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
// IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
// THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 

/// @file mist/filter/region_growing.h
//!
//! @brief ẻ摜ɑΉCCӂ̈ʒuCӂ̍\vfƔCӂ̏pė̈g@Kp邽߂̃Cu
//!

#ifndef __INCLUDE_MIST_REGIONGROWING__
#define __INCLUDE_MIST_REGIONGROWING__


#ifndef __INCLUDE_MIST_H__
#include "../mist.h"
#endif

#ifndef __INCLUDE_MIST_LIMITS__
#include "../limits.h"
#endif

#ifndef __INCLUDE_MIST_VECTOR__
#include "../vector.h"
#endif

#include <deque>
#include <vector>
#include <cmath>


// mistOԂ̎n܂
_MIST_BEGIN



//! @addtogroup region_growing_group ̈g@
//!
//! @code ̃wb_CN[h
//! #include <mist/filter/region_growing.h>
//! @endcode
//!
//! @code ̈g@̎gp
//! typedef mist::region_growing_utility::point_type    point_type;     // ̈g̊Jn_w肷^imist::vector2  mist::vector3 őpj
//! typedef mist::region_growing_utility::circle        component_type; // ̈gɗp\vf
//! typedef mist::region_growing_utility::less< short > condition_type; // ̈ĝ̊g
//!
//! mist::region_growing(
//!                       in,                    // ͉摜
//!                       out,                   // ̈gsʂi[摜	
//!                       point_type( x, y ),    // gJn_iXgɂ邱Ƃŕw\j
//!                       128,                   // ̈ǧʂɑl
//!                       component_type( 20 ),  // ̈gɗp\vfifC~CȂǁj
//!                       condition_type( 128 )  // ̈ĝ̊giCȏC͈͂Ȃǁj
//!                     );
//! @endcode
//!
//!  @{


/// @brief ̈g@ɗp\vf֘A
//! 
//! ̈g@ɗp\vf̒`
//! 
namespace region_growing_utility
{
	/// @brief ̒ړ_̒ߕ
	enum expand_mode_type
	{
		NC4,	///< @brief SߖTiQj
		NC8,	///< @brief WߖTiQj
		NC6,	///< @brief UߖTiRj
		NC18,	///< @brief PWߖTiRj
		NC26,	///< @brief QUߖTiRj
		ALL		///< @brief \vfׂ̂Ă̓_
	};


	/// @brief ̈g@ŗpQf̍\vf
	struct pixel
	{
		typedef size_t		size_type;				///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;		///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ

		size_type width( ) const { return( 1 ); }	///< @brief \vf̕Ԃ
		size_type height( ) const { return( 1 ); }	///< @brief \vf̍Ԃ
		size_type depth( ) const { return( 1 ); }	///< @brief \vf̐[Ԃ

		expand_mode_type __expand_mode__;

		/// @brief Cӂ̈ʒu\vfǂ𔻒肷֐
		bool operator ()( difference_type /* i */, difference_type /* j */, difference_type /* k */ ) const { return( true ); }

		/// @brief ̒ړ_̒ߕ
		expand_mode_type expand_mode( ) const { return( __expand_mode__ ); }

		/// @brief NC4 ߖT^̃{NZ
		pixel( ) : __expand_mode__( NC4 ){ }

		/// @brief w肵ߖT̃{NZƂď
		pixel( expand_mode_type mode ) : __expand_mode__( mode ){ }
	};


	/// @brief ̈g@ŗpRf̍\vf
	struct voxel
	{
		typedef size_t		size_type;				///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;		///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ

		size_type width( ) const { return( 1 ); }	///< @brief \vf̕Ԃ
		size_type height( ) const { return( 1 ); }	///< @brief \vf̍Ԃ
		size_type depth( ) const { return( 1 ); }	///< @brief \vf̐[Ԃ

		expand_mode_type __expand_mode__;

		/// @brief Cӂ̈ʒu\vfǂ𔻒肷֐
		bool operator ()( difference_type /* i */, difference_type /* j */, difference_type /* k */ ) const { return( true ); }

		/// @brief ̒ړ_̒ߕ
		expand_mode_type expand_mode( ) const { return( __expand_mode__ ); }

		/// @brief NC6 ߖT^̃{NZ
		voxel( ) : __expand_mode__( NC6 ){ }

		/// @brief w肵ߖT̃{NZƂď
		voxel( expand_mode_type mode ) : __expand_mode__( mode ){ }
	};


	/// @brief ̈g@ŗp~̍\vf
	class circle
	{
	public:
		typedef size_t		size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ

	private:
		double radius_;
		double resoX_;
		double resoY_;
		size_type width_;
		size_type height_;

	public:
		size_type width( ) const { return( width_ ); }		///< @brief \vf̕Ԃ
		size_type height( ) const { return( height_ ); }	///< @brief \vf̍Ԃ
		size_type depth( ) const { return( 1 ); }			///< @brief \vf̐[Ԃ

		/// @brief Cӂ̈ʒu\vfǂ𔻒肷֐
		bool operator ()( difference_type i, difference_type j, difference_type k ) const
		{
			if( k != 0 )
			{
				return( false );
			}
			else
			{
				return( i * i * resoX_ * resoX_ + j * j * resoY_ * resoY_ <= radius_ * radius_ );
			}
		}

		/// @brief ̒ړ_̒ߕ
		expand_mode_type expand_mode( ) const { return( NC4 ); }

		/// @brief aƉ𑜓xw肵ċiȉ~̍쐬\CCPʂ͉fj
		circle( double radius, double resoX = 1.0, double resoY = 1.0, bool radiusInPhysicalCoords = false ) : radius_( radius )
		{
			if( radiusInPhysicalCoords )
			{
				resoX_ = resoX;
				resoY_ = resoY;
			}
			else
			{
				double max_reso = resoX > resoY ? resoX: resoY;

				resoX_ = resoX / max_reso;
				resoY_ = resoY / max_reso;
			}
			width_  = static_cast< size_type >( std::ceil( radius_ / resoX_ ) ) * 2 + 1;
			height_ = static_cast< size_type >( std::ceil( radius_ / resoY_ ) ) * 2 + 1;
		}
	};


	/// @brief ̈g@ŗp鋅̍\vf
	class sphere
	{
	public:
		typedef size_t		size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ

	private:
		double radius_;		///< @brief a
		double resoX_;		///< @brief X̉𑜓x
		double resoY_;		///< @brief Ỷ𑜓x
		double resoZ_;		///< @brief Z̉𑜓x
		size_type width_;
		size_type height_;
		size_type depth_;

	public:
		size_type width( ) const { return( width_ ); }		///< @brief \vf̕Ԃ
		size_type height( ) const { return( height_ ); }	///< @brief \vf̍Ԃ
		size_type depth( ) const { return( depth_ ); }		///< @brief \vf̐[Ԃ

		/// @brief Cӂ̈ʒu\vfǂ𔻒肷֐
		bool operator ()( difference_type i, difference_type j, difference_type k ) const
		{
			return( i * i * resoX_ * resoX_ + j * j * resoY_ * resoY_+ k * k * resoZ_ * resoZ_ <= radius_ * radius_ );
		}

		/// @brief ̒ړ_̒ߕ
		expand_mode_type expand_mode( ) const { return( NC6 ); }

		/// @brief aƉ𑜓xw肵ċiȉ~̂̍쐬\CCPʂ͉fj
		sphere( double radius, double resoX = 1.0, double resoY = 1.0, double resoZ = 1.0, bool radiusInPhysicalCoords = false ) : radius_( radius )
		{
			if (radiusInPhysicalCoords)
			{
				resoX_ = resoX;
				resoY_ = resoY;
				resoZ_ = resoZ;
			}
			else
			{
				double max_reso = resoX > resoY ? resoX: resoY;
				max_reso = max_reso > resoZ ? max_reso : resoZ;

				resoX_ = resoX / max_reso;
				resoY_ = resoY / max_reso;
				resoZ_ = resoZ / max_reso;
			}
			width_  = static_cast< size_type >( std::ceil( radius_ / resoX_ ) ) * 2 + 1;
			height_ = static_cast< size_type >( std::ceil( radius_ / resoY_ ) ) * 2 + 1;
			depth_  = static_cast< size_type >( std::ceil( radius_ / resoZ_ ) ) * 2 + 1;
		}
	};


	/// @brief ̈g@ŗpg
	//!
	//! 胊Xg̑SẲfl
	//!
	template < class T >
	class less
	{
	private:
		typedef size_t		size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
		typedef T			value_type;			///< @brief ɗpf[^^

	protected:
		value_type th_;		///< @brief l

	public:
		/// @brief 胊Xg̉f𖞂ǂ𒲂ׂ
		template < class VerifyList >
		bool operator()( const VerifyList &elements, size_type num ) const
		{
			typedef typename VerifyList::value_type verify_value_type;

			for( size_type i = 0 ; i < num ; i++ )
			{
				const verify_value_type &v = elements[ i ];
				if( !( v < th_ ) )
				{
					return ( false );
				}
			}

			return ( true );
		}

		/// @brief sۂɁC\vf̑SfKvȏꍇ true Ԃ
		bool require_all_elements( ) const { return( false ); }

		/// @brief Cӂ̂lŏ
		less( value_type threshold ) : th_( threshold ){ }
	};


	/// @brief ̈g@ŗpg
	//!
	//! 胊Xg̑SẲfl傫
	//!
	template < class T >
	class greater
	{
	private:
		typedef size_t		size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
		typedef T			value_type;			///< @brief ɗpf[^^

	protected:
		value_type th_;		///< @brief l

	public:
		/// @brief 胊Xg̉f𖞂ǂ𒲂ׂ
		template < class VerifyList >
		bool operator()( const VerifyList &elements, size_type num ) const
		{
			typedef typename VerifyList::value_type verify_value_type;

			for( size_type i = 0 ; i < num ; i++ )
			{
				const verify_value_type &v = elements[ i ];
				if( v < th_ )
				{
					return ( false );
				}
			}

			return ( true );
		}

		/// @brief sۂɁC\vf̑SfKvȏꍇ true Ԃ
		bool require_all_elements( ) const { return( false ); }

		/// @brief Cӂ̂lŏ
		greater( value_type threshold ) : th_( threshold ){ }
	};


	/// @brief ̈g@ŗpg
	//!
	//! 胊Xg̑SẲflƓ
	//!
	template < class T >
	class equal
	{
	private:
		typedef size_t		size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
		typedef T			value_type;			///< @brief ɗpf[^^

	protected:
		value_type th_;		///< @brief l

	public:
		/// @brief 胊Xg̉f𖞂ǂ𒲂ׂ
		template < class VerifyList >
		bool operator()( const VerifyList &elements, size_type num ) const
		{
			typedef typename VerifyList::value_type verify_value_type;

			for( size_type i = 0 ; i < num ; i++ )
			{
				const verify_value_type &v = elements[ i ];
				if( v < th_ || th_ < v )
				{
					return ( false );
				}
			}

			return ( true );
		}

		/// @brief sۂɁC\vf̑SfKvȏꍇ true Ԃ
		bool require_all_elements( ) const { return( false ); }

		/// @brief Cӂ̂lŏ
		equal( value_type threshold ) : th_( threshold ){ }
	};


	/// @brief ̈g@ŗpg
	//!
	//! 胊Xg̑SẲf͈͓ɑ݂
	//!
	template < class T >
	class range
	{
	private:
		typedef size_t		size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
		typedef ptrdiff_t	difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
		typedef T			value_type;			///< @brief ɗpf[^^

	protected:
		value_type min_;	///< @brief ͈͂̉
		value_type max_;	///< @brief ͈͂̏

	public:
		/// @brief 胊Xg̉f𖞂ǂ𒲂ׂ
		template < class VerifyList >
		bool operator()( const VerifyList &elements, size_type num ) const
		{
			typedef typename VerifyList::value_type verify_value_type;

			for( size_type i = 0 ; i < num ; i++ )
			{
				const verify_value_type &v = elements[ i ];
				if( v < min_ || max_ < v )
				{
					return ( false );
				}
			}

			return ( true );
		}

		/// @brief sۂɁC\vf̑SfKvȏꍇ true Ԃ
		bool require_all_elements( ) const { return( false ); }

		/// @brief Cӂ̉ƏŔ͈͂
		range( value_type min, value_type max ) : min_( min ), max_( max ){ }
	};
}


// ̈g@̓ŗp^Ȃ
namespace __region_growing_utility__
{
	struct pointer_diff2
	{
		typedef ptrdiff_t	difference_type;

		ptrdiff_t diff1;
		ptrdiff_t diff2;

		pointer_diff2( ptrdiff_t d1 = 0, ptrdiff_t d2 = 0 ) : diff1( d1 ), diff2( d2 ){ }
	};

	typedef vector3< ptrdiff_t >			point_type;			///< @brief ̈g@ŗpW\^
	typedef std::vector< point_type >		point_list_type;	///< @brief ̈g@ŗpW̃Xg^
	typedef std::vector< pointer_diff2 >	ptrdiff_list_type;


	struct no_mask
	{
		typedef size_t		size_type;
		typedef ptrdiff_t	difference_type;

		bool empty( ) const { return( true ); }
		bool operator ()( difference_type /* i */, difference_type /* j */, difference_type /* k */ ) const { return( false ); }
	};


	template < class Array1, class Array2, class Component >
	ptrdiff_list_type create_component_list( const Array1 &in1, const Array2 &in2, const Component &components )
	{
		typedef typename Array1::const_pointer				const_pointer1;
		typedef typename Array2::const_pointer				const_pointer2;
		typedef typename ptrdiff_list_type::size_type		size_type;
		typedef typename ptrdiff_list_type::difference_type	difference_type;
		typedef typename ptrdiff_list_type::value_type		value_type;

		ptrdiff_list_type list;

		difference_type w = components.width( );
		difference_type h = components.height( );
		difference_type d = components.depth( );
		difference_type cx = w / 2;
		difference_type cy = h / 2;
		difference_type cz = d / 2;
		difference_type ox1 = in1.width( ) / 2;
		difference_type oy1 = in1.height( ) / 2;
		difference_type oz1 = in1.depth( ) / 2;
		difference_type ox2 = in2.width( ) / 2;
		difference_type oy2 = in2.height( ) / 2;
		difference_type oz2 = in2.depth( ) / 2;

		const_pointer1 p1 = &in1( ox1, oy1, oz1 );
		const_pointer2 p2 = &in2( ox2, oy2, oz2 );

		for( difference_type k = 0 ; k < d ; k++ )
		{
			for( difference_type j = 0 ; j < h ; j++ )
			{
				for( difference_type i = 0 ; i < w ; i++ )
				{
					if( components( i - cx, j - cy, k - cz ) )
					{
						ptrdiff_t d1 = &in1( ox1 + i - cx, oy1 + j - cy, oz1 + k - cz ) - p1;
						ptrdiff_t d2 = &in2( ox2 + i - cx, oy2 + j - cy, oz2 + k - cz ) - p2;
						list.push_back( pointer_diff2( d1, d2 ) );
					}
				}
			}
		}

		return( list );
	}


	template < class Array1, class Array2, class Component >
	ptrdiff_list_type create_update_list( const Array1 &in1, const Array2 &in2, const Component &components )
	{
		typedef typename Array1::const_pointer				const_pointer1;
		typedef typename Array2::const_pointer				const_pointer2;
		typedef typename ptrdiff_list_type::size_type		size_type;
		typedef typename ptrdiff_list_type::difference_type	difference_type;
		typedef typename ptrdiff_list_type::value_type		value_type;

		ptrdiff_list_type list;

		difference_type ox1 = in1.width( ) / 2;
		difference_type oy1 = in1.height( ) / 2;
		difference_type oz1 = in1.depth( ) / 2;
		difference_type ox2 = in2.width( ) / 2;
		difference_type oy2 = in2.height( ) / 2;
		difference_type oz2 = in2.depth( ) / 2;

		const_pointer1 p1 = &in1( ox1, oy1, oz1 );
		const_pointer2 p2 = &in2( ox2, oy2, oz2 );

		switch( components.expand_mode( ) )
		{
		case region_growing_utility::NC4:
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1, oz1 ) - p1, &in2( ox2 - 1, oy2, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1, oz1 ) - p1, &in2( ox2 + 1, oy2, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1, oy1 - 1, oz1 ) - p1, &in2( ox2, oy2 - 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1, oy1 + 1, oz1 ) - p1, &in2( ox2, oy2 + 1, oz2 ) - p2 ) );
			break;

		case region_growing_utility::NC8:
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 - 1, oz1 ) - p1, &in2( ox2 - 1, oy2 - 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1 ) - p1, &in2( ox2    , oy2 - 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 - 1, oz1 ) - p1, &in2( ox2 + 1, oy2 - 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1 ) - p1, &in2( ox2 - 1, oy2    , oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1 ) - p1, &in2( ox2 + 1, oy2    , oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 + 1, oz1 ) - p1, &in2( ox2 - 1, oy2 + 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1 ) - p1, &in2( ox2    , oy2 + 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 + 1, oz1 ) - p1, &in2( ox2 + 1, oy2 + 1, oz2 ) - p2 ) );
			break;

		case region_growing_utility::NC6:
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1, oz1 ) - p1, &in2( ox2 - 1, oy2, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1, oz1 ) - p1, &in2( ox2 + 1, oy2, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1, oy1 - 1, oz1 ) - p1, &in2( ox2, oy2 - 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1, oy1 + 1, oz1 ) - p1, &in2( ox2, oy2 + 1, oz2 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1, oy1, oz1 - 1 ) - p1, &in2( ox2, oy2, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1, oy1, oz1 + 1 ) - p1, &in2( ox2, oy2, oz2 + 1 ) - p2 ) );
			break;

		case region_growing_utility::NC18:
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1 - 1 ) - p1, &in2( ox2    , oy2 - 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1 - 1 ) - p1, &in2( ox2 - 1, oy2    , oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1    , oz1 - 1 ) - p1, &in2( ox2    , oy2    , oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1 - 1 ) - p1, &in2( ox2 + 1, oy2    , oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1 - 1 ) - p1, &in2( ox2    , oy2 + 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 - 1, oz1     ) - p1, &in2( ox2 - 1, oy2 - 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1     ) - p1, &in2( ox2    , oy2 - 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 - 1, oz1     ) - p1, &in2( ox2 + 1, oy2 - 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1     ) - p1, &in2( ox2 - 1, oy2    , oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1     ) - p1, &in2( ox2 + 1, oy2    , oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 + 1, oz1     ) - p1, &in2( ox2 - 1, oy2 + 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1     ) - p1, &in2( ox2    , oy2 + 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 + 1, oz1     ) - p1, &in2( ox2 + 1, oy2 + 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1 + 1 ) - p1, &in2( ox2    , oy2 - 1, oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1 + 1 ) - p1, &in2( ox2 - 1, oy2    , oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1    , oz1 + 1 ) - p1, &in2( ox2    , oy2    , oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1 + 1 ) - p1, &in2( ox2 + 1, oy2    , oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1 + 1 ) - p1, &in2( ox2    , oy2 + 1, oz2 + 1 ) - p2 ) );
			break;

		case region_growing_utility::NC26:
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 - 1, oz1 - 1 ) - p1, &in2( ox2 - 1, oy2 - 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1 - 1 ) - p1, &in2( ox2    , oy2 - 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 - 1, oz1 - 1 ) - p1, &in2( ox2 + 1, oy2 - 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1 - 1 ) - p1, &in2( ox2 - 1, oy2    , oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1    , oz1 - 1 ) - p1, &in2( ox2    , oy2    , oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1 - 1 ) - p1, &in2( ox2 + 1, oy2    , oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 + 1, oz1 - 1 ) - p1, &in2( ox2 - 1, oy2 + 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1 - 1 ) - p1, &in2( ox2    , oy2 + 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 + 1, oz1 - 1 ) - p1, &in2( ox2 + 1, oy2 + 1, oz2 - 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 - 1, oz1     ) - p1, &in2( ox2 - 1, oy2 - 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1     ) - p1, &in2( ox2    , oy2 - 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 - 1, oz1     ) - p1, &in2( ox2 + 1, oy2 - 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1     ) - p1, &in2( ox2 - 1, oy2    , oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1     ) - p1, &in2( ox2 + 1, oy2    , oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 + 1, oz1     ) - p1, &in2( ox2 - 1, oy2 + 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1     ) - p1, &in2( ox2    , oy2 + 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 + 1, oz1     ) - p1, &in2( ox2 + 1, oy2 + 1, oz2     ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 - 1, oz1 + 1 ) - p1, &in2( ox2 - 1, oy2 - 1, oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 - 1, oz1 + 1 ) - p1, &in2( ox2    , oy2 - 1, oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 - 1, oz1 + 1 ) - p1, &in2( ox2 + 1, oy2 - 1, oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1    , oz1 + 1 ) - p1, &in2( ox2 - 1, oy2    , oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1    , oz1 + 1 ) - p1, &in2( ox2    , oy2    , oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1    , oz1 + 1 ) - p1, &in2( ox2 + 1, oy2    , oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 - 1, oy1 + 1, oz1 + 1 ) - p1, &in2( ox2 - 1, oy2 + 1, oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1    , oy1 + 1, oz1 + 1 ) - p1, &in2( ox2    , oy2 + 1, oz2 + 1 ) - p2 ) );
			list.push_back( pointer_diff2( &in1( ox1 + 1, oy1 + 1, oz1 + 1 ) - p1, &in2( ox2 + 1, oy2 + 1, oz2 + 1 ) - p2 ) );
			break;

		case region_growing_utility::ALL:
		default:
			list = create_component_list( in1, in2, components );
			break;
		}

		return( list );
	}


	template < class T, class Allocator, class Array, class PointType >
	bool check_component_list( const array2< T, Allocator > &in1, const Array &in2, const PointType &pt, pointer_diff2 &ptr )
	{
		typedef typename array2< T, Allocator >::const_pointer	const_pointer1;
		typedef typename Array::const_pointer					const_pointer2;
		typedef typename ptrdiff_list_type::difference_type		difference_type;

		difference_type x = static_cast< difference_type >( pt.x );
		difference_type y = static_cast< difference_type >( pt.y );
		difference_type w = in1.width( );
		difference_type h = in1.height( );

		if( x < 0 || x >= w || y < 0 || y >= h )
		{
			return( false );
		}
		else if( in2( x, y ) != 0 )
		{
			return( false );
		}
		else
		{
			ptr.diff1 = &in1( x, y) - &in1[ 0 ];
			ptr.diff2 = &in2( x, y ) - &in2[ 0 ];
			return( true );
		}
	}


	template < class T, class Allocator, class Array, class PointType >
	bool check_component_list( const array3< T, Allocator > &in1, const Array &in2, const PointType &pt, pointer_diff2 &ptr )
	{
		typedef typename array3< T, Allocator >::const_pointer	const_pointer1;
		typedef typename Array::const_pointer					const_pointer2;
		typedef typename ptrdiff_list_type::difference_type		difference_type;

		difference_type x = static_cast< difference_type >( pt.x );
		difference_type y = static_cast< difference_type >( pt.y );
		difference_type z = static_cast< difference_type >( pt.z );
		difference_type w = in1.width( );
		difference_type h = in1.height( );
		difference_type d = in1.depth( );

		if( x < 0 || x >= w || y < 0 || y >= h || z < 0 || z >= d )
		{
			return( false );
		}
		else if( in2( x, y, z ) != 0 )
		{
			return( false );
		}
		else
		{
			ptr.diff1 = &in1( x, y, z ) - &in1[ 0 ];
			ptr.diff2 = &in2( x, y, z ) - &in2[ 0 ];
			return( true );
		}
	}


	template < class PointType >
	struct point_list_converter
	{
		typedef std::vector< PointType > point_list_type;

		static point_list_type create_point_list( const PointType &pt )
		{
			point_list_type list( 1 );
			list[ 0 ] = pt;
			return( list );
		}
	};


	template < class PointType >
	struct point_list_converter< std::vector< PointType > >
	{
		typedef std::vector< PointType > point_list_type;

		static point_list_type create_point_list( const point_list_type &list )
		{
			return( list );
		}
	};

	template < class PointType >
	struct point_list_converter< std::deque< PointType > >
	{
		typedef std::deque< PointType > point_list_type;

		static point_list_type create_point_list( const point_list_type &list )
		{
			return( list );
		}
	};


	template < class T >
	inline const T maximum( const T &v0, const typename type_trait< T >::value_type &v1 )
	{
		return( v0 > v1 ? v0 : v1 );
	}

	template < class T >
	inline const T maximum( const T &v0, const typename type_trait< T >::value_type &v1, const typename type_trait< T >::value_type &v2 )
	{
		return( v0 > v1 ? ( v0 > v2 ? v0 : v2 ) : ( v1 > v2 ? v1 : v2 ) );
	}

	template < class T >
	inline const T maximum( const T &v0, const typename type_trait< T >::value_type &v1, const typename type_trait< T >::value_type &v2, const typename type_trait< T >::value_type &v3 )
	{
		return( maximum( maximum( v0, v1 ), maximum( v2, v3 ) ) );
	}
}


/// @brief Cӂ̍\vfƏpė̈gs֐
//!
//! @attention ̓sCmax_paint Ŏw肵f኱߂ɗ̈悪g\܂D
//!
//! @param[in]  in           c ͉摜
//! @param[out] out          c o̓}[Nf[^
//! @param[in]  mask         c ΏۊO}XN
//! @param[in]  start_points c ̈g̊Jn_̃Xgiw肷ꍇ́Cstd::vectorȂǂ̃Xgɑ邱Ɓj
//! @param[in]  output_value c o̓}[Nf[^ɏޒl
//! @param[in]  components   c ̈gɗp\vf
//! @param[in]  condition    c \vf̉fׂ
//! @param[in]  max_paint    c őœhԂf
//!
//! @return ͉摜sK؂ȏꍇCő唽񐔂sĂI𖞂Ȃꍇ -1 ԂDȊȌꍇ́ChԂfԂD
//! 
template < class Array1, class Array2, class MaskType, class PointList, class Component, class Condition >
typename Array1::difference_type region_growing( const Array1 &in, Array2 &out, const MaskType &mask, const PointList &start_points, typename Array2::value_type output_value,
														const Component &components, const Condition &condition, typename Array1::size_type max_paint )
{
	if( in.empty( ) || is_same_object( in, out ) )
	{
		return( -1 );
	}

	typedef typename Array1::template rebind< unsigned char >::other mask_type;
	typedef __region_growing_utility__::pointer_diff2 pointer_diff_type;

	typedef typename Array1::size_type			size_type;
	typedef typename Array1::value_type			value_type;
	typedef typename Array1::difference_type	difference_type;
	typedef typename Array2::value_type			out_value_type;

	typedef typename Array1::const_pointer		const_pointer;
	typedef typename mask_type::pointer			work_pointer;
	typedef typename Array2::pointer			output_pointer;

	size_type rx = components.width( ) / 2;
	size_type ry = components.height( ) / 2;
	size_type rz = components.depth( ) / 2;

	// ړ_͈͊OɏoȂƂĎ}XN
	// ŒłOɂPf̈S̈쐬
	marray< mask_type > work( in, __region_growing_utility__::maximum( rx, ry, rz, 1 ) );

	__region_growing_utility__::ptrdiff_list_type clist = __region_growing_utility__::create_component_list( in, work, components );	// \vf̉f̃Xg
	__region_growing_utility__::ptrdiff_list_type ulist = __region_growing_utility__::create_update_list( in, work, components );		// 钍ړ_̎ɒړ_ɂȂf̃Xg

	std::deque< pointer_diff_type > que;									// ړ_f̃Xg
	std::vector< value_type > element( clist.size( ) );						// g̔ɗpf̔z
	__region_growing_utility__::ptrdiff_list_type elist( clist.size( ) );	// g̔ɗpfւ̃|C^oz

	// 摜̊OɊgȂ悤Ƀ}XNݒ肷
	work.fill( );
	work.fill_margin( 255 );

	// }XN̓e𒍖ړ_̃f[^ɔf
	if( !mask.empty( ) )
	{
		for( size_type k = 0 ; k < in.depth( ) ; k++ )
		{
			for( size_type j = 0 ; j < in.height( ) ; j++ )
			{
				for( size_type i = 0 ; i < in.width( ) ; i++ )
				{
					if( mask( i, j, k ) )
					{
						work( i, j, k ) = 255;
					}
				}
			}
		}
	}

	typedef __region_growing_utility__::point_list_converter< PointList > start_point_list_converter;
	typedef typename start_point_list_converter::point_list_type start_point_list_type;
	start_point_list_type sps = start_point_list_converter::create_point_list( start_points );

	// gJn_摜ɑ݂Ă邩ǂ`FbN
	for( typename start_point_list_type::const_iterator ite = sps.begin( ) ; ite != sps.end( ) ; ++ite )
	{
		__region_growing_utility__::pointer_diff2 ptr;
		if( __region_growing_utility__::check_component_list( in, work, *ite, ptr ) )
		{
			que.push_back( ptr );
		}
	}

	// o͉摜̑傫`FbN
	if( in.size( ) != out.size( ) )
	{
		out.resize( in.width( ), in.height( ), in.depth( ) );
		out.reso1( in.reso1( ) );
		out.reso2( in.reso2( ) );
		out.reso3( in.reso3( ) );
	}

	
	size_type num_painted = 0;	// hԂf̃JE^


	// ̈g̏ɂāC\vf̑Svf𔻒ɗp邩ǂ߂
	// require_all_elements  false ̏ꍇɂ́Cłɔς݂̉f2xڂ̔sȂ
	if( condition.require_all_elements( ) )
	{
		// \vfׂ̂Ă̓_pĕ]s
		while( !que.empty( ) )
		{
			if( num_painted >= max_paint )
			{
				return( num_painted );
			}

			// Xg̐擪fo
			pointer_diff_type cur = que.front( );
			que.pop_front( );

			const_pointer  pi = &in[ 0 ] + cur.diff1;
			work_pointer   pw = &work[ 0 ] + cur.diff2;
			output_pointer po = &out[ 0 ] + cur.diff1;

			// ȍ~̏Œړ_ƂȂȂ悤ɁC݂̒ړ_}XN
			pw[ 0 ] |= 0xf0;

			// gɗpf񋓂
			// ŁC}XNĂ͔͈̈͂Ɋ܂߂Ȃi摜OȂǁj
			size_type num = 0;
			for( size_type i = 0 ; i < clist.size( ) ; i++ )
			{
				const pointer_diff_type &d = clist[ i ];
				if( ( pw[ d.diff2 ] & 0x0f ) == 0 )
				{
					element[ num ] = pi[ d.diff1 ];
					elist[ num ]   = d;
					num++;
				}
			}

			// g̔s
			if( num != 0 && condition( element, num ) )
			{
				// 𖞂\vf̉fׂĂhԂ
				for( size_type i = 0 ; i < num ; i++ )
				{
					const pointer_diff_type &d = elist[ i ];
					num_painted += po[ d.diff1 ] != output_value ? 1 : 0;
					po[ d.diff1 ] = output_value;
				}

				// \vfɂĎ̒ړ_肵CXg̍Ōɒǉ
				for( size_type i = 0 ; i < ulist.size( ) ; i++ )
				{
					const pointer_diff_type &d = ulist[ i ];
					if( ( pw[ d.diff2 ] & 0xf0 ) == 0 )
					{
						pw[ d.diff2 ] |= 0xf0;
						que.push_back( pointer_diff_type( cur.diff1 + d.diff1, cur.diff2 + d.diff2 ) );
					}
				}
			}
		}
	}
	else
	{
		// łɔ肳ꂽfɂĂ͕]sȂ
		while( !que.empty( ) )
		{
			if( num_painted >= max_paint )
			{
				return( num_painted );
			}

			// Xg̐擪fo
			pointer_diff_type cur = que.front( );
			que.pop_front( );

			const_pointer  pi = &in[ 0 ] + cur.diff1;
			work_pointer   pw = &work[ 0 ] + cur.diff2;
			output_pointer po = &out[ 0 ] + cur.diff1;

			// ȍ~̏Œړ_ƂȂȂ悤ɁC݂̒ړ_}XN
			pw[ 0 ] |= 0xf0;

			// gɗpf񋓂
			// ŁC}XNĂ͔͈̈͂Ɋ܂߂Ȃi摜OȂǁj
			// ܂Cłɔς݂̉fO
			// fƍXṼXg쐬
			size_type num = 0;
			for( size_type i = 0 ; i < clist.size( ) ; i++ )
			{
				const pointer_diff_type &d = clist[ i ];
				if( ( pw[ d.diff2 ] & 0x0f ) == 0 )
				{
					element[ num ] = pi[ d.diff1 ];
					elist[ num ]   = d;
					num++;
				}
			}

			// g̔s
			if( condition( element, num ) )
			{
				// 𖞂\vf̉fׂĂhԂ
				for( size_type i = 0 ; i < num ; i++ )
				{
					const pointer_diff_type &d = elist[ i ];
					num_painted += po[ d.diff1 ] != output_value ? 1 : 0;
					pw[ d.diff2 ] |= 1;
					po[ d.diff1 ] = output_value;
				}

				// \vfɂĎ̒ړ_肵CXg̍Ōɒǉ
				for( size_type i = 0 ; i < ulist.size( ) ; i++ )
				{
					const pointer_diff_type &d = ulist[ i ];
					if( ( pw[ d.diff2 ] & 0xf0 ) == 0 )
					{
						pw[ d.diff2 ] |= 0xf0;
						que.push_back( pointer_diff_type( cur.diff1 + d.diff1, cur.diff2 + d.diff2 ) );
					}
				}
			}
		}
	}

	return( num_painted );
}



/// @brief Cӂ̍\vfƏpė̈gs֐
//!
//! ΏۊO}XNݒ肵Ȃ̈g@
//!
//! @attention ̓sCmax_paint Ŏw肵f኱߂ɗ̈悪g\܂D
//!
//! @param[in]  in           c ͉摜
//! @param[out] out          c o̓}[Nf[^
//! @param[in]  start_points c ̈g̊Jn_̃Xgiw肷ꍇ́Cstd::vectorȂǂ̃Xgɑ邱Ɓj
//! @param[in]  output_value c o̓}[Nf[^ɏޒl
//! @param[in]  components   c ̈gɗp\vf
//! @param[in]  condition    c \vf̉fׂ
//! @param[in]  max_paint    c őœhԂfiȗꍇ͏𖞂_݂ȂȂ܂Ŏsj
//!
//! @return ͉摜sK؂ȏꍇCő唽񐔂sĂI𖞂Ȃꍇ -1 ԂDȊȌꍇ́ChԂfԂD
//!
template < class Array1, class Array2, class PointList, class Component, class Condition >
typename Array1::difference_type region_growing( const Array1 &in, Array2 &out, const PointList &start_points, typename Array2::value_type output_value,
										const Component &components, const Condition &condition, typename Array1::size_type max_paint = type_limits< typename Array1::size_type >::maximum( ) )
{
	return( region_growing( in, out, __region_growing_utility__::no_mask( ), start_points, output_value, components, condition, max_paint ) );
}


/// @}
//  ̈g@O[v̏I


// mistOԂ̏I
_MIST_END


#endif // __INCLUDE_MIST_REGIONGROWING__
