// 
// Copyright (c) 2003-2010, MIST Project, Nagoya University
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice,
// this list of conditions and the following disclaimer in the documentation
// and/or other materials provided with the distribution.
// 
// 3. Neither the name of the Nagoya University nor the names of its contributors
// may be used to endorse or promote products derived from this software
// without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
// IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
// THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 

/// @file mist/hash_algorithm.h
//!
//! @brief Cӂ̃oCg̃nbV֐lvZ
//!
//! - Ql
//!   - MD2, RFC1319, http://www.ietf.org/rfc/rfc1319.txt
//!   - MD4, RFC1320, http://www.ietf.org/rfc/rfc1320.txt
//!   - MD5, RFC1321, http://www.ietf.org/rfc/rfc1321.txt
//!   - SHA-1, RFC3174, http://www.ietf.org/rfc/rfc3174.txt
//!   - SHA-256, 384, 512, "Descriptions of SHA-256, SHA-384, and SHA-512." http://csrc.nist.gov/cryptval/shs/sha256-384-512.pdf 
//!
#ifndef __INCLUDE_HASH_ALGORITHM__
#define __INCLUDE_HASH_ALGORITHM__

#ifndef __INCLUDE_MIST_CONF_H__
#include "config/mist_conf.h"
#endif

#include <string>


// mistOԂ̎n܂
_MIST_BEGIN


//! @defgroup hash_group nbV֐
//!
//! @code ̃wb_CN[h
//! #include <mist/hash_algorithm.h>
//! @endcode
//!
//!  @{


/// @brief nbV֐vZNX
//!
//! MD2CMD4CMD5CSHA1CSHA-256CSHA-384CSHA-512 vZ邽߂̊NX
//!
//! @attention SHA-384CSHA-512 vZۂ 64 rbg𗘗p邽߁CRpC 64 rbgT|[gKvD
//!
//! @code nbV֐̌vZ
//! // nbV֐ SHA1 ŏ
//! mist::hash_algorithm &h = mist::sha1( "a" );
//! 
//! // nbV֐l̃_CWFXgo͂
//! std::cout << h << std::endl;
//! 
//! // ʂ̕ōČvZ
//! h.compute_hash( "" );
//! 
//! // nbV֐l̃_CWFXgo͂
//! std::cout << h << std::endl;
//! 
//! @endcode
//!
class hash_algorithm
{
public:
	typedef size_t					size_type;			///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
	typedef ptrdiff_t				difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ
	typedef unsigned char			uint8;				///< @brief Ȃ8rbg\^D̃nbV֐lvẐɗpDIɂ unsigned char ^ƓD
	typedef unsigned int			uint32;				///< @brief Ȃ32rbg\^D̃nbV֐lvẐɗpDIɂ unsigned int ^ƓD
	typedef unsigned long long int	uint64;				///< @brief Ȃ64rbg\^D̃nbV֐lvẐɗpDIɂ unsigned long long int ^ƓD

protected:
	size_type nbytes;
	unsigned char *digest;

public:
	/// @brief data[ 0 ]  data[ len - 1 ]  len oCg̃nbV֐lvZD
	virtual void compute_hash( const void *bytes, uint64 len ) = 0;

	/// @brief  str ̃nbV֐lvZD
	void compute_hash( const std::string &str ){ compute_hash( str.c_str( ), str.length( ) ); }


	/// @brief nbV֐Ԃ
	virtual const std::string name( ) const = 0;

	/// @brief _CWFXgoCg̒
	size_type size( ) const { return( nbytes ); }

	/// @brief _CWFXg̊eoCg擾
	unsigned char operator []( size_type index ) const { return( digest[ index ] ); }


	/// @brief _CWFXgԂ
	std::string to_string( ) const
	{
		static char x16[ 16 ] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
		std::string str;

		for( size_type i = 0 ; i < size( ) ; i++ )
		{
			str += x16[ ( digest[ i ] >> 4 ) & 0x0f ];
			str += x16[ digest[ i ] & 0x0f ];
		}

		return( str );
	}

protected:
	// nbV֐̃Rs[͋֎~II
	hash_algorithm &operator =( const hash_algorithm &h );
	hash_algorithm( const hash_algorithm &h );

protected:
	/// @brief w肳ꂽ_CWFXg̃oCgŏihNXł̂ݗp\j
	hash_algorithm( size_type num_digest_bytes ) : nbytes( num_digest_bytes ), digest( new unsigned char[ nbytes ] ){ }

	/// @brief w肳ꂽ_CWFXgŏihNXł̂ݗp\j
	hash_algorithm( const std::string &str ) : nbytes( str.length( ) / 2 ), digest( new unsigned char[ nbytes ] )
	{
		static uint8 x16[ 256 ] = {	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

		for( size_type i = 0, indx = 0 ; i < size( ) ; i++, indx += 2 )
		{
			size_type s1 = static_cast< uint8 >( str[ indx ] );
			size_type s2 = static_cast< uint8 >( str[ indx + 1 ] );
			digest[ i ] = ( ( x16[ s1 ] << 4 ) & 0xf0 ) | ( x16[ s2 ] & 0x0f );
		}
	}

	/// @brief _CWFXgɗpf[^폜
	virtual ~hash_algorithm( )
	{
		delete [] digest;
	}


public:
	/// @brief 2̃nbV֐lꂩǂ𔻒肷
	bool operator ==( const hash_algorithm &h ) const
	{
		if( size( ) != h.size( ) )
		{
			return( false );
		}
		else
		{
			for( size_type i = 0 ; i < size( ) ; i++ )
			{
				if( digest[ i ] != h[ i ] )
				{
					return( false );
				}
			}
			return( true );
		}
	}

	/// @brief 2̃_CWFXg񂪓ꂩǂ𔻒肷
	bool operator ==( const std::string &str ) const
	{
		return( str == to_string( ) );
	}
};


/// @brief w肳ꂽXg[Ƀf[^o͂
//! 
//! @param[in,out] out c ͂Əo͂sXg[
//! @param[in]     h   c nbV֐IuWFNg
//! 
//! @return ͂ꂽXg[
//! 
inline std::ostream &operator <<( std::ostream &out, const hash_algorithm &h )
{
	out << h.to_string( );
	return( out );
}


/// @}
//  nbV֐O[v̏I


// mistOԂ̏I
_MIST_END


#ifndef __INCLUDE_MD5__
#include "md5.h"
#endif

#ifndef __INCLUDE_SHA__
#include "sha.h"
#endif


#endif // __INCLUDE_HASH_ALGORITHM__
