// 
// Copyright (c) 2003-2010, MIST Project, Nagoya University
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without modification,
// are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright notice,
// this list of conditions and the following disclaimer in the documentation
// and/or other materials provided with the distribution.
// 
// 3. Neither the name of the Nagoya University nor the names of its contributors
// may be used to endorse or promote products derived from this software
// without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
// IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF
// THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 

/// @file mist/thread.h
//!
//! @brief ȕւɃXbh𗘗p\ɂ邽߂̃Cu
//!
#ifndef __INCLUDE_MIST_THREAD__
#define __INCLUDE_MIST_THREAD__


#ifndef __INCLUDE_MIST_CONF_H__
#include "config/mist_conf.h"
#endif

#ifndef __INCLUDE_MIST_SINGLETON__
#include "singleton.h"
#endif


// XbhpCũCN[h
// UNIXnWindowspg
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	// XbhT|[g͂Ȃ̂œɕKvȃCN[ht@C͖
	#define __THREAD_POOL_SUPPORT__		0
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	#if !defined( _WIN32_WINNT )
		#define __THREAD_POOL_SUPPORT__		0
		#pragma message( "To use mist thread pool functionality, you must define _WIN32_WINNT (>=0x0400) in the project setting page." )
	#elif _WIN32_WINNT < 0x0400
		#define __THREAD_POOL_SUPPORT__		0
		#pragma message( "To use mist thread pool functionality, _WIN32_WINNT must be greater than 0x0400." )
	#else
		#define __THREAD_POOL_SUPPORT__		1
	#endif

	#include <windows.h>
	#include <process.h>
	#include <mmsystem.h>
	#pragma comment ( lib, "winmm.lib" )
#else
	#include <pthread.h>
	#include <unistd.h>
	#include <time.h>
	#include <errno.h>
	#include <sys/time.h>
	#define __THREAD_POOL_SUPPORT__		1
#endif

#ifndef INFINITE
#define INFINITE	( ( unsigned long ) -1 )
#endif

#include <map>
#include <vector>
#include <list>
#include <string>


// mistOԂ̎n܂
_MIST_BEGIN



// Xbhp֐
typedef unsigned long ThreadExitCode;
typedef ThreadExitCode ( LPTHREADFUNC ) ( void *thread_param );


// Xbh𑀍삷Cʂϐ
struct thread_dmy_class{ };

// Xbh𑀍삷ԃx[XƂȂNX
struct thread_object
{
	virtual bool create( ) = 0;
	virtual bool wait( unsigned long dwMilliseconds = INFINITE ) = 0;
	virtual bool close( ) = 0;
	virtual ~thread_object( ){ }
};


//! @addtogroup thread_group Xbh
//!
//! @code ̃wb_CN[h
//! #include <mist/thread.h>
//! @endcode
//!
//!  @{


/// @brief p\CPU擾
//! 
//! Windows̏ꍇ GetSystemInfo ֐𗘗pCPU擾D
//! Linuxn̏ꍇ́Csysconf ֐𗘗pCPU擾D
//! 
//! @return p\CPU
//! 
inline size_t get_cpu_num( )
{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	// XbhT|[g͂Ȃ̂CPU͏1
	return( 1 );
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	SYSTEM_INFO sysInfo;
	GetSystemInfo( &sysInfo );
	return( static_cast< size_t >( sysInfo.dwNumberOfProcessors ) );
#else
	return( static_cast< size_t >( sysconf( _SC_NPROCESSORS_ONLN ) ) );
#endif
}


/// @brief w肵ԂX[vi~bPʁj
inline void sleep( size_t dwMilliseconds )
{
#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	SleepEx( static_cast< DWORD >( dwMilliseconds ), false );
#else
	timespec treq, trem;
	treq.tv_sec = static_cast< time_t >( dwMilliseconds / 1000 );
	treq.tv_nsec = static_cast< long >( ( dwMilliseconds % 1000 ) * 1000000 );
	
	while( nanosleep( &treq, &trem ) != 0 )
	{
		treq = trem;
	}
#endif
}


/// @brief Xbh̔rT|[głPȃNX
//! 
//! bNIuWFNgLꍇɎgpłVvȃNX
//! 
//! @attention bNIuWFNg̐_ł́CbN͂ȂƂɒ
//! 
//! - Windows c NeBJZNV𗘗p
//! - Linuxn c ~[ebNX𗘗p
//! 
//! @code gp
//! simple_lock_object l;
//! 
//! ... 炩̏ ...
//! 
//! l.lock( );   // <- ȍ~̏1r䂷
//! 
//! ... 1 ...
//! 
//! l.unlock( ); // <- ܂ł̏1r䂳
//! 
//! 
//! ... 炩̏ ...
//! 
//! 
//! l.lock( );   // <- ȍ~̏2r䂷
//! 
//! ... 2 ...
//! 
//! l.unlock( ); // <- ܂ł̏2r䂳
//! 
//! @endcode
//! 
struct simple_lock_object
{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	typedef char lock_object_type;					// XbhT|[g͂Ȃ̂Ń_~[ϐp
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	typedef CRITICAL_SECTION lock_object_type;		// Windowsp̃bNIuWFNg(CRITIFCALSECTION𗘗p)
#else
	typedef pthread_mutex_t lock_object_type;		// pthreadCuł̃bNIuWFNg
#endif

	lock_object_type __lock__;	///< @brief bNpIuWFNg

public:
	/// @brief RXgN^
	simple_lock_object( )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		//InitializeCriticalSectionAndSpinCount( &__lock__, 4000 );	// NeBJZNVIuWFNg
		InitializeCriticalSection( &__lock__ );	// NeBJZNVIuWFNg
#else
		pthread_mutex_init( &__lock__, NULL );		// pthreadpMutexIuWFNg
#endif
	}

	/// @brief fXgN^
	~simple_lock_object( )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		DeleteCriticalSection( &__lock__ );		// NeBJZNVIuWFNg폜
#else
		pthread_mutex_destroy( &__lock__ );		// pthreadpMutexIuWFNg
#endif
	}

#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0

	// XbhT|[g͂Ȃ̂œɕKȂ
	bool lock( ){ return( true ); }
	bool unlock( ){ return( true ); }
	bool try_lock( ){ return( true ); }

#else

	/// @brief rpIuWFNgbN
	//! 
	//! @retval true  c bNɐ
	//! @retval false c bNɎs
	//! 
	bool lock( )
	{
#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		EnterCriticalSection( &__lock__ );		// NeBJZNVIuWFNgbN
#else
		pthread_mutex_lock( &__lock__ );		// pthreadpMutexIuWFNgbN
#endif

		return( true );
	}

#if defined( __THREAD_POOL_SUPPORT__ ) && __THREAD_POOL_SUPPORT__ != 0
	/// @brief rpIuWFNg̃bN݂
	//! 
	//! ɃbNĂꍇ́CbN false ԂD
	//! 
	//! @retval true  c bNɐ
	//! @retval false c bNɎs
	//! 
	bool try_lock( )
	{
		// NeBJZNVIuWFNg̃bN\𒲂ׂ
#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		return( TryEnterCriticalSection( &__lock__ ) != FALSE );
#else
		return( pthread_mutex_trylock( &__lock__ ) == 0 );
#endif
	}
#endif

	/// @brief rpIuWFNg̃bN
	bool unlock( )
	{
#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		LeaveCriticalSection( &__lock__ );		// NeBJZNVIuWFNgAbN
#else
		pthread_mutex_unlock( &__lock__ );		// pthreadpMutexIuWFNgAbN
#endif

		return( true );
	}

#endif
};

/// @}
//  XbhO[v̏I

template < class MUTEX >
class lock_object_table : public ::std::map< ::std::string, MUTEX >
{
private:
	typedef ::std::map< ::std::string, MUTEX > base;
	typedef MUTEX lock_object_type;

public:
	lock_object_table( )
	{
	}

	~lock_object_table( )
	{
		typename base::iterator ite = base::begin( );
		for( ; ite != base::end( ) ; ++ite )
		{
			destroy( ite->second );
		}
		base::clear( );
	}

protected:
	static void destroy( lock_object_type &l )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		DeleteCriticalSection( &l );		// NeBJZNVIuWFNg폜
#else
		pthread_mutex_destroy( &l );		// pthreadpMutexIuWFNg
#endif
	}
};

//! @addtogroup thread_group Xbh
//!
//!  @{


/// @brief Xbh̔rT|[gNX
//! 
//! bNIuWFNg𖼑OpċL邱Ƃ\ȃNXD
//! ̃bNIuWFNg쐬邱ƂŕʁX̃Xbh֐ł̔r䂪\ƂȂD
//! 
//! @attention bNIuWFNg̐_ł́CbN͂ȂƂɒ
//! 
//! - Windows c NeBJZNV𗘗p
//! - Linuxn c ~[ebNX𗘗p
//! 
//! @code gp
//! lock_object l( "bNIuWFNg̖O" );  // <- Ow肵ȂƁCftHg̃bNIuWFNggp
//! 
//! ... 炩̏ ...
//! 
//! l.lock( );   // <- ȍ~̏1r䂷
//! 
//! ... 1 ...
//! 
//! l.unlock( ); // <- ܂ł̏1r䂳
//! 
//! 
//! ... 炩̏ ...
//! 
//! 
//! l.lock( );   // <- ȍ~̏2r䂷
//! 
//! ... 2 ...
//! 
//! l.unlock( ); // <- ܂ł̏2r䂳
//! 
//! @endcode
//! 
struct lock_object
{
protected:
	::std::string lock_name_;

#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	typedef char lock_object_type;					// XbhT|[g͂Ȃ̂Ń_~[ϐp
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	typedef CRITICAL_SECTION lock_object_type;		// Windowsp̃bNIuWFNg(CRITIFCALSECTION𗘗p)
#else
	typedef pthread_mutex_t lock_object_type;		// pthreadCuł̃bNIuWFNg
#endif

	typedef lock_object_table< lock_object_type > lock_table;

public:
	lock_object( ) : lock_name_( "mist default lock object!!" ){ }
	lock_object( const std::string &name ) : lock_name_( name ){ }

#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0

	// XbhT|[g͂Ȃ̂œɕKȂ
	bool lock( ){ return( true ); }
	bool unlock( ){ return( true ); }

#else

	/// @brief rpIuWFNgbN
	//! 
	//! @retval true  c bNɐ
	//! @retval false c bNɎs
	//! 
	bool lock( )
	{
		static lock_object_type &__double_lock__ = double_lock_object( );

		// e[ǔŃXbhՓ˂Ȃ悤ɃbN
		lock( __double_lock__ );

		lock_table &table = singleton< lock_table >::get_instance( );
		lock_table::iterator ite = table.find( lock_name_ );
		if( ite == table.end( ) )
		{
			// ܂bNIuWFNgĂȂ̂ŏ
			::std::pair< lock_table::iterator, bool > p = table.insert( lock_table::value_type( lock_name_, lock_object_type( ) ) );
			if( p.second )
			{
				lock_object_type &obj = p.first->second;
				initialize( obj );

				// e[up̃bNJ
				unlock( __double_lock__ );

				lock( obj );
			}
			else
			{
				// e[up̃bNJ
				unlock( __double_lock__ );

				// bNIuWFNge[uɒǉ邱Ƃł܂łEEE
				return( false );
			}
		}
		else
		{
			// e[up̃bNJ
			unlock( __double_lock__ );

			// łɓ̃bNIuWFNĝ݂łbN
			lock( ite->second );
		}

		return( true );
	}

#if defined( __THREAD_POOL_SUPPORT__ ) && __THREAD_POOL_SUPPORT__ != 0
	/// @brief rpIuWFNg̃bN݂
	//! 
	//! ɃbNĂꍇ́CbN false ԂD
	//! 
	//! @retval true  c bNɐ
	//! @retval false c bNɎs
	//! 
	bool try_lock( )
	{
		static lock_object_type &__double_lock__ = double_lock_object( );

		// e[ǔŃXbhՓ˂Ȃ悤ɃbN
		lock( __double_lock__ );

		lock_table &table = singleton< lock_table >::get_instance( );
		lock_table::iterator ite = table.find( lock_name_ );
		if( ite == table.end( ) )
		{
			// ܂bNIuWFNgĂȂ̂ŏ
			::std::pair< lock_table::iterator, bool > p = table.insert( lock_table::value_type( lock_name_, lock_object_type( ) ) );
			if( p.second )
			{
				lock_object_type &obj = p.first->second;
				initialize( obj );

				// e[up̃bNJ
				unlock( __double_lock__ );

				return( try_lock( obj ) );
			}
			else
			{
				// e[up̃bNJ
				unlock( __double_lock__ );

				// bNIuWFNge[uɒǉ邱Ƃł܂łEEE
				return( false );
			}
		}
		else
		{
			// e[up̃bNJ
			unlock( __double_lock__ );

			// łɓ̃bNIuWFNĝ݂łbN
			return( try_lock( ite->second ) );
		}
	}
#endif

	/// @brief rpIuWFNg̃bN
	bool unlock( )
	{
		lock_table &table = singleton< lock_table >::get_instance( );
		lock_table::iterator ite = table.find( lock_name_ );
		if( ite == table.end( ) )
		{
			// w肳ꂽbNIuWFNgȂ̂ŉȂ
			return( false );
		}
		else
		{
			unlock( ite->second );
		}
		return( true );
	}

#endif


protected:
	/// @brief Ŏgp_ubNIuWFNg擾
	static lock_object_type &double_lock_object( )
	{
		static bool isFirst = true;
		static lock_object_type __double_lock__;
		if( isFirst )
		{
			isFirst = false;
			initialize( __double_lock__ );
		}
		return( __double_lock__ );
	}

	/// @brief bNIuWFNg̏s
	//! 
	//! @param[in,out] l c bNIuWFNg
	//! 
	static void initialize( lock_object_type &l )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		InitializeCriticalSection( &l );	// NeBJZNVIuWFNg
#else
		pthread_mutex_init( &l, NULL );		// pthreadpMutexIuWFNg
#endif
	}


	/// @brief bNIuWFNgbN
	//! 
	//! @param[in,out] l c bNIuWFNg
	//! 
	static void lock( lock_object_type &l )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		EnterCriticalSection( &l );		// NeBJZNVIuWFNgbN
#else
		pthread_mutex_lock( &l );		// pthreadpMutexIuWFNgbN
#endif
	}

#if defined( __THREAD_POOL_SUPPORT__ ) && __THREAD_POOL_SUPPORT__ != 0
	/// @brief rpIuWFNg̃bN݂
	//! 
	//! ɃbNĂꍇ́CbN false ԂD
	//! 
	//! @param[in,out] l c bNIuWFNg
	//! 
	//! @retval true  c bNɐ
	//! @retval false c bNɎs
	//! 
	static bool try_lock( lock_object_type &l )
	{
		// NeBJZNVIuWFNg̃bN\𒲂ׂ
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
		return( true );
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		return( TryEnterCriticalSection( &l ) != FALSE );
#else
		return( pthread_mutex_trylock( &l ) == 0 );
#endif
	}
#endif

	/// @brief bNIuWFNgbN
	//! 
	//! @param[in,out] l c bNIuWFNg
	//! 
	static void unlock( lock_object_type &l )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂œɕKȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		LeaveCriticalSection( &l );		// NeBJZNVIuWFNgAbN
#else
		pthread_mutex_unlock( &l );		// pthreadpMutexIuWFNgAbN
#endif
	}
};




/// @brief Xbh̔rȕւɋLq邽߂̃wpNX
//! 
//! @code gp
//! ... 炩̏ ...
//! {
//!     // ̃ubNr䂷
//!     lock l( "bNIuWFNg̖O" );  // <- Ow肵ȂƁCftHg̃bNIuWFNggp
//! 
//!     ... r䂵 ...
//! 
//!     // ܂ł̏r䂳
//! }
//! @endcode
//! 
class lock
{
protected:
	lock_object lock_object_;	///< @brief bNIuWFNg

public:
	/// @brief ftHg̖OŃbNIuWFNg𐶐CbNIuWFNgbN
	lock( ) : lock_object_( )
	{
		lock_object_.lock( );
	}


	/// @brief w肵 name ̖ObNIuWFNg𐶐CbNIuWFNgbN
	lock( const std::string &name ) : lock_object_( name )
	{
		lock_object_.lock( );
	}


	/// @brief JɁCbNIuWFNg̃bN
	~lock( )
	{
		lock_object_.unlock( );
	}
};


#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
/// @brief XbhԂŃVOiL邽߂̃NX
class signal
{
protected:
	HANDLE handle_;

public:
	/// @brief RXgN^
	signal( ) : handle_( CreateEvent( NULL, FALSE, FALSE, NULL ) )
	{
	}

	/// @brief fXgN^
	~signal( )
	{
		CloseHandle( handle_ );
	}

	/// @brief VOiԂɂȂ܂őҋ@
	bool wait( unsigned long dwMilliseconds = INFINITE )
	{
		DWORD ret = WaitForSingleObject( handle_, dwMilliseconds );
		if( SUCCEEDED( ret ) )
		{
			ResetEvent( handle_ );
			return ( true );
		}
		else
		{
			return( false );
		}
	}

	/// @brief VOi𑗐M
	void send( )
	{
		SetEvent( handle_ );
	}

	/// @brief VOiNAɂ
	void reset( )
	{
		ResetEvent( handle_ );
	}
};
#else
class signal
{
protected:
	pthread_cond_t     cond_;
	pthread_mutex_t    mutex_;
	volatile bool      flag_;

public:
	/// @brief RXgN^
	signal( ) : flag_( false )
	{
		pthread_mutex_init( &mutex_, NULL );
		pthread_cond_init( &cond_, NULL );
		pthread_mutex_lock ( &mutex_ );
	}

	/// @brief fXgN^
	~signal( )
	{
		pthread_cond_destroy( &cond_ );
		pthread_mutex_destroy( &mutex_ );
	}

	/// @brief VOiԂɂȂ܂őҋ@
	bool wait( unsigned long dwMilliseconds = INFINITE )
	{
		if( dwMilliseconds == INFINITE )
		{
			while( !flag_ )
			{
				if( pthread_cond_wait( &cond_, &mutex_ ) == 0 )
				{
					break;
				}
			}
			flag_ = false;
			return( true );
		}
		else
		{
			timeval now;
			gettimeofday( &now, NULL );

			timespec tm;
			tm.tv_sec  = now.tv_sec + static_cast< time_t >( dwMilliseconds / 1000 );
			tm.tv_nsec = now.tv_usec * 1000 + static_cast< long >( ( dwMilliseconds % 1000 ) * 1000000 );

			tm.tv_sec += tm.tv_nsec / 1000000000;
			tm.tv_nsec = tm.tv_nsec % 1000000000;

			while( !flag_ )
			{
				if( pthread_cond_timedwait( &cond_, &mutex_, &tm ) == 0 )
				{
					break;
				}
				else
				{
					return( false );
				}
			}

			flag_ = false;
			return( true );
		}
	}

	/// @brief VOi𑗐M
	void send( )
	{
		flag_ = true;
		pthread_cond_broadcast( &cond_ );
	}

	/// @brief VOiNAɂipthread̏ꍇ͉Ȃj
	void reset( )
	{
		flag_ = false;
	}
};
#endif


/// @brief template ^̃f[^ƂłXbhNX
//! 
//! ڍׂȐ֐̎gp
//! 
//! @attention _MIST_THREAD_SUPPORT_  1 ̏ꍇ́CXbhT|[g͍sȂ߁CVOXbhpɓύX
//! 
//! @param thread_parameter c ̐
//! 
template < class thread_parameter = thread_dmy_class >
class thread : public thread_object
{
public:
	typedef unsigned long thread_exit_type;		///< @brief XbhI̖߂ľ^

private:

#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	// XbhT|[g͂Ȃ̂œɕKvȕϐ͖
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	HANDLE       thread_handle_;		// Windowsp̃Xbhʂnh
	unsigned int thread_id_;			// Windowsp̃XbhʂID
#else
	pthread_t thread_id_;				// pthreadCuŃXbhʂID
	signal    finish_;					// Xbh̏I҂sVOi
	bool      joined_;					// Xbhɑ΂ pthread_join IĂ邩ǂ
#endif

	simple_lock_object exit_;				// I҂VOi
	thread_exit_type   thread_exit_code_;	// Xbh̖߂l

public:

	/// @brief XbhIɕԂ߂l擾
	thread_exit_type exit_code( ) const { return( thread_exit_code_ ); }


	/// @brief ̃XbhIuWFNgƓ̂쐬
	//! 
	//! @attention PȂRs[łCʓrXbh킯ł͂Ȃ
	//! 
	//! @param[in] t c Rs[̃XbhIuWFNg
	//!
	//! @return g
	//! 
	const thread &operator =( const thread &t )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		thread_handle_ = t.thread_handle_;
		thread_id_ = t.thread_id_;
#else
		thread_id_ = t.thread_id_;
		joined_    = t.joined_;
#endif
		thread_exit_code_ = t.thread_exit_code_;
		return( *this );
	}


	/// @brief 2̃XbhIuWFNĝǂ𔻒肷
	//! 
	//! ڍׂȐ֐̎gp
	//! 
	//! @param[in] t c rΏۂ̃XbhIuWFNg
	//!
	//! @retval true  c Xbh
	//! @retval false c ʂ̃Xbh
	//! 
	bool operator ==( const thread &t ) const
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ
		return( this == &t );
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		return( thread_id_ == t.thread_id_ );
#else
		return( pthread_equal( thread_id_, t.thread_id_ ) != 0 );
#endif
	}

	bool operator !=( const thread &t ) const
	{
		return( !( *this == t ) );
	}


#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	// XbhT|[g͂Ȃ
	thread( const thread &t ) : thread_exit_code_( t.thread_exit_code_ ){ }
	thread( ) : thread_exit_code_( 0 ){ }
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	thread( const thread &t ) : thread_handle_( t.thread_handle_ ), thread_id_( t.thread_id_ ), thread_exit_code_( t.thread_exit_code_ ){ }
	thread( ) : thread_handle_( NULL ), thread_id_( ( unsigned int )-1 ), thread_exit_code_( 0 ){ }
#else
	thread( const thread &t ) : thread_id_( t.thread_id ), joined_( false ), thread_exit_code_( t.thread_exit_code ){ }
	thread( ) : thread_id_( ( pthread_t ) ( -1 ) ), joined_( false ), thread_exit_code_( 0 ){ }
#endif

	virtual ~thread( )
	{
		this->close( );
	}


	/// @brief Xbh𐶐
	//! 
	//! @retval true  c Xbh̍쐬ɐ
	//! @retval false c Xbh̍쐬Ɏs
	//! 
	virtual bool create( )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂Œڊ֐Ăяo
		bool ret = true;
		thread_exit_code_ = thread_function( );
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		if( thread_handle_ != NULL ) return( false );
		thread_handle_ = ( HANDLE )_beginthreadex( NULL, 0, map_thread_function, ( void * )this, 0, &thread_id_ );
		bool ret = thread_handle_ != NULL ? true : false;
#else
		if( thread_id_ != ( pthread_t ) ( -1 ) ) return( false );
		bool ret = pthread_create( &( thread_id_ ), NULL, map_thread_function, ( void * )this ) == 0 ? true : false;
		joined_ = false;
#endif

		return ( ret );
	}


	/// @brief XbhŊ֐s
	//! 
	//! @retval true  c Xbh̍쐬ɐ
	//! @retval false c Xbh̍쐬Ɏs
	//! 
	virtual bool create_without_thread( )
	{
		thread_exit_code_ = thread_function( );
		return ( true );
	}


	/// @brief XbhI邩C^CAEgɂȂ܂őҋ@
	//! 
	//! ^CAEg INFINITE ɐݒ肷邱ƂŁCXbhI܂ő҂
	//! 
	//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
	//! 
	//! @retval true  c Xbh^CAEgOɏI
	//! @retval false c ^CAEgC̑̃G[
	//! 
	virtual bool wait( unsigned long dwMilliseconds = INFINITE )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂ŉȂ
		// XbhȂ߁CdwMilliseconds ͏ INFINITE ƂȂ
		return( true );
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		DWORD ret = WaitForSingleObject( thread_handle_, dwMilliseconds );
		return ( ret == WAIT_OBJECT_0 );
#else
		if( dwMilliseconds == INFINITE )
		{
			if( pthread_join( thread_id_, NULL ) == 0 )
			{
				joined_ = true;
				return( true );
			}
			else
			{
				return( false );
			}
		}
		else
		{
			if( finish_.wait( dwMilliseconds ) )
			{
				if( pthread_join( thread_id_, NULL ) == 0 )
				{
					joined_ = true;
					return( true );
				}
				else
				{
					return( false );
				}
			}
			else
			{
				return( false );
			}
		}
#endif
	}


	/// @brief XbhgpĂ\[XJ
	//! 
	//! Xbhgp\[XJ邽߂ɁCK1̃Xbhɑ΂1xĂ΂Kv
	//! 
	//! @retval true  c ɃXbhI
	//! @retval false c Xbh̏IɎs
	//! 
	virtual bool close( )
	{
#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
		// XbhT|[g͂Ȃ̂ŏ true Ԃ
		return( true );
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
		if( thread_handle_ != NULL )
		{
			// Xbh̊SIҋ@
#if defined( __THREAD_POOL_SUPPORT__ ) && __THREAD_POOL_SUPPORT__ != 0
			while( !exit_.try_lock( ) ){}
			exit_.unlock( );
#endif

			BOOL ret = CloseHandle( thread_handle_ );
			thread_handle_ = NULL;
			return ( ret != 0 );
		}
		else
		{
			return( true );
		}
#else
		if( thread_id_ != ( pthread_t ) ( -1 ) )
		{
			// Xbh̊SIҋ@
			while( !exit_.try_lock( ) ){}
			exit_.unlock( );

			if( !joined_ )
			{
				pthread_join( thread_id_, NULL );
			}

			thread_id_ = ( pthread_t ) ( -1 );
			return( true );
		}
		else
		{
			return( true );
		}
#endif
	}


protected:
	/// @brief pŕKXbh֐
	//! 
	//! {NXpCXbh̋@\ꍇɕKȂĂ͂ȂȂ֐D
	//! Xbh̐ƂƂɖ{֐Ă΂D
	//! 
	//! @return XbhI̖߂l
	//! 
	virtual thread_exit_type thread_function( ) = 0;


#if !defined( _MIST_THREAD_SUPPORT_ ) || _MIST_THREAD_SUPPORT_ == 0
	// XbhT|[g͂Ȃ̂ŉȂ
#elif defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	static unsigned int __stdcall map_thread_function( void *p )
	{
		thread *obj = static_cast< thread * >( p );
		obj->exit_.lock( );
		obj->thread_exit_code_ = obj->thread_function( );
		obj->exit_.unlock( );
		return( 0 );
	}
#else
	static void *map_thread_function( void *p )
	{
		thread *obj = static_cast< thread * >( p );
		obj->exit_.lock( );
		obj->thread_exit_code_ = obj->thread_function( );
		obj->exit_.unlock( );
		obj->finish_.send( );
		return ( NULL );
	}
#endif
};


/// @brief XbhEIҋ@EXbh̔j܂łA̗Ƃčs
//! 
//! @param[in] threads        c XbhIuWFNg
//! @param[in] num_threads    c Xbh
//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c ̃Xbh̎sɐ
//! @retval false c ̃Xbh̎sɎs
//! 
template < class Thread >
inline bool do_threads_( Thread *threads, size_t num_threads, unsigned long dwMilliseconds = INFINITE )
{
	bool ret = true;
	size_t i = 0;

	// Xbh̐
	for( i = 1 ; i < num_threads ; i++ )
	{
		if( !threads[ i ].create( ) )
		{
			ret = false;
		}
	}
	if( num_threads > 0 )
	{
		// 擪̃p[^̂ݎsXbhōs
		threads[ 0 ].create_without_thread( );
	}

	// Xbh̏I҂
	for( i = 1 ; i < num_threads ; i++ )
	{
		if( !threads[ i ].wait( dwMilliseconds ) )
		{
			ret = false;
		}
	}

	// \[X̊J
	for( i = 1 ; i < num_threads ; i++ )
	{
		if( !threads[ i ].close( ) )
		{
			ret = false;
		}
	}

	return( ret );
}

/// @brief XbhEIҋ@EXbh̔j܂łA̗Ƃčs
//! 
//! @param[in] threads        c XbhIuWFNg
//! @param[in] num_threads    c Xbh
//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c ̃Xbh̎sɐ
//! @retval false c ̃Xbh̎sɎs
//! 
template < class Thread >
inline bool do_threads_( Thread **threads, size_t num_threads, unsigned long dwMilliseconds = INFINITE )
{
	bool ret = true;
	size_t i = 0;

	// Xbh̐
	for( i = 1 ; i < num_threads ; i++ )
	{
		if( !threads[ i ]->create( ) )
		{
			ret = false;
		}
	}
	if( num_threads > 0 )
	{
		// 擪̃p[^̂ݎsXbhōs
		threads[ 0 ]->create_without_thread( );
	}

	// Xbh̏I҂
	for( i = 1 ; i < num_threads ; i++ )
	{
		if( !threads[ i ]->wait( dwMilliseconds ) )
		{
			ret = false;
		}
	}

	// \[X̊J
	for( i = 1 ; i < num_threads ; i++ )
	{
		if( !threads[ i ]->close( ) )
		{
			ret = false;
		}
	}

	return( ret );
}



/// @brief XbhEIҋ@EXbh̔j܂łA̗Ƃčs
//! 
//! @param[in] threads        c XbhIuWFNg
//! @param[in] num_threads    c Xbh
//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c ̃Xbh̎sɐ
//! @retval false c ̃Xbh̎sɎs
//! 
template < class Thread >
inline bool do_threads( Thread *threads, size_t num_threads, unsigned long dwMilliseconds = INFINITE )
{
	bool ret = true;
	size_t i = 0;

	// Xbh̐
	for( i = 0 ; i < num_threads ; i++ )
	{
		if( !threads[ i ].create( ) )
		{
			ret = false;
		}
	}

	// Xbh̏I҂
	for( i = 0 ; i < num_threads ; i++ )
	{
		if( !threads[ i ].wait( dwMilliseconds ) )
		{
			ret = false;
		}
	}

	// \[X̊J
	for( i = 0 ; i < num_threads ; i++ )
	{
		if( !threads[ i ].close( ) )
		{
			ret = false;
		}
	}

	return( ret );
}


/// @}
//  XbhO[v̏I

// Xbh𕁒ʂ̊֐`̌ĂяoŊȕւɗp邽߂̊֐Q
namespace __thread_controller__
{
#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
	inline unsigned long _timeGetTime_( )
	{
		return( timeGetTime( ) );
	}
#else
	inline unsigned long _timeGetTime_( )
	{
		timeval dmy;
		gettimeofday( &dmy, NULL );
		return( dmy.tv_sec );
	}
#endif

	template < class Param, class Functor >
	class thread_object_functor : public thread< thread_object_functor< Param, Functor > >
	{
	public:
		typedef thread< thread_object_functor< Param, Functor > > base;
		typedef typename base::thread_exit_type thread_exit_type;

	private:
		Param   &param_;
		Functor func_;

	public:
		thread_object_functor( Param &p, Functor f ) : param_( p ), func_( f ){ }
		virtual ~thread_object_functor( ){ }

	protected:
		// pŕKXbh֐
		virtual thread_exit_type thread_function( )
		{
			func_( param_ );
			return( 0 );
		}
	};

#if defined( __THREAD_POOL_SUPPORT__ ) && __THREAD_POOL_SUPPORT__ != 0
	struct __thread_pool_functor__
	{
		typedef size_t    size_type;
		typedef ptrdiff_t difference_type;

		virtual void run( size_type id, size_type nthreads ) = 0;

		virtual ~__thread_pool_functor__( ){}
	};

	template < class Param, class Functor >
	class thread_pool_functor_base : public __thread_pool_functor__
	{
	private:
		Param   param_;
		Functor func_;

	public:
		thread_pool_functor_base( Param p, Functor f ) : param_( p ), func_( f ){ }
		virtual ~thread_pool_functor_base( ){}

	protected:
		// pŕKXbh֐
		virtual void run( size_type /* id */, size_type /* nthreads */ )
		{
			func_( param_ );
		}
	};

	template < class Param1, class Param2, class Functor >
	class thread_pool_functor_base2 : public __thread_pool_functor__
	{
	private:
		Param1  param1_;
		Param2  param2_;
		Functor func_;

	public:
		thread_pool_functor_base2( Param1 p1, Param2 p2, Functor f ) : param1_( p1 ), param2_( p2 ), func_( f ){ }
		virtual ~thread_pool_functor_base2( ){ }

	protected:
		// pŕKXbh֐
		virtual void run( size_type /* id */, size_type /* nthreads */ )
		{
			func_( param1_, param2_ );
		}
	};

	template < class Param1, class Param2, class Param3, class Functor >
	class thread_pool_functor_base3 : public __thread_pool_functor__
	{
	private:
		Param1  param1_;
		Param2  param2_;
		Param3  param3_;
		Functor func_;

	public:
		thread_pool_functor_base3( Param1 p1, Param2 p2, Param3 p3, Functor f ) : param1_( p1 ), param2_( p2 ), param3_( p3 ), func_( f ){ }
		virtual ~thread_pool_functor_base3( ){ }

	protected:
		// pŕKXbh֐
		virtual void run( size_type /* id */, size_type /* nthreads */ )
		{
			func_( param1_, param2_, param3_ );
		}
	};

	template < class Param1, class Param2, class Param3, class Param4, class Functor >
	class thread_pool_functor_base4 : public __thread_pool_functor__
	{
	private:
		Param1  param1_;
		Param2  param2_;
		Param3  param3_;
		Param4  param4_;
		Functor func_;

	public:
		thread_pool_functor_base4( Param1 p1, Param2 p2, Param3 p3, Param4 p4, Functor f ) : param1_( p1 ), param2_( p2 ), param3_( p3 ), param4_( p4 ), func_( f ){ }
		virtual ~thread_pool_functor_base4( ){ }

	protected:
		// pŕKXbh֐
		virtual void run( size_type /* id */, size_type /* nthreads */ )
		{
			func_( param1_, param2_, param3_, param4_ );
		}
	};


	template < class Functor >
	class thread_pool_void_functor_base : public __thread_pool_functor__
	{
	private:
		Functor func_;

	public:
		thread_pool_void_functor_base( Functor f ) : func_( f ){ }
		virtual ~thread_pool_void_functor_base( ){ }

	protected:
		// pŕKXbh֐
		virtual void run( size_type /* id */, size_type /* nthreads */ )
		{
			func_( );
		}
	};

	class thread_pool_functor : public thread< thread_pool_functor >
	{
	public:
		typedef thread< thread_pool_functor > base;
		typedef base::thread_exit_type thread_exit_type;
		typedef size_t    size_type;
		typedef ptrdiff_t difference_type;

	private:
		std::list< __thread_pool_functor__ * > &functors_;
		simple_lock_object                     &lock_;
		simple_lock_object                     suspend_lock_;
		simple_lock_object                     wait_lock_;
		bool                                   is_running_;
		bool                                   is_end_;
		size_type                              id_;
		size_type                              nthreads_;
		signal                                 signal_;

	public:
		thread_pool_functor( std::list< __thread_pool_functor__ * > &functors, simple_lock_object &l, size_type id, size_type nthreads )
			: functors_( functors ), lock_( l ), is_running_( false ), is_end_( false ), id_( id ), nthreads_( nthreads )
		{
			// bNԂŃX^[g
			suspend_lock_.lock( );
		}

		virtual ~thread_pool_functor( ){ }

		size_type num_jobs( ) const { return( functors_.size( ) ); }
		bool is_end( ) const { return( is_end_ ); }

		bool is_suspended( )
		{
			if( wait_lock_.try_lock( ) )
			{
				wait_lock_.unlock( );
				return( true );
			}
			else
			{
				return( false );
			}
		}

		virtual bool create( )
		{
			if( base::create( ) )
			{
				// Xbh̊Jn܂őҋ@
				while( true )
				{
					lock_.lock( );
					if( is_running_ )
					{
						lock_.unlock( );
						break;
					}
					else
					{
						lock_.unlock( );
					}
				}

				return( true );
			}
			else
			{
				return( false );
			}
		}

		virtual bool close( )
		{
			lock_.lock( );
			// ItO𗧂ĂăXbhċN
			is_end_ = true;
			this->resume( );
			lock_.unlock( );

			// XbhI܂ő҂
			while( !wait_lock_.try_lock( ) ){}
			wait_lock_.unlock( );

			return( base::close( ) );
		}

		void resume( )
		{
			if( is_suspended( ) )
			{
				suspend_lock_.unlock( );
				while( wait_lock_.try_lock( ) )
				{
					wait_lock_.unlock( );
				}
				suspend_lock_.lock( );
			}
		}

		bool wait( unsigned long dwMilliseconds = INFINITE )
		{
			lock_.lock( );
			if( !wait_lock_.try_lock( ) )
			{
				lock_.unlock( );
				return( signal_.wait( dwMilliseconds ) );
			}
			else
			{
				wait_lock_.unlock( );
				lock_.unlock( );
				return( true );
			}
		}

	protected:
		// pŕKXbh֐
		virtual thread_exit_type thread_function( )
		{
			wait_lock_.lock( );

			lock_.lock( );
			is_running_ = true;
			lock_.unlock( );

			while( true )
			{
				lock_.lock( );
				if( is_end_ )
				{
					lock_.unlock( );
					break;
				}
				else if( functors_.empty( ) )
				{
					// ҂
					wait_lock_.unlock( );
					lock_.unlock( );
					signal_.send( );
					suspend_lock_.lock( );
					signal_.reset( );
					wait_lock_.lock( );
					suspend_lock_.unlock( );
				}
				else
				{
					// L[̐擪f[^o
					__thread_pool_functor__ *f = functors_.front( );
					functors_.pop_front( );
					lock_.unlock( );

					if( f != NULL )
					{
						f->run( id_, nthreads_ );
						delete f;
					}
				}
			}

			wait_lock_.unlock( );
			signal_.send( );

			return( 0 );
		}
	};
#endif
}


//! @addtogroup thread_group Xbh
//!
//!  @{

#if defined( __THREAD_POOL_SUPPORT__ ) && __THREAD_POOL_SUPPORT__ != 0
/// @brief Xbhv[𗘗păXbh̍ėp\ɂNX
//! 
//! Xbhv[𗘗pďsꍇCpɃXbhv[̃Xbh蓖Ă܂D
//! Iꍇ́CXbh̓Xbhv[ɉCʂ̏֊蓖Ă܂D
//! Xbhėp邱ƂɂCXbh̐jɕKvȃI[o[wbh팸܂D
//! 
//! @attention N̓XbhɔI[o[wbh܂
//! 
//! @code gp
//! struct parameter{ ... 炩̃p[^ ... };   // Xbh֐ɓnp[^iɍ\̂łKv͖j
//! 
//! void thread_function( const parameter &p );       // Xbh֐iŉ炩̏j
//! 
//! // ȍ~ŃXbh쐬
//! parameter param[ s ];
//! 
//! for( i = 0 ; i < s ; i++ )
//! {
//!     p[^ɉ炩̒lݒ肷
//! }
//! 
//! // Xbhv[̃Xbhw肵ď
//! mist::thread_pool pool( 2 );
//! 
//! // ֐p[^ςĕ񏈗
//! pool.executes( thread_function, param, s );
//! 
//! // Xbhv[̃XbhsĂ鏈SďI܂őҋ@
//! pool.wait( );
//! @endcode
//! 
class thread_pool
{
public:
	typedef size_t    size_type;		///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
	typedef ptrdiff_t difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ

private:
	typedef __thread_controller__::thread_pool_functor thread_pool_functor;
	typedef __thread_controller__::__thread_pool_functor__ __thread_pool_functor__;

	std::vector< thread_pool_functor * >   threads_;		///< @brief Xbhv[̃XbhǗϐ
	std::list< __thread_pool_functor__ * > functors_;		///< @brief Xbhv[̏҂̃WuǗ郊Xg
	simple_lock_object                     lock_;			///< @brief Xbhv[̃XbhԂœ邽߂̃bNIuWFNg
	bool                                   initialized_;	///< @brief Xbhv[̏IĂ邩ǂ

public:
	/// @brief Xbhv[̃IuWFNg쐬邪CXbhv[̃Xbh0̏Ԃŏiinitialize֐ĂяoKvj
	thread_pool( ) : initialized_( false )
	{
	}

	/// @brief Xbhv[̃Xbhw肵CɃXbhv[gԂŏ
	thread_pool( size_type number_of_max_threads ) : initialized_( false )
	{
		initialize( number_of_max_threads );
	}

	/// @brief Xbhv[ŎgpĂSẴ\[XJ
	virtual ~thread_pool( )
	{
		uninitialize( );
	}

	/// @brief Xbhv[ŎgpXbh̐w肵ăXbhv[
	//! 
	//! ɃXbhv[Ăꍇ́C݂̃XbhSďI܂őҋ@
	//! 
	//! @param[in] number_of_max_threads c Xbhv[ɂ߂ĂXbh̐
	//! 
	bool initialize( size_type number_of_max_threads )
	{
		// ݎs̃XbhSďI
		uninitialize( );

		if( number_of_max_threads == 0 )
		{
			// 1ȏw肷Kv
			return( false );
		}

		threads_.resize( number_of_max_threads );

		// XbhsăAChԂɂ
		for( size_type i = 0 ; i < threads_.size( ) ; i++ )
		{
			threads_[ i ] = new thread_pool_functor( functors_, lock_, i, threads_.size( ) );
			threads_[ i ]->create( );
			//threads_[ i ]->wait( );
		}

		initialized_ = true;

		return( true );
	}

	/// @brief Xbhv[ŎgpĂ郊\[XSĊJ
	//! 
	//! ݂̃XbhSďI܂őҋ@CXbhv[ŎgpĂ郊\[XSĊJ
	//! 
	bool uninitialize( )
	{
		if( initialized_ )
		{
			for( size_type i = 0 ; i < threads_.size( ) ; i++ )
			{
				// Xbh̃nh
				threads_[ i ]->close( );

				// gpĂ̈J
				delete threads_[ i ];
			}

			threads_.clear( );

			// L[ɎcĂf[^폜
			lock_.lock( );
			while( !functors_.empty( ) )
			{
				__thread_pool_functor__ *f = functors_.front( );
				functors_.pop_front( );
				delete f;
			}
			lock_.unlock( );
		}

		initialized_ = false;

		return( true );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p c Xbh̊֐ɓnp[^
	//! @param[in]     f c sXbh֐
	//! 
	template < class Functor, class Param >
	bool execute( Functor f, Param p )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base< Param, Functor >( p, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p1 c Xbh̊֐ɓnp[^
	//! @param[in,out] p2 c Xbh̊֐ɓnp[^
	//! @param[in]     f  c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2 >
	bool execute( Functor f, Param1 p1, Param2 p2 )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base2< Param1, Param2, Functor >( p1, p2, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p1 c Xbh̊֐ɓnp[^
	//! @param[in,out] p2 c Xbh̊֐ɓnp[^
	//! @param[in,out] p3 c Xbh̊֐ɓnp[^
	//! @param[in]     f  c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2, class Param3 >
	bool execute( Functor f, Param1 p1, Param2 p2, Param3 p3 )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base3< Param1, Param2, Param3, Functor >( p1, p2, p3, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p1 c Xbh̊֐ɓnp[^
	//! @param[in,out] p2 c Xbh̊֐ɓnp[^
	//! @param[in,out] p3 c Xbh̊֐ɓnp[^
	//! @param[in,out] p4 c Xbh̊֐ɓnp[^
	//! @param[in]     f  c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2, class Param3, class Param4 >
	bool execute( Functor f, Param1 p1, Param2 p2, Param3 p3, Param4 p4 )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base4< Param1, Param2, Param3, Param4, Functor >( p1, p2, p3, p4, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! 
	//! @param[in] f c sXbh֐
	//! 
	template < class Functor >
	bool execute( Functor f )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_void_functor_base< Functor >( f ) ) );
	}

	/// @brief ֐ƃp[^𕡐w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] param       c Xbh̊֐ɓnp[^
	//! @param[in]     num_threads c Xbh
	//! @param[in]     f           c sXbh֐
	//! 
	template < class Functor, class Param >
	bool executes( Functor f, Param *param, size_t num_threads )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		// L[ɒǉ
		lock_.lock( );
		for( size_type i = 0 ; i < num_threads ; i++ )
		{
			functors_.push_back( new __thread_controller__::thread_pool_functor_base< Param, Functor >( param[ i ], f ) );
		}
		lock_.unlock( );

		resume_thread_from_queue( num_threads );

		return( true );
	}

	/// @brief ֐ƃp[^𕡐w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] param1      c Xbh̊֐ɓnp[^
	//! @param[in,out] param2      c Xbh̊֐ɓnp[^
	//! @param[in]     num_threads c Xbh
	//! @param[in]     f           c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2 >
	bool executes( Functor f, Param1 *param1, Param2 *param2, size_t num_threads )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		// L[ɒǉ
		lock_.lock( );
		for( size_type i = 0 ; i < num_threads ; i++ )
		{
			functors_.push_back( new __thread_controller__::thread_pool_functor_base2< Param1, Param2, Functor >( param1[ i ], param2[ i ], f ) );
		}
		lock_.unlock( );

		resume_thread_from_queue( num_threads );

		return( true );
	}

	/// @brief ֐ƃp[^𕡐w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] param1      c Xbh̊֐ɓnp[^
	//! @param[in,out] param2      c Xbh̊֐ɓnp[^
	//! @param[in,out] param3      c Xbh̊֐ɓnp[^
	//! @param[in]     num_threads c Xbh
	//! @param[in]     f           c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2, class Param3 >
	bool executes( Functor f, Param1 *param1, Param2 *param2, Param3 *param3, size_t num_threads )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		// L[ɒǉ
		lock_.lock( );
		for( size_type i = 0 ; i < num_threads ; i++ )
		{
			functors_.push_back( new __thread_controller__::thread_pool_functor_base3< Param1, Param2, Param3, Functor >( param1[ i ], param2[ i ], param3[ i ], f ) );
		}
		lock_.unlock( );

		resume_thread_from_queue( num_threads );

		return( true );
	}

	/// @brief ֐ƃp[^𕡐w肵ăXbhs
	//! 
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] param1      c Xbh̊֐ɓnp[^
	//! @param[in,out] param2      c Xbh̊֐ɓnp[^
	//! @param[in,out] param3      c Xbh̊֐ɓnp[^
	//! @param[in,out] param4      c Xbh̊֐ɓnp[^
	//! @param[in]     num_threads c Xbh
	//! @param[in]     f           c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2, class Param3, class Param4 >
	bool executes( Functor f, Param1 *param1, Param2 *param2, Param3 *param3, Param4 *param4, size_t num_threads )
	{
		if( threads_.empty( ) || !initialized_ )
		{
			return( false );
		}

		// L[ɒǉ
		lock_.lock( );
		for( size_type i = 0 ; i < num_threads ; i++ )
		{
			functors_.push_back( new __thread_controller__::thread_pool_functor_base4< Param1, Param2, Param3, Param4, Functor >( param1[ i ], param2[ i ], param3[ i ], param4[ i ], f ) );
		}
		lock_.unlock( );

		resume_thread_from_queue( num_threads );

		return( true );
	}

	/// @brief SẴXbhI邩C^CAEgɂȂ܂őҋ@
	//! 
	//! ^CAEg INFINITE ɐݒ肷邱ƂŁCXbhI܂ő҂D
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! 
	//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
	//! 
	//! @retval true  c Xbh^CAEgOɏI
	//! @retval false c ^CAEgC̑̃G[
	//! 
	virtual bool wait( unsigned long dwMilliseconds = INFINITE )
	{
		if( !initialized_ )
		{
			return( false );
		}
		else if( threads_.empty( ) )
		{
			return( false );
		}

		unsigned long st = __thread_controller__::_timeGetTime_( );
		size_type i = 0;
		for( ; i < threads_.size( ) ; i++ )
		{
			if( !threads_[ i ]->wait( dwMilliseconds ) )
			{
				break;
			}
			else if( dwMilliseconds != INFINITE )
			{
				unsigned long ct = __thread_controller__::_timeGetTime_( );
				dwMilliseconds -= ct - st;
				st = ct;
			}
		}

		return( i < threads_.size( ) );
	}

protected:
	/// @brief ֐Xbhv[̃XbhɊ蓖ĂĎs
	//! 
	//! @param[in] func c sXbh֐
	//! 
	bool exec( __thread_controller__::__thread_pool_functor__ *func )
	{
		// L[ɒǉ
		{
			// r䂷
			lock_.lock( );
			functors_.push_back( func );
			lock_.unlock( );
		}

		resume_thread_from_queue( 1 );

		return( true );
	}

	/// @brief Xbhv[KvȐXbhĊJ
	//! 
	//! @param[in] num_threads c W[Xbh
	//! 
	void resume_thread_from_queue( size_type num_threads )
	{
		size_type count = 0;
		for( size_type i = 0 ; i < threads_.size( ) ; i++ )
		{
			thread_pool_functor &t = *threads_[ i ];

			if( t.is_suspended( ) )
			{
				// AChԂ̃Xbh΍ĊJ
				t.resume( );

				count++;
			}

			if( count >= num_threads )
			{
				break;
			}
		}
	}
};



/// @brief obNOEhŎs郏[J[XbhNX
//! 
//! [J[Xbhɂ͕̃Wuݒ肷邱ƂłCeWus܂D
//! SẴWuIƁC[J[Xbh͑ҋ@ԂɓC̃Wu҂܂D
//! 
//! @attention N̓XbhɔI[o[wbh܂
//! 
//! @code gp
//! struct parameter{ ... 炩̃p[^ ... };   // Xbh֐ɓnp[^iɍ\̂łKv͖j
//! 
//! void thread_function( const parameter &p );       // Xbh֐iŉ炩̏j
//! 
//! // ȍ~ŃXbh쐬
//! parameter param;
//! 
//! // [J[Xbh
//! mist::worker_thread work;
//! 
//! // ֐s
//! work.execute( thread_function, param );
//! 
//! // [J[XbhsĂ鏈SďI܂őҋ@
//! work.wait( );
//! @endcode
//! 
class worker_thread
{
public:
	typedef size_t    size_type;		///< @brief Ȃ̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ size_t ^Ɠ
	typedef ptrdiff_t difference_type;	///< @brief t̐\^DRei̗vfCevfw肷ƂȂǂɗpCIɂ ptrdiff_t ^Ɠ

private:
	__thread_controller__::thread_pool_functor                    *thread_;		///< @brief [J[XbhǗ郁CXbh
	std::list< __thread_controller__::__thread_pool_functor__ * >  functors_;	///< @brief [J[Xbh̏҂̃WuǗ郊Xg
	simple_lock_object                                             lock_;		///< @brief [J[Xbh̃XbhԂœ邽߂̃bNIuWFNg

public:
	/// @brief [J[Xbhp̃Xbh쐬Ԃőҋ@
	worker_thread( ) : thread_( NULL )
	{
		thread_ = new __thread_controller__::thread_pool_functor( functors_, lock_, 0, 1 );
		thread_->create( );
		//thread_->wait( );
	}

	/// @brief [J[XbhŎgpĂSẴ\[XJ
	virtual ~worker_thread( )
	{
		close( );
	}

	/// @brief [J[XbhTXyhĂ邩ǂ
	bool is_suspended( )
	{
		if( thread_ == NULL )
		{
			return( true );
		}

		return( thread_->is_suspended( ) );
	}

	/// @brief [J[XbhŎgpĂ郊\[XSĊJ
	//! 
	//! Ƀ[J[XbhIC[J[XbhŎgpĂ郊\[XSĊJ
	//! 
	bool close( )
	{
		if( thread_ == NULL )
		{
			return( false );
		}

		// Xbh̃nh
		thread_->close( );

		// gpĂ̈J
		delete thread_;
		thread_ = NULL;

		// L[ɎcĂf[^폜
		lock_.lock( );
		while( !functors_.empty( ) )
		{
			__thread_controller__::__thread_pool_functor__ *f = functors_.front( );
			functors_.pop_front( );
			delete f;
		}
		lock_.unlock( );

		return( true );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! [J[XbhIԂł false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p c Xbh̊֐ɓnp[^
	//! @param[in]     f c sXbh֐
	//! 
	template < class Functor, class Param >
	bool execute(  Functor f, Param p )
	{
		if( thread_ == NULL )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base< Param, Functor >( p, f ) ) );
	}


	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! [J[XbhIԂł false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in] p1 c Xbh̊֐ɓnp[^1
	//! @param[in] p2 c Xbh̊֐ɓnp[^2
	//! @param[in] f  c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2 >
	bool execute( Functor f, Param1 p1, Param2 p2 )
	{
		if( thread_ == NULL )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base2< Param1, Param2, Functor >( p1, p2, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! [J[XbhIԂł false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p1 c Xbh̊֐ɓnp[^
	//! @param[in,out] p2 c Xbh̊֐ɓnp[^
	//! @param[in,out] p3 c Xbh̊֐ɓnp[^
	//! @param[in]     f  c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2, class Param3 >
	bool execute( Functor f, Param1 p1, Param2 p2, Param3 p3 )
	{
		if( thread_ == NULL )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base3< Param1, Param2, Param3, Functor >( p1, p2, p3, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! [J[XbhIԂł false ԂD
	//! Xbhs֐̈̓Rs[ēnD
	//! 
	//! @param[in,out] p1 c Xbh̊֐ɓnp[^
	//! @param[in,out] p2 c Xbh̊֐ɓnp[^
	//! @param[in,out] p3 c Xbh̊֐ɓnp[^
	//! @param[in,out] p4 c Xbh̊֐ɓnp[^
	//! @param[in]     f  c sXbh֐
	//! 
	template < class Functor, class Param1, class Param2, class Param3, class Param4 >
	bool execute( Functor f, Param1 p1, Param2 p2, Param3 p3, Param4 p4 )
	{
		if( thread_ == NULL )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_functor_base4< Param1, Param2, Param3, Param4, Functor >( p1, p2, p3, p4, f ) ) );
	}

	/// @brief ֐ƃp[^w肵ăXbhs
	//! 
	//! [J[XbhIԂł false ԂD
	//! 
	//! @param[in]     f     c sXbh֐
	//! 
	template < class Functor >
	bool execute( Functor f )
	{
		if( thread_ == NULL )
		{
			return( false );
		}

		return( exec( new __thread_controller__::thread_pool_void_functor_base< Functor >( f ) ) );
	}

	/// @brief SĂ̏I邩C^CAEgɂȂ܂őҋ@
	//! 
	//! ^CAEg INFINITE ɐݒ肷邱ƂŁCXbhI܂ő҂D
	//! Xbhv[̏IĂȂꍇ false ԂD
	//! 
	//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
	//! 
	//! @retval true  c Xbh^CAEgOɏI
	//! @retval false c ^CAEgC̑̃G[
	//! 
	virtual bool wait( unsigned long dwMilliseconds = INFINITE )
	{
		if( thread_ == NULL )
		{
			return( true );
		}
		else
		{
			return( thread_->wait( dwMilliseconds ) );
		}
	}

protected:
	/// @brief ֐Xbhv[̃XbhɊ蓖ĂĎs
	//! 
	//! @param[in] func c sXbh֐
	//! 
	bool exec( __thread_controller__::__thread_pool_functor__ *func )
	{
		// L[ɒǉ
		{
			// r䂷
			lock_.lock( );
			functors_.push_back( func );
			lock_.unlock( );
		}

		// XbhĊJ
		thread_->resume( );


		return( true );
	}
};
#endif

/// @brief XbhǗNX
class thread_handle
{
private:
	thread_object *thread_;		///< @brief Xbh̎

public:

	/// @brief Xbh쐬
	bool create( ){ return( thread_ == NULL ? false : thread_->create( ) ); }


	/// @brief XbhI邩C^CAEg܂őҋ@
	//! 
	//! @param[in] dwMilliseconds c ^CAEgԁi~bPʁj
	//! 
	bool wait( unsigned long dwMilliseconds = INFINITE ){ return( thread_ == NULL ? false : thread_->wait( dwMilliseconds ) ); }


	/// @brief XbhgpĂ\[XJ
	bool close( )
	{
		if( thread_ == NULL )
		{
			return( false );
		}
		bool b = thread_->close( );
		delete thread_;
		thread_ = NULL;
		return( b );
	}

public:
	thread_handle( ) : thread_( NULL ){ }
	thread_handle( thread_object *t ) : thread_( t ){ }

	const thread_handle &operator =( const thread_handle &t )
	{
		if( &t != this )
		{
			thread_ = t.thread_;
		}
		return( *this );
	}
};

/// @brief w肵Xbh֐𗘗pCXbh쐬
//! 
//! @param[in,out] param c Xbh̊֐ɓnp[^
//! @param[in]     f     c sXbh֐
//!
//! @return XbhǗ thread_handle IuWFNg
//! 
//! @code gp
//! struct parameter{ ... 炩̃p[^ ... };   // Xbh֐ɓnp[^iɍ\̂łKv͖j
//! 
//! void thread_function( const parameter &p );       // Xbh֐iŉ炩̏j
//! 
//! // ȍ~ŃXbh쐬
//! parameter param[ 쐬Xbh ];
//! mist::thread_handle t[ 쐬Xbh ];
//! 
//! for( i = 0 ; i < 쐬Xbh ; i++ )
//! {
//!     p[^ɉ炩̒lݒ肷
//! }
//! 
//! for( i = 0 ; i < 쐬Xbh ; i++ )
//! {
//!     // Xbh쐬
//!     t[ i ] = mist::create_thread( param[ i ], thread_function );
//! }
//! 
//! for( i = 0 ; i < 쐬Xbh ; i++ )
//! {
//!     // XbhI܂ő҂
//!     mist::wait_thread( t[ i ] );
//! }
//! 
//! for( i = 0 ; i < 쐬Xbh ; i++ )
//! {
//!     // XbhgpĂ\[XJ
//!     mist::close_thread( t[ i ] );
//! }
//! @endcode
//! 
template < class Param, class Functor >
inline thread_handle create_thread( Param &param, Functor f )
{
	thread_handle thread_( new __thread_controller__::thread_object_functor< Param, Functor >( param, f ) );
	thread_.create( );
	return( thread_ );
}

/// @brief w肵Xbh֐𗘗pCXbh𕡐쐬
//! 
//! @param[out]    handles     c XbhIuWFNg
//! @param[in,out] param       c Xbh̊֐ɓnp[^
//! @param[in]     num_threads c Xbh
//! @param[in]     f           c sXbh֐
//! 
template < class Param, class Functor >
inline void create_threads( thread_handle *handles, Param *param, size_t num_threads, Functor f )
{
	for( size_t i = 0 ; i < num_threads ; i++ )
	{
		handles[ i ] = thread_handle( new __thread_controller__::thread_object_functor< Param, Functor >( param[ i ], f ) );
        handles[ i ].create( );
	}
}


/// @brief XbhgpĂ\[XJ
//! 
//! XbhgpŁCKĂяoKv
//! 
//! @param[in,out] thread_ c XbhIuWFNg
//!
//! @retval true  c \[X̊Jɐ
//! @retval false c \[X̊JɎs
//! 
inline bool close_thread( thread_handle &thread_ ){ return( thread_.close( ) ); }

/// @brief ̃XbhgpĂ\[XJ
//! 
//! @param[out]    handles     c XbhIuWFNg
//! @param[in]     num_threads c Xbh
//!
//! @retval true  c SẴ\[X̊Jɐ
//! @retval false c SẴ\[X̊JɎs
//! 
inline bool close_threads( thread_handle *handles, size_t num_threads )
{
	bool ret = true;
	for( size_t i = 0 ; i < num_threads ; i++ )
	{
        if( !handles[ i ].close( ) )
		{
			ret = false;
		}
	}
	return( ret );
}


/// @brief XbhI邩C^CAEg܂őҋ@
//! 
//! ^CAEg INFINITE ɐݒ肷邱ƂŁCXbhI܂ő҂
//! 
//! @param[in,out] thread_        c XbhIuWFNg
//! @param[in]     dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c Xbh^CAEgԓɐIꍇ
//! @retval false c Xbh^CAEgԓɏIȂꍇ
//! 
inline bool wait_thread( thread_handle &thread_, unsigned long dwMilliseconds = INFINITE ){ return( thread_.wait( dwMilliseconds ) ); }

/// @brief ̃XbhgpĂ\[XJ
//! 
//! @param[out]    handles        c XbhIuWFNg
//! @param[in]     num_threads    c Xbh
//! @param[in]     dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c ̃Xbh^CAEgԓɐIꍇ
//! @retval false c ̃Xbh^CAEgԓɏIȂꍇ
//! 
inline bool wait_threads( thread_handle *handles, size_t num_threads, unsigned long dwMilliseconds = INFINITE )
{
	if( dwMilliseconds == INFINITE )
	{
		bool ret = true;
		for( size_t i = 0 ; i < num_threads ; i++ )
		{
			if( !handles[ i ].wait( INFINITE ) )
			{
				ret = false;
			}
		}
		return( ret );
	}
	else
	{
		bool ret = true;
		for( size_t i = 0 ; i < num_threads ; i++ )
		{
#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
			DWORD st = timeGetTime( );
#else
			timeval dmy;
			gettimeofday( &dmy, NULL );
			unsigned long st = dmy.tv_sec * 1000 + dmy.tv_usec / 1000;
#endif
			if( !handles[ i ].wait( dwMilliseconds ) )
			{
				ret = false;
			}

#if defined( __MIST_WINDOWS__ ) && __MIST_WINDOWS__ > 0
			DWORD et = timeGetTime( );
#else
			gettimeofday( &dmy, NULL );
			unsigned long et = dmy.tv_sec * 1000 + dmy.tv_usec / 1000;
#endif

			if( st + dwMilliseconds <= et )
			{
				break;
			}
			else
			{
				dwMilliseconds -= et - st;
			}
		}
		return( ret );
	}
}


/// @brief XbhEIҋ@EXbh̔j܂łA̗Ƃčs
//! 
//! @param[in,out] param          c Xbh̊֐ɓnp[^
//! @param[in]     f              c sXbh֐
//! @param[in]     dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c Xbh̎sɐ
//! @retval false c Xbh̎sɎs
//! 
template < class Param, class Functor >
inline bool do_thread( Param &param, Functor f, unsigned long dwMilliseconds = INFINITE )
{
	bool ret = true;

	// Xbh̐
	thread_handle thread_ = create_thread( param, f );

	// Xbh̏I҂
	if( !wait_thread( thread_, dwMilliseconds ) )
	{
		ret = false;
	}

	// \[X̊J
	if( !close_thread( thread_ ) )
	{
		ret = false;
	}
	return( ret );
}


/// @brief XbhEIҋ@EXbh̔j܂łA̗Ƃčs
//! 
//! @param[in,out] params         c Xbh̊֐ɓnp[^
//! @param[in]     num_threads    c Xbh
//! @param[in]     f              c sXbh֐
//! @param[in]     dwMilliseconds c ^CAEgԁi~bPʁj
//!
//! @retval true  c ̃Xbh̎sɐ
//! @retval false c ̃Xbh̎sɎs
//! 
template < class Param, class Functor >
inline bool do_threads( Param *params, size_t num_threads, Functor f, unsigned long dwMilliseconds = INFINITE )
{
	bool ret = true;
	thread_handle *threads_ = new thread_handle[ num_threads ];

	// Xbh̐
	create_threads( threads_, params, num_threads, f );

	// Xbh̏I҂
	if( !wait_threads( threads_, num_threads, dwMilliseconds ) )
	{
		ret = false;
	}

	// \[X̊J
	if( !close_threads( threads_, num_threads ) )
	{
		ret = false;
	}

	delete [] threads_;

	return( ret );
}


/// @}
//  XbhO[v̏I


// mistOԂ̏I
_MIST_END


#endif // __INCLUDE_MIST_THREAD__
