/*
 * KY-Logger
 * 2014/3/10
 * 酒井先生プログラムをベース
 * 
 * センサ
 * 　LPS331
 * 　SHT1x or 7x
 * 
 * 湿度センサーSHTにはV3とV4の２種類あり係数が違うので注意
 * 　Sensirion.cppに係数の定義があるので適宜切り替える
 * 
 */ 
// 2013.12.03 気圧センサ付きシールドに対応
// 2013.12.21 気圧センサ　温度補正　LGGER_PRO08
//
#include <Arduino.h>
#include <avr/sleep.h>      // IDE に同封されているライブラリをインクルード
#include <SPI.h>
#include <Wire.h>
#include "Sensirion.h"	// ソース(.c .h)を同じディレクトリに配置
#include <SD.h>
#include <EEPROM.h>
#define BCD2Decimal(x)		(((x>>4)*10)+(x&0xf))

// Loger ID
byte ID;
float Ver=1.0;

// LED
const int LED_RED = 8;
const int LED_BLUE = 9;

// SHT1xデバイス
const uint8_t dataPin1  =  4;  // データ端子定義
const uint8_t clockPin1 =  5;  // クロック端子定義

float temperature1; // 温度格納用変数
float humidity1;    // 湿度格納用変数
float dewpoint1;    // 露点温度格納用変数

Sensirion tempSensor1 = Sensirion(dataPin1, clockPin1);

// 気圧センサ
int sensorAddress = 0xBA >> 1; // LPS331APのアドレス，SA0=GNDで0xB8,SA0=VDDで0xBA
float pressure=0, tempPress, dp; 
int8_t Ap,Bp,Cp;

// SD カード
const int chipSelect = 10;     // SD port Sparkfun のシールドは 8
File dataFile;

// RTC
int RTC8564_SLAVE_ADRS=0x51; // RTC のスレーブアドレス
byte cntl1, cntl2, sec, minute, hour, day, week, month, year;
char* chWeek[] = {
  "SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"} ;
char fName[13]={
  'I','Y','m', 'd', 'd','h', 'm', 'm', '.', 'C', 'S', 'V', '\0'};

// 電源モニタ

float v_batt=0;

/*
 * DEBUG Serial OUT
 * mySoftwareSerail.cppを使用
 */
#define	DEBUG_SERIAL	// デバッグ用シリアル出力　使用時に定義する

#ifdef	DEBUG_SERIAL
	#define	DEBUG_SERIAL_INIT()	Serial.begin(9600)
	// 文字列をFLASHとRAMに格納する
	#define	PDEBUG(s)		Serial.print(s)
	#define	PDEBUG_ARG(s,t)	Serial.print(s,t)
	#define	PDEBUGLN(s)		Serial.println(s)
	#define	PDEBUGLN_ARG(s,t)	Serial.println(s,t)
	// 文字列をFLASHのみに格納する
	#define	PDEBUG_PGM(s)	PDEBUG_PGM0(PSTR(s))
	#define	PDEBUGLN_PGM(s)	PDEBUGLN_PGM0(PSTR(s))
#else
	#define	DEBUG_SERIAL_INIT()
	#define	PDEBUG(s)	
	#define	PDEBUG_ARG(s,t)
	#define	PDEBUGLN(s)	
	#define	PDEBUGLN_ARG(s,t)

	#define	PDEBUG_PGM(s)
	#define	PDEBUGLN_PGM(s)
#endif


//=================================================================
void setup()
{
//  LED
  pinMode(LED_BLUE,OUTPUT);
  pinMode(LED_RED,OUTPUT);
  digitalWrite(LED_BLUE,LOW);
  digitalWrite(LED_RED,LOW);
  
  for (byte i=0; i<2; i++){
    blinkLED(LED_RED,100);
  }
  DEBUG_SERIAL_INIT();
//  Serial.begin(9600);
  Wire.begin();
  delay(100);

// Logger ID
  ID = EEPROM.read(0);

 // 気圧センサ
  i2cWriteByte(sensorAddress, 0x20, 0x00); // パワーダウン（クリーンスタート）
  i2cWriteByte(sensorAddress, 0x10, 0x7a); // high precision (512(p),128(t); default)
  i2cWriteByte(sensorAddress, 0x20, 0x84); // one shot
  Ap = EEPROM.read(1);
  Bp = EEPROM.read(2);
  Cp = EEPROM.read(3);

  PDEBUG("ID:");
  PDEBUG(ID);
  PDEBUG(" Ap:");
  PDEBUG(Ap);
  PDEBUG(" Bp:");
  PDEBUG(Bp);
  PDEBUG(" Cp:");
  PDEBUGLN(Cp);

  //  インタラプト設定
  pinMode(2,INPUT_PULLUP);               // インタラプトピン
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);   // スリープモード設定

  // SD card 
  pinMode(SS, OUTPUT);// S(10)Sピン出力にする必要があり。
  if (!SD.begin(chipSelect)) {
    while(1){ // 失敗、(無限ループ）
      blinkLED(LED_RED,20);
    }
  }
  
  getTime();
  setFileName();
  SdFile::dateTimeCallback( &dateTime );  // 日付と時刻を返す関数を登録
  dataFile = SD.open(fName, FILE_WRITE);  // ファイルを開く
  if (dataFile) {                         // もしファイルが開けたら値を書き込む
    dataFile.println("Logger ID, Vich ver., Ap, Bp, Cp");
    dataFile.print(ID); 
    dataFile.print(", ");
    dataFile.print(Ver);
    dataFile.print(", ");
    dataFile.print(Ap);
    dataFile.print(", ");
    dataFile.print(Bp);
    dataFile.print(", ");
    dataFile.println(Cp);

    dataFile.println("DATE,TIME,Temp,Dewpt,Humid,Pressure,P_Temp,dP,Battery");
    dataFile.close();                     // 終了処理  
  }
  
  // Timer off
  i2cWriteByte(RTC8564_SLAVE_ADRS, 0x0E, 0x00);   // TE ->0
  i2cWriteByte(RTC8564_SLAVE_ADRS, 0x01, 0x00);   // TF,TIE->0
  // Timer start
  i2cWriteByte(RTC8564_SLAVE_ADRS, 0x01, 0x11);   // Periodic Timer Interrupt Enable
  i2cWriteByte(RTC8564_SLAVE_ADRS, 0x0F, 0x3c);   // Timer preset 測定間隔(10s:0a, 30s:1e, 60s:3c)
  i2cWriteByte(RTC8564_SLAVE_ADRS, 0x0E, 0x82);   // Timer start, Freq = 1Hz  
}
//==============================================================
void loop()
{
  blinkLED(LED_BLUE,50);
  getTime();  // 時間取得
  tempSensor1.measure(&temperature1, &humidity1, &dewpoint1);  // 温湿度測定
  getPress(); // 気圧測定

  int v = analogRead(0); // バッテリモニタ
//  v_batt = v*9.98/1024.;
  v_batt = v*9.9/1024.;	// 1/3に分圧しているので
  
  // ファイル書き込み
  writeDataSD();
  printData();

  for (byte i=0; i<2; i++){
    blinkLED(LED_BLUE,50);
  }

  //  待機
  attachInterrupt(0, wakeup, FALLING);  // D2 HIGH → LOW で割り込み
  sleep_mode();  // ここでスリープ
}
//=============================================================
void wakeup()
{
  detachInterrupt(0);  // 割り込み停止
}
//-------------------------------------------------
void getPress()
{
  unsigned long read2c, read2b, read2a, read29, read28;
  float P_reading=0, T_reading=0;

  pressure=0;
  tempPress=0;
  byte num_read=32;
  for(byte i=0; i<num_read; i++){
    i2cWriteByte(sensorAddress, 0x21, 0x01); // one shot start
    byte stat;
    do {
      delay(10);
      i2cReadBegin(sensorAddress, 0x21, 1);
      stat = Wire.read(); 
    } while(stat != 0);

    i2cReadBegin(sensorAddress, 0x28, 1);
    read28 = Wire.read(); // 気圧の下位バイト 
    i2cReadBegin(sensorAddress, 0x29, 1);
    read29 = Wire.read(); // 気圧の中位バイト 
    i2cReadBegin(sensorAddress, 0x2a, 1);
    read2a = Wire.read(); // 気圧の上位バイト
    i2cReadBegin(sensorAddress, 0x2b, 1);
    read2b = Wire.read(); // 温度の下位バイト 
    i2cReadBegin(sensorAddress, 0x2c, 1);
    read2c = Wire.read(); // 温度の上位バイト 

    P_reading =(read2a<<16) + (read29<<8) + read28;
    P_reading /=4096; // hPa単位に直す
    pressure += P_reading;
  
    int temp = (read2c<<8) + read2b;
    T_reading = temp;
    T_reading= 42.5 + T_reading/480;
    tempPress += T_reading;
  }
  pressure  /= num_read;
  tempPress /= num_read;  
  dp = (0.005+0.00001*Ap) * (tempPress-20)*(tempPress-20)
        + (-0.1+0.001*Bp) * (tempPress-20) + 0.01*Cp;
  PDEBUG("P:");
  PDEBUG(pressure);
  PDEBUG(" dp:");
  PDEBUGLN(dp);
  
  pressure -= dp;  
}
//-------------------------------------------------
void getTime()
{
  //　時刻データを取得
  i2cReadBegin(RTC8564_SLAVE_ADRS, 0x02, 7);
  sec    = Wire.read() & 0x7F ;   //  2
  minute = Wire.read() & 0x7F ;   //  3
  hour   = Wire.read() & 0x3F ;   //  4
  day    = Wire.read() & 0x3F ;   //  5
  week   = Wire.read() & 0x07 ;   //  6
  month  = Wire.read() & 0x1F ;   //  7
  year   = Wire.read();           //  8    
}
//-------------------------------------------------
void dateTime(uint16_t* date, uint16_t* time)
{
  uint16_t yearF;
  uint8_t monthF, dayF, hourF, minuteF, secondF;

  yearF   = BCD2Decimal(year)+2000;
  monthF  = BCD2Decimal(month);
  dayF    = BCD2Decimal(day);
  hourF   = BCD2Decimal(hour);
  minuteF = BCD2Decimal(minute);
  secondF = BCD2Decimal(sec);

  *date = FAT_DATE(yearF, monthF, dayF);
  *time = FAT_TIME(hourF, minuteF, secondF);
}
//-------------------------------------------------
void setFileName()
{
  fName[0] = chDigit(ID);
  fName[1] = chDigit(BCD2Decimal(year));
  fName[2] = chDigit(BCD2Decimal(month));
  fName[3] = (day   >>  4) + 48;
  fName[4] = (day   & 0xF) + 48;
  fName[5] = chDigit(BCD2Decimal(hour));
  fName[6] = (minute >>  4) + 48;
  fName[7] = (minute & 0xF) + 48;
}
//------------------------------------------------
char chDigit(byte num)
{
  char CD;
  if(num<10) {CD = num+48;}
  else{CD = num+55;}
  return (CD);
}
//-------------------------------------------------
void writeDataSD()
{
  File dataFile = SD.open(fName, FILE_WRITE);
  if (dataFile) {
    dataFile.print(year,HEX);   // 日付
    dataFile.print("/");
    dataFile.print(month,HEX);
    dataFile.print("/");
    dataFile.print(day,HEX);
    dataFile.print(",");
  
    dataFile.print(hour,HEX);    // 時刻
    dataFile.print(":");
    dataFile.print(minute,HEX);
    dataFile.print(":");
    dataFile.print(sec,HEX);
    dataFile.print(","); 
  
    dataFile.print(temperature1);  // 測定データ1
    dataFile.print(",");
    dataFile.print(dewpoint1);
    dataFile.print(",");
    dataFile.print(humidity1);
    dataFile.print(",");         
    dataFile.print(pressure);  // 気圧
    dataFile.print(",");         
    dataFile.print(tempPress);  // 気圧センサ温度
    dataFile.print(",");         
    dataFile.print(dp);  // 気圧補正値
    dataFile.print(",");         

    dataFile.println(v_batt);    // 電源電圧
    dataFile.close();            // 終了処理
  }
  else {
    //PDEBUGln(F("error opening datalog.txt"));
    while(1){ // 失敗、(無限ループ）
      blinkLED(LED_RED,20);
    }
  } 
}
//-------------------------------------------------
void blinkLED(byte pin, int period)
{
  digitalWrite(pin, HIGH);
  delay(period);
  digitalWrite(pin,LOW);
  delay(period*4);
}
//--------------------- I2C -------------------
// I2Cで指定したセンサの指定アドレスから１バイト読み出す
void i2cReadBegin(int deviceAddress, int registerAddress, int numBytes){
  Wire.beginTransmission(deviceAddress); 
  Wire.write(registerAddress);
  Wire.endTransmission();
  Wire.requestFrom(deviceAddress, numBytes);
}

// I2Cで指定したセンサの指定アドレスに１バイト書き込む 
void i2cWriteByte(int deviceAddress, int registerAddress, int data){
  Wire.beginTransmission(deviceAddress); 
  Wire.write(registerAddress);
  Wire.write(data);
  Wire.endTransmission();
} 
//-------------------------------------------------
void printData()
{  
  PDEBUG_ARG(year,HEX);
  PDEBUG("/");
  PDEBUG_ARG(month,HEX);
  PDEBUG("/");
  PDEBUG_ARG(day,HEX);
  PDEBUG(" (");
  PDEBUG(chWeek[week]);
  PDEBUG(")\t");

  PDEBUG_ARG(hour,HEX);
  PDEBUG(":");
  PDEBUG_ARG(minute,HEX);
  PDEBUG(":");
  PDEBUG_ARG(sec,HEX);

  PDEBUG("   \t");
  PDEBUG(temperature1);
  PDEBUG("C ");
  PDEBUG(dewpoint1);
  PDEBUG("C ");
  PDEBUG(humidity1);
  PDEBUG("% ");
  PDEBUG(pressure);
  PDEBUG("hPa ");
  PDEBUG(tempPress);
  PDEBUG("C ");
  PDEBUG(v_batt);
  PDEBUGLN("V ");
}
