/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.TreeMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.RawLocalFileSystem;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;

public class ProcessBasedLocalHBaseCluster {
    private static final String DEFAULT_WORKDIR = "/tmp/hbase-" + System.getenv("USER");
    private final String hbaseHome;
    private final String workDir;
    private int numRegionServers;
    private final int zkClientPort;
    private final int masterPort;
    private final Configuration conf;
    private static final int MAX_FILE_SIZE_OVERRIDE = 10000000;
    private static final Log LOG = LogFactory.getLog(ProcessBasedLocalHBaseCluster.class);
    private List<String> daemonPidFiles = Collections.synchronizedList(new ArrayList());
    private boolean shutdownHookInstalled;
    private String hbaseDaemonScript;

    public ProcessBasedLocalHBaseCluster(Configuration conf, String hbaseHome, int numRegionServers) {
        this.conf = conf;
        this.hbaseHome = hbaseHome;
        this.numRegionServers = numRegionServers;
        this.workDir = DEFAULT_WORKDIR;
        this.hbaseDaemonScript = hbaseHome + "/bin/hbase-daemon.sh";
        this.zkClientPort = HBaseTestingUtility.randomFreePort();
        this.masterPort = HBaseTestingUtility.randomFreePort();
        conf.set("hbase.zookeeper.quorum", "localhost");
        conf.setInt("hbase.zookeeper.property.clientPort", this.zkClientPort);
    }

    public void start() throws IOException {
        this.cleanupOldState();
        LOG.info((Object)"Starting ZooKeeper");
        this.startZK();
        HBaseTestingUtility.waitForHostPort("localhost", this.zkClientPort);
        this.startMaster();
        ZKUtil.waitForBaseZNode((Configuration)this.conf);
        for (int idx = 0; idx < this.numRegionServers; ++idx) {
            this.startRegionServer(HBaseTestingUtility.randomFreePort());
        }
        LOG.info((Object)"Waiting for HBase startup by scanning META");
        int attemptsLeft = 10;
        while (attemptsLeft-- > 0) {
            try {
                new HTable(this.conf, HConstants.META_TABLE_NAME);
            }
            catch (Exception e) {
                LOG.info((Object)("Waiting for HBase to startup. Retries left: " + attemptsLeft), (Throwable)e);
                Threads.sleep((long)1000L);
            }
        }
        LOG.info((Object)("Process-based HBase Cluster with " + this.numRegionServers + " region servers up and running... \n\n"));
    }

    public void startRegionServer(int port) {
        this.startServer("regionserver", port);
    }

    public void startMaster() {
        this.startServer("master", 0);
    }

    public void killRegionServer(int port) throws IOException {
        this.killServer("regionserver", port);
    }

    public void killMaster() throws IOException {
        this.killServer("master", 0);
    }

    public void startZK() {
        this.startServer("zookeeper", 0);
    }

    private void executeCommand(String command) {
        this.ensureShutdownHookInstalled();
        this.executeCommand(command, null);
    }

    private void executeCommand(String command, Map<String, String> envOverrides) {
        LOG.debug((Object)("Command : " + command));
        try {
            String[] envp = null;
            if (envOverrides != null) {
                HashMap<String, String> map = new HashMap<String, String>(System.getenv());
                map.putAll(envOverrides);
                envp = new String[map.size()];
                int idx = 0;
                for (Map.Entry e : map.entrySet()) {
                    envp[idx++] = (String)e.getKey() + "=" + (String)e.getValue();
                }
            }
            Process p = Runtime.getRuntime().exec(command, envp);
            BufferedReader stdInput = new BufferedReader(new InputStreamReader(p.getInputStream()));
            BufferedReader stdError = new BufferedReader(new InputStreamReader(p.getErrorStream()));
            String s = null;
            while ((s = stdInput.readLine()) != null) {
                System.out.println(s);
            }
            while ((s = stdError.readLine()) != null) {
                System.out.println(s);
            }
        }
        catch (IOException e) {
            LOG.error((Object)("Error running: " + command), (Throwable)e);
        }
    }

    private void shutdownAllProcesses() {
        LOG.info((Object)"Killing daemons using pid files");
        ArrayList<String> pidFiles = new ArrayList<String>(this.daemonPidFiles);
        for (String pidFile : pidFiles) {
            int pid = 0;
            try {
                pid = ProcessBasedLocalHBaseCluster.readPidFromFile(pidFile);
            }
            catch (IOException ex) {
                LOG.error((Object)("Could not kill process with pid from " + pidFile));
            }
            if (pid <= 0) continue;
            LOG.info((Object)("Killing pid " + pid + " (" + pidFile + ")"));
            this.killProcess(pid);
        }
        LOG.info((Object)"Waiting a bit to let processes terminate");
        Threads.sleep((long)5000L);
    }

    private void ensureShutdownHookInstalled() {
        if (this.shutdownHookInstalled) {
            return;
        }
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable(){

            @Override
            public void run() {
                ProcessBasedLocalHBaseCluster.this.shutdownAllProcesses();
            }
        }));
        this.shutdownHookInstalled = true;
    }

    private void cleanupOldState() {
        this.executeCommand("rm -rf " + this.workDir);
    }

    private void writeStringToFile(String s, String fileName) {
        try {
            BufferedWriter out = new BufferedWriter(new FileWriter(fileName));
            out.write(s);
            out.close();
        }
        catch (IOException e) {
            LOG.error((Object)("Error writing to: " + fileName), (Throwable)e);
        }
    }

    private String serverWorkingDir(String serverName, int port) {
        String dir = serverName.equals("regionserver") ? this.workDir + "/" + serverName + "-" + port : this.workDir + "/" + serverName;
        return dir;
    }

    private int getServerPID(String serverName, int port) throws IOException {
        String pidFile = this.pidFilePath(serverName, port);
        return ProcessBasedLocalHBaseCluster.readPidFromFile(pidFile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static int readPidFromFile(String pidFile) throws IOException {
        Scanner scanner = new Scanner(new File(pidFile));
        try {
            int n = scanner.nextInt();
            return n;
        }
        finally {
            scanner.close();
        }
    }

    private String pidFilePath(String serverName, int port) {
        String dir = this.serverWorkingDir(serverName, port);
        String user = System.getenv("USER");
        String pidFile = String.format("%s/hbase-%s-%s.pid", dir, user, serverName);
        return pidFile;
    }

    private void killServer(String serverName, int port) throws IOException {
        int pid = this.getServerPID(serverName, port);
        if (pid > 0) {
            LOG.info((Object)("Killing " + serverName + "; pid=" + pid));
            this.killProcess(pid);
        }
    }

    private void killProcess(int pid) {
        String cmd = "kill -s KILL " + pid;
        this.executeCommand(cmd);
    }

    private void startServer(String serverName, int rsPort) {
        String conf = this.generateConfig(rsPort);
        String dir = this.serverWorkingDir(serverName, rsPort);
        this.executeCommand("mkdir -p " + dir);
        this.writeStringToFile(conf, dir + "/" + "hbase-site.xml");
        HashMap<String, String> envOverrides = new HashMap<String, String>();
        envOverrides.put("HBASE_LOG_DIR", dir);
        envOverrides.put("HBASE_PID_DIR", dir);
        try {
            FileUtils.copyFile((File)new File(this.hbaseHome, "conf/log4j.properties"), (File)new File(dir, "log4j.properties"));
        }
        catch (IOException ex) {
            LOG.error((Object)("Could not install log4j.properties into " + dir));
        }
        this.executeCommand(this.hbaseDaemonScript + " --config " + dir + " start " + serverName, envOverrides);
        this.daemonPidFiles.add(this.pidFilePath(serverName, rsPort));
    }

    private final String generateConfig(int rsPort) {
        StringBuilder sb = new StringBuilder();
        TreeMap<String, Object> confMap = new TreeMap<String, Object>();
        confMap.put("hbase.cluster.distributed", true);
        if (rsPort > 0) {
            confMap.put("hbase.regionserver.port", rsPort);
            confMap.put("hbase.regionserver.info.port.auto", true);
        }
        confMap.put("hbase.zookeeper.property.clientPort", this.zkClientPort);
        confMap.put("hbase.master.port", this.masterPort);
        confMap.put("hbase.hregion.max.filesize", 10000000);
        confMap.put("fs.file.impl", RawLocalFileSystem.class.getName());
        sb.append("<configuration>\n");
        for (Map.Entry entry : confMap.entrySet()) {
            sb.append("  <property>\n");
            sb.append("    <name>" + (String)entry.getKey() + "</name>\n");
            sb.append("    <value>" + entry.getValue() + "</value>\n");
            sb.append("  </property>\n");
        }
        sb.append("</configuration>\n");
        return sb.toString();
    }

    public Configuration getConf() {
        return this.conf;
    }
}

