package net.osdn.util.sql;

import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.sql.ConnectionPoolDataSource;
import javax.sql.RowSet;
import javax.sql.rowset.CachedRowSet;
import javax.sql.rowset.RowSetFactory;
import javax.sql.rowset.RowSetProvider;

import net.osdn.util.concurrent.Monitor;
import net.osdn.util.concurrent.Monitor.MultipleObjects;

/** JDBCデータソースを扱うためのユーティリティークラスです。
 * 
 * <p>このクラスはJDBCを扱い易くするユーティリティーメソッドの集合であり、
 * javax.sql.DataSourceインターフェースを実装していないことに注意してください。</p>
 * 
 * <p>このクラスはインスタンスを生成せずにスタティック・メソッドを呼び出す使い方と、
 * インスタンスを生成してインスタンス・メソッドを呼び出す使い方の両方をサポートしています。
 * アプリケーションが扱うデータソースが1つだけの場合はスタティック・メソッドを使うのが便利です。</p>
 * 
 */
public class DataSource {
	
	public static final int OPTION_MSSQL_SET_NOCOUNT_ON = 1;

	private static Pattern TABLE_NAME = Pattern.compile("(?:FROM|INTO|UPDATE)\\s+(.+?)(?:$|\\s)", Pattern.CASE_INSENSITIVE);
	private static final Monitor monitor = new Monitor();

	private static DataSource.Instance defaultInstance;

	/** 既定のJNDIリソース名 "jdbc/datasource" を検索してデータソースを構成します。
	 * 
	 * <p>JNDIリソース名 "jdbc/datasource" によって取得されるオブジェクトはjavax.sql.DataSourceインターフェース、
	 * javax.sql.ConnectionPoolDataSourceインターフェースのいずれかもしくは両方を実装している必要があります。</p>
	 * 
	 * <p>このメソッドによって見つかったデータソースはスタティック・フィールドに保持され、各スタティック・メソッドでの呼び出し時に使用されます。
	 * 既に他のデータソースが保持されている場合、以前のデータソースは閉じられます。</p>
	 * 
	 * @throws NamingException ネーミング例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static synchronized void initialize() throws NamingException, SQLException {
		initialize("jdbc/datasource");
	}
	
	/** 指定したJNDIリソース名でデータソースを検索して構成します。
	 * 
	 * <p>指定されたJNDIリソース名によって取得されるオブジェクトはjavax.sql.DataSourceインターフェース、
	 * javax.sql.ConnectionPoolDataSourceインターフェースのいずれかもしくは両方を実装している必要があります。</p>
	 * 
	 * <p>このメソッドによって見つかったデータソースはスタティック・フィールドに保持され、各スタティック・メソッドでの呼び出し時に使用されます。
	 * 既に他のデータソースが保持されている場合、以前のデータソースは閉じられます。</p>
	 * 
	 * @param name データソースを取得するためのJNDIリソース名
	 * @throws NamingException ネーミング例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static void initialize(String name) throws NamingException, SQLException {
		InitialContext context = new InitialContext();
		Object datasource = context.lookup(name);
		if(datasource instanceof javax.sql.DataSource) {
			initialize((javax.sql.DataSource)datasource);
		} else if(datasource instanceof javax.sql.ConnectionPoolDataSource) {
			initialize((javax.sql.ConnectionPoolDataSource)datasource);
		} else {
			throw new IllegalArgumentException();
		}
	}
	
	/** 指定したデータソースでユーティリティを構成します。
	 * 
	 * <p>指定されたデータソースはスタティック・フィールドに保持され、各スタティック・メソッドでの呼び出し時に使用されます。
	 * 既に他のデータソースが保持されている場合、以前のデータソースは閉じられます。</p>
	 * 
	 * @param datasource このユーティリティで使用するデータソース
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static void initialize(javax.sql.DataSource datasource) throws SQLException {
		if(defaultInstance != null) {
			defaultInstance.close();
		}
		defaultInstance = new DataSource.Instance(datasource);
	}
	
	/** 接続プールを提供可能なデータソースを指定してユーティリティを構成します。
	 * 
	 * <p>指定されたデータソースはスタティック・フィールドに保持され、各スタティック・メソッドでの呼び出し時に使用されます。
	 * 既に他のデータソースが保持されている場合、以前のデータソースは閉じられます。</p>
	 * 
	 * @param datasource このユーティリティで使用するデータソース
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static void initialize(javax.sql.ConnectionPoolDataSource datasource) throws SQLException {
		if(defaultInstance != null) {
			defaultInstance.close();
		}
		defaultInstance = new DataSource.Instance(datasource);
	}
	
	/* package private */ static Monitor getMonitor() {
		return monitor;
	}
	
	/** 新しいデータベースへの接続を作成してトランザクションを開始します。
	 * トランザクション分離レベルは READ COMMITTED に設定されます。
	 * 
	 * @return トランザクション
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static Transaction beginTransaction() throws SQLException {
		return defaultInstance.beginTransaction();
	}
	
	/** 新しいデータベースへの接続を作成して指定された分離レベルのトランザクションを開始します。
	 * 
	 * @param level トランザクション分離レベル
	 * @return トランザクション
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static Transaction beginTransaction(IsolationLevel level) throws SQLException {
		return defaultInstance.beginTransaction(level);
	}
	
	/** データソースへの接続を取得します。
	 * 
	 * <p>データソースがjavax.sql.ConnectionPoolDataSourceインターフェースを実装している場合、
	 * このメソッドはプールされている接続を返します。</p>
	 * 
	 * @return データソースへの接続
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static Connection getConnection() throws SQLException {
		return defaultInstance.getConnection(0);
	}
	
	public static Connection getConnection(int options) throws SQLException {
		return defaultInstance.getConnection(options);
	}
	
	/** 指定したResultSetオブジェクトのデータを読み込んでCachedRowSetを作成します。
	 * 
	 * <p>CachedRowSet オブジェクトは、メモリー内に行をキャッシュする、データ行のコンテナであり、常にそのデータソースに接続していなくても処理を実行できます。</p>
	 * 
	 * @param rs ResultSetオブジェクト
	 * @return ResultSetオブジェクトのデータを読み込んだCachedRowSet
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public static CachedRowSet populate(ResultSet rs) throws SQLException {
		return defaultInstance.populate(rs);
	}
	
	/** 単一のResultSetオブジェクトを返す、指定されたSQL文を実行します。
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static ResultSet executeQuery(String sql) throws SQLException, IOException {
		return defaultInstance.executeQuery(sql);
	}
	
	/** 単一のResultSetオブジェクトを返す、指定されたSQL文にパラメーターを設定して実行します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。</p>
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static ResultSet executeQuery(String sql, Object... parameters) throws SQLException, IOException {
		return defaultInstance.executeQuery(sql, parameters);
	}
	
	/** 単一のResultSetオブジェクトを返す、指定されたSQL文にパラメーターを設定して実行します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。</p>
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static ResultSet executeQuery(String sql, Collection<?> parameters) throws SQLException, IOException {
		return defaultInstance.executeQuery(sql, parameters);
	}
	
	/** 指定された名前付きパラメーターステートメントを実行し、単一のResultSetオブジェクトを返します。
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public static RowSet executeQuery(NamedParameterStatement statement) throws SQLException, IOException {
		return defaultInstance.executeQuery(statement);
	}
	
	/** SQL文を実行します。
	 * 
	 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters　プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static int executeUpdate(String sql, Object... parameters) throws SQLException, IOException {
		return defaultInstance.executeUpdate(sql, parameters);
	}
	
	/** SQL文を実行します。
	 * 
	 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters　プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static int executeUpdate(String sql, Collection<?> parameters) throws SQLException, IOException {
		return defaultInstance.executeUpdate(sql, parameters);
	}
	
	/** SQL文を実行します。
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
	 * 
	 * @param statement 名前付きパラメーターステートメント
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static int executeUpdate(NamedParameterStatement statement) throws SQLException, IOException {
		return defaultInstance.executeUpdate(statement);
	}
	
	/** SQL文を実行して待機スレッドを再開させます。
	 * 
	 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
	 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
	 * 
	 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static int executeUpdate(Object notifier, String sql, Object... parameters) throws SQLException, IOException {
		return defaultInstance.executeUpdate(notifier, sql, parameters);
	}
	
	/** SQL文を実行して待機スレッドを再開させます。
	 * 
	 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
	 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
	 * 
	 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static int executeUpdate(Object notifier, String sql, Collection<?> parameters) throws SQLException, IOException {
		return defaultInstance.executeUpdate(notifier, sql, parameters);
	}
	
	/** SQL文を実行して待機スレッドを再開させます。
	 * 
	 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
	 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
	 * 
	 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
	 * @param statement 結果セットを返さないSQL文
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public static int executeUpdate(Object notifier, NamedParameterStatement statement) throws SQLException, IOException {
		return defaultInstance.executeUpdate(notifier, statement);
	}

	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
	 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
	 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(String)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static RowSet executeQueryWaitForRows(long timeout, String sql, Object... parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
		return defaultInstance.executeQueryWaitForRows(timeout, sql, parameters);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
	 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
	 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(String)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static RowSet executeQueryWaitForRows(long timeout, String sql, Collection<?> parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
		return defaultInstance.executeQueryWaitForRows(timeout, sql, parameters);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
	 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
	 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public static RowSet executeQueryWaitForRows(long timeout, NamedParameterStatement statement) throws SQLException, TimeoutException, InterruptedException, IOException {
		return defaultInstance.executeQueryWaitForRows(timeout, statement);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
	 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static RowSet executeQueryWaitForRows(MultipleObjects waiters, String sql, Object... parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
		return defaultInstance.executeQueryWaitForRows(waiters, sql, parameters);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
	 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static RowSet executeQueryWaitForRows(MultipleObjects waiters, String sql, Collection<?> parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
		return defaultInstance.executeQueryWaitForRows(waiters, sql, parameters);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
	 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public static RowSet executeQueryWaitForRows(MultipleObjects waiters, NamedParameterStatement statement) throws SQLException, TimeoutException, InterruptedException, IOException {
		return defaultInstance.executeQueryWaitForRows(waiters, statement);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static <T> T executeScalar(String sql) throws SQLException, IOException {
		return defaultInstance.executeScalar(sql);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static <T> T executeScalar(String sql, Object... parameters) throws SQLException, IOException {
		return defaultInstance.executeScalar(sql, parameters);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 */
	public static <T> T executeScalar(String sql, Collection<?> parameters) throws SQLException, IOException {
		return defaultInstance.executeScalar(sql, parameters);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public static <T> T executeScalar(NamedParameterStatement statement) throws SQLException, IOException {
		return defaultInstance.executeScalar(statement);
	}

	/** 指定した結果セットおよび、ステートメント、コネクションを閉じます。
	 * 
	 * @param rs 結果セット
	 * @throws SQLException SQLException
	 */
	public static void close(ResultSet rs) throws SQLException {
		if(rs instanceof CachedRowSet) {
			return;
		}
		
		Statement st = null;
		if(rs != null) {
			st = rs.getStatement();
		}
		Connection cn = null;
		if(st != null) {
			cn = st.getConnection();
		}
		if(rs != null && !rs.isClosed()) {
			rs.close();
		}
		if(st != null && !st.isClosed()) {
			st.close();
		}
		if(cn != null && !cn.isClosed()) {
			cn.close();
		}
	}

	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行から指定された列名の値を返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>id列とname列を持つPersonテーブルに対してSELECT文を実行するには以下の方法があります。</p>
	 * 
	 * <pre><code>
	 * String name = DataSource.select("name", new Person() {
	 *     int id = 1;
	 * });
	 * </code></pre>
	 * 
	 * または
	 * 
	 * <pre><code>
	 * String name = DataSource.select("name", new Table("Person") {
	 *     int id = 1;
	 * });
	 * </code></pre>
	 * 
	 * <p>ただし、以下のようにPersonクラスのインスタンスを初期化した場合は上記と同じ結果になりません。</p>
	 * 
	 * <pre><code>
	 * Person person = new Person();
	 * person.id = 1;
	 * String name = DataSource.select("name", person);
	 * </code></pre>
	 * 
	 * <p>上記のケースではPersonクラスが持つ他のフィールドnameがWHERE句に追加されてしまうためです。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSet#getObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param columnName 列名
	 * @param obj SELECT文を構築するオブジェクト
	 * @return クエリーが返す結果セットの最初の行にある指定された列名の値。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalArgumentException 指定されたオブジェクトが null の場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static <T> T select(String columnName, Object obj) throws SQLException, IOException, IllegalArgumentException, IllegalAccessException {
		return defaultInstance.select(columnName, obj);
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行をオブジェクトとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOオブジェクトです。
	 * オブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>id列とname列を持つPersonテーブルに対してSELECT文を実行するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * Person person = DataSource.select(new Person() {
	 *     int id = 1;
	 * });
	 * </code></pre>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param obj SELECT文を構築するオブジェクト
	 * @return クエリーが返す結果セットの最初の行をオブジェクトのクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(obj);
	}

	/** 指定したクラスでSELECT文を構築し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * <p>このメソッドはレコードが1行しかないテーブルのデータを取得するのに適しています。</p>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Class<T> returnClass) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(returnClass);
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param obj SELECT文を構築するオブジェクト
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Class<T> returnClass, Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(returnClass, obj);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Class<T> returnClass, String sql) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(returnClass, sql);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Class<T> returnClass, String sql, Object... parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(returnClass, sql, parameters);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Class<T> returnClass, String sql, Collection<?> parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(returnClass, sql, parameters);
	}
	
	/** 指定された名前付きパラメーターステートメントを実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param statement 名前付きパラメーターステートメント
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> T select(Class<T> returnClass, NamedParameterStatement statement) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.select(returnClass, statement);
	}
	
	/** 指定されたクラスでSELECT文を構築し、そのクエリーが返す結果セットをクラスのリストとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するクラスのクラス名をテーブル名とします。
	 * このメソッドに匿名クラスを渡すことはできません。
	 * このメソッドはテーブルのすべての行を返します。</p>
	 * 
	 * <p>Personテーブルのすべての行をList&lt;Person&gt;として取得するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * List&lt;Person&gt; persons = DataSource.selectAsList(Person.class);
	 * </code></pre>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス。クラス名がテーブル名と一致している必要があります。
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Class<T> returnClass) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(returnClass);
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットをオブジェクトのクラスのリストとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOオブジェクトです。
	 * オブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>Personテーブルのdeleted列の値が1の行をList&lt;Person&gt;として取得するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * List&lt;Person&gt; persons = DataSource.selectAsList(new Person() {
	 *     int deleted = 1;
	 * });
	 * </code></pre>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param obj SELECT文を構築するオブジェクト。このオブジェクトのクラス名がテーブル名と一致している必要があります。
	 * @return クエリーが返す結果セットの各行を指定されたオブジェクトのクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(obj);
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>Personテーブルのdeleted列の値が1の行をList&lt;Person&gt;として取得するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * List&lt;Person&gt; persons = DataSource.selectAsList(Person.class, new Table("Person") {
	 *     int deleted = 1;
	 * });
	 * </code></pre>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param obj SELECT文を構築するオブジェクト。
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Class<T> returnClass, Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(returnClass, obj);
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Class<T> returnClass, String sql) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(returnClass, sql);
	}
	
	/** 指定されたSQL文にパラメーターを設定して実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Class<T> returnClass, String sql, Object... parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(returnClass, sql, parameters);
	}
	
	/** 指定されたSQL文にパラメーターを設定して実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Class<T> returnClass, String sql, Collection<?> parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(returnClass, sql, parameters);
	}
	
	/** 指定された名前付きパラメーターステートメントを実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param statement 名前付きパラメーターステートメント
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public static <T> List<T> selectAsList(Class<T> returnClass, NamedParameterStatement statement) throws SQLException, ParseException, IOException, ReflectiveOperationException {
		return defaultInstance.selectAsList(returnClass, statement);
	}
	
	/** 指定されたオブジェクトでINSERT文を構築し、データソースへ行の追加を試みます。
	 * 
	 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
	 * 
	 * <p>列 id, name, age を持つPersonテーブルに行を追加するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * DataSource.insert(new Person() {
	 *     int    id   = 4;
	 *     String name = "四郎";
	 *     int    age  = 19; 
	 * });
	 * </code></pre>
	 * 
	 * または
	 * 
	 * <pre><code>
	 * DataSource.insert(new Table("Person") {
	 *     int    id   = 4;
	 *     String name = "四郎";
	 *     int    age  = 19; 
	 * });
	 * </code></pre>
	 * 
	 * または
	 * 
	 * <pre><code>
	 * Person person = new Person();
	 * person.id   = 4;
	 * person.name = "四郎";
	 * person.age  = 19;
	 * DataSource.insert(person);
	 * </code></pre>
	 * 
	 * <p>3つ目のコード例ではPersonクラスが列名と一致する他のフィールドを持っている場合に、
	 * 列名に 0 や null が指定されることに注意してください。</p>
	 * 
	 * @param obj INSERT文を構築するオブジェクト
	 * @return INSERT文の実行により影響を受けた行数。つまり行の追加が成功した場合は 1 です。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int insert(Object obj) throws SQLException, IOException, IllegalAccessException {
		return defaultInstance.insert(obj);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してINSERT文を構築し、データソースへ行の追加を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけINSERT文が実行されますが、これらのINSERT文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
	 * 
	 * @param objects INSERT文を構築するオブジェクト(複数)
	 * @return INSERT文の実行により影響を受けた行数。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] insert(Object... objects) throws SQLException, IOException, IllegalAccessException {
		return defaultInstance.insert(objects);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してINSERT文を構築し、データソースへ行の追加を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけINSERT文が実行されますが、これらのINSERT文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
	 * 
	 * @param objects INSERT文を構築するオブジェクト(複数)
	 * @return INSERT文の実行により影響を受けた行数。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] insert(Collection<?> objects) throws SQLException, IOException, IllegalAccessException {
		return defaultInstance.insert(objects.toArray());
	}
	
	/** 指定されたオブジェクトでUPDATE文を構築し、データソースへ行の更新を試みます。
	 * 
	 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
	 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>主キー列 id と通常の列 name を持つPersonテーブルを更新するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * DataSource.update(new Person() {
	 *     int    id   = 4;
	 *     String name = "史郎";
	 * });
	 * </code></pre>
	 * 
	 * <p>上記コード例では主キー列 id の値が 4 となっている行の name 列が "史郎" に更新されます。</p>
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param obj UPDATE文を構築するオブジェクト
	 * @return UPDATE文の実行により影響を受けた行数。つまり行の更新が成功した場合は 1 です。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int update(Object obj) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.update(obj);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してUPDATE文を構築し、データソースへ行の更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけUPDATE文が実行されますが、これらのUPDATE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
	 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects UPDATE文を構築するオブジェクト(複数)
	 * @return UPDATE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] update(Object... objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.update(objects);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してUPDATE文を構築し、データソースへ行の更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけUPDATE文が実行されますが、これらのUPDATE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
	 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects UPDATE文を構築するオブジェクト(複数)
	 * @return UPDATE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] update(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.update(objects);
	}
	
	/** 指定されたオブジェクトでDELETE文を構築し、データソースから行の削除を試みます。
	 * 
	 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param obj DELETE文を構築するオブジェクト
	 * @return DELETE文の実行により影響を受けた行数。つまり行の削除が成功した場合は 1 です。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int delete(Object obj) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.delete(obj);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してDELETE文を構築し、データソースから行の削除を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけDELETE文が実行されますが、これらのDELETE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects DELETE文を構築するオブジェクト(複数)
	 * @return DELETE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] delete(Object... objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.delete(objects);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してDELETE文を構築し、データソースから行の削除を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけDELETE文が実行されますが、これらのDELETE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects DELETE文を構築するオブジェクト(複数)
	 * @return DELETE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] delete(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.delete(objects);
	}
	
	/** 指定されたオブジェクトでMERGE文を構築し、データソースへ行の追加または更新を試みます。
	 * 
	 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を更新または追加することを目的としています。
	 * 行を一意に特定するためのフィールドが指定されている必要があります。
	 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>主キー列 id と通常の列 name を持つPersonテーブルを追加または更新するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * DataSource.merge(new Person() {
	 *     int    id   = 5;
	 *     String name = "五郎";
	 * });
	 * </code></pre>
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param obj MERGE文を構築するオブジェクト
	 * @return MERGE文の実行により影響を受けた行数。つまり行の追加または更新が成功した場合は 1 です。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int merge(Object obj) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.merge(obj);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してMERGE文を構築し、データソースへ行の追加または更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけMERGE文が実行されますが、これらのMERGE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を更新または追加することを目的としています。
	 * 行を一意に特定するためのフィールドが指定されている必要があります。
	 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects MERGE文を構築するオブジェクト(複数)
	 * @return MERGE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] merge(Object... objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.merge(objects);
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してMERGE文を構築し、データソースへ行の追加または更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけMERGE文が実行されますが、これらのMERGE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を更新または追加することを目的としています。
	 * 行を一意に特定するためのフィールドが指定されている必要があります。
	 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects MERGE文を構築するオブジェクト(複数)
	 * @return MERGE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] merge(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
		return defaultInstance.merge(objects);
	}
	
	/** 待機時間を指定して複数のオブジェクトを対象としてスレッドを待機・再開させるための複合オブジェクトを作成します。
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param objects スレッドを待機・再開させるための対象オブジェクト
	 * @return 指定されたオブジェクトをまとめたMultipleObjects
	 * 
	 * @see MultipleObjects
	 */
	public static MultipleObjects newMultipleObjects(long timeout, Object... objects) {
		return monitor.newMultipleObjects(timeout, objects);
	}
	
	/** 待機時間を指定して複数のオブジェクトを対象としてスレッドを待機・再開させるための複合オブジェクトを作成します。
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param objects スレッドを待機・再開させるための対象オブジェクト
	 * @return 指定されたオブジェクトをまとめたMultipleObjects
	 * 
	 * @see MultipleObjects
	 */
	public static MultipleObjects newMultipleObjects(long timeout, Collection<?> objects) {
		return monitor.newMultipleObjects(timeout, objects);
	}
	
	/** 指定したオブジェクトをまとめているMultipleObjectsのwaitForMultipleObjects()メソッドで待機しているすべてのスレッドを再開させます。
	 * 
	 * @param object MultipleObjectsに含まれているいずれかのオブジェクト
	 */
	public static void notifyAll(Object object) {
		monitor.notifyAll(object);
	}
	
	/* package private */ static String findTableName(String sql) {
		String tableName = null;
		Matcher matcher = TABLE_NAME.matcher(sql);
		if(matcher.find()) {
			tableName = matcher.group(1);
			int pos = tableName.lastIndexOf('.');
			if(0 <= pos && pos < tableName.length() - 1) {
				tableName = tableName.substring(pos + 1);
			}
			if(tableName.length() >= 2 && tableName.startsWith("\"") && tableName.endsWith("\"")) {
				tableName = tableName.substring(1, tableName.length() - 1);
			}
			if(tableName.length() >= 2 && tableName.startsWith("[") && tableName.endsWith("]")) {
				tableName = tableName.substring(1, tableName.length() - 1);
			}
			tableName = tableName.toUpperCase();
		}
		return tableName;
	}
	
	/** データソース・ユーティリティーの実装クラスです。
	 * 
	 * <p>複数のデータソースを扱う場合は、明示的にDataSource.Instanceのインスタンスを作成する必要があります。</p>
	 * 
	 */
	public static class Instance {
		private ConnectionPool pool;
		private javax.sql.DataSource datasource;
		private volatile RowSetFactory rowSetFactory;
		
		/** {@link javax.sql.DataSource}インターフェースを実装したデータソースを持つDataSource.Instanceを作成します。
		 * 
		 * @param datasource {@link javax.sql.DataSource}インターフェースを実装しているデータソース
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 */
		public Instance(javax.sql.DataSource datasource) throws SQLException {
			if(datasource instanceof ConnectionPoolDataSource) {
				this.pool = new ConnectionPool((ConnectionPoolDataSource)datasource);
			} else {
				this.datasource = datasource;
			}
		}
		
		/** {@link javax.sql.ConnectionPoolDataSource}インターフェースを実装したデータソースを持つDataSource.Instanceを作成します。
		 * 
		 * @param datasource {@link javax.sql.ConnectionPoolDataSource}インターフェースを実装しているデータソース
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 */
		public Instance(javax.sql.ConnectionPoolDataSource datasource) throws SQLException {
			this.pool = new ConnectionPool(datasource);
		}
		
		/** 新しいデータベースへの接続を作成してトランザクションを開始します。
		 * トランザクション分離レベルは READ COMMITTED に設定されます。
		 * 
		 * @return トランザクション
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 */
		public Transaction beginTransaction() throws SQLException {
			Transaction transaction = new Transaction(this, IsolationLevel.ReadCommitted);
			return transaction;
		}

		/** 新しいデータベースへの接続を作成して指定された分離レベルのトランザクションを開始します。
		 * 
		 * @param level トランザクション分離レベル
		 * @return トランザクション
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 */
		public Transaction beginTransaction(IsolationLevel level) throws SQLException {
			Transaction transaction = new Transaction(this, level);
			return transaction;
		}
		
		/** データソースへの接続を取得します。
		 * 
		 * <p>データソースがjavax.sql.ConnectionPoolDataSourceインターフェースを実装している場合、
		 * このメソッドはプールされている接続を返します。</p>
		 * 
		 * @return データソースへの接続
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 */
		public Connection getConnection() throws SQLException {
			return getConnection(0);
		}
		
		public Connection getConnection(int options) throws SQLException {
			Connection cn = (pool != null) ? pool.getConnection() : datasource.getConnection();
			
			if((options & OPTION_MSSQL_SET_NOCOUNT_ON) != 0) {
				Statement st = cn.createStatement();
				st.execute("SET NOCOUNT ON");
				st.close();
			}
			
			return cn;
		}
		
		/** データソースがjavax.sql.ConnectionPoolDataSrouceを実装している場合、このメソッドは接続プールを閉じます。
		 * 
		 * <p>接続プールが閉じられると、プールされているすべての接続も閉じられます。
		 * ただし、getConnection()メソッドによって払い出されているアクティブな接続が強制的に閉じられることはありません。
		 * このメソッドの呼び出し以降、getConnection()メソッドによって払い出されているアクティブな接続のclose()メソッドを呼び出すと、
		 * 接続はプールに戻されるのではなく、データソースとの物理的な接続が閉じられます。</p>
		 */
		public void close() {
			if(pool != null) {
				pool.close();
			}
		}
		
		/** 指定したResultSetオブジェクトのデータを読み込んでCachedRowSetを作成します。
		 * 
		 * <p>CachedRowSet オブジェクトは、メモリー内に行をキャッシュする、データ行のコンテナであり、常にそのデータソースに接続していなくても処理を実行できます。</p>
		 * 
		 * @param rs ResultSetオブジェクト
		 * @return ResultSetオブジェクトのデータを読み込んだCachedRowSet
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 */
		public CachedRowSet populate(ResultSet rs) throws SQLException {
			if(rs == null) {
				return null;
			}
			if(rowSetFactory == null) {
				synchronized (this) {
					if(rowSetFactory == null) {
						rowSetFactory = RowSetProvider.newFactory();
					}
				}
			}
			CachedRowSet rowSet = rowSetFactory.createCachedRowSet();
			rowSet.populate(rs);
			return rowSet;
		}
		
		/** 指定されたSQL文を実行し、単一のRowSetオブジェクトを返す、します。
		 * 
		 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
		 * 
		 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
		 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public ResultSet executeQuery(String sql) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQuery(sql);
			}
		}
		
		/** 指定されたSQL文にパラメーターを設定して実行し、単一のRowSetオブジェクトを返します。
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。</p>
		 * 
		 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
		 * 
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public RowSet executeQuery(String sql, Object... parameters) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQuery(sql, parameters);
			}
		}
		
		/** 指定されたSQL文にパラメーターを設定して実行し、単一のRowSetオブジェクトを返します。
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。</p>
		 * 
		 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
		 * 
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public RowSet executeQuery(String sql, Collection<?> parameters) throws SQLException, IOException {
			return executeQuery(sql, parameters.toArray());
		}
		
		/** 指定された名前付きパラメーターステートメントを実行し、単一のRowSetオブジェクトを返します。
		 * 
		 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
		 * 
		 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
		 * 
		 * @param statement 名前付きパラメーターステートメント
		 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * 
		 * @see NamedParameterStatement
		 */
		public RowSet executeQuery(NamedParameterStatement statement) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQuery(statement);
			}
		}
		
		/** SQL文を実行します。
		 * 
		 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param sql 結果セットを返さないSQL文
		 * @param parameters　プレースホルダーに設定されるパラメーター
		 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public int executeUpdate(String sql, Object... parameters) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeUpdate(sql, parameters);
			}
		}
		
		/** SQL文を実行します。
		 * 
		 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param sql 結果セットを返さないSQL文
		 * @param parameters　プレースホルダーに設定されるパラメーター
		 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public int executeUpdate(String sql, Collection<?> parameters) throws SQLException, IOException {
			return executeUpdate(sql, parameters.toArray());
		}
		
		/** SQL文を実行します。
		 * 
		 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
		 * 
		 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
		 * 
		 * @param statement 名前付きパラメーターステートメント
		 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public int executeUpdate(NamedParameterStatement statement) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeUpdate(statement);
			}
		}
		
		/** SQL文を実行して待機スレッドを再開させます。
		 * 
		 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
		 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
		 * 
		 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
		 * @param sql 結果セットを返さないSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public int executeUpdate(Object notifier, String sql, Object... parameters) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeUpdate(notifier, sql, parameters);
			}
		}
		
		/** SQL文を実行して待機スレッドを再開させます。
		 * 
		 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
		 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
		 * 
		 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
		 * @param sql 結果セットを返さないSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public int executeUpdate(Object notifier, String sql, Collection<?> parameters) throws SQLException, IOException {
			return executeUpdate(notifier, sql, parameters.toArray());
		}
		
		/** SQL文を実行して待機スレッドを再開させます。
		 * 
		 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
		 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
		 * 
		 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
		 * 
		 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
		 * 
		 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
		 * @param statement 結果セットを返さないSQL文
		 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * 
		 * @see NamedParameterStatement
		 */
		public int executeUpdate(Object notifier, NamedParameterStatement statement) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeUpdate(notifier, statement);
			}
		}

		/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
		 * 
		 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
		 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
		 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(String)メソッドを呼び出すと、
		 * スレッドが再開され、再び、SQL文が実行されます。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param timeout ミリ秒単位の待機時間
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws TimeoutException 指定された待機時間が経過した場合
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public RowSet executeQueryWaitForRows(long timeout, String sql, Object... parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQueryWaitForRows(timeout, sql, parameters);
			}
		}
		
		/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
		 * 
		 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
		 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
		 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(String)メソッドを呼び出すと、
		 * スレッドが再開され、再び、SQL文が実行されます。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param timeout ミリ秒単位の待機時間
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws TimeoutException 指定された待機時間が経過した場合
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public RowSet executeQueryWaitForRows(long timeout, String sql, Collection<?> parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
			return executeQueryWaitForRows(timeout, sql, parameters.toArray());
		}
		
		/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
		 * 
		 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
		 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
		 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
		 * スレッドが再開され、再び、SQL文が実行されます。</p>
		 * 
		 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
		 * 
		 * @param timeout ミリ秒単位の待機時間
		 * @param statement 名前付きパラメーターステートメント
		 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws TimeoutException 指定された待機時間が経過した場合
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * 
		 * @see NamedParameterStatement
		 */
		public RowSet executeQueryWaitForRows(long timeout, NamedParameterStatement statement) throws SQLException, TimeoutException, InterruptedException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQueryWaitForRows(timeout, statement);
			}
		}
		
		/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
		 * 
		 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
		 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
		 * スレッドが再開され、再び、SQL文が実行されます。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws TimeoutException 指定された待機時間が経過した場合
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public RowSet executeQueryWaitForRows(MultipleObjects waiters, String sql, Object... parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQueryWaitForRows(waiters, sql, parameters);
			}
		}
		
		/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
		 * 
		 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
		 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
		 * スレッドが再開され、再び、SQL文が実行されます。</p>
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws TimeoutException 指定された待機時間が経過した場合
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public RowSet executeQueryWaitForRows(MultipleObjects waiters, String sql, Collection<?> parameters) throws SQLException, TimeoutException, InterruptedException, IOException {
			return executeQueryWaitForRows(waiters, sql, parameters.toArray());
		}
		
		/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
		 * 
		 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
		 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
		 * スレッドが再開され、再び、SQL文が実行されます。</p>
		 * 
		 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
		 * 
		 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
		 * @param statement 名前付きパラメーターステートメント
		 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws TimeoutException 指定された待機時間が経過した場合
		 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * 
		 * @see NamedParameterStatement
		 */
		public RowSet executeQueryWaitForRows(MultipleObjects waiters, NamedParameterStatement statement) throws SQLException, TimeoutException, InterruptedException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeQueryWaitForRows(waiters, statement);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
		 * 
		 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
		 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
		 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public <T> T executeScalar(String sql) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeScalar(sql);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public <T> T executeScalar(String sql, Object... parameters) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeScalar(sql, parameters);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
		 * 
		 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
		 * パラメーターのプレースホルダーは {@literal ?} で指定します。
		 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
		 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
		 * 
		 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 */
		public <T> T executeScalar(String sql, Collection<?> parameters) throws SQLException, IOException {
			return executeScalar(sql, parameters.toArray());
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
		 * 
		 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
		 * 
		 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
		 * @param statement 名前付きパラメーターステートメント
		 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * 
		 * @see NamedParameterStatement
		 */
		public <T> T executeScalar(NamedParameterStatement statement) throws SQLException, IOException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.executeScalar(statement);
			}
		}
		
		/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行から指定された列名の値を返します。
		 * 
		 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
		 * 
		 * <p>id列とname列を持つPersonテーブルに対してSELECT文を実行するには以下の方法があります。</p>
		 * 
		 * <pre><code>
		 * String name = DataSource.select("name", new Person() {
		 *     int id = 1;
		 * });
		 * </code></pre>
		 * 
		 * または
		 * 
		 * <pre><code>
		 * String name = DataSource.select("name", new Table("Person") {
		 *     int id = 1;
		 * });
		 * </code></pre>
		 * 
		 * <p>ただし、以下のようにPersonクラスのインスタンスを初期化した場合は上記と同じ結果になりません。</p>
		 * 
		 * <pre><code>
		 * Person person = new Person();
		 * person.id = 1;
		 * String name = DataSource.select("name", person);
		 * </code></pre>
		 * 
		 * <p>上記のケースではPersonクラスが持つ他のフィールドnameがWHERE句に追加されてしまうためです。</p>
		 * 
		 * @param <T> 返される列の値をキャストする型。列の値はResultSet#getObject(int)によって取得されて{@literal <T>}にキャストされます。
		 * @param columnName 列名
		 * @param obj SELECT文を構築するオブジェクト
		 * @return クエリーが返す結果セットの最初の行にある指定された列名の値。行がない場合は null。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalArgumentException 指定されたオブジェクトが null の場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public <T> T select(String columnName, Object obj) throws SQLException, IOException, IllegalArgumentException, IllegalAccessException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(columnName, obj);
			}
		}
		
		/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行をオブジェクトとして返します。
		 * 
		 * <p>SELECT文を構築するために指定するオブジェクトはPOJOオブジェクトです。
		 * オブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
		 * 
		 * <p>id列とname列を持つPersonテーブルに対してSELECT文を実行するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * Person person = DataSource.select(new Person() {
		 *     int id = 1;
		 * });
		 * </code></pre>
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param obj SELECT文を構築するオブジェクト
		 * @return クエリーが返す結果セットの最初の行をオブジェクトのクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(obj);
			}
		}
		
		/** 指定したクラスでSELECT文を構築し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
		 * 
		 * <p>このメソッドはレコードが1行しかないテーブルのデータを取得するのに適しています。</p>
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス
		 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Class<T> returnClass) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(returnClass);
			}
		}
		
		/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
		 * 
		 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス
		 * @param obj SELECT文を構築するオブジェクト
		 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Class<T> returnClass, Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(returnClass, obj);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス
		 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
		 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Class<T> returnClass, String sql) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(returnClass, sql);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Class<T> returnClass, String sql, Object... parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(returnClass, sql, parameters);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Class<T> returnClass, String sql, Collection<?> parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			return select(returnClass, sql, parameters.toArray());
		}
		
		/** 指定された名前付きパラメーターステートメントを実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
		 * 
		 * @param <T> 返されるオブジェクトの型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス
		 * @param statement 名前付きパラメーターステートメント
		 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> T select(Class<T> returnClass, NamedParameterStatement statement) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.select(returnClass, statement);
			}
		}

		/** 指定されたクラスでSELECT文を構築し、そのクエリーが返す結果セットをクラスのリストとして返します。
		 * 
		 * <p>SELECT文を構築するために指定するクラスのクラス名をテーブル名とします。
		 * このメソッドに匿名クラスを渡すことはできません。
		 * このメソッドはテーブルのすべての行を返します。</p>
		 * 
		 * <p>Personテーブルのすべての行をList&lt;Person&gt;として取得するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * List&lt;Person&gt; persons = DataSource.selectAsList(Person.class);
		 * </code></pre>
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param returnClass クエリーが返す結果セットをマッピングするクラス。クラス名がテーブル名と一致している必要があります。
		 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Class<T> returnClass) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.selectAsList(returnClass, (Object)null);
			}
		}
		
		/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットをオブジェクトのクラスのリストとして返します。
		 * 
		 * <p>SELECT文を構築するために指定するオブジェクトはPOJOオブジェクトです。
		 * オブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
		 * 
		 * <p>Personテーブルのdeleted列の値が1の行をList&lt;Person&gt;として取得するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * List&lt;Person&gt; persons = DataSource.selectAsList(new Person() {
		 *     int deleted = 1;
		 * });
		 * </code></pre>
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param obj SELECT文を構築するオブジェクト。このオブジェクトのクラス名がテーブル名と一致している必要があります。
		 * @return クエリーが返す結果セットの各行を指定されたオブジェクトのクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.selectAsList(obj);
			}
		}
		
		/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
		 * 
		 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
		 * 
		 * <p>Personテーブルのdeleted列の値が1の行をList&lt;Person&gt;として取得するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * List&lt;Person&gt; persons = DataSource.selectAsList(Person.class, new Table("Person") {
		 *     int deleted = 1;
		 * });
		 * </code></pre>
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param returnClass 返されるリストの要素の型
		 * @param obj SELECT文を構築するオブジェクト。
		 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Class<T> returnClass, Object obj) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.selectAsList(returnClass, obj);
			}
		}
		
		/** 指定されたSQL文を実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param returnClass 返されるリストの要素の型
		 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
		 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Class<T> returnClass, String sql) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.selectAsList(returnClass, sql);
			}
		}
		
		/** 指定されたSQL文にパラメーターを設定して実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param returnClass 返されるリストの要素の型
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Class<T> returnClass, String sql, Object... parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.selectAsList(returnClass, sql, parameters);
			}
		}
		
		/** 指定されたSQL文にパラメーターを設定して実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param returnClass 返されるリストの要素の型
		 * @param sql プレースホルダー {@literal ?} を含むSQL文
		 * @param parameters プレースホルダーに設定されるパラメーター
		 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Class<T> returnClass, String sql, Collection<?> parameters) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			return selectAsList(returnClass, sql, parameters.toArray());
		}
		
		/** 指定された名前付きパラメーターステートメントを実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
		 * 
		 * @param <T> 返されるリストの要素の型。
		 * @param returnClass 返されるリストの要素の型
		 * @param statement 名前付きパラメーターステートメント
		 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws ParseException 列の値をフィールドの型に変換できない場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
		 */
		public <T> List<T> selectAsList(Class<T> returnClass, NamedParameterStatement statement) throws SQLException, ParseException, IOException, ReflectiveOperationException {
			try(Transaction transaction = beginTransaction(null)) {
				return transaction.selectAsList(returnClass, statement);
			}
		}
		
		/** 指定されたオブジェクトでINSERT文を構築し、データソースへ行の追加を試みます。
		 * 
		 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
		 * 
		 * <p>列 id, name, age を持つPersonテーブルに行を追加するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * DataSource.insert(new Person() {
		 *     int    id   = 4;
		 *     String name = "四郎";
		 *     int    age  = 19; 
		 * });
		 * </code></pre>
		 * 
		 * または
		 * 
		 * <pre><code>
		 * DataSource.insert(new Table("Person") {
		 *     int    id   = 4;
		 *     String name = "四郎";
		 *     int    age  = 19; 
		 * });
		 * </code></pre>
		 * 
		 * または
		 * 
		 * <pre><code>
		 * Person person = new Person();
		 * person.id   = 4;
		 * person.name = "四郎";
		 * person.age  = 19;
		 * DataSource.insert(person);
		 * </code></pre>
		 * 
		 * <p>3つ目のコード例ではPersonクラスが列名と一致する他のフィールドを持っている場合に、
		 * 列名に 0 や null が指定されることに注意してください。</p>
		 * 
		 * @param obj INSERT文を構築するオブジェクト
		 * @return INSERT文の実行により影響を受けた行数。つまり行の追加が成功した場合は 1 です。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int insert(Object obj) throws SQLException, IOException, IllegalAccessException {
			int[] r = insert(new Object[] { obj });
			return r[0];
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してINSERT文を構築し、データソースへ行の追加を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけINSERT文が実行されますが、これらのINSERT文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
		 * 
		 * @param objects INSERT文を構築するオブジェクト(複数)
		 * @return INSERT文の実行により影響を受けた行数。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] insert(Object... objects) throws SQLException, IOException, IllegalAccessException {
			int[] results;
			
			try(Transaction transaction = beginTransaction(IsolationLevel.ReadUncommitted)) {
				results = transaction.insert(objects);
				transaction.commit();
			}
			return results;
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してINSERT文を構築し、データソースへ行の追加を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけINSERT文が実行されますが、これらのINSERT文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
		 * 
		 * @param objects INSERT文を構築するオブジェクト(複数)
		 * @return INSERT文の実行により影響を受けた行数。
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] insert(Collection<?> objects) throws SQLException, IOException, IllegalAccessException {
			return insert(objects.toArray());
		}
		
		/** 指定されたオブジェクトでUPDATE文を構築し、データソースへ行の更新を試みます。
		 * 
		 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
		 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
		 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>主キー列 id と通常の列 name を持つPersonテーブルを更新するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * DataSource.update(new Person() {
		 *     int    id   = 4;
		 *     String name = "史郎";
		 * });
		 * </code></pre>
		 * 
		 * <p>上記コード例では主キー列 id の値が 4 となっている行の name 列が "史郎" に更新されます。</p>
		 * 
		 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
		 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
		 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
		 * 
		 * @param obj UPDATE文を構築するオブジェクト
		 * @return UPDATE文の実行により影響を受けた行数。つまり行の更新が成功した場合は 1 です。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int update(Object obj) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			int[] r = update(new Object[] { obj });
			return r[0];
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してUPDATE文を構築し、データソースへ行の更新を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけUPDATE文が実行されますが、これらのUPDATE文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
		 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
		 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
		 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
		 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
		 * 
		 * @param objects UPDATE文を構築するオブジェクト(複数)
		 * @return UPDATE文の実行により影響を受けた行数。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] update(Object... objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			int[] results;
			
			try(Transaction transaction = beginTransaction(IsolationLevel.ReadUncommitted)) {
				results = transaction.update(objects);
				transaction.commit();
			}
			return results;
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してUPDATE文を構築し、データソースへ行の更新を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけUPDATE文が実行されますが、これらのUPDATE文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
		 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
		 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
		 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
		 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
		 * 
		 * @param objects UPDATE文を構築するオブジェクト(複数)
		 * @return UPDATE文の実行により影響を受けた行数。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] update(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			return update(objects.toArray());
		}
		
		/** 指定されたオブジェクトでDELETE文を構築し、データソースから行の削除を試みます。
		 * 
		 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
		 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
		 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
		 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
		 * 
		 * @param obj DELETE文を構築するオブジェクト
		 * @return DELETE文の実行により影響を受けた行数。つまり行の削除が成功した場合は 1 です。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int delete(Object obj) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			int[] r = delete(new Object[] { obj });
			return r[0];
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してDELETE文を構築し、データソースから行の削除を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけDELETE文が実行されますが、これらのDELETE文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
		 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
		 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
		 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
		 * 
		 * @param objects DELETE文を構築するオブジェクト(複数)
		 * @return DELETE文の実行により影響を受けた行数。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] delete(Object... objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			int[] results;
			
			try(Transaction transaction = beginTransaction(IsolationLevel.ReadUncommitted)) {
				results = transaction.delete(objects);
				transaction.commit();
			}
			return results;
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してDELETE文を構築し、データソースから行の削除を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけDELETE文が実行されますが、これらのDELETE文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
		 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
		 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
		 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
		 * 
		 * @param objects DELETE文を構築するオブジェクト(複数)
		 * @return DELETE文の実行により影響を受けた行数。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] delete(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			return delete(objects.toArray());
		}
		
		/** 指定されたオブジェクトでMERGE文を構築し、データソースへ行の追加または更新を試みます。
		 * 
		 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * このメソッドは単一行を更新または追加することを目的としています。
		 * 行を一意に特定するためのフィールドが指定されている必要があります。
		 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>主キー列 id と通常の列 name を持つPersonテーブルを追加または更新するには以下のようにします。</p>
		 * 
		 * <pre><code>
		 * DataSource.merge(new Person() {
		 *     int    id   = 5;
		 *     String name = "五郎";
		 * });
		 * </code></pre>
		 * 
		 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 	 * 
		 * @param obj MERGE文を構築するオブジェクト
		 * @return MERGE文の実行により影響を受けた行数。つまり行の追加または更新が成功した場合は 1 です。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int merge(Object obj) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			int[] r = merge(new Object[] { obj });
			return r[0];
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してMERGE文を構築し、データソースへ行の追加または更新を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけMERGE文が実行されますが、これらのMERGE文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * このメソッドは単一行を更新または追加することを目的としています。
		 * 行を一意に特定するためのフィールドが指定されている必要があります。
		 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 	 * 
		 * @param objects MERGE文を構築するオブジェクト(複数)
		 * @return MERGE文の実行により影響を受けた行数。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] merge(Object... objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			int[] results;
			
			try(Transaction transaction = beginTransaction(IsolationLevel.ReadUncommitted)) {
				results = transaction.merge(objects);
				transaction.commit();
			}
			return results;
		}
		
		/** 指定された複数のオブジェクトそれぞれに対してMERGE文を構築し、データソースへ行の追加または更新を試みます。
		 * 
		 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
		 * 
		 * <p>指定されたオブジェクトの数だけMERGE文が実行されますが、これらのMERGE文は1つのトランザクションとして実行されます。</p>
		 * 
		 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
		 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
		 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
		 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
		 * このメソッドは単一行を更新または追加することを目的としています。
		 * 行を一意に特定するためのフィールドが指定されている必要があります。
		 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
		 * 
		 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 	 * 
		 * @param objects MERGE文を構築するオブジェクト(複数)
		 * @return MERGE文の実行により影響を受けた行数。
		 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
		 * @throws SQLException データベースアクセスエラーが発生した場合
		 * @throws IOException 暗黙的トランザクションで自動クローズ例外が発生した場合
		 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
		 */
		public int[] merge(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IOException, IllegalAccessException {
			return merge(objects.toArray());
		}
	}
}
