package net.osdn.util.sql;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Timestamp;
import java.text.ParseException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.YearMonth;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.osdn.util.sql.OptimisticConcurrencyException.Op;

/** オブジェクト(POJO)とリレーショナル・データベースのマッピングを提供するクラスです。
 * 
 * <p>マッピングにはオブジェクトとしてPOJOを使用することができます。
 * getter/setterなどのBean作法やアノテーションは必要ありません。
 * 代わりにクラス名がテーブル名になる、フィールド名が列名になるといった規約があります。</p>
 * 
 * <p>このクラスはデータベースへのアクセスはせずに、
 * オブジェクトからSELECT、INSERT、UPDATE、DELETE、MERGEの各SQL文を作成します。</p>
 * 
 */
public class ORMapper {

	private static ConcurrentHashMap<Class<?>, String> cacheTableNames = new ConcurrentHashMap<Class<?>, String>();
	private static ConcurrentHashMap<Class<?>, String> cacheSelectStatements = new ConcurrentHashMap<Class<?>, String>();
	private static ConcurrentHashMap<Class<?>, String> cacheInsertStatements = new ConcurrentHashMap<Class<?>, String>();
	private static ConcurrentHashMap<Class<?>, String> cacheUpdateStatements = new ConcurrentHashMap<Class<?>, String>();
	private static ConcurrentHashMap<Class<?>, String> cacheDeleteStatements = new ConcurrentHashMap<Class<?>, String>();
	private static ConcurrentHashMap<Class<?>, String> cacheMergeStatements = new ConcurrentHashMap<Class<?>, String>();
	
	private ResultSet rs;
	private Class<?> cls;
	private Object obj;
	private NamedParameterStatement selectStatement;
	private NamedParameterStatement insertStatement;
	private NamedParameterStatement updateStatement;
	private NamedParameterStatement deleteStatement;
	private NamedParameterStatement mergeStatement;
	
	/** 指定したResultSetをオブジェクトに変換するためのORMapperインスタンスを作成します。
	 * 
	 * @param rs ResultSet
	 */
	public ORMapper(ResultSet rs) {
		if(rs == null) {
			throw new IllegalArgumentException();
		}
		this.rs = rs;
	}
	
	/** 指定したクラスを使ってSQL文を構築するORMapperインスタンスを作成します。
	 * 
	 * @param cls SQL文の構築に使用するクラス
	 */
	public ORMapper(Class<?> cls) {
		if(cls == null) {
			throw new IllegalArgumentException();
		}
		this.cls = cls;
	}
	
	/** 指定したオブジェクト(POJO)を使ってSQL文を構築するORMapperインスタンスを作成します。
	 * 
	 * @param obj SQL文の構築に使用するオブジェクト
	 */
	public ORMapper(Object obj) {
		if(obj == null) {
			throw new IllegalArgumentException();
		}
		this.obj = obj;
		this.cls = obj.getClass();
	}
	
	// ResultSet -> Object

	/** 保持しているResultSetから指定したクラスのインスタンスを作成します。
	 * 
	 * <p>コンストラクタにResultSetを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>ResultSetがデータのある行を指している場合、その行の値を使用します。
	 * ResultSetが先頭行の前に設定されている場合は、ResultSetのnext()メソッドを呼び出して先頭行を指すように試みます。
	 * 行データが存在しない場合は null を返します。</p>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass 返されるオブジェクトの型。
	 * @return ResultSetの現在の行の値が設定されたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。returnClassには引数を持たないコンストラクタが必要です。
	 */
	public <T> T get(Class<T> returnClass) throws SQLException, ParseException, ReflectiveOperationException {
		T obj = null;
		
		Map<String, Field> fields = getFields(returnClass);
		String[] columnNames = getColumnNames(rs.getMetaData());
		int columnCount = rs.getMetaData().getColumnCount();
		
		if(rs.isBeforeFirst()) {
			if(!rs.next()) {
				return null;
			}
		}
		try {
			if(rs.getRow() == 0) {
				return null;
			}
		} catch(SQLFeatureNotSupportedException e) {}
		
		obj = createInstance(returnClass);
		for(int i = 1; i <= columnCount; i++) {
			Field field = fields.get(columnNames[i]);
			if(field == null) {
				continue;
			}
			Object value = getValue(rs, i, field);
			field.set(obj, value);
		}
		return obj;
	}
	
	/** 保持しているResultSetから指定したクラスのインスタンスを要素に持つリストを返します。
	 * 
	 * <p>コンストラクタにResultSetを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>ResultSetのnext()メソッドを呼び出してからデータの読み取りを最終行まで繰り返します。
	 * ResultSetが先頭行の前ではなくデータ行を指している場合、その行は対象にならず、次の行から読み取りがおこなわれることに注意してください。</p>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型。
	 * @return ResultSetから読み取られた行の値を設定したオブジェクトのリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws IllegalArgumentException 指定されたクラスが null の場合
	 * @throws ReflectiveOperationException リフレクション操作例外。returnClassには引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> getList(Class<T> returnClass) throws SQLException, ParseException, IllegalArgumentException, ReflectiveOperationException {
		List<T> list = new ArrayList<T>();
		
		Map<String, Field> fields = getFields(returnClass);
		String[] columnNames = getColumnNames(rs.getMetaData());
		int columnCount = rs.getMetaData().getColumnCount();

		while(rs.next()) {
			T obj = createInstance(returnClass);
			for(int i = 1; i <= columnCount; i++) {
				Field field = fields.get(columnNames[i]);
				if(field == null) {
					continue;
				}
				Object value = getValue(rs, i, field);
				field.set(obj, value);
			}
			list.add(obj);
		}
		return list;
	}
	
	/** 保持しているResultSetから指定したクラスのインスタンスを作成する反復可能なオブジェクトを返します。
	 * 
	 * <p>返された反復可能なオブジェクトは拡張for文を使って要素を取り出すことができます。
	 * このメソッドは{@link #getList(Class)}メソッドとは異なり、
	 * すぐにResultSetの読み取りをおこないません。
	 * イテレーターのnext()メソッドが呼び出される都度、
	 * ResultSetの行を進めてオブジェクトに変換していきます。
	 * 大量の行データを扱う場合、{@link #getList(Class)}メソッドよりもメモリー効率が良くなります。</p>
	 * 
	 * @param <T> イテレータから返される要素の型
	 * @param returnClass ResultSetから読み取られた行を変換する型
	 * @return 型Tの要素の反復可能なオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public <T> ObjectIterable<T> getIterable(Class<T> returnClass) throws SQLException {
		ObjectIterator<T> iterator = new ObjectIterator<T>(returnClass, rs);
		ObjectIterable<T> iterable = new ObjectIterable<T>(iterator);
		return iterable;
	}
	
	/** 保持しているResultSetから指定したクラスのインスタンスを作成するイテレーターを返します。
	 * 
	 * <p>返されたイテレーターのnext()メソッドが呼び出される都度、
	 * ResultSetの行を進めてオブジェクトに変換していきます。
	 * 大量の行データを扱う場合、{@link #getList(Class)}メソッドよりもメモリー効率が良くなります。</p>
	 * 
	 * @param <T> イテレータから返される要素の型
	 * @param returnClass ResultSetから読み取られた行を変換する型
	 * @return 型Tの要素の返すイテレーター
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public <T> ObjectIterator<T> getIterator(Class<T> returnClass) throws SQLException {
		ObjectIterator<T> iterator = new ObjectIterator<T>(returnClass, rs);
		return iterator;
	}
	
	// Object -> NamedParameterStatement
	
	/** SELECT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @return 名前付きパラメーターを持つSELECT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getSelectStatement() throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			String tableName = getTableName(cn, cls);
			return getSelectStatement(cn, tableName, this.obj);
		}
	}
	
	/** 指定したオブジェクトを使用してSELECT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @param obj SELECT文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つSELECT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getSelectStatement(Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			String tableName = getTableName(cn, cls);
			return getSelectStatement(cn, tableName, obj);
		}
	}
	
	/** 指定した接続を使用してSELECT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @return 名前付きパラメーターを持つSELECT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getSelectStatement(Connection cn) throws SQLException, IllegalArgumentException, IllegalAccessException {
		String tableName = getTableName(cn, cls);
		return getSelectStatement(cn, tableName, this.obj);
	}
	
	/** 指定した接続、テーブル名、オブジェクトを使用してSELECT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @param tableName テーブル名
	 * @param obj SELECT文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つSELECT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getSelectStatement(Connection cn, String tableName, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(selectStatement == null) {
			String sql = cacheSelectStatements.get(cls);
			if(sql == null) {
				if(tableName == null) {
					if(obj instanceof Table) {
						tableName = ((Table)obj).getTableName();
					} else {
						tableName = getTableName(cn, cls);
					}
				}
				DatabaseMetaData md = cn.getMetaData();
				NamedParameterStatement npst = StatementBuilder.getInstance(md).createSelectStatement(md, cls, tableName, obj);
				sql = npst.getOriginalSql();
				cacheSelectStatements.put(cls,  sql);
				selectStatement = npst;
			} else {
				selectStatement = new NamedParameterStatement(sql);
			}
		}
		
		selectStatement.clearParameters();
		if(obj != null) {
			if(!cls.isInstance(obj)) {
				throw new IllegalArgumentException();
			}
			
			Map<String, Field> fields = getFields(cls);
			for(String parameterName : selectStatement.getParameterNames()) {
				Field field = fields.get(parameterName.toLowerCase());
				if(field == null) {
					continue;
				}
				Object value = field.get(obj);
				selectStatement.setObject(parameterName, value);
			}
		}
		
		return selectStatement;
	}
	
	/** INSERT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @return 名前付きパラメーターを持つINSERT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getInsertStatement() throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getInsertStatement(cn, this.obj);
		}
	}
	
	/** 指定したオブジェクトを使用してINSERT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @param obj INSERT文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つINSERT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getInsertStatement(Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getInsertStatement(cn, obj);
		}
	}
	
	/** 指定した接続を使用してINSERT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @return 名前付きパラメーターを持つINSERT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getInsertStatement(Connection cn) throws SQLException, IllegalArgumentException, IllegalAccessException {
		return getInsertStatement(cn, this.obj);
	}

	/** 指定した接続とオブジェクトを使用してINSERT文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @param obj INSERT文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つINSERT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getInsertStatement(Connection cn, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(insertStatement == null) {
			String sql = cacheInsertStatements.get(cls);
			if(sql == null) {
				String tableName = null;
				if(obj instanceof Table) {
					tableName = ((Table)obj).getTableName();
				}
				DatabaseMetaData md = cn.getMetaData();
				NamedParameterStatement npst = StatementBuilder.getInstance(md).createInsertStatement(md, cls, tableName);
				sql = npst.getOriginalSql();
				cacheInsertStatements.put(cls, sql);
				insertStatement = npst;
			} else {
				insertStatement = new NamedParameterStatement(sql);
			}
		}

		insertStatement.clearParameters();
		if(obj != null) {
			if(!cls.isInstance(obj)) {
				throw new IllegalArgumentException();
			}
			
			Map<String, Field> fields = getFields(cls);
			for(String parameterName : insertStatement.getParameterNames()) {
				Field field = fields.get(parameterName.toLowerCase());
				if(field == null) {
					continue;
				}
				Object value = field.get(obj);
				insertStatement.setObject(parameterName, value);
			}
		}
		
		return insertStatement;
	}
	
	/** UPDATE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @return 名前付きパラメーターを持つUPDATE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getUpdateStatement() throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getUpdateStatement(cn, this.obj);
		}
	}
	
	/** 指定したオブジェクトを使用してUPDATE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @param obj UPDATE文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つUPDATE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getUpdateStatement(Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getUpdateStatement(cn, obj);
		}
	}
	
	/** 指定した接続を使用してUPDATE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @return 名前付きパラメーターを持つUPDATE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getUpdateStatement(Connection cn) throws SQLException, IllegalArgumentException, IllegalAccessException {
		return getUpdateStatement(cn, this.obj);
	}
	
	/** 指定した接続とオブジェクトを使用してUPDATE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @param obj UPDATE文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つUPDATE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getUpdateStatement(Connection cn, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(updateStatement == null) {
			String sql = cacheUpdateStatements.get(cls);
			if(sql == null) {
				String tableName = null;
				if(obj instanceof Table) {
					tableName = ((Table)obj).getTableName();
				}
				DatabaseMetaData md = cn.getMetaData();
				NamedParameterStatement npst = StatementBuilder.getInstance(md).createUpdateStatement(md, cls, tableName);
				sql = npst.getOriginalSql();
				cacheUpdateStatements.put(cls, sql);
				updateStatement = npst;
			} else {
				updateStatement = new NamedParameterStatement(sql);
			}
		}
		
		updateStatement.clearParameters();
		if(obj != null) {
			if(!cls.isInstance(obj)) {
				throw new IllegalArgumentException();
			}
			Map<String, Field> fields = getFields(cls);
			for(String parameterName : updateStatement.getParameterNames()) {
				Field field = fields.get(parameterName.toLowerCase());
				if(field == null) {
					continue;
				}
				Object value = field.get(obj);
				updateStatement.setObject(parameterName, value);
			}
		}
		
		return updateStatement;
	}
	
	/** DELETE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @return 名前付きパラメーターを持つDELETE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getDeleteStatement() throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getDeleteStatement(cn, this.obj);
		}
	}

	/** 指定したオブジェクトを使用してDELETE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @param obj DELETE文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つDELETE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getDeleteStatement(Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getDeleteStatement(cn, obj);
		}
	}
	
	/** 指定した接続を使用してDELETE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @return 名前付きパラメーターを持つDELETE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getDeleteStatement(Connection cn) throws SQLException, IllegalArgumentException, IllegalAccessException {
		return getDeleteStatement(cn, this.obj);
	}
	
	/** 指定した接続とオブジェクトを使用してDELETE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @param obj DELETE文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つDELETE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getDeleteStatement(Connection cn, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(deleteStatement == null) {
			String sql = cacheDeleteStatements.get(cls);
			if(sql == null) {
				String tableName = null;
				if(obj instanceof Table) {
					tableName = ((Table)obj).getTableName();
				}
				DatabaseMetaData md = cn.getMetaData();
				NamedParameterStatement npst = StatementBuilder.getInstance(md).createDeleteStatement(md, cls, tableName);
				sql = npst.getOriginalSql();
				cacheDeleteStatements.put(cls, sql);
				deleteStatement = npst;
			} else {
				deleteStatement = new NamedParameterStatement(sql);
			}
		}

		deleteStatement.clearParameters();
		if(obj != null) {
			if(!cls.isInstance(obj)) {
				throw new IllegalArgumentException();
			}
			Map<String, Field> fields = getFields(cls);
			for(String parameterName : deleteStatement.getParameterNames()) {
				Field field = fields.get(parameterName.toLowerCase());
				if(field == null) {
					continue;
				}
				Object value = field.get(obj);
				deleteStatement.setObject(parameterName, value);
			}
		}
		
		return deleteStatement;
	}

	/** MERGE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @return 名前付きパラメーターを持つMERGE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getMergeStatement() throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getMergeStatement(cn, this.obj);
		}
	}
	
	/** 指定したオブジェクトを使用してMERGE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドではデータベース・スキーマを参照するため、
	 * 暗黙的に {@link DataSource#getConnection()}メソッドが呼び出します。
	 * {@link DataSource#initialize()}メソッドを使用して{@link DataSource}が適切に初期化されている必要があります。</p>
	 * 
	 * @param obj MERGE文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つMERGE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getMergeStatement(Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		try(Connection cn = DataSource.getConnection()) {
			return getMergeStatement(cn, obj);
		}
	}
	
	/** 指定した接続を使用してMERGE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @return 名前付きパラメーターを持つMERGE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getMergeStatement(Connection cn) throws SQLException, IllegalArgumentException, IllegalAccessException {
		return getMergeStatement(cn, this.obj);
	}
	
	/** 指定した接続とオブジェクトを使用してMERGE文を構築します。
	 * 
	 * <p>コンストラクタにクラスまたはResultSet以外のオブジェクトを指定してORMapperインスタンスを作成している必要があります。</p>
	 * 
	 * <p>このメソッドでは指定された接続を使ってデータベース・スキーマを参照するため、
	 * {@link DataSource}が初期化されている必要がありません。</p>
	 * 
	 * @param cn データベース・スキーマの参照に使用する接続
	 * @param obj MERGE文の構築に使用するオブジェクト
	 * @return 名前付きパラメーターを持つMERGE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public NamedParameterStatement getMergeStatement(Connection cn, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(mergeStatement == null) {
			String sql = cacheMergeStatements.get(cls);
			if(sql == null) {
				String tableName = null;
				if(obj instanceof Table) {
					tableName = ((Table)obj).getTableName();
				}
				DatabaseMetaData md = cn.getMetaData();
				NamedParameterStatement npst = StatementBuilder.getInstance(md).createMergeStatement(md, cls, tableName);
				sql = npst.getOriginalSql();
				cacheMergeStatements.put(cls, sql);
				mergeStatement = npst;
			} else {
				mergeStatement = new NamedParameterStatement(sql);
			}
		}

		mergeStatement.clearParameters();
		if(obj != null) {
			if(!cls.isInstance(obj)) {
				throw new IllegalArgumentException();
			}
			Map<String, Field> fields = getFields(cls);
			for(String parameterName : mergeStatement.getParameterNames()) {
				Field field = fields.get(parameterName.toLowerCase());
				if(field == null) {
					continue;
				}
				Object value = field.get(obj);
				mergeStatement.setObject(parameterName, value);
			}
		}
		
		return mergeStatement;
	}

	/** 指定した接続とクラスを使用してSELECT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param cls SELECT文の構築に使用するクラス
	 * @return SELECT文を実行して返されるResultSet
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static ResultSet select(Connection cn, Class<?> cls) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(cls == null) {
			throw new IllegalArgumentException();
		}

		String tableName = getTableName(cn, cls);
		return select(cn, tableName, null);
	}
	
	/** 指定した接続とオブジェクトを使用してSELECT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param obj SELECT文の構築に使用するオブジェクト
	 * @return SELECT文を実行して返されるResultSet
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static ResultSet select(Connection cn, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(obj == null) {
			throw new IllegalArgumentException();
		}
		
		String tableName = getTableName(cn, obj.getClass());
		if(obj instanceof Table) {
			tableName = ((Table)obj).getTableName();
		}
		return select(cn, tableName, obj);
	}
	
	/** 指定した接続、クラス、オブジェクトを使用してSELECT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param cls SELECT文の構築に使用するクラス
	 * @param obj SELECT文の構築に使用するオブジェクト
	 * @return SELECT文を実行して返されるResultSet
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static ResultSet select(Connection cn, Class<?> cls, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(cls == null) {
			throw new IllegalArgumentException();
		}
		
		String tableName = getTableName(cn, cls);
		return select(cn, tableName, obj);
	}
	
	/** 指定した接続、テーブル名、オブジェクトを使用してSELECT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param tableName テーブル名
	 * @param obj SELECT文の構築に使用するオブジェクト
	 * @return SELECT文を実行して返されるResultSet
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static ResultSet select(Connection cn, String tableName, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(tableName == null) {
			throw new IllegalArgumentException();
		}
		
		Class<?> cls = (obj != null) ? obj.getClass() : (new Table(tableName) {}).getClass();
		NamedParameterStatement npst = new ORMapper(cls).getSelectStatement(cn, tableName, obj);
		PreparedStatement st = cn.prepareStatement(npst.getSql());
		for(NamedParameter parameter : npst.getParameters()) {
			parameter.applyTo(st);
		}

		return st.executeQuery();
	}
	
	/** 指定した接続とオブジェクトを使用してINSERT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param obj INSERT文の構築に使用するオブジェクト
	 * @return INSERT文の実行により影響を受けた行数
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int insert(Connection cn, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		int[] r = insert(cn, new Object[] { obj });
		return r[0];
	}	
	
	/** 指定した接続とオブジェクト(複数)を使用してINSERT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param objects INSERT文の構築に使用するオブジェクト(複数)
	 * @return INSERT文の実行により影響を受けた行数
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] insert(Connection cn, Object... objects) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(objects == null) {
			throw new IllegalArgumentException();
		}
		
		PreparedStatement st = null;
		NamedParameterStatement nps = null;
		Class<?> prevClass = null;
		int batchCount = 0;
		List<Integer> results = new ArrayList<Integer>();
		
		try {
			for(Object obj : objects) {
				if(obj == null) {
					continue;
				}
				if(batchCount > 0 && obj.getClass() != prevClass) {
					int[] r = st.executeBatch();
					st.close();
					st = null;
					for(int i = 0; i < r.length; i++) {
						results.add(r[i]);
					}
					batchCount = 0;
				}
				if(st == null) {
					nps = new ORMapper(obj).getInsertStatement(cn);
					st = cn.prepareStatement(nps.getSql());
				}
				nps.clearParameters();
				Map<String, Object> parameters = ORMapper.createParameters(obj, nps.getParameterNames());
				for(Entry<String, Object> parameter : parameters.entrySet()) {
					nps.setObject(parameter.getKey(), parameter.getValue());
				}
				st.clearParameters();
				for(NamedParameter parameter : nps.getParameters()) {
					parameter.applyTo(st);
				}
				st.addBatch();
				batchCount++;
				prevClass = obj.getClass();
			}
			if(batchCount > 0) {
				int[] r = st.executeBatch();
				for(int i = 0; i < r.length; i++) {
					results.add(r[i]);
				}
			}
		} finally {
			if(st != null) {
				st.close();
			}
		}
		
		int[] r = new int[results.size()];
		for(int i = 0; i < results.size(); i++) {
			r[i] = results.get(i);
		}
		return r;
	}

	/** 指定した接続とオブジェクト(複数)を使用してINSERT文を構築し、実行します。
	 * 
	 * @param cn データベースへの接続
	 * @param objects INSERT文の構築に使用するオブジェクト(複数)
	 * @return INSERT文の実行により影響を受けた行数
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] insert(Connection cn, Collection<?> objects) throws SQLException, IllegalArgumentException, IllegalAccessException {
		return insert(cn, objects.toArray());
	}
	
	/** 指定した接続とオブジェクトを使用してUPDATE文を構築し、実行します。
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param obj UPDATE文の構築に使用するオブジェクト
	 * @return UPDATE文の実行により影響を受けた行数
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int update(Connection cn, Object obj) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		int[] r = update(cn, new Object[] { obj });
		return r[0];
	}
	
	/** 指定した接続とオブジェクト(複数)を使用してUPDATE文を構築し、実行します。
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param objects UPDATE文の構築に使用するオブジェクト(複数)
	 * @return UPDATE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] update(Connection cn, Object... objects) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		if(objects == null) {
			throw new IllegalArgumentException();
		}
		
		PreparedStatement st = null;
		NamedParameterStatement nps = null;
		Class<?> prevClass = null;
		int batchCount = 0;
		List<Integer> results = new ArrayList<Integer>();
		
		try {
			boolean[] hasRowVersion = new boolean[objects.length];
			for(Object obj : objects) {
				if(obj == null) {
					continue;
				}
				if(batchCount > 0 && obj.getClass() != prevClass) {
					int[] r = st.executeBatch();
					st.close();
					st = null;
					for(int i = 0; i < r.length; i++) {
						if(hasRowVersion[i] && r[i] == 0) {
							throw new OptimisticConcurrencyException(Op.Update);
						}
						results.add(r[i]);
					}
					batchCount = 0;
					hasRowVersion = new boolean[objects.length];
				}
				if(st == null) {
					nps = new ORMapper(obj).getUpdateStatement(cn);
					st = cn.prepareStatement(nps.getSql());
				}
				nps.clearParameters();
				Map<String, Object> parameters = ORMapper.createParameters(obj, nps.getParameterNames());
				for(Entry<String, Object> parameter : parameters.entrySet()) {
					nps.setObject(parameter.getKey(), parameter.getValue());
				}
				st.clearParameters();
				for(NamedParameter parameter : nps.getParameters()) {
					parameter.applyTo(st);
				}
				st.addBatch();
				hasRowVersion[batchCount] = nps.hasRowVersionColumn;
				batchCount++;
				prevClass = obj.getClass();
			}
			if(batchCount > 0) {
				int[] r = st.executeBatch();
				for(int i = 0; i < r.length; i++) {
					if(hasRowVersion[i] && r[i] == 0) {
						throw new OptimisticConcurrencyException(Op.Update);
					}
					results.add(r[i]);
				}
			}
		} finally {
			if(st != null) {
				st.close();
			}
		}
		
		int[] r = new int[results.size()];
		for(int i = 0; i < results.size(); i++) {
			r[i] = results.get(i);
		}
		return r;
	}
	
	/** 指定した接続とオブジェクト(複数)を使用してUPDATE文を構築し、実行します。
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param objects UPDATE文の構築に使用するオブジェクト(複数)
	 * @return UPDATE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] update(Connection cn, Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		return update(cn, objects.toArray());
	}
	
	/** 指定した接続とオブジェクトを使用してDELETE文を構築し、実行します。
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param obj DELETE文の構築に使用するオブジェクト
	 * @return DELETE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int delete(Connection cn, Object obj) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		int[] r = delete(cn, new Object[] { obj });
		return r[0];
	}

	/** 指定した接続とオブジェクト(複数)を使用してDELETE文を構築し、実行します。
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param objects DELETE文の構築に使用するオブジェクト(複数)
	 * @return DELETE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] delete(Connection cn, Object... objects) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		if(objects == null) {
			throw new IllegalArgumentException();
		}
		
		PreparedStatement st = null;
		NamedParameterStatement nps = null;
		Class<?> prevClass = null;
		int batchCount = 0;
		List<Integer> results = new ArrayList<Integer>();
		
		try {
			boolean[] hasRowVersion = new boolean[objects.length];
			for(Object obj : objects) {
				if(obj == null) {
					continue;
				}
				if(batchCount > 0 && obj.getClass() != prevClass) {
					int[] r = st.executeBatch();
					st.close();
					st = null;
					for(int i = 0; i < r.length; i++) {
						if(hasRowVersion[i] && r[i] == 0) {
							throw new OptimisticConcurrencyException(Op.Delete);
						}
						results.add(r[i]);
					}
					batchCount = 0;
					hasRowVersion = new boolean[objects.length];
				}
				if(st == null) {
					nps = new ORMapper(obj).getDeleteStatement(cn);
					st = cn.prepareStatement(nps.getSql());
				}
				nps.clearParameters();
				Map<String, Object> parameters = ORMapper.createParameters(obj, nps.getParameterNames());
				for(Entry<String, Object> parameter : parameters.entrySet()) {
					nps.setObject(parameter.getKey(), parameter.getValue());
				}
				st.clearParameters();
				for(NamedParameter parameter : nps.getParameters()) {
					parameter.applyTo(st);
				}
				st.addBatch();
				hasRowVersion[batchCount] = nps.hasRowVersionColumn;
				batchCount++;
				prevClass = obj.getClass();
			}
			if(batchCount > 0) {
				int[] r = st.executeBatch();
				for(int i = 0; i < r.length; i++) {
					if(hasRowVersion[i] && r[i] == 0) {
						throw new OptimisticConcurrencyException(Op.Delete);
					}
					results.add(r[i]);
				}
			}
		} finally {
			if(st != null) {
				st.close();
			}
		}
		
		int[] r = new int[results.size()];
		for(int i = 0; i < results.size(); i++) {
			r[i] = results.get(i);
		}
		return r;
	}
	
	/** 指定した接続とオブジェクト(複数)を使用してDELETE文を構築し、実行します。
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param objects DELETE文の構築に使用するオブジェクト(複数)
	 * @return DELETE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] delete(Connection cn, Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		return delete(cn, objects.toArray());
	}
	
	/** 指定した接続とオブジェクトを使用してMERGE文を構築し、実行します。
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param obj MERGE文の構築に使用するオブジェクト
	 * @return MERGE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int merge(Connection cn, Object obj) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		int[] r = merge(cn, new Object[] { obj });
		return r[0];
	}
	
	/** 指定した接続とオブジェクト(複数)を使用してMERGE文を構築し、実行します。
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param objects MERGE文の構築に使用するオブジェクト(複数)
	 * @return MERGE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] merge(Connection cn, Object... objects) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		if(objects == null) {
			throw new IllegalArgumentException();
		}
		
		PreparedStatement st = null;
		NamedParameterStatement nps = null;
		Class<?> prevClass = null;
		int batchCount = 0;
		List<Integer> results = new ArrayList<Integer>();
		
		try {
			boolean[] hasRowVersion = new boolean[objects.length];
			for(Object obj : objects) {
				if(obj == null) {
					continue;
				}
				if(batchCount > 0 && obj.getClass() != prevClass) {
					int[] r = st.executeBatch();
					st.close();
					st = null;
					for(int i = 0; i < r.length; i++) {
						if(hasRowVersion[i] && r[i] == 0) {
							throw new OptimisticConcurrencyException(Op.Update);
						}
						results.add(r[i]);
					}
					batchCount = 0;
					hasRowVersion = new boolean[objects.length];
				}
				if(st == null) {
					nps = new ORMapper(obj).getMergeStatement(cn);
					st = cn.prepareStatement(nps.getSql());
				}
				nps.clearParameters();
				Map<String, Object> parameters = ORMapper.createParameters(obj, nps.getParameterNames());
				for(Entry<String, Object> parameter : parameters.entrySet()) {
					nps.setObject(parameter.getKey(), parameter.getValue());
				}
				st.clearParameters();
				for(NamedParameter parameter : nps.getParameters()) {
					parameter.applyTo(st);
				}
				st.addBatch();
				hasRowVersion[batchCount] = nps.hasRowVersionColumn;
				batchCount++;
				prevClass = obj.getClass();
			}
			if(batchCount > 0) {
				int[] r = st.executeBatch();
				for(int i = 0; i < r.length; i++) {
					if(hasRowVersion[i] && r[i] == 0) {
						throw new OptimisticConcurrencyException(Op.Update);
					}
					results.add(r[i]);
				}
			}
		} finally {
			if(st != null) {
				st.close();
			}
		}
		
		int[] r = new int[results.size()];
		for(int i = 0; i < results.size(); i++) {
			r[i] = results.get(i);
		}
		return r;
	}
	
	/** 指定した接続とオブジェクト(複数)を使用してMERGE文を構築し、実行します。
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param cn データベースへの接続
	 * @param objects MERGE文の構築に使用するオブジェクト(複数)
	 * @return MERGE文の実行により影響を受けた行数
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 下位メソッドに不正な引数が渡された場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public static int[] merge(Connection cn, Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IllegalArgumentException, IllegalAccessException {
		return merge(cn, objects.toArray());
	}
	
	protected static Map<String, Object> createParameters(Object obj, Set<String> parameterNames) throws IllegalArgumentException, IllegalAccessException {
		if(obj == null) {
			throw new IllegalArgumentException();
		}
		
		Map<String, Object> parameters = new LinkedHashMap<String, Object>();
		
		Map<String, Field> fields = getFields(obj.getClass());
		for(String parameterName : parameterNames) {
			Field field = fields.get(parameterName.toLowerCase());
			if(field == null) {
				continue;
			}
			Object value = field.get(obj);
			parameters.put(parameterName, value);
		}
		
		return parameters;
	}

	protected static String getTableName(Connection cn, Class<?> cls) throws SQLException {
		String tableName = cacheTableNames.get(cls);
		if(tableName == null) {
			Set<String> tableNames = new HashSet<String>();
			ResultSet rs = null;
			try {
				rs = cn.getMetaData().getTables(null, null, null, new String[]{"TABLE"});
				while(rs.next()) {
					String s = rs.getString("TABLE_NAME");
					tableNames.add(s.toLowerCase());
				}
			} finally {
				if(rs != null) {
					rs.close();
				}
			}

			Class<?> c = cls;
			while(c != null) {
				String s = c.getSimpleName().toLowerCase();
				if(tableNames.contains(s)) {
					tableName = s;
					break;
				}
				c = c.getSuperclass();
			}
			cacheTableNames.put(cls, tableName);
		}
			
		return tableName;
	}
	
	protected static String[] getColumnNames(ResultSetMetaData md) throws SQLException {
		List<String> names = new ArrayList<String>();
		names.add("");
		int columnCount = md.getColumnCount();
		for(int i = 1; i <= columnCount; i++) {
			String name = md.getColumnLabel(i);
			if(name == null || name.length() == 0) {
				name = md.getColumnName(i);
			}
			if(name == null || name.length() == 0) {
				name = "";
			}
			names.add(name.toLowerCase());
		}
		return names.toArray(new String[]{});
	}
	
	protected static Map<String, Field> getFields(Class<?> cls) {
		Map<String, Field> map = new HashMap<String, Field>();
		
		for(Field field : cls.getDeclaredFields()) {
			field.setAccessible(true);
			map.put(field.getName().toLowerCase(), field);
		}
		return map;
	}
	
	protected static Object getValue(ResultSet rs, int columnIndex, Field target) throws SQLException, ParseException {
		Object value = null;
		if(target.getType().equals(byte.class)) {
			value = rs.getByte(columnIndex);
			if(rs.wasNull()) {
				value = (byte)0;
			}
		} else if(target.getType().equals(Byte.class)) {
			value = rs.getByte(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(short.class)) {
			value = rs.getShort(columnIndex);
			if(rs.wasNull()) {
				value = (short)0;
			}
		} else if(target.getType().equals(Short.class)) {
			value = rs.getShort(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(int.class)) {
			value = rs.getInt(columnIndex);
			if(rs.wasNull()) {
				value = (int)0;
			}
		} else if(target.getType().equals(Integer.class)) {
			value = rs.getInt(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(long.class)) {
			value = rs.getLong(columnIndex);
			if(rs.wasNull()) {
				value = (long)0L;
			}
		} else if(target.getType().equals(Long.class)) {
			value = rs.getLong(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(float.class)) {
			value = rs.getFloat(columnIndex);
			if(rs.wasNull()) {
				value = (float)0f;
			}
		} else if(target.getType().equals(Float.class)) {
			value = rs.getFloat(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(double.class)) {
			value = rs.getDouble(columnIndex);
			if(rs.wasNull()) {
				value = (double)0.0d;
			}
		} else if(target.getType().equals(Double.class)) {
			value = rs.getDouble(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(BigDecimal.class)) {
			value = rs.getBigDecimal(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(boolean.class)) {
			value = rs.getBoolean(columnIndex);
			if(rs.wasNull()) {
				value = false;
			}
		} else if(target.getType().equals(Boolean.class)) {
			value = rs.getBoolean(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(String.class)) {
			value = rs.getString(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().isArray() && target.getType().getComponentType().equals(byte.class)) {
			value = rs.getBytes(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(java.util.Date.class)) {
			Timestamp ts = rs.getTimestamp(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				value = new Date(ts.getTime());
			}
		} else if(target.getType().equals(java.sql.Date.class)) {
			value = rs.getDate(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(java.sql.Time.class)) {
			value = rs.getTime(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(java.sql.Timestamp.class)) {
			value = rs.getTimestamp(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		} else if(target.getType().equals(LocalDateTime.class)) {
			value = rs.getTimestamp(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				value = ((java.sql.Timestamp)value).toLocalDateTime();
			}
		} else if(target.getType().equals(LocalDate.class)) {
			value = rs.getDate(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				value = ((java.sql.Date)value).toLocalDate();
			}
		} else if(target.getType().equals(LocalTime.class)) {
			value = rs.getDate(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				value =((java.sql.Time)value).toLocalTime();
			}
		} else if(target.getType().equals(OffsetDateTime.class)) {
			String s = rs.getString(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				value = DateTimeParser.parseOffsetDateTime(s);
			}
		} else if(target.getType().equals(OffsetTime.class)) {
			String s = rs.getString(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				value = DateTimeParser.parseOffsetTime(s);
			}
		} else if(target.getType().equals(YearMonth.class)) {
			java.sql.Date d = rs.getDate(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				LocalDate ld = d.toLocalDate();
				value = YearMonth.of(ld.getYear(), ld.getMonth());
			}
		} else if(target.getType().isEnum()) {
			String s = rs.getString(columnIndex);
			if(rs.wasNull()) {
				value = null;
			} else {
				//value = Enum.valueOf(target.getType().asSubclass(Enum.class), s);
				@SuppressWarnings("rawtypes")
				Class<? extends Enum> cls = target.getType().asSubclass(Enum.class);
				@SuppressWarnings("unchecked")
				Enum<?> e = Enum.valueOf(cls, s);
				value = e;
			}
		} else {
			value = rs.getObject(columnIndex);
			if(rs.wasNull()) {
				value = null;
			}
		}
		return value;
	}
	
	/* package private */ static <T> T createInstance(Class<T> cls) throws InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException, NoSuchMethodException {
		
		LOCALCLASS: if(cls.isLocalClass()) {
			Method enclosingMethod = cls.getEnclosingMethod();
			if(enclosingMethod != null) {
				if(Modifier.isStatic(enclosingMethod.getModifiers())) {
					break LOCALCLASS;
				}
			}
			Constructor<?> enclosingConstructor = cls.getEnclosingConstructor();
			if(enclosingConstructor != null) {
				if(Modifier.isStatic(enclosingConstructor.getModifiers())) {
					break LOCALCLASS;
				}
			}
			
			Class<?> enclosingClass = cls.getEnclosingClass();
			Constructor<T> constructor = cls.getDeclaredConstructor(enclosingClass);
			constructor.setAccessible(true);
			// FIXME: どうやったらエンクロージング・クラスのインスタンスを取得することができる？
			// エンクロージング・クラスのインスタンスが取得できないので、とりあえず null でインスタンスを作成。
			T obj = constructor.newInstance((Object)null);
			return obj;
		}
		
		Constructor<T> constructor = cls.getDeclaredConstructor();
		constructor.setAccessible(true);
		T obj = constructor.newInstance();
		return obj;
	}
	
	protected static class ParseResult {
		
		public Integer year;
		public Integer month;
		public Integer dayOfMonth;
		public Integer hour;
		public Integer minute;
		public Integer second;
		public Integer nanoOfSecond;
		public ZoneOffset offset;
		
	}
	
	protected static class DateTimeParser {

		private static final ZoneOffset DEFAULT_OFFSET = ZoneOffset.ofTotalSeconds(TimeZone.getDefault().getRawOffset() / 1000);
		private static final Pattern DATETIME = Pattern.compile("(?:|(\\d{1,4})[-/](\\d{1,2})(?:|[-/](\\d{1,2})))(?:|(?:(?:^|[ T])(\\d{1,2}):(\\d{1,2})(?:|(?::(\\d{1,2})))(?:|(?:\\.(\\d{1,3})))(?:|(?:([^:\\.].*)))))");
		
		public static OffsetDateTime parseOffsetDateTime(String string) throws ParseException {
			int year;
			int month;
			int dayOfMonth;
			int hour = 0;
			int minute = 0;
			int second = 0;
			int nanoOfSecond = 0;
			ZoneOffset offset = DEFAULT_OFFSET;
			
			ParseResult result = parse(string);
			
			if(result.year != null) {
				year = result.year;
			} else {
				throw new ParseException(string, 0);
			}
			if(result.month != null) {
				month = result.month;
			} else {
				throw new ParseException(string, 0);
			}
			if(result.dayOfMonth != null) {
				dayOfMonth = result.dayOfMonth;
			} else {
				throw new ParseException(string, 0);
			}
			if(result.hour != null) {
				hour = result.hour;
				if(result.minute != null) {
					minute = result.minute;
				} else {
					throw new ParseException(string, 0);
				}
			}
			if(result.second != null) {
				second = result.second;
			}
			if(result.nanoOfSecond != null) {
				nanoOfSecond = result.nanoOfSecond;
			}
			if(result.offset != null) {
				offset = result.offset;
			}
			OffsetDateTime datetime = OffsetDateTime.of(year, month, dayOfMonth, hour, minute, second, nanoOfSecond, offset);
			return datetime;
		}
		
		public static OffsetTime parseOffsetTime(String string) throws ParseException {
			int year = 1970;
			int month = 1;
			int dayOfMonth = 1;
			int hour = 0;
			int minute = 0;
			int second = 0;
			int nanoOfSecond = 0;
			ZoneOffset offset = DEFAULT_OFFSET;
			
			ParseResult result = parse(string);
			
			if(result.year != null) {
				year = result.year;
			}
			if(result.month != null) {
				month = result.month;
			}
			if(result.dayOfMonth != null) {
				dayOfMonth = result.dayOfMonth;
			}
			if(result.hour != null) {
				hour = result.hour;
				if(result.minute != null) {
					minute = result.minute;
				} else {
					throw new ParseException(string, 0);
				}
			} else {
				throw new ParseException(string, 0);
			}
			if(result.second != null) {
				second = result.second;
			}
			if(result.nanoOfSecond != null) {
				nanoOfSecond = result.nanoOfSecond;
			}
			if(result.offset != null) {
				offset = result.offset;
			}
			OffsetDateTime datetime = OffsetDateTime.of(year, month, dayOfMonth, hour, minute, second, nanoOfSecond, offset);
			return datetime.toOffsetTime();
		}
		
		public static ParseResult parse(String string) throws ParseException {
			ParseResult result = new ParseResult();
			
			Matcher m = DATETIME.matcher(string);
			if(m.matches()) {
				if(m.group(1) != null) {
					result.year = Integer.parseInt(m.group(1));
				}
				if(m.group(2) != null) {
					result.month = Integer.parseInt(m.group(2));
				}
				if(m.group(3) != null) {
					result.dayOfMonth = Integer.parseInt(m.group(3));
				}
				if(m.group(4) != null) {
					result.hour = Integer.parseInt(m.group(4));
					result.minute = Integer.parseInt(m.group(5));
				}
				if(m.group(6) != null) {
					result.second = Integer.parseInt(m.group(6));
				}
				if(m.group(7) != null) {
					result.nanoOfSecond = Integer.parseInt((m.group(7) + "000000000").substring(0, 9));
				}
				if(m.group(8) != null) {
					String tz = m.group(8);
					if(tz.charAt(0) == ' ') {
						tz = "+" + tz.substring(1);
					}
					result.offset = ZoneOffset.of(tz);
				}
				return result;
			} else {
				throw new ParseException(string, 0);
			}
		}
	}
}
