package net.osdn.util.sql;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import net.osdn.util.sql.h2.H2StatementBuilder;
import net.osdn.util.sql.mssql.MssqlStatementBuilder;

/** オブジェクトからSQL文を構築するためのビルダークラスです。
 * 
 * <p>データベースごとのSQL構文差異を吸収するために、
 * データベース固有のビルダーを追加することができます。</p>
 * 
 * <p>既定で以下のビルダーが追加されています。</p>
 * <ul>
 * <li>{@link MssqlStatementBuilder} (SQL Server)</li>
 * <li>{@link H2StatementBuilder} (H2)</li>
 * </ul>
 * 
 */
public abstract class StatementBuilder {
	
	private static StatementBuilder defaultBuilder = new DefaultStatementBuilder();
	private static Set<StatementBuilder> builders = new HashSet<StatementBuilder>();
	
	static {
		add(new MssqlStatementBuilder());
		add(new H2StatementBuilder());
	}
	
	/** SQLを構築するビルダーを追加します。
	 * 
	 * @param builder SQLを構築するビルダー
	 */
	public static void add(StatementBuilder builder) {
		builders.add(builder);
	}
	
	/** 指定したデータベース・メタデータに適合する構文ビルダーを取得します。
	 * 
	 * @param md データベース・メタデータ
	 * @return 適合する構文ビルダー。適合する構文ビルダーが追加されていない場合は既定の構文ビルダーが返されます。
	 */
	public static StatementBuilder getInstance(DatabaseMetaData md) {
		for(StatementBuilder builder : builders) {
			try {
				if(builder.isAcceptable(md)) {
					return builder;
				}
			} catch(Exception e) {
				e.printStackTrace();
			}
		}
		return defaultBuilder;
	}
	
	/** 指定されたデータベース・メタデータが、この構文ビルダーに適合するかどうかをチェックします。
	 * 
	 * <p>データベース固有の構文ビルダーはデータベース・メタデータからデータベースの製品名などの情報を参照して、
	 * 構文ビルダーがサポートするデータベースかどうかをチェックする必要があります。</p>
	 * 
	 * @param md データベース製品を特定するために参照されるデータベース・メタデータ
	 * @return この構文ビルダーが指定されたメタデータからデータベースをサポートする場合はtrue、そうでなければfalse
	 * @throws Exception 構文ビルダー固有の例外がスローされることがあります。(主にSQLException)
	 */
	public abstract boolean isAcceptable(DatabaseMetaData md) throws Exception;
	
	/** 指定されたデータベース・メタデータ、クラス、テーブル名、オブジェクトから
	 * 名前付きパラメーターをサポートするSELECT文を構築します。
	 * 
	 * <p>tableNameにnull以外の値が指定された場合、これをテーブル名として扱います。
	 * tableNameがnullの場合にはclsをテーブル名として扱います。</p>
	 * 
	 * <p>objのフィールドはSELECT文のWHERE句を構成する列名として使用されます。
	 * objがnullの場合にはWHERE句を持たない、つまりすべての行を返すSELECT文が構築されます。</p>
	 * 
	 * <p>このメソッドが返す名前付きパラメーター・ステートメントはパラメーターの値が設定されていない状態です。
	 * objのフィールドはWHERE句に出現する列名を決定することにのみに使用され、列の値には作用しません。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名を表すクラス
	 * @param tableName テーブル名。nullを指定した場合はclsをテーブル名として扱います。
	 * @param obj WHERE句に指定される列を指定するオブジェクトです。
	 * @return 名前付きパラメーターを持つ
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public NamedParameterStatement createSelectStatement(DatabaseMetaData md, Class<?> cls, String tableName, Object obj) throws SQLException {
		if(tableName == null) {
			tableName = getTableName(md, cls);
		}
		if(tableName == null) {
			throw new IllegalArgumentException();
		}
		
		Set<String> fieldNames = new HashSet<String>();
		if(obj != null) {
			Field[] fields = getFields(obj.getClass());
			for(int i = 0; i < fields.length; i++) {
				fieldNames.add(fields[i].getName().toLowerCase());
			}
		}
		
		List<Column> columns = getColumns(md, tableName, fieldNames);
		
		if(columns.size() == 0) {
			throw new IllegalStateException();
		}
		
		boolean hasRowVersionColumn = false;
		List<Column> whereClauseColumns = new ArrayList<Column>();
		for(Column column : columns) {
			if(!fieldNames.contains(column.name.toLowerCase())) {
				continue;
			}
			if(column.isRowVersion) {
				hasRowVersionColumn = true;
			}
			
			whereClauseColumns.add(column);
		}
		
		StringBuilder sb = new StringBuilder();
		sb.append("SELECT * FROM \"");
		sb.append(tableName);
		sb.append("\"");
		if(whereClauseColumns.size() >= 1) {
			sb.append(" WHERE ");
			for(int i = 0; i < whereClauseColumns.size(); i++) {
				Column column = whereClauseColumns.get(i);
				if(column.isNullable) {
					sb.append("(\"");
					sb.append(column.name);
					sb.append("\" = :");
					sb.append(column.name);
					sb.append(" OR (\"");
					sb.append(column.name);
					sb.append("\" IS NULL AND :");
					sb.append(column.name);
					sb.append(" IS NULL))");
				} else {
					sb.append('"');
					sb.append(column.name);
					sb.append("\" = :");
					sb.append(column.name);
				}
				if(i + 1 < whereClauseColumns.size()) {
					sb.append(" AND ");
				}
			}
		}
		
		NamedParameterStatement npst = new NamedParameterStatement(sb.toString());
		npst.hasRowVersionColumn = hasRowVersionColumn;
		return npst;
	}
	
	/** 指定されたデータベース・メタデータ、クラス、テーブル名から
	 * 名前付きパラメーターをサポートするINSERT文を構築します。
	 * 
	 * <p>tableNameにnull以外の値が指定された場合、これをテーブル名として扱います。
	 * tableNameがnullの場合にはclsをテーブル名として扱います。</p>
	 * 
	 * <p>clsのフィールドはINSERT文のVALUES句を構成する列名として使用されます。</p>
	 * 
	 * <p>このメソッドが返す名前付きパラメーター・ステートメントはパラメーターの値が設定されていない状態です。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名を表すクラス。フィールドはVALUES句の列名として扱われます。
	 * @param tableName テーブル名。nullを指定した場合はclsをテーブル名として扱います。
	 * @return 名前付きパラメーターを持つINSERT文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public NamedParameterStatement createInsertStatement(DatabaseMetaData md, Class<?> cls, String tableName) throws SQLException {
		if(tableName == null) {
			tableName = getTableName(md, cls);
		}
		if(tableName == null) {
			throw new IllegalArgumentException();
		}

		Set<String> fieldNames = new HashSet<String>();
		Field[] fields = getFields(cls);
		for(int i = 0; i < fields.length; i++) {
			fieldNames.add(fields[i].getName().toLowerCase());
		}
		
		List<Column> columns = getColumns(md, tableName, fieldNames);
		
		if(columns.size() == 0) {
			throw new IllegalStateException();
		}
		
		List<Column> valuesClauseColumns = new ArrayList<Column>();
		for(Column column : columns) {
			if(!fieldNames.contains(column.name.toLowerCase())) {
				continue;
			}
			if(column.isAutoincrement) {
				continue;
			}
			if(column.isRowVersion) {
				continue;
			}
			valuesClauseColumns.add(column);
		}
		if(valuesClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		
		StringBuilder sb = new StringBuilder();
		sb.append("INSERT INTO \"");
		sb.append(tableName);
		sb.append("\"(");
		for(int i = 0; i < valuesClauseColumns.size(); i++) {
			Column column = valuesClauseColumns.get(i);
			sb.append('"');
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < valuesClauseColumns.size()) {
				sb.append(", ");
			}
		}
		sb.append(") VALUES(");
		for(int i = 0; i < valuesClauseColumns.size(); i++) {
			Column column = valuesClauseColumns.get(i);
			sb.append(':');
			sb.append(column.name);
			if(i + 1 < valuesClauseColumns.size()) {
				sb.append(", ");
			}
		}
		sb.append(')');
		
		return new NamedParameterStatement(sb.toString());
	}
	
	/** 指定されたデータベース・メタデータ、クラス、テーブル名から
	 * 名前付きパラメーターをサポートするUPDATE文を構築します。
	 * 
	 * <p>tableNameにnull以外の値が指定された場合、これをテーブル名として扱います。
	 * tableNameがnullの場合にはclsをテーブル名として扱います。</p>
	 * 
	 * <p>clsのフィールドはUPDATE文のSET句またはWHERE句を構成する列名として使用されます。
	 * データベース・メタデータからテーブルのスキーマを参照しテーブルの行を一意に特定するための列を求めます。
	 * 行を一意に特定するための列と行バージョン列はWHERE句に指定されます。WHERE句に指定される列が少なくとも1つ以上必要です。
	 * その他の列はSET句に指定されます。(ただし自動インクリメント列は除外されます。)</p>
	 * 
	 * <p>このメソッドが返す名前付きパラメーター・ステートメントはパラメーターの値が設定されていない状態です。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名を表すクラス。フィールドはSET句またはWHERE句の列名として扱われます。
	 * @param tableName テーブル名。nullを指定した場合はclsをテーブル名として扱います。
	 * @return 名前付きパラメーターを持つUPDATE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public NamedParameterStatement createUpdateStatement(DatabaseMetaData md, Class<?> cls, String tableName) throws SQLException {
		if(tableName == null) {
			tableName = getTableName(md, cls);
		}
		if(tableName == null) {
			throw new IllegalArgumentException();
		}

		Set<String> fieldNames = new HashSet<String>();
		Field[] fields = getFields(cls);
		for(int i = 0; i < fields.length; i++) {
			fieldNames.add(fields[i].getName().toLowerCase());
		}
		
		List<Column> columns = getColumns(md, tableName, fieldNames);
		if(columns.size() == 0) {
			throw new IllegalStateException();
		}
		
		boolean hasRowVersionColumn = false;
		List<Column> setClauseColumns = new ArrayList<Column>();
		List<Column> whereClauseColumns = new ArrayList<Column>();
		for(Column column : columns) {
			if(!fieldNames.contains(column.name.toLowerCase())) {
				continue;
			}
			if(column.isRowVersion) {
				hasRowVersionColumn = true;
			}
			if(!column.isRowIdentifier && !column.isAutoincrement && !column.isRowVersion) {
				setClauseColumns.add(column);
			}
			if(column.isRowIdentifier || column.isRowVersion) {
				whereClauseColumns.add(column);
			}
		}
		if(setClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		if(whereClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		
		StringBuilder sb = new StringBuilder();
		sb.append("UPDATE \"");
		sb.append(tableName);
		sb.append("\" SET ");
		for(int i = 0; i < setClauseColumns.size(); i++) {
			Column column = setClauseColumns.get(i);
			sb.append('"');
			sb.append(column.name);
			sb.append("\" = :");
			sb.append(column.name);
			if(i + 1 < setClauseColumns.size()) {
				sb.append(", ");
			}
		}
		
		sb.append(" WHERE ");
		for(int i = 0; i < whereClauseColumns.size(); i++) {
			Column column = whereClauseColumns.get(i);
			if(column.isNullable) {
				sb.append("(\"");
				sb.append(column.name);
				sb.append("\" = :");
				sb.append(column.name);
				sb.append(" OR (\"");
				sb.append(column.name);
				sb.append("\" IS NULL AND :");
				sb.append(column.name);
				sb.append(" IS NULL))");
			} else {
				sb.append('"');
				sb.append(column.name);
				sb.append("\" = :");
				sb.append(column.name);
			}
			if(i + 1 < whereClauseColumns.size()) {
				sb.append(" AND ");
			}
		}
		
		NamedParameterStatement npst = new NamedParameterStatement(sb.toString());
		npst.hasRowVersionColumn = hasRowVersionColumn;
		return npst;
	}
	
	/** 指定されたデータベース・メタデータ、クラス、テーブル名から
	 * 名前付きパラメーターをサポートするDELETE文を構築します。
	 * 
	 * <p>tableNameにnull以外の値が指定された場合、これをテーブル名として扱います。
	 * tableNameがnullの場合にはclsをテーブル名として扱います。</p>
	 * 
	 * <p>clsのフィールドはDELETE文のWHERE句を構成する列名として使用されます。
	 * データベース・メタデータからテーブルのスキーマを参照しテーブルの行を一意に特定するための列を求めます。
	 * 行を一意に特定するための列と行バージョン列はWHERE句に指定されます。WHERE句に指定される列が少なくとも1つ以上必要です。
	 * その他の列は無視されます。</p>
	 * 
	 * <p>このメソッドが返す名前付きパラメーター・ステートメントはパラメーターの値が設定されていない状態です。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名を表すクラス。フィールドの一部はWHERE句の列名として扱われます。
	 * @param tableName テーブル名。nullを指定した場合はclsをテーブル名として扱います。
	 * @return 名前付きパラメーターを持つDELETE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public NamedParameterStatement createDeleteStatement(DatabaseMetaData md, Class<?> cls, String tableName) throws SQLException {
		if(tableName == null) {
			tableName = getTableName(md, cls);
		}
		if(tableName == null) {
			throw new IllegalArgumentException();
		}

		Set<String> fieldNames = new HashSet<String>();
		Field[] fields = getFields(cls);
		for(int i = 0; i < fields.length; i++) {
			fieldNames.add(fields[i].getName().toLowerCase());
		}
		
		List<Column> columns = getColumns(md, tableName, fieldNames);
		if(columns.size() == 0) {
			throw new IllegalStateException();
		}
		
		boolean hasRowVersionColumn = false;
		List<Column> whereClauseColumns = new ArrayList<Column>();
		for(Column column : columns) {
			if(!fieldNames.contains(column.name.toLowerCase())) {
				continue;
			}
			if(column.isRowVersion) {
				hasRowVersionColumn = true;
			}
			if(column.isRowIdentifier || column.isRowVersion) {
				whereClauseColumns.add(column);
			}
		}
		if(whereClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		
		StringBuilder sb = new StringBuilder();
		sb.append("DELETE FROM \"");
		sb.append(tableName);
		sb.append("\" WHERE ");
		for(int i = 0; i < whereClauseColumns.size(); i++) {
			Column column = whereClauseColumns.get(i);
			if(column.isNullable) {
				sb.append("(\"");
				sb.append(column.name);
				sb.append("\" = :");
				sb.append(column.name);
				sb.append(" OR (\"");
				sb.append(column.name);
				sb.append("\" IS NULL AND :");
				sb.append(column.name);
				sb.append(" IS NULL))");
			} else {
				sb.append('"');
				sb.append(column.name);
				sb.append("\" = :");
				sb.append(column.name);
			}
			if(i + 1 < whereClauseColumns.size()) {
				sb.append(" AND ");
			}
		}
		
		NamedParameterStatement npst = new NamedParameterStatement(sb.toString());
		npst.hasRowVersionColumn = hasRowVersionColumn;
		return npst;
	}
	
	/** 指定されたデータベース・メタデータ、クラス、テーブル名から
	 * 名前付きパラメーターをサポートするMERGE文を構築します。
	 * 
	 * <p>tableNameにnull以外の値が指定された場合、これをテーブル名として扱います。
	 * tableNameがnullの場合にはclsをテーブル名として扱います。</p>
	 * 
	 * <p>clsのフィールドはMERGE文を構成する列名として使用されます。</p>
	 * 
	 * <p>このメソッドが返す名前付きパラメーター・ステートメントはパラメーターの値が設定されていない状態です。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名を表すクラス。フィールドの一部は列名として扱われます。
	 * @param tableName テーブル名。nullを指定した場合はclsをテーブル名として扱います。
	 * @return 名前付きパラメーターを持つMERGE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public NamedParameterStatement createMergeStatement(DatabaseMetaData md, Class<?> cls, String tableName) throws SQLException {
		if(tableName == null) {
			tableName = getTableName(md, cls);
		}
		if(tableName == null) {
			throw new IllegalArgumentException();
		}

		Set<String> fieldNames = new HashSet<String>();
		Field[] fields = getFields(cls);
		for(int i = 0; i < fields.length; i++) {
			fieldNames.add(fields[i].getName().toLowerCase());
		}
		
		List<Column> columns = getColumns(md, tableName, fieldNames);
		if(columns.size() == 0) {
			throw new IllegalStateException();
		}

		List<Column> selectClauseColumns = new ArrayList<Column>();
		List<Column> setClauseColumns = new ArrayList<Column>();
		List<Column> whereClauseColumns = new ArrayList<Column>();
		List<Column> valuesClauseColumns = new ArrayList<Column>();
		for(Column column : columns) {
			if(!fieldNames.contains(column.name.toLowerCase())) {
				continue;
			}

			selectClauseColumns.add(column);
			
			if(!column.isRowIdentifier && !column.isAutoincrement && !column.isRowVersion) {
				setClauseColumns.add(column);
			}
			if(column.isRowIdentifier || column.isRowVersion) {
				whereClauseColumns.add(column);
			}
			if(!column.isAutoincrement && !column.isRowVersion) {
				valuesClauseColumns.add(column);
			}
		}
		if(selectClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		if(setClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		if(whereClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		if(valuesClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		
		StringBuilder sb = new StringBuilder();
		sb.append("MERGE INTO \"");
		sb.append(tableName);
		sb.append("\" AS Target \r\n");
		sb.append("USING ( \r\n");
		sb.append("  SELECT \r\n");
		for(int i = 0; i < selectClauseColumns.size(); i++) {
			Column column = selectClauseColumns.get(i);
			sb.append("    \"");
			sb.append(column.name);
			sb.append("\" = :");
			sb.append(column.name);
			if(i + 1 < selectClauseColumns.size()) {
				sb.append(", \r\n");
			}
		}
		sb.append("\r\n) AS Source \r\n");
		sb.append("ON (");
		for(int i = 0; i < whereClauseColumns.size(); i++) {
			Column column = whereClauseColumns.get(i);
			sb.append("Target.\"");
			sb.append(column.name);
			sb.append("\" = Source.\"");
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < whereClauseColumns.size()) {
				sb.append(" AND ");
			}
		}
		sb.append(") \r\n");
		sb.append("WHEN MATCHED THEN \r\n");
		sb.append("UPDATE SET \r\n");
		for(int i = 0; i < setClauseColumns.size(); i++) {
			Column column = setClauseColumns.get(i);
			sb.append("  \"");
			sb.append(column.name);
			sb.append("\" = Source.\"");
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < setClauseColumns.size()) {
				sb.append(", \r\n");
			}
		}
		sb.append(" \r\n");
		sb.append("WHEN NOT MATCHED THEN \r\n");
		sb.append("INSERT (");
		for(int i = 0; i < valuesClauseColumns.size(); i++) {
			Column column = valuesClauseColumns.get(i);
			sb.append('"');
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < valuesClauseColumns.size()) {
				sb.append(", ");
			}
		}
		sb.append(") VALUES ( \r\n");
		for(int i = 0; i < valuesClauseColumns.size(); i++) {
			Column column = valuesClauseColumns.get(i);
			sb.append("  Source.\"");
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < valuesClauseColumns.size()) {
				sb.append(", \r\n");
			}
		}
		sb.append(" \r\n");
		sb.append(");\r\n");
		
		return new NamedParameterStatement(sb.toString());
	}
	
	/** 指定したデータベース・メタデータ、クラスからテーブルを取得します。
	 * 
	 * <p>指定したクラス名を部分文字列として含むテーブルが返されます。
	 * 該当するテーブルがない場合はnullが返されます。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名(の部分文字列)
	 * @return テーブル名。該当がない場合は null
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	protected String getTableName(DatabaseMetaData md, Class<?> cls) throws SQLException {
		Set<String> tableNames = new HashSet<String>();
		ResultSet rs = null;
		try {
			rs = md.getTables(null, null, null, new String[]{"TABLE"});
			while(rs.next()) {
				String s = rs.getString("TABLE_NAME");
				tableNames.add(s.toLowerCase());
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
		}

		while(cls != null) {
			String s = cls.getSimpleName().toLowerCase();
			if(tableNames.contains(s)) {
				return s;
			}
			cls = cls.getSuperclass();
		}
		return null;
	}

	/** 指定されたデータベース・メタデータ、テーブル名からテーブル列情報のリストを取得します。
	 * 
	 * 
	 * @param md データベース・メタデータ
	 * @param tableName テーブル名
	 * @return 列情報のリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * 
	 * @see Column
	 */
	protected List<Column> getColumns(DatabaseMetaData md, String tableName, Set<String> fieldNames) throws SQLException {
		List<Column> columns = new ArrayList<Column>();
		
		Set<String> primaryKeys = getPrimaryKeys(md, tableName);
		Set<String> rowVersionColumns = getVersionColumns(md, tableName);
		Set<String> rowIdentifiers = null;
		
		if(fieldNames.containsAll(primaryKeys)) {
			rowIdentifiers = primaryKeys;
		} else {
			Map<String, Set<String>> uniqueIndexes = getUniqueIndexes(md, tableName);
			for(Map.Entry<String, Set<String>> entry : uniqueIndexes.entrySet()) {
				Set<String> uniqueKeys = entry.getValue();
				if(fieldNames.containsAll(uniqueKeys)) {
					rowIdentifiers = uniqueKeys;
					break;
				}
			}
		}
		
		ResultSet rs = null;
		try {
			rs = md.getColumns(null, null, tableName, null);
			while(rs.next()) {
				String columnName = rs.getString("COLUMN_NAME");
				String isNullable = rs.getString("IS_NULLABLE");
				String isAutoincrement = rs.getString("IS_AUTOINCREMENT");
				Column column = new Column();
				column.name = columnName;
				column.isPrimaryKey = primaryKeys.contains(columnName.toLowerCase());
				column.isRowIdentifier = (rowIdentifiers != null) && rowIdentifiers.contains(columnName.toLowerCase());
				column.isAutoincrement = !isAutoincrement.equals("NO");
				column.isRowVersion = rowVersionColumns.contains(columnName.toLowerCase());
				column.isNullable = !isNullable.equals("NO");
				
				columns.add(column);
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
		}
		return columns;
	}
	
	/** 指定されたデータベース・メタデータ、テーブル名から主キー列を取得します。
	 * 列名は小文字に変換されています。
	 * 
	 * @param md データベース・メタデータ
	 * @param tableName テーブル名
	 * @return 主キー列セット。null が返されることはありませんが、セット内の要素数が 0 になることはあります。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	protected Set<String> getPrimaryKeys(DatabaseMetaData md, String tableName) throws SQLException {
		Set<String> columns = new LinkedHashSet<String>();
		
		ResultSet rs = null;
		try {
			rs = md.getPrimaryKeys(null, null, tableName);
			while(rs.next()) {
				String s = rs.getString("COLUMN_NAME");
				columns.add(s.toLowerCase());
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
		}
		return columns;
	}
	
	/** 指定されたデータベース・メタデータ、テーブル名から一意キーのマップを取得します。
	 *  マップのkeyはユニーク制約名、valueは列名のリストです。
	 *  列名は小文字に変換されています。
	 *  
	 * @param md データベース・メタデータ
	 * @param tableName テーブル名
	 * @return 一意キーのマップ
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	protected Map<String, Set<String>> getUniqueIndexes(DatabaseMetaData md, String tableName) throws SQLException {
		Map<String, Set<String>> uniqueIndexes = new LinkedHashMap<String, Set<String>>();
		ResultSet rs = null;
		try {
			rs = md.getIndexInfo(null, null, tableName, true, false);
			while(rs.next()) {
				String indexName = rs.getString("INDEX_NAME");
				if(indexName == null) {
					continue;
				}
				String columnName = rs.getString("COLUMN_NAME");
				if(columnName == null) {
					continue;
				}
				boolean isUnique = (rs.getShort("NON_UNIQUE") == 0);
				if(rs.wasNull()) {
					continue;
				}
				if(!isUnique) {
					continue;
				}
				Set<String> columnNames = uniqueIndexes.get(indexName);
				if(columnNames == null) {
					columnNames = new LinkedHashSet<String>();
					uniqueIndexes.put(indexName, columnNames);
				}
				columnNames.add(columnName.toLowerCase());
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
		}
		return uniqueIndexes;
	}

	/** 指定されたデータベース・メタデータ、テーブル名から行バージョン列セットを取得します。
	 * 
	 * <p>行バージョン列は楽観性同時実行制御に使用することができます。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param tableName テーブル名
	 * @return 行バージョン列セット。null が返されることはありませんが、セット内の要素数が 0 になることはあります。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	protected Set<String> getVersionColumns(DatabaseMetaData md, String tableName) throws SQLException {
		Set<String> columns = new LinkedHashSet<String>();
		
		ResultSet rs = null;
		try {
			rs = md.getVersionColumns(null, null, tableName);
			while(rs.next()) {
				String s = rs.getString("COLUMN_NAME");
				columns.add(s.toLowerCase());
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
		}
		return columns;
	}
	
	/** 指定したクラスのフィールド配列を取得します。
	 * 
	 * <p>スタティック・フィールドとプライベート・フィールドは除外されます。</p>
	 * 
	 * @param cls フィールド取得対象のクラス
	 * @return 指定したクラスのフィールド配列。null が返されることはありませんが、配列の要素数が 0 になることはあります。
	 */
	protected Field[] getFields(Class<?> cls) {
		List<Field> fields = new ArrayList<Field>();
		for(Field field : cls.getDeclaredFields()) {
			int mod = field.getModifiers();
			if(!Modifier.isStatic(mod) && !Modifier.isPrivate(mod)) {
				fields.add(field);
			}
		}
		return fields.toArray(new Field[] {});
	}
	
	/** テーブルの列情報を表すクラスです。
	 * 
	 */
	protected static class Column {
		
		/** テーブル名 */
		public String name;
		
		/** 主キー列かどうかを表します。true の場合は主キー列です。*/
		public boolean isPrimaryKey;
		
		/** 行を特定するための列かどうかを表します。true の場合は行を特定するための列です。*/
		public boolean isRowIdentifier;
		
		/** 自動インクリメント列かどうかを表します。true の場合は自動インクリメント列です。*/
		public boolean isAutoincrement;
		
		/** 行バージョン列かどうかを表します。true の場合は行バージョン列です。*/
		public boolean isRowVersion;
		
		/** ヌル許容列かどうかを表します。true の場合はヌル許容列です。*/
		public boolean isNullable;
	}
}
