package net.osdn.util.sql;

import java.io.Closeable;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeoutException;

import javax.sql.RowSet;
import javax.sql.rowset.CachedRowSet;

import net.osdn.util.concurrent.Monitor;
import net.osdn.util.concurrent.Monitor.MultipleObjects;

/** トランザクションを提供します。
 * 
 */
public class Transaction implements Closeable {

	private DataSource.Instance datasource;
	private IsolationLevel isolationLevel;
	private Connection cn;
	private boolean isCommitted = false;
	
	/* package private */ Transaction(DataSource.Instance datasource, IsolationLevel isolationLevel) throws SQLException {
		this.datasource = datasource;
		this.isolationLevel = isolationLevel;
		this.cn = datasource.getConnection();
		
		if(isolationLevel != null) {
			if(cn.getTransactionIsolation() != isolationLevel.getValue()) {
				cn.setTransactionIsolation(isolationLevel.getValue());
			}
			
			if(isolationLevel != IsolationLevel.None) {
				if(this.cn.getAutoCommit()) {
					this.cn.setAutoCommit(false);
				}
			}
		}
	}
	
	@Override
	public void close() throws IOException {
		try {
			if(cn != null && !cn.isClosed()) {
				if(isolationLevel != null) {
					if(!isCommitted) {
						cn.rollback();
					}
				}
				cn.close();
			}
		} catch(SQLException e) {
			throw new IOException(e);
		}
	}
	
	public void commit() throws SQLException {
		if(isolationLevel == null) {
			throw new IllegalStateException();
		}

		cn.commit();
		isCommitted = true;
	}
	
	/** 指定されたSQL文を実行して、単一のRowSetオブジェクトを返します。
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。結果が0行の場合もnullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public RowSet executeQuery(String sql) throws SQLException {
		CachedRowSet rowSet = null;
		Statement st = null;
		ResultSet rs = null;
		try {
			st = cn.createStatement();
			rs = st.executeQuery(sql);
			rowSet = datasource.populate(rs);
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
		return rowSet;
	}
	
	/** 指定されたSQL文にパラメーターを設定して実行し、単一のRowSetオブジェクトを返します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。</p>
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。結果が0行の場合もnullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public RowSet executeQuery(String sql, Object... parameters) throws SQLException {
		CachedRowSet rowSet = null;
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(sql);
			if(parameters != null) {
				if(parameters instanceof NamedParameter[]) {
					for(NamedParameter parameter : (NamedParameter[])parameters) {
						parameter.applyTo(st);
					}
				} else {
					int i = 0;
					for(Object parameter : parameters) {
						st.setObject(++i, parameter);
					}
				}
			}
			rs = st.executeQuery();
			rowSet = datasource.populate(rs);
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
		return rowSet;
	}

	/** 指定されたSQL文にパラメーターを設定して実行し、単一のRowSetオブジェクトを返します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。</p>
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。結果が0行の場合もnullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public RowSet executeQuery(String sql, Collection<?> parameters) throws SQLException {
		return executeQuery(sql, parameters.toArray());
	}
	
	/** 指定された名前付きパラメーターステートメントを実行し、単一のRowSetオブジェクトを返します。
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * <p>返されるRowSetオブジェクトはメモリーにキャッシュされた非接続の行セットです。</p>
	 * 
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーによって作成されたデータを含むResultSetオブジェクト。結果が0行の場合もnullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public RowSet executeQuery(NamedParameterStatement statement) throws SQLException {
		RowSet rowSet = null;
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			rs = st.executeQuery();
			rowSet = datasource.populate(rs);
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
		return rowSet;
	}
	
	/** SQL文を実行します。
	 * 
	 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public int executeUpdate(String sql, Object... parameters) throws SQLException {
		String notifier = DataSource.findTableName(sql);
		return executeUpdate(notifier, sql, parameters);
	}
	
	/** SQL文を実行します。
	 * 
	 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public int executeUpdate(String sql, Collection<?> parameters) throws SQLException {
		return executeUpdate(sql, parameters.toArray());
	}
	
	/** SQL文を実行します。
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * <p>それはSQLデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さないSQL文でなければなりません。</p>
	 * 
	 * @param statement 名前付きパラメーターステートメント
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public int executeUpdate(NamedParameterStatement statement) throws SQLException {
		int rowCount;
		PreparedStatement st = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			rowCount = st.executeUpdate();
		} finally {
			if(st != null) {
				st.close();
			}
		}
		return rowCount;
	}
	
	/** SQL文を実行して待機スレッドを再開させます。
	 * 
	 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
	 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
	 * 
	 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public int executeUpdate(Object notifier, String sql, Object... parameters) throws SQLException {
		int rowCount;
		PreparedStatement st = null;
		try {
			st = cn.prepareStatement(sql);
			if(parameters != null) {
				if(parameters instanceof NamedParameter[]) {
					for(NamedParameter parameter : (NamedParameter[])parameters) {
						parameter.applyTo(st);
					}
				} else {
					int i = 0;
					for(Object parameter : parameters) {
						st.setObject(++i, parameter);
					}
				}
			}
			rowCount = st.executeUpdate();
			if(rowCount > 0 && notifier != null) {
				DataSource.getMonitor().notifyAll(notifier);
			}
		} finally {
			if(st != null) {
				st.close();
			}
		}
		return rowCount;
	}
	
	/** SQL文を実行して待機スレッドを再開させます。
	 * 
	 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
	 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
	 * 
	 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第2引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第2引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
	 * @param sql 結果セットを返さないSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public int executeUpdate(Object notifier, String sql, Collection<?> parameters) throws SQLException {
		return executeUpdate(notifier, sql, parameters.toArray());
	}
	
	/** SQL文を実行して待機スレッドを再開させます。
	 * 
	 * <p>SQL文を実行して影響を受けた行数が1行以上あった場合、このデータソースが保持する {@link Monitor} を使って
	 * notifierオブジェクトに通知を送ります。notifierオブジェクトを含むMultipleObjectsで待機しているスレッドがあれば再開されます。</p>
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * <p>SQL文はデータ操作言語(DML)文(INSERT文、UPDATE文、DELETE文など)であるか、DDL文のような何も返さない文でなければなりません。</p>
	 * 
	 * @param notifier 待機中のスレッドを再開させるためのオブジェクト
	 * @param statement 結果セットを返さないSQL文
	 * @return (1) SQLデータ操作言語(DML)文の場合は行数、(2) 何も返さないSQL文の場合は 0
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public int executeUpdate(Object notifier, NamedParameterStatement statement) throws SQLException {
		int rowCount;
		PreparedStatement st = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			rowCount = st.executeUpdate();
			if(rowCount > 0 && notifier != null) {
				DataSource.getMonitor().notifyAll(notifier);
			}
		} finally {
			if(st != null) {
				st.close();
			}
		}
		return rowCount;
	}

	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
	 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
	 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(String)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 */
	public RowSet executeQueryWaitForRows(long timeout, String sql, Object... parameters) throws SQLException, TimeoutException, InterruptedException {
		MultipleObjects waiters = null;
		String tableName = DataSource.findTableName(sql);
		if(tableName != null) {
			waiters = DataSource.newMultipleObjects(timeout, tableName);
		}
		return executeQueryWaitForRows(waiters, sql, parameters);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
	 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
	 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(String)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 */
	public RowSet executeQueryWaitForRows(long timeout, String sql, Collection<?> parameters) throws SQLException, TimeoutException, InterruptedException {
		return executeQueryWaitForRows(timeout, sql, parameters.toArray());
	}
	
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、暗黙的な待機オブジェクトを使ってこのスレッドを待機させます。
	 * 暗黙的な待機オブジェクトはSQL文に最初に出現するテーブル名を持つString型です。
	 * ほかのスレッドがこのテーブル名を指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * @param timeout ミリ秒単位の待機時間
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * 
	 * @see NamedParameterStatement
	 */
	public RowSet executeQueryWaitForRows(long timeout, NamedParameterStatement statement) throws SQLException, TimeoutException, InterruptedException {
		MultipleObjects waiters = null;
		String tableName = DataSource.findTableName(statement.getSql());
		if(tableName != null) {
			waiters = DataSource.newMultipleObjects(timeout, tableName);
		}
		return executeQueryWaitForRows(waiters, statement);
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
	 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 */
	public RowSet executeQueryWaitForRows(MultipleObjects waiters, String sql, Object... parameters) throws SQLException, TimeoutException, InterruptedException {
		CachedRowSet rowSet = null;
		PreparedStatement st = null;
		try {
			st = cn.prepareStatement(sql);
			if(parameters != null) {
				if(parameters instanceof NamedParameter[]) {
					for(NamedParameter parameter : (NamedParameter[])parameters) {
						parameter.applyTo(st);
					}
				} else {
					int i = 0;
					for(Object parameter : parameters) {
						st.setObject(++i, parameter);
					}
				}
			}
			
			do {
				ResultSet rs = null;
				try {
					rs = st.executeQuery();
					rowSet = DataSource.populate(rs);
				} finally {
					if(rs != null) {
						rs.close();
					}
				}
				if(rowSet.next()) {
					rowSet.beforeFirst();
					break;
				} else {
					rowSet = null;
				}
				
				if(waiters == null) {
					break;
				}
			} while(waiters.waitForMultipleObjects() != null);
			
		} finally {
			if(st != null) {
				st.close();
			}
		}
		
		if(rowSet == null) {
			throw new TimeoutException();
		}
		
		return rowSet;
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
	 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 */
	public RowSet executeQueryWaitForRows(MultipleObjects waiters, String sql, Collection<?> parameters) throws SQLException, TimeoutException, InterruptedException {
		return executeQueryWaitForRows(waiters, sql, parameters.toArray());
	}
	
	/** 1行以上の結果が得られるか、あらかじめ指定されている量の実時間が経過するまで、指定されたSQLを繰り返し実行します。
	 * 
	 * <p>SQL文を実行して返された結果行数が 0 である場合、指定された {@link MultipleObjects} waitersを使ってこのスレッドを待機させます。
	 * ほかのスレッドがwaitersに含まれるオブジェクトのいずれかを指定して、データソースが保持する {@link Monitor} のnotifyAll(Object)メソッドを呼び出すと、
	 * スレッドが再開され、再び、SQL文が実行されます。</p>
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * @param waiters 待機中のスレッドを再開させるためのオブジェクトを含むMultipleObjects
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーによって作成されたデータを含むRowSetオブジェクト。nullにはなりません。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws TimeoutException 指定された待機時間が経過した場合
	 * @throws InterruptedException 現在のスレッドが通知を待機する前または待機中に、いずれかのスレッドが現在のスレッドに割り込んだ場合。この例外がスローされると、現在のスレッドの「割り込みステータス」はクリアされます。
	 * 
	 * @see NamedParameterStatement
	 */
	public RowSet executeQueryWaitForRows(MultipleObjects waiters, NamedParameterStatement statement) throws SQLException, TimeoutException, InterruptedException {
		CachedRowSet rowSet = null;
		PreparedStatement st = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			do {
				ResultSet rs = null;
				try {
					rs = st.executeQuery();
					rowSet = DataSource.populate(rs);
				} finally {
					if(rs != null) {
						rs.close();
					}
				}
				if(rowSet.next()) {
					rowSet.beforeFirst();
					break;
				} else {
					rowSet = null;
				}
				
				if(waiters == null) {
					break;
				}
			} while(waiters.waitForMultipleObjects() != null);
		} finally {
			if(st != null) {
				st.close();
			}
		}
		
		if(rowSet == null) {
			throw new TimeoutException();
		}
		
		return rowSet;
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public <T> T executeScalar(String sql) throws SQLException {
		Object value = null;
		Statement st = null;
		ResultSet rs = null;
		try {
			st = cn.createStatement();
			rs = st.executeQuery(sql);
			if(rs.next()) {
				value = rs.getObject(1);
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
		
		@SuppressWarnings("unchecked")
		T result = (T)value;
		return result;
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public <T> T executeScalar(String sql, Object... parameters) throws SQLException {
		Object value = null;
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(sql);
			if(parameters != null) {
				if(parameters instanceof NamedParameter[]) {
					for(NamedParameter parameter : (NamedParameter[])parameters) {
						parameter.applyTo(st);
					}
				} else {
					int i = 0;
					for(Object parameter : parameters) {
						st.setObject(++i, parameter);
					}
				}
			}
			rs = st.executeQuery();
			if(rs.next()) {
				value = rs.getObject(1);
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
		
		@SuppressWarnings("unchecked")
		T result = (T)value;
		return result;
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * <p>このメソッドに指定するSQL文はパラメータのプレースホルダーを使うことができます。
	 * パラメーターのプレースホルダーは {@literal ?} で指定します。
	 * パラメーターのプレースホルダーには第3引数で指定したparametersが順番に割り当てられます。
	 * パラメーターが必要ない場合は第3引数を省略してこのメソッドを呼び出します。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public <T> T executeScalar(String sql, Collection<?> parameters) throws SQLException {
		return executeScalar(sql, parameters.toArray());
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行にある最初の列を返します。
	 * 
	 * <p>このメソッドには名前付きパラメーター機能を提供する{@link NamedParameterStatement}を指定します。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSetのgetObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param statement 名前付きパラメーターステートメント
	 * @return 指定されたクエリーが返す結果セットの最初の行にある最初の列。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * 
	 * @see NamedParameterStatement
	 */
	public <T> T executeScalar(NamedParameterStatement statement) throws SQLException {
		Object value = null;
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			rs = st.executeQuery();
			if(rs.next()) {
				value = rs.getObject(1);
			}
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
		
		@SuppressWarnings("unchecked")
		T result = (T)value;
		return result;
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行から指定された列名の値を返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>id列とname列を持つPersonテーブルに対してSELECT文を実行するには以下の方法があります。</p>
	 * 
	 * <pre><code>
	 * String name = DataSource.select("name", new Person() {
	 *     int id = 1;
	 * });
	 * </code></pre>
	 * 
	 * または
	 * 
	 * <pre><code>
	 * String name = DataSource.select("name", new Table("Person") {
	 *     int id = 1;
	 * });
	 * </code></pre>
	 * 
	 * <p>ただし、以下のようにPersonクラスのインスタンスを初期化した場合は上記と同じ結果になりません。</p>
	 * 
	 * <pre><code>
	 * Person person = new Person();
	 * person.id = 1;
	 * String name = DataSource.select("name", person);
	 * </code></pre>
	 * 
	 * <p>上記のケースではPersonクラスが持つ他のフィールドnameがWHERE句に追加されてしまうためです。</p>
	 * 
	 * @param <T> 返される列の値をキャストする型。列の値はResultSet#getObject(int)によって取得されて{@literal <T>}にキャストされます。
	 * @param columnName 列名
	 * @param obj SELECT文を構築するオブジェクト
	 * @return クエリーが返す結果セットの最初の行にある指定された列名の値。行がない場合は null。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalArgumentException 指定されたオブジェクトが null の場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public <T> T select(String columnName, Object obj) throws SQLException, IllegalArgumentException, IllegalAccessException {
		if(obj == null) {
			throw new IllegalArgumentException();
		}
		
		ResultSet rs = null;
		try {
			String tableName;
			if(obj instanceof Table) {
				tableName = ((Table)obj).getTableName();
			} else {
				tableName = ORMapper.getTableName(cn, obj.getClass());
			}
			rs = ORMapper.select(cn, tableName, obj);
			if(rs.next()) {
				Object value = rs.getObject(columnName);
				if(!rs.wasNull()) {
					@SuppressWarnings("unchecked")
					T result = (T)value;
					return result;
				}
			}
			return null;
		} finally {
			if(rs != null) {
				Statement st = rs.getStatement();
				rs.close();
				if(st != null) {
					st.close();
				}
			}
		}
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行をオブジェクトとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOオブジェクトです。
	 * オブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>id列とname列を持つPersonテーブルに対してSELECT文を実行するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * Person person = DataSource.select(new Person() {
	 *     int id = 1;
	 * });
	 * </code></pre>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param obj SELECT文を構築するオブジェクト
	 * @return クエリーが返す結果セットの最初の行をオブジェクトのクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Object obj) throws SQLException, ParseException, ReflectiveOperationException {
		Class<?> cls = obj.getClass();
		while(cls.isAnonymousClass()) {
			cls = cls.getSuperclass();
		}
		@SuppressWarnings("unchecked")
		Class<T> returnClass = (Class<T>)cls;
		return select(returnClass, obj);
	}

	/** 指定したクラスでSELECT文を構築し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * <p>このメソッドはレコードが1行しかないテーブルのデータを取得するのに適しています。</p>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Class<T> returnClass) throws SQLException, ParseException, ReflectiveOperationException {
		ResultSet rs = null;
		try {
			rs = ORMapper.select(cn, returnClass);
			if(rs.next()) {
				return new ORMapper(rs).get(returnClass);
			}
			return null;
		} finally {
			if(rs != null) {
				Statement st = rs.getStatement();
				rs.close();
				if(st != null) {
					st.close();
				}
			}
		}
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param obj SELECT文を構築するオブジェクト
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Class<T> returnClass, Object obj) throws SQLException, ParseException, ReflectiveOperationException {
		ResultSet rs = null;
		try {
			String tableName;
			if(obj instanceof Table) {
				tableName = ((Table)obj).getTableName();
			} else {
				tableName = ORMapper.getTableName(cn, returnClass);
			}
			rs = ORMapper.select(cn, tableName, obj);
			if(rs.next()) {
				return new ORMapper(rs).get(returnClass);
			}
			return null;
		} finally {
			if(rs != null) {
				Statement st = rs.getStatement();
				rs.close();
				if(st != null) {
					st.close();
				}
			}
		}
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Class<T> returnClass, String sql) throws SQLException, ParseException, ReflectiveOperationException {
		Statement st = null;
		ResultSet rs = null;
		try {
			st = cn.createStatement();
			rs = st.executeQuery(sql);
			if(rs.next()) {
				return new ORMapper(rs).get(returnClass);
			}
			return null;
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Class<T> returnClass, String sql, Object... parameters) throws SQLException, ParseException, ReflectiveOperationException {
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(sql);
			if(parameters != null) {
				if(parameters instanceof NamedParameter[]) {
					for(NamedParameter parameter : (NamedParameter[])parameters) {
						parameter.applyTo(st);
					}
				} else {
					int i = 0;
					for(Object parameter : parameters) {
						st.setObject(++i, parameter);
					}
				}
			}
			rs = st.executeQuery();
			if(rs.next()) {
				return new ORMapper(rs).get(returnClass);
			}
			return null;
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Class<T> returnClass, String sql, Collection<?> parameters) throws SQLException, ParseException, ReflectiveOperationException {
		return select(returnClass, sql, parameters.toArray());
	}
	
	/** 指定された名前付きパラメーターステートメントを実行し、そのクエリーが返す結果セットの最初の行を指定されたクラスのオブジェクトとして返します。
	 * 
	 * @param <T> 返されるオブジェクトの型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス
	 * @param statement 名前付きパラメーターステートメント
	 * @return クエリーが返す結果セットの最初の行を指定されたクラス型にマッピングしたオブジェクト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> T select(Class<T> returnClass, NamedParameterStatement statement) throws SQLException, ParseException, ReflectiveOperationException {
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			rs = st.executeQuery();
			if(rs.next()) {
				return new ORMapper(rs).get(returnClass);
			}
			return null;
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
	}

	/** 指定されたクラスでSELECT文を構築し、そのクエリーが返す結果セットをクラスのリストとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するクラスのクラス名をテーブル名とします。
	 * このメソッドに匿名クラスを渡すことはできません。
	 * このメソッドはテーブルのすべての行を返します。</p>
	 * 
	 * <p>Personテーブルのすべての行をList&lt;Person&gt;として取得するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * List&lt;Person&gt; persons = DataSource.selectAsList(Person.class);
	 * </code></pre>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass クエリーが返す結果セットをマッピングするクラス。クラス名がテーブル名と一致している必要があります。
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Class<T> returnClass) throws SQLException, ParseException, ReflectiveOperationException {
		return selectAsList(returnClass, (Object)null);
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットをオブジェクトのクラスのリストとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOオブジェクトです。
	 * オブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>Personテーブルのdeleted列の値が1の行をList&lt;Person&gt;として取得するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * List&lt;Person&gt; persons = DataSource.selectAsList(new Person() {
	 *     int deleted = 1;
	 * });
	 * </code></pre>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param obj SELECT文を構築するオブジェクト。このオブジェクトのクラス名がテーブル名と一致している必要があります。
	 * @return クエリーが返す結果セットの各行を指定されたオブジェクトのクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Object obj) throws SQLException, ParseException, ReflectiveOperationException {
		Class<?> cls = obj.getClass();
		while(cls.isAnonymousClass()) {
			cls = cls.getSuperclass();
		}
		@SuppressWarnings("unchecked")
		Class<T> returnClass = (Class<T>)cls;
		return selectAsList(returnClass, obj);
	}
	
	/** 指定されたオブジェクトでSELECT文を構築し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * <p>SELECT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはWHERE句に指定されます。</p>
	 * 
	 * <p>Personテーブルのdeleted列の値が1の行をList&lt;Person&gt;として取得するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * List&lt;Person&gt; persons = DataSource.selectAsList(Person.class, new Table("Person") {
	 *     int deleted = 1;
	 * });
	 * </code></pre>
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param obj SELECT文を構築するオブジェクト。
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Class<T> returnClass, Object obj) throws SQLException, ParseException, ReflectiveOperationException {
		ResultSet rs = null;
		try {
			String tableName;
			if(obj instanceof Table) {
				tableName = ((Table)obj).getTableName();
			} else {
				tableName = ORMapper.getTableName(cn, returnClass);
			}
			rs = ORMapper.select(cn, tableName, obj);
			return new ORMapper(rs).getList(returnClass);
		} finally {
			if(rs != null) {
				Statement st = rs.getStatement();
				rs.close();
				if(st != null) {
					st.close();
				}
			}
		}
	}
	
	/** 指定されたSQL文を実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param sql データベースに送られるSQL文。通常は静的なSELECT文です。
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Class<T> returnClass, String sql) throws SQLException, ParseException, ReflectiveOperationException {
		Statement st = null;
		ResultSet rs = null;
		try {
			st = cn.createStatement();
			rs = st.executeQuery(sql);
			return new ORMapper(rs).getList(returnClass);
		} finally {
			if(st != null) {
				st.close();
			}
			if(rs != null) {
				rs.close();
			}
		}
	}
	
	/** 指定されたSQL文にパラメーターを設定して実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Class<T> returnClass, String sql, Object... parameters) throws SQLException, ParseException, ReflectiveOperationException {
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(sql);
			if(parameters != null) {
				if(parameters instanceof NamedParameter[]) {
					for(NamedParameter parameter : (NamedParameter[])parameters) {
						parameter.applyTo(st);
					}
				} else {
					int i = 0;
					for(Object parameter : parameters) {
						st.setObject(++i, parameter);
					}
				}
			}
			rs = st.executeQuery();
			return new ORMapper(rs).getList(returnClass);
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
	}
	
	/** 指定されたSQL文にパラメーターを設定して実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param sql プレースホルダー {@literal ?} を含むSQL文
	 * @param parameters プレースホルダーに設定されるパラメーター
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Class<T> returnClass, String sql, Collection<?> parameters) throws SQLException, ParseException, ReflectiveOperationException {
		return selectAsList(returnClass, sql, parameters.toArray());
	}
	
	/** 指定された名前付きパラメーターステートメントを実行し、そのクエリーが返す結果セットを指定されたクラスのリストとして返します。
	 * 
	 * @param <T> 返されるリストの要素の型。
	 * @param returnClass 返されるリストの要素の型
	 * @param statement 名前付きパラメーターステートメント
	 * @return クエリーが返す結果セットの各行を指定されたクラス型にマッピングしたリスト
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws ParseException 列の値をフィールドの型に変換できない場合
	 * @throws ReflectiveOperationException リフレクション操作例外。{@literal <T>}には引数を持たないコンストラクタが必要です。
	 */
	public <T> List<T> selectAsList(Class<T> returnClass, NamedParameterStatement statement) throws SQLException, ParseException, ReflectiveOperationException {
		PreparedStatement st = null;
		ResultSet rs = null;
		try {
			st = cn.prepareStatement(statement.getSql());
			for(NamedParameter parameter : statement.getParameters()) {
				parameter.applyTo(st);
			}
			rs = st.executeQuery();
			return new ORMapper(rs).getList(returnClass);
		} finally {
			if(rs != null) {
				rs.close();
			}
			if(st != null) {
				st.close();
			}
		}
	}
	
	/** 指定されたオブジェクトでINSERT文を構築し、データソースへ行の追加を試みます。
	 * 
	 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
	 * 
	 * <p>列 id, name, age を持つPersonテーブルに行を追加するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * DataSource.insert(new Person() {
	 *     int    id   = 4;
	 *     String name = "四郎";
	 *     int    age  = 19; 
	 * });
	 * </code></pre>
	 * 
	 * または
	 * 
	 * <pre><code>
	 * DataSource.insert(new Table("Person") {
	 *     int    id   = 4;
	 *     String name = "四郎";
	 *     int    age  = 19; 
	 * });
	 * </code></pre>
	 * 
	 * または
	 * 
	 * <pre><code>
	 * Person person = new Person();
	 * person.id   = 4;
	 * person.name = "四郎";
	 * person.age  = 19;
	 * DataSource.insert(person);
	 * </code></pre>
	 * 
	 * <p>3つ目のコード例ではPersonクラスが列名と一致する他のフィールドを持っている場合に、
	 * 列名に 0 や null が指定されることに注意してください。</p>
	 * 
	 * @param obj INSERT文を構築するオブジェクト
	 * @return INSERT文の実行により影響を受けた行数。つまり行の追加が成功した場合は 1 です。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int insert(Object obj) throws SQLException, IllegalAccessException {
		int[] r = insert(new Object[] { obj });
		return r[0];
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してINSERT文を構築し、データソースへ行の追加を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけINSERT文が実行されますが、これらのINSERT文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
	 * 
	 * @param objects INSERT文を構築するオブジェクト(複数)
	 * @return INSERT文の実行により影響を受けた行数。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] insert(Object... objects) throws SQLException, IllegalAccessException {
		int[] results = ORMapper.insert(cn, objects);
		return results;
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してINSERT文を構築し、データソースへ行の追加を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけINSERT文が実行されますが、これらのINSERT文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>INSERT文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはVALUES句に指定されます。</p>
	 * 
	 * @param objects INSERT文を構築するオブジェクト(複数)
	 * @return INSERT文の実行により影響を受けた行数。
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] insert(Collection<?> objects) throws SQLException, IllegalAccessException {
		return insert(objects.toArray());
	}
	
	/** 指定されたオブジェクトでUPDATE文を構築し、データソースへ行の更新を試みます。
	 * 
	 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
	 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>主キー列 id と通常の列 name を持つPersonテーブルを更新するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * DataSource.update(new Person() {
	 *     int    id   = 4;
	 *     String name = "史郎";
	 * });
	 * </code></pre>
	 * 
	 * <p>上記コード例では主キー列 id の値が 4 となっている行の name 列が "史郎" に更新されます。</p>
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param obj UPDATE文を構築するオブジェクト
	 * @return UPDATE文の実行により影響を受けた行数。つまり行の更新が成功した場合は 1 です。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int update(Object obj) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		int[] r = update(new Object[] { obj });
		return r[0];
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してUPDATE文を構築し、データソースへ行の更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけUPDATE文が実行されますが、これらのUPDATE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
	 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects UPDATE文を構築するオブジェクト(複数)
	 * @return UPDATE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] update(Object... objects) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		int[] results = ORMapper.update(cn, objects);
		return results;
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してUPDATE文を構築し、データソースへ行の更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけUPDATE文が実行されますが、これらのUPDATE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>UPDATE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * テーブルの列名と一致するオブジェクトのフィールドはSET句に指定されます。
	 * このメソッドは単一行を更新することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたUPDATE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、UPDATE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects UPDATE文を構築するオブジェクト(複数)
	 * @return UPDATE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] update(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		return update(objects.toArray());
	}
	
	/** 指定されたオブジェクトでDELETE文を構築し、データソースから行の削除を試みます。
	 * 
	 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param obj DELETE文を構築するオブジェクト
	 * @return DELETE文の実行により影響を受けた行数。つまり行の削除が成功した場合は 1 です。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int delete(Object obj) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		int[] r = delete(new Object[] { obj });
		return r[0];
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してDELETE文を構築し、データソースから行の削除を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけDELETE文が実行されますが、これらのDELETE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects DELETE文を構築するオブジェクト(複数)
	 * @return DELETE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] delete(Object... objects) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		int[] results = ORMapper.delete(cn, objects);
		return results;
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してDELETE文を構築し、データソースから行の削除を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけDELETE文が実行されますが、これらのDELETE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>DELETE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を削除することを目的としています。行を一意に特定するためのフィールドが指定されている必要があります。
	 * WHERE句を構成する(行を一意に特定するための)列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたDELETE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、DELETE文によって影響を受ける行数は0になります。
	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
	 * 
	 * @param objects DELETE文を構築するオブジェクト(複数)
	 * @return DELETE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] delete(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		return delete(objects.toArray());
	}
	
	/** 指定されたオブジェクトでMERGE文を構築し、データソースへ行の追加または更新を試みます。
	 * 
	 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を更新または追加することを目的としています。
	 * 行を一意に特定するためのフィールドが指定されている必要があります。
	 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>主キー列 id と通常の列 name を持つPersonテーブルを追加または更新するには以下のようにします。</p>
	 * 
	 * <pre><code>
	 * DataSource.merge(new Person() {
	 *     int    id   = 5;
	 *     String name = "五郎";
	 * });
	 * </code></pre>
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
 	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
 	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
 	 * 
	 * @param obj MERGE文を構築するオブジェクト
	 * @return MERGE文の実行により影響を受けた行数。つまり行の追加または更新が成功した場合は 1 です。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int merge(Object obj) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		int[] r = merge(new Object[] { obj });
		return r[0];
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してMERGE文を構築し、データソースへ行の追加または更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけMERGE文が実行されますが、これらのMERGE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を更新または追加することを目的としています。
	 * 行を一意に特定するためのフィールドが指定されている必要があります。
	 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
 	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
 	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
 	 * 
	 * @param objects MERGE文を構築するオブジェクト(複数)
	 * @return MERGE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] merge(Object... objects) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		int[] results = ORMapper.merge(cn, objects);
		return results;
	}
	
	/** 指定された複数のオブジェクトそれぞれに対してMERGE文を構築し、データソースへ行の追加または更新を試みます。
	 * 
	 * <p>それぞれのオブジェクトのクラスが異なっていても構いません。</p>
	 * 
	 * <p>指定されたオブジェクトの数だけMERGE文が実行されますが、これらのMERGE文は1つのトランザクションとして実行されます。</p>
	 * 
	 * <p>MERGE文を構築するために指定するオブジェクトはPOJOまたは{@link Table}オブジェクトです。
	 * オブジェクトが{@link Table}である場合、{@link Table}のコンストラクタで指定された文字列をテーブル名とします。
	 * オブジェクトが{@link Table}以外のPOJOである場合はオブジェクトのクラス名をテーブル名とします。
	 * オブジェクトのクラスが匿名クラスである場合は、スーパークラスのクラス名をテーブル名とします。
	 * このメソッドは単一行を更新または追加することを目的としています。
	 * 行を一意に特定するためのフィールドが指定されている必要があります。
	 * 行を一意に特定するための列はテーブルのスキーマに基づいて自動的に決定されます。</p>
	 * 
	 * <p>構築されたMERGE文が行バージョン列を含む場合、楽観的同時実行制御をおこないます。
 	 * オブジェクトの取得以降に行バージョン列の値が更新されていた場合、MERGE文によって影響を受ける行数は0になります。
 	 * これを検出した場合、楽観的同時実行制御違反として{@link OptimisticConcurrencyException}をスローします。</p>
 	 * 
	 * @param objects MERGE文を構築するオブジェクト(複数)
	 * @return MERGE文の実行により影響を受けた行数。
	 * @throws OptimisticConcurrencyException 楽観的同時実行制御例外が発生した場合
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 * @throws IllegalAccessException オブジェクトのフィールドにアクセスできない場合
	 */
	public int[] merge(Collection<?> objects) throws OptimisticConcurrencyException, SQLException, IllegalAccessException {
		return merge(objects.toArray());
	}
}
