package net.osdn.util.sql;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.ConcurrentLinkedQueue;

import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import javax.sql.ConnectionPoolDataSource;
import javax.sql.PooledConnection;

/** javax.sql.ConnectionPoolDataSourceインターフェースを実装したデータソースで接続プールを構成するクラスです.
 * 
 */
public class ConnectionPool implements ConnectionEventListener {
	
	private ConnectionPoolDataSource datasource;
	private String username;
	private String password;
	private ConcurrentLinkedQueue<PooledConnection> pool;
	
	
	/** 指定したデータソースで接続プールを構成します。
	 * 
	 * @param datasource javax.sql.ConnectionPoolDataSourceインターフェースを実装しているデータソース
	 */
	public ConnectionPool(ConnectionPoolDataSource datasource) {
		this(datasource, null, null);
	}
	
	/** 指定したデータソースで接続プールを構成します。
	 * データベースへの接続には username と password を使用します。
	 * 
	 * 
	 * @param datasource javax.sql.ConnectionPoolDataSourceインターフェースを実装しているデータソース
	 * @param username データベースへの接続に使用するユーザー名
	 * @param password データベースへの接続に使用するユーザーのパスワード
	 */
	public ConnectionPool(ConnectionPoolDataSource datasource, String username, String password) {
		this.datasource = datasource;
		this.username = username;
		this.password = password;
		this.pool = new ConcurrentLinkedQueue<PooledConnection>();
	}
	
	/** データソースへの接続を返します.
	 * 
	 * <p>このメソッドはプールされている接続があればプールされている接続を返します。
	 * プールされている接続がなければデータベースへの物理接続の確立を試みて、確立された新しい接続を返します。
	 * このメソッドによって返された接続のcloseメソッドを呼び出してもデータソースへの物理的な接続は閉じられることはなく、
	 * 接続はプールに戻されます。</p>
	 * 
	 * @return データソースへの接続
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public Connection getConnection() throws SQLException {
		if(datasource == null) {
			throw new IllegalStateException();
		}
		PooledConnection pooledConnection = pool.poll();
		if(pooledConnection == null) {
			if(username != null || password != null) {
				pooledConnection = datasource.getPooledConnection();
			} else {
				pooledConnection = datasource.getPooledConnection(username, password);
			}
			pooledConnection.addConnectionEventListener(this);
		}
		return pooledConnection.getConnection();
	}
	
	/** プールされているすべての接続を閉じます.
	 * 
	 * <p>このメソッドを呼び出してもgetConnection()メソッドによって払い出されているアクティブな接続が強制的に閉じられることはありません。
	 * このメソッドの呼び出し以降、 getConnection()メソッドによって払い出されているアクティブな接続のclose()メソッドを呼び出すと、
	 * 接続はプールに戻されるのではなく、データソースとの物理的な接続が閉じられます。</p>
	 * 
	 */
	public void close() {
		datasource = null;
		
		PooledConnection pooledConnection;
		while((pooledConnection = pool.poll()) != null) {
			close(pooledConnection);
		}
	}
	
	private void close(PooledConnection pooledConnection) {
		pooledConnection.removeConnectionEventListener(this);
		try {
			pooledConnection.close();
		} catch(SQLException e) {
			e.printStackTrace();
		}
	}

	@Override
	public void connectionClosed(ConnectionEvent event) {
		PooledConnection pooledConnection = (PooledConnection)event.getSource();
		if(datasource == null) {
			close(pooledConnection);
		} else {
			boolean isSucceeded = pool.offer(pooledConnection);
			if(!isSucceeded) {
				close(pooledConnection);
			}
		}
	}

	@Override
	public void connectionErrorOccurred(ConnectionEvent event) {
		PooledConnection pooledConnection = (PooledConnection)event.getSource();
		pooledConnection.removeConnectionEventListener(this);
	}
}
