package net.osdn.util.sql;

import java.lang.reflect.Field;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

/** ResultSetの各行をオブジェクトに変換していくイテレーターです。
 * 
 * @param <E> このイテレーターで返される要素の型
 */
public class ObjectIterator<E> implements Iterator<E> {

	private Class<E> type;
	private boolean isScalar = false;
	private ResultSet rs;

	private String[] columnNames;
	private int columnCount;
	private Map<String, Field> fields;
	
	private Boolean hasNext;
	
	/** 指定したクラスとResultSetでイテレーターを作成します。
	 * 
	 * @param type ResultSetの各行を変換する型
	 * @param rs ResultSet
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	public ObjectIterator(Class<E> type, ResultSet rs) throws SQLException {
		if(ORMapper.isScalar(type)) {
			if(rs.getMetaData().getColumnCount() != 1) {
				throw new IllegalArgumentException("To get as a list of scalar values, it must be a result set with a single column.");
			}
			this.isScalar = true;
		}
		this.type = type;
		this.rs = rs;
		this.columnNames = ORMapper.getColumnNames(rs.getMetaData());
		this.columnCount = rs.getMetaData().getColumnCount();
		this.fields = ORMapper.getFields(type);
	}
	
	/** 反復処理でさらに要素がある場合にtrueを返します。
	 * 
	 * @return 反復処理でさらに要素がある場合はtrue
	 */
	@Override
	public boolean hasNext() {
		if(hasNext == null) {
			try {
				hasNext = rs.next();
			} catch (SQLException e) {
				throw new RuntimeException(e);
			}
		}
		return hasNext;
	}

	/** 反復処理で次の要素を返します。
	 * 
	 * <p>このメソッドはResultSetから現在の行を読み取ってオブジェクトに変換します。
	 * この処理は内部でSQLException、ParseException、IllegalArgumentException、IllegalAccessException、InstantiationExceptionをスローする可能性があります。
	 * 内部でこれらの例外がスローされた場合、これらの例外がRuntimeExceptionでラップされて再スローされます。</p>
	 * 
	 * @throws NoSuchElementException 反復処理で要素がない場合
	 */
	@Override
	public E next() throws NoSuchElementException {
		if(!hasNext()) {
			throw new NoSuchElementException();
		}
		hasNext = null;
		
		try {
			if(isScalar) {
				@SuppressWarnings("unchecked")
				E obj = (E)ORMapper.getValue(rs, 1, type);
				return obj;
			} else {
				E obj = ORMapper.createInstance(type);
				for(int i = 1; i <= columnCount; i++) {
					Field field = fields.get(columnNames[i]);
					if(field == null) {
						continue;
					}
					Object value = ORMapper.getValue(rs, i, field);
					field.set(obj, value);
				}
				return obj;
			}
		} catch (SQLException e) {
			throw new RuntimeException(e);
		} catch (ParseException e) {
			throw new RuntimeException(e);
		} catch (IllegalArgumentException e) {
			throw new RuntimeException(e);
		} catch (ReflectiveOperationException e) {
			throw new RuntimeException(e);
		}
	}
}
