//	VirtualDub - Video processing and capture application
//	Copyright (C) 1998-2001 Avery Lee
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
#include "VirtualDub.h"

#include <io.h>
#include <crtdbg.h>

#include <windows.h> 
#include <commctrl.h>
#include <commdlg.h>
#include <shellapi.h>
#include <shlobj.h>
#include <time.h> 

#include "resource.h"

#include "VideoSequenceCompressor.h"
#include "gui.h"
#include "job.h" 
#include "command.h"
#include "dub.h"
#include "script.h" 
#include <winsock2.h>
#include "oshelper.h"
#include "prefs.h"
#include "crash.h"
#include "audio.h"
#include <MMREG.H>
#include "misc.h"

//==============================//
//=====  MODIFICATION OGM  =====//
//=====    -= Cyrius =-    =====//
//BEGIN ========================//
#include "OGM/OGMDub.h"
extern ogm_stream *ogm_streams;
extern comment_list *video_comments;
extern comment_list *audio_comments;
extern comment_list *audio2_comments;
// 13/11/2002, Cyrius
extern bool nandub_compatibility;
extern AudioSource *inputAudioAC3;
extern AudioSource *inputAudio2AC3;
//END ==========================//

//////////////////////////////////////////////////////////////////////////
class VDJob : public ListNode {
private:
	static List job_list;
	static long job_count;
	static bool fModified;
	static bool fSlaveMode;
	static bool fMasterMode;

public:
	static bool fRunAllStop;
	static bool fRunInProgress;
	static long job_number;

	enum {
		WAITING		= 0,
		INPROGRESS	= 1,
		DONE		= 2,
		POSTPONED	= 3,
		ABORTED		= 4,
		ERR			= 5,
	};

	char szName[64];
	char szInputFile[MAX_PATH];
	char szOutputFile[MAX_PATH];
	char *szInputFileTitle;
	char *szOutputFileTitle;
	char szError[256];
	int iState;
	SYSTEMTIME stStart, stEnd;
	char *script;

	/////

	VDJob();
	~VDJob();

	void Add(bool force_no_update = false);
	void Delete(bool force_no_update = false);
	void Refresh();


	void Run();


	static VDJob *ListGet(int index);
	static int ListFind(VDJob *vdj_find);
	static long ListSize();
	static void ListClear(bool force_no_update = false);
	static void ListLoad(char *lpszName = NULL);

	static bool IsModified() {
		return fModified;
	}

	static bool IsRunInProgress() {
		return fRunInProgress;
	}

	static bool IsRunAllStop() {
		return fRunAllStop;
	}

	static bool IsSlaveMode() {
		return fSlaveMode;
	}

	static bool IsMasterMode() {
		return fMasterMode;
	}

	static bool IsNetworkJob() {
		return fMasterMode || fSlaveMode;
	}

	static void SetModified();
	static void SetSlaveMode(bool Mode);
	static void SetMasterMode(bool Mode);

	static void Flush(char *lpfn =NULL);
	static void RunAll();
	static void RunAllStop();
};
///////////////////////////////////////////////////////////////////////////
extern HWND g_hWnd;
extern HINSTANCE g_hInst;
extern FilterFunctions g_filterFuncs;
extern char g_szInputAVIFile[];
extern char g_szInputWAVFile[];
extern char g_szInputCBRMP3File[];
extern char g_szInputMP3File[];
extern char g_szInputAC3File[];
extern char g_szInputOGGFile[];
extern char g_szInput2WAVFile[];
extern char g_szInput2CBRMP3File[];
extern char g_szInput2MP3File[];
extern char g_szInput2AC3File[];
extern char g_szInput2OGGFile[];
extern char g_szInputAVIFileTitle[];
extern InputFileOptions *g_pInputOpts;

HWND g_hwndJobs;

struct JobsRegConfig {
	int	PartLength;
	bool Shutdown;
	bool ShutdownClients;
	int	Priority;
	bool RestartJobs;
	signed char Loglevel;
	bool ProcessAudio;
	bool ScanForSilence;
	bool SendToTray;
} g_JobsDlgConfig;

struct HostsRegConfig {
	bool NeverShutdown;
} g_HostsConfig;

bool g_fJobMode;
bool g_fJobAborted;
int g_iLocalProgress;
int g_FramesPerSec=25;

HMENU	HostsContextMenu;
int		HostsContextMenuItemPos;

#define BLOCKFRAMECOUNT			(g_FramesPerSec*g_JobsDlgConfig.PartLength)
#define	MAXBLOCKFRAMECOUNT		(BLOCKFRAMECOUNT*2)

#define SCRIPTLEN	32768
#define LINELEN		8192


///////////////////////////////////////////////////////////////////////////
static BOOL CALLBACK JobCtlDlgProc(HWND hdlg, UINT uiMsg, WPARAM wParam, LPARAM lParam);
int NetworkRunScript(char *script,char *error=NULL);
void UpdateProgressInformation();
void SetJobWindowButtons(int Mode=JWS_DETECT);
void EnableMenuEntries();
void SetMenuEntries(bool Save=false);
void Job_LoadHostConfig(int Host);
void Job_SaveHostConfig(int Host);
///////////////////////////////////////////////////////////////////////////

int __sign(int s) {
	if (s>0) return 1;
	if (s<0) return -1;
	return 0;
}

#define __max3(a,b,c) \
	__max(a, __max(b, c))

////////////////////////////////////////////////////////////////////////////
void strUnCify(char *string) {
	char *s,*t;
	int c;
	s=t=string;

	while(*s) {
		if (*s=='\\') {
			++s;
			switch (*s) {
			case 'x': {
				sscanf(s,"x%02x",&c);
				s+=3;
				*t++=(char)(c&0xff);
					  } break;
				default:*t++=*s++;
			}
		}
		else *t++=*s++;
	}
	*t=0;
}

///////////////////////////////////////////////////////////////////////////

class JobScriptOutputBlock : public ListNode {
public:
	long size;
	long ptr;
	char data[];
};

class JobScriptOutput {
private:
	List listScript;
	JobScriptOutputBlock *jsob;
	long total;
public:
	JobScriptOutput();
	~JobScriptOutput();

	void clear();
	void write(const char *s, long l);
	void adds(const char *s);
	void addf(const char *fmt, ...);
	char *getscript();
};

///////

JobScriptOutput::JobScriptOutput() {
	clear();
}

JobScriptOutput::~JobScriptOutput() {
	clear();
}

void JobScriptOutput::clear() {
	JobScriptOutputBlock *jsob;

	while(jsob = (JobScriptOutputBlock *)listScript.RemoveHead())
		freemem(jsob);

	total = 0;
	this->jsob = NULL;
}

void JobScriptOutput::write(const char *s, long l) {
	long to_copy;

	total += l;

	while(l) {
		if (jsob && jsob->ptr<jsob->size) {
			to_copy = jsob->size-jsob->ptr;
			if (to_copy > l) to_copy = l;

			memcpy(jsob->data + jsob->ptr, s, to_copy);
			jsob->ptr += to_copy;
			l -= to_copy;
			s += to_copy;
		} else {
			jsob = (JobScriptOutputBlock *)allocmem(16384);

			if (!jsob) throw MyMemoryError();

			jsob->size = 16384 - sizeof(JobScriptOutputBlock);
			jsob->ptr = 0;
			listScript.AddTail(jsob);
		}
	}
}

void JobScriptOutput::adds(const char *s) {
	long l = strlen(s);

	write(s, l);
	write("\n",1);
//	_RPT1(0,">%s\n",s);
}

void JobScriptOutput::addf(const char *fmt, ...) {
	char buf[16384];
	va_list val;
	long l;

	va_start(val, fmt);
	_vsnprintf(buf, sizeof buf, fmt, val);
	va_end(val);
	buf[sizeof buf-1]=0;

	l = strlen(buf);
	adds(buf);
}

char *JobScriptOutput::getscript() {
	char *mem = (char *)allocmem(total+1), *t=mem;
	JobScriptOutputBlock *jsobptr;
	if (!mem) throw MyMemoryError();

	jsobptr = (JobScriptOutputBlock *)listScript.head.prev;

	while(jsobptr->prev) {
		memcpy(t, jsobptr->data, jsobptr->ptr);
		t += jsobptr->ptr;

		jsobptr = (JobScriptOutputBlock *)jsobptr->prev;
	}
	*t = 0;

	return mem;
}

//////////////////////////////////////////////////////////////////////////
bool JobIsNetworkMode() {
	return VDJob::IsNetworkJob();
}


List VDJob::job_list;
long VDJob::job_count;
long VDJob::job_number=1;
bool VDJob::fModified = false;
bool VDJob::fRunInProgress = false;
bool VDJob::fRunAllStop;
bool VDJob::fSlaveMode = false;
bool VDJob::fMasterMode = false;

VDJob::VDJob() {
	szName[0]=0;
	szInputFile[0]=0;
	szOutputFile[0]=0;
	iState = VDJob::WAITING;
	stStart.wYear = stEnd.wYear = 0;
	script = NULL;
}

VDJob::~VDJob() {
	delete script;
}

void VDJob::Add(bool force_no_update) {
	char c, *s;

	s = szInputFileTitle = szInputFile;
	while(c=*s++) if (c=='\\' || c==':') szInputFileTitle=s;
	
	s = szOutputFileTitle = szOutputFile;
	while(c=*s++) if (c=='\\' || c==':') szOutputFileTitle=s;

	job_list.AddHead(this);
	++job_count;

	if (g_hwndJobs) {
		LVITEM li;

		li.mask		= LVIF_TEXT;
		li.iSubItem	= 0;
		li.iItem	= job_count-1;
		li.pszText	= LPSTR_TEXTCALLBACK;

		ListView_InsertItem(GetDlgItem(g_hwndJobs, IDC_JOBS), &li);
	}

	if (!force_no_update) SetModified();
}

void VDJob::Delete(bool force_no_update) {
	int index = ListFind(this);

	if (index>=0 && g_hwndJobs)
		ListView_DeleteItem(GetDlgItem(g_hwndJobs, IDC_JOBS), index);

	ListNode::Remove();
	--job_count;
	
	if (!force_no_update) SetModified();
}

void VDJob::Refresh() {
	int index = ListFind(this);
//	bool fSelected;

	if (index>=0 && g_hwndJobs) {
		HWND hwndItem = GetDlgItem(g_hwndJobs, IDC_JOBS);

		ListView_Update(hwndItem, index);
	}
}

void VDJob::Run() {
	iState = INPROGRESS;
	GetLocalTime(&stStart);
	memset(&stEnd, 0, sizeof(SYSTEMTIME));
	Refresh();
	Flush();

	strcpy(g_szInputAVIFile, szInputFile);
	strcpy(g_szInputAVIFileTitle, szInputFileTitle);

	iState=NetworkRunScript(script,szError);

	SetJobWindowButtons();
	GetLocalTime(&stEnd);
	Refresh();
	Flush();
}

////////

VDJob *VDJob::ListGet(int index) {
	VDJob *vdj = (VDJob *)job_list.tail.next, *vdj_next;

	while((vdj_next = (VDJob *)vdj->next) && index--)
		vdj = vdj_next;

	if (!vdj_next) return NULL;

	return vdj;
}

int VDJob::ListFind(VDJob *vdj_find) {
	VDJob *vdj = (VDJob *)job_list.tail.next, *vdj_next;
	int index=0;

	while((vdj_next = (VDJob *)vdj->next) && vdj != vdj_find) {
		vdj = vdj_next;
		++index;
	}

	if (vdj == vdj_find) return index;

	return -1;
}

long VDJob::ListSize() {
	return job_count;
}

// VDJob::ListClear()
//
// Clears all jobs from the list.

void VDJob::ListClear(bool force_no_update) {
	VDJob *vdj;

	while((vdj = (VDJob *)job_list.tail.next)->next) {
		vdj->Delete(true);
		delete vdj;
	}

	if (!force_no_update) SetModified();
}

// VDJob::ListLoad()
//
// Loads the list from a file.

static char *findcmdline(char *s) {
	while(isspace(*s)) ++s;

	if (s[0] != '/' || s[1] != '/')
		return NULL;

	s+=2;

	while(isspace(*s)) ++s;
	if (*s++ != '$') return NULL;

	return s;
}

static void strgetarg(char *buf, long bufsiz, const char *s) {
	const char *t = s;
	long l;

	if (*t == '"') {
		s = ++t;
		while(*s && *s!='"') ++s;
	} else
		while(*s && !isspace(*s)) ++s;

	l = s-t;
	if (l > bufsiz-1)
		l = bufsiz-1;

	memcpy(buf, t, l);
	buf[l]=0;
}

void VDJob::ListLoad(char *lpszName) {
	FILE *f = NULL;
	char szName[MAX_PATH], szVDPath[MAX_PATH], *lpFilePart;
	char linebuf[LINELEN];
	VDJob *job = NULL;

	// 13/11/2002, Cyrius : changed to VirtualDubMod.jobs
	// Try to create VirtualDubMod.jobs in the same directory as VirtualDub.

	if (!lpszName) {
		lpszName = szName;

		if (!GetModuleFileName(NULL, szVDPath, sizeof szVDPath))
			return;

		if (!GetFullPathName(szVDPath, sizeof szName, szName, &lpFilePart))
			return;

		// 19/12/2002, Cyrius : use the .exe filename as basis for the jobs file
		// This may be used as a workaround for current version of GKnot which
		// uses VirtualDub.jobs
		char *lpExt = lpFilePart + strlen(lpFilePart);
		while( (lpExt >= lpFilePart) && (*lpExt != '.'))
			lpExt--;
		if(*lpExt != '.')
			lpExt = lpFilePart + strlen(lpFilePart);
		strcpy(lpExt, ".jobs");
	}
	try {
		BOOL script_capture = false;
		JobScriptOutput jso;

		f = fopen(lpszName, "r");
		if (!f) return;

		ListClear(true);

		while(fgets(linebuf, sizeof linebuf, f)) {
			char *s;

			// kill the ending newline

			if (s = strchr(linebuf,'\n'))
				*s = 0;

			// scan for a command

			if (s = findcmdline(linebuf)) {
				char *t = s;

				while(isalpha(*t) || *t=='_') ++t;

				if (*t) *t++=0;
				while(isspace(*t)) ++t;

				if (!stricmp(s, "job")) {
					if (!(job = new VDJob)) throw "out of memory";
					job->szError[0]=0;

					strgetarg(job->szName, sizeof job->szName, t);

				} else if (!stricmp(s, "input")) {

					strgetarg(job->szInputFile, sizeof job->szInputFile, t);

				} else if (!stricmp(s, "output")) {

					strgetarg(job->szOutputFile, sizeof job->szOutputFile, t);

				} else if (!stricmp(s, "error")) {

					strgetarg(job->szError, sizeof job->szError, t);

				} else if (!stricmp(s, "state")) {

					job->iState = atoi(t);

					// Make sure "In Progress" states change to Aborted

					if (job->iState == INPROGRESS)
						job->iState = ABORTED;

				} else if (!stricmp(s, "start_time")) {
					FILETIME ft;

					if (2 != sscanf(t, "%08lx %08lx", &ft.dwHighDateTime, &ft.dwLowDateTime))
						throw "invalid start time";

					if (!ft.dwHighDateTime && !ft.dwLowDateTime)
						memset(&job->stStart, 0, sizeof(SYSTEMTIME));
					else
						FileTimeToSystemTime(&ft, &job->stStart);

				} else if (!stricmp(s, "end_time")) {
					FILETIME ft;

					if (2 != sscanf(t, "%08lx %08lx", &ft.dwHighDateTime, &ft.dwLowDateTime))
						throw "invalid start time";

					if (!ft.dwHighDateTime && !ft.dwLowDateTime)
						memset(&job->stEnd, 0, sizeof(SYSTEMTIME));
					else
						FileTimeToSystemTime(&ft, &job->stEnd);

				} else if (!stricmp(s, "script")) {

					script_capture = true;

				} else if (!stricmp(s, "endjob")) {
					if (script_capture) {
						job->script = jso.getscript();
						jso.clear();
						script_capture = false;
					}

					job->Add(true);
					job = NULL;;
				}
			} else if (script_capture) {
				// kill starting spaces

				s = linebuf;

				while(isspace(*s)) ++s;

				// don't add blank lines

				if (*s)
					jso.adds(s);
			}
		}

	} catch(int e) {
		_RPT0(0,"I/O error on job load\n");
		if (lpszName != szName) {
			if (f) fclose(f);
			throw MyError("Failure loading job list: %s.", strerror(e));
		}
	} catch(char *s) {
		_RPT0(0,s);
		if (lpszName != szName) {
			if (f) fclose(f);
			throw MyError(s);
		}
	}

	if (f) fclose(f);
	delete job;
}

void VDJob::SetModified() {
	fModified = true;

	if (!g_hwndJobs)
		Flush();
}

void VDJob::SetSlaveMode(bool Mode)
{
if (!fMasterMode) fSlaveMode=Mode;
}

void VDJob::SetMasterMode(bool Mode)
{
if (!fSlaveMode) fMasterMode=Mode;
}


// VDJob::Flush()
//
// Flushes the job list out to disk.
//
// We store the job list in a file called VirtualDubMod.jobs.  It's actually a
// human-readable, human-editable Sylia script with extra comments to tell
// VirtualDub about each of the scripts.
// 13/11/2002, Cyrius : changed to VirtualDubMod.jobs

void VDJob::Flush(char *lpszFileName) {
	FILE *f = NULL;
	char szName[MAX_PATH], szVDPath[MAX_PATH], *lpFilePart;

	_RPT0(0,"VDJob::Flush()\n");

	// Try to create VirtualDubMod.jobs in the same directory as VirtualDub.

	if (!lpszFileName) {
		if (!GetModuleFileName(NULL, szVDPath, sizeof szVDPath))
			return;

		if (!GetFullPathName(szVDPath, sizeof szName, szName, &lpFilePart))
			return;

		// 19/12/2002, Cyrius : use the .exe filename as basis for the jobs file
		// This may be used as a workaround for current version of GKnot which
		// uses VirtualDub.jobs
		char *lpExt = lpFilePart + strlen(lpFilePart);
		while( (lpExt >= lpFilePart) && (*lpExt != '.'))
			lpExt--;
		if(*lpExt != '.')
			lpExt = lpFilePart + strlen(lpFilePart);
		strcpy(lpExt, ".jobs");

		lpszFileName = szName;
	}

	try {
		VDJob *vdj, *vdj_next;

		f = fopen(lpszFileName, "w");
		if (!f) throw errno;

		if (fprintf(f,
				"// VirtualDub job list (Sylia script format)\n"
				"// This is a program generated file -- edit at your own risk.\n"
				"//\n"
				"// $numjobs %d\n"
				"//\n\n"
				,VDJob::ListSize()
				)<0)
			throw errno;

		vdj = (VDJob *)job_list.tail.next;

		while(vdj_next = (VDJob *)vdj->next) {
			FILETIME ft;
			char *s, *t, c;

			if (fprintf(f,"// $job \"%s\""			"\n", vdj->szName)<0) throw errno;
			if (fprintf(f,"// $input \"%s\""		"\n", vdj->szInputFile)<0) throw errno;
			if (fprintf(f,"// $output \"%s\""		"\n", vdj->szOutputFile)<0) throw errno;
			if (fprintf(f,"// $state %d"			"\n", vdj->iState)<0) throw errno;

			if (vdj->stStart.wYear) {
				SystemTimeToFileTime(&vdj->stStart, &ft);
				if (fprintf(f,"// $start_time %08lx %08lx"	"\n", ft.dwHighDateTime, ft.dwLowDateTime)<0) throw errno;
			} else
				if (fprintf(f,"// $start_time 0 0\n")<0) throw errno;

			if (vdj->stEnd.wYear) {
				SystemTimeToFileTime(&vdj->stEnd, &ft);
				if (fprintf(f,"// $end_time %08lx %08lx"	"\n", ft.dwHighDateTime, ft.dwLowDateTime)<0) throw errno;
			} else
				if (fprintf(f,"// $end_time 0 0\n")<0) throw errno;

			if (vdj->iState == ERR)
				if (fprintf(f,"// $error \"%s\"\n", vdj->szError)<0) throw errno;

			if (fprintf(f,"// $script\n\n")<0) throw errno;

			// Dump script

			s = vdj->script;

			while(*s) {
				t=s;

				while((c=*t) && c!='\r' && c!='\n')
					++t;

				if (t>s)
					if (1!=fwrite(s, t-s, 1, f)) throw errno;
				if (EOF==putc('\n', f)) throw errno;

				// handle CR, CR/LF, LF, and NUL terminators

				if (c == '\r') ++t;
				if (c == '\n') ++t;

				s=t;
			}

			// Next...

			if (fputs(
					"\n"
					"// $endjob\n"
					"//\n"
					"//--------------------------------------------------\n"
				,f)==EOF) throw errno;

			vdj = (VDJob *)vdj->next;
		}

		if (fprintf(f,"// $done\n")<0) throw errno;

		if (fflush(f)) throw errno;

		if (lpszFileName == szName) fModified = false;

	} catch(int) {
		_RPT0(0,"I/O error on job flush\n");
		if (lpszFileName) {
			if (f) fclose(f);
			throw MyError("Job list flush failed: %s.", strerror(errno));
		}
	}

	if (f) fclose(f);
}

void VDJob::RunAll() {
	VDJob *vdj = (VDJob *)job_list.tail.next;

	fRunInProgress	= true;
	fRunAllStop		= false;

	if (g_hwndJobs) SetDlgItemText(g_hwndJobs, IDC_START, "Stop");

	SetJobWindowButtons();
	EnableMenuEntries();

	while((VDJob *)vdj->next && !fRunAllStop) {
		if (vdj->iState == WAITING) {
			vdj->Run();
			if (((vdj->iState==VDJob::ERR)||(vdj->iState==VDJob::ABORTED))&&(!fRunAllStop)) 
				RunAllStop();
		}


		vdj = (VDJob *)vdj->next;
	}

	fRunInProgress = false;

	if (g_hwndJobs)	SetDlgItemText(g_hwndJobs, IDC_START, "Start");

	EnableMenuEntries();
	SetJobWindowButtons();


	// Shutdown all Clients 
	if ((!vdj->next)&&(g_JobsDlgConfig.ShutdownClients)&&(g_hwndJobs))
		SendMessage(g_hwndJobs,IDC_SHUTDOWN_CLIENTS,0,0);

	// Shutdown after all Jobs 
	if ((!vdj->next)&&(g_JobsDlgConfig.Shutdown))
	{
		char *ShutDownScript="VirtualDub.Shutdown();";
		_CrtCheckMemory();
		RunScriptMemory(ShutDownScript);
	}

}

void VDJob::RunAllStop() {
	fRunAllStop = true;
	if (g_hwndJobs)
	{
		CheckDlgButton(g_hwndJobs, IDC_JOBS_SHUTDOWN, BST_UNCHECKED);
		CheckDlgButton(g_hwndJobs, IDC_JOBS_SHUTDOWN_ALL, BST_UNCHECKED);
		g_JobsDlgConfig.Shutdown=g_JobsDlgConfig.ShutdownClients=false;
	}
}

///////////////////////////////////////////////////////////////////////////

static BOOL CALLBACK JobErrorDlgProc(HWND hdlg, UINT uiMsg, WPARAM wParam, LPARAM lParam) {

	switch(uiMsg) {
	case WM_INITDIALOG:
		{
			VDJob *vdj = (VDJob *)lParam;
			char buf[1024];

			_snprintf(buf, sizeof buf, "VirtualDub - Job \"%s\"", vdj->szName);
			SetWindowText(hdlg, buf);

			SetDlgItemText(hdlg, IDC_ERROR, vdj->szError);
		}
		return TRUE;

	case WM_COMMAND:
		if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL) {
			EndDialog(hdlg, 0);
			return TRUE;
		}
		break;
	}

	return FALSE;
}

static void Job_GetDispInfo(NMLVDISPINFO *nldi) {
	VDJob *vdj = VDJob::ListGet(nldi->item.iItem);
	SYSTEMTIME *st = &vdj->stEnd;
	SYSTEMTIME ct;
	static const char *dow[]={"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};

	nldi->item.mask			= LVIF_TEXT;
	nldi->item.pszText[0]	= 0;

	switch(nldi->item.iSubItem) {
	case 0:
		nldi->item.pszText = vdj->szName;
		break;
	case 1:		// file in
		nldi->item.pszText = vdj->szInputFileTitle;
		break;
	case 2:		// file out
		nldi->item.pszText = vdj->szOutputFileTitle;
		break;
	case 3:		// time in
		st = &vdj->stStart;
	case 4:		// time out
		GetLocalTime(&ct);
		if (!st->wYear)
			nldi->item.pszText = "-";
		else if (ct.wYear != st->wYear
			|| ct.wMonth != st->wMonth
			|| ct.wDay != st->wDay) {

			_snprintf(nldi->item.pszText, nldi->item.cchTextMax, "%s %d %d:%02d%c"
						,dow[st->wDayOfWeek]
						,st->wDay
						,st->wHour==12||!st->wHour ? 12 : st->wHour%12
						,st->wMinute
						,st->wHour>=12 ? 'p' : 'a');
		} else {
			_snprintf(nldi->item.pszText, nldi->item.cchTextMax, "%d:%02d%c"
						,st->wHour==12||!st->wHour ? 12 : st->wHour%12
						,st->wMinute
						,st->wHour>=12 ? 'p' : 'a');
		}
		break;
	case 5:		// status
		switch(vdj->iState) {
		case VDJob::WAITING:	nldi->item.pszText = "Waiting"		; break;
		case VDJob::INPROGRESS:	nldi->item.pszText = "In progress"	; break;
		case VDJob::DONE:		nldi->item.pszText = "Done"			; break;
		case VDJob::POSTPONED:	nldi->item.pszText = "Postponed"	; break;
		case VDJob::ABORTED:	nldi->item.pszText = "Aborted"		; break;
		case VDJob::ERR:		nldi->item.pszText = "Error"		; break;
		}
		break;
	}
}


static void JobProcessDirectory(HWND hDlg) {
	char szSourceDir[MAX_PATH];
	char szDestDir[MAX_PATH];
	char *lpszFileName;
	BROWSEINFO bi;
	LPITEMIDLIST pidlBrowse;
	LPMALLOC pMalloc;
	bool fAbort = false;

	if (SUCCEEDED(SHGetMalloc(&pMalloc))) {
		if (lpszFileName = (char *)pMalloc->Alloc(MAX_PATH)) {
			bi.hwndOwner		= hDlg;
			bi.pidlRoot			= NULL;
			bi.pszDisplayName	= lpszFileName;
			bi.lpszTitle		= "Select source directory";
			bi.ulFlags			= BIF_RETURNONLYFSDIRS;
			bi.lpfn				= NULL;

			if (pidlBrowse = SHBrowseForFolder(&bi)) {
				if (SHGetPathFromIDList(pidlBrowse, lpszFileName))
					strcpy(szSourceDir, lpszFileName);
				else
					fAbort = true;

				pMalloc->Free(pidlBrowse);
			}

			bi.lpszTitle		= "Select destination directory";

			if (!fAbort && (pidlBrowse = SHBrowseForFolder(&bi))) {
				if (SHGetPathFromIDList(pidlBrowse, lpszFileName))
					strcpy(szDestDir, lpszFileName);

				pMalloc->Free(pidlBrowse);
			}
			pMalloc->Free(lpszFileName);
		}
	}

	if (fAbort) return;
	JobAddBatchDirectory(szSourceDir, szDestDir);
}

static struct ReposItem jobCtlPosData[]={
	{ IDOK			, REPOS_MOVERIGHT },
	{ IDC_MOVE_UP	, REPOS_MOVERIGHT },
	{ IDC_MOVE_DOWN	, REPOS_MOVERIGHT },
	{ IDC_POSTPONE	, REPOS_MOVERIGHT },
	{ IDC_DELETE	, REPOS_MOVERIGHT },
	{ IDC_START		, REPOS_MOVERIGHT },
	{ IDC_ABORT		, REPOS_MOVERIGHT },
	{ IDC_JOBS_SHUTDOWN, REPOS_MOVERIGHT },
	{ IDC_JOBS		, REPOS_SIZERIGHT | REPOS_SIZEDOWN },
	{ IDC_CURRENTPART, REPOS_MOVEDOWN },
	{ IDC_PROGRESS	, REPOS_MOVEDOWN | REPOS_SIZERIGHT },
	{ IDC_PERCENT	, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ IDC_HOSTS		, REPOS_MOVEDOWN | REPOS_SIZERIGHT },
	{ IDC_UPDATE	, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ IDC_MASTER_MODE, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ IDC_SLAVE_MODE, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ IDC_MSMODE_ABORT, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ IDC_JOBS_SHUTDOWN_ALL, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ IDC_STATUSMAP,	REPOS_MOVEDOWN | REPOS_SIZERIGHT },
	{ IDC_NETWORKJOB, REPOS_MOVEDOWN },
	{ IDC_NETWORK_PROGRESS, REPOS_MOVEDOWN | REPOS_SIZERIGHT },
	{ IDC_NETWORK_PERCENT, REPOS_MOVEDOWN | REPOS_MOVERIGHT },
	{ 0 }
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
bool ScriptSearchLine(char *script, char *scanline, char *output)
{
	char *s=script, *t;
	int len=strlen(scanline);
	while (*s) {
		t = s;
		while(*t && *t!='\n') ++t;
		if (*t && *t=='\n') ++t;
		if (0==memcmp(s,scanline,len))
		{
			if (output) {
				memcpy(output, s, t-s);
				output[t-s]=0; }
			return true;
		}
		s = t;
	}
return false;
}

bool ScriptRemoveLine(char *script, char *scanline)
{
	char *s=script, *t;
	int len=strlen(scanline);
	while (*s) {
		t = s;
		while(*t && *t!='\n') ++t;
		if (*t=='\n') ++t;
		if (0==memcmp(s,scanline,len))
		{
			memmove(s, t, strlen(t));
			s[strlen(t)]=0;
			return true;
		}
		s = t;
	}
return false;
}

bool ScriptReplaceLine(char *script, char *scanline, char *newline)
{		// replace the line containing "scanline" with "newline"
	char *s=script, *t;
	int scanlen=strlen(scanline);
	int newlen=strlen(newline);

	while (*s) {
		t = s;
		while(*t && *t!='\n') ++t;
		if (*t=='\n') ++t;
		if (0==memcmp(s,scanline,scanlen))
		{
			memmove(s+newlen,t,strlen(t)+1);
			memmove(s,newline,newlen);
			return true;
		}
		s = t;
	}
return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
#define SHEDULER_SINGLEMODE	-1
#define MAXPARTCOUNT	512
#define MAXHOSTCOUNT	128

//////////////////////////////////////////////////////////////////////////////////////////////////////////

struct PartInfo {
	int		From, To;		// Framerange
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////

struct ShedulePartInfo {
	int			SourceNum;
	int			From;
	int			To;

	int			Host;
	int			Progress;
	int			State;

	int			OutFrom;		// set by recalcFrameCount
	int			OutTo;

	// only used by sheduler:
	int			BlockSize;
	int			PartCount;
	int			PartSize;
	int			FramesSheduled;

#ifdef NANDUB
	int			Bitrate;
	int			Keyframe;
#endif


};

//////////////////////////////////////////////////////////////////////////////////////////////////////////

struct SheduleHostInfo {
	int		FPS;
	int		HostSpeed;
	int		tmpFPS;
	int		Speed;		
	int		Frames;
	bool	Active;
	__int64	startTime;
	__int64	stopTime;

	// only used by sheduler:
	int		framesToGo;
	int		framesToShedule;
	int		ticksToGo;

};

//////////////////////////////////////////////////////////////////////////////////////////////////////////

struct InputFileInfo {
	char	filename[256];
	int		From;
	int		To;
	int		msStartTime;
	int		msLength;
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////

struct NetworkHostInfo {
	bool	Running;
	char	Name[32];
	int		Type;
	int		From;
	int		To;
	int		Progress;
	int		FPS;
	bool	Valid;
	DWORD	IP;
	SOCKET	CtrlSocket;
	unsigned char*	MsgBuf;
	int		MsgBufLen;
	bool	ReScanFound;
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////

struct NetworkTransferInfo {
	bool	Running;
	char	Name[32];
	int		Type;
	int		From;
	int		To;
	int		Progress;
	int		FPS;
};


//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////

class PartSheduler {
private:
	ShedulePartInfo *Parts;
	SheduleHostInfo *Hosts;
	InputFileInfo	*IFInfo;

	int		iActiveHosts;
	int		iInputFiles;

	int		iPartCount;
	int		iValidPartCount;

	bool	bExtAudioSource;

	int		iAudioInterleave;

	char	AudioInterleave[LINELEN];

	bool	fSingleMode;

	char	*workscript;
	char	*origscript;

	char	tmpscript[SCRIPTLEN];

	int		iGlobalProgress;

	int		iFrames;
	int		iOpenFrames;

	int		getFreePartNum();
	int		getHostPart(int Host);

	bool	mergePart(int P1, int P2);
	void	tryMergeParts();
	void	recalcFrameCount();
	__int64 getLocalTime();

	bool	get1IntParamFromString(char *line, int *p);
	bool	get2IntParamFromString(char *line, int *p1, int *p2);
	bool	getStrParamFromString(char *line, char *string=NULL);

	void	getUniquePartfileName(char *name);
	void	savePartInfo();
	bool	loadPartInfo();

	bool	testValidPathInfo(char *filename);

	bool	setInputFileInfo();

	FILE	*logfile;
	void	openlogfile(bool erase=false);
	void	closelogfile();

	int		splitPart(int Part, int Blocksize);
	bool	partsLocked(int P1, int P2, int State=0);
	bool	pairCompare(int A1, int A2, int B1, int B2);
	int		scanSilence(int Part, int StartFrame);

#ifdef NANDUB
	bool	fUseStats;
	bool	fGenStats;
	bool	fGenOutput;
	char	statsfile[256];

	void	correctBitrate(int BitRate,int Keyframe, int CreditsBitrate, int CreditsStart);

	bool	checkStatsFile(char *filename);
	void 	tryRebuildStatsfile();
	void	storeStatsFilePart(int Num);
	void	createStatsFilePart(int Num);
	void	removeStatsFilePart(int Num);
	
#endif

public:
	void	log(int level, char *s, ... );
	void	ClearLogFile();

	PartSheduler(int HostCount);								// Constructor, (Number of Hosts)
	~PartSheduler();											// Destructor

	int		GetFrameCount();
	int		GetFrameState(int Part, int Progress);				// Progress=0..999
	int		GetPartCount();
	int		GetPartFirstFrame(int Part);
	bool	IsBorderSheduled(int P1, int P2);
	int		GetPartState(int Part);

	void	SetActiveHost(int Host, bool Active=true);

	void	Clear();												// Restart

	void	AddBlock(int From, int To);								// add a movie-segment
	void	SheduleJob();	
	void	rekursiveSheduleBlock(int BlockFree, int HostCount, int ChangePos, int depth=0);
	bool	GetNextPart(int Host);									// get a part of a segment to a host

	bool	GetPartInfo(int Host, PartInfo *PI);					// get info about part associated with a host
	int		GetPartFrom(int Host);									// the "from"-Info
	int		GetPartTo(int Host);									// the "to"-Info

	int		GetPartNumbers(int **Numbers, int Mode=PI_DONE);		// return the right order of all parts

	void	PartSuccess(int Host);									// Part successfully completed
	void	PartAborted(int Host);									// Part aborted -> will be returned again by "GetNextPart"

	void	SetHostProgress(int Host, int Progress);				// set progressinformation (range: 0..1000)
	int		GetGlobalProgress();									// get progress for current job
	int		GetHostProgress(int Host);								// get progress for one host and associated part
	int		GetHostFPS(int Host);									// returns -1 if failure
	void	SaveHostSpeed(int Host);
	int		LoadHostSpeed(int Host);

	bool	SetOriginalScript(int LocalHost, char *script);			// deliver the job-script to sheduler
	void	GetModifiedScript(int Host, char *script);				// get modified script for a part

	void	GetConcatScript(char *script);							// generate script to merge all parts
	void	RemoveTempFiles();										// remove all temp. parts (the .avi-files)

	bool	IsSingleMode() { return fSingleMode; }
	bool	IsHostJob(int Host);
	bool	IsAnyHostJob();
	bool	IsAllPartsComplete();

	bool	ModifyFileName(int Num, char *filename);

	// Singlemode functions
	void	SetPartInfo(int From, int To);
	void	SetGlobalProgress(int Progress);

#ifdef NANDUB
	bool	IsGenStats()	{ return fGenStats; }
	bool	IsUseStats()	{ return fUseStats; }
	bool	IsGenOutput()	{ return fGenOutput; }
#else
	bool	IsGenOutput()	{ return true; }
#endif

} *psSheduler;


//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////

class NetworkManager {
private:
	NetworkHostInfo *HostInfo;
	int		getHostNum(char *HostName);
	bool	fSingleMode;
	int		HostCount;
	int		LocalHost;
	DWORD	LocalIP; 
	int		LocalState;

	void	LoadHostList();
	void	RekusiveScanNetwork();

	DWORD	RecursiveScanNetwork(LPNETRESOURCE RootResource=NULL, int depth=1);
	void	RemoveDeadHost(int Host);

	SOCKET	SelectSocket;

public:
	NetworkManager();
	~NetworkManager();
	void	Clear();

	void	GetTransferInfo(int Host, NetworkTransferInfo *NTI);
	void	SetTransferInfo(NetworkTransferInfo *NTI);
	void	SetTransferInfo(int Host, NetworkTransferInfo *NTI);

	bool	StartMSMode();
	bool	SetMasterMode();
	bool	SetSlaveMode();
	bool	SetSingleMode();

	SOCKET	GetSock(int Host);
	void	SetSock(int Host,SOCKET s);

	DWORD	GetIP(int Host);
	void	SetIP(int Host, DWORD IP);

	sockaddr* GetIPstruct(int Port);

	int		GetType(int Host);
	void	SetType(int Host, int Type);
	bool	IsType(int Host, int Type1, int Type2=-1, int Type3=-1);

	bool	GetWorkState(int Host);
	void	SetWorkState(int Host, bool Running);

	int		GetMsgbufLen(int Host);
	int		GetMsgID(int Host);
	void	SyncMsgQueue(int Host);
	void	MsgbufRecv(int Host, SOCKET s, int Flags);
	void*	GetMsgbufDataPtr(int Host, int Offset=0);
	void	RemoveMessage(int Host, int Len);

	char*	GetName(int Host);
	int		GetHostCount() { return HostCount; }
	int		GetLocalHost() { return LocalHost; }

	void	UpdateHostList();
	bool	TryAddHost(char *name);

	bool	CreateListenSocket();
	bool	DestroyListenSocket();

} *netMan;


////////////////////////////////////////////////////////////////////
PartSheduler::PartSheduler(int HostCount) {
	if (fSingleMode=(HostCount==SHEDULER_SINGLEMODE)) {
		iPartCount=1;
		iGlobalProgress=0;
	} else {
		iPartCount=MAXPARTCOUNT;
	}

	Parts=(ShedulePartInfo*)calloc(iPartCount*sizeof ShedulePartInfo,1);
	Hosts=(SheduleHostInfo*)calloc(MAXHOSTCOUNT*sizeof SheduleHostInfo,1);
	IFInfo=0;
	iInputFiles=0;

	workscript=origscript=NULL;

#ifdef NANDUB
	*statsfile=0;
#endif

	logfile=NULL;
	openlogfile();
	Clear();

}

////////////////////////////////////////////////////////////////////
PartSheduler::~PartSheduler() {
	if (Parts) free(Parts);
	if (Hosts) free(Hosts);
	if (workscript) free(workscript);
	if (origscript) free(origscript);
	if (IFInfo) free(IFInfo);

	log(3,"Destructor... bye bye\n\n");
	closelogfile();

}

////////////////////////////////////////////////////////////////////
void PartSheduler::openlogfile(bool erase) {
	if (logfile) return;
	if (g_JobsDlgConfig.Loglevel==0) return;

	int ret;

	char szVDPath[256],*lpFilePart;

	GetModuleFileName(NULL, szVDPath, sizeof szVDPath);
	GetFullPathName(szVDPath, sizeof szVDPath, szVDPath, &lpFilePart);

	strcpy(lpFilePart, "jobs.log");

	if (erase)
		ret=remove(szVDPath);

	logfile=fopen(szVDPath,"a+t");

	log(3,"Constructor, %s", (fSingleMode) ? "Single Mode (Single, Slave)" : "Multihostmode (Master)");
}

////////////////////////////////////////////////////////////////////
void PartSheduler::closelogfile() {
	if (logfile) {
		fflush(logfile);
		fclose(logfile);
		logfile=NULL;
	}
}

////////////////////////////////////////////////////////////////////
void PartSheduler::ClearLogFile() {
	closelogfile();
	openlogfile(true);	// reopen and erase
}

////////////////////////////////////////////////////////////////////
void PartSheduler::log(int level, char *s, ... ) {
	if (g_JobsDlgConfig.Loglevel==0) {
		if (logfile) 
			closelogfile();
		return;
	}

	if (!logfile)
		openlogfile();

	if (!logfile)
		return;

	if (level>g_JobsDlgConfig.Loglevel)
		return;

	va_list param;
	va_start(param, s);
	char ti[20];
	_strtime(ti);
	strcat(ti,"-");
	fprintf(logfile,ti);
	vfprintf(logfile,s,param);
	if (param[strlen(param)]!='\n')
		fprintf(logfile,"\n");
	fflush(logfile);
}

////////////////////////////////////////////////////////////////////
int PartSheduler::getFreePartNum() { // number of first free partspace
	for (int i=0;i<iPartCount;i++)
		if (Parts[i].State==PI_NONE)
			return i;
	return -1;
}

////////////////////////////////////////////////////////////////////
int	PartSheduler::getHostPart(int Host) {
	for (int i=0;i<iPartCount;i++)
		if ((Parts[i].State==PI_PROGRESS)&&(Parts[i].Host==Host))
			return i;
	return -1;
}

////////////////////////////////////////////////////////////////////
__int64 PartSheduler::getLocalTime() {
	SYSTEMTIME	sTime;
	__int64 iTime;
	GetLocalTime(&sTime);
	SystemTimeToFileTime(&sTime,(FILETIME*)&iTime);
	return iTime;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::get2IntParamFromString(char *line, int *p1, int *p2) {  // string(p1,p2)string
	char *s=strchr(line,'(');
	char *t=strchr(line,',');
	char *u=strchr(line,')');
	if ((!s++)||(!t++)||(!u++)) return false;
	char param[20];
	memset(param,0,20);
	*p1=atoi(strncpy(param,s,t-s-1));
	memset(param,0,20);
	*p2=atoi(strncpy(param,t,u-t-1));
	return true;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::get1IntParamFromString(char *line, int *p) {
	char *s=strrchr(line,',');		// ...("Hello",0);
	if (!s) s=strchr(line,'(');		// ...(0);
	char *t=strrchr(line,')');
	if ((!s++)||(!t++)) return false;
	char param[128];
	memset(param,0,128);
	*p=atoi(strncpy(param,s,t-s-1));
	return true;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::getStrParamFromString(char *line, char *string) {	// text("string")text
	char *s=strchr(line,'\"');
	//char *t=strrchr(line,'\"');
	// 2002-12-26 bug fix by stream
	char *t = NULL;
	if (s) t = strchr(s+1, '\"');
	// end bug fix
	char *out=(string)?string:line;
	if ((!s)||(!t)) return false;
	memmove(out,s+1,t-s-1);
	out[t-s-1]=0;
	return true;
}

WAVEFORMATEX	*iFormat,	*oFormat;
bool			acmDriverFound;
HACMDRIVER		acmDriver;
HACMSTREAM		acmStreamHandle;
HACMDRIVERID	adiDriverID;

BOOL CALLBACK JobsACMStreamOpenCallback(HACMDRIVERID hadid, DWORD dwInstance, DWORD fdwSupport) {
	bool success=false;
	if (fdwSupport & ACMDRIVERDETAILS_SUPPORTF_CODEC) {

		MMRESULT res;

		// Attempt to open driver.

		res = acmDriverOpen(&acmDriver, hadid, 0);

		if (!res) {

			// Attempt our stream open!
			res = acmStreamOpen(&acmStreamHandle, acmDriver, iFormat, oFormat, NULL, 0, 0, ACM_STREAMOPENF_NONREALTIME);
			if (!res)
				success = true;
			else {
				res = acmStreamOpen(&acmStreamHandle, acmDriver, iFormat, oFormat, NULL, 0, 0, 0);
				if (!res)
					success = true;
				else
					acmDriverClose(acmDriver, 0);
			}
		}
	}

	acmDriverFound=success;
	adiDriverID=hadid;

	return !success;
}


////////////////////////////////////////////////////////////////////
int PartSheduler::scanSilence(int Part, int StartFrame) {
#ifdef NANDUB
	if (!fGenOutput)
		return StartFrame;
#endif

	if ((!g_JobsDlgConfig.ProcessAudio)||		// no audio encoding
		(!origscript)||							// bad error.... 
		(ScriptSearchLine(origscript,"VirtualDub.audio.SetSource(0)",NULL)))	// no Audio
		return StartFrame;
		
	int *Num,Cnt;
	Cnt=GetPartNumbers(&Num,PI_VALID);

	// Prepare filename
	char filename[256];

	strcpy(filename,IFInfo[Parts[Part].SourceNum].filename);

	if (bExtAudioSource) {
		if (ScriptSearchLine(origscript,"VirtualDub.audio.SetSource",filename)) {
			getStrParamFromString(filename);				
		}
	}

	strUnCify(filename);


	// Init AVI-System
	AVIFileInit();

	log(3,"ScanForSilence: started, File=\"%s\"",filename);

	// Open the AVI-File
	PAVIFILE f;
	if (0==AVIFileOpen(&f, filename, OF_SHARE_DENY_NONE|OF_READ, NULL)) {

		// Get 1st. Audiostream from AVI-File
		PAVISTREAM aviStream;
		if (0==AVIFileGetStream(f,&aviStream,streamtypeAUDIO,0)) {

			AVISTREAMINFO StreamInfo;
			AVIStreamInfo(aviStream, &StreamInfo, sizeof(StreamInfo));

			iFormat=(WAVEFORMATEX*)calloc(512,1);
			oFormat=(WAVEFORMATEX*)calloc(512,1);
			long bufsize=512;

			// Get Stream-Info
			AVIStreamReadFormat(aviStream, 0, iFormat, &bufsize);

			bool pcmAudio=(iFormat->wFormatTag==1);
			AudioFormatConverter pcmDecoder;
			if (pcmAudio)
				pcmDecoder=AudioPickConverter(iFormat,false, false);

			memcpy(oFormat,iFormat,512);

			// Calculate Outputstream-Info
			oFormat->wFormatTag=1;
			oFormat->nBlockAlign=1;
			oFormat->wBitsPerSample=8;
			oFormat->nChannels=1;
			oFormat->cbSize=0;
			oFormat->nAvgBytesPerSec=(oFormat->wBitsPerSample*oFormat->nChannels*oFormat->nSamplesPerSec)/8;

			log(5,"ScanForSilence: Stream opened, Audiotype=%s, %dBit/s %s",pcmAudio ? "PCM-Audio" : "Compressed audio",
																		   iFormat->nAvgBytesPerSec*8, 
																		   (iFormat->nChannels==1) ? "Mono" : (iFormat->nChannels==2) ? "Stereo" : "(>2 Channels, ACM?)"); 
			// Try all drivers, open Audio-Decompressor
			acmDriverFound=false;
			if ((pcmAudio)||(!acmDriverEnum(JobsACMStreamOpenCallback, 0, 0))) {

				if ((acmDriverFound)||(pcmAudio)) {

					if (acmDriverFound) {
						ACMDRIVERDETAILS addInfo;
						addInfo.cbStruct=sizeof(addInfo);
						acmDriverDetails(adiDriverID, &addInfo, 0);
						log(5,"ScanForSilence: Decompressor found\n\t%s\n\t%s\n\t%s\n\t%s\n\t%s",
							addInfo.szShortName, addInfo.szLongName, 
							addInfo.szCopyright, addInfo.szLicensing,
							addInfo.szFeatures);
					}

					int ScanTime=__min( g_JobsDlgConfig.PartLength/2, 120);
					if (pcmAudio)
						ScanTime=__min(ScanTime,60);

					int InputBufferSize=iFormat->nAvgBytesPerSec*ScanTime;
					int OutputBufferSize=oFormat->nAvgBytesPerSec*ScanTime;

					long InputSamplesToScan=iFormat->nSamplesPerSec * ScanTime;		
					long OutputSamplesToScan=oFormat->nSamplesPerSec * ScanTime;		

					LPBYTE InputBuf=(LPBYTE)calloc(InputBufferSize,1);
					LPBYTE OutputBuf=(LPBYTE)calloc(OutputBufferSize,1);

					ACMSTREAMHEADER	ashInfo;
					memset(&ashInfo,0,sizeof(ashInfo));

					ashInfo.cbStruct=sizeof(ashInfo);

					ashInfo.pbSrc=InputBuf;
					ashInfo.cbSrcLength=InputBufferSize;
					ashInfo.pbDst=OutputBuf;
					ashInfo.cbDstLength=OutputBufferSize;

					long BytesRead=0,SamplesRead=0;

					// Calc. Starttime
					long StartTime=MulDiv(StartFrame,1000,g_FramesPerSec)-IFInfo[Parts[Part].SourceNum].msStartTime;
					long SilentTime=StartTime;

					long StartSample;
					long ScanLength=oFormat->nSamplesPerSec/2;		// 500ms Silence
					long SilentValue=0x7F*ScanLength;						
					BYTE *ScanWin=(BYTE*)calloc(ScanLength,1);
														
					// Prepare Conversion-Stream
					if ((pcmAudio)||(!acmStreamPrepareHeader(acmStreamHandle,&ashInfo,0))) {

						// Calc. StartSample
						StartSample=AVIStreamTimeToSample(aviStream,StartTime);

						// Read Data from AVI-Stream
						AVIStreamRead(aviStream, StartSample, InputSamplesToScan, InputBuf, InputBufferSize, &BytesRead, &SamplesRead);

						// Decompress audio
						ashInfo.cbSrcLength=BytesRead;
						ashInfo.cbSrcLengthUsed=0;
						ashInfo.cbDstLengthUsed=0;
		
						if ((pcmAudio)&&(SamplesRead))
							pcmDecoder(OutputBuf, InputBuf, SamplesRead);
							
						if ((pcmAudio)||
							(!acmStreamConvert(acmStreamHandle,&ashInfo,ACM_STREAMCONVERTF_BLOCKALIGN | ACM_STREAMCONVERTF_START ))) {

							long Samples=(pcmAudio) ? SamplesRead : 8*ashInfo.cbDstLengthUsed/(oFormat->wBitsPerSample*oFormat->nChannels);

							// Search for "silent" part
							memset(ScanWin,0x7F,ScanLength);
							long ScanWinPos=0, sum=(0x7F*ScanLength);

							for (int i=0;i<Samples;i++) {
								ScanWinPos=i%ScanLength;
							
								int OldVal=ScanWin[ScanWinPos];
								int NewVal=OutputBuf[i];
								NewVal=abs(NewVal-128);

								sum-=OldVal;
								sum+=NewVal;

								ScanWin[ScanWinPos]=NewVal;

								if ((i>=ScanLength)&&(SilentValue>sum)) {
									SilentValue=sum;
									SilentTime=StartTime+MulDiv(i-ScanLength/2,1000,oFormat->nSamplesPerSec)-iAudioInterleave;
								}
							}
						if (!pcmAudio) 
							acmStreamUnprepareHeader(acmStreamHandle,&ashInfo,0);
						}
					}

					log(5,"ScanForSilence: Scan complete");
					StartFrame=MulDiv(SilentTime+IFInfo[Parts[Part].SourceNum].msStartTime,g_FramesPerSec,1000);
					StartFrame=__max(StartFrame,Parts[Part].From);
					StartFrame=__min(StartFrame,Parts[Part].To);

					// Close StreamConverter and acmDriver
					if (!pcmAudio) {
						acmStreamClose(acmStreamHandle, 0);
						acmDriverClose(acmDriver,0);
					}

					// dealloc
					free(InputBuf);
					free(OutputBuf);
				} else log(1,"ScanForSilence: no decompressor found");

			} else log(1,"ScanForSilence: DriverEnum failed");

			AVIStreamRelease(aviStream);
		} else log(1,"ScanForSilence: no audiostream found");

		AVIFileRelease(f);

	} else log(1,"ScanForSilence: unable to open file");
	AVIFileExit();
	return StartFrame;

}

////////////////////////////////////////////////////////////////////
bool PartSheduler::partsLocked(int P1, int P2, int State) {
	if ((Parts[P1].To+1==Parts[P2].From)&&
#ifdef NANDUB
		(Parts[P1].Bitrate==Parts[P2].Bitrate)&&
#endif
		(Parts[P1].SourceNum==Parts[P2].SourceNum)) 

		if ((State==0)||
			((Parts[P1].State==State)&&(Parts[P2].State==State)))
			return true;
return false;
}

////////////////////////////////////////////////////////////////////
int PartSheduler::splitPart(int Part, int Blocksize) {
	if (Parts[Part].To-Parts[Part].From+1<=Blocksize)
		return Part;

	int PartNum=getFreePartNum();

	int CutFrame=Parts[Part].From+Blocksize-1;

	if ((g_JobsDlgConfig.ScanForSilence) && 
		(g_JobsDlgConfig.ProcessAudio) && 
		(Parts[Part].State!=PI_SHEDULED)) {

		CutFrame=scanSilence(Part, CutFrame);
		if (CutFrame==Parts[Part].To) {
			log(5,"SplitPart, Part %d (%d-%d) not splitted",Part,Parts[Part].From,Parts[Part].To);
			return Part;
		}
	}

	Parts[PartNum]=Parts[Part];
	Parts[Part].To=CutFrame;
	Parts[PartNum].From=CutFrame+1;
	log(5,"SplitPart, Part %d (now: %d-%d) splitted, new part=%d (%d-%d)",Part, Parts[Part].From, Parts[Part].To, PartNum, Parts[PartNum].From, Parts[PartNum].To);
	return PartNum;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::mergePart(int P1, int P2) {
	if ((Parts[P1].To+1==Parts[P2].From)&&
#ifdef NANDUB
		(Parts[P1].Bitrate==Parts[P2].Bitrate)&&
#endif
		(Parts[P1].SourceNum==Parts[P2].SourceNum))
	{
		// Merge in P1, delete P2
		Parts[P1].To=Parts[P2].To;
		Parts[P2].State=PI_NONE;
		Parts[P1].State=PI_TODO;  // forget sheduling
		log(5,"mergePart, merged part %d and %d",P1,P2);
		return true;
	}
return false;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::pairCompare(int A1, int A2, int B1, int B2) {
	return  (((A1==B1)&&(A2==B2))||
		     ((A1==B2)&&(A2==B1)));

}
////////////////////////////////////////////////////////////////////
void PartSheduler::tryMergeParts() {
	if (fSingleMode) return; 
	bool Merged;
	do {
		Merged=false;

		int *Num,Cnt,i;		
		Cnt=GetPartNumbers(&Num,PI_TODO+PI_SHEDULED);
		for (i=0;i<Cnt;i++)
			Parts[Num[i]].State=PI_TODO;	// forget sheduling

		// Search Partpair with p[n].To+1==p[n+1].From, same bitrate(NANDUB), same sourcefile
		for (i=0;i<(Cnt-1);i++) {
			Merged=mergePart(Num[i],Num[i+1]);
			if (Merged) break;
		}
		free(Num);
	} while (Merged);
	recalcFrameCount();
}

////////////////////////////////////////////////////////////////////
void PartSheduler::recalcFrameCount() {
	iFrames=iOpenFrames=iValidPartCount=0;
	int *Num,Cnt;
	Cnt=GetPartNumbers(&Num,PI_VALID);

	for (int i=0;i<Cnt;i++) {
		switch (Parts[Num[i]].State) {
		case PI_NONE: break;
		case PI_SHEDULED:
		case PI_TODO: iOpenFrames+=(Parts[Num[i]].To-Parts[Num[i]].From+1); ++iValidPartCount;
		case PI_PROGRESS: 
		case PI_DONE: iFrames+=(Parts[Num[i]].To-Parts[Num[i]].From+1);
		}
		Parts[Num[i]].OutFrom=(i==0) ? 0 : Parts[Num[i-1]].OutTo+1;
		Parts[Num[i]].OutTo=Parts[Num[i]].OutFrom+Parts[Num[i]].To-Parts[Num[i]].From;
	}
	free(Num);
}

#ifdef NANDUB
////////////////////////////////////////////////////////////////////
void PartSheduler::correctBitrate(int Bitrate,int Keyframe, int CreditsBitrate, int CreditsStart) {
	recalcFrameCount();
	if ((CreditsStart==0)||(CreditsStart>=iFrames)) {
		for (int i=0;i<iPartCount;i++) 
			if (Parts[i].State==PI_TODO) {
				Parts[i].Bitrate=Bitrate;
				Parts[i].Keyframe=Keyframe;
			}
		log(5,"correctBitRate, simple case, Bitrate=%d, Keyframe=%d",Bitrate,Keyframe);
		return;
	}

	int *Numbers,Cnt;
	Cnt=GetPartNumbers(&Numbers,PI_VALID);

	// Calculate real bitrate, 
		// (iFrames-iCreditsStart=number of Creditframes, CreditsStart=number of normal frames
		// at 1 fps:	filesize	=	BitRate*iFrames,
		//				creditssize	=	CreditsBitrate*(iFrames-CreditsStart)
		//				NewBitrate	=	(filesize-creditsize)/CreditsStart
	log(5,"correctBitrate, old bitrate=%d, Credisbitrate=%d, CreditsState=%d",Bitrate,CreditsBitrate,CreditsStart);
	Bitrate=MulDiv(Bitrate,iFrames,CreditsStart)-MulDiv(CreditsBitrate,(iFrames-CreditsStart),CreditsStart);
	log(5,"correctBitrate, new bitrate=%d",Bitrate);

	// Determine real framenumber of CreditsStart
	int df=0,ef=-1;
	for (int i=0;i<Cnt;i++) {
		df+=Parts[Numbers[i]].From-ef-1;
		if ((CreditsStart>=Parts[Numbers[i]].From-df)&&
			(CreditsStart<=Parts[Numbers[i]].To-df)) break;
		ef=Parts[Numbers[i]].To;
	}
	CreditsStart+=df;

	// Set all undivided parts
	for (i=0;i<Cnt;i++) 
		if (Parts[Numbers[i]].State==PI_TODO) {
			Parts[Numbers[i]].Keyframe=Keyframe;
			if (Parts[Numbers[i]].To<CreditsStart)
				Parts[Numbers[i]].Bitrate=Bitrate;
			if (Parts[Numbers[i]].From>=CreditsStart)
				Parts[Numbers[i]].Bitrate=CreditsBitrate;
		}

	// Search part and divide (if found)
	for (i=0;i<Cnt;i++)
		if ((Parts[Numbers[i]].State==PI_TODO)&&
			(Parts[Numbers[i]].From<CreditsStart)&&
			(Parts[Numbers[i]].To>=CreditsStart)) {

			int PartNum=getFreePartNum();

			Parts[PartNum]=Parts[Numbers[i]];

			Parts[Numbers[i]].To=CreditsStart-1;
			Parts[Numbers[i]].Bitrate=Bitrate;

			Parts[PartNum].From=CreditsStart;
			Parts[PartNum].Bitrate=CreditsBitrate;
			log(3,"correctBitrate, divided Part %d, new Part= %d",Numbers[i],PartNum);
			break;
		}
	free(Numbers);
	recalcFrameCount();
	savePartInfo();
}
#endif

#ifdef NANDUB
////////////////////////////////////////////////////////////////////
bool PartSheduler::checkStatsFile(char *filename) {
	FILE *f;
	bool err=false;
	int version;
	const int size=(sizeof(DWORD)+sizeof(frame_info)+sizeof(double));
	DWORD buffer[size/4];

	log(3,"checkStatsFile called (%s)",filename);

	// open statsfile
	if (!(f=fopen(filename,"r+b" ))) {
		log(1,"checkStatsFile, could not open file");
		return false; // File not found
	}

	fread(&version,sizeof(DWORD),1,f);
	if (version!=-20) {  // wrong version. See Dub.cpp 
		log(1,"checkStatsFile, wrong fileversion (%d)",version);
		fclose(f); 
		return false; 
	}			
	
	int frame=0, blstart=-1, blend;
	while (!feof(f)) {
		fread(buffer,size,1,f);
		frame++;
		int i,zcnt=0;
		for (i=0;i<15;i++) if (!buffer[i]) zcnt++;
		if (zcnt>8) {	// more than 6 zerofields -> may be an error
			if (blstart==-1)  // Blockstart	  
				blstart=frame;
			blend=frame;
		} 
		if ((zcnt<=6)||(feof(f)))
			if (blstart!=-1) {
				if (blend-blstart>750) {
					log(1,"checkStatsFile, Errorblock found: %d-%d",blstart,blend);
					err=true;
				}
				blstart=-1;
			}
	}
	fclose(f);
	if (!err) log(5,"checkStatsFile, no errors found, %d frames", frame);
	return !err;
}

////////////////////////////////////////////////////////////////////
#define __overlap(a,b,x,y) \
	((a<y)&&(b>x))
	
void PartSheduler::tryRebuildStatsfile() {
	FILE *f;
	bool err=false;
	int version;
	const int size=(sizeof(DWORD)+sizeof(frame_info)+sizeof(double));
	DWORD buffer[size/4];

	// open statsfile
	if (!(f=fopen(statsfile,"r+b" ))) {
		log(1,"tryRebuildStatsfile, could not open file");
		err=true;
	}

	if (!err) {
		fread(&version,sizeof(DWORD),1,f);
		if (version!=-20) {  // wrong version. See Dub.cpp 
			log(1,"tryRebuildStatsfile, wrong fileversion (%d)",version);
			err=true;
			fclose(f);
		}			
	}

	if (err) {	// no valid statsfile->all parts TODO
		int *Num,Cnt;
		Cnt=GetPartNumbers(&Num,PI_VALID);
		for (int i=0;i<Cnt;i++)
			Parts[Num[i]].State=PI_TODO;
		free(Num);
		return;
	}
	recalcFrameCount();
	int frame=0, blstart=-1, blend;
	while (!feof(f)) {
		fread(buffer,size,1,f);
		int i,zcnt=0;
		for (i=0;i<15;i++) if (!buffer[i]) zcnt++;
		if (zcnt>6) {	// more than 6 zerofields -> may be an error
			if (blstart==-1)  // Blockstart
				blstart=frame;
			blend=frame;
		} else {
			if (blstart!=-1) {
				if (blend-blstart>200) {	// more than 200 "may-be-errors"->real error...
					log(1,"checkStatsFile, Errorblock found: %d-%d",blstart,blend);

					int *Num,Cnt,NewPart;
					Cnt=GetPartNumbers(&Num,PI_VALID);
					for (int i=0;i<Cnt;i++) {
						if (__overlap(blstart,blend, Parts[Num[i]].OutFrom, Parts[Num[i]].OutTo)) {
							// split left? (if >1000 Frames)
							if (blstart-1000>Parts[Num[i]].OutFrom) {
								NewPart=splitPart(Num[i],blstart-Parts[Num[i]].OutFrom);
								Num[i]=NewPart;
								recalcFrameCount();
								log(5,"checkStatsFile, Block divided (left)");
							}
							// split right?
							if (blend+1000<Parts[Num[i]].OutTo) {
								NewPart=splitPart(Num[i],blend-Parts[Num[i]].OutFrom);
								recalcFrameCount();
								log(5,"checkStatsFile, Block divided (right)");
							}
							Parts[Num[i]].State=PI_TODO;
						}
					}
					free(Num);
					err=true;
				}
				blstart=-1;
			}
		}
		frame++;
	}
	fclose(f);
	tryMergeParts();
	if (!err) log(5,"checkStatsFile, no errors found");
}

////////////////////////////////////////////////////////////////////
void PartSheduler::storeStatsFilePart(int Num) {
	if ((!*statsfile)||(!fGenStats)||(fSingleMode)) return;
	FILE *f,*g;
	int version;
	const int size=(sizeof(DWORD)+sizeof(frame_info)+sizeof(double));
	char buffer[size],name[128];

	log(3,"storeStatsFilePart(%d) called",Num);

	// Check, open or create statsfile
	if (!(f=fopen(statsfile,"r+b" ))) { // File not found -> create
		if (!(f=fopen(statsfile,"w+b"))) return; // something really went wrong
		version=-20;
		fwrite(&version,4,1,f);
		memset(buffer,0,size);
		for (int i=0;i<iFrames;i++)
			fwrite(buffer,size,1,f);
	}

	// Open and check statspartfile
	strcpy(name,statsfile);
	ModifyFileName(Num,name);
	if (!(g=fopen(name,"rb"))) { fclose(f);return; }	// file not found

	fread(&version,sizeof(DWORD),1,g);
	if (version!=-20) { fclose(f); return; }			// wrong version. See Dub.cpp 

	int *Numbers,Cnt,Pos=0;
	Cnt=GetPartNumbers(&Numbers,PI_VALID);
	
	for (int i=0;i<Cnt;i++) {
		if (Num!=Numbers[i]) { // Calculcate real framenumber
			Pos+=(Parts[Numbers[i]].To-Parts[Numbers[i]].From+1);
		} else {	// copy file
			fseek(f,4+size*Pos,SEEK_SET);
	
			for (int j=0;j<Parts[Num].To-Parts[Num].From+1;j++) {
				fread(buffer,size,1,g);
				fwrite(buffer,size,1,f);
			}
			log(5,"storeStatsFilePart, Part %d saved, first frame=%d, from %s, to %s",Numbers[i],Pos,name,statsfile);
			break;
		}
	}
	free(Numbers);
	fclose(g);
	fclose(f);
	remove(name);
}
#endif

#ifdef NANDUB
////////////////////////////////////////////////////////////////////
void PartSheduler::createStatsFilePart(int Num) {
	// copy stats-information for one part from statsfile

	FILE *f,*g;
	char name[130];
	int version;

	log(3,"createStatsFilePart called");

	if ((!*statsfile)||(!fUseStats)||(fSingleMode)) return;

	if (!(f=fopen(statsfile,"rb" ))) return;
	fread(&version,sizeof DWORD,1,f);
	if (version!=-20) { fclose(f); return; }

	int *Numbers,Cnt,Pos=0;
	Cnt=GetPartNumbers(&Numbers,PI_VALID);
	for (int i=0;i<Cnt;i++) {
		if (Num!=Numbers[i]) { // Calculcate real framenumber
			Pos+=(Parts[Numbers[i]].To-Parts[Numbers[i]].From+1);
		} else {	// create and copy file
			const int size=(sizeof(DWORD)+sizeof(frame_info)+sizeof(double));
			char buffer[size];
			fseek(f,4+size*Pos,SEEK_SET);

			// create file
			strcpy(name,statsfile);
			ModifyFileName(Num,name);
			if (!(g=fopen(name,"wb"))) { free(Numbers);fclose(f);return; }

			version=-20;
			fwrite(&version,4,1,g);
			for (int j=0;j<Parts[Num].To-Parts[Num].From+1;j++) {
				fread(buffer,size,1,f);
				fwrite(buffer,size,1,g);
			}
			fclose(g);
			log(5,"createStatsFilePart, file for part %d created, first frame=%d, from %s, to %s",Numbers[i],Pos,statsfile,name);

			break;
		}	// else
	}
	free(Numbers);
	fclose(f);
}
#endif

#ifdef NANDUB
////////////////////////////////////////////////////////////////////
void PartSheduler::removeStatsFilePart(int Num) {
	if ((fSingleMode)||(!fUseStats)) return;
	if (Parts[Num].State!=PI_DONE) return;
	if (!*statsfile) return;

	char name[128];
	strcpy(name,statsfile);
	ModifyFileName(Num,name);
	remove(name);

	log(3,"removeStatsFilePart, Statsfile for part %d erased (filename:%s)",Num,name);
}
#endif

////////////////////////////////////////////////////////////////////
void PartSheduler::getUniquePartfileName(char *name) {
	// Nandub: Video.avi --> Video.XYZ.parts (X=Output?2:1, Y=UseStats?2:1, Z=GetStats?2:1)
	// VirtualDub: Video.avi --> Video.parts
	char lbuf[LINELEN]="";
	*name=0;

#ifdef NANDUB
	if (fGenOutput) {
		ScriptSearchLine(origscript,"VirtualDub.SaveAVI",lbuf);					// one-pass-mode
		ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",lbuf);
	}
	if (fGenStats) ScriptSearchLine(origscript,"VirtualDub.video.GenStats",lbuf);			// two-pass, first pass
	if (fUseStats) ScriptSearchLine(origscript,"VirtualDub.video.SetCurveFile",lbuf);		// two-pass, second pass
#else
	ScriptSearchLine(origscript,"VirtualDub.SaveAVI",lbuf);
	ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",lbuf);
#endif

	if (!lbuf[0]) return;		// Oooops

	getStrParamFromString(lbuf,name);
	strUnCify(name);
	strcpy(strrchr(name,'.'),".parts");

#ifdef NANDUB
	int UniNum=(fGenStats?2:1)+(fUseStats?20:10)+(fGenOutput?200:100);		// single mode: 211, 1st pass: 112, 2nd pass: 221
	ModifyFileName(UniNum,name);
#endif
}

////////////////////////////////////////////////////////////////////
void PartSheduler::savePartInfo() {
	if (!origscript) return;

	char filename[_MAX_PATH*4];
	FILE *F;

	getUniquePartfileName(filename);
	if (!*filename) return;

	if (!(F=fopen(filename,"wt"))) return;

	log(5,"savePartInfo, saved in file %s",filename);

	//first line: Inputfile
	ScriptSearchLine(origscript,"VirtualDub.Open",filename);	
	getStrParamFromString(filename);
	strUnCify(filename);
	fprintf(F,"Input:%s\n",filename);

	//second line: Outputfile
	ScriptSearchLine(origscript,"VirtualDub.SaveAVI",filename);	
	ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",filename);	
	getStrParamFromString(filename);
	strUnCify(filename);
	fprintf(F,"Output:%s\n",filename);

	//third line: Options
	fprintf(F,"Options: Audio=%d\n",g_JobsDlgConfig.ProcessAudio);


	for (int i=0;i<iPartCount;i++) {
		if (Parts[i].State!=PI_NONE)
#ifdef NANDUB
			fprintf(F,"%d,%d,%d,%d,%d,%d,%d\n",i,Parts[i].SourceNum,Parts[i].From,Parts[i].To,Parts[i].State,Parts[i].Bitrate,Parts[i].Keyframe);
#else
			fprintf(F,"%d,%d,%d,%d,%d\n",i,Parts[i].SourceNum,Parts[i].From,Parts[i].To,Parts[i].State);
#endif
	}
	fclose(F);

} 

////////////////////////////////////////////////////////////////////
bool PartSheduler::loadPartInfo(){

	char filename[128],lbuf[LINELEN];
	FILE *F;

	if (g_JobsDlgConfig.RestartJobs) return false;

#ifdef NANDUB
	if ((fGenStats)||(fUseStats)) {
		// check if statsfile exists
		if (!*statsfile) return false;
		if (_access(statsfile,0)) return false;		// file not found or no access
	}
#endif

	getUniquePartfileName(filename);
	if (!*filename) return false;

	if (!(F=fopen(filename,"rt"))) return false;

	// check first line: Inputfile
	strcpy(lbuf,"Input:");
	int len=strlen(lbuf);
	ScriptSearchLine(origscript,"VirtualDub.Open",&lbuf[len]);	
	getStrParamFromString(&lbuf[len]);
	strUnCify(lbuf);
	fgets(filename,128,F);
	if (filename[strlen(filename)-1]=='\n') filename[strlen(filename)-1]=0;
	if (strcmp(lbuf,filename))	{ fclose(F); return false; }

	// check second line: Outputfile
	strcpy(lbuf,"Output:");
	len=strlen(lbuf);
	ScriptSearchLine(origscript,"VirtualDub.SaveAVI",&lbuf[len]);	
	ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",&lbuf[len]);	
	getStrParamFromString(&lbuf[len]);
	strUnCify(lbuf);
	fgets(filename,128,F);
	if (filename[strlen(filename)-1]=='\n') filename[strlen(filename)-1]=0;
	if (strcmp(lbuf,filename))	{ fclose(F); return false; }

	// check third line: Options
	bool ProcessAudio;
	if (1!=fscanf(F,"Options: Audio=%d\n",&ProcessAudio)) { fclose(F); return false; }
	g_JobsDlgConfig.ProcessAudio=ProcessAudio;	// Audio-Settings
	SetMenuEntries(true);

	// File seems ok....
	getUniquePartfileName(filename);

	log(5,"loadPartInfo, all checks successful, filename:%s",filename);

	for (int i=0;i<iPartCount;i++) {
		memset(&Parts[i],0,sizeof ShedulePartInfo);
		Parts[i].Host=PI_NONE;
		Parts[i].State=PI_NONE;
	}

	while (!(feof(F))) {
#ifdef NANDUB
		int a,b,c,d,e,f,g;
		if (7==fscanf(F,"%d,%d,%d,%d,%d,%d,%d\n",&a,&b,&c,&d,&e,&f,&g)) {
			Parts[a].SourceNum=b;
			Parts[a].From=c;
			Parts[a].To=d;
			Parts[a].State=e;
			Parts[a].Bitrate=f;
			Parts[a].Keyframe=g;
			if (Parts[a].State==PI_PROGRESS) Parts[a].State=PI_TODO;
			if (Parts[a].State==PI_SHEDULED) Parts[a].State=PI_TODO;
		} 
#else  // VirtualDub, 5 Values
		int a,b,c,d,e;
		if (5==fscanf(F,"%d,%d,%d,%d,%d\n",&a,&b,&c,&d,&e)) {
			Parts[a].SourceNum=b;
			Parts[a].From=c;
			Parts[a].To=d;
			Parts[a].State=e;
			if (Parts[a].State==PI_PROGRESS) Parts[a].State=PI_TODO;
		} 
#endif
	}
	fclose(F);
	tryMergeParts();
	recalcFrameCount();

#ifdef NANDUB
	if (fGenStats)
		tryRebuildStatsfile();
#endif

	return true;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::testValidPathInfo(char *filename) {
	if ((!filename)||(!filename[0])) return false;

	char path[256],*file;

	if (GetFullPathName(filename, sizeof path, path, &file)<1)
		return false;

	if (strncmp(filename,path,strlen(path))!=0)
		return false;

	return true;
}
////////////////////////////////////////////////////////////////////
bool PartSheduler::setInputFileInfo() {
	if ((fSingleMode)||(!workscript)) return false;

	if (IFInfo) free(IFInfo);
	IFInfo=NULL;
	iInputFiles=0;

	char lbuf[LINELEN];

	strcpy(tmpscript,workscript);


	for (;;) {
		if (ScriptSearchLine(tmpscript,"VirtualDub.Open",lbuf)) {
			ScriptRemoveLine(tmpscript,"VirtualDub.Open");
			IFInfo=(InputFileInfo*)realloc(IFInfo,(iInputFiles+1)*sizeof(InputFileInfo));

			getStrParamFromString(lbuf,IFInfo[iInputFiles].filename);
			++iInputFiles;
		} else
			if (ScriptSearchLine(tmpscript,"VirtualDub.Append",lbuf)) {
				ScriptRemoveLine(tmpscript,"VirtualDub.Append");
				IFInfo=(InputFileInfo*)realloc(IFInfo,(iInputFiles+1)*sizeof(InputFileInfo));

				getStrParamFromString(lbuf,IFInfo[iInputFiles].filename);
				++iInputFiles;
			} else
				break;
	}

	log(5,"SetInputFileInfo, found %d inputfiles",iInputFiles);

	AVIFileInit();    

	for (int i=0;i<iInputFiles;i++) {
		PAVIFILE f;
		AVIFILEINFO afiInfo;
		bool fail=false;
		char filename[256];
		strcpy(filename,IFInfo[i].filename);
		strUnCify(filename);
		DWORD dwResult;

		if (dwResult=AVIFileOpen(&f, filename, OF_SHARE_DENY_NONE|OF_READ, NULL))
			fail=true;

		if ((!fail)&&(dwResult=AVIFileInfo(f,&afiInfo,sizeof(afiInfo)))) {
			AVIFileRelease(f);
			fail=true;
		}

		if (i==0) {
			IFInfo[i].From=0;
			IFInfo[i].msStartTime=0;
		} else {
			IFInfo[i].From=IFInfo[i-1].To+1;
			IFInfo[i].msStartTime=IFInfo[i-1].msStartTime+IFInfo[i-1].msLength;
		}

		if (fail) {
			IFInfo[i].To=IFInfo[i].From-1;
			IFInfo[i].msLength=0;
			log(5,"SetInputFileInfo, File %d: FAILED Startframe=%d, Endframe=%d, Filename=%s",i,IFInfo[i].From,IFInfo[i].To,filename);
		} else {
			g_FramesPerSec=afiInfo.dwRate/afiInfo.dwScale;
			IFInfo[i].To=IFInfo[i].From+afiInfo.dwLength-1;
			IFInfo[i].msLength=MulDiv(afiInfo.dwLength,1000*afiInfo.dwScale,afiInfo.dwRate);
			AVIFileRelease(f);
			log(5,"SetInputFileInfo, File %d: Startframe=%d, Endframe=%d, Filename=%s",i,IFInfo[i].From,IFInfo[i].To,filename);
		}

	}

	AVIFileExit();
	return true;
							
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////
int PartSheduler::GetFrameCount() {
	recalcFrameCount();
	return iFrames;
}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetFrameState(int Part, int Progress) {
	switch (Parts[Part].State) {
		case PI_SHEDULED:
		case PI_TODO: return FT_TODO;
		case PI_DONE: return FT_DONE;
		case PI_PROGRESS: {
			if ( Progress > Parts[Part].Progress )
				return FT_DOING_TODO;
			else
				return FT_DOING_DONE;
						  }
	}
	return FT_TODO;
}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetPartCount() {
	return iPartCount;
}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetPartFirstFrame(int Part) {
	if (Parts[Part].State!=PI_NONE)
		return Parts[Part].OutFrom;
	else
		return 0;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::IsBorderSheduled(int P1, int P2) {

	if ((Parts[P1].State==PI_SHEDULED)&&
		(Parts[P2].State==PI_SHEDULED)&&
		(Parts[P1].To+1==Parts[P2].From)&&
#ifdef NANDUB
		(Parts[P1].Bitrate==Parts[P2].Bitrate)&&
#endif
		(Parts[P1].SourceNum==Parts[P2].SourceNum))
		return true;
	return false;
}

int PartSheduler::GetPartState(int Part) {
	return Parts[Part].State;
}


////////////////////////////////////////////////////////////////////
bool PartSheduler::ModifyFileName(int Num, char *filename) {  // file.ext -> file.<Num>.ext
	char num[10];
	sprintf(num,".%d",Num);
	if (char *t=strrchr(filename,'.')) {
		memmove(t+strlen(num),t,strlen(t)+1);
		memmove(t,num,strlen(num));
		return true;
	}
	return false;
}

////////////////////////////////////////////////////////////////////
void PartSheduler::SetActiveHost(int Host, bool Active) {
	if (fSingleMode) return;
	
	if (Hosts[Host].Active!=Active) {
		iActiveHosts+=(Active)?1:-1;
		Hosts[Host].Active=Active;
	}
}
////////////////////////////////////////////////////////////////////
void PartSheduler::Clear() {
	for (int i=0;i<iPartCount;i++) {
		memset(&Parts[i],0,sizeof ShedulePartInfo);
		Parts[i].Host=PI_NONE;
		Parts[i].State=PI_NONE;
	}

	for (i=0;i<MAXHOSTCOUNT;i++)  {
			Hosts[i].FPS=Hosts[i].tmpFPS=Hosts[i].Frames=0;
			Hosts[i].startTime=Hosts[i].stopTime=0;
			Hosts[i].Active=false;
	}

	if (workscript) { 
		free(workscript);
		workscript=NULL;
	}
	if (origscript) { 
		free(origscript);
		origscript=NULL;
	}

	if (IFInfo) {
		free(IFInfo);
		IFInfo=NULL;
	}
	iInputFiles=NULL;

#ifdef NANDUB
	*statsfile=0;

	fUseStats=fGenStats=fGenOutput=false;
#endif

	iActiveHosts=0;

	log(3,"Clear called, object reset");

	recalcFrameCount();
}

////////////////////////////////////////////////////////////////////
void PartSheduler::AddBlock(int From, int To) {

	if ((From==-1)||(To==-1)) {	// Blocksize unknown, use whole file... 
		From=0;
		To=IFInfo[iInputFiles-1].To;
	}

	int i, PartNum, StartSrc=0,EndSrc=iInputFiles-1;
	for (i=0; i<iInputFiles; i++) {
		if ((From>=IFInfo[i].From)&&(From<=IFInfo[i].To))
			StartSrc=i;
		if ((To>=IFInfo[i].From)&&(To<=IFInfo[i].To))
			EndSrc=i;
	}

	log(3,"AddBlock, Frames From: %d, To: %d, Sourcefile: %d-%d", From, To, StartSrc, EndSrc);
	
	for (i=StartSrc; i<=EndSrc; i++) {
		PartNum=getFreePartNum();
		if (PartNum<0) return;
		Parts[PartNum].Host=PI_NONE;
		Parts[PartNum].Progress=0;
		Parts[PartNum].State=PI_TODO;

		Parts[PartNum].From= (i==StartSrc) ? From : IFInfo[i].From;
		Parts[PartNum].To= (i==EndSrc) ? To : IFInfo[i].To;

		Parts[PartNum].SourceNum=i;

#ifdef NANDUB
		Parts[PartNum].Bitrate=0;
		Parts[PartNum].Keyframe=0;
#endif
		log(5,"AddBlock, block %d added, From: %d, To: %d, Sourcefile: %d", PartNum, Parts[PartNum].From, Parts[PartNum].To, i);

	}

	recalcFrameCount();
	tryMergeParts();
}

int *BestSol, *ActSol, *MaxParts, *PartSize, BestFree, calls;


////////////////////////////////////////////////////////////////////
void PartSheduler::rekursiveSheduleBlock(int BlockFree, int HostCount, int ChangePos, int depth) {

	if (calls>3000000) return;
	++calls;
	// Change
	if (ChangePos>=0) {
		ActSol[ChangePos]++;
		BlockFree-=PartSize[ChangePos];
	}

	// Check
	if (abs(BlockFree)<abs(BestFree)) {
		log(7,"RekursiveSheduleBlock, BlockFree=%d, Best=%d, Host=%d, Depth=%d, %d Calls", BlockFree, BestFree, ChangePos, depth, calls);
		memcpy(BestSol, ActSol, HostCount*4);
		BestFree=BlockFree;
	}

	// Rekurse
	for (int i=0;i<HostCount;++i) {
//		int j=(i+ChangePos+1)%HostCount;
		int j=i;
		if (//(abs(BestFree)>BLOCKFRAMECOUNT/10)&&
			(ActSol[j]<MaxParts[j])&&(BlockFree>(-BLOCKFRAMECOUNT/2))) {
			rekursiveSheduleBlock(BlockFree, HostCount, j, depth+1);
		}
	}

	// Unchange
	if (ChangePos>=0) {
		ActSol[ChangePos]--;
		BlockFree+=PartSize[ChangePos];

	}
}
////////////////////////////////////////////////////////////////////
void PartSheduler::SheduleJob() {

	char logstr[8192],strnum[16];
	recalcFrameCount();
	if (iValidPartCount==0) return;
	if (iActiveHosts<1) return;
	int ValidHosts[MAXHOSTCOUNT];
	int HostCount=0;

	int *Num,Cnt,i,j;

	// find active Hosts
	for (i=0;i<MAXHOSTCOUNT;i++)
		if (Hosts[i].Active) 
			ValidHosts[HostCount++]=i;

	// 1. only one host
	if (HostCount==1) {
		tryMergeParts();
		Cnt=GetPartNumbers(&Num,PI_TODO);
		int Host=ValidHosts[0];

		for (i=0;i<Cnt;i++) {
			int BlockSize=Parts[Num[i]].To-Parts[Num[i]].From+1;
			int BlockCount=__max(BlockSize/BLOCKFRAMECOUNT,1);
			int PartSize=BlockSize/BlockCount+1;
			Parts[Num[i]].State=PI_SHEDULED;
			Parts[Num[i]].Host=Host;
			int Part=-1,NewPart=Num[i];
			while (NewPart!=Part) {
				Part=NewPart;
				NewPart=splitPart(Part,PartSize);
			}
		}
		recalcFrameCount();
		free(Num);
		return;
	}

	// > 1 host
 	int SumSpeed=0,SumFrames=iOpenFrames,LocalHost=netMan->GetLocalHost();
	
	// first loop: get overall values, init running jobs, O(n)
	for (i=0;i<HostCount;i++) {
		int PartNum=getHostPart(ValidHosts[i]);
		SumSpeed+=Hosts[ValidHosts[i]].Speed;
		if (PartNum>=0) {
			Hosts[ValidHosts[i]].framesToGo=MulDiv( Hosts[ValidHosts[i]].Frames , (1000-Parts[PartNum].Progress) , 1000);
			SumFrames+=Hosts[ValidHosts[i]].framesToGo;
			Hosts[ValidHosts[i]].ticksToGo=Hosts[ValidHosts[i]].framesToGo/Hosts[ValidHosts[i]].Speed;
		} else {
			Hosts[ValidHosts[i]].framesToGo=0;
			Hosts[ValidHosts[i]].ticksToGo=0;
		}
	}

	// second loop: calc overall time for job, O(n)
	for (i=0;i<HostCount;i++) {
		Hosts[ValidHosts[i]].framesToShedule=__max( MulDiv(SumFrames, Hosts[ValidHosts[i]].Speed, SumSpeed)-Hosts[ValidHosts[i]].framesToGo, 0);
	}

	// Check sheduling...
	bool ReShedule=false;
	Cnt=GetPartNumbers(&Num,PI_SHEDULED);
	for (i=0;i<HostCount;i++) {
		int SheduledFrames=0;
		for (j=0;j<Cnt;j++) {
			if (Parts[Num[j]].Host==ValidHosts[i])
				SheduledFrames+= (Parts[Num[j]].To- Parts[Num[j]].From +1);
		}
		if (abs(SheduledFrames-Hosts[ValidHosts[i]].framesToShedule)>30*25)
			ReShedule=true;
	}
	free(Num);
	if (!ReShedule) {
		log(5,"SheduleJob, no sheduling needed");
		return;
	}

	tryMergeParts();		// forget older sheduling
	Cnt=GetPartNumbers(&Num,PI_TODO);
	for (i=0;i<Cnt;i++) {
		Parts[Num[i]].BlockSize=Parts[Num[i]].To-Parts[Num[i]].From+1;
		Parts[Num[i]].FramesSheduled=0;
		Parts[Num[i]].PartCount=__max(((Parts[Num[i]].BlockSize)/BLOCKFRAMECOUNT),1);
		Parts[Num[i]].PartSize=Parts[Num[i]].BlockSize/Parts[Num[i]].PartCount;
	}
	free(Num);

	strcpy(logstr,"SheduleJob, ToShedule:\t");
	for (i=0;i<HostCount;i++) {sprintf(strnum, "%d\t", Hosts[ValidHosts[i]].framesToShedule);strcat(logstr, strnum);}
	log(7,logstr);


	for (;;) {
		Cnt=GetPartNumbers(&Num,PI_TODO);
		if (Cnt==0) break;

		int BlockSize=Parts[Num[0]].To-Parts[Num[0]].From+1;
		BestSol =(int*)calloc(HostCount,4);
		ActSol  =(int*)calloc(HostCount,4);
		MaxParts=(int*)calloc(HostCount,4);
		PartSize=(int*)calloc(HostCount,4);
		BestFree=BlockSize*100;
		calls=0;

		int PartCount=0;

		for (j=0;j<HostCount;j++) {
			PartSize[j]=Hosts[ValidHosts[j]].framesToShedule/ __max( Hosts[ValidHosts[j]].framesToShedule/BLOCKFRAMECOUNT, 1);
			if (PartSize[j]<BLOCKFRAMECOUNT/2) MaxParts[j]=0;
										  else MaxParts[j]=__max(1,__min(BlockSize, Hosts[ValidHosts[j]].framesToShedule) / PartSize[j]);
																 

		PartCount+=MaxParts[j];
		}

		if (PartCount==0) {
			for (j=0;j<HostCount;j++) {
				PartSize[j]=BLOCKFRAMECOUNT;
				MaxParts[j]=1;
			}
		}


		rekursiveSheduleBlock(BlockSize, HostCount, -1);

		// Logging...
		log(3,"SheduleJob, Blocksize=%d, %d Calls", BlockSize, calls);
		strcpy(logstr,"SheduleJob, ToShedule:\t");
		for (j=0;j<HostCount;j++) {sprintf(strnum, "%d\t", Hosts[ValidHosts[j]].framesToShedule);strcat(logstr, strnum);}
		log(7,logstr);
		strcpy(logstr,"SheduleJob, Partsize:\t");
		for (j=0;j<HostCount;j++) {sprintf(strnum, "%d\t", PartSize[j]);strcat(logstr, strnum);}
		log(7,logstr);
		strcpy(logstr,"SheduleJob, MaxParts:\t");
		for (j=0;j<HostCount;j++) {sprintf(strnum, "%d\t", MaxParts[j]);strcat(logstr, strnum);}
		log(7,logstr);
		strcpy(logstr,"SheduleJob, BestSol:\t");
		for (j=0;j<HostCount;j++) {sprintf(strnum, "%d\t", BestSol[j]);strcat(logstr, strnum);}
		log(7,logstr);

// Divide Block
		int SheduleSum=0, SatSum=0;
		bool CanBuffer=false;
		for (i=0;i<HostCount;i++) {
			SheduleSum+=(PartSize[i]*BestSol[i]);
			CanBuffer|=(BestSol[i]<MaxParts[i])&&(BestSol[i]>0);
			if (BestSol[i]==MaxParts[i])
				SatSum+=(PartSize[i]*BestSol[i]);
		}

		if ((CanBuffer)&&(SatSum+BLOCKFRAMECOUNT/2>BlockSize))
			CanBuffer=false;

		log(5,"SheduleJob, BlockFree=%d, Sheduled=%d", BestFree, SheduleSum);

		Parts[Num[0]].State=PI_SHEDULED;
		// find saturated hosts
		for (i=0;i<HostCount;i++) {
			while ((BestSol[i]==MaxParts[i])&&
				   (BlockSize>0)&&
				   (Hosts[ValidHosts[i]].framesToShedule>0)&&
				   (BestSol[i]>0)) {

				int ScaledPartSize;

				if (CanBuffer) {							// no scale
					if (BestSol[i]==1)
						ScaledPartSize=__min(BlockSize, Hosts[ValidHosts[i]].framesToShedule);
					else
						ScaledPartSize=PartSize[i];

				} else {
					ScaledPartSize=MulDiv(PartSize[i], BlockSize, SheduleSum);
					ScaledPartSize=__min(Hosts[ValidHosts[i]].framesToShedule, ScaledPartSize);
					if (BlockSize-ScaledPartSize<BLOCKFRAMECOUNT/2)
						ScaledPartSize=BlockSize;
				}

				ScaledPartSize=__min(BlockSize, ScaledPartSize);

				int NewPart=splitPart(Num[0], ScaledPartSize);
				Parts[Num[0]].Host=ValidHosts[i];

				Hosts[ValidHosts[i]].framesToShedule-=ScaledPartSize;

				if (NewPart==Num[0]) BlockSize=0;
								else BlockSize=Parts[NewPart].To-Parts[NewPart].From+1;

				--BestSol[i];
				--MaxParts[i];
				SheduleSum-=PartSize[i];

				Num[0]=NewPart;	// keep remaining part
			} // while
		} // for(HostCount)

		SheduleSum=0;
		for (i=0;i<HostCount;i++)
			SheduleSum+=(PartSize[i]*BestSol[i]);

		// all other hosts
		for (i=0;i<HostCount;i++) {
			while ((BlockSize>0)&&
				   (BestSol[i]<MaxParts[i])&&
				   (Hosts[ValidHosts[i]].framesToShedule>0)&&
				   (BestSol[i]>0)) {

				int ScaledPartSize=MulDiv(PartSize[i], BlockSize, SheduleSum);

				if (BlockSize-ScaledPartSize<BLOCKFRAMECOUNT/2)
					ScaledPartSize=BlockSize;

				ScaledPartSize=__min(BlockSize, ScaledPartSize);

				int NewPart=splitPart(Num[0], ScaledPartSize);
				Parts[Num[0]].Host=ValidHosts[i];

				Hosts[ValidHosts[i]].framesToShedule-=ScaledPartSize;

				if (NewPart==Num[0]) BlockSize=0;
								else BlockSize=Parts[NewPart].To-Parts[NewPart].From+1;
				--BestSol[i];
				--MaxParts[i];
				SheduleSum-=PartSize[i];

				Num[0]=NewPart;	// keep remaining part
			}
		}

		// bad hack: something remaining?
		if (BlockSize>0) {
			int BestHost=ValidHosts[0];
			for (i=1;i<HostCount;i++)
				if (Hosts[ValidHosts[i]].framesToShedule>Hosts[BestHost].framesToShedule)
					BestHost=ValidHosts[i];
			Parts[Num[0]].Host=BestHost;
			Parts[Num[0]].State=PI_SHEDULED;
			log(5,"SheduleJob, remaining: %d", BlockSize);
			BlockSize=0;
		}
	
		free(BestSol);
		free(ActSol);
		free(MaxParts);
		free(PartSize);
		free(Num);
	}
	free(Num);

}
////////////////////////////////////////////////////////////////////
bool PartSheduler::GetNextPart(int Host) {
	if (fSingleMode) Host=0; 
	if (getHostPart(Host)>=0) return false;

	SetActiveHost(Host);
	SheduleJob();
	
	if (iValidPartCount==0) return false;

	int *Num,Cnt,i; 
	Cnt=GetPartNumbers(&Num,PI_SHEDULED);

	if (Cnt==0) {
		log(3,"GetNextPart(%d), no more parts", Host);
		free(Num);
		return false;			// nothing to do
	}

	for (i=0;i<Cnt;i++)
		if (Parts[Num[i]].Host==Host) {

			int PartSize,MergedHost;
			if (i<Cnt-1) {
				PartSize=Parts[Num[i]].To-Parts[Num[i]].From+1;
				MergedHost=Parts[Num[i+1]].Host;
				if (mergePart(Num[i],Num[i+1])) {
					Num[i+1]=splitPart(Num[i],PartSize);		// scan for silence!
					if (Num[i+1]!=Num[i]) {
						log(5,"GetNextPart(%d), Right border merged and splitted successful", Host);
						Parts[Num[i+1]].State=PI_SHEDULED;
						Parts[Num[i+1]].Host=MergedHost;
					}
				}
			}

			if (i>0) {
				PartSize=Parts[Num[i-1]].To-Parts[Num[i-1]].From+1;
				if (mergePart(Num[i-1],Num[i])) {
					Num[i]=splitPart(Num[i-1],PartSize);		// scan for silence!
					if (Num[i-1]!=Num[i]) {
						log(5,"GetNextPart(%d), Left border merged and splitted successful", Host);
						Parts[Num[i-1]].State=PI_SHEDULED;
						Parts[Num[i]].Host=Host;
					} else
					{
						log(5,"GetNextPart(%d), Right border merged, split failed, restarting loop", Host);
						free(Num);
						Cnt=GetPartNumbers(&Num,PI_SHEDULED);
						i=-1;
						continue;
					}

				}
			}

			Parts[Num[i]].State=PI_PROGRESS;
			Parts[Num[i]].Progress=0;
			
			Hosts[Host].Frames=Parts[Num[i]].To-Parts[Num[i]].From+1;

			log(3,"GetNextPart(%d), part %d returned, range: %d to %d", Host, Num[i], Parts[Num[i]].From, Parts[Num[i]].To);
			free(Num);

			SheduleJob();
			recalcFrameCount();
			savePartInfo();

			return true;
		}

	free(Num);
// 	log("GetNextPart: no sheduled jobs found");
	return false;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::GetPartInfo(int Host, PartInfo *PI) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return false;

	if (PI) {
		PI->From=Parts[PartNum].From;
		PI->To=Parts[PartNum].To;
	}
	return true;
}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetPartFrom(int Host) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return 0;

	return Parts[PartNum].From;
}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetPartTo(int Host) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return 0;

	return Parts[PartNum].To;
}
////////////////////////////////////////////////////////////////////
int PartSheduler::GetPartNumbers(int **Numbers,int Mode) {
	int *Nums=NULL;
	int Cnt=0,i,j;
	for (i=0;i<iPartCount;i++) {
		if ((abs(Parts[i].State)&abs(Mode))!=0) {
			Nums=(int*)realloc(Nums,(Cnt+1)*sizeof(int));
			// Simple Insertion-Sort
			for (j=0;((j<Cnt)&&(Parts[Nums[j]].From<Parts[i].From));j++) {}
			memmove(&Nums[j+1],&Nums[j],(Cnt-j)*sizeof(int) );
			Nums[j]=i;
			++Cnt;
		}
	}
	*Numbers=Nums;
	return Cnt;
}

////////////////////////////////////////////////////////////////////
void PartSheduler::PartSuccess(int Host) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return;

#ifdef NANDUB
	char name[128];
	strcpy(name,statsfile);
	ModifyFileName(PartNum,name);
	if ((fGenStats)&&(!checkStatsFile(name))) {
		log(1,"PartSuccess, statsfile-check failed!");
		PartAborted(Host);
		return;
	}
#endif

	Parts[PartNum].State=PI_DONE;
	Parts[PartNum].Host=PI_NONE;

	Hosts[Host].stopTime=getLocalTime();
	int dt=__max(1,(int)((Hosts[Host].stopTime-Hosts[Host].startTime)/10000000));
	Hosts[Host].tmpFPS=Hosts[Host].FPS=Hosts[Host].Frames/dt;

	if (!fSingleMode) SaveHostSpeed(Host);

	log(3,"PartSuccess, host %d, part %d, time=%ds, speed=%d fps", Host, PartNum, (int)((Hosts[Host].stopTime-Hosts[Host].startTime)/10000000), Hosts[Host].FPS);

	recalcFrameCount();
	savePartInfo();

#ifdef NANDUB
	storeStatsFilePart(PartNum);
	removeStatsFilePart(PartNum);
#endif

}

////////////////////////////////////////////////////////////////////
void PartSheduler::PartAborted(int Host) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return;

	Parts[PartNum].State=PI_TODO;
	Parts[PartNum].Host=PI_NONE;

	log(3,"PartAborted, host %d, part %d", Host, PartNum);

	recalcFrameCount();
	tryMergeParts();

	savePartInfo();
}

////////////////////////////////////////////////////////////////////
void PartSheduler::SetHostProgress(int Host, int Progress) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return;

	if (Parts[PartNum].Progress==0)
		Hosts[Host].startTime=getLocalTime();	// skip startuptime

	Parts[PartNum].Progress=Progress;

	__int64 ti=getLocalTime();
	int diffTi=(int)((ti-Hosts[Host].startTime)/10000000);

	if ((Progress)&&(diffTi))
		Hosts[Host].tmpFPS=MulDiv(Hosts[Host].Frames,Progress,1000*diffTi);

}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetGlobalProgress() {
	if (fSingleMode) {
		return iGlobalProgress;
	}

	if (!workscript) return 0;

	int Frames=0;
	for (int i=0;i<iPartCount;i++) {
		switch (Parts[i].State) {
		case PI_NONE: 
		case PI_TODO: break;
		case PI_DONE: Frames+=(Parts[i].To-Parts[i].From+1); break;
		case PI_PROGRESS: Frames+=MulDiv(Parts[i].To-Parts[i].From+1, Parts[i].Progress, 1000);
		}
	}
	return MulDiv(Frames,1000,iFrames);
}

////////////////////////////////////////////////////////////////////
int PartSheduler::GetHostProgress(int Host) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return 0;
	return Parts[PartNum].Progress;
}

////////////////////////////////////////////////////////////////////
int	PartSheduler::GetHostFPS(int Host) {
	if (fSingleMode) Host=0;
	int PartNum=getHostPart(Host);
	if (PartNum<0) return -1;
	return Hosts[Host].tmpFPS;
}

////////////////////////////////////////////////////////////////////
void PartSheduler::SaveHostSpeed(int Host) {
	if (fSingleMode) return;

	int	LocalHost=netMan->GetLocalHost();
	if (Hosts[LocalHost].FPS==0) return;

	WORD LastRes[5];
	memset(LastRes,0,sizeof(LastRes));

	DWORD IPAd;
	HOSTENT *HostInfo;

	IPAd=ntohl(netMan->GetIP(Host));
	HostInfo=gethostbyaddr((char*)&IPAd,4,AF_INET);
	
	QueryConfigBinary("HostInfo", HostInfo->h_name, (char*)LastRes,sizeof(LastRes));

	memmove(&LastRes[1],&LastRes[0],sizeof(LastRes)-sizeof(LastRes[0]));
	LastRes[0]=MulDiv(Hosts[Host].FPS,100,Hosts[LocalHost].FPS);

	SetConfigBinary("HostInfo", HostInfo->h_name , (char*)LastRes,sizeof(LastRes));
}

////////////////////////////////////////////////////////////////////
int PartSheduler::LoadHostSpeed(int Host) {

	Hosts[Host].Speed=100;

	if (fSingleMode) return 100;

	WORD LastRes[5];
 	memset(LastRes,0,sizeof(LastRes));

	DWORD IPAd;
	HOSTENT *HostInfo;

	IPAd=ntohl(netMan->GetIP(Host));
	HostInfo=gethostbyaddr((char*)&IPAd,4,AF_INET);

	QueryConfigBinary("HostInfo", HostInfo->h_name, (char*)LastRes, sizeof(LastRes));

	int i,cnt,sum;
	for (i=cnt=sum=0;i<5;i++) {
		sum+=LastRes[i];
		if (LastRes[i]) ++cnt;
	}
	if (cnt)
		sum=sum/cnt;
	else
		sum=100;

	Hosts[Host].Speed=sum;
	return sum;
}

////////////////////////////////////////////////////////////////////
void PartSheduler::SetPartInfo(int From, int To) {
	if (!fSingleMode) return;
	Parts[0].Host=0;
	Parts[0].From=From;
	Parts[0].To=To;
	Parts[0].Progress=0;
	Parts[0].State=PI_PROGRESS;
	Parts[0].SourceNum=0;
	recalcFrameCount();
}
////////////////////////////////////////////////////////////////////
void PartSheduler::SetGlobalProgress(int Progress) {
	if (!fSingleMode) return;
	iGlobalProgress=Progress;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::SetOriginalScript(int LocalHost, char *script) {

	log(3,"SetOriginalScript");
	log(5,"   script follows...\n\n%s\nEnd of Script\n\n",script);

	Clear();

	char lbuf[LINELEN];
	bool fFirstSubset=true;

	if (workscript) free(workscript);
	workscript=(char*)calloc(SCRIPTLEN,1);

	if (origscript) free(origscript);
	origscript=(char*)calloc(SCRIPTLEN,1);

#ifdef NANDUB
	*statsfile=0;
#endif

	memcpy(workscript,script,strlen(script)+1);
	memcpy(origscript,script,strlen(script)+1);

#ifdef NANDUB
	fGenOutput=ScriptSearchLine(workscript,"VirtualDub.video.NoAVIOutput(0)",NULL);
#endif

	setInputFileInfo();	

	///////////////////////////////////////////
	// Modify workscript (using token)
	///////////////////////////////////////////

	// Open -> Token
	ScriptReplaceLine(workscript,"VirtualDub.Open","NETWORK-INSERT-OPEN-HERE\n");

	// remove "Append"
	while (ScriptRemoveLine(workscript,"VirtualDub.Append"));

	// Check for subset
#ifdef NANDUB
 #define SUBSETSYNTAX "VirtualDub.subset.AddFrame"
#else
 #define SUBSETSYNTAX "VirtualDub.subset.AddRange"
#endif
	if (ScriptSearchLine(workscript,SUBSETSYNTAX, lbuf))	{
		log(5,"SetOriginalScript, using subsets");
		// Read subset
		while (ScriptSearchLine(workscript,SUBSETSYNTAX,lbuf)) {
			int subFrom,subLen;
			if (get2IntParamFromString(lbuf,&subFrom,&subLen)) {
				AddBlock(subFrom,subFrom+subLen-1);
				if (fFirstSubset) {
					ScriptReplaceLine(workscript,SUBSETSYNTAX,"NETWORK-INSERT-SUBSET-HERE\n");
					fFirstSubset=false;
				} else
					ScriptRemoveLine(workscript,SUBSETSYNTAX);
			}
		}
	} else {  // whole File
		log(5,"SetOriginalScript, using whole file");
		AddBlock(-1,-1);
		ScriptReplaceLine(workscript,"VirtualDub.subset.Delete","VirtualDub.subset.Clear();\nNETWORK-INSERT-SUBSET-HERE\n");
	}

	// Save-statements
	ScriptReplaceLine(workscript,"VirtualDub.SaveAVI","NETWORK-INSERT-SAVENAME-HERE\n");
	ScriptReplaceLine(workscript,"VirtualDub.SaveCompatibleAVI","NETWORK-INSERT-SAVENAMECOMP-HERE\n");

#ifdef NANDUB
	fUseStats=!ScriptSearchLine(workscript,"VirtualDub.video.SetCurveFile(\"\")",NULL);

	if (ScriptSearchLine(workscript,"VirtualDub.video.GenStats",lbuf)) {
		int iGenStats;
		this->get1IntParamFromString(lbuf,&iGenStats);
		fGenStats=(iGenStats!=0);
	}

	if (fUseStats) fGenStats=false;
	if (fGenOutput) fGenStats=false;

	if (fUseStats) {
		ScriptSearchLine(workscript,"VirtualDub.video.SetCurveFile",lbuf);
		getStrParamFromString(lbuf,statsfile);
		strUnCify(statsfile);
		if (testValidPathInfo(statsfile)) 
			ScriptReplaceLine(workscript,"VirtualDub.video.SetCurveFile","NETWORK-INSERT-CURVEFILE-HERE\n");
		else {
			fUseStats=false;
			ScriptRemoveLine(workscript,"VirtualDub.video.SetCurveFile");
			log(1,"SetOriginalScript, fUseStats, wrong statsfilename: %s",statsfile);
		}
	}

	if (fGenStats) {
		ScriptSearchLine(workscript,"VirtualDub.video.GenStats",lbuf);
		getStrParamFromString(lbuf,statsfile);
		strUnCify(statsfile);
		if (testValidPathInfo(statsfile)) {
 			ScriptReplaceLine(workscript,"VirtualDub.video.GenStats","NETWORK-INSERT-STATSFILE-HERE\n");
			if (ScriptSearchLine(origscript,"VirtualDub.SaveAVI",lbuf)||	// remove path from tempfilenames
				ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",lbuf)) {		
				char *s=strchr(lbuf,'\"');
				char *t=strrchr(lbuf,'\\');

				if ((s)&&(t)) {
					memmove(s+1,t+1,strlen(t)+1);
					ScriptReplaceLine(origscript,"VirtualDub.SaveAVI",lbuf);
					ScriptReplaceLine(origscript,"VirtualDub.SaveCompatibleAVI",lbuf);
				}
			}
		} else {
			fGenStats=false;
			ScriptRemoveLine(workscript,"VirtualDub.video.GenStats");
			log(1,"SetOriginalScript, fGenStats, wrong statsfilename: %s",statsfile);
		}
	}

	if ((ScriptSearchLine(workscript,"VirtualDub.video.SetDivX",lbuf))) {
		int DivXOpt1,DivXOpt2,CreditsOpt1=0,CreditsOpt2=0;

		get2IntParamFromString(lbuf,&DivXOpt1,&DivXOpt2);

		if  ((fGenOutput)&&
			 (ScriptSearchLine(workscript,"VirtualDub.video.SetCurveCredits",lbuf)) &&
			(!ScriptSearchLine(workscript,"VirtualDub.video.SetCurveCredits(0,",NULL))) {  // CurveCredits used.
			
			get2IntParamFromString(lbuf,&CreditsOpt1,&CreditsOpt2);

			ScriptReplaceLine(workscript,"VirtualDub.video.SetDivX","NETWORK-INSERT-BITRATE-HERE\n");
			ScriptReplaceLine(workscript,"VirtualDub.video.SetCurveCredits","VirtualDub.video.SetCurveCredits(0,300);\n");
		}
		if (!loadPartInfo())
			correctBitrate(DivXOpt1,DivXOpt2,CreditsOpt2,CreditsOpt1);
	}
#else
	loadPartInfo();		// VirtualDub
#endif

	// Audio
	if (!g_JobsDlgConfig.ProcessAudio) {	
		if ((ScriptSearchLine(workscript,"VirtualDub.audio.SetMode(0)",NULL))&&
			(!ScriptSearchLine(workscript,"VirtualDub.audio.SetSource(0)",NULL))) {
			log(1,"SetOriginalScript, error: audio-mode==direct-stream-copy");
			return false;
		}

		// use audio, recompress to pcm
		ScriptReplaceLine(workscript,"VirtualDub.audio.SetMode","VirtualDub.audio.SetMode(1);\n");
		ScriptReplaceLine(workscript,"VirtualDub.audio.SetCompression","VirtualDub.audio.SetCompression();\n");


	}
	// forget audio2 (Nandub only)
	ScriptReplaceLine(workscript,"VirtualDub.audio2.SetSource","VirtualDub.audio2.SetSource(0);\n");
	ScriptReplaceLine(workscript,"VirtualDub.audio2.SetMode","VirtualDub.audio2.SetMode(0);\n");
	ScriptRemoveLine(workscript,"VirtualDub.audio2.SetCompression");


	// Interleave, audio
	if (ScriptSearchLine(workscript,"VirtualDub.audio.SetInterleave",lbuf)) {
		ScriptReplaceLine(workscript,"VirtualDub.audio.SetInterleave","NETWORK-INSERT-INTERLEAVE-HERE\n");
		int a,b,c,d;
		sscanf(lbuf,"VirtualDub.audio.SetInterleave(%d,%d,%d,%d,%d);\n",&a,&b,&c,&d,&iAudioInterleave);
		sprintf(AudioInterleave,"VirtualDub.audio.SetInterleave(%d,%d,%d,%d,%%d);\n",a,b,c,d);

		ScriptSearchLine(workscript,"VirtualDub.audio.SetSource",lbuf);
 		bExtAudioSource=!((strstr(lbuf,"(0)"))||(strstr(lbuf,"(1)")));
	}

	log(5,"SetOriginalScript, changed script follows...\n\n%s\nEnd of Script\n\n",workscript);

	return true;

}
////////////////////////////////////////////////////////////////////
void PartSheduler::GetModifiedScript(int Host, char *script) {
	if ((!workscript)||(!origscript)) { memset(script,0,128);return; }
	if (fSingleMode) Host=0;
	int StartFrame,PartNum=getHostPart(Host);

	if (PartNum<0) return;
	if (fSingleMode) {
		strcpy(script,workscript);
		return;
	}

	log(3,"GetModifiedScript, host=%d", Host);

	char modscript[SCRIPTLEN],line[LINELEN];
	*script=0;

	strcpy(modscript,workscript);
	StartFrame=IFInfo[Parts[PartNum].SourceNum].From;

// NETWORK-INSERT-OPEN-HERE
	if (ScriptSearchLine(modscript,"NETWORK-INSERT-OPEN-HERE",NULL)) {
		sprintf(line,"VirtualDub.Open(\"%s\",0,0);\n",IFInfo[Parts[PartNum].SourceNum].filename);
		ScriptReplaceLine(modscript,"NETWORK-INSERT-OPEN-HERE",line);
	}

// NETWORK-INSERT-INTERLEAVE-HERE
	if (ScriptSearchLine(modscript,"NETWORK-INSERT-INTERLEAVE-HERE",NULL)) {
		if (bExtAudioSource)
			sprintf(line,AudioInterleave,iAudioInterleave-IFInfo[Parts[PartNum].SourceNum].msStartTime);
		else
			sprintf(line,AudioInterleave,0);
		ScriptReplaceLine(modscript,"NETWORK-INSERT-INTERLEAVE-HERE",line);
	}

// NETWORK-INSERT-SUBSET-HERE --> VirtualDub.subset.AddFrame(<From>,<Count>);
	if (ScriptSearchLine(modscript,"NETWORK-INSERT-SUBSET-HERE",NULL)) {
#ifdef NANDUB
		sprintf(line,"VirtualDub.subset.AddFrame(%d,%d);\n",Parts[PartNum].From-StartFrame,
															Parts[PartNum].To-Parts[PartNum].From+1);
#else
		sprintf(line,"VirtualDub.subset.AddRange(%d,%d);\n",Parts[PartNum].From-StartFrame,
															Parts[PartNum].To-Parts[PartNum].From+1);
#endif

		ScriptReplaceLine(modscript,"NETWORK-INSERT-SUBSET-HERE",line);
		}
	
// NETWORK-INSERT-SAVENAME-HERE --> VirtualDub.SaveAVI("<Filename>");
	if ((ScriptSearchLine(modscript,"NETWORK-INSERT-SAVENAME-HERE",NULL))&&
		(ScriptSearchLine(origscript,"VirtualDub.SaveAVI",line))) {
		if (ModifyFileName(PartNum,line))
			ScriptReplaceLine(modscript,"NETWORK-INSERT-SAVENAME-HERE",line);
	}

// NETWORK-INSERT-SAVENAMECOMP-HERE --> VirtualDub.SaveCompatibleAVI("<Filename>");
	if ((ScriptSearchLine(modscript,"NETWORK-INSERT-SAVENAMECOMP-HERE",NULL))&&
		(ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",line))) {
		if (ModifyFileName(PartNum,line))
			ScriptReplaceLine(modscript,"NETWORK-INSERT-SAVENAMECOMP-HERE",line);
	}
	
// NETWORK-INSERT-CURVEFILE-HERE --> VirtualDub.video.SetCurveFile("<Filename>");
	if ((ScriptSearchLine(modscript,"NETWORK-INSERT-CURVEFILE-HERE",NULL))&&
	    (ScriptSearchLine(origscript,"VirtualDub.video.SetCurveFile",line))) {
		if (ModifyFileName(PartNum,line))
			ScriptReplaceLine(modscript,"NETWORK-INSERT-CURVEFILE-HERE",line);
	}
	
// NETWORK-INSERT-STATSFILE-HERE --> VirtualDub.video.GenStats("<Filename>",1);
	if ((ScriptSearchLine(modscript,"NETWORK-INSERT-STATSFILE-HERE",NULL))&&
		(ScriptSearchLine(origscript,"VirtualDub.video.GenStats",line))) {
		if (ModifyFileName(PartNum,line))
			ScriptReplaceLine(modscript,"NETWORK-INSERT-STATSFILE-HERE",line);
	}
#ifdef NANDUB
// NETWORK-INSERT-BITRATE-HERE
	if (ScriptSearchLine(modscript,"NETWORK-INSERT-BITRATE-HERE",NULL)) {
		sprintf(line,"VirtualDub.video.SetDivX(%d,%d);\n",Parts[PartNum].Bitrate,Parts[PartNum].Keyframe);
		ScriptReplaceLine(modscript,"NETWORK-INSERT-BITRATE-HERE",line);
		}

	if (fUseStats)
		createStatsFilePart(PartNum);
#endif

	Hosts[Host].startTime=getLocalTime();

	strcpy(script,modscript);

	log(5,"GetModifiedScript, script to be executed:\n\n%s\nEnd of Script\n\n",script);
}

////////////////////////////////////////////////////////////////////
void PartSheduler::GetConcatScript(char *script) {
	char *ConScript=(char*)calloc(SCRIPTLEN,1);
	char *tmpscript=(char*)calloc(SCRIPTLEN,1);
	char filename[128],lbuf[LINELEN],line[LINELEN],*t;
	strcpy(tmpscript,origscript);

	log(3,"GetConcatScript called");

	// get order of parts
	int iPP[MAXPARTCOUNT],Cnt=0,i,j;
	for (i=0;i<iPartCount;i++) 
		if (Parts[i].State==PI_DONE) {
			// Simple Insertion-Sort
			for (j=0;((j<Cnt)&&(Parts[iPP[j]].From<Parts[i].From));j++) {}
			memmove(&iPP[j+1],&iPP[j],(Cnt-j)*sizeof(int) );
			iPP[j]=i;
			++Cnt;
		}
	
	// generate Script
	ScriptSearchLine(tmpscript,"VirtualDub.SaveAVI",lbuf);
	ScriptSearchLine(tmpscript,"VirtualDub.SaveCompatibleAVI",lbuf);

	getStrParamFromString(lbuf,filename);		// get filename

	t=strrchr(filename,'.');	// prepare 
	memmove(t+3,t,strlen(t)+1);
	memmove(t,".%d",3);			// filename--> X:\Dir\Name.%d.avi

	// Open first file
	sprintf(lbuf,filename,iPP[0]);
	sprintf(line,"VirtualDub.Open(\"%s\",0,0);\n",lbuf);
	strcat(ConScript,line);

	// Append all other files
	for (i=1;i<Cnt;i++) {
		sprintf(lbuf,filename,iPP[i]);
		sprintf(line,"VirtualDub.Append(\"%s\");\n",lbuf);
		strcat(ConScript,line);
	}


	if (g_JobsDlgConfig.ProcessAudio) {
		
		// First audio-section (VirtualDub.audio)
		while (ScriptSearchLine(tmpscript,"VirtualDub.audio.",lbuf)) {
			if (strstr(lbuf,".audio.SetSource"))
				if (strstr(lbuf,"SetSource(0)"))
					strcat(ConScript,"VirtualDub.audio.SetSource(0);\n");
				else
					strcat(ConScript,"VirtualDub.audio.SetSource(1);\n");
			else if (strstr(lbuf,".audio.SetMode"))
				strcat(ConScript,"VirtualDub.audio.SetMode(0);\n");
			else if (strstr(lbuf,".audio.SetCompression"))
				strcat(ConScript,"VirtualDub.audio.SetCompression();\n");
			else if (strstr(lbuf,".audio.SetInterleave")) {
				if (bExtAudioSource) sprintf(lbuf,AudioInterleave,0);
								else sprintf(lbuf,AudioInterleave,iAudioInterleave);
				strcat(ConScript,lbuf);
			} else strcat(ConScript,lbuf);
			ScriptRemoveLine(tmpscript,"VirtualDub.audio.");
		}

	} else {

		// First audio-section (VirtualDub.audio)
		while (ScriptSearchLine(tmpscript,"VirtualDub.audio.",lbuf)) {
			if (strstr(lbuf,".audio.SetSource"))
				if (strstr(lbuf,"SetSource(0)"))
					strcat(ConScript,"VirtualDub.audio.SetSource(0);\n");
				else
					strcat(ConScript,"VirtualDub.audio.SetSource(1);\n");
			else if (strstr(lbuf,".audio.SetMode"))
				strcat(ConScript,"VirtualDub.audio.SetMode(1);\n");
			else if (strstr(lbuf,".audio.SetInterleave")) {
				if (bExtAudioSource) sprintf(lbuf,AudioInterleave,0);
								else sprintf(lbuf,AudioInterleave,iAudioInterleave);
				strcat(ConScript,lbuf);
			} else strcat(ConScript,lbuf);
			ScriptRemoveLine(tmpscript,"VirtualDub.audio.");
		}

		// no Second audio-section (VirtualDub.audio2)
		while (ScriptSearchLine(tmpscript,"VirtualDub.audio2.",lbuf)) {
			ScriptRemoveLine(tmpscript,"VirtualDub.audio2.");
		}
		strcat(ConScript,"VirtualDub.audio2.SetSource(0);\n");
		strcat(ConScript,"VirtualDub.audio2.SetMode(0);\n");
	
	}

	// Video-Part (VirtualDub.video)
	while (ScriptSearchLine(tmpscript,"VirtualDub.video.",lbuf)) {
		if (strstr(lbuf,".video.SetMode")) 
			strcat(ConScript,"VirtualDub.video.SetMode(0);\n");
		else if (strstr(lbuf,".video.SetRange")) 
			strcat(ConScript,"VirtualDub.video.SetRange(0,0);\n");
		else if (strstr(lbuf,".video.SetCurve"));
		else if (strstr(lbuf,".video.filters.Add"));
		else if (strstr(lbuf,".video.filters.instance"));
		else strcat(ConScript,lbuf);
		ScriptRemoveLine(tmpscript,"VirtualDub.video.");
	}

	// No subset
	while (ScriptRemoveLine(tmpscript,"VirtualDub.subset."));
	strcat(ConScript,"VirtualDub.subset.Delete();\n");
	
	// Complete brc (Nandub only)
	while (ScriptSearchLine(tmpscript,"VirtualDub.brc.",lbuf)) {
		ScriptRemoveLine(tmpscript,"VirtualDub.brc.");
		strcat(ConScript,lbuf);
	}
	
	// Save File
	if (ScriptSearchLine(tmpscript,"VirtualDub.SaveAVI",lbuf)||
		ScriptSearchLine(tmpscript,"VirtualDub.SaveCompatibleAVI",lbuf))
		strcat(ConScript,lbuf);
	
	// Close File
	if (ScriptSearchLine(tmpscript,"VirtualDub.Close",lbuf))
		strcat(ConScript,lbuf);

	strcpy(script,ConScript);

	log(5,"GetConcatScript: returned script:\n\n%s\nEnd of Script\n\n",ConScript);
	free(ConScript);
	free(tmpscript);

}

////////////////////////////////////////////////////////////////////
void PartSheduler::RemoveTempFiles() {

#ifdef NANDUB
	if (!fGenOutput) return;
#endif

	if ((!origscript)||(fSingleMode)) return;

	int *Numbers, Cnt;
	char lbuf[LINELEN],filename[128];
	if (ScriptSearchLine(origscript,"VirtualDub.SaveAVI",lbuf)||
		ScriptSearchLine(origscript,"VirtualDub.SaveCompatibleAVI",lbuf)) {
		getStrParamFromString(lbuf,filename);
		strUnCify(filename);

		Cnt=GetPartNumbers(&Numbers);
		for (int i=0;i<Cnt;i++) {
			strcpy(lbuf,filename);
			ModifyFileName(Numbers[i],lbuf);
			remove(lbuf);
			log(5,"RemoveTempFiles, removed %s",lbuf);
		}

		// remove parts-info-file
		getUniquePartfileName(filename);
		if (*filename) {
			remove(filename);
			log(5,"RemoveTempFiles, removed parts-info-file %s",filename);
		}

	free(Numbers);

	}
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::IsAllPartsComplete() {
	for (int i=0;i<iPartCount;i++) {
		if (Parts[i].State==PI_PROGRESS) return false;
		if (Parts[i].State==PI_TODO) return false;
		if (Parts[i].State==PI_SHEDULED) return false;

	}
	return true;
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::IsHostJob(int Host) {
	if (fSingleMode) Host=0;
	return (getHostPart(Host)>=0);
}

////////////////////////////////////////////////////////////////////
bool PartSheduler::IsAnyHostJob() {
	for (int i=0;i<iPartCount;i++)
		if (Parts[i].State==PI_PROGRESS) return true;
	return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// Same configuration for all sockets													   //
/////////////////////////////////////////////////////////////////////////////////////////////
static void ConfigSocket(SOCKET s)
{
bool opt;
LINGER linger;
linger.l_onoff=1;
linger.l_linger=0;
setsockopt(s,SOL_SOCKET,SO_DONTLINGER,(char*)&linger,sizeof(linger));
setsockopt(s,SOL_SOCKET,SO_REUSEADDR,(char*)&(opt=true),sizeof(opt));
}

////////////////////////////////////////////////////
// close socket, uses windowmessages
////////////////////////////////////////////////////
static void CleanCloseSocket(SOCKET s,HWND hwnd,unsigned int Msg)
{
	WSAAsyncSelect(s,hwnd,Msg,FD_MOSTFLAGS);
	shutdown(s,SD_SEND);
}

////////////////////////////////////////////////////
// Create and configure socket
////////////////////////////////////////////////////
static SOCKET NewSocket(void)
{
	SOCKET s=socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
	ConfigSocket(s);
	return s;
}


////////////////////////////////////////////////////////////////////
NetworkManager::NetworkManager() {
	HostInfo=(NetworkHostInfo*)calloc(MAXHOSTCOUNT*sizeof(NetworkHostInfo),1);
	SetSingleMode();
	for (int i=0;i<MAXHOSTCOUNT;i++) {
		HostInfo[i].MsgBuf=(unsigned char*)calloc(SCRIPTLEN,1);
		HostInfo[i].MsgBufLen=0;
	}
	Clear();
	SetSingleMode();
}

////////////////////////////////////////////////////////////////////
NetworkManager::~NetworkManager() {
	for (int i=0;i<MAXHOSTCOUNT;i++)
		free(HostInfo[i].MsgBuf);

	if (HostInfo) free(HostInfo);
}

////////////////////////////////////////////////////////////////////
void NetworkManager::Clear() {
	for (int i=0;i<MAXHOSTCOUNT;i++) {
		HostInfo[i].Valid=false;
		HostInfo[i].Running=false;
		memset(HostInfo[i].MsgBuf,0,SCRIPTLEN);
		HostInfo[i].MsgBufLen=0;
	}
	HostCount=0;
	LocalHost=0;
	LocalState=HI_LOCALHOST;
}

////////////////////////////////////////////////////////////////////
int NetworkManager::getHostNum(char *HostName) {
	for (int i=0;i<MAXHOSTCOUNT;i++)
		if (strfuzzycompare(HostInfo[i].Name,HostName))
			return i;
	return -1;
}

////////////////////////////////////////////////////////////////////
void NetworkManager::GetTransferInfo(int Host, NetworkTransferInfo *NTI) {
	if (!fSingleMode) {
		HostInfo[Host].From=psSheduler->GetPartFrom(Host);
		HostInfo[Host].To=psSheduler->GetPartTo(Host);
		HostInfo[Host].Progress=psSheduler->GetHostProgress(Host);
		HostInfo[Host].FPS=psSheduler->GetHostFPS(Host);
	}

	memcpy(NTI,&HostInfo[Host],sizeof(NetworkTransferInfo));
}

////////////////////////////////////////////////////////////////////
void NetworkManager::SetTransferInfo(NetworkTransferInfo *NTI) {
	if (LocalState==HI_LOCALHOST) return;

	if (getHostNum(NTI->Name)==-1)
		if (!TryAddHost(NTI->Name)) return;

	SetTransferInfo(getHostNum(NTI->Name), NTI);
}

////////////////////////////////////////////////////////////////////
void NetworkManager::SetTransferInfo(int Host, NetworkTransferInfo *NTI) {
	if (LocalState==HI_LOCALHOST) return;
	if ((Host<0)||(Host>127)) return;
	memcpy(&HostInfo[Host],NTI,sizeof(NetworkTransferInfo));
	HostInfo[Host].Valid=true;
	ListView_Update(GetDlgItem(g_hwndJobs,IDC_HOSTS),Host);
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::SetMasterMode() {
	fSingleMode=false;
	LocalState=HI_MASTER;
	if (!StartMSMode()) {
		SetSingleMode();
		return false;
	}
	return true;
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::SetSlaveMode() {
	fSingleMode=true;
	LocalState=HI_SLAVE;
	if (!StartMSMode()) {
		SetSingleMode();
		return false;
	}
	return true;
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::SetSingleMode() {
	fSingleMode=true;
	for (int i=0;i<MAXHOSTCOUNT;i++) {
		HostInfo[i].Running=false;
		HostInfo[i].FPS=HostInfo[i].From=HostInfo[i].To=HostInfo[i].Progress=0;
	}
	LocalState=HI_LOCALHOST;
	return true;
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::StartMSMode() {
	if (LocalState==HI_LOCALHOST) return false;
	bool Master=(LocalState==HI_MASTER);
	
	if ((VDJob::IsRunInProgress())||(VDJob::IsNetworkJob())) 
		return false;

	if (! CreateListenSocket()) return false;

	if (Master) {
		if (psSheduler) delete psSheduler;
		psSheduler=new PartSheduler(HostCount);
	}

	// Scan for Master/Slaves
	for (int i=0;i<HostCount;i++)
		if (IsType(i,HI_OFFLINE))
		{
			// Try to connect 
			SOCKET s=NewSocket();
			bind(s,GetIPstruct((Master) ? MASTER_CONNECT_PORT : SLAVE_CONNECT_PORT ),sizeof(sockaddr));
			sockaddr_in RemoteAddr;
			RemoteAddr.sin_family=AF_INET;
			RemoteAddr.sin_addr.s_addr=htonl(HostInfo[i].IP);
			RemoteAddr.sin_port=htons((Master) ? SLAVE_LISTEN_PORT : MASTER_LISTEN_PORT );
			if (connect(s,(sockaddr*)&RemoteAddr,sizeof(RemoteAddr))==0)
			{
				// Master/Slave found
				HostInfo[i].Type=(Master) ? HI_SLAVE : HI_MASTER;
				if (!Master)
					for (int k=0;k<HostCount;k++)
						if ((i!=k)&&(IsType(k,HI_MASTER)))
							HostInfo[i].Type=HI_ILLEGAL_MASTER;
				HostInfo[i].CtrlSocket=s;
				WSAAsyncSelect(s,g_hwndJobs,(Master) ? IDC_MASTER_SOCKET : IDC_SLAVE_SOCKET ,FD_MOSTFLAGS);
				ListView_Update(GetDlgItem(g_hwndJobs, IDC_HOSTS),i);
				psSheduler->log(7,"Connecting.... %s(%s): success(%s)",inet_ntoa(RemoteAddr.sin_addr),HostInfo[i].Name,Master?"Master":"Slave");

			} else
			{
				CleanCloseSocket(s,g_hwndJobs, (Master) ? IDC_MASTER_SOCKET : IDC_SLAVE_SOCKET);
				psSheduler->log(7,"Connecting.... %s(%s): failed(%s)",inet_ntoa(RemoteAddr.sin_addr),HostInfo[i].Name,Master?"Master":"Slave");
			}
		}
	if (Master) VDJob::SetMasterMode(true);
		   else VDJob::SetSlaveMode(true);

	// Update Controls 
	ListView_Update(GetDlgItem(g_hwndJobs, IDC_HOSTS),LocalHost);
	return true;
}

////////////////////////////////////////////////////////////////////
SOCKET NetworkManager::GetSock(int Host) { return HostInfo[Host].CtrlSocket; }

////////////////////////////////////////////////////////////////////
void NetworkManager::SetSock(int Host,SOCKET s){ HostInfo[Host].CtrlSocket=s; }

////////////////////////////////////////////////////////////////////
DWORD NetworkManager::GetIP(int Host){ return HostInfo[Host].IP; }

////////////////////////////////////////////////////////////////////
void NetworkManager::SetIP(int Host, DWORD IP){ HostInfo[Host].IP=IP;}

////////////////////////////////////////////////////////////////////
sockaddr* NetworkManager::GetIPstruct(int Port) {
	static sockaddr_in IPAddr;

	IPAddr.sin_family=AF_INET;
	IPAddr.sin_addr.s_addr=htonl(LocalIP);
	IPAddr.sin_port=htons(Port);

	return (sockaddr*)&IPAddr;
}


////////////////////////////////////////////////////////////////////
int NetworkManager::GetType(int Host){ return HostInfo[Host].Type; }

////////////////////////////////////////////////////////////////////
void NetworkManager::SetType(int Host, int Type){ HostInfo[Host].Type=Type; }

////////////////////////////////////////////////////////////////////
bool NetworkManager::IsType(int Host, int Type1, int Type2, int Type3){
	return ((HostInfo[Host].Type==Type1)||
		    (HostInfo[Host].Type==Type2)||
			(HostInfo[Host].Type==Type3));
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::GetWorkState(int Host){ return HostInfo[Host].Running; }

////////////////////////////////////////////////////////////////////
void NetworkManager::SetWorkState(int Host, bool Running){	HostInfo[Host].Running=Running; }

////////////////////////////////////////////////////////////////////
char* NetworkManager::GetName(int Host){ return HostInfo[Host].Name; }

////////////////////////////////////////////////////////////////////
int NetworkManager::GetMsgbufLen(int Host) { 
	return HostInfo[Host].MsgBufLen; 
}

////////////////////////////////////////////////////////////////////
int NetworkManager::GetMsgID(int Host) {
	if (HostInfo[Host].MsgBufLen<4) return 0;
	return *(int*)HostInfo[Host].MsgBuf;
}

////////////////////////////////////////////////////////////////////
void* NetworkManager::GetMsgbufDataPtr(int Host, int Offset) {
	if (HostInfo[Host].MsgBufLen<4) return 0;
	return (HostInfo[Host].MsgBuf)+4+Offset;
}

////////////////////////////////////////////////////////////////////
void NetworkManager::MsgbufRecv(int Host, SOCKET s, int Flags){
	char FAR *lastchar=(char*)HostInfo[Host].MsgBuf+HostInfo[Host].MsgBufLen;
	int rcvd=recv(s,lastchar,SCRIPTLEN-HostInfo[Host].MsgBufLen,0);
	HostInfo[Host].MsgBufLen+=rcvd;
}

////////////////////////////////////////////////////////////////////
void NetworkManager::SyncMsgQueue(int Host) {
	while ((HostInfo[Host].MsgBufLen>0)&&
		   (HostInfo[Host].MsgBuf[0]!=0xff))
		RemoveMessage(Host,1);
	while ((HostInfo[Host].MsgBufLen>0)&&
		   (HostInfo[Host].MsgBuf[0]==0xff))
		RemoveMessage(Host,1);
}


////////////////////////////////////////////////////////////////////
void NetworkManager::RemoveMessage(int Host, int Len){
	if (Len>HostInfo[Host].MsgBufLen) return;
	memmove((char*)HostInfo[Host].MsgBuf,(char*)HostInfo[Host].MsgBuf+Len,HostInfo[Host].MsgBufLen);
	HostInfo[Host].MsgBufLen-=Len;
}

////////////////////////////////////////////////////////////////////
void NetworkManager::UpdateHostList() {
	hostent *LocalHostEnt;
	char LocalHostName[100];
	gethostname(LocalHostName,100);
	LocalHostEnt=gethostbyname(LocalHostName);
	LocalIP=ntohl(*(int*)LocalHostEnt->h_addr_list[0]);
	HWND hwndItem = GetDlgItem(g_hwndJobs, IDC_HOSTS);
	for (int i=0;i<HostCount;i++)
		HostInfo[i].ReScanFound=false;

	LoadHostList();					
	RecursiveScanNetwork();
/*	Need PartSheduler::RemoveDeadHost
	for (i=HostCount-1;i>=0;i--)
		if (!HostInfo[i].ReScanFound)
			RemoveDeadHost(i);
*/

}
////////////////////////////////////////////////////////////////////
void NetworkManager::RemoveDeadHost(int Host) {
	if (Host==LocalHost) return;
	
	memmove(&HostInfo[Host],&HostInfo[Host+1],(HostCount-Host-1)*sizeof(NetworkHostInfo));
	
	if (LocalHost>Host) --LocalHost;
	--HostCount;
	ListView_DeleteItem(GetDlgItem(g_hwndJobs, IDC_HOSTS), HostCount+1);
}


////////////////////////////////////////////////////////////////////
void NetworkManager::LoadHostList() {
	char szVDPath[256],*lpFilePart;

	GetModuleFileName(NULL, szVDPath, sizeof szVDPath);
	GetFullPathName(szVDPath, sizeof szVDPath, szVDPath, &lpFilePart);
	strcpy(lpFilePart, "hosts.info");

	FILE *infofile=fopen(szVDPath,"rt");
	if (!infofile) return;

	while (!feof(infofile)) {
		char lbuf[128];
		lbuf[0]=0;
		fgets(lbuf,128,infofile);
		if (lbuf[strlen(lbuf)-1]=='\n')
			lbuf[strlen(lbuf)-1]=0;
		if ((lbuf[0]!='#')&&(strlen(lbuf)>0)) {
			if (psSheduler) psSheduler->log(7,"LoadHostList, Add Host %s",lbuf);
			DWORD iIP=inet_addr(lbuf);
			hostent *hHost;
			if (iIP==INADDR_NONE)
				hHost=gethostbyname(lbuf); else
				hHost=gethostbyaddr((char*)&iIP,4,AF_INET);
			if (hHost) TryAddHost(hHost->h_name);
		}
	}
}

////////////////////////////////////////////////////////////////////
DWORD NetworkManager::RecursiveScanNetwork(LPNETRESOURCE RootResource, int depth)
{
	NETRESOURCE nr;
	if (depth==1) {

		nr.dwScope = RESOURCE_GLOBALNET;
		nr.dwType = RESOURCETYPE_ANY; 
		nr.dwUsage = 0;
		nr.lpLocalName = 0;
		nr.lpRemoteName = 0;
		nr.lpComment = 0;
		nr.lpProvider = 0;  
		RootResource=&nr;
	}

	DWORD dwResult, dwResultEnum;
	HANDLE hEnum;
	DWORD cbBuffer = 16384;
	DWORD i, cEntries = -1;
	static BOOL bFirstLoop = TRUE;
	static HTREEITEM hRoot;
	static HTREEITEM hChildItem;
 
	dwResult=WNetOpenEnum(RESOURCE_GLOBALNET, RESOURCETYPE_ANY, 0,RootResource, &hEnum);
	if (dwResult==NO_ERROR) {
		LPNETRESOURCE lpnrLocal = (LPNETRESOURCE) GlobalAlloc(0,cbBuffer);
		do {
			memset(lpnrLocal, 0, cbBuffer);

			dwResultEnum=WNetEnumResource(hEnum, &cEntries, lpnrLocal,&cbBuffer);

			if (dwResultEnum==NO_ERROR) {
				for(i = 0; i < cEntries; i++) {
					if ((lpnrLocal[i].lpRemoteName)&&(lpnrLocal[i].lpRemoteName[0]=='\\')) {
						if (psSheduler) psSheduler->log(7,"RecursiveScanNetwork, Add host: %s",&(lpnrLocal[i].lpRemoteName[2]));
						TryAddHost(&(lpnrLocal[i].lpRemoteName[2]));
					} else {
						if ((psSheduler)&&(lpnrLocal[i].lpRemoteName))
							psSheduler->log(7,"ScanNetwork(%d), Entry found:%s", depth, lpnrLocal[i].lpRemoteName);
						if (depth<20)
							RecursiveScanNetwork(&lpnrLocal[i], depth+1);
					}
				}
			} else if (dwResultEnum != ERROR_NO_MORE_ITEMS) 
				psSheduler->log(1,"ScanNetwork(%d), Error in WNetEnumResource:%d", depth, dwResultEnum);

		} while(dwResultEnum != ERROR_NO_MORE_ITEMS);
 
		GlobalFree(lpnrLocal);
 		WNetCloseEnum(hEnum);

	} else psSheduler->log(1,"ScanNetwork(%d), Error in WNetOpenEnum:%d", depth, dwResult);
 
	return 0;	
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::TryAddHost(char *name) {
	bool dbl=false;
	_strupr(name);
	for (int i=0;i<HostCount;i++) 
		if (strfuzzycompare(HostInfo[i].Name,name)) {	// known host...
			if (psSheduler) psSheduler->log(7,"TryAddHost, host already found: %s", name);
			HostInfo[i].ReScanFound=true;
			return false;
		}

	hostent *hHost=gethostbyname(name);
	_strupr(name); 
	if ((hHost)&&(HostCount<255))
	{
		if (psSheduler) psSheduler->log(7,"TryAddHost, Valid host found:%s", name);
		HostInfo[HostCount].ReScanFound=true;


		HostInfo[HostCount].IP=ntohl(*(int*)hHost->h_addr_list[0]);
		strcpy(HostInfo[HostCount].Name,name);
		HostInfo[HostCount].FPS=0;
		HostInfo[HostCount].From=0;
		HostInfo[HostCount].To=0;
		HostInfo[HostCount].Progress=0;
		HostInfo[HostCount].Running=false;

		if (HostInfo[HostCount].IP==LocalIP) {
			HostInfo[HostCount].Type=HI_LOCALHOST;
			LocalHost=HostCount;
		} else
			HostInfo[HostCount].Type=HI_OFFLINE;
		HostInfo[HostCount].CtrlSocket=0;

		LVITEM li;
		li.mask		= LVIF_TEXT;
		li.iSubItem	= 0;
		li.iItem	= HostCount;
		li.pszText	= LPSTR_TEXTCALLBACK;

		ListView_InsertItem(GetDlgItem(g_hwndJobs, IDC_HOSTS), &li);
		ListView_Update(GetDlgItem(g_hwndJobs, IDC_HOSTS), HostCount);
		HostCount++;
		return true;
	}
	if (psSheduler) psSheduler->log(1,"TryAddHost, Host not found: %s", name);
	return false;
}
////////////////////////////////////////////////////////////////////
bool NetworkManager::CreateListenSocket() {
	int ret;
	if (LocalState==HI_LOCALHOST) return false;

	int Port=(LocalState==HI_MASTER) ? MASTER_LISTEN_PORT : SLAVE_LISTEN_PORT;

	if (!(SelectSocket=NewSocket())) {
		ret=WSAGetLastError();
		psSheduler->log(1,"CreateListenSocket failed, NewSocket returned %d",ret);
		return false;
	}

	if (ret=bind(SelectSocket,GetIPstruct(Port),sizeof(sockaddr))) {
		ret=WSAGetLastError();
		psSheduler->log(1,"CreateListenSocket failed, BIND returned %d",ret);
		return false;
	}

	if (ret=listen(SelectSocket,SOMAXCONN)) {
		ret=WSAGetLastError();
		psSheduler->log(1,"CreateListenSocket failed, LISTEN returned %d",ret);
		return false;
	}
		
	WSAAsyncSelect(SelectSocket, g_hwndJobs, (LocalState==HI_MASTER) ? IDC_MASTER_SOCKET : IDC_SLAVE_SOCKET, FD_MOSTFLAGS);

	return true;
}

////////////////////////////////////////////////////////////////////
bool NetworkManager::DestroyListenSocket() {
	WSAAsyncSelect(SelectSocket, g_hwndJobs,0,0);
	closesocket(SelectSocket);
	return true;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////


//////////////////////////////////////////////
//  Enable/Disable jobdialog-buttons 
//////////////////////////////////////////////
static void SetJobWindowButtons(int Mode)
{
	int jobButtonMode[][2] = {
		{ IDOK,					JWS_SINGLE	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_MOVE_UP,			JWS_SINGLE	| JWS_MASTER	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_MOVE_DOWN,		JWS_SINGLE	| JWS_MASTER	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_POSTPONE,			JWS_SINGLE	| JWS_MASTER	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_DELETE,			JWS_SINGLE	| JWS_MASTER	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_START,			JWS_SINGLE	| JWS_MASTER	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_ABORT,			JWS_SINGLE	| JWS_MASTER	| JWS_RUNNING },
		{ IDC_UPDATE,			JWS_SINGLE	| JWS_MASTER	| JWS_SLAVE	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_MASTER_MODE,		JWS_SINGLE	| JWS_NOT_RUNNING },
		{ IDC_SLAVE_MODE,		JWS_SINGLE	| JWS_NOT_RUNNING },
		{ IDC_MSMODE_ABORT,		JWS_MASTER	| JWS_SLAVE	| JWS_NOT_RUNNING },
		{ IDC_PROGRESS,			JWS_SINGLE	| JWS_MASTER	| JWS_SLAVE	| JWS_RUNNING },
		{ IDC_PERCENT,			JWS_SINGLE	| JWS_MASTER	| JWS_SLAVE	| JWS_RUNNING},
		{ IDC_NETWORK_PROGRESS,	JWS_MASTER	| JWS_SLAVE	| JWS_RUNNING },
		{ IDC_NETWORK_PERCENT,	JWS_SINGLE	| JWS_MASTER	| JWS_SLAVE	| JWS_RUNNING },
		{ IDC_JOBS_SHUTDOWN,	JWS_MASTER	| JWS_SINGLE	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ IDC_JOBS_SHUTDOWN_ALL,JWS_MASTER	| JWS_RUNNING	| JWS_NOT_RUNNING },
		{ 0,0 }
		};

	if (!g_hwndJobs) return;

	if (Mode==JWS_DETECT)
	{
		Mode= VDJob::IsNetworkJob()?(VDJob::IsMasterMode()?JWS_MASTER:JWS_SLAVE):JWS_SINGLE;
		Mode|=((VDJob::IsRunInProgress())&&(!VDJob::IsSlaveMode()))?JWS_RUNNING:JWS_NOT_RUNNING;
	}
	for (int i=0;jobButtonMode[i][0];i++)
	{
		HWND hwnd=GetDlgItem(g_hwndJobs,jobButtonMode[i][0]);
		ShowWindow(hwnd,SW_SHOW);
		EnableWindow(hwnd,((Mode&jobButtonMode[i][1])==Mode));
	}
}

////////////////////////////////////////////////////
// Scan Networkresources for Client
////////////////////////////////////////////////////
////////////////////////////////////////////////////
// Initialize running a script over multiple Clients
////////////////////////////////////////////////////
int NetworkRunScript(char *script, char *error)
{
	if (!VDJob::IsMasterMode())
	{
		if ((psSheduler)&&(!(VDJob::IsSlaveMode()))) psSheduler->SetPartInfo(0,0);


		g_fJobMode = true;
		g_fJobAborted = false; 
		SetJobWindowButtons();
		g_prefs.main.iDubPriority=g_JobsDlgConfig.Priority;
		try {
			_CrtCheckMemory();
			RunScriptMemory(script);
			_CrtCheckMemory();
			g_fJobMode = false;
		} catch(MyError err) {
			if (error) strcpy(error, err.gets());
			return VDJob::ERR;
		}
		if (psSheduler) psSheduler->PartSuccess(0);
		netMan->SetWorkState(0,false);

		return (g_fJobAborted || VDJob::IsRunAllStop()) ? VDJob::ABORTED : VDJob::DONE;
	}
	
	if (!psSheduler) return VDJob::ERR;

	psSheduler->log(3,"NetworkRunScript called, running in Mastermode");

	if (!psSheduler->SetOriginalScript(netMan->GetLocalHost(), script)) {
		psSheduler->log(1,"NetworkRunScript, error setting script");
		if (error) strcpy(error,"invalid networkjob-script, see logfile for details");
		return VDJob::ERR;
	}

	for (int i=0;i<netMan->GetHostCount();i++) {
		if (netMan->IsType(i,HI_SLAVE,HI_LOCALHOST)) {
			psSheduler->SetActiveHost(i);
			psSheduler->LoadHostSpeed(i);
		} else 
			psSheduler->SetActiveHost(i,false);
	}

	////////////////////////////////////////////
	// start all clients
	////////////////////////////////////////////
	SendMessage(g_hwndJobs, IDC_START_NETWORKJOB, 0, 0);

	////////////////////////////////////////////
	// process local, wait for clients
	////////////////////////////////////////////
	MSG msg;
	char workscript[SCRIPTLEN];
	while ((!psSheduler->IsAllPartsComplete())&&(!VDJob::IsRunAllStop())) {
		if (psSheduler->GetNextPart(netMan->GetLocalHost())) {
			netMan->SetWorkState(netMan->GetLocalHost(),true);
			psSheduler->GetModifiedScript(netMan->GetLocalHost(),workscript);

			psSheduler->log(3,"NetworkRunScript, execute script local");
			g_prefs.main.iDubPriority=g_JobsDlgConfig.Priority;
			g_fJobMode=true;
			g_fJobAborted = false; 
			SetJobWindowButtons();
			try {
				_CrtCheckMemory();
				RunScriptMemory(workscript);
				_CrtCheckMemory();
			} catch(MyError err) {
				char szError[256];
				strcpy(szError, err.gets());

				psSheduler->log(1,"NetworkRunScript, job-error %s",szError);

				SendMessage(g_hwndJobs, IDC_STOP_NETWORKJOB, 0, 0);
				if (error) strcpy(error,err.gets());
				return VDJob::ERR;

			}

			g_fJobMode=false;

			if (g_fJobAborted) psSheduler->log(3,"NetworkRunScript, Job done, normal exit");
						  else psSheduler->log(1,"NetworkRunScript, Job aborted, normal exit");

			if (g_fJobAborted) psSheduler->PartAborted(netMan->GetLocalHost());
						  else psSheduler->PartSuccess(netMan->GetLocalHost());

			netMan->SetWorkState(netMan->GetLocalHost(),false);
			
			// Remove "___temp__.?.avi" Files
			if (!psSheduler->IsGenOutput()) {
				char lbuf[LINELEN];
				if (ScriptSearchLine(workscript,"VirtualDub.SaveAVI",lbuf)||
					ScriptSearchLine(workscript,"VirtualDub.SaveCompatibleAVI",lbuf)) {
					char *t,*s;
					s=strchr(lbuf,'\"');
					t=strrchr(s,'\"');
					s++;
					if ((s)&&(t)) {
						t[0]=0;
						remove(s);
					}
				}
			}

		} else {	// no Parts to do
			Sleep(5);
			if (GetQueueStatus(QS_ALLEVENTS)==0) continue;
			if (GetMessage(&msg, (HWND) NULL, 0, 0)) {
				if (guiCheckDialogs(&msg)) continue;
				if (!IsWindow(g_hwndJobs) || !IsDialogMessage(g_hwndJobs, &msg)) { 
					TranslateMessage(&msg); 
				    DispatchMessage(&msg); 
				}
			}		
		}
	}

	g_fJobMode = false;


	psSheduler->log(3,"NetworkRunScript, job completed");

	if (VDJob::IsRunAllStop()) {
		psSheduler->Clear();
		return VDJob::ABORTED;
	}

////////////////////////////////////////////
// Check and concatenate all files, process audio
////////////////////////////////////////////
	if (psSheduler->IsGenOutput()) {
		psSheduler->GetConcatScript(workscript);
		
		psSheduler->log(3,"NetworkRunScript, processing audio");
		
		g_fJobMode = true;
		g_fJobAborted = false; 
		SetJobWindowButtons();
		g_prefs.main.iDubPriority=g_JobsDlgConfig.Priority;

		try {
			_CrtCheckMemory();
			RunScriptMemory(workscript);
			_CrtCheckMemory();
		} catch(MyError err) {
			psSheduler->log(1,"NetworkRunScript, concatenation-job-error %s",err.gets());
			if (error) strcpy(error, err.gets());
			return VDJob::ERR;
		}

		g_fJobMode = false;
		if (psSheduler) psSheduler->PartSuccess(0);
		netMan->SetWorkState(0,false);

		if (g_fJobAborted) return VDJob::ABORTED;

/////////////////////////
// Remove Files
/////////////////////////
		psSheduler->log(3,"NetworkRunScript, removing files");
		psSheduler->RemoveTempFiles();
	}
	return VDJob::DONE;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
int __send(SOCKET s, const char FAR * buf, int len, int flags) {
	int syncsent=0,sent=0,result;

	char Sync[16];
	memset(Sync,0xff,16);
	do {
		result=send(s,Sync,4,flags);
		if (result>0) syncsent+=result; 
		if (result==SOCKET_ERROR) {
			result=WSAGetLastError();
			if (result!=WSAEWOULDBLOCK)
				return 0;
		}
	}while (syncsent<4);

	do {
		result=send(s,buf+sent,len,flags);
		if (result>0) sent+=result; 
		if (result==SOCKET_ERROR) {
			result=WSAGetLastError();
			if (result!=WSAEWOULDBLOCK)
				return 0;
		}
	}while (sent<len);

	return sent;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void Jobs_SendToTray(bool totray) {
	NOTIFYICONDATA	IconData;
	
	IconData.cbSize=sizeof(IconData);
	IconData.hWnd=g_hwndJobs;
	IconData.uID=1;

	if (totray) {
		IconData.uFlags=NIF_ICON | NIF_MESSAGE | NIF_TIP;
		IconData.uCallbackMessage=MYWM_TRAYICON;
		IconData.hIcon=(HICON) LoadImage(g_hInst, MAKEINTRESOURCE(IDI_VIRTUALDUB) , IMAGE_ICON, 16, 16, 0);
		strcpy(IconData.szTip,"Click to restore window\0");

		Shell_NotifyIcon(NIM_ADD, &IconData); 

		ShowWindow(g_hwndJobs, SW_HIDE);
		ShowWindow(g_hWnd, SW_HIDE);
	} else {
		IconData.uFlags=0;
		Shell_NotifyIcon(NIM_DELETE, &IconData);

		ShowWindow(g_hWnd, SW_SHOW);
		ShowWindow(g_hwndJobs, SW_RESTORE);
	}
}
/////////////////////////////////////////////////////////////////////////////////////////////////
int LENID[]={ID_NETWORK_PARTLENGTH_1MIN,60,
			 ID_NETWORK_PARTLENGTH_2MIN,120,
			 ID_NETWORK_PARTLENGTH_3MIN,180,
			 ID_NETWORK_PARTLENGTH_5MIN,300,
			 ID_NETWORK_PARTLENGTH_10MIN,600,
			 ID_NETWORK_PARTLENGTH_15MIN,900,
			 0};

int PRIID[]={ID_NETWORK_PRIORITY_1,1,
			 ID_NETWORK_PRIORITY_2,2,
			 ID_NETWORK_PRIORITY_3,3,
			 ID_NETWORK_PRIORITY_4,4,
			 ID_NETWORK_PRIORITY_5,5,
			 ID_NETWORK_PRIORITY_6,6,
			 ID_NETWORK_PRIORITY_7,7,
			 ID_NETWORK_PRIORITY_8,8,
			 0};

int LOGID[]={ID_DEBUG_LOGLEVEL_0,0,
			 ID_DEBUG_LOGLEVEL_1,1,
			 ID_DEBUG_LOGLEVEL_3,3,
			 ID_DEBUG_LOGLEVEL_5,5,
			 ID_DEBUG_LOGLEVEL_7,7,
			 ID_DEBUG_LOGLEVEL_9,9,
		     0};

/////////////////////////////////////////////////////////////////////////////////////////////////
void Job_MenuHit(HWND hdlg, WPARAM wParam) {
	static char *fileFilters=
		"VirtualDub job list (*.jobs)\0"		"*.jobs\0"
		"Sylia script for VirtualDub (*.syl)\0"	"*.syl\0"
		"All files (*.*)\0"						"*.*\0";
	VDJob *vdj, *vdj_next;

	try {
		switch(LOWORD(wParam)) {
			////////////////////////////////////////////
			// Shutdown one Client
			////////////////////////////////////////////
			case IDC_CONTXT_SHUTDOWN:
				if (VDJob::IsMasterMode()) {
					int Message=IDN_SHUTDOWN;
					__send(netMan->GetSock(HostsContextMenuItemPos),(char*)&Message,4,0);
				}
				break;
			////////////////////////////////////////////
			// Close Client
			////////////////////////////////////////////
			case IDC_CONTXT_CLOSE:
				if (VDJob::IsMasterMode()) {
					int Message=IDN_CLOSE;
					__send(netMan->GetSock(HostsContextMenuItemPos),(char*)&Message,4,0);
				}
				break;
			////////////////////////////////////////////
			// Invert selection
			////////////////////////////////////////////
			case IDC_CONTXT_NEVER_SHUTDOWN:
				Job_LoadHostConfig(HostsContextMenuItemPos);
				g_HostsConfig.NeverShutdown=!g_HostsConfig.NeverShutdown;
				Job_SaveHostConfig(HostsContextMenuItemPos);
				break;

			case ID_FILE_LOADJOBLIST:
				{
					OPENFILENAME ofn;
					char szFile[MAX_PATH];

					///////////////

					szFile[0] = 0;

					ofn.lStructSize			= sizeof(OPENFILENAME);
					ofn.hwndOwner			= hdlg;
					ofn.lpstrFilter			= fileFilters;
					ofn.lpstrCustomFilter	= NULL;
					ofn.nFilterIndex		= 1;
					ofn.lpstrFile			= szFile;
					ofn.nMaxFile			= sizeof szFile;
					ofn.lpstrFileTitle		= NULL;
					ofn.nMaxFileTitle		= 0;
					ofn.lpstrInitialDir		= NULL;
					ofn.lpstrTitle			= "Load job list";
					ofn.Flags				= OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_ENABLESIZING;
					ofn.lpstrDefExt			= NULL;

					if (GetOpenFileName(&ofn))
						VDJob::ListLoad(szFile);
				}
				break;

			case ID_FILE_SAVEJOBLIST:
				{
					OPENFILENAME ofn;
					char szFile[MAX_PATH];

					///////////////

					szFile[0] = 0;

					ofn.lStructSize			= sizeof(OPENFILENAME);
					ofn.hwndOwner			= hdlg;
					ofn.lpstrFilter			= fileFilters;
					ofn.lpstrCustomFilter	= NULL;
					ofn.nFilterIndex		= 1;
					ofn.lpstrFile			= szFile;
					ofn.nMaxFile			= sizeof szFile;
					ofn.lpstrFileTitle		= NULL;
					ofn.nMaxFileTitle		= 0;
					ofn.lpstrInitialDir		= NULL;
					ofn.lpstrTitle			= "Save job list";
					ofn.Flags				= OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_ENABLESIZING;
					ofn.lpstrDefExt			= NULL;

					if (GetSaveFileName(&ofn))
						vdj->Flush(szFile);
				}
				break;

			case ID_EDIT_CLEARLIST:
				if (IDOK != MessageBox(hdlg, "Really clear job list?", "VirtualDub job system", MB_OKCANCEL | MB_ICONEXCLAMATION))
					break;

				VDJob::ListClear(false);
				break;

			case ID_EDIT_DELETEDONEJOBS:

				if (!(vdj = VDJob::ListGet(0)))
					break;

				while(vdj_next = (VDJob *)vdj->next) {
					if (vdj->iState == VDJob::DONE) {
						vdj->Delete();
						delete vdj;
					}

					vdj = vdj_next;
				}

				break;

			case ID_EDIT_FAILEDTOWAITING
				:

				if (!(vdj = VDJob::ListGet(0)))
					break;

				while(vdj_next = (VDJob *)vdj->next) {
					if (vdj->iState == VDJob::ABORTED || vdj->iState == VDJob::ERR) {
						vdj->iState = VDJob::WAITING;
						vdj->Refresh();
					}

					vdj = vdj_next;
				}
				break;

			case ID_EDIT_WAITINGTOPOSTPONED:

				if (!(vdj = VDJob::ListGet(0)))
					break;

				while(vdj_next = (VDJob *)vdj->next) {
					if (vdj->iState == VDJob::WAITING) {
						vdj->iState = VDJob::POSTPONED;
						vdj->Refresh();
					}

					vdj = vdj_next;
				}
				break;

			case ID_EDIT_POSTPONEDTOWAITING:

				if (!(vdj = VDJob::ListGet(0)))
					break;

				while(vdj_next = (VDJob *)vdj->next) {
					if (vdj->iState == VDJob::POSTPONED) {
						vdj->iState = VDJob::WAITING;
						vdj->Refresh();
					}

					vdj = vdj_next;
				}
				break;

			case ID_EDIT_DONETOWAITING:

				if (!(vdj = VDJob::ListGet(0)))
					break;

				while(vdj_next = (VDJob *)vdj->next) {
					if (vdj->iState == VDJob::DONE) {
						vdj->iState = VDJob::WAITING;
						vdj->Refresh();
					}

					vdj = vdj_next;
				}
				break;

			case ID_EDIT_PROCESSDIRECTORY:
				JobProcessDirectory(hdlg);
				break;
			case ID_ALWAYS_RESTART:
				g_JobsDlgConfig.RestartJobs=!g_JobsDlgConfig.RestartJobs;
				SetMenuEntries(true);
				break;
			case ID_DEBUG_ERASE_LOGFILE:
				if (g_hwndJobs)
					PostMessage(g_hwndJobs,IDC_CLEAR_LOGFILE,0,0);
				break;
			case ID_OPTIONS_PROCESS_AUDIO:
				g_JobsDlgConfig.ProcessAudio=!g_JobsDlgConfig.ProcessAudio;
				SetMenuEntries(true);
				break;
			case ID_OPTIONS_SILENCE_SCAN:
				g_JobsDlgConfig.ScanForSilence=!g_JobsDlgConfig.ScanForSilence;
				SetMenuEntries(true);
				break;
			case ID_SENDTOTRAY:
				g_JobsDlgConfig.SendToTray=!g_JobsDlgConfig.SendToTray;
				SetMenuEntries(true);
				if ((g_JobsDlgConfig.SendToTray) && (VDJob::IsSlaveMode()))
					Jobs_SendToTray(true);
				break;
		}
	} catch(MyError e) {
		e.post(hdlg, "Job system error");
	}

	int i;
	for (i=0;LENID[i];i+=2) 
		if (LENID[i]==LOWORD(wParam)) {
			g_JobsDlgConfig.PartLength=LENID[i+1];
			SetMenuEntries(true);
		}

	for (i=0;PRIID[i];i+=2) 
		if (PRIID[i]==LOWORD(wParam)) {
			g_JobsDlgConfig.Priority=PRIID[i+1];
			SetMenuEntries(true);
		}

	for (i=0;LOGID[i];i+=2) 
		if (LOGID[i]==LOWORD(wParam)) {
			g_JobsDlgConfig.Loglevel=LOGID[i+1];
			SetMenuEntries(true);
		}
}

/////////////////////////////////////////////////////////////////////////////////////////////////
#define CHECKMENUITEM(__MENU, __ID, __ENABLE)	\
	CheckMenuItem(__MENU, __ID, ((__ENABLE) ? MF_CHECKED : MF_UNCHECKED) | MF_BYCOMMAND)

#define ENABLEMENUITEM(__MENU, __ID, __ENABLE)	\
	EnableMenuItem(__MENU, __ID, ((__ENABLE) ? MF_ENABLED : MF_GRAYED) | MF_BYCOMMAND)

/////////////////////////////////////////////////////////////////////////////////////////////////
void EnableMenuEntries() {
	HMENU hMenu = GetMenu(g_hwndJobs);

	ENABLEMENUITEM(hMenu, ID_OPTIONS_PROCESS_AUDIO, !VDJob::fRunInProgress);
	ENABLEMENUITEM(hMenu, ID_OPTIONS_SILENCE_SCAN, !VDJob::fRunInProgress && g_JobsDlgConfig.ProcessAudio);

}

/////////////////////////////////////////////////////////////////////////////////////////////////
void Job_SaveDlgConfig() {
#ifdef NANDUB
	SetConfigBinary("", "JobDlgPrefsNandub", (char*)&g_JobsDlgConfig, sizeof g_JobsDlgConfig);
#else
	SetConfigBinary("", "JobDlgPrefsVirtualDub", (char*)&g_JobsDlgConfig, sizeof g_JobsDlgConfig);
#endif
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void Job_LoadDlgConfig() {
	// default Values:
	g_JobsDlgConfig.PartLength=300;
	g_JobsDlgConfig.Priority=5;
	g_JobsDlgConfig.RestartJobs=true;
	g_JobsDlgConfig.Shutdown=false;
	g_JobsDlgConfig.ShutdownClients=false;
	g_JobsDlgConfig.Loglevel=1;
	g_JobsDlgConfig.ProcessAudio=true;
	g_JobsDlgConfig.ScanForSilence=true;
	g_JobsDlgConfig.SendToTray=false;

	QueryConfigBinary("", "JobDlgPrefs", (char*)&g_JobsDlgConfig, sizeof g_JobsDlgConfig);
#ifdef NANDUB
	QueryConfigBinary("", "JobDlgPrefsNandub", (char*)&g_JobsDlgConfig, sizeof g_JobsDlgConfig);
#else
	QueryConfigBinary("", "JobDlgPrefsVirtualDub", (char*)&g_JobsDlgConfig, sizeof g_JobsDlgConfig);
#endif
	SetMenuEntries();
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void Job_LoadHostConfig(int Host) {
	// Get hostname
	char *HostName=netMan->GetName(Host);

	// default options
	g_HostsConfig.NeverShutdown=false;

	// query registry
	QueryConfigBinary("HostOptions", HostName, (char*)&g_HostsConfig,sizeof(g_HostsConfig));
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void Job_SaveHostConfig(int Host) {
	// Get hostname
	char *HostName=netMan->GetName(Host);

	// query registry
	SetConfigBinary("HostOptions", HostName , (char*)&g_HostsConfig,sizeof(g_HostsConfig));
}

/////////////////////////////////////////////////////////////////////////////////////////////////
void SetMenuEntries(bool Save) {
	HMENU hMenu = GetMenu(g_hwndJobs);
	int i;

	CHECKMENUITEM(hMenu,ID_OPTIONS_PROCESS_AUDIO, g_JobsDlgConfig.ProcessAudio);
	CHECKMENUITEM(hMenu,ID_ALWAYS_RESTART, g_JobsDlgConfig.RestartJobs);
	CHECKMENUITEM(hMenu,ID_OPTIONS_SILENCE_SCAN, g_JobsDlgConfig.ScanForSilence);
	CHECKMENUITEM(hMenu,ID_SENDTOTRAY, g_JobsDlgConfig.SendToTray);

	for (i=0;PRIID[i];i+=2)
		CHECKMENUITEM(hMenu, PRIID[i], PRIID[i+1]==g_JobsDlgConfig.Priority);

	for (i=0;LENID[i];i+=2)
		CHECKMENUITEM(hMenu, LENID[i], LENID[i+1]==g_JobsDlgConfig.PartLength);

	for (i=0;LOGID[i];i+=2)
		CHECKMENUITEM(hMenu, LOGID[i], LOGID[i+1]==g_JobsDlgConfig.Loglevel);

	CheckDlgButton(g_hwndJobs, IDC_JOBS_SHUTDOWN, g_JobsDlgConfig.Shutdown ? BST_CHECKED : BST_UNCHECKED);
	CheckDlgButton(g_hwndJobs, IDC_JOBS_SHUTDOWN_ALL, g_JobsDlgConfig.ShutdownClients ? BST_CHECKED : BST_UNCHECKED);

	EnableMenuEntries();

	if (Save)
		Job_SaveDlgConfig();
}

/////////////////////////////////////////////////////////////////////////////////////////////////
POINT jobCtlPos[22];
bool f_CloseClient;
int i_StartupTimer;

static BOOL CALLBACK JobCtlDlgProc(HWND hdlg, UINT uiMsg, WPARAM wParam, LPARAM lParam) {
	static char *szColumnNames[]={ "Name","Source","Dest","Start","End","Status" };
	static int iColumnWidths[]={ 60,75,75,50,50,90 };

	static char *szHostsColumnNames[]={"Name","State","Range","%","FPS"};
	static int iHostsColumnWidths[]={80,110,100,40,40};

	static UINT uiUpdateTimer;
	static int iUpdateVal;
	static bool fUpdateDisable;
	static RECT rInitial;

	VDJob *vdj, *vdj_p, *vdj_p2, *vdj_n, *vdj_n2;
	int index;
	HWND hwndItem;


	switch(uiMsg) {
/////////////////////////////////////////////////////////////////////////////
	case WM_INITDIALOG:
		{
			g_hwndJobs=hdlg;
			g_JobsDlgConfig.Shutdown=false;
			g_JobsDlgConfig.ShutdownClients=false;
			HWND hwndItem;
			LV_COLUMN lvc;
			int i;

			i_StartupTimer=-1;
			GetWindowRect(hdlg, &rInitial);

			hwndItem = GetDlgItem(hdlg, IDC_JOBS);
			fUpdateDisable = false;

			ListView_SetExtendedListViewStyleEx(hwndItem, LVS_EX_FULLROWSELECT , LVS_EX_FULLROWSELECT);

			for (i=0; i<6; i++) {
				lvc.mask = LVCF_FMT | LVCF_SUBITEM | LVCF_TEXT | LVCF_WIDTH;
				lvc.fmt = LVCFMT_LEFT;
				lvc.cx = iColumnWidths[i];
				lvc.pszText = szColumnNames[i];

				ListView_InsertColumn(hwndItem, i, &lvc);
			}

			for(i=0; i<VDJob::ListSize(); i++) {
				LVITEM li;

				li.mask		= LVIF_TEXT;
				li.iSubItem	= 0;
				li.iItem	= i;
				li.pszText	= LPSTR_TEXTCALLBACK;

				ListView_InsertItem(hwndItem, &li);
			}

			// Init Host-List
			hwndItem = GetDlgItem(hdlg, IDC_HOSTS);
			ListView_SetExtendedListViewStyleEx(hwndItem, LVS_EX_FULLROWSELECT , LVS_EX_FULLROWSELECT);

			for (i=0;i<5;i++) {
				lvc.mask = LVCF_FMT | LVCF_SUBITEM | LVCF_TEXT | LVCF_WIDTH;
				lvc.fmt = LVCFMT_LEFT;
				lvc.cx = iHostsColumnWidths[i];
				lvc.pszText = szHostsColumnNames[i];
  			    ListView_InsertColumn(hwndItem, i, &lvc);
			}

			HostsContextMenu=LoadMenu(g_hInst, MAKEINTRESOURCE(IDR_JOBS_HOSTCONTEXT));

			Job_LoadDlgConfig();
			psSheduler=new PartSheduler(SHEDULER_SINGLEMODE);
			netMan=new NetworkManager();

			// Init Windows-Socket-Functions
			WORD wVersionRequested;
			WSADATA wsaData; 
			wVersionRequested = MAKEWORD(2,2);
			WSAStartup( wVersionRequested, &wsaData );

			// Fill Host-List (by pressing Update-Button)
			// pulco 13/12/2002 : deactivated by default. user must push update to update the list
			// PostMessage(hdlg, WM_COMMAND, MAKELONG(IDC_UPDATE,BN_CLICKED), 1);

			if (g_dubber) {
				EnableWindow(GetDlgItem(hdlg, IDC_START), FALSE);
				EnableWindow(GetDlgItem(hdlg, IDC_ABORT), FALSE);
			} else if (VDJob::IsRunInProgress()) SetDlgItemText(hdlg, IDC_START, "Stop");

			SendDlgItemMessage(hdlg, IDC_PROGRESS, PBM_SETRANGE, 0, MAKELPARAM(0, 1000));
			SendDlgItemMessage(hdlg, IDC_NETWORK_PROGRESS, PBM_SETRANGE, 0, MAKELPARAM(0, 1000));

			// 11/11/2002, Cyrius  (find it better with an icon :) )
			SendMessage(hdlg, WM_SETICON, (WPARAM)ICON_SMALL, (LPARAM)LoadIcon(g_hInst, MAKEINTRESOURCE(IDI_VIRTUALDUB)));

			guiReposInit(hdlg, jobCtlPosData, jobCtlPos);
			
			uiUpdateTimer = SetTimer(hdlg, 1, 5000, NULL);


			iUpdateVal = 0;
			SetJobWindowButtons();
			return FALSE;
		}
/////////////////////////////////////////////////////////////////////////////
	case WM_TIMER:

		// Wait for two intervals to flush.  That way, if the job list
		// is set modified right before a timer message, we don't flush
		// immediately.
		if (VDJob::IsModified()) {
			if (iUpdateVal) {
				VDJob::Flush();
				iUpdateVal = 0;
			} else
				++iUpdateVal;
		} else
			iUpdateVal = 0;

		// Update Master/Client-Progressbar 
		if (VDJob::IsNetworkJob()) {
			PostMessage(g_hwndJobs,IDC_SEND_STATUS, 0, 0);
			if (i_StartupTimer>0) --i_StartupTimer;
			if (i_StartupTimer==0) {
				i_StartupTimer=-2;
  				int Msg[2];
				Msg[0]=IDN_REQUESTPART;
				for (int i=0;i<netMan->GetHostCount();i++)
					if (netMan->IsType(i,HI_MASTER))
						__send(netMan->GetSock(i),(char*)Msg,4,0);
			}
		}



		return TRUE;
/////////////////////////////////////////////////////////////////////////////
	case WM_DRAWITEM: {
		if (wParam==IDC_STATUSMAP) {
			if ((!psSheduler)||(!netMan)) return true;
			DRAWITEMSTRUCT *diInfo=(DRAWITEMSTRUCT*)lParam;
			if (diInfo->itemAction==ODA_DRAWENTIRE) {
				HDC dc=diInfo->hDC;
				
				int FrameCount,i;
				FrameCount=psSheduler->GetFrameCount();

				int *Num,Cnt,x,y;
				Cnt=psSheduler->GetPartNumbers(&Num,PI_VALID);

				int PartXPos[512];
				int xh=diInfo->rcItem.right-diInfo->rcItem.left+1;
				int yh=diInfo->rcItem.bottom-diInfo->rcItem.top+1;

				if (FrameCount>0) {
					for (i=1;i<Cnt;i++) {
						int xpos=MulDiv(psSheduler->GetPartFirstFrame(Num[i]),xh,FrameCount);
						PartXPos[i]=xpos;
					}
				}
				PartXPos[0]=-1;
				PartXPos[Cnt]=xh;
				bool NetMode=(psSheduler)&&(VDJob::IsMasterMode())&&(psSheduler->IsAnyHostJob());
				COLORREF col;
				if (NetMode) {
					// Paint part-borders
					for (i=1;i<Cnt;i++) {
						col=0x00000000;
						if (psSheduler->IsBorderSheduled(Num[i-1],Num[i]))
							col=0x008080F0;
						for (y=0;y<yh;y++)
							SetPixel(dc,PartXPos[i],y,col);
					}
					// Paint blocks
					for (i=0;i<Cnt;i++) {
						for (x=PartXPos[i]+1;x<PartXPos[i+1];x++) {
							for (y=0;y<yh;y++) {
								switch (psSheduler->GetFrameState(Num[i], MulDiv((x-PartXPos[i]-1)*yh+y,1000,yh*(PartXPos[i+1]-PartXPos[i]-1)))) {
								case FT_TODO:col=GetSysColor(COLOR_BTNFACE);break;
								case FT_DONE:col=0x0030D030;break;
								case FT_DOING_TODO:col=0x005050F0;break;
								case FT_DOING_DONE:col=0x0070F070;break;
								}
							SetPixel(dc,x,y,col);
							} 
						}
					}
				} else {
					for (int x=0;x<xh;x++) {
						for (int y=0;y<yh;y++) {
							col=GetSysColor( (x*yh+y<MulDiv(g_iLocalProgress,xh*yh,1000)) ? 
											 COLOR_ACTIVECAPTION : COLOR_BTNFACE);
							SetPixel(dc,x+diInfo->rcItem.left,y+diInfo->rcItem.top,col);
						}
					}
				}
				free(Num);
				return true;
			}
			return true;
		}
		return false;
		}
		break;
/////////////////////////////////////////////////////////////////////////////
	case WM_NOTIFY:
		{
			NMHDR *nm = (NMHDR *)lParam;
			if (nm->idFrom == IDC_JOBS) {
				NMLVDISPINFO *nldi = (NMLVDISPINFO *)nm;
				NMLISTVIEW *nmlv;
				VDJob *vdj;
				bool fSelected;

				switch(nm->code) {

				case LVN_GETDISPINFO:
					Job_GetDispInfo(nldi);
					return TRUE;
				case LVN_ENDLABELEDIT:
					SetWindowLong(hdlg, DWL_MSGRESULT, TRUE);
					vdj = VDJob::ListGet(nldi->item.iItem);

					if (vdj && nldi->item.pszText) {
						strncpy(vdj->szName, nldi->item.pszText, sizeof vdj->szName);
						vdj->szName[sizeof vdj->szName-1] = 0;
					}
					return TRUE;
				case LVN_ITEMCHANGED:

					if (fUpdateDisable) return TRUE;

					nmlv = (NMLISTVIEW *)lParam;
					vdj = VDJob::ListGet(nmlv->iItem);

					_RPT3(0,"Item %d, subitem %d: new state is %s\n",nmlv->iItem,nmlv->iSubItem,nmlv->uNewState & LVIS_SELECTED ? "selected" : "unselected");
					fSelected = !!(nmlv->uNewState & LVIS_SELECTED);

					EnableWindow(GetDlgItem(hdlg, IDC_MOVE_UP), fSelected && nmlv->iItem>0);
					EnableWindow(GetDlgItem(hdlg, IDC_MOVE_DOWN), fSelected && nmlv->iItem<VDJob::ListSize()-1);
					EnableWindow(GetDlgItem(hdlg, IDC_DELETE), fSelected && (!vdj || vdj->iState != VDJob::INPROGRESS));
					EnableWindow(GetDlgItem(hdlg, IDC_POSTPONE), fSelected && (!vdj || vdj->iState != VDJob::INPROGRESS));

					return TRUE;

				case LVN_KEYDOWN:
					switch(((LPNMLVKEYDOWN)lParam)->wVKey) {
					case VK_DELETE:
						SendMessage(hdlg, WM_COMMAND, IDC_DELETE, (LPARAM)GetDlgItem(hdlg, IDC_DELETE));
					}
					return TRUE;

				case NM_DBLCLK:

					//	Previous state		Next state		Action
					//	--------------		----------		------
					//	Error				Waiting			Show error message
					//	Done				Waiting
					//	Postponed			Waiting
					//	Aborted				Waiting
					//	All others			Postponed

					index = ListView_GetNextItem(GetDlgItem(hdlg, IDC_JOBS), -1, LVNI_ALL | LVNI_SELECTED);
					if (index>=0) {
						vdj = VDJob::ListGet(index);

						switch(vdj->iState) {
						case VDJob::ERR:
							DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_JOBERROR), hdlg, JobErrorDlgProc, (LPARAM)vdj);
						case VDJob::DONE:
						case VDJob::ABORTED:
							vdj->iState = VDJob::WAITING;
							vdj->Refresh();
							VDJob::SetModified();
						case VDJob::INPROGRESS:
							break;
						default:
							SendMessage(hdlg, WM_COMMAND, MAKELONG(IDC_POSTPONE, BN_CLICKED), (LPARAM)GetDlgItem(hdlg, IDC_POSTPONE));
						}
					}

					return TRUE;
				}

			}
			if ((nm->idFrom == IDC_HOSTS)&&(psSheduler)&&(netMan)) {
				NMLVDISPINFO *nldi = (NMLVDISPINFO *)nm;
				switch(nm->code) {
				case LVN_GETDISPINFO: 
					{
						NetworkTransferInfo NTI;
						int Host=nldi->item.iItem;
						netMan->GetTransferInfo(Host,&NTI);

						switch(nldi->item.iSubItem) {
						case 0:	nldi->item.pszText = NTI.Name;	break;
						case 1: // State
							switch (NTI.Type) {
							case HI_OFFLINE:	nldi->item.pszText="not connected";		break;
							case HI_DISCONNECT:	nldi->item.pszText="disconnecting...";	break;
							case HI_LOCALHOST:
								if (VDJob::IsNetworkJob())
									nldi->item.pszText=(VDJob::IsMasterMode())?"Localhost (Master)":"Localhost (Slave)";
								else
									nldi->item.pszText="Localhost";
								break;
							case HI_SLAVE:		nldi->item.pszText="Slave";				break;
							case HI_MASTER:		nldi->item.pszText="Master";			break;
							case HI_ILLEGAL_MASTER:	nldi->item.pszText="Illegal (multiple) master";	break;
							} break;
						case 2: // Range
							switch (NTI.Type) {
							case HI_OFFLINE:
							case HI_DISCONNECT:
								break;
							case HI_LOCALHOST:
							case HI_MASTER:
							case HI_SLAVE:
								if (NTI.Running)
									_snprintf(nldi->item.pszText,100,"%d-%d",NTI.From,NTI.To);
							} break;
						case 3: // Percent
							switch (NTI.Type) {
							case HI_OFFLINE:
							case HI_DISCONNECT:
								break;
							case HI_LOCALHOST:
							case HI_MASTER:
							case HI_SLAVE:
									if (NTI.Running)
										_snprintf(nldi->item.pszText,100,"%d%%",NTI.Progress/10);
							} break;
						case 4: // FPS
							switch (NTI.Type) {
							case HI_OFFLINE:
							case HI_DISCONNECT:
								break;
							case HI_LOCALHOST:
						 	case HI_SLAVE:
							case HI_MASTER:
								if (NTI.Running)
									_snprintf(nldi->item.pszText,100,"%d",NTI.FPS);
							} break;
						}  // switch subitem
						return TRUE;
					}
				}
			}
		}
		break;

/////////////////////////////////////////////////////////////////////////////
	case WM_DESTROY:
		_RPT0(0,"Destroy caught -- flushing job list.\n");
		g_hwndJobs = NULL;
		VDJob::Flush();

		return TRUE;
/////////////////////////////////////////////////////////////////////////////
	case WM_CONTEXTMENU: 
		{
			LVHITTESTINFO HitInfo;
			HitInfo.pt.x=LOWORD(lParam);
			HitInfo.pt.y=HIWORD(lParam);
			ScreenToClient(GetDlgItem(hdlg, IDC_HOSTS), &HitInfo.pt);
			ListView_HitTest(GetDlgItem(hdlg, IDC_HOSTS), &HitInfo);

			if ((HitInfo.flags && LVHT_ONITEM)&&(HitInfo.iItem>=0)) {

				HostsContextMenuItemPos=HitInfo.iItem;
				int SubMenu=2;

				switch (netMan->GetType(HitInfo.iItem)) {
				case HI_SLAVE: SubMenu=0; break;
				case HI_LOCALHOST: SubMenu=1; break;
				}

				// Select Submenu
				HMENU HostsContextSubMenu = GetSubMenu( HostsContextMenu, SubMenu);

				// Load options
				Job_LoadHostConfig(HitInfo.iItem);

				// Set Menuoptions
				CHECKMENUITEM(HostsContextSubMenu, IDC_CONTXT_NEVER_SHUTDOWN,g_HostsConfig.NeverShutdown);

				// Show menu
				TrackPopupMenu(HostsContextSubMenu, TPM_LEFTALIGN | TPM_BOTTOMALIGN | TPM_LEFTBUTTON,
							   LOWORD(lParam), HIWORD(lParam), 0, g_hwndJobs, NULL);
			}
		}
		return true;
/////////////////////////////////////////////////////////////////////////////
	case WM_CLOSE:
		if ((!VDJob::IsSlaveMode())&&(!VDJob::IsMasterMode()))
		{
			if (psSheduler) delete psSheduler;
			if (netMan) delete netMan;
			DestroyMenu(HostsContextMenu);
			DestroyWindow(hdlg);
			return TRUE;
		}
		return false;
/////////////////////////////////////////////////////////////////////////////
	case WM_COMMAND:
		if (!lParam) {
			// menu hit

			Job_MenuHit(hdlg, wParam);
	
			return TRUE;

		} else if (HIWORD(wParam) == BN_CLICKED) {
			hwndItem = GetDlgItem(hdlg, IDC_JOBS);

			index = ListView_GetNextItem(hwndItem, -1, LVNI_ALL | LVNI_SELECTED);
			if (index>=0)
				vdj = VDJob::ListGet(index);
			else
				vdj = NULL;

			switch(LOWORD(wParam)) {

			case IDOK:
				if ((!VDJob::IsSlaveMode())&&(!VDJob::IsMasterMode()))
				{
					DestroyWindow(hdlg);
					return TRUE;
				}
				return FALSE;

			case IDC_DELETE:
				if (vdj) {
					// Do not delete jobs that are in progress!

					if (vdj->iState != VDJob::INPROGRESS) {
						fUpdateDisable = true;
						vdj->Delete();
						delete vdj;
						VDJob::SetModified();
						fUpdateDisable = false;
						if (VDJob::ListSize() > 0)
							ListView_SetItemState(hwndItem, index==(unsigned int)VDJob::ListSize() ? index-1 : index, LVIS_SELECTED, LVIS_SELECTED);
					}
				}
				return TRUE;

			case IDC_POSTPONE:
				if (vdj) {
					// Do not postpone jobs in progress

					if (vdj->iState != VDJob::INPROGRESS) {
						if (vdj->iState == VDJob::POSTPONED)
							vdj->iState = VDJob::WAITING;
						else
							vdj->iState = VDJob::POSTPONED;

						vdj->Refresh();

						VDJob::SetModified();
					}
				}
				return TRUE;

			case IDC_MOVE_UP:
				if (!vdj || index <= 0)
					return TRUE;

				vdj_n	= (VDJob *)vdj->next;
				vdj_p	= (VDJob *)vdj->prev;
				vdj_p2	= (VDJob *)vdj_p->prev;

				vdj_p2->next = vdj;		vdj->prev = vdj_p2;
				vdj->next = vdj_p;		vdj_p->prev = vdj;
				vdj_p->next = vdj_n;	vdj_n->prev = vdj_p;

				ListView_SetItemState(hwndItem, index  , 0, LVIS_SELECTED | LVIS_FOCUSED);
				ListView_SetItemState(hwndItem, index-1, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
				ListView_RedrawItems(hwndItem, index-1, index);

				VDJob::SetModified();

				return TRUE;

			case IDC_MOVE_DOWN:
				if (!vdj || index >= VDJob::ListSize()-1)
					return TRUE;

				vdj_p	= (VDJob *)vdj->prev;
				vdj_n	= (VDJob *)vdj->next;
				vdj_n2	= (VDJob *)vdj_n->next;

				vdj_p->next = vdj_n;	vdj_n->prev = vdj_p;
				vdj->prev = vdj_n;		vdj_n->next = vdj;
				vdj_n2->prev = vdj;		vdj->next = vdj_n2;
				
				ListView_SetItemState(hwndItem, index  , 0, LVIS_SELECTED | LVIS_FOCUSED);
				ListView_SetItemState(hwndItem, index+1, LVIS_SELECTED | LVIS_FOCUSED, LVIS_SELECTED | LVIS_FOCUSED);
				ListView_RedrawItems(hwndItem, index, index+1);

				VDJob::SetModified();

				return TRUE;

			case IDC_START:
				if (VDJob::IsRunInProgress()) {
					VDJob::RunAllStop();
					EnableWindow((HWND)lParam, FALSE);
				} else
					VDJob::RunAll();
				return TRUE;

			case IDC_ABORT:
				if (VDJob::IsRunInProgress()) {
					VDJob::RunAllStop();
					EnableWindow(GetDlgItem(hdlg, IDC_START), FALSE);
					EnableWindow((HWND)lParam, FALSE);
					if (g_dubber) g_dubber->Abort();
					if (VDJob::IsMasterMode())
						SendMessage(g_hwndJobs, IDC_STOP_NETWORKJOB, 0, 0);
				}
				return TRUE;

			///////////////////////////////
			// Shutdown Localhost after all Jobs
			///////////////////////////////
			case IDC_JOBS_SHUTDOWN:
				{
					g_JobsDlgConfig.Shutdown=(BST_CHECKED==IsDlgButtonChecked(hdlg, IDC_JOBS_SHUTDOWN));
					SetMenuEntries(true);
				}
				break;
			///////////////////////////////
			// Shutdown Slaves after all Jobs
			///////////////////////////////
			case IDC_JOBS_SHUTDOWN_ALL:
				{
					g_JobsDlgConfig.ShutdownClients=(BST_CHECKED==IsDlgButtonChecked(hdlg, IDC_JOBS_SHUTDOWN_ALL));
					SetMenuEntries(true);
				}
				break;
			///////////////////////////////
			// Reload Host-List
			///////////////////////////////
			case IDC_UPDATE: {
				SetJobWindowButtons(JWS_WAITING);
				netMan->UpdateHostList();
				SetJobWindowButtons();
				return true;
				}
				
			///////////////////////////////
			// Start Slave-Mode
			///////////////////////////////
			case IDC_SLAVE_MODE:
				if (g_JobsDlgConfig.SendToTray)
					Jobs_SendToTray(true);
				SetJobWindowButtons(JWS_WAITING);
				netMan->SetSlaveMode();
				SetJobWindowButtons();
				return true;
			///////////////////////////////
			// Start Master-Mode
			///////////////////////////////
			case IDC_MASTER_MODE:
					SetJobWindowButtons(JWS_WAITING);
					netMan->SetMasterMode();
					SetJobWindowButtons();
				return true;
			////////////////////////////////////////////
			// Stop Master- or Slave-Mode
			////////////////////////////////////////////
			case IDC_MSMODE_ABORT:
				if (VDJob::IsNetworkJob()) {
					// Check and abort Jobs
					if (VDJob::fRunInProgress) 
						SendMessage(hdlg,WM_COMMAND, MAKELONG(IDC_ABORT,BN_CLICKED), 1);
					// Close all Sockets
					for (int i=0;i<netMan->GetHostCount();i++)
					{
						if (netMan->IsType(i,HI_SLAVE,HI_MASTER))
							CleanCloseSocket(netMan->GetSock(i),hdlg,(VDJob::IsSlaveMode())?IDC_SLAVE_SOCKET:IDC_MASTER_SOCKET);
						if (netMan->IsType(i,HI_ILLEGAL_MASTER))
							netMan->SetType(i,HI_OFFLINE);
					}
					netMan->DestroyListenSocket();

					// Update Controls
					VDJob::SetSlaveMode(false);
					VDJob::SetMasterMode(false);
					netMan->SetSingleMode();

					if (psSheduler) delete psSheduler;
					psSheduler=new PartSheduler(SHEDULER_SINGLEMODE);

					SetJobWindowButtons();
					ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),netMan->GetLocalHost());
				}
				return true;
			}
		}
		break;

		
		////////////////////////////////////////////
		// Clear Logfile
		////////////////////////////////////////////
		case IDC_CLEAR_LOGFILE:
			if (psSheduler)
				psSheduler->ClearLogFile();
			break;
		////////////////////////////////////////////
		// Start distributed processing
		////////////////////////////////////////////
		case IDC_START_NETWORKJOB: {
			if (VDJob::IsMasterMode()) {
				int Message=IDN_STARTNETJOB;
				for (int i=0;i<netMan->GetHostCount();i++) {
					if (netMan->IsType(i,HI_SLAVE)) {
						__send(netMan->GetSock(i),(char*)&Message,4,0);
					}
				}
			}
		}
		break;
		////////////////////////////////////////////
		// Stop job, cancel all slaves
		////////////////////////////////////////////
		case IDC_STOP_NETWORKJOB: {
			if (VDJob::IsMasterMode()) {
				int Message=IDN_ABORTJOB;
				for (int i=0;i<netMan->GetHostCount();i++) {
					if (netMan->IsType(i,HI_SLAVE)) {    
						psSheduler->PartAborted(i);
						netMan->SetWorkState(i,false);
						__send(netMan->GetSock(i),(char*)&Message,4,0);
					}
				}
			}
		} break;
		////////////////////////////////////////////
		// Shutdown Clients
		////////////////////////////////////////////
		case IDC_SHUTDOWN_CLIENTS: {
			if (VDJob::IsMasterMode())
			{
				int Message=IDN_SHUTDOWN;
				for (int i=0;i<netMan->GetHostCount();i++) {
					if (netMan->IsType(i,HI_SLAVE)) {
						Job_LoadHostConfig(i);
						if (!g_HostsConfig.NeverShutdown)
							__send(netMan->GetSock(i),(char*)&Message,4,0);
					}
				}
			}
		} break;

		////////////////////////////////////////////
		// Send Progress
		////////////////////////////////////////////
		case IDC_SEND_STATUS: {
			if ((!VDJob::IsNetworkJob())||(!psSheduler)) return true;
			int Message[2+(sizeof(NetworkTransferInfo)/4)];
			if (VDJob::IsMasterMode()) {
				// Send Progress to all Slaves
				for (int i=0;i<netMan->GetHostCount();i++) 
					if (netMan->IsType(i,HI_SLAVE)) {
						Message[0]=IDN_PROGRESSINFO;
						Message[1]=psSheduler->GetGlobalProgress();
						__send(netMan->GetSock(i),(char*)Message,8,0);
					}
				for (i=0;i<netMan->GetHostCount();i++) {
					if (netMan->IsType(i,HI_SLAVE)) {
						for (int j=0;j<netMan->GetHostCount();j++) {
							NetworkTransferInfo NTI;
							netMan->GetTransferInfo(j,&NTI);
							if (NTI.Type==HI_LOCALHOST)
								NTI.Type=HI_MASTER;
							if (i==j)
								NTI.Type=HI_LOCALHOST;
							Message[0]=IDN_STATUSINFO;
							memcpy(&Message[1],&NTI,sizeof(NetworkTransferInfo));
							__send(netMan->GetSock(i),(char*)Message,4+sizeof(NetworkTransferInfo),0);
						}
					}
				}
			}
			if (VDJob::IsSlaveMode()) {
				// Send Progress to Master
				for (int i=0;i<netMan->GetHostCount();i++) {
					if (netMan->IsType(i,HI_MASTER)) {
						Message[0]=IDN_PROGRESSINFO;
						Message[1]=psSheduler->GetHostProgress(0);
						__send(netMan->GetSock(i),(char*)Message,8,0);
					}
				}
			}
		} break;

			
		////////////////////////////////////////////
		// Socket-Event in Server-Mode
		////////////////////////////////////////////
		case IDC_MASTER_SOCKET: {
			SOCKET s=(SOCKET)wParam;
			int Host=-1;
			for (int i=0;i<netMan->GetHostCount();i++)
				if (s==netMan->GetSock(i))
					Host=i;
			psSheduler->log(9,"Master-Socket-Event received, Host=%d, Socket=%d, Event=%d",Host,s,WSAGETSELECTEVENT(lParam));
			////////////////////////////
			//       Close Socket, Server-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_CLOSE)==FD_CLOSE)
			{
			closesocket(s);
			if (Host>=0) {
					netMan->SetSock(Host,INVALID_SOCKET);
					netMan->SetType(Host,HI_OFFLINE);
					netMan->SetWorkState(Host,false);
					if (psSheduler) {
						psSheduler->PartAborted(Host);
						psSheduler->SetActiveHost(Host,false);
						// Check if other slaves wait
						if ((!VDJob::IsRunAllStop())&&(VDJob::IsRunInProgress())) {
							for (int i=0;i<netMan->GetHostCount();i++) {
								if ((netMan->IsType(i,HI_SLAVE))&&(!psSheduler->IsHostJob(i))) {
									int Message=IDN_STARTNETJOB;
									__send(netMan->GetSock(i),(char*)&Message,4,0);
								}
							}
						}
					}
					ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),Host);
				}
			}

			////////////////////////////
			// Connection accepted, Server-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_ACCEPT)==FD_ACCEPT)
			{
				sockaddr_in RemoteHost;
				int AddrLen=sizeof(RemoteHost);
				SOCKET newsock=accept(s,(sockaddr*)&RemoteHost,&AddrLen);
				ConfigSocket(newsock);

 				// Scan for (up to now) unknown hosts
				SendMessage(g_hwndJobs, WM_COMMAND, MAKELONG(IDC_UPDATE,BN_CLICKED), 1);

				for (int i=0;i<netMan->GetHostCount();i++)
					if (netMan->GetIP(i)==htonl(RemoteHost.sin_addr.s_addr)) {
						netMan->SetSock(i,newsock);
						netMan->SetType(i,HI_SLAVE);
						netMan->SetWorkState(i,false);

						WSAAsyncSelect(newsock,hdlg,IDC_MASTER_SOCKET,FD_MOSTFLAGS);
						ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),i);

						int Message=IDN_STARTNETJOB;
						__send(netMan->GetSock(i),(char*)&Message,4,0);

						if (psSheduler) {
							psSheduler->SetActiveHost(i);
							psSheduler->LoadHostSpeed(i);
						}

						return true;
					}
				// unknown host --> disconnect 
				closesocket(newsock);
			}

			////////////////////////////
			// Read, Server-Mode == Message reveived
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_READ)==FD_READ)
			{
				if ((Host<0)||									// illegal Host
					(!VDJob::IsMasterMode())||					// illegal Mode
					(!netMan->IsType(Host,HI_SLAVE))) { // illegal Hostmode
					char tmpbuf[256];
					recv(s,tmpbuf,256,0);
					return true;
				}
				netMan->MsgbufRecv(Host,s,0);
				while (netMan->GetMsgbufLen(Host)>=4) {
					switch (netMan->GetMsgID(Host)) {
					//////////////////////////////////
					// Slave succesfully completed part
					//////////////////////////////////
					case IDN_PARTCOMPLETE:	{	// noparam
						if (psSheduler)	psSheduler->PartSuccess(Host);
						netMan->SetWorkState(Host,false);
						PostMessage(g_hwndJobs, IDC_SEND_STATUS, 0, 0);
						ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),Host);
						netMan->RemoveMessage(Host,4);
						} break;
					//////////////////////////////////
					// Slave aborted current part
					//////////////////////////////////
					case IDN_PARTABORTED:	{	// noparam
						if (psSheduler)	{
							psSheduler->PartAborted(Host);
							netMan->SetWorkState(Host,false);
							PostMessage(g_hwndJobs, IDC_SEND_STATUS, 0, 0);
							// Check if other slaves wait
							if ((VDJob::IsMasterMode())&&(!VDJob::IsRunAllStop())&&(VDJob::IsRunInProgress())) {
								for (int i=0;i<netMan->GetHostCount();i++) {
									if ((netMan->IsType(i,HI_SLAVE))&&(psSheduler->IsHostJob(i))) {
										int Message=IDN_STARTNETJOB;
										__send(netMan->GetSock(i),(char*)&Message,4,0);
									}
								}
							}
						}

						ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),Host);
						netMan->RemoveMessage(Host,4);
						} break;
					//////////////////////////////////
					// Slave requests next part
					//////////////////////////////////
					case IDN_REQUESTPART:	{	// noparam
						char newscript[SCRIPTLEN];
						int Msg[4+SCRIPTLEN>>2];
						PostMessage(g_hwndJobs, IDC_SEND_STATUS, 0, 0);
						if ((psSheduler)&&(psSheduler->GetNextPart(Host)))
						{
							PartInfo PI;
							netMan->SetWorkState(Host,true);
							psSheduler->GetPartInfo(Host,&PI);
							psSheduler->GetModifiedScript(Host,newscript);
							// Send EXECSCRIPT Message, Param: size,range_from,range_to,script[size]
							int len=strlen(newscript);
							Msg[0]=IDN_EXECSCRIPT;
							Msg[1]=len+1;
							Msg[2]=PI.From;
							Msg[3]=PI.To;
							memmove((char*)(&Msg[4]),newscript,len+1);
							__send(netMan->GetSock(Host),(char*)&Msg,len+16+1,0);
							Sleep(5);
							ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),Host);
						} else {
							// Send NOMOREPARTS Message
							Msg[0]=IDN_NOMOREPARTS;
							__send(netMan->GetSock(Host),(char*)&Msg,4,0);
						}
						netMan->RemoveMessage(Host,4);
						} break;
					//////////////////////////////////
					// Progressinfo from Slave
					//////////////////////////////////
					case IDN_PROGRESSINFO:	{	// (int)promill
						// Received Progressinfo from Slave
						if (netMan->GetMsgbufLen(Host)<8) break;
						if (psSheduler) psSheduler->SetHostProgress(Host,*(int*)netMan->GetMsgbufDataPtr(Host));
						ListView_Update(GetDlgItem(g_hwndJobs,IDC_HOSTS), Host);
						UpdateProgressInformation();
						PostMessage(g_hwndJobs, IDC_SEND_STATUS, 0, 0);
						netMan->RemoveMessage(Host,8);
						} break;
					default:
						netMan->SyncMsgQueue(Host);
					}
				}
			}

			////////////////////////////
			// Connected, Server-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_CONNECT)==FD_CONNECT) {}

			////////////////////////////
			// Write, Server-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_WRITE)==FD_WRITE) {}
		}
		return true;


		////////////////////////////////////////////
		// Socket-Event in Slave-Mode
		////////////////////////////////////////////
		case IDC_SLAVE_SOCKET: {
			SOCKET s=(SOCKET)wParam;
			int Host=-1;
			for (int i=0;i<netMan->GetHostCount();i++)
				if (s==netMan->GetSock(i))
					Host=i;
			psSheduler->log(9,"Slave-Socket-Event received, Host=%d, Socket=%d, Event=%d",Host,s,WSAGETSELECTEVENT(lParam));

			////////////////////////////
			// Close Socket, Slave-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_CLOSE)==FD_CLOSE)
			{
			closesocket(s);
			if (Host>=0)
				{
					netMan->SetSock(Host,INVALID_SOCKET);
					netMan->SetType(Host,HI_OFFLINE);
					ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),Host);
					for (i=0;i<netMan->GetHostCount();i++) 
						if (netMan->IsType(i,HI_SLAVE,HI_MASTER)) {
							netMan->SetType(i,HI_OFFLINE);
							ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),i);
						}
				}
			}

			////////////////////////////
			// Connection accepted, Slave-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_ACCEPT)==FD_ACCEPT)
			{
				sockaddr_in RemoteHost;
				int iRemoteState=HI_MASTER;
				int AddrLen=sizeof(RemoteHost);
				SOCKET newsock=accept(s,(sockaddr*)&RemoteHost,&AddrLen);
				ConfigSocket(newsock);

				for (int i=0;i<netMan->GetHostCount();i++)
					if (netMan->IsType(i,HI_MASTER))
						iRemoteState=HI_ILLEGAL_MASTER;
				for (i=0;i<netMan->GetHostCount();i++)
					if (netMan->GetIP(i)==ntohl(RemoteHost.sin_addr.s_addr)) {
						netMan->SetType(i,iRemoteState);
						if (HI_MASTER==iRemoteState) {
							netMan->SetSock(i,newsock);
							WSAAsyncSelect(newsock,hdlg,IDC_SLAVE_SOCKET,FD_MOSTFLAGS);
						} else
							closesocket(newsock);
						ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),i);
					}
				i_StartupTimer=-2;
			}

			////////////////////////////
			// Read, Slave-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_READ)==FD_READ)
			{
				if ((Host<0)||										// illegal Host
					(!VDJob::IsSlaveMode())||						// illegal Mode
					(!netMan->IsType(Host,HI_MASTER))) {			// Mission impossible
					char buf[256];
					recv(s,buf,256,0);								// Read and throw away
					return true;
				}
				netMan->MsgbufRecv(Host,s,0);
				while (netMan->GetMsgbufLen(Host)>=4) {
					switch (netMan->GetMsgID(Host)) {

					////////////////////////////////////
					// Close Client
					////////////////////////////////////
					case IDN_CLOSE: 
						// Abort current operation
						if (VDJob::IsRunInProgress()) 
							f_CloseClient=true;
						SendMessage(hdlg,WM_COMMAND, MAKELONG(IDC_MSMODE_ABORT,BN_CLICKED),1);

						// Close program
						if (!f_CloseClient)
							_exit(0);
						netMan->RemoveMessage(Host,4);
						break;

					////////////////////////////////////
					// Shutdown now
					////////////////////////////////////
					case IDN_SHUTDOWN: {	// noparam
						// Abort current operation
						SendMessage(hdlg,WM_COMMAND, MAKELONG(IDC_MSMODE_ABORT,BN_CLICKED),1);
						// shutdown
						char *ShutDownScript="VirtualDub.Shutdown();";
						_CrtCheckMemory();
						RunScriptMemory(ShutDownScript);
						netMan->RemoveMessage(Host,4);
						} break;	
					
					////////////////////////////////////
					// Init Slavefunction, request 1st part
					////////////////////////////////////
					case IDN_STARTNETJOB: {	// noparam
						if (i_StartupTimer==-2) {
							int Msg[2];
							Msg[0]=IDN_REQUESTPART;
							__send(netMan->GetSock(Host),(char*)Msg,4,0);
						} else
							i_StartupTimer=6;	// 30 seconds
						netMan->RemoveMessage(Host,4);
						} break;
					
					////////////////////////////////////
					// Abort current job
					////////////////////////////////////
					case IDN_ABORTJOB:	{	// noparam
						PostMessage(hdlg, WM_COMMAND, IDC_ABORT, (LPARAM)GetDlgItem(hdlg, IDC_ABORT));
						netMan->RemoveMessage(Host,4);
						} break;

					////////////////////////////////////
					// Execute Script
					////////////////////////////////////
					case IDN_EXECSCRIPT:	{	// (int)size, (int)From, (int)To, script[size]
						int msgsize=*(int*)netMan->GetMsgbufDataPtr(Host);
						if (netMan->GetMsgbufLen(Host)<msgsize+16) return true;
						if (psSheduler) psSheduler->SetPartInfo(*(int*)netMan->GetMsgbufDataPtr(Host,4),
																*(int*)netMan->GetMsgbufDataPtr(Host,8));
						char *NetScript=(char*)calloc(SCRIPTLEN,1);
						strcpy(NetScript,(char*)netMan->GetMsgbufDataPtr(Host,12));
						netMan->RemoveMessage(Host,16+msgsize);

						if (psSheduler) psSheduler->log(5,"Slave, received script: \n%s\n\n",NetScript);

						VDJob::fRunInProgress=true;
						VDJob::fRunAllStop=false;
						f_CloseClient=false;
						char szError[256];

						int State=NetworkRunScript(NetScript,szError);
						VDJob::fRunInProgress=false;

						SetJobWindowButtons();
						ListView_Update(GetDlgItem(hdlg, IDC_HOSTS),netMan->GetLocalHost());

						// Remove "___temp.?.avi" Files
						if (ScriptSearchLine(NetScript,"VirtualDub.video.NoAVIOutput(1)",NULL)) {
							char lbuf[LINELEN];
							if (ScriptSearchLine(NetScript,"VirtualDub.SaveAVI",lbuf)||
								ScriptSearchLine(NetScript,"VirtualDub.SaveCompatibleAVI",lbuf)) {
								char *t,*s=strstr(lbuf,"\"");
								if (s) {
									s++;
									t=strstr(s,"\"");
									if (t) {
										t[0]=0;
										remove(s);
									}
								}
							}
						}

						free(NetScript);
						int Msg;

						if (State==VDJob::DONE) {
							Msg=IDN_PARTCOMPLETE;
							__send(netMan->GetSock(Host),(char*)&Msg,4,0);
							if (!VDJob::fRunAllStop) {
								Msg=IDN_REQUESTPART;
								__send(netMan->GetSock(Host),(char*)&Msg,4,0);
							}
						} else {
							Msg=IDN_PARTABORTED;
							__send(netMan->GetSock(Host),(char*)&Msg,4,0);
							if (State==VDJob::ERR)
								MessageBox(g_hwndJobs,szError,"Joberror",MB_OK|MB_ICONERROR|MB_APPLMODAL);
						}
						if (f_CloseClient)
							_exit(0);

					} break;
					
					////////////////////////////////////
					// Request for next part failed
					////////////////////////////////////
					case IDN_NOMOREPARTS: {	// noparam
						netMan->RemoveMessage(Host,4);
						} break;

					////////////////////////////////////
					// Received progressinfo from master
					////////////////////////////////////
					case IDN_PROGRESSINFO: {	// (int)promill
						if (netMan->GetMsgbufLen(Host)<8) return true;
						if (psSheduler) psSheduler->SetGlobalProgress(*(int*)netMan->GetMsgbufDataPtr(Host));
						UpdateProgressInformation();
						netMan->RemoveMessage(Host,8);
						} break;
					////////////////////////////////////
					// Received status from master
					////////////////////////////////////
					case IDN_STATUSINFO: {		// (NetworkTransferInfo)
						if (netMan->GetMsgbufLen(Host)<4+sizeof(NetworkTransferInfo)) return true;
						netMan->SetTransferInfo( (NetworkTransferInfo*) netMan->GetMsgbufDataPtr(Host) );
						netMan->RemoveMessage(Host,4+sizeof(NetworkTransferInfo));
						} break;
					default: 
						netMan->SyncMsgQueue(Host);
					}
				}
			}

			////////////////////////////
			// Connected, Slave-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_CONNECT)==FD_CONNECT) {}
			////////////////////////////
			// Write, Slave-Mode
			////////////////////////////
			if ((WSAGETSELECTEVENT(lParam)&FD_WRITE)==FD_WRITE) {}
		}
		return true;

		/////////////////////////////////
		// Tray activity
		/////////////////////////////////
		case MYWM_TRAYICON:
			switch (lParam) {
			case WM_RBUTTONDOWN:
			case WM_LBUTTONDOWN:
				Jobs_SendToTray(false);
			}
			return true;

		case WM_GETMINMAXINFO:
		{
			LPMINMAXINFO lpmmi = (LPMINMAXINFO)lParam;

			lpmmi->ptMinTrackSize.x = rInitial.right - rInitial.left;
			lpmmi->ptMinTrackSize.y = rInitial.bottom - rInitial.top;
		}
		return TRUE;

		case WM_SIZE: {
			guiReposResize(hdlg, jobCtlPosData, jobCtlPos);
			if ((wParam==SIZE_MINIMIZED)&&(g_JobsDlgConfig.SendToTray)&&
				(VDJob::IsSlaveMode()))
				Jobs_SendToTray(true);
			}
		return true;

		}
	return false;
}

///////////////////////////////////////////////////////////////////////////
//==============================//
//=====  MODIFICATION OGM  =====//
//=====    -= Cyrius =-    =====//
//BEGIN ========================//
void JobWriteOGMConfiguration(JobScriptOutput& output) {
	ogm_stream *stream = ogm_streams;
	comment_list *comment = NULL;
	while(stream) {
		if(stream->use) {
			if(stream->type != AUDIO_TYPE_INPUTFILE) {
				output.addf("VirtualDub.OpenOGMInput(\"%s\", %d, %d);", strCify(stream->audio->getName()), stream->type, stream->offset);
			} else {
				if(stream->audio->isText)
					output.addf("VirtualDub.AddInputTextStream(%d, %d);", ((OGMTextSourceOGM *)stream->audio)->Stream(), stream->offset);
				else
					output.addf("VirtualDub.AddInputAudioStream(%d, %d);", ((OGMAudioSourceOGM *)stream->audio)->Stream(), stream->offset);
			}
		}
		stream = stream->next;
	}

	// Delete all the comments
	output.addf("VirtualDub.DeleteComments();");
	// Then add the new ones
	comment = video_comments;
	while(comment) {
		output.addf("VirtualDub.AddVideoComment(\"%s\", \"%s\");"
				, (comment->tag ? comment->tag : "")
				, (comment->comment ? comment->comment : ""));
		comment = comment->next;
	}
	comment = audio_comments;
	while(comment) {
		output.addf("VirtualDub.AddAudioComment(\"%s\", \"%s\");"
				, (comment->tag ? comment->tag : "")
				, (comment->comment ? comment->comment : ""));
		comment = comment->next;
	}
	comment = audio2_comments;
	while(comment) {
		output.addf("VirtualDub.AddAudio2Comment(\"%s\", \"%s\");"
				, (comment->tag ? comment->tag : "")
				, (comment->comment ? comment->comment : ""));
		comment = comment->next;
	}

	stream = ogm_streams;
	int i=0;
	while(stream) {
		if(stream->use) {
			comment = stream->comments;
			while(comment) {
				output.addf("VirtualDub.AddComment(%d, %d, \"%s\", \"%s\");", i, stream->type
					, (comment->tag ? comment->tag : "")
					, (comment->comment ? comment->comment : ""));
				comment = comment->next;
			}
			i++;
		}
		stream = stream->next;
	}
}
//END ==========================//

void JobCreateScript(JobScriptOutput& output, const DubOptions *opt) {
	char *mem= NULL;
	char buf[2048];

//==============================//
//=====  MODIFICATION OGM  =====//
//=====    -= Cyrius =-    =====//
//BEGIN ========================//
	// First we will need to delete the input streams
	output.addf("VirtualDub.RemoveInputStreams();");
	// Then add our configuration
	JobWriteOGMConfiguration(output);
//END ==========================//

	switch(audioInputMode) {
	case AUDIOIN_WAVE:
		output.addf("VirtualDub.audio.SetSource(\"%s\");", strCify(g_szInputWAVFile));
		break;
	case AUDIOIN_CBRMP3://pulco  09/12/2002
		output.addf("VirtualDub.audio.SetSource(\"%s\",%d);", strCify(g_szInputCBRMP3File),0);// 0 for CBR
		break;
//#ifdef NANDUB
	case AUDIOIN_MP3:
		output.addf("VirtualDub.audio.SetSource(\"%s\",%d);", strCify(g_szInputMP3File),1); // 1 for VBR
		break;
	case AUDIOIN_AC3:
		// 13/11/2002, Cyrius
		output.addf("VirtualDub.SetAC3FrameMode(%d);", (((AudioSourceAC3 *)inputAudioAC3)->usingFrameMode() ? (int)1 : (int)0));
		output.addf("VirtualDub.audio.SetSource(\"%s\");", strCify(g_szInputAC3File));
		break;
	case AUDIOIN_OGG:
		output.addf("VirtualDub.audio.SetSource(\"%s\");", strCify(g_szInputOGGFile));
		break;
//#endif
	default:
		output.addf("VirtualDub.audio.SetSource(%d);", audioInputMode);
		break;
	
	}

	output.addf("VirtualDub.audio.SetMode(%d);", opt->audio.mode);

	output.addf("VirtualDub.audio.SetInterleave(%d,%d,%d,%d,%d);",
			opt->audio.enabled,
			opt->audio.preload,
			opt->audio.interval,
			opt->audio.is_ms,
			opt->audio.offset);

	output.addf("VirtualDub.audio.SetClipMode(%d,%d);",
			opt->audio.fStartAudio,
			opt->audio.fEndAudio);

	output.addf("VirtualDub.audio.SetConversion(%d,%d,%d,%d,%d);",
			opt->audio.new_rate,
			opt->audio.newPrecision,
			opt->audio.newChannels,
			opt->audio.integral_rate,
			opt->audio.fHighQuality);

	if (opt->audio.volume)
		output.addf("VirtualDub.audio.SetVolume(%d);", opt->audio.volume);
	else
		output.addf("VirtualDub.audio.SetVolume();");

	if (g_ACompressionFormat) {
		if (g_ACompressionFormat->cbSize) {
			mem = (char *)allocmem(((g_ACompressionFormat->cbSize+2)/3)*4 + 1);
			if (!mem) throw MyMemoryError();

			membase64(mem, (char *)(g_ACompressionFormat+1), g_ACompressionFormat->cbSize);
			output.addf("VirtualDub.audio.SetCompression(%d,%d,%d,%d,%d,%d,%d,\"%s\");"
						,g_ACompressionFormat->wFormatTag
						,g_ACompressionFormat->nSamplesPerSec
						,g_ACompressionFormat->nChannels
						,g_ACompressionFormat->wBitsPerSample
						,g_ACompressionFormat->nAvgBytesPerSec
						,g_ACompressionFormat->nBlockAlign
						,g_ACompressionFormat->cbSize
						,mem
						);

			freemem(mem);
		} else
			output.addf("VirtualDub.audio.SetCompression(%d,%d,%d,%d,%d,%d);"
						,g_ACompressionFormat->wFormatTag
						,g_ACompressionFormat->nSamplesPerSec
						,g_ACompressionFormat->nChannels
						,g_ACompressionFormat->wBitsPerSample
						,g_ACompressionFormat->nAvgBytesPerSec
						,g_ACompressionFormat->nBlockAlign
						);
	} else
		output.addf("VirtualDub.audio.SetCompression();");

//#ifdef NANDUB	
	switch(audio2InputMode) {
	case AUDIOIN_WAVE:
		output.addf("VirtualDub.audio2.SetSource(\"%s\");", strCify(g_szInput2WAVFile));
		break;
	case AUDIOIN_CBRMP3://pulco  09/12/2002
		output.addf("VirtualDub.audio2.SetSource(\"%s\",%d);", strCify(g_szInput2CBRMP3File),0);// 0 for CBR
		break;
	case AUDIOIN_MP3:
		output.addf("VirtualDub.audio2.SetSource(\"%s\",%d);", strCify(g_szInput2MP3File),1);
		break;
	case AUDIOIN_AC3:
		// 13/11/2002, Cyrius
		output.addf("VirtualDub.SetAC3FrameMode(%d);", (((AudioSourceAC3 *)inputAudio2AC3)->usingFrameMode() ? (int)1 : (int)0));
		output.addf("VirtualDub.audio2.SetSource(\"%s\");", strCify(g_szInput2AC3File));
		break;
	case AUDIOIN_OGG:
		output.addf("VirtualDub.audio2.SetSource(\"%s\");", strCify(g_szInput2OGGFile));
		break;
	default:
		output.addf("VirtualDub.audio2.SetSource(%d);", audio2InputMode);
		break;
	}

	if( audio2InputMode ) {

		output.addf("VirtualDub.audio2.SetMode(%d);", opt->audio2.mode);

		output.addf("VirtualDub.audio2.SetInterleave(%d,%d,%d,%d,%d);",
				opt->audio2.enabled,
				opt->audio2.preload,
				opt->audio2.interval,
				opt->audio2.is_ms,
				opt->audio2.offset);

		output.addf("VirtualDub.audio2.SetClipMode(%d,%d);",
				opt->audio2.fStartAudio,
				opt->audio2.fEndAudio);

		output.addf("VirtualDub.audio2.SetConversion(%d,%d,%d,%d,%d);",
				opt->audio2.new_rate,
				opt->audio2.newPrecision,
				opt->audio2.newChannels,
				opt->audio2.integral_rate,
				opt->audio2.fHighQuality);

		if (opt->audio2.volume)
			output.addf("VirtualDub.audio2.SetVolume(%d);", opt->audio2.volume);
		else
			output.addf("VirtualDub.audio2.SetVolume();");

		if (g_A2CompressionFormat) {
			if (g_A2CompressionFormat->cbSize) {
				mem = (char *)allocmem(((g_A2CompressionFormat->cbSize+2)/3)*4 + 1);
				if (!mem) throw MyMemoryError();

				membase64(mem, (char *)(g_A2CompressionFormat+1), g_A2CompressionFormat->cbSize);
				output.addf("VirtualDub.audio2.SetCompression(%d,%d,%d,%d,%d,%d,%d,\"%s\");"
							,g_A2CompressionFormat->wFormatTag
							,g_A2CompressionFormat->nSamplesPerSec
							,g_A2CompressionFormat->nChannels
							,g_A2CompressionFormat->wBitsPerSample
							,g_A2CompressionFormat->nAvgBytesPerSec
							,g_A2CompressionFormat->nBlockAlign
							,g_A2CompressionFormat->cbSize
							,mem
							);

				freemem(mem);
			} else
				output.addf("VirtualDub.audio2.SetCompression(%d,%d,%d,%d,%d,%d);"
							,g_A2CompressionFormat->wFormatTag
							,g_A2CompressionFormat->nSamplesPerSec
							,g_A2CompressionFormat->nChannels
							,g_A2CompressionFormat->wBitsPerSample
							,g_A2CompressionFormat->nAvgBytesPerSec
							,g_A2CompressionFormat->nBlockAlign
							);
		} else
			output.addf("VirtualDub.audio2.SetCompression();");

	} // if( audio2InputMode )

//#endif //NANDUB


	output.addf("VirtualDub.video.SetDepth(%d,%d);",
			16+8*opt->video.inputDepth,
			16+8*opt->video.outputDepth);

	output.addf("VirtualDub.video.SetMode(%d);",
			opt->video.mode);

	output.addf("VirtualDub.video.SetFrameRate(%d,%d);",
			opt->video.frameRateNewMicroSecs,
			opt->video.frameRateDecimation);

	output.addf("VirtualDub.video.SetIVTC(%d,%d,%d,%d);",
			opt->video.fInvTelecine,
			opt->video.fIVTCMode,
			opt->video.nIVTCOffset,
			opt->video.fIVTCPolarity);

	output.addf("VirtualDub.video.SetRange(%d,%d);",
			opt->video.lStartOffsetMS,
			opt->video.lEndOffsetMS);

#ifndef NANDUB
	DWORD l;

	if ((g_Vcompression.dwFlags & ICMF_COMPVARS_VALID) && g_Vcompression.fccHandler) {
		output.addf("VirtualDub.video.SetCompression(0x%08lx,%d,%d,%d);",
				g_Vcompression.fccHandler,
				g_Vcompression.lKey,
				g_Vcompression.lQ,
				g_Vcompression.lDataRate);

		l = ICGetStateSize(g_Vcompression.hic);

		if (l>0) {
			mem = (char *)allocmem(l + ((l+2)/3)*4 + 1);
			if (!mem) throw MyMemoryError();

			if (ICGetState(g_Vcompression.hic, mem, l)<0) {
				freemem(mem);
//				throw MyError("Bad state data returned from compressor");

				// Fine then, be that way.  Stupid Pinnacle DV200 driver.
			}

			if (mem) {
				membase64(mem+l, mem, l);
				output.addf("VirtualDub.video.SetCompData(%d,\"%s\");", l, mem+l);
				freemem(mem);
			}
		}

	} else
		output.addf("VirtualDub.video.SetCompression();");

#endif

#ifdef NANDUB

	output.addf( opt->video.codecUseMP42 ?
			"VirtualDub.video.SetMPEG_four_two(%d,%d);" :
			"VirtualDub.video.SetDivX(%d,%d);",
			opt->video.codecBitrate,
			opt->video.codecKFI);

	output.addf( "VirtualDub.video.SetQualityControl(%d,%d,%d,%d);",
			opt->video.codecQCMode,
			opt->video.codecAntiShit,
			opt->video.codecMinPsnr,
			opt->video.codecMotionModulation);

	output.addf("VirtualDub.video.SetMotionDetection(%d,%d,%d,%d);",
			opt->video.motionSpan,
			opt->video.motionSensitivity,
			opt->video.motionFast,
			opt->video.motionLow);

	output.addf("VirtualDub.video.SetCrispness(%d,%d);",
			opt->video.motionCrisp,
			opt->video.motionSetLimit);

	output.addf("VirtualDub.video.SpaceKF(%d);",
			opt->video.spaceKF);

	output.addf("VirtualDub.video.InternalSCD(%d);",
			opt->video.internalSCD);

	output.addf("VirtualDub.video.SetMinKBPS(%d);",
			opt->video.codecMinKBPS);

	output.addf("VirtualDub.video.SetCurveFile(\"%s\");",
			strCify(opt->video.statsFile) );

	output.addf("VirtualDub.video.SetCurveMcFactor(%d);",
			opt->video.curveMcFactor);

	if( opt->video.curveCompressLo==opt->video.curveCompressHi ) {
		output.addf("VirtualDub.video.SetCurveCompression(%d,%d);",
				opt->video.curveCompressLo,
				opt->video.curveSmooth);
	} else {
		output.addf("VirtualDub.video.SetCurveCompression(%d,%d,%d);",
				opt->video.curveCompressLo,
				opt->video.curveCompressHi,
				opt->video.curveSmooth);
	}

	output.addf("VirtualDub.video.SetCurveFilter(%d,%d);",
			opt->video.curveHighpass,
			opt->video.curveLowpass);
	output.addf("VirtualDub.video.SetCurveCredits(%d,%d);",
			opt->video.curveCredits,
			opt->video.curveCredRate);

	output.addf("VirtualDub.video.SetLumaCorrectionAmp(%d,%d,%d);",
			opt->video.curveLuma,
			opt->video.curveLumaT,
			opt->video.curveLumaG);

	output.addf("VirtualDub.video.SetCurveRedist(%d);",
			opt->video.curveRedistMode);


	output.addf("// VirtualDub.video.CalcCurveCompression();");

	output.addf("VirtualDub.video.SetCompLevelsMain(%d,%d);",
			opt->video.minCompress,
			opt->video.maxCompress);

	output.addf("VirtualDub.video.SetCompLevelsA(%d,%d,%d);",
			opt->video.motionFloor1,
			opt->video.minCompress2,
			opt->video.maxCompress2);

	output.addf("VirtualDub.video.SetCompLevelsB(%d,%d,%d);",
			opt->video.motionFloor2,
			opt->video.minCompress3,
			opt->video.maxCompress3);

	output.addf("VirtualDub.video.SetCompLevelsC(%d,%d,%d);",
			opt->video.motionFloor3,
			opt->video.minCompress4,
			opt->video.maxCompress4);

	output.addf("VirtualDub.video.SetCompLevelsD(%d,%d,%d);",
			opt->video.motionFloor4,
			opt->video.minCompress5,
			opt->video.maxCompress5);

	output.addf("VirtualDub.video.SetCompLevelsE(%d,%d,%d);",
			opt->video.motionFloor5,
			opt->video.minCompress6,
			opt->video.maxCompress6);

	output.addf("VirtualDub.video.SetCompLevelK(%d,%d);",
			opt->video.kfQuality,
			opt->video.kfMinQuality);


	output.addf("VirtualDub.video.SetBitsReservoir(%d,%d,%d,%d,%d,%d);",
			opt->video.boostKF,
			opt->video.startGauge,
			opt->video.minGauge,
			opt->video.maxGauge,
			opt->video.paybackDelay,
			opt->video.freezeBR);

	output.addf("VirtualDub.video.SetLowBrCorrection(%d,%d);",
			opt->video.lbrCorrect,
			opt->video.lbrModulated);


	output.addf("VirtualDub.video.NoAVIOutput(%d);",
			opt->video.fNoAVI);

	output.addf("VirtualDub.video.GenStats(\"%s\",%d);",
			strCify(opt->video.statsCollect),
			opt->video.collectKF);

	output.addf("VirtualDub.video.SetEncodingControl(\"%s\");",
			strCify(opt->video.ecfFilename) );

#endif

	output.addf("VirtualDub.video.filters.Clear();");

	// Add filters

	FilterInstance *fa = (FilterInstance *)g_listFA.tail.next, *fa_next;
	int iFilter = 0;

	while(fa_next = (FilterInstance *)fa->next) {
		output.addf("VirtualDub.video.filters.Add(\"%s\");", strCify(fa->filter->name));

		if (fa->x1 || fa->y1 || fa->x2 || fa->y2)
			output.addf("VirtualDub.video.filters.instance[%d].SetClipping(%d,%d,%d,%d);"
						,iFilter
						,fa->x1
						,fa->y1
						,fa->x2
						,fa->y2
						);

		if (fa->filter->fssProc && fa->filter->fssProc(fa, &g_filterFuncs, buf, sizeof buf))
			output.addf("VirtualDub.video.filters.instance[%d].%s;", iFilter, buf);

		++iFilter;
		fa = fa_next;
	}

	// Add subset information

	if (inputSubset) {
		FrameSubsetNode *pfsn;

		output.addf("VirtualDub.subset.Clear();");

		if (pfsn = inputSubset->getFirstFrame())
			do {
#ifdef NANDUB
				output.addf("VirtualDub.subset.AddFrame(%ld,%ld);", pfsn->start, pfsn->len);
#else
				output.addf("VirtualDub.subset.Add%sRange(%ld,%ld);", pfsn->bMask ? "Masked" : "", pfsn->start, pfsn->len);
#endif
			} while(pfsn = inputSubset->getNextFrame(pfsn));
	} else
		output.addf("VirtualDub.subset.Delete();");

#ifdef NANDUB
/* KBRC START */
	output.addf( "VirtualDub.brc.Set( 0, %d );", opt->brc.finalSize );
	output.addf( "VirtualDub.brc.Set( 1, %d );", opt->brc.multSize );
	output.addf( "VirtualDub.brc.Set( 2, %d );", opt->brc.audioKbps );
	output.addf( "VirtualDub.brc.Set( 3, %d );", opt->brc.audioSize );
	output.addf( "VirtualDub.brc.Set( 4, %d );", opt->brc.useAudioSize );
/* KBRC END */
#endif

}

void JobAddConfigurationInputs(JobScriptOutput& output, const char *szFileInput, int iFileMode, List2<InputFilenameNode> *pListAppended) {
	do {
		if (g_pInputOpts) {
			int l;
			char buf[256];

			l = g_pInputOpts->write(buf, (sizeof buf)/7*3);

			if (l) {
				membase64(buf+l, (char *)buf, l);
				output.addf("VirtualDub.Open(\"%s\",%d,0,\"%s\");",strCify(szFileInput), iFileMode, buf+l);
				break;
			}
		}

		output.addf("VirtualDub.Open(\"%s\",%d,0);",strCify(szFileInput), iFileMode);

	} while(false);

	if (pListAppended) {
		InputFilenameNode *ifn = pListAppended->AtHead(), *ifn_next;

		if (ifn = ifn->NextFromHead())
			while(ifn_next = ifn->NextFromHead()) {
				output.addf("VirtualDub.Append(\"%s\");", strCify(ifn->name));
				ifn = ifn_next;
			}
	}
}

//==============================//
//=====  MODIFICATION OGM  =====//
//=====    -= Cyrius =-    =====//
//BEGIN ========================//
void JobAddConfiguration(const DubOptions *opt, const char *szFileInput, int iFileMode, const char *szFileOutput, _avi_info *infos, bool fCompatibility, List2<InputFilenameNode> *pListAppended, long lSpillThreshold, long lSpillFrameThreshold, const char* outputType, int audioOnly) {
//END ==========================//
	VDJob *vdj = new VDJob;
	JobScriptOutput output;

	try {

		// 13/11/2002, Cyrius
		output.addf("VirtualDub.SetNandubCompatibility(%d);", (nandub_compatibility ? (int)1 : (int)0));

		JobAddConfigurationInputs(output, szFileInput, iFileMode, pListAppended);
		JobCreateScript(output, opt);

		// Add actual run option

//==============================//
//=====  MODIFICATION OGM  =====//
//=====    -= Cyrius =-    =====//
//BEGIN ========================//
		if(audioOnly == 0) {
	    if(!infos) {
				if (lSpillThreshold)
				  output.addf("VirtualDub.SaveSegmented%s(\"%s\", %d, %d);", outputType, strCify(szFileOutput), lSpillThreshold, lSpillFrameThreshold);
			  else
				  output.addf("VirtualDub.Save%s%s(\"%s\");", fCompatibility ? "Compatible" : "", outputType, strCify(szFileOutput));
	    } else {
				if (lSpillThreshold)
				  output.addf("VirtualDub.SaveSegmented%s(\"%s\", %d, %d);", outputType, strCify(szFileOutput), lSpillThreshold, lSpillFrameThreshold);
				else {
					static char szTemp[4*(256+4)]; // because of strCify...
					strcpy( szTemp, strCify(infos->szSubject) );
					strcat( szTemp, "\", \"" );
					strcat( szTemp, strCify(infos->szAuthor) );
					strcat( szTemp, "\", \"" );
					strcat( szTemp, strCify(infos->szCopy) );
					strcat( szTemp, "\", \"" );
					strcat( szTemp, strCify(infos->szComment) );

					output.addf("VirtualDub.Save%s%s(\"%s\",\"%s\");"
						, fCompatibility ? "Compatible" : ""
						, outputType
						, strCify(szFileOutput)
						, szTemp);
				}
			}
		} else {
			if(audioOnly & 0xF0) {
				// Demux without WAV header
				output.addf("VirtualDub.DemuxAudio(\"%s\");", strCify(szFileOutput));
			} else {
				// Demux with WAV header
				output.addf("VirtualDub.SaveWAV(\"%s\");", strCify(szFileOutput));
			}
		}
//END ==========================//
		output.adds("VirtualDub.Close();");

		///////////////////

		strncpy(vdj->szInputFile, szFileInput, sizeof vdj->szInputFile);
		strncpy(vdj->szOutputFile, szFileOutput, sizeof vdj->szOutputFile);
		sprintf(vdj->szName, "Job %d", VDJob::job_number++);

		vdj->script = output.getscript();
		vdj->Add();
	} catch(...) {
		freemem(vdj);
		throw;
	}
}

void JobAddConfigurationImages(const DubOptions *opt, const char *szFileInput, int iFileMode, const char *szFilePrefix, const char *szFileSuffix, int minDigits, int imageFormat, List2<InputFilenameNode> *pListAppended) {
	VDJob *vdj = new VDJob;
	JobScriptOutput output;
	char *s = NULL;

	try {
		JobAddConfigurationInputs(output, szFileInput, iFileMode, pListAppended);
		JobCreateScript(output, opt);

		// Add actual run option

		s = strdup(strCify(szFilePrefix));		// I swear I will clean this mess up in 1.5....

		output.addf("VirtualDub.SaveImageSequence(\"%s\", \"%s\", %d, %d);", s, strCify(szFileSuffix), minDigits, imageFormat);

		free(s);
		s = NULL;

		output.adds("VirtualDub.Close();");

		///////////////////

		strncpy(vdj->szInputFile, szFileInput, sizeof vdj->szInputFile);
		_snprintf(vdj->szOutputFile, sizeof vdj->szOutputFile, "%s*%s", szFilePrefix, szFileSuffix);
		sprintf(vdj->szName, "Job %d", VDJob::job_number++);

		vdj->script = output.getscript();
		vdj->Add();
	} catch(...) {
		free(s);
		freemem(vdj);
		throw;
	}
}

void JobWriteConfiguration(FILE *f, DubOptions *opt) {
	JobScriptOutput output;
	char *scr;

	JobCreateScript(output, opt);

	scr = output.getscript();

	if (fputs(scr, f)<0 || fflush(f)) {
		freemem(scr);
		throw MyError("Can't write configuration: %s.", strerror(errno));
	}

	freemem(scr);
}

///////////////////////////////////////////////////////////////////////////

bool InitJobSystem() {
	VDJob::ListLoad();

	return true;
}

void DeinitJobSystem() {
	VDJob::ListClear(true);
}

void OpenJobWindow() {
	if (g_hwndJobs) return;

	g_hwndJobs = CreateDialog(g_hInst, MAKEINTRESOURCE(IDD_JOBCONTROL), NULL, JobCtlDlgProc);

}

void StartSlaveMode() {
	PostMessage(g_hwndJobs, WM_COMMAND, MAKELONG(IDC_SLAVE_MODE,BN_CLICKED), 1);
}


void CloseJobWindow() {
	if (g_hwndJobs)
		DestroyWindow(g_hwndJobs);
}

void JobLockDubber() {
	if (g_hwndJobs) {
		EnableWindow(GetDlgItem(g_hwndJobs, IDC_START), FALSE);
		EnableWindow(GetDlgItem(g_hwndJobs, IDC_ABORT), FALSE);
	}
}

void JobUnlockDubber() {
	if (g_hwndJobs) {
		EnableWindow(GetDlgItem(g_hwndJobs, IDC_START), TRUE);
		EnableWindow(GetDlgItem(g_hwndJobs, IDC_ABORT), TRUE);
	}
}

void UpdateProgressInformation() {
	if (!g_hwndJobs) return;

	char buf[8];
	int local =g_iLocalProgress;

	SendMessage(GetDlgItem(g_hwndJobs, IDC_PROGRESS), PBM_SETPOS, local, 0);
	wsprintf(buf, "%d%%", local/10);
	SetDlgItemText(g_hwndJobs, IDC_PERCENT, buf);

	if (!psSheduler) return;

	int global=psSheduler->GetGlobalProgress();

	SendMessage(GetDlgItem(g_hwndJobs, IDC_NETWORK_PROGRESS), PBM_SETPOS, global, 0);
	wsprintf(buf, "%d%%", global/10);
	SetDlgItemText(g_hwndJobs, IDC_NETWORK_PERCENT, buf);

	InvalidateRect(GetDlgItem(g_hwndJobs,IDC_STATUSMAP),0,false);

}


void JobPositionCallback(LONG start, LONG cur, LONG end) {
	g_iLocalProgress=MulDiv(cur-start, 1000, end-start);
	if (psSheduler) {
		psSheduler->SetHostProgress(netMan->GetLocalHost(),g_iLocalProgress);
		if (!VDJob::IsNetworkJob()) psSheduler->SetGlobalProgress(g_iLocalProgress);
	}
	UpdateProgressInformation();
	ListView_Update(GetDlgItem(g_hwndJobs,IDC_HOSTS),netMan->GetLocalHost());
}

void JobPositionCallback(LONG start, LONG cur, LONG end, int progress) {
	// 27/11/2002, Cyrius
	JobPositionCallback(0, progress, 12288/*8192 = 2 (audio + video) * 4096*/);
}

void JobClearList() {
	VDJob::ListClear();
}

void JobRunList() {
	VDJob::RunAll();
}

void JobAddBatchDirectory(const char *lpszSrc, const char *lpszDst) {
	// Scan source directory

	HANDLE				h;
	WIN32_FIND_DATA		wfd;
	char *s, *t;
	char szSourceDir[MAX_PATH], szDestDir[MAX_PATH];

	strcpy(szSourceDir, lpszSrc);
	strcpy(szDestDir, lpszDst);

	s = szSourceDir;
	t = szDestDir;

	if (*s) {

		// If the path string is just \ or starts with x: or ends in a slash
		// then don't append a slash

		while(*s) ++s;

		if ((s==szSourceDir || s[-1]!='\\') && (!isalpha(szSourceDir[0]) || szSourceDir[1]!=':' || szSourceDir[2]))
			*s++ = '\\';

	}
	
	if (*t) {

		// If the path string is just \ or starts with x: or ends in a slash
		// then don't append a slash

		while(*t) ++t;

		if ((t==szDestDir || t[-1]!='\\') && (!isalpha(szDestDir[0]) || szDestDir[1]!=':' || szDestDir[2]))
			*t++ = '\\';

	}

	strcpy(s,"*.*");

	h = FindFirstFile(szSourceDir,&wfd);

	if (INVALID_HANDLE_VALUE != h) {
		do {
			if (!(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
				char *t2, *dot = NULL;

				strcpy(s, wfd.cFileName);
				strcpy(t, wfd.cFileName);
 
				// Replace extension with .avi

				t2 = t;
				while(*t2) if (*t2++ == '.') dot = t2;

				if (dot)
					strcpy(dot, "avi");
				else
					strcpy(t2, ".avi");

				// Add job!

				JobAddConfiguration(&g_dubOpts, szSourceDir, 0, szDestDir, NULL, false, NULL, 0, 0);
			}
		} while(FindNextFile(h,&wfd));
		FindClose(h);
	}
}










































