#ifndef __BUNDLE_DEPEND_H_INCLUDED
#define __BUNDLE_DEPEND_H_INCLUDED 1

#include "version.h"
#include <string>
#include <list>
#include <boost/shared_ptr.hpp>

namespace RUNES {


class BundleManager;


// Base Class
class BundleDepend {
public:
	BundleDepend() {}
	virtual ~BundleDepend() {}
public:
	typedef enum {
		DEP_AND,
		DEP_OR,
		DEP_BUNDLE,
		DEP_FILE,
		DEP_LIB,
		DEP_BIN,
	} depend_type;
public:
	// 共通インターフェース
	virtual depend_type getType(void) const = 0;
	virtual void show(void) const = 0;
};


// Logic Base Class
class BDLogic : public BundleDepend {
public:
	BDLogic() {}
	virtual ~BDLogic() {}
public:
	typedef std::list< boost::shared_ptr<BundleDepend> > entries_t;
public:
	// 論理依存インターフェース
	template <typename T> T* addTarget(const T& target)
	{
		T* p = new T(target);
		m_entries.push_back( boost::shared_ptr<BundleDepend>(p) );
		return p;
	}
	const entries_t& getEntries(void) const { return m_entries; }

	// BundleHoldable
	void getRecursivePreOrder (std::list<std::string>* order_list) const;
	void getRecursivePostOrder(std::list<std::string>* order_list) const;
private:
	entries_t m_entries;
};


class BDLogicAND : public BDLogic {
public:
	BDLogicAND() {}
	~BDLogicAND() {}
public:
	// 共通インターフェース
	depend_type getType(void) const { return DEP_AND; }
	void show(void) const;
};

class BDLogicOR : public BDLogic {
public:
	BDLogicOR() {}
	~BDLogicOR() {}
public:
	// 共通インターフェース
	depend_type getType(void) const { return DEP_OR; }
	void show(void) const;
};


// Terminal Node Base Class
class BDTerm : public BundleDepend {
public:
	BDTerm() {}
	virtual ~BDTerm() {}
};

class BDTBundle : public BDTerm {
public:
	typedef enum {
		ORDER_ANY,
		ORDER_PRE,
		ORDER_POST,
	} order_t;
public:
	BDTBundle(
			const std::string& name,
			const VersionRange& version_range,
			order_t order = ORDER_PRE
			) :
		m_name(name),
		m_version_range(version_range),
		m_order(order)
	{}
	~BDTBundle() {}
public:
	const std::string& getName(void) const { return m_name; }
	const VersionRange& getVersionRange(void) const { return m_version_range; }
	order_t getOrder(void) const { return m_order; }
private:
	const std::string m_name;
	const VersionRange m_version_range;
	const order_t m_order;
public:
	// 共通インターフェース
	depend_type getType(void) const { return DEP_BUNDLE; }
	virtual void show(void) const;
private:
	BDTBundle();
};


class BDTHost : public BDTerm {
public:
	BDTHost(const std::string& name) : m_name(name) {}
	virtual ~BDTHost() {}
public:
	const std::string& getName(void) const { return m_name; }
private:
	const std::string m_name;
};

class BDTHostFile : public BDTHost {
public:
	BDTHostFile(const std::string& path) : BDTHost(path) {}
	~BDTHostFile() {}
public:
	// 共通インターフェース
	depend_type getType(void) const { return DEP_FILE; }
	virtual void show(void) const;
private:
	BDTHostFile();
};

class BDTHostBin : public BDTHost {
public:
	BDTHostBin(const std::string& name) : BDTHost(name) {};
	~BDTHostBin() {};
public:
	// 共通インターフェース
	depend_type getType(void) const { return DEP_BIN; }
	virtual void show(void) const;
private:
	BDTHostBin();
};


class BDTHostLib : public BDTHost {
public:
	// 共通インターフェース
	BDTHostLib(const std::string& name) : BDTHost(name) {};
	~BDTHostLib() {};
public:
	depend_type getType(void) const { return DEP_LIB; }
	virtual void show(void) const;
private:
	BDTHostLib();
};


}  // namespace RUNES

#endif /* bundle_depend.h */
