#include "bundlemgr_impl.h"
#include "bundle.h"
#include "runestree.h"
#include "filesystem.h"
#include "exception.h"
#include <sys/types.h>
#include <dirent.h>
#include <string.h>

#include <iostream>

namespace RUNES {


// コンストラクタ
BundleManagerIMPL::BundleManagerIMPL(const std::string& bundles_dir) :
	m_bundles_dir(bundles_dir)
{}


// デストラクタ
BundleManagerIMPL::~BundleManagerIMPL() {}


// public:
void BundleManagerIMPL::initialize(void)
{
	DIR *dir_bundles;
	dir_bundles = ::opendir( m_bundles_dir.c_str() );
	if( dir_bundles == NULL ) {
		throw std::runtime_error( std::string("Can't open ") + m_bundles_dir );
	}

	struct dirent* di;
	while( (di = ::readdir(dir_bundles)) != NULL ) {
		const std::string dirname(di->d_name);
		if( ! FileSystem::isSuffix(dirname, BundleComponent::BUNDLE_SUFFIX) ) {
			continue;
		}

		try {	// XXX: 例外のクラスを作る
			Bundle bundle(m_bundles_dir + "/" + dirname);
			bundle.initialize();
			m_bundles.insert( std::make_pair(bundle.getName(), bundle) );
		}
		catch( const std::runtime_error& ex ) {
			// XXX: 例外処理？
		}
	}

	::closedir(dir_bundles);
}

const Bundle* BundleManagerIMPL::find(const std::string& name) const
{
	name_bundle_map_t::const_iterator found( m_bundles.find(name) );
	if( found == m_bundles.end() ) {
		return NULL;
	} else {
		return &(found->second);
	}
}

const std::string& BundleManagerIMPL::getBaseDir(void) const
{
	return m_bundles_dir;
}

const std::list<const Bundle*> BundleManagerIMPL::getList(void) const
{
	std::list<const Bundle*> list;
	for( name_bundle_map_t::const_iterator bdl( m_bundles.begin() );
			bdl != m_bundles.end();
			++bdl ) {
		list.push_back( &(bdl->second) );
	}
	return list;
}

void BundleManagerIMPL::show(void) const
{
	// XXX: std::for_each( mem_fun(Bundle::show)
	for( name_bundle_map_t::const_iterator bdl( m_bundles.begin() );
			bdl != m_bundles.end();
			++bdl ) {
		bdl->second.show();
	}
}


}  // namespace RUNES
