#include "sax.h"
#include "libparsifal/parsifal.h"
#include <exception>
#include <stdexcept>
#include <stdio.h>
#include <errno.h>

namespace RUNES {


class SAXMapper {
public:
	SAXMapper();
	~SAXMapper();
public:
	static int StartElement(
			void *UserData,
			const XMLCH *uri,
			const XMLCH *localName,
			const XMLCH *qName,
			LPXMLVECTOR atts
			);

	static int EndElement(
			void *UserData,
			const XMLCH *uri,
			const XMLCH *localName,
			const XMLCH *qName
			);

	static int Characters(
			void *UserData,
			const XMLCH *Chars,
			int cbChars
			);

	static void ErrorHandler(
			LPXMLPARSER parser
			);

	static int cstream(
			BYTE *buf,
			int cBytes,
			int *cBytesActual,
			void *inputData
			);
};


SAXMapper::SAXMapper() {}
SAXMapper::~SAXMapper() {}

int SAXMapper::StartElement(
		void *UserData,
		const XMLCH *uri,
		const XMLCH *localName,
		const XMLCH *qName,
		LPXMLVECTOR atts
		)
{
	// qName: unsigned char*  =>  char*
	std::string name( reinterpret_cast<const char*>(qName) );

	SAX::attributes_t attributes;
	for( int i=0; i < atts->length; i++ ) {
		LPXMLRUNTIMEATT att = static_cast<LPXMLRUNTIMEATT>( XMLVector_Get(atts,i) );
		attributes.push_back(
				std::make_pair(
					reinterpret_cast<const char*>(att->qname),
					reinterpret_cast<const char*>(att->value)
					)
				);
	}

	SAX* self = static_cast<SAX*>(UserData);
	self->startElementHandler(name, attributes);
	return 0;
}

int SAXMapper::EndElement(
		void *UserData,
		const XMLCH *uri,
		const XMLCH *localName,
		const XMLCH *qName
		)
{
	// qName: unsigned char*
	std::string name( reinterpret_cast<const char*>(qName) );

	SAX* self = static_cast<SAX*>(UserData);
	self->endElementHandler(name);
	return 0;
}

int SAXMapper::Characters(
		void *UserData,
		const XMLCH *Chars,
		int cbChars
		)
{
	std::string text(reinterpret_cast<const char*>(Chars), cbChars);

	SAX* self = static_cast<SAX*>(UserData);
	self->textHandler(text);
	return 0;
}

void SAXMapper::ErrorHandler(
		LPXMLPARSER parser
		)
{
	std::string message( reinterpret_cast<const char*>(parser->ErrorString) );
	unsigned long long line = parser->ErrorLine;
	unsigned long long column = parser->ErrorColumn;

	SAX* self = static_cast<SAX*>(parser->UserData);
	self->errorHandler(message, line, column);
}

int SAXMapper::cstream(
		BYTE *buf,
		int cBytes,
		int *cBytesActual,
		void *inputData
		)
{
	std::istream& stream( *static_cast<std::istream*>(inputData) );
	stream.read( (char*)buf, cBytes);
	if( stream.bad() ) {
		return BIS_ERR_INPUT;
	}
	*cBytesActual = stream.gcount();

	if( stream.eof() ) {
		return BIS_EOF;
	} else {
		return 0;
	}
}



SAX::SAX(std::istream& source_stream) : m_stream(source_stream) {}
SAX::~SAX() {}

void SAX::parse(void)
{
	LPXMLPARSER parser;

	if( !XMLParser_Create(&parser) ) {
		throw std::runtime_error("Can't create xml parser");
	}

	parser->startElementHandler = SAXMapper::StartElement;
	parser->endElementHandler = SAXMapper::EndElement;
	parser->charactersHandler = SAXMapper::Characters;
	parser->errorHandler = SAXMapper::ErrorHandler;
	parser->UserData = this;

	initializer();
	if( !XMLParser_Parse(parser, SAXMapper::cstream, &m_stream, 0) ) {
		throw std::runtime_error("Can't parse xml");
	}
	finalizer();
}


}  // namespace RUNES
