#include "targetmgr.h"
#include "bundle.h"
#include "role.h"
#include "runestree.h"
#include <boost/bind.hpp>

#include <iostream>

namespace RUNES {

namespace Applyer {


// コンストラクタ
TargetManager::TargetManager(const BundleManager& bmgr, const RoleManager& rmgr, const std::list<std::string> use_roles) :
	m_bmgr(bmgr),
	m_rmgr(rmgr),
	m_use_roles(use_roles)
{}


// デストラクタ
TargetManager::~TargetManager() {}


// public:
void TargetManager::initialize(void)
{
	// roleは後に指定されたものを優先する
	for( std::list<std::string>::const_reverse_iterator rl(m_use_roles.rbegin()), rl_end(m_use_roles.rend());
			rl != rl_end;
			++rl ) {
		const Role* found_role( m_rmgr.find(*rl) );
		if( found_role == NULL ) {
			// 指定されたRoleが見つからない
			// XXX: 例外処理
			// XXX: COLOR!
			std::cout << "No such role: " << *rl << std::endl;
			continue;
		}
		std::list<std::string> parts( found_role->getParts() );
		for( std::list<std::string>::const_iterator pt(parts.begin()), pt_end(parts.end());
				pt != pt_end;
				++pt ) {
			const Bundle* found_bundle( m_bmgr.find(*pt) );
			if( found_bundle == NULL ) {
				// Roleの中にconfが置いてあるのに対応するBundleが無い
				// XXX: 例外処理
				std::cout << "No such bundle (role-bundle linking): " << *pt << std::endl;
				continue;
			}
			// 失敗しても無視（roleは後に指定されたものを優先する）
			m_targets.insert(	// class Targetのコピー演算
					std::make_pair(
						found_bundle->getName(),
						Target(	*found_bundle,
							found_role->getPath()+"/"+(*pt)+RolesDir::PARTS_SUFFIX
							)
						)
					);
		}
	}
}


// for linking
void TargetManager::link(void)
{
	for( name_target_map_t::iterator tgt_pair( m_targets.begin() );
			tgt_pair != m_targets.end();
			++tgt_pair ) {
		Target& tgt(tgt_pair->second);

		// pre orderをリンク
		const std::list<std::string> pre_orders( tgt.getSourcePreOrder() );
		for( std::list<std::string>::const_iterator pre( pre_orders.begin() );
				pre != pre_orders.end();
				++pre ) {
			Target* found_target( find(*pre) );
			if( found_target == NULL ) {
				// XXX: 例外処理
				// XXX: COLOR!
				std::cout << "No such bundle (pre order resolving): " << *pre << std::endl;
			} else {
				tgt.pushPreRequre(found_target);
			}
		}

		// post orderを相手先にpush
		const std::list<std::string>& post_orders( tgt.getSourcePostOrder() );
		for( std::list<std::string>::const_iterator post( post_orders.begin() );
				post != post_orders.end();
				++post ) {
			Target* found_target( find(*post) );
			if( found_target == NULL ) {
				// XXX: 例外処理
				// XXX: COLOR!
				std::cout << "No such bundle (post order resolving): " << *post << std::endl;
			} else {
				found_target->pushPreRequre( &tgt );
			}
		}

	}
}

// for applying
void TargetManager::apply(void)
{
	for( name_target_map_t::iterator tgt( m_targets.begin() );
			tgt != m_targets.end();
			++tgt ) {
		tgt->second.start();
	}

	for( name_target_map_t::iterator tgt( m_targets.begin() );
			tgt != m_targets.end();
			++tgt ) {
		tgt->second.collect();
	}
}


// for resolving
void TargetManager::resolve(void)
{
	bool all_resolved = true;
	do {
		try {
			for( name_target_map_t::iterator tgt_pair( m_targets.begin() );
					tgt_pair != m_targets.end();
					++tgt_pair ) {
				Target& tgt(tgt_pair->second);
				if( ! tgt.isResolved() ) {
					std::list<Target*> circle_list( tgt.resolve() );
					if( circle_list.empty() ) {
						all_resolved = false;
					}
				}
			}
		}
		catch( const CircularPreRequresError& circular ) {
			std::cout << circular.what() << std::endl;
			std::list<Target*> list( circular.getList() ); // コピー
			for( std::list<Target*>::iterator ptgt( list.begin() );
					ptgt != list.end();
					++ptgt ) {
				driveAway(*ptgt);
			}
		}
	} while( ! isAllResolved() );
}

bool TargetManager::isAllResolved(void) const
{
	for( name_target_map_t::const_iterator tgt_pair( m_targets.begin() );
			tgt_pair != m_targets.end();
			++tgt_pair ) {
		if( ! tgt_pair->second.isResolved() ) {
			return false;
		}
	}
	return true;
}

void TargetManager::driveAway(Target* target)
{
	for( name_target_map_t::iterator tgt_pair( m_targets.begin() );
			tgt_pair != m_targets.end();
			++tgt_pair ) {
		if( &(tgt_pair->second) == target ) {
			std::cout << "Driving away " << tgt_pair->second.getName() << std::endl;
			m_targets.erase(tgt_pair);	// XXX: 
		}
	}
}


// for debugging
void TargetManager::show(void) const
{
	for( name_target_map_t::const_iterator tgt( m_targets.begin() );
			tgt != m_targets.end();
			++tgt ) {
		tgt->second.show();
	}
}


// private:
Target* TargetManager::find(const std::string& name)
{
	name_target_map_t::iterator found( m_targets.find(name) );
	if( found == m_targets.end() ) {
		return NULL;
	} else {
		return &(found->second);
	}
}


}  // namespace Applyer

}  // namespace RUNES
