#include "controler.h"
#include "color.h"
#include "constant.h"
#include "display.h"
#include "sdpci.h"
#include "sdusb.h"
#include "modules.h"
#include "linearizer.h"
#include "shadowmapper.h"
#include <cstdio>
#include <sstream>
#include <fstream>
#include <sys/utsname.h>
#include <cmath>

#define VIVER_CL_IMPL(func)	VIVER_CL_RET Controler::func VIVER_CL_ARG

namespace VIVER {


VIVER_CL_IMPL( initial )
{
	// procをマウント
	vd_proc = new VirtualDevice("proc", "proc", 0, "");
	vd_proc->mount(MP::PROC);

	// sysをマウント
	vd_sys = new VirtualDevice("sys", "sysfs", 0, "");
	vd_sys->mount(MP::SYS);

	// カーネルバージョンを得る
	{
		struct utsname uts;
		if( ::uname(&uts) < 0 ) {
			throw ExecuteFailedError(CANNOT + "detect kernel version" + errorDescribe());
		}
		kernel_release = uts.release;
	}

	// ブートパラメータをパース
	parameters = new Parameters( Component::Kernel::PATH_BOOT_PARAMETER_FILE );
	parameters->parse();

	// displayオブジェクトを初期化
	host::display.configure(*parameters);
}

VIVER_CL_IMPL( mountArk )
{
	// Arkをマウント
	vd_ark = new VirtualDevice("tmpfs", "tmpfs", 0, "size=2m");
	vd_ark->mount(MP::ARK);

	// /bin
	FileSystem::makeDirForce(MP::ARK + ARK::BIN, 0700);

	// /dev
	FileSystem::makeDirForce(MP::ARK + ARK::DEV, 0755);
	FileSystem::makeDirForce(MP::ARK + ARK::DEV + "mapper", 0755);
	FileSystem::makeDirForce(MP::ARK + ARK::DEV + "cow", 0755);
	FileSystem::makeCharacterNodeForce(  1,   3, MP::ARK + ARK::DEV + "null",	0666);
	FileSystem::makeCharacterNodeForce(  5,   1, MP::ARK + ARK::DEV + "console",	0600);
	FileSystem::makeCharacterNodeForce( 10,  63, MP::ARK + ARK::DEV + "mapper/control", 0660);
	FileSystem::makeBlockNodeForce    (241, 255, MP::ARK + ARK::DEV + "cow/ctl",	0640);

	// マウントポイント
	FileSystem::makeDirForce(MP::ARK + ARK::MP_BOOTDISK,	0750);
	FileSystem::makeDirForce(MP::ARK + ARK::MP_BOOTDISK,	0750);
	FileSystem::makeDirForce(MP::ARK + ARK::MP_COMPRESSED,	0750);
	FileSystem::makeDirForce(MP::ARK + ARK::MP_SCREEN,	0750);
	FileSystem::makeDirForce(MP::ARK + ARK::MP_SCREEN_CONTAINER,	0750);
	FileSystem::makeDirForce(MP::ARK + ARK::MP_TORAM,	0750);
}


VIVER_CL_IMPL( startLogd )
{
	// syslogdを開始
	logd = new LogDaemon( (MP::ARK + "viver.log").c_str() );
	logd->start();
}

VIVER_CL_IMPL( newDevmanager )
{
	devmanager = new DeviceManager();
}

VIVER_CL_IMPL( mountModules )
{
	dn_modules = devmanager->loopSetup(
			Component::Resource::MODULE_DISK,
			Component::Resource::FSTYPE_MODULE_DISK,
			MS_RDONLY,
			""
			);
	dn_modules->mount(MP::MODULE_DISK);
}

VIVER_CL_IMPL( detectHardware )
{
	if( parameters->getValueString("nodetect", NULL) != NULL ) return;

	using namespace SimpleDetect;
	Modules::LoadModuleSD loader;

	sdPCI sdpci(Component::Resource::PCI_IDS, (MP::MODULE_DISK + kernel_release + "modules.pcimap").str());

	if( parameters->getValueString("pccard", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::BRIDGE_PCCARD, sdPCI::FORCE);
	else if( parameters->getValueString("nopccard", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::BRIDGE_PCCARD, sdPCI::NEVER);

	if( parameters->getValueString("usb", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::BUS_USB, sdPCI::FORCE);
	else if( parameters->getValueString("nousb", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::BUS_USB, sdPCI::NEVER);

	if( parameters->getValueString("firewire", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::BUS_FIREWIRE, sdPCI::FORCE);
	else if( parameters->getValueString("nofirewire", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::BUS_FIREWIRE, sdPCI::NEVER);

	if( parameters->getValueString("scsi", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::STORAGE_SCSI, sdPCI::FORCE);
	else if( parameters->getValueString("noscsi", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::STORAGE_SCSI, sdPCI::NEVER);

	if( parameters->getValueString("ide", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::STORAGE_IDE, sdPCI::FORCE);
	else if( parameters->getValueString("noide", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::STORAGE_IDE, sdPCI::NEVER);

	if( parameters->getValueString("sata", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::STORAGE_SATA, sdPCI::FORCE);
	else if( parameters->getValueString("nosata", NULL) != NULL )
		sdpci.setKnownClassBasedLoadConfig(sdPCI::STORAGE_SATA, sdPCI::FORCE);

	sdpci.initialize();
	sdpci.detect(loader);
	sdpci.showDetectedDevices( host::display.verbose() );


	if( parameters->getValueString("nousb", NULL) != NULL ) return;
	sdUSB sdusb(Component::Resource::USB_IDS, (MP::MODULE_DISK + kernel_release + "modules.usbmap").str());
	sdusb.initialize();
	sdusb.detect(loader);
	sdusb.showDetectedDevices( host::display.verbose() );
}

VIVER_CL_IMPL( loadSystemModules )
{
	// ファイルシステムドライバ
	static const char* fsmodules[] = {
		"ntfs",
		"vfat",
		"ext3",
		"reiserfs",
		"reiser4",
		"xfs",
		"jfs",
		"ufs",
		"udf",
		"hfs",
		"hfsplus",
		NULL,
	};
	Modules::loadModule(fsmodules);

	// ブートパラメータで指定されたモジュール
	try {
		std::vector<std::string> modules;
		for( CSV mod( parameters->getValueString("modprobe","") );
				!mod.end();
				++mod ) {
			modules.push_back(*mod);
		}
		Modules::loadModule(modules);
	} catch(const std::bad_alloc& ex) {
		std::cerr << ex.what() << std::endl;
	}
}

bool Controler::isBootFormDisk(void)
{
	if( parameters->getValueString("bootstrap", NULL ) == NULL )
		return true;
	else
		return false;
}

VIVER_CL_IMPL( mountBootDisk )
{
	// ディスクから起動
	boot_from_disk = true;

	// CompressedImageのBootDisk上のパスを決定
	compressed_on_bootdisk = parameters->getValueString(
			"compressed",
			compressed_on_bootdisk.c_str()
			);

	// BootDiskを探してマウント
	mb_bootdisk = devmanager->searchDeviceByFileRO( compressed_on_bootdisk );
	mb_bootdisk->mount( MP::ARK + ARK::MP_BOOTDISK );
}

VIVER_CL_IMPL( wakeNetwork )
{
	// ネットワークを起動
	Modules::loadModule( Network::KERNEL_MODULE );
	network = new Network(*parameters);
	network->wakeup();
}

VIVER_CL_IMPL( losetupCompressedOnDisk )
{
	// CompressedImageをlosetup
	dn_compressed = devmanager->loopSetup(
			mb_bootdisk->getMountPoint() + compressed_on_bootdisk,
			Component::Image::FSTYPE_COMPRESSED,
			MS_RDONLY,
			""
			);
}

VIVER_CL_IMPL( connectSBD )
{
	// ネットワークから起動
	boot_from_disk = false;

	// CompressedImageを取得（SharedBlockDeviceを接続）
	Modules::loadModule( SharedBlockDevice::KERNEL_MODULE );
	sbd = new SharedBlockDevice(*parameters, *devmanager);
	sbd->configure();

	dn_compressed = sbd->connect(
			Component::Image::FSTYPE_COMPRESSED,
			MS_RDONLY,
			""
			);
}

VIVER_CL_IMPL( mountToRAMFS )
{
	// 物理RAMサイズをKBで測定
	unsigned int ram_total = 0;
	{
		std::ifstream proc_meminfo(Component::Kernel::PATH_PROC_MEMINFO);
		if( !proc_meminfo.is_open() ) {
			throw NoSuchFileError(CANNOT + "open " + Component::Kernel::PATH_PROC_MEMINFO);
		}
		char buf[256];
		std::string size;
		std::string kb;
		while( !proc_meminfo.eof() ) {
			std::string name;
			proc_meminfo.getline(buf, sizeof(buf));
			std::istringstream line_stream(buf);
			line_stream >> name >> size >> kb;
			if( name == "MemTotal:" ) {
				std::istringstream size_stream(size);
				size_stream >> ram_total;
				host::display.debug() << "MemTotal: " << ram_total << " KB" << std::endl;
				break;
			}
		}
	}

	// CompressedImageのサイズをKBで測定
	uint64_t compressed_size = 0;
	if( boot_from_disk ) {
		const Path if_file_path( mb_bootdisk->getMountPoint() + compressed_on_bootdisk );
		compressed_size = FileSystem::getFileSize( if_file_path ) / 1024;
	} else {
		sleep(3);  // XXX XXX: NBD接続が完了するまで待つ
		compressed_size = dn_compressed->getSize() / 1024;
	}
	if( compressed_size == 0 ) {  // XXX: throwする？
		host::display.warn() << "Can't calc the block size of CompressedImage" << errorDescribe() << std::endl;
	}
	host::display.debug() << "CompressedImage size: " << compressed_size << " KB" << std::endl;

	if( ram_total < compressed_size ) {
		// メモリが足りないのでtoramできない
		throw InsufficientMemoryError("RAM size is insufficent to copy CompressedImage to RAM");
	}

	// vd_toramを作成
	{
		std::ostringstream opt_size;
		opt_size << "size=" << (compressed_size + 1024) << "k";	// XXX: 余分の分量は割合での計算が必要
		vd_toram = new VirtualDevice("tmpfs", "tmpfs", 0, opt_size.str() );
	}

	// FIXME: ブートパラメータで指定できる必要があるか？
	compressed_on_toram = DefaultInfo::Image::COMPRESSED_ON_BOOTDISK;

	vd_toram->mount(MP::ARK + ARK::MP_TORAM);
}

VIVER_CL_IMPL( toRAM )
{
	host::display.notice() << "Copying CompressedImage to RAM" << std::endl;

	Process ps_dd_toram(Component::Program::DD);

	Path if_file_path;
	if( boot_from_disk )
		if_file_path = mb_bootdisk->getMountPoint() + compressed_on_bootdisk;
	else
		if_file_path = dn_compressed->getNodePath();

	Path of_file_path( vd_toram->getMountPoint()    + compressed_on_toram );

	{
		// FIXME: bsとcoutはどうする？
		std::string opt_if( std::string("if=") + if_file_path.str() );
		std::string opt_of( std::string("of=") + of_file_path.str() );
		ps_dd_toram % opt_if.c_str() % opt_of.c_str();
	}

	MultiPipe mp_dd_toram( ps_dd_toram.exec(true,true), host::display.debug(), host::display.debug() );
	mp_dd_toram.display();
	if( ps_dd_toram.wait() != 0 ) {
		// XXX: 別のVIVER_MakeRuntimeErrorが必要
		throw NoSuchFileError( CANNOT + "copy " + compressed_on_bootdisk + " to RAM" );
	}

	if( boot_from_disk ) {
		// BootDiskをアンマウント
		mb_bootdisk->umount();
		//delete mb_bootdisk;
		//mb_bootdisk = NULL;
	} else {
		// SBDを接続解除
		devmanager->returnNode(dn_compressed);
		sbd->disconnect();
		//delete sbd;
		//sbd = NULL;
	}
}

VIVER_CL_IMPL( losetupCompressedOnRAM )
{
	// CompressedImageをlosetup
	dn_compressed = devmanager->loopSetup(
			vd_toram->getMountPoint() + compressed_on_toram,
			Component::Image::FSTYPE_COMPRESSED,
			MS_RDONLY,
			""
			);
}


VIVER_CL_IMPL( mountCompressed )
{
	// CompressedImageをマウント
	dn_compressed->mount( MP::ARK + ARK::MP_COMPRESSED );
}

bool Controler::isRAMScreen(void)
{
	if( parameters->getValueString("shadow", NULL) == NULL )
		return true;
	else
		return false;
}

VIVER_CL_IMPL( mountDirectScreen )
{
	std::string keyfile;
	std::string dev_csv;		// XXX: 未実装
	std::string options;		// XXX: 未実装
	std::string fstype_csv;		// XXX: 未実装

	const char* pa = parameters->getValueString("shadow",NULL);
	std::string line(  ( (pa == NULL || pa[0] == '\0') ? DefaultInfo::Image::SHADOW_ON_SCREEN : pa )  );
	std::string current(line);

	std::string::size_type pos_sem;
	if( (pos_sem=current.find(';')) == std::string::npos ) {
		screen_keyfile = current;
		goto search_direct_screen;
	}

	dev_csv = current.substr(0, pos_sem);
	current = current.substr(pos_sem+1);

	if( (pos_sem=current.find(';')) == std::string::npos ) {
		screen_keyfile = current;
		goto search_direct_screen;
	}

	fstype_csv = current.substr(0, pos_sem);
	current = current.substr(pos_sem+1);

	if( (pos_sem=current.find(';')) == std::string::npos ) {
		screen_keyfile = current;
		goto search_direct_screen;
	}

	options = current.substr(0, pos_sem);
	current = current.substr(pos_sem+1);

	screen_keyfile = current;
	goto search_direct_screen;

search_direct_screen:
	DeviceManager::keyfile_type_t screen_keyfile_type;
	mb_container = devmanager->searchDeviceByFileRW(screen_keyfile, &screen_keyfile_type);

	if( screen_keyfile_type == DeviceManager::FILE_KEY ) {
		// ファイルなので、ループバックShadowにする
		mb_container->mount( MP::ARK + ARK::MP_SCREEN_CONTAINER );
		// shadow_on_screenはデフォルトを使う（container１つにはShadowImageは１つしか入れられない）
	} else {
		// ディレクトリなので、そのままDirectShadow
		mb_screen = mb_container;
		mb_container = NULL;
	}
}

VIVER_CL_IMPL( losetupLoopScreen )
{
	if( mb_container == NULL ) return;	// DirectShadowのようなので何もしない

	// screen_keyfileはcontainerなので、これをループバックマウントする
	std::string fstype_container(
			parameters->getValueString("containerfs", DefaultInfo::Image::FSTYPE_SCREEN_CONTAINER)
			);
	devmanager->loopSetup(mb_container->getMountPoint() + screen_keyfile,
			fstype_container,
			0,
			"" );
}

VIVER_CL_IMPL( configureRAMScreen )
{
	Parameters::with_unit_t sz( parameters->getValueByte("ramsize", Parameters::with_unit_t(0, Parameters::INVALID_UNIT)) );

	unsigned long number;
	char unit_chr;
	switch( sz.second ) {
	case Parameters::KB_UNIT:
		number = sz.first;
		unit_chr = 'k';
		break;
	case Parameters::MB_UNIT:
		number = sz.first;
		unit_chr = 'm';
		break;
	case Parameters::GB_UNIT:
		number = sz.first * 1024;
		unit_chr = 'm';
		break;
	case Parameters::TB_UNIT:
		number = sz.first * 1024 * 1024;
		unit_chr = 'm';
		break;
	case Parameters::INVALID_UNIT:
	default:
		// デフォルトのサイズを決定
		{
			FILE* f_pm;
			f_pm = ::fopen("/proc/meminfo", "r");
			if( f_pm == NULL ) {
				number = 64;
				unit_chr = 'm';
			} else {
				char label_trash[128];
				int mem_total;
				char unit_trash[32];
				fscanf(f_pm, "%s %d %s", label_trash, &mem_total, unit_trash);
				// unitは常にkB (linux/fs/proc/proc_misc.c :153)

				//double smart_size = ::log( (double)mem_total / 1024 / 180 + 1) * 115 * 1024;
				//double smart_size = ::log( (double)mem_total / 1024 / 200 + 1) * 150 * 1024;
				//double smart_size = ::log( (double)mem_total / 1024 / 180 + 1) * 135 * 1024;
				double smart_size = ::log( (double)mem_total / 1024 / 300 + 1) * 250 * 1024;
				double min_size = 1024;
				double max_size = (double)mem_total * 0.9;
				if( smart_size < min_size ) {
					number = static_cast<unsigned long>(min_size);
				} else if( smart_size > max_size ) {
					number = static_cast<unsigned long>(max_size);
				} else {
					number = static_cast<unsigned long>(smart_size);
				}
				unit_chr = 'k';
			}
			::fclose(f_pm);
		}
		break;
	}

	std::ostringstream size_opt;
	size_opt << "size=" << number << unit_chr;
	mb_screen = new VirtualDevice("tmpfs", "tmpfs", 0, size_opt.str());
	// shadow_on_screenはデフォルトを使う
}

VIVER_CL_IMPL( mountScreen )
{
	mb_screen->mount( MP::ARK + ARK::MP_SCREEN );
}

bool Controler::isToRAM(void)
{
	if( parameters->getValueString("toram", NULL ) != NULL )
		return true;
	else
		return false;
}

VIVER_CL_IMPL( losetupFixed )
{
	fixed_on_compressed = parameters->getValueString(
			"compressed",
			fixed_on_compressed.c_str()
			);
	dn_fixed = devmanager->loopSetup(
			dn_compressed->getMountPoint() + fixed_on_compressed,
			Component::Image::FSTYPE_FIXED,
			0,
			""
			);
}

VIVER_CL_IMPL( makeSparse )
{
	// SparseImageのサイズを決定
	Parameters::with_unit_t param(
			parameters->getValueByte("sparse", Parameters::with_unit_t(0, Parameters::MB_UNIT))
			);

	unsigned long sparse_size_kb = 0;

	if( param.first == 0 ) {		// 指定なし
		sparse_size_kb = decideDefaultSparseSize();
	} else {
		switch( param.second ) {
		case Parameters::KB_UNIT:
			sparse_size_kb = param.first;
			break;
		case Parameters::MB_UNIT:
			sparse_size_kb = param.first * 1024;
			break;
		case Parameters::GB_UNIT:
			sparse_size_kb = param.first * 1024 * 1024;
			break;
		case Parameters::TB_UNIT:
			sparse_size_kb = param.first * 1024 * 1024 * 1024;
			break;
		case Parameters::INVALID_UNIT:
			sparse_size_kb = decideDefaultSparseSize();
			std::string msg( std::string("Invalid sparse size unit: ") + parameters->getValueString("sparse", "") );
			throw UnreadableParameterError( msg );
		}
	}


	Path sparse_path( mb_screen->getMountPoint() + sparse_on_screen );
	std::string opt_of( std::string("of=") + sparse_path.str() );

	std::ostringstream opt_seek_stream;
	opt_seek_stream << "seek=" << sparse_size_kb;
	std::string opt_seek( opt_seek_stream.str() );

	Process ps_dd_sparse(Component::Program::DD);
	ps_dd_sparse % "if=/dev/null" % opt_of.c_str() % "bs=1024" % "count=1" % opt_seek.c_str();
	MultiPipe mp_dd_sparse( ps_dd_sparse.exec(true,true), host::display.debug(), host::display.debug() );
	mp_dd_sparse.display();
	ps_dd_sparse.wait();
}
unsigned long Controler::decideDefaultSparseSize(void) const
{
	// ShadowScreenのサイズからSparseImageのサイズを決定し、KB単位で返す
	// XXX: 未実装
	return 1024 * 1024;	// 1GB
}

VIVER_CL_IMPL( connectSparse )
{
	Path sparse_path( mb_screen->getMountPoint() + sparse_on_screen );
	dn_sparse = devmanager->loopSetup(sparse_path, "sparse", 0, "");

	Modules::loadModule(Linearizer::KERNEL_MODULE);

	Linearizer linear(*devmanager);
	linear.addDevice(dn_fixed);
	linear.addDevice(dn_sparse);

	// XXX: マウントオプションは指定許可？
	dn_sparsed_fixed = linear.linearize(dn_fixed->getFSType(), 0, "");
}

VIVER_CL_IMPL( formShadow )
{
	host::display.notice() << "Forming Copy-on-Write root disk" << std::endl;

	Modules::loadModule( ShadowMapper::KERNEL_MODULE );

	ShadowMapper shadow_mapper(
			*devmanager,
			dn_sparsed_fixed,
			mb_screen->getMountPoint() + shadow_on_screen,
			Component::Image::FSTYPE_FIXED,
			0,
			""
			);

	dn_formed = shadow_mapper.form();
}

VIVER_CL_IMPL( mountFormedImage )
{
	dn_formed->mount( MP::FORMED );
}

VIVER_CL_IMPL( expandFormedImage )
{
	Process ps_expand(Component::Program::XFS_GROWFS);
	ps_expand % "-t" % "/proc/mounts" % MP::FORMED.c_str();
	MultiPipe mp_expand( ps_expand.exec(true,true), host::display.debug(), host::display.warn() );
	mp_expand.display();
	if( ps_expand.wait() != 0 ) {
		throw ExecuteFailedError("Can't expand FormedImage");
	}
}

VIVER_CL_IMPL( umountModuleDisk )
{
	dn_modules->umount();
	devmanager->loopRemove(dn_modules);
}

VIVER_CL_IMPL( switchRoot )
{
	devmanager->umountTrialPoints();

	Path to_ark(MP::FORMED + FORMED::MP_ARK);
	Path to_sys(MP::FORMED + FORMED::MP_SYS);
	Path to_proc(MP::FORMED + FORMED::MP_PROC);
	FileSystem::makeDirForce(to_ark.str(),  0755);
	FileSystem::makeDirForce(to_sys.str(),  0755);
	FileSystem::makeDirForce(to_proc.str(), 0755);
	vd_ark->move( to_ark );
	vd_sys->move( to_sys );
	vd_proc->move( to_proc );	// procは必ず最後にmove

	Path put_old( parameters->getValueString("rdput", DefaultInfo::Formed::PIVOT_OLD_DIR) );

	FileSystem::cd( MP::FORMED );
	Mountable::pivot_root( static_cast<Mountable*>(dn_formed), put_old );
	FileSystem::cd("/");

	vd_sys->pivotNotice(FORMED::MP_SYS);
	vd_proc->pivotNotice(FORMED::MP_PROC);
}

VIVER_CL_IMPL( editSysRoot )
{
	// エラーが起きても続行
	Path put_old( parameters->getValueString("rdput", DefaultInfo::Formed::PIVOT_OLD_DIR) );

	// fstab, mtab
	FileSystem::move("/etc/fstab", "/etc/fstab.orig");
	FileSystem::move("/etc/mtab", "/etc/mtab.orig");

	// XXX: DeviceManager::makeMtab()を作る？
	FileSystem::copyFile("/proc/mounts", "/etc/mtab");

	// XXX: DeviceManager::makeFstab()を作る？
	std::ofstream fstab("/etc/fstab");
	if( fstab.is_open() ) {
		fstab << "/VIVER/dev/cow0 / " << Component::Image::FSTYPE_FIXED << " defaults 0 0" << std::endl;
		fstab << "proc /proc proc defaults 0 0" << std::endl;
		fstab << "sysfs /sys sysfs defaults 0 0" << std::endl;
	}


	// デバイスノードを作成
	FileSystem::makeDir("/dev", 0755);
	FileSystem::makeCharacterNode(  1,   3, "/dev/null",    0666);
	FileSystem::makeCharacterNode(  1,   5, "/dev/zero",    0666);
	FileSystem::makeCharacterNode(  1,   8, "/dev/random",  0666);
	FileSystem::makeCharacterNode(  1,   9, "/dev/urandom", 0444);
	FileSystem::makeCharacterNode(  5,   1, "/dev/console", 0600);
	FileSystem::makeDir("/dev/mapper", 0755);
	FileSystem::makeCharacterNode( 10,  63, "/dev/mapper/control", 0660);
	FileSystem::makeDir("/dev/cow", 0755);
	FileSystem::makeBlockNode(241, 255, "/dev/cow/ctl", 0640);
	devmanager->makeNodesOn( Path("/dev") );

	// /etc/resolv.confをコピー
	try {
		FileSystem::remove("/etc/resolv.conf");
		FileSystem::copyFile( (put_old+"/etc/resolv.conf").str(), "/etc/resolv.conf");
	} catch (...) {
		// エラー無視
	}
}

VIVER_CL_IMPL( outputParameters )
{
	// ブートパラメータを書き出す
	std::string output_dir = parameters->getValueString(
			"paramout",
			DefaultInfo::Formed::PARAMETER_OUTPUT_DIR
			);
	FileSystem::prepareDir(output_dir);

	host::display.notice() << "Writing boot parameters to " << output_dir << std::endl;
	parameters->writeToFile(output_dir);
}

VIVER_CL_IMPL( stopLogd )
{
	logd->stop();
	delete logd;
	logd = NULL;
}

VIVER_CL_IMPL( rc )
{
	std::cout << Color::NORMAL << std::flush;
	const char* rc_path = parameters->getValueString(
			"viverrc",
			DefaultInfo::Formed::RC_PATH
			);
	Process ps_rc(rc_path);

	/*
	{ // XXX: viverrc 0.1 互換モード
		std::string preset;

		preset = "all";
		if( boot_from_disk ) preset += ",disk";
		else preset += ",netboot";
		if( ram_shadow ) preset += ",ramcow";
		else preset += ",phycow";

		ps_rc % "/etc/VIVER" % preset.c_str();

		for( CSV optional( parameters->getValueString("role", "") );
				!optional.end();
				++optional ) {
			ps_rc % *optional;
		}
	}
	*/

	{ // RUNES 0.1
		ps_rc % "/etc/RUNES/runes" % "/etc/RUNES/roles" % "/etc/RUNES/interface";
		ps_rc % "_all";
		if( boot_from_disk ) ps_rc % "_disk";
		else ps_rc % "_netboot";
		//if( ram_shadow ) ps_rc % "_ramcow";
		//else ps_rc % "_phycow";
		for( CSV optional( parameters->getValueString("role", "") );
				!optional.end();
				++optional ) {
			ps_rc % *optional;
		}
	}

	ps_rc.exec(false,false);
	ps_rc.wait();
}

VIVER_CL_IMPL( post )
{
	vd_sys->umount();
	vd_proc->umount();	// procは最後
}


}  // namespace VIVER
