#include "devicemanager.h"
#include "process.h"
#include "display.h"
#include "constant.h"
#include <fstream>
#include <sstream>
#include <dirent.h>
#include <string.h>

namespace VIVER {


// コンストラクタ
DeviceManager::DeviceManager() :
	m_dev_path(MP::ARK + ARK::DEV)
{}


// デストラクタ
DeviceManager::~DeviceManager()
{}


// public:
DeviceNode* DeviceManager::loopSetup(const Path& path,
		const std::string& fstype, unsigned long mount_flags, const std::string& data_string)
{
	// 未使用のループバックデバイスを検索
	FileSystem::minor_t loop_num;
	for( loop_num = Component::Kernel::LOOP_MIN; loop_num <= Component::Kernel::LOOP_MAX; ++loop_num ) {
		std::pair<used_loop_t::iterator, bool> ins_pair( m_used_loop.insert(loop_num) );
		if( ins_pair.second == true ) break;
	}

	std::ostringstream loop_name;
	loop_name << "loop" << loop_num;
	DeviceNode* dn_loop = requestNode(Component::Kernel::LOOP_MAJOR, loop_num,
			loop_name.str(),
			fstype, mount_flags, data_string);

	if( dn_loop->makeNode() < 0 ) {
		throw NoSuchFileError( CANNOT + "make device node " + dn_loop->getNodePath().str() + errorDescribe() );
	}

	// losetup
	Process ps_losetup(Component::Program::LOSETUP);
	ps_losetup % dn_loop->getNodePath().c_str() % path.c_str();
	MultiPipe mp( ps_losetup.exec(true,true), host::display.debug(), host::display.warn() );
	mp.display();
	if( ps_losetup.wait() != 0 ) {
		throw ExecuteFailedError( CANNOT + "losetup " + path.str() );
	}

	return dn_loop;
}
void DeviceManager::loopRemove(DeviceNode* dn_loop)
{
	Process ps_losetup(Component::Program::LOSETUP);
	ps_losetup % "-d" % dn_loop->getNodePath().c_str();
	MultiPipe mp( ps_losetup.exec(true,true), host::display.debug(), host::display.warn() );
	mp.display();
	if( ps_losetup.wait() != 0 ) {
		// エラー続行
		host::display.warn() << "Failed to losetup -d " << dn_loop->getNodePath().c_str() << std::endl;
	}
}

DeviceNode* DeviceManager::requestNode(FileSystem::major_t major, FileSystem::minor_t minor,
		const std::string& node_name,
		const std::string& fstype, unsigned long mount_flags, const std::string& data_string)
{
	// デバイスノードファイルは作成しない
	DeviceNode dev(major, minor,
			m_dev_path, node_name,
			fstype, mount_flags, data_string);
	return registerNode(dev);
}
DeviceNode* DeviceManager::requestNode(FileSystem::major_t major, FileSystem::minor_t minor,
		const std::string& node_name,
		const std::string& fstype, unsigned long mount_flags, const std::string& data_string,
		const Path& mount_point)
{
	// デバイスノードファイルは作成しない
	DeviceNode dev(major, minor,
			m_dev_path, node_name,
			fstype, mount_flags, data_string,
			mount_point);
	return registerNode(dev);
}

void DeviceManager::returnNode(DeviceNode* node)
{
	node_list_t::iterator nodite( m_node_list.begin() );
	while( nodite != m_node_list.end() ) {
		if( &(*nodite) == node )
			nodite = m_node_list.erase(nodite);
		else
			++nodite;
	}
}

VirtualDevice* DeviceManager::requestVirtual(const std::string& target_name,
		const std::string& fstype, unsigned long mount_flags, const std::string& data_string)
{
	VirtualDevice vdev(target_name, fstype, mount_flags, data_string);
	return registerVirtual(vdev);
}

void DeviceManager::makeNodesOn(const Path& dev_path)
{
	for( node_list_t::iterator node( m_node_list.begin() );
			node != m_node_list.end();
			++node ) {
		// エラーが起きても続行
		node->makeNodeOn(dev_path);
	}
	// candidateのデバイスノードは要らない
	//for( candidate_list_t::const_iterator cand( m_candidate_list.begin() );
	//		cand != m_candidate_list.end();
	//		++cand ) {
	//}
}

void DeviceManager::umountTrialPoints(void)
{
	for( node_list_t::iterator node( m_node_list.begin() );
			node != m_node_list.end();
			++node ) {
		if( node->getMountPoint().isChildOf(Component::Resource::MOUNT_TRIAL_POINT) ) {
			// trial/以下にマウントされているので、アンマウント
			// 例外が発生する可能性がある
			try {
				node->umount();
			}
			catch(const std::runtime_error& ex) {
				host::display.warn() << "Can't umount " << node->getNodeName()
					<< " (mounted on " << node->getMountPoint().str() << ")"
					<< errorDescribe() << std::endl;
			}
		}
	}
	for( virtual_list_t::iterator node( m_virtual_list.begin() );
			node != m_virtual_list.end();
			++node ) {
		if( node->getMountPoint().isChildOf(Component::Resource::MOUNT_TRIAL_POINT) ) {
			// trial/以下にマウントされているので、アンマウント
			// 例外が発生する可能性がある
			try {
				node->umount();
			}
			catch(const std::runtime_error& ex) {
				host::display.warn() << "Can't umount " << node->getTargetName()
					<< " (mounted on " << node->getMountPoint().str() << ")"
					<< errorDescribe() << std::endl;
			}
		}
	}
}

// public:
Mountable* DeviceManager::searchDeviceByFileRO(const std::string& key_file)
{
	// 例外は呼出元で捕捉
	// 例外が発生しなければ、found != NULL
	Mountable* found = searchDeviceByFile(key_file, NULL);

	if( found->getMountPoint().isChildOf(Component::Resource::MOUNT_TRIAL_POINT) ) {
		// trial/以下にマウントされているので、move可能。返す。
		return found;
	} else {
		// 既にどこか別の場所にmoveされている。move不可。bindしたものを返す。
		// XXX: 第一引数はパスなのだが、Virtualだからstringになっている
		return requestVirtual( found->getMountPoint().str(),
				"bind", MS_BIND, found->getDataString() );
	}
}

Mountable* DeviceManager::searchDeviceByFileRW(const std::string& key_file,
		keyfile_type_t* return_keyfile_type )
{
	// 例外は呼出元で捕捉
	// 例外が発生しなければ、found != NULL
	Mountable* found = searchDeviceByFile(key_file, return_keyfile_type);
	if( found->getMountFlags() & MS_RDONLY != 0 ) {
		// 書き込み可能でリマウント
		// リマウントできなければ例外が発生する
		found->remount( found->getMountFlags() ^ MS_RDONLY, found->getDataString() );
	}

	if( found->getMountPoint().isChildOf(Component::Resource::MOUNT_TRIAL_POINT) ) {
		// trial/以下にマウントされているので、move可能。返す。
		return found;
	} else {
		// 既にどこか別の場所にmoveされている。move不可。bindしたものを返す。
		// XXX: 第一引数はパスなのだが、Virtualだからstringになっている
		return requestVirtual( found->getMountPoint().str(),
				"bind", MS_BIND, found->getDataString() );
	}
}


// private:
DeviceNode* DeviceManager::searchDeviceByFile(const std::string& key_file,
		keyfile_type_t* return_keyfile_type )
{
	host::display.debug() << "Searching " << key_file << std::endl;

	if( m_candidate_list.empty() ) lookupDeviceNodes();

	// まず m_certain から調べる
	if( !m_certain.empty() ) {
		for( certain_class_t::iterator cert( m_certain.begin() );
				cert != m_certain.end();
				++cert ) {
			if( ! (*cert)->isMounted() ) continue;		// XXX: どこかにマウントする？
			// key_pathを含むか確認
			switch( FileSystem::checkFileExists( (*cert)->getMountPoint() + key_file ) ) {
			case FileSystem::REGULAR_FILE:	// 通常ファイルとして含む
				*return_keyfile_type = FILE_KEY;
				return *cert;
			case FileSystem::DIRECTORY:	// ディレクトリとして含む
				*return_keyfile_type = DIR_KEY;
				return *cert;
			case FileSystem::OTHER:		// 含むがFIFOなどの特殊ファイル
			case FileSystem::NOT_EXISTS:	// 含まない
				continue;
			}
		}
	}

	DeviceNode* return_node = NULL;
	// m_unclear_candidatesを調べる
	DeviceNodeCandidate::test_stat_t stat;
	std::string found_fstype;
	unsigned long found_mount_flags;
	std::string found_data_string;
	while( return_node == NULL && !m_unclear_candidates.empty() ) {	// 見つかっていない上に、候補がまだある

		for( candidate_class_t::iterator p_node( m_unclear_candidates.begin() );
				p_node != m_unclear_candidates.end();
				++p_node ) {
			(*p_node)->startTestMount();
		}

		{
			candidate_class_t::iterator p_node( m_unclear_candidates.begin() );
			while( p_node != m_unclear_candidates.end() ) {

				stat = (*p_node)->waitTestMount(&found_fstype, &found_mount_flags, &found_data_string);

				if( stat == DeviceNodeCandidate::MOUNT_FAILED ) {
					// マウント失敗
					++p_node;

				} else if( stat == DeviceNodeCandidate::MOUNT_SUCCESS ) {
					// マウント成功
					host::display.verbose() << "Proved device: "
						<< (*p_node)->getNodeName()
						<< " (" << found_fstype << ")" << std::endl;

					// 判明したので DeviceNodeCandidate ではなくなる
					// m_candidate_list, m_unclear_candidatesから削除し、m_certainに追加
					Path found_mount_point( (*p_node)->getTestPoint() );
					DeviceNode* found_node = requestNode(
							(*p_node)->getMajor(), (*p_node)->getMinor(),
							(*p_node)->getNodeName(),
							found_fstype, found_mount_flags, found_data_string,
							found_mount_point
							);
					m_certain.push_back(found_node);
					for( candidate_list_t::iterator cand( m_candidate_list.begin() );
							cand != m_candidate_list.end();
							++cand ) {
						if( &(*cand) == *p_node )
							m_candidate_list.erase(cand);
						break;
					}
						// ここで *p_node にアクセスすると Segmentation Fault
					p_node = m_unclear_candidates.erase(p_node);

					switch( FileSystem::checkFileExists(found_mount_point + key_file) ) {
					case FileSystem::REGULAR_FILE:		// 通常ファイルとして含む
						if( return_node == NULL) {	// 最初に見つかったデバイスを返す
							if( return_keyfile_type )
								*return_keyfile_type = FILE_KEY;
							return_node = found_node;
						}
						break;
					case FileSystem::DIRECTORY:		// ディレクトリとして含む
						if( return_node == NULL) {	// 最初に見つかったデバイスを返す
							if( return_keyfile_type )
								*return_keyfile_type = DIR_KEY;
							return_node = found_node;
						}
						break;
					case FileSystem::OTHER:		// 含むがFIFOなどの特殊ファイル
						break;
					case FileSystem::NOT_EXISTS:	// 含まない
						break;
					}

				} else {
					// すべてテストしたか、エラー
					// m_unusable_candidatesに入れ、
					// m_unclear_candidatesから削除
					host::display.debug() << "Unusable candidate "
						<< (*p_node)->getNodeName() << std::endl;
					m_unusable_candidates.push_back(*p_node);
					p_node = m_unclear_candidates.erase(p_node);
				}

			}
		}

	}

	if( return_node != NULL ) {
		host::display.notice() << "Found device by file " << key_file << ": "
			<< return_node->getNodeName() << " (" << return_node->getFSType() << ")" << std::endl;
		return return_node;
	} else {
		throw DeviceNotFoundError( CANNOT + "find device by file " + key_file );
	}
}

void DeviceManager::lookupDeviceNodes(void)
{
	// 並列化するなら m_candidate_list と m_unclear_candidates のロックが必要
	// FIXME: HDDのREMOVABLEとFIXEDの判別ができない。
	std::set<std::string> skipped_disk;
	lookupProcPartitions(&skipped_disk);
	lookupSysBlocks(skipped_disk);
}

void DeviceManager::lookupProcPartitions(std::set<std::string>* p_skipped_disk)
{
	std::ifstream f_partitions(Component::Kernel::PATH_PROC_PARTITIONS);
	if( !f_partitions.is_open() ) {
		host::display.warn() << "Can't open " << Component::Kernel::PATH_PROC_PARTITIONS << std::endl;
		return;
	}

	std::string name;
	char buf[256];
	FileSystem::major_t major = 0;
	while( !f_partitions.eof() ) {
		f_partitions.getline(buf, sizeof(buf));

		unsigned int blocks = 0;
		FileSystem::minor_t minor = 0;
		std::istringstream line(buf);
		line >> major >> minor >> blocks >> name;

		if( blocks < 2 )
			continue;	// 読み取りに失敗したか、拡張パーティションのヘッダ

		if( !host::debug_mode && name.substr(0,::strlen("loop")) == "loop" )
			continue;	// デバッグモードでなければloopは飛ばす

		if( minor == 0 ) {
			// hda,sdaなど、パーティションでは無くディスク自体
			// /proc/partitionsに列挙されている時点で光学ディスクでは無い
			p_skipped_disk->insert(name);
			continue;
		}


		DeviceNodeCandidate candidate(major, minor, m_dev_path, name, DeviceNodeCandidate::FIXED_HDD);
		registerCandidate(candidate);
		host::display.verbose() << "Partition found: " << major << ":" << minor << "  " << name << std::endl;
	}
}

void DeviceManager::lookupSysBlocks(const std::set<std::string>& skipped_disk)
{
	// TODO: ブロッククサイズでDVDとCDを判別することも可能

	DIR *block_dir;
	struct dirent *block_dirent;

	block_dir = ::opendir(Component::Kernel::PATH_SYS_BLOCK_DIR);
	if( block_dir == NULL ) {
		host::display.warn() << "Can't open" << Component::Kernel::PATH_SYS_BLOCK_DIR << std::endl;
		return;
	}

	while( (block_dirent = ::readdir(block_dir)) != NULL ) {
		const char* dirname = block_dirent->d_name;
		if( dirname[0] == '0' ) continue;    // "." と ".." を飛ばす
		if( ::strncmp(dirname, "loop", ::strlen("loop")) == 0 ) continue;    // loopは飛ばす
		if( ::strncmp(dirname, "ram",  ::strlen("ram")) == 0 ) continue;     // ramは飛ばす
		if( ::strncmp(dirname, "dm",   ::strlen("dm"))  == 0 ) continue;     // dmは飛ばす
		if( ::strncmp(dirname, "md",   ::strlen("md"))  == 0 ) continue;     // mdは飛ばす
		if( ::strncmp(dirname, "fd",   ::strlen("fd"))  == 0 ) continue;     // XXX: fdはどうせ違うだろうから飛ばして良いだろうけど、どうする？

		std::ifstream block_dev_file(
				( Path(Component::Kernel::PATH_SYS_BLOCK_DIR) + dirname + "dev" ).c_str()
			);
		char buf[256];
		block_dev_file.getline(buf, sizeof(buf));
		if( block_dev_file.fail() ) {
			block_dev_file.close();
			continue;
		}
		block_dev_file.close();

		FileSystem::major_t major = 0;
		FileSystem::minor_t minor = 0;
		{
			std::string dev_string(buf);
			std::string::size_type pos_colon;
			if( (pos_colon = dev_string.find(':')) == std::string::npos ) continue;
			std::istringstream major_stream( dev_string.substr(0,pos_colon) );
			std::istringstream minor_stream( dev_string.substr(pos_colon+1) );
			major_stream >> major;
			minor_stream >> minor;
		}
		if( major == 0 ) continue;    // 読み取りに失敗した
		if( skipped_disk.find(dirname) != skipped_disk.end() ) continue; 

		DeviceNodeCandidate candidate(major, minor, m_dev_path, dirname, DeviceNodeCandidate::CD);
		registerCandidate(candidate);
		host::display.verbose() << "Optical disk found: " << major << ":" << minor << "  " << dirname << std::endl;
	}

	::closedir(block_dir);
}


void DeviceManager::registerCandidate(const DeviceNodeCandidate& dev)
{
	m_candidate_list.push_back(dev);
	DeviceNodeCandidate* p_node = &( *(--m_candidate_list.end() ) );
	m_unclear_candidates.push_back(p_node);
}


// private:
DeviceNode* DeviceManager::registerNode(const DeviceNode& dev)
{
	m_node_list.push_back(dev);    // DeviceNodeのコピー演算
	return &( *(--m_node_list.end() ) );
}

VirtualDevice* DeviceManager::registerVirtual(const VirtualDevice& vdev)
{
	m_virtual_list.push_back(vdev);    // VirtualDeviceのコピー演算
	return &( *(--m_virtual_list.end() ) );
}


}  // namespace VIVER
