#ifndef __DEVICE_MANAGER_H_INCLUDED
#define __DEVICE_MANAGER_H_INCLUDED 1

#include "devicenode.h"
#include "devicenode_candidate.h"
#include "virtualdevice.h"
#include "filesystem.h"
#include "csv.h"
#include "exception.h"
#include <string>
#include <list>
#include <set>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <boost/utility.hpp>

namespace VIVER {


// 例外
VIVER_MakeRuntimeError( DeviceNotFoundError );


class DeviceManager : private boost::noncopyable {
public:
	DeviceManager();		// throw()
	~DeviceManager();		// throw()
public:
	DeviceNode* loopSetup(const Path& path,
			const std::string& fstype, unsigned long mount_flags, const std::string& data_string);
			// throw(ExecuteFailedError)
	void loopRemove(DeviceNode* dn_loop);

	// まだマウントされていないDeviceNodeを作成
	DeviceNode* requestNode(FileSystem::major_t major, FileSystem::minor_t minor,
			const std::string& node_name,
			const std::string& fstype, unsigned long mount_flags, const std::string& data_string);
		// throw()

	// 既にマウントされているDeviceNodeを作成
	DeviceNode* requestNode(FileSystem::major_t major, FileSystem::minor_t minor,
			const std::string& node_name,
			const std::string& fstype, unsigned long mount_flags, const std::string& data_string,
			const Path& mount_point);
		// throw()

	// requestしたNodeを返却
	void returnNode(DeviceNode* node);	// throw()

	// まだマウントされていないVirtualDevcieを作成
	VirtualDevice* requestVirtual(const std::string& target_name,
			const std::string& fstype, unsigned long mount_flags, const std::string& data_string);

	void makeNodesOn(const Path& dev_path);	// throw()
	void umountTrialPoints(void);		// throw()
public:
	typedef enum {
		FILE_KEY,
		DIR_KEY,
	} keyfile_type_t;
	Mountable* searchDeviceByFileRO(const std::string& key_file);	// XXX: マウントオプションは指定できない
			// throw( DeviceNotFoundError )
	Mountable* searchDeviceByFileRW(const std::string& key_file,
			keyfile_type_t* return_keyfile_type );		// XXX: マウントオプションは指定できない
			// throw( DeviceNotFoundError )
public:
	const std::list<DeviceNode>& getNodeList(void) { return m_node_list; }	// throw()
private:
	typedef std::list<DeviceNode> node_list_t;
	node_list_t m_node_list;

	typedef std::list<VirtualDevice> virtual_list_t;
	virtual_list_t m_virtual_list;

	typedef std::set<FileSystem::minor_t> used_loop_t;
	used_loop_t m_used_loop;

	Path m_dev_path;
private:
	typedef std::list<DeviceNodeCandidate> candidate_list_t;
	typedef std::list<DeviceNodeCandidate*> candidate_class_t;
	typedef std::list<DeviceNode*> certain_class_t;
	candidate_list_t m_candidate_list;
	candidate_class_t m_unclear_candidates;
	candidate_class_t m_unusable_candidates;
	certain_class_t m_certain;
private:
	DeviceNode* registerNode(const DeviceNode& dev);
	VirtualDevice* registerVirtual(const VirtualDevice& vdev);
private:
	DeviceNode* searchDeviceByFile(const std::string& key_file,
			keyfile_type_t* return_keyfile_type );

	void registerCandidate(const DeviceNodeCandidate& dev);
	void lookupDeviceNodes(void);
	void lookupProcPartitions(std::set<std::string>* p_skipped_disk);
	void lookupSysBlocks(const std::set<std::string>& skipped_disk);
};


}  // namespace VIVER

#endif /* devicemanager.h */
