#include "mountable.h"
#include "display.h"
#include "constant.h"
#include <string.h>
#include <sys/mount.h>
#include <linux/fs.h>

// for pivot_root
#include <fstream>
#include <linux/unistd.h>
_syscall2(int, pivot_root, const char *, new_root, const char *, put_old);

namespace VIVER {


// コンストラクタ
Mountable::Mountable(const std::string& fstype, unsigned long mount_flags, const std::string& data_string) :
	m_mounted(false),
	m_fstype(fstype),
	m_mount_flags(mount_flags),
	m_data_string(data_string)
{}

Mountable::Mountable(const std::string& fstype, unsigned long mount_flags, const std::string& data_string,
			const Path& mount_point) :
	m_mounted(true),
	m_fstype(fstype),
	m_mount_flags(mount_flags),
	m_data_string(data_string),
	m_mount_point(mount_point)
{}

Mountable::Mountable()
{}


// デストラクタ
Mountable::~Mountable()
{}


// public:
void Mountable::pivot_root(Mountable* new_root, const Path& put_old)
{
	host::display.notice() << "pivot_root "
		<< new_root->getMountPoint().str() << " "
		<< (new_root->getMountPoint() + put_old).str()
		<< std::endl;

	if( host::debug_mode ) {
		// デバッグモードのときは pivot_root しない
		throw MountError( "DEBUG MODE" );
	}

	FileSystem::makeDirForce(new_root->getMountPoint() + put_old, 0700);
	if( ::pivot_root(new_root->getMountPoint().c_str(), (new_root->getMountPoint() + put_old).c_str()) < 0 ) {
		throw MountError( CANNOT + "pivot_root" + errorDescribe() );
	}
	new_root->m_mount_point = ("/");

	std::ofstream realroot(Component::Kernel::PATH_PROC_REALROOT);
	if( ! realroot.is_open() ) {
		throw NoSuchFileError(
				CANNOT + "open " + Component::Kernel::PATH_PROC_REALROOT + errorDescribe() );
	}
	realroot << "0x0100";		// XXX: この値は？
	realroot.close();
}
void Mountable::pivotNotice(const Path& current_mount_point)
{
	m_mount_point = current_mount_point;
}

// protected:
void Mountable::mount(const std::string& device, const std::string& mount_point, const std::string& fstype,
		unsigned long mount_flags, const std::string& data_string)
{
	host::display.notice() << "Mounting " << device
			       << " on " << mount_point
			       << " type " << fstype
			       << " (" << makeHandOptions(mount_flags, data_string) << ")"
			       << std::endl;

	int retval;
	unsigned short i=1;
	const char* mount_data = ( data_string.empty() ? NULL : data_string.c_str() );
	while( (retval = ::mount(device.c_str(), mount_point.c_str(), fstype.c_str(), mount_flags, mount_data)) != 0 ) {
		// 4回までリトライ
		if( i > 4 ) {
			throw MountError( CANNOT + "mount " + device
					+ " on " + mount_point
					+ " type " + fstype
					+ " (" + makeHandOptions(mount_flags, data_string) + ")"
					+ errorDescribe() );
		}
		host::display.debug() << "Can't mount " << mount_point << errorDescribe() << std::endl;
		::usleep(i<<19);
		i++;
	}
}

void Mountable::mount(const std::string& device, const std::string& mount_point, const std::string& fstype,
		const std::string& hand_options)
{
	std::pair<unsigned long, std::string> opt(
			splitHandOptions(hand_options)
			);

	return mount(device, mount_point, fstype, opt.first, opt.second);
}



void Mountable::move(const std::string& current_point, const std::string& new_point)
{
	host::display.notice() << "Moving mount point from " << current_point << " to " << new_point << std::endl;

	int retval;
	int i=1;
	while( (retval = ::mount(current_point.c_str(), new_point.c_str(), NULL, MS_MOVE, NULL)) != 0 ) {
		// filesystemtype, mountflags, dataは無視される
		// 4回までリトライ
		if( i > 4 ) {
			throw MountError(
				CANNOT + "move " + current_point + " to " + new_point + errorDescribe() );
		}
		host::display.debug() << "Can't move mount point from " << current_point << " to " << new_point << errorDescribe() << std::endl;
		::usleep(i<<19);
		i++;
	}
}

void Mountable::umount(const std::string& mount_point)
{
	host::display.notice() << "Unmounting " << mount_point << std::endl;
	if( ::umount(mount_point.c_str()) < 0 ) {
		throw MountError( CANNOT + "umount " + mount_point + errorDescribe() );
	}
}

void Mountable::remount(const std::string& device, const std::string& mount_point,
		unsigned long mount_flags, const std::string& data_string)
{
	host::display.notice() << "Remounting " << device
		               << " on " << mount_point
			       << " (" << makeHandOptions(mount_flags, data_string) << ")"
			       << std::endl;

	const char* mount_data = ( data_string.empty() ? NULL : data_string.c_str() );
	int retval;
	int i=1;
	while( (retval = ::mount(device.c_str(), mount_point.c_str(), NULL, MS_REMOUNT | mount_flags, mount_data)) != 0 ) {
		// filesystemtypeは無視される
		// 4回までリトライ
		if( i > 4 ) {
			throw MountError( CANNOT + "remount " + device
					+ " on " + mount_point
					+ " (" + makeHandOptions(mount_flags, data_string) + ")"
					+ errorDescribe() );
		}
		host::display.debug() << "Can't remount " << mount_point << errorDescribe() << std::endl;
		::usleep(i<<19);
		i++;
	}
}

void Mountable::remount(const std::string& device, const std::string& mount_point,
		const std::string& hand_options)
{
	std::pair<unsigned long, std::string> opt( splitHandOptions(hand_options) );
	remount(device, mount_point, opt.first, opt.second );
}


// private:
std::pair<unsigned long, std::string> Mountable::splitHandOptions(const std::string& hand_options) const
{
	// hand_optionsからVFSオプションとファイルシステム固有オプションを分離する
	std::string::size_type pos;

	std::string cs( std::string(",") + hand_options );
	unsigned long mount_flags = 0;

	if( (pos = cs.find(",ro")) != std::string::npos ) {
		mount_flags = mount_flags & MS_RDONLY;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",ro") );

	} else if( (pos = cs.find(",noatime")) != std::string::npos ) {
		mount_flags = mount_flags & MS_NOATIME;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",noatime") );

	} else if( (pos = cs.find(",nodev")) != std::string::npos ) {
		mount_flags = mount_flags & MS_NODEV;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",nodev") );

	} else if( (pos = cs.find(",mand")) != std::string::npos ) {
		mount_flags = mount_flags & MS_MANDLOCK;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",mand") );

	} else if( (pos = cs.find(",noexec")) != std::string::npos ) {
		mount_flags = mount_flags & MS_NOEXEC;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",noexec") );

	} else if( (pos = cs.find(",nosuid")) != std::string::npos ) {
		mount_flags = mount_flags & MS_NOSUID;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",nosuid") );

	} else if( (pos = cs.find(",sync")) != std::string::npos ) {
		mount_flags = mount_flags & MS_SYNCHRONOUS;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",sync") );

	} else if( (pos = cs.find(",dirsync")) != std::string::npos ) {
		mount_flags = mount_flags & MS_DIRSYNC;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",dirsync") );

	/*  // FIXME: 文字列が不明
	} else if( (pos = cs.find(",")) != std::string::npos ) {
		mount_flags = mount_falgs & MS_NODIRATIME;
		cs = cs.substr(0,pos) + cs.substr( pos + ::strlen(",") );
	*/
	}

	return std::make_pair(mount_flags, cs.substr(1));
}

std::string Mountable::makeHandOptions(unsigned long mount_flags, const std::string& data_string) const
{
	// 見た目に良いように成形したマウントオプション文字列を返す
	std::string vfs_options;
	if( mount_flags & MS_RDONLY      ) vfs_options += ",ro";
	if( mount_flags & MS_NOATIME     ) vfs_options += ",noatime";
	if( mount_flags & MS_NODEV       ) vfs_options += ",nodev";
	if( mount_flags & MS_MANDLOCK    ) vfs_options += ",mand";
	if( mount_flags & MS_NOEXEC      ) vfs_options += ",noexec";
	if( mount_flags & MS_NOSUID      ) vfs_options += ",nosuid";
	if( mount_flags & MS_SYNCHRONOUS ) vfs_options += ",sync";
	if( mount_flags & MS_DIRSYNC     ) vfs_options += ",dirsync";
	//if( mount_flags & MS_NODIRATIME ) vfs_options += ",";  // FIXME: 文字列が不明

	if( vfs_options.empty() ) {
		if( data_string.empty() ) return "rw";
		else return data_string;
	} else {
		// vfs_options.substr(1)は先頭の,を取り除く
		if( data_string.empty() ) return vfs_options.substr(1);
		else return vfs_options.substr(1) + "," + data_string;
	}
}


}  // namespace VIVER

