#ifndef __MOUNTABLE_H_INCLUDED
#define __MOUNTABLE_H_INCLUDED 1

#include "path.h"
#include "filesystem.h"
#include "exception.h"
#include <string>

// for mount_flags
#include <sys/mount.h>
#include <linux/fs.h>

namespace VIVER {


// 例外
VIVER_MakeRuntimeError( MountError );


class Mountable {
public:
	Mountable(const std::string& fstype, unsigned long mount_flags, const std::string& data_string);	// throw()
	Mountable(const std::string& fstype, unsigned long mount_flags, const std::string& data_string,
			const Path& mount_point);	// throw()
	Mountable();	// throw()
	virtual ~Mountable();	// throw()
public:
	virtual void mount(const Path& mount_point) = 0;
	virtual void umount(void) = 0;
	virtual void move(const Path& mount_point) = 0;
	virtual void remount(unsigned long mount_flags, const std::string& data_string) = 0;

	const std::string& getFSType(void) const { return m_fstype; }		// throw()
	unsigned long getMountFlags(void) const { return m_mount_flags; }	// throw()
	const std::string& getDataString(void) const { return m_data_string; }	// throw()
	const std::string getHandOptions(void) const				// throw()
	{ return Mountable::makeHandOptions(m_mount_flags, m_data_string); }

	// XXX: マウントされていなかったら？
	const Path& getMountPoint(void) const { return m_mount_point; }		// throw()

	bool isMounted(void) const { return m_mounted; }			// throw()
public:
	static void pivot_root(Mountable* new_root, const Path& put_old);	// throw(MountError)
	void pivotNotice(const Path& current_mount_point);			// throw()
protected:
	bool m_mounted;
	const std::string m_fstype;
	unsigned long m_mount_flags;
	std::string m_data_string;
	Path m_mount_point;
protected:
	virtual void mount(const std::string& device, const std::string& mount_point, const std::string& fstype,
			unsigned long mount_flags, const std::string& data_string);
			// throw(MountError)
	virtual void mount(const std::string& device, const std::string& mount_point, const std::string& fstype,
			const std::string& hand_options);
			// throw(MountError)

	virtual void move(const std::string& current_point, const std::string& new_point);
			// throw(MountError)

	virtual void umount(const std::string& mount_point);
			// throw(MountError)

	virtual void remount(const std::string& device, const std::string& mount_point,
			unsigned long mount_flags, const std::string& data_string);
			// throw(MountError)
	virtual void remount(const std::string& device, const std::string& mount_point,
			const std::string& hand_options);
			// throw(MountError)
protected:
	std::pair<unsigned long, std::string> splitHandOptions(const std::string& hand_options) const;
			// throw()

	std::string makeHandOptions(unsigned long mount_flags, const std::string& data_string) const;
			// throw()
};


}  // namespace VIVER

#endif /* mountable.h */
