#include "constant.h"
#include "parameters.h"
#include "exception.h"
#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <string.h>
#include <strings.h>    // for ::strcasecmp()
#include <boost/format.hpp>

namespace VIVER {


// コンストラクタ
Parameters::Parameters(const std::string& boot_parameter_file) :
	m_boot_parameter_file(boot_parameter_file)
{}


// デストラクタ
Parameters::~Parameters()
{}


// public:
void Parameters::parse(void)
{
	std::ifstream bootparam(m_boot_parameter_file.c_str());
	if ( !bootparam.is_open() ) {
		throw NoSuchFileError( CANNOT + "open " + m_boot_parameter_file + errorDescribe() );
	}

	// 初期化
	m_parameters.clear();

	// GLOBAL名前空間を登録
	std::pair<parameters_t::iterator, bool> ins_init(
			m_parameters.insert(
				parameters_t::value_type(
					"GLOBAL",
					valmap_t()
					)
				)
			);
	if( ins_init.second == false ) return;    // XXX: 例外
	parameters_t::iterator current_ns( ins_init.first );

	std::string param;
	std::pair<parameters_t::iterator,bool> ins;
	while (bootparam >> param) {
		if( param.empty() ) continue;

		std::string::size_type poseq( param.find_first_of('=') );
		if (poseq != std::string::npos) {
			// key=val
			// 値付きパラメータ
			std::pair<valmap_t::iterator, bool> ins(
					current_ns->second.insert(
						valmap_t::value_type(
							param.substr(0,poseq),
							param.substr(poseq + 1)
							)
						)
					);
			if( !ins.second ) {
				// map::insertが失敗 -> キーが重複 -> valを上書き
				current_ns->second[param.substr(0,poseq)] = param.substr(poseq + 1);
			}

		} else if(param[param.size()-1] == ':') {
			// ns:
			// 名前空間指定
			std::string ns = param.substr(0,param.size()-1);
			if( ns.empty() ) ns = "GLOBAL";
			parameters_t::iterator new_ns( m_parameters.find(ns) );
			if( new_ns == m_parameters.end() ) {
				// 指定された名前空間は登録されていない
				std::pair<parameters_t::iterator,bool> ins_ns(
						m_parameters.insert(
							parameters_t::value_type(
								ns,
								valmap_t()
								)
							)
						);
				if( ins_ns.second == false ) continue;
				current_ns = ins_ns.first;
			} else {
				// 指定された名前空間は登録済み
				current_ns = new_ns;
			}

		} else {
			// key
			// 単発のパラメータ
			std::pair<valmap_t::iterator, bool> ins(
					current_ns->second.insert(
						valmap_t::value_type(
							param,
							std::string()
							)
						)
					);
			if( !ins.second ) {
				// map::insertが失敗 -> キーが重複
				current_ns->second[param] = std::string();
			}
		}
	}
	bootparam.close();
}


// public:
const char* Parameters::getValueString(const char* key, const char* default_value) const
{
	parameters_t::const_iterator found_ns_viver( m_parameters.find(PARAMETER_NAMESPACE) );
	if( found_ns_viver != m_parameters.end() ) {
		valmap_t::const_iterator found_key = found_ns_viver->second.find(key);
		if( found_key != found_ns_viver->second.end() )
			return found_key->second.c_str();
	}

	parameters_t::const_iterator found_ns_global( m_parameters.find("GLOBAL") );
	if( found_ns_global != m_parameters.end() ) {
		valmap_t::const_iterator found_key = found_ns_global->second.find(key);
		if( found_key != found_ns_global->second.end() )
			return found_key->second.c_str();
	}

	return default_value;
}

bool Parameters::getValueBool(const char* key, bool default_value) const
{
	// keyが存在しなければdefault_value
	// keyが存在して、false || n || N || no || 0 ならfalse
	// それ以外はtrue
	const char* val( getValueString(key, NULL) );
	if ( val == NULL ) {
		// keyが存在しない
		return default_value;
	}
	// FIXME: 重いし冗長。他の解決方法を
	if( strcmp(val,"false")==0 || strcmp(val,"n")==0 || strcmp(val,"N")==0 || strcmp(val,"no")==0 || strcmp(val,"0")==0 ) {
		return false;
	} else {
		return true;
	}
}

int Parameters::getValueInt(const char* key, int default_value) const
{
	const char* val = getValueString(key, (char*)NULL);
	if ( val == NULL || strlen(val) != strspn(val,"0123456789") ) {
		// keyが存在しないか、valに[0-9]以外の文字が含まれている
		return default_value;
	} else {
		return atoi(val);
	}
}

double Parameters::getValueDouble(const char* key, double default_value) const
{
	const char* val = getValueString(key, (char*)NULL);
	if ( val == NULL || strlen(val) != strspn(val,"0123456789.") || strchr(val,'.') == strrchr(val,'.') ) {
		// keyが存在しないか、valに[0-9.]以外の文字が含まれているか、. が二つ以上ある(前から . を探した位置と後ろから探した位置が一致しない)
		return default_value;
	} else {
		return atof(val);
	}
}

Parameters::with_unit_t Parameters::getValueByte(const char* key, with_unit_t default_value) const
{
	const char* val = getValueString(key, (char*)NULL);
	if ( val == NULL ) return default_value;	// keyが存在しない

	size_t num_length = ::strspn(val, "0123456789");

	with_unit_t::first_type num = default_value.first;
	std::istringstream num_stream( std::string(val).substr(0,num_length) );
	num_stream >> num;
	if( num_stream.fail() ) return default_value;

	with_unit_t::second_type unit;
	const char* unit_str = &val[num_length];
	if( unit_str[0] == '\0' ) {	// 単位指定なし
		unit = default_value.second;
	} else if( ::strcasecmp(unit_str, "m") == 0 || ::strcasecmp(unit_str, "mb") == 0 ) {
		unit = MB_UNIT;
	} else if( ::strcasecmp(unit_str, "g") == 0 || ::strcasecmp(unit_str, "gb") == 0 ) {
		unit = GB_UNIT;
	} else if( ::strcasecmp(unit_str, "k") == 0 || ::strcasecmp(unit_str, "kb") == 0 ) {
		unit = KB_UNIT;
	} else if( ::strcasecmp(unit_str, "t") == 0 || ::strcasecmp(unit_str, "tb") == 0 ) {
		unit = TB_UNIT;
	} else {
		// 解釈できない単位
		unit = INVALID_UNIT;
	}

	return with_unit_t(num, unit);
}

void Parameters::writeToFile(const std::string& dir) const
{
	// TODO: 並列実行
	writeToFileCompatible(dir);

	// シェルスクリプト
	writeToFileFormat(dir, ".sh",
			"param_%1%_%2%=\"%3%\";",
			"system_%1%=\"%2%\";",
			"#!/bin/sh",
			"");

	// XML
	writeToFileFormat(dir, ".xml",
			"\t<Parameter namespace=\"%1%\" key=\"%2%\">%3%</Parameter>",
			"\t<System key=\"%1%\">%2%</System>",
			"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<Parameters>",
			"</Paramters>");

	// Ruby		XXX: 未実装
	//writeToFileFormatRb(dir, ".rb"

	// Perl		XXX: 未実装
	//writeToFileFormatPl(dir, ".pl"

	// Python	XXX: 未実装
	//writeToFileFormatPy(dir, ".py"
}


// private:
void Parameters::writeToFileFormat(const std::string& dir,
		const char* file_suffix,
		const char* parameter_format,
		const char* system_format,
		const char* header, const char* footer) const
{
	std::ofstream file( (dir + "/parameter" + file_suffix).c_str() );
	if( !file.is_open() ) return;

	file << header << std::endl;

	for( parameters_t::const_iterator current_ns( m_parameters.begin() );
			current_ns != m_parameters.end();
			++current_ns ) {
		std::string ns( current_ns->first );
		for( valmap_t::const_iterator param( current_ns->second.begin() );
				param != current_ns->second.end();
				++param ) {
			// XXX: "や>や<をエスケープしていない
			file << (boost::format(parameter_format) % ns % param->first % param->second) << std::endl;
		}
	}
	file << ( boost::format(system_format) % "MP_ARK"         % MP::ARK.str()         ) << std::endl;
	file << ( boost::format(system_format) % "MP_MODULE_DISK" % MP::MODULE_DISK.str() ) << std::endl;
	file << ( boost::format(system_format) % "MP_FORMED"      % MP::FORMED.str()      ) << std::endl;

	file << footer << std::endl;

	file.close();
}

void Parameters::writeToFileCompatible(const std::string& dir) const
{
	// viverrc 0.1 compatible
	FileSystem::makeDirForce("/etc/VIVER/var/VIVER", 0700);
	std::ofstream file("/etc/VIVER/var/VIVER/bootparams");
	if( !file.is_open() ) return;
	for( parameters_t::const_iterator current_ns( m_parameters.begin() );
			current_ns != m_parameters.end();
			++current_ns ) {
		std::string ns( current_ns->first );
		for( valmap_t::const_iterator param( current_ns->second.begin() );
				param != current_ns->second.end();
				++param ) {
			// XXX: "をエスケープしていない
			file << "param_" << param->first << "=\"" << param->second << "\"" << std::endl;
		}
	}
	file.close();

	// XXX: 場当たり的 viverrc 0.1 互換モード
	std::ofstream sysparams("/etc/VIVER/var/VIVER/sysparams");
	if( !sysparams.is_open() ) return;
	sysparams << "MTSYS=\"" << MP::FORMED.str()  << "\"" << std::endl;
	sysparams << "MTDISK=\"" << ARK::MP_BOOTDISK.str() << "\"" << std::endl;
	sysparams << "MTSQUASH=\"" << ARK::MP_COMPRESSED.str() << "\"" << std::endl;
	sysparams << "MTCOW=\"" << ARK::MP_SCREEN.str() << "\"" << std::endl;
	sysparams << "MTNOAH=\"" << FORMED::MP_ARK.str() << "\"" << std::endl;
	sysparams << "RCDIR=\"/etc/VIVER\"" << std::endl;
	sysparams << "DEVDIR=\"/etc/VIVER/dev\"" << std::endl;
	sysparams << "looppath=\"" << "/" << DefaultInfo::Image::COMPRESSED_ON_BOOTDISK << "\"" << std::endl;
	sysparams << "rootserver=\"" << getValueString("bootstrap", "") << "\"" << std::endl;
	sysparams << "BB=\"" << "/initrd/bin/busybox" << "\"" << std::endl;
	sysparams.close();
}


}  // namespace VIVER
