#include "vtable/vtable.h"
#include "rpc/rpcserver.h"
#include "rpc/rpcclient.h"
#include "storage/storage.h"
#include "stream/stream.h"
#include "search/controller.h"
#include "interface/interface.h"
#include "interface/nbd/if_nbd.h"
#include "vfield.h"
#include "threadpool.h"
#include "exception.h"
#include "argparser.h"
#include "log.h"
#include "color.h"
#include <signal.h>
#include <unistd.h>
#include <iostream>
#include <boost/format.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/scoped_ptr.hpp>

using namespace VFIELD;

namespace {
// XXX: daemonizeが必要


volatile sig_atomic_t stop_flag;
void signal_handler(int)
{
	stop_flag = 1;
	// 2回目は強制終了
}

struct Parameters {
	Parameters();
	std::string port;
	std::string bootstrap;
	std::string image;
	std::string netif;
	std::string self;
	uint64_t storelimit;
	uint16_t nbd_port;
	bool nonbd;
	bool nobcast;
	unsigned short init_threads;
	size_t max_connection;
	unsigned int sweep_interval;
	unsigned int first_join_delay;
};

// デフォルトパラメータ
Parameters::Parameters() :
	port( boost::lexical_cast<std::string>(DEFAULT_LISTEN_PORT) ),
	// bootstrap = empty
	// image = empty
	// netif = 後処理
	// self  = netifパラメータから得る
	// storelimit = 後処理
	nbd_port(8990),
	nonbd(false),
	nobcast(false),
	init_threads(THREAD_POOL_INITIAL_THREADS),
	max_connection(STREAM_MANAGER_MAX_CONNECTION),
	sweep_interval(STREAM_MANAGER_SWEEP_INTERVAL),
	first_join_delay(0)
{
	// ここではLog*が使えないので注意

	// デフォルトのstorelimitを計算
	// XXX: 未実装
	storelimit = 64;

	// デフォルトのnetifを得る
	std::vector<std::string> iflist;
	getIfList(iflist);
	if( iflist.empty() ) {
		netif = "";
	} else if( iflist.size() == 1 ) {
		netif = *iflist.begin();
	} else {
		netif = *(++iflist.begin());
	}
}

}  // noname namespace


int main(int argc, char* argv[])
{
	stop_flag = 0;

	Parameters parameters;
	{  // コマンドラインパラメータのパース
		ArgParser parser;

		parser.addKey( StringKey("file",
					"file to export (this node becames initial node when this parameter is set)",
					CopyOperator(parameters.image)
					) );
		parser.addKey( StringKey("bs",
					"bootstrap node address",
					CopyOperator(parameters.bootstrap)
					) );
		parser.addKey( StringKey("port",
					boost::io::str(
						boost::format("Port number for listening and sending broadcast [%1%]")
						% parameters.port ),
					CopyOperator(parameters.port)
					) );
		parser.addKey( StringKey("store",
					boost::io::str(
						boost::format("limit of data storing size in MB [%1%]")
						% parameters.storelimit ),
					NumericCopyOperator(parameters.storelimit)
					) );
		parser.addKey( StringKey("if",
					boost::io::str(
						boost::format("network interface to use [%1%]")
						% parameters.netif ),
					CopyOperator(parameters.netif)
					) );
		parser.addKey( StringKey("nbd_port",
					 boost::io::str(
						 boost::format("port number of NBD server [%1%]")
						 % parameters.nbd_port ),
					 NumericCopyOperator(parameters.nbd_port)
					) );
		parser.addKey( ExistKey("nonbd",
					"don't use NBD",
					SetOperator(parameters.nonbd)
				       ) );
		parser.addKey( StringKey("max_connection",
					boost::io::str(
						boost::format("max number of tcp connection [%1%]")
						% parameters.max_connection ),
					NumericCopyOperator(parameters.max_connection)
					) );
		parser.addKey( StringKey("sweep",
					boost::io::str(
						boost::format("interval time of sweeping tcp connection in seconds [%1%]")
						% parameters.sweep_interval ),
					NumericCopyOperator(parameters.sweep_interval)
					) );
		parser.addKey( StringKey("join_delay",
					boost::io::str(
						boost::format("delay time of first join in seconds [%1%]")
						% parameters.first_join_delay ),
					NumericCopyOperator(parameters.first_join_delay)
					) );
		parser.addKey( StringKey("init_thread",
					boost::io::str(
						boost::format("initial number of threads of threadpool [%1%]")
						% parameters.init_threads ),
					NumericCopyOperator(parameters.init_threads)
					) );
		/*
		parser.addKey( ExistKey("nobcast",
					"don't use broadcast (not impremented)",
					SetOperator(parameters.nobcast)
				       ) );
				       */


		try {
			parser.parse(argc, argv);
		} catch (const ArgumentException& ex) {
			std::cout << ex.what() << std::endl;
			parser.showUsage();
			return 1;
		}

		if( parameters.bootstrap.empty() && parameters.image.empty() ) {
			std::cout << "-bs or -file parameter must be set" << std::endl;
			parser.showUsage();
			return 1;
		}
	}  // コマンドラインパラメータのパース


	// ログオブジェクトを初期化
	// 終了直前まで使うので、ここで確保
	boost::scoped_ptr<Log::LogInitializer> loginit;
	try {
		loginit.reset( new Log::LogInitializer() );
	} catch (...) {
		std::cerr << "Can't initialize logging function" << std::endl;
		exit(1);
	}


	try {

		if( signal(SIGUSR1, signal_handler) == SIG_ERR ) {
			throw std::runtime_error("Failed to set signal USR1 handler");
		}
		if( signal(SIGINT, signal_handler) == SIG_ERR ) {
			throw std::runtime_error("Failed to set signal INT handler");
		}
		if( signal(SIGTERM, signal_handler) == SIG_ERR ) {
			throw std::runtime_error("Failed to set signal TERM handler");
		}
		// SIGPIPEは無視する
		if( signal(SIGPIPE, SIG_IGN) == SIG_ERR ) {
			throw std::runtime_error("Failed to set signal PIPE handler");
		}


		bool is_root = !parameters.image.empty();

		// selfのアドレスをparameters.netifから得る
		NodeIdentity self(getIfAddr(parameters.netif).c_str(), parameters.port.c_str());

		// ThreadPool
		ThreadPool threadpool(parameters.init_threads);

		// RPCClient
		// XXX: image idは？
		RPCClient rpcc(parameters.port.c_str(), self);

		// VTable
		VTable vtable(rpcc);

		// Storage
		boost::scoped_ptr<RootStorage> rstorage;
		boost::scoped_ptr<EdgeStorage> estorage;
		Storage* storage;
		if( is_root ) {
			rstorage.reset( new RootStorage(parameters.image, rpcc) );
			storage = rstorage.get();
			// EdgeStorage::invokeJoinに対応する処理
			rstorage->initialize(self, vtable);
		} else {
			// parameters.storelimitはMB単位、EdgeStorageToRAMはbyte単位
			estorage.reset( new EdgeStorageToRAM(parameters.storelimit * 1024 * 1024, rpcc, vtable, threadpool) );
			storage = estorage.get();
			// VTableへの自分の追加は、EdgeStorage::invokeJoinで行われる
		}

		// StreamManager
		StreamManager smgr(parameters.port.c_str(), *storage, vtable, threadpool, parameters.max_connection, parameters.sweep_interval);

		// SearchController
		SearchController engine(smgr, rpcc, vtable, threadpool);

		RPCServer rpcs(parameters.port.c_str(), smgr, vtable, *storage, engine, self, threadpool);

		boost::scoped_ptr<Interface> viff;
		if( is_root ) {
			viff.reset( new Interface(*rstorage) );
		} else {
			viff.reset( new Interface(*estorage, engine) );
		}

		boost::scoped_ptr<InterfaceNBD> nbd;
		if( !parameters.nonbd ) {
			nbd.reset( new InterfaceNBD(*viff, parameters.nbd_port, threadpool) );
		}

		if( !is_root ) {
			// invokeJoinは失敗したときに例外を投げる
			estorage->invokeJoin(
					smgr,
					rpcs,
					engine,
					NodeIdentity(
						parameters.bootstrap.c_str(),
						parameters.port.c_str()
						),
					parameters.first_join_delay,
					true	// 最初joinはmain()のスレッドで実行し、例外を受け取る
					);
		}

		// TODO: V-FIELDモニタ
		while(!stop_flag) {
			pause();	// シグナルを待つ
		}

		// RPCNotifySelfDownを送るのはVTableのデストラクタ

	} catch(const JoinFailedException& ex) {
		LogError( Log::format("Can't complete Stream Join %1%, abort") % ex.what() );
		exit(1);
	} catch(const std::runtime_error& ex) {
		std::cerr << Color::RED << ex.what() << Color::NORMAL << std::endl;
		LogError(ex.what());
		exit(1);
	}
	// FIXME: catchにbad_allocが必要

	return 0;
}

