require "log"
require "fsutils"
require "vfs"

class SharedBlockError < RuntimeError
	def initialize(msg)
		@msg = msg
	end
	def to_s
		return @msg
	end
end

class SharedBlockFS < VFS
	def initialize(sharedblock)
		@sharedblock = sharedblock
		super()
	end
	def mount(path, fstype, opt)
		self.mountdev(@sharedblock, path, fstype, opt)
	end
end


class SharedBlock
	DEFAULT_PORT = "19876"
	NBD_PORT = "8990"

	def initialize(network, initrdfs, arkfs, procfs, kmodule)
		@network = network
		@initrdfs = initrdfs
		@arkfs = arkfs
		@procfs = procfs
		@kmodule = kmodule
	end


	def getSmartStoreSize
		# MB単位で返す
		return (@procfs.memtotal.to_f / 5.5 / 1024).to_i  # ProcFS::memtotalはKB単位で返してくる
	end

	def genVFIELDArgs
		# vfieldの引数を作成

		# -bs  -port
		# bootstrap=192.168.0.1
		# bootstrap=192.168.0.1:19770
		ip, port = $param.getValue("bootstrap", "").split(":", 2)
		if ip === nil || ip.empty?
			raise SharedBlockError.new("Invalid bootstrap ip: #{ip}")
		end
		if port == nil || port.empty?
			port = DEFAULT_PORT
		end

		# -store=
		store = 64  # vfieldのサイズ指定はMB単位
		sizeline = $param.getValue("store", "")
		if !sizeline.empty?
			begin
				match = /^([1-9][0-9]*)(.?)[bB]?$/.match(sizeline)
				size_user = Integer(match[1])
				unit_user = match[2].upcase
				case unit_user
				when ""
					# デフォルトの単位はMB
					store = size_user.to_i
				when "K"
					store = size_user.to_i / 1024
				when "M"
					store = size_user.to_i
				when "G"
					store = size_user.to_i * 1024
				when "T"
					store = size_user.to_i * 1024 * 1024
				else
					raise
				end
			rescue
				raise InvalidParameterError.new(sizeline, "Invalid size")
			end
		else
			store = getSmartStoreSize
		end


		# -nbd_port
		nbd_port = "8990"


		other_args = ""


		# -join_delay
		join_delay_str = $param.getValue("join_delay", nil)
		if join_delay_str != nil
			other_args << " -join_delay #{join_delay_str}"
		end


		# -if
		net, route = @network.getInfo
		if !net.empty?
			useif = ""
			# method == dhcp -> autoip -> fallback の順で探す
			["dhcp", "autoip", "fallback"].each {|md|
				net.each {|netif|
					netif.name == "lo" && next
					if netif.method == md
						useif = netif.name
						break
					end
				}
				if !useif.empty?
					other_args << " -if #{useif}"
					break		# 最初に見つかったものを採用
				end
			}
		# XXX: genVFIELDArgs: -if: 一つも見つからなかったら？（netifがloしかない）
		# XXX: genVFIELDArgs: -if: netInfoがemtpyの場合は？
		end


		max_connection_user = $param.getValue("max_connection", nil)
		if max_connection_user != nil
			other_args << " -max_connection #{max_connection}"
		end

		init_thred_user = $param.getValue("init_thread", nil)
		if init_thred_user != nil
			other_args << " -init_thread #{init_thred_user}"
		end

		sweep_user = $param.getValue("sweep", nil)
		if sweep_user != nil
			other_args << " -sweep #{sweep_user}"
		end

		return "-bs #{ip} -port #{port} -store #{store} -nbd_port #{nbd_port} #{other_args}"
	end
	private :genVFIELDArgs


	def genNBDArgs(nbd_dev)
		return "127.0.0.1 #{NBD_PORT} #{nbd_dev}"
	end
	private :genNBDArgs



	def chrootBackExec(root, relative_cmd, args)
		pid = fork {
			#Process.daemon(true,true) rescue $log.warn $! # XXX: Process.daemonはruby 1.9からだった
			#@initrdfs.cmd_chroot(root, relative_cmd, args)
			# XXX: vfield/nbd-client起動方法はどうする？
			$log.debug "executing vfield #{args}"
			Kernel.exec("#{@initrdfs.chroot_file} #{root} #{relative_cmd} #{args}")
		}
		$log.debug0 "vfield chroot pid: #{pid}"
	end
	def chrootExec(root, relative_cmd, args)
		# XXX: nbd-clientはfrok要らない？
		pid = fork {
			#Process.daemon(true,true) rescue $log.warn $! # XXX: Process.daemonはruby 1.9からだった
			#@initrdfs.cmd_chroot(root, relative_cmd, args)
			$log.debug "executing nbd-client #{args}"
			Kernel.exec("#{@initrdfs.chroot_file} #{root} #{relative_cmd} #{args}")
		}
		$log.debug0 "nbd-client chroot pid: #{pid}"
	end

	def waitVFIELDConnect
		# XXX: vfield側の改良が必要: sleepするのはスマートでない
		sleep(4)
	end

	def waitNBDConnect(nbd_dev)
		# XXX: waitNBDConnect: ここのsleepは必要？
		sleep(3)
	end

	def getBlock(force_wakenet = false)
		@network.wakeup(force_wakenet)  # 例外は呼び出し元でキャッチ

		# nbdをロード
		@kmodule.load("nbd")


		# vfieldとnbd-clientをarkfsにコピー
		onark_vfield    = @arkfs.bin_relative + "/vfield"
		onark_nbdclient = @arkfs.bin_relative + "/nbd-client"
		if !File.executable?(@arkfs.point.to_s + onark_vfield)
			copy_file(@initrdfs.vfield_file, @arkfs.point.to_s + onark_vfield, 0755)
		end
		if !File.executable?(@arkfs.point.to_s + onark_nbdclient)
			copy_file(@initrdfs.nbdclient_file, @arkfs.point.to_s + onark_nbdclient, 0755)
		end


		# XXX: vfieldが動作していたらkillするコードが必要
		$log.info "Starting V-FIELD"
		chrootBackExec(@arkfs.point, onark_vfield, genVFIELDArgs)
		waitVFIELDConnect


		# XXX: nbd-clientが動作していたらkillするコードが必要
		nbd_dev = @arkfs.dev_nbd
		onark_nbd_dev = @arkfs.dev_nbd_relative
		chrootExec(@arkfs.point, onark_nbdclient, genNBDArgs(onark_nbd_dev))
		waitNBDConnect(nbd_dev)

		return SharedBlockFS.new(nbd_dev)
	end
end


$log.debug "#{File.basename(__FILE__, ".*")} loaded"

