# vfs.rb
require "log"

class MountError < RuntimeError
	def initialize(dev, path, fstype, opt, message)
		@dev = dev
		@path = path
		@fstype = fstype
		@opt = opt
		@message = message
	end
	def to_s
		return "Can't mount #{@dev} on #{@path} type #{@fstype} (#{@opt}): #{@message}"
	end
	attr_reader :dev, :path, :fstype, :opt, :message
end

class MountMoveError < MountError
	def initialize(dev, old_path, new_path, fstype, opt, message)
		@dev = dev
		@old_path = old_path
		@new_path = new_path
		@fstype = fstype
		@opt = opt
		@message = message
	end
	def to_s
		return "Can't move #{@dev} from #{@old_path} to #{@new_path} type #{@fstype} (#{@opt}): #{@message}"
	end
	attr_reader :dev, :old_path, :new_path, :fstype, :opt, :message
end

class RemountError < MountError
	def initialize(dev, path, fstype, old_opt, new_opt, message)
		@dev = dev
		@path = path
		@fstype = fstype
		@old_opt = old_opt
		@new_opt = new_opt
		@message = message
	end
	def to_s
		return "Can't remount #{@dev} on #{@path} type #{@fstype} (from #{@old_opt} to #{@new_opt}): #{@message}"
	end
	attr_reader :dev, :path, :fstype, :old_opt, :new_opt, :message
end

class UnmountError < MountError
	def initialize(dev, path, fstype, opt, message)
		@dev = dev
		@path = path
		@fstype = fstype
		@opt = opt
		@message = message
	end
	def to_s
		return "Can't unmount #{@dev} on #{@path} type #{@fstype} (#{@opt}): #{@message}"
	end
	attr_reader :dev, :path, :fstype, :opt, :message
end

class LoopSetupError < MountError
	def initialize(loop_dev, image, message)
		@loop_dev = loop_dev
		@image = image
		@message = message
	end
	def to_s
		return "Can't loop setup #{@image} to #{@loop_dev}: #{@message}"
	end
	attr_reader :loop_dev, :image, :message
end

class LoopDetachError < MountError
	def initialize(loop_dev, message)
		@loop_dev = loop_dev
		@message = message
	end
	def to_s
		return "Can't detach #{@loop_dev}: #{@message}"
	end
	attr_reader :loop_dev, :message
end

class NotMountedError < MountError
	def initialize(cmd)
		@cmd = cmd
	end
	def to_s
		return "Can't #{cmd}: not mounted"
	end
	attr_reader :cmd
end

class AlreadyMountedError < MountError
	def initialize(cur_dev, cur_point, cur_fstype, cur_opt, new_dev, new_path, new_fstype, new_opt)
		@cur_dev = cur_dev
		@cur_point = cur_point
		@cur_fstype = cur_fstype
		@cur_opt = cur_opt
		@new_dev = new_dev
		@new_path = new_path
		@new_opt = new_opt
	end
	def to_s
		return "Can't mount #{@new_dev} on #{@new_path} type #{@new_fstype} (#{@new_opt}): #{@cur_dev} is already mounted on #{@cur_path} type #{@cur_fstype} (#{@cur_opt})"
	end
end



class VFS
	def self.setInitrdFS(initrdfs)
		@@initrdfs = initrdfs
	end

	def initalize
		@dev    = nil
		@point  = nil
		@fstype = nil
		@opt    = nil
	end
	attr_reader :dev, :point, :fstype, :opt

	class IgnoreOut
		def write
		end
	end

	def mounted?
		return @point != nil
	end

	def mountdev(dev, path, fstype, opt)
		if mounted?; raise AlreadyMountedError.new(@dev, @point, @fstype, @opt, dev, path, fstype, opt); end
		opt.empty? && opt = "defaults"
		$log.debug0 "mount #{dev} on #{path} type #{fstype} (#{opt})"

		begin
			@@initrdfs.cmd_mount("-t #{fstype} -o #{opt} #{dev} #{path}")
			$log.debug "mount success: #{dev} on #{path} type #{fstype} (#{opt})"
			@dev    = dev
			@point  = path    # DynamicPath aware
			@fstype = fstype
			@opt    = opt
		rescue
			raise MountError.new(dev, path, fstype, opt, $!)
		end
	end
	protected :mountdev

	def move(new_path)
		if !mounted?; raise NotMountedError.new("unmount"); end
		$log.debug0 "move #{@dev} from #{@point} to #{new_path} type #{@fstype} (#{@opt})"

		begin
			@@initrdfs.cmd_mount("-o move #{@point} #{new_path}")
			$log.debug "move success: #{@dev} from #{@point} to #{new_path} type #{@fstype} (#{@opt})"
			@point.move(new_path)		# DynamicPath
		rescue
			raise MountMoveError.new(@dev, @point, new_path, @fstype, @opt,$!)
		end
	end

	def remount(new_opt)
		if !mounted?; raise NotMountedError.new("unmount"); end
		new_opt.empty? && new_opt = "defaults"
		$log.debug0 "remount #{@dev} on #{@path} type #{@fstype} (from #{@opt} to #{new_opt})"

		begin
			spec_opt = ( ["remount"] + new_opt.split(",") ).join(",")
			@@initrdfs.cmd_mount("-o #{spec_opt} #{@dev} #{@point}")
			$log.debug "remount success: #{@dev} on #{@path} type #{@fstype} (from #{@opt} to #{new_opt})"
			@opt = new_opt
		rescue
			raise RemountError.new(@dev, @point, @fstype, @opt, new_opt, $!)
		end
	end

	def umount
		if !mounted?; raise NotMountedError.new("unmount"); end
		$log.debug0 "unmount #{@dev} from #{@point} type #{@fstype} (#{opt})"

		begin
			@@initrdfs.cmd_umount(@point)
			$log.debug "unmount success: #{@dev} from #{@point} type #{@fstype} (#{@opt})"
			@dev    = nil
			@point  = nil
			@fstype = nil
			@opt    = nil
		rescue
			raise UnmountError.new(@dev, @point, @fstype, @opt, $!)
		end
	end

	def to_s
		return @point.to_s
	end
end


class BindFS < VFS
	def mount(source_dir, path)
		# XXX: 未テストBindFS
		self.mountdev(source_dir, path, "bind", "bind")
	end
end


class LoopFS < VFS
	def self.setInitrdFS(initrdfs)
		@@initrdfs = initrdfs
	end

	def initialize
		@loopdev = nil
		super()
	end

	def mount(loopdev, image, path, fstype, opt)
		begin
			@@initrdfs.cmd_losetup(loopdev, image)
			@loopdev = loopdev
			$log.debug "losetup success: #{image} to #{@loopdev}"
		rescue
			LoopSetupError.new(loopdev, image, $!)
		end
		begin
			self.mountdev(@loopdev, path, fstype, opt)
		rescue
			ex = $!
			@@initrdfs.cmd_losetup_d(@loopdev) rescue $log.error $!
			raise ex
		end
	end

	def umount
		super
		begin
			@@initrdfs.cmd_losetup_d(@loopdev)
		rescue
			LoopDetachError.new(@loopdev, $!)
		end
		$log.debug "losetup -d success: #{@loopdev}"
	end
end


class VFSWrapper
	def initialize
		@basefs = nil
	end
	def mounted?
		if @basefs == nil; return false; end
		return @basefs.mounted?
	end
	def move(new_path)
		if @basefs == nil; raise NotMountedError.new("move"); end
		return @basefs.move(new_path)
	end
	def remount(new_opt)
		if @basefs == nil; raise NotMountedError.new("remount"); end
		return @basefs.mounted?
	end
	def umount
		if @basefs == nil; raise NotMountedError.new("unmount"); end
		return @basefs.umount
	end
	def point
		if @basefs == nil
			return nil
		else
			return @basefs.point
		end
	end
end


$log.debug "#{File.basename(__FILE__, ".*")} loaded"

