require "log"

class NetworkManager

	class NetInfo
		def initialize
			@name   = nil
			@method = nil
			@ipaddr = nil
			@mask   = nil
			@bcast  = nil
			@hwaddr = nil
		end
		def parse(netif, lines, method)
			@method = method
			@name = netif
			# TODO: 正規表現が甘い
			if (match = /#{netif}.+HWaddr ([0-9a-fA-F:\-]+)/.match(lines[0]))
				# FIXME: IP over FireWireはHWaddrの書式が違う。無視?
				# FIXME:  ifconfigの結果はLink ecap: で分岐が必要
				@hwaddr = match[1]
			end
			if (match = / +inet addr:([0-9\.]+)  Bcast:([0-9\.]+)  Mask:([0-9\.]+)/.match(lines[1]))
				@ipaddr = match[1]
				@bcast  = match[2]
				@mask   = match[3]
			elsif (match = / +inet addr:([0-9\.]+)  Mask:([0-9\.]+)/.match(lines[1]))
				# loにはBcastが無い
				@ipaddr = match[1]
				@mask   = match[2]
			end
		end
		attr_reader :name, :method, :ipaddr, :mask, :bcast, :hwaddr
	end

	class RouteInfo
		IPV4_OCT   = "[0-9]{1,3}"
		REGEX_IPV4_ADDR  = /#{IPV4_OCT}\.#{IPV4_OCT}\.#{IPV4_OCT}\.#{IPV4_OCT}/

		class RouteEntry
			def initialize(dest, gw, mask, metric, netif)
				@dest   = dest
				@gw     = gw
				@mask   = mask
				@metric = metric
				@netif  = netif
			end
			attr_reader :dest, :gw, :mask, :metric, :netif
		end

		def initialize
			@entries = Array.new
			@default = nil
		end
		attr_reader :entries, :default
		def parse(lines)
			# Destination     Gateway         Genmask         Flags Metric Ref    Use Iface
			# 192.168.0.0     0.0.0.0         255.255.255.0   U     10     0        0 eth0
			# 0.0.0.0         192.168.0.1     0.0.0.0         UG    10     0        0 eth0
			lines.each {|line|
				if (match = /(#{REGEX_IPV4_ADDR}) +(#{REGEX_IPV4_ADDR}) +(#{REGEX_IPV4_ADDR}) +([^ ]*) +([0-9]+) +([0-9]+) +([0-9]+) +(.+)/.match(line))
					@entries.push( RouteEntry.new(match[1], match[2], match[3], match[5], match[8]) )
					if match[1] == "0.0.0.0"
						@default = @entries.last
					end
				end
			}
		end

		def each
			return @entries.each {|e|
				yield e
			}
		end
	end


	def getNetInfo(netif, method)
		net = NetInfo.new
		line = @initrdfs.cmd_ifconfig("#{netif}")
		net.parse( netif, line.split("\n"), method )
		# TODO: フォーマットが美しくない
		$log.success "Network #{net.name}:  addr:#{net.ipaddr}  bcast:#{net.bcast}  mask:#{net.mask}  (#{net.method})"
		return net
	end
	private :getNetInfo

	def getRouteInfo
		route = RouteInfo.new
		line = @initrdfs.cmd_route("-n")
		route.parse( line.split("\n") )
		route.each {|r|
			# TODO: フォーマットが美しくない
			$log.success "Route #{r.netif}:  dest:#{r.dest}  gw:#{r.gw}  mask:#{r.mask}  metric:#{r.metric}"
		}
		return route
	end
	private :getRouteInfo


	def setupUser(netif, userspec)
		$log.info "Setting user-specified setting for #{netif} (#{userspec})"
		param_mask = (userspec.mask.empty? ? "" : " netmask #{userspec.mask}")
		# userspec.addrなどの判定はわざと甘い（ifconfigに判断を委ねる&例外で処理）
		@initrdfs.cmd_ifconfig("#{netif} #{userspec.addr}#{param_mask}")
	end
	private :setupUser

	def setupDHCP(netif)
		$log.info "Start DHCP client for #{netif}, waiting response..."
		# XXX: udhcpc: メッセージをリアルタイムで表示できない
		# XXX: bootstrap by DHCP 要精査
		selectbylease = $param.getValue("dhcpselect", nil)
		if selectbylease != nil && !selectbylease.empty?
			@initrdfs.cmd_udhcpc("-nfq -T 2 -t 60 -i #{netif} -s #{@initrdfs.dhcp_script} -L #{selectbylease}")
		else
			@initrdfs.cmd_udhcpc("-nfq -T 3 -t 5 -i #{netif} -s #{@initrdfs.dhcp_script}")
		end
	end
	private :setupDHCP

	def setupAutoIP(netif)
		$log.info "Start Auto-IP setting for #{netif}, waiting result..."
		# リンクアップ
		@initrdfs.cmd_ip("link set #{netif} up")
		# XXX: zcip: メッセージをリアルタイムで表示できない
		@initrdfs.cmd_zcip("-f -q #{netif} #{@initrdfs.zcip_script}")
	end
	private :setupAutoIP

	def setupStaticIP(netif, id)
		staticip = "192.168.#{id}.254"
		$log.info "Setting #{staticip} (fallback) for #{netif}"
		@initrdfs.cmd_ifconfig("#{netif} #{staticip} up")
	end
	private :setupStaticIP



	def wakeupInterface(netif, id, user)
		# リンクアップ
		begin
			@initrdfs.cmd_ip("link set #{netif} up")
		rescue
			$log.warn $!
		end
	
		# User
		if !user.empty?
			begin
				setupUser(netif)
				@@config_lock.synchronize {
					@configured.push( getNetInfo(netif, "user") )
					@not_configured.delete(netif)
				}
				return
			rescue
				$log.warn "User specified IP for #{netif} failed: #{$!}"
			end
		end

		# DHCP
		begin
			setupDHCP(netif)
			@config_lock.synchronize {
				@configured.push( getNetInfo(netif, "dhcp") )
				@not_configured.delete(netif)
			}
			return
		rescue
			$log.warn "DHCP for #{netif} failed: #{$!}"
		end

		# AutoIP
		begin
			setupAutoIP(netif)
			@config_lock.synchronize {
				@configured.push( getNetInfo(netif, "autoip") )
				@not_configured.delete(netif)
			}
			return
		rescue
			$log.warn "Auto-IP setting for #{netif} failed: #{$!}"
		end

		# Fallback
		begin
			setupStaticIP(netif, id)
			@config_lock.synchronize {
				@configured.push( getNetInfo(netif, "fallback") )
				@not_configured.delete(netif)
			}
			return
		rescue
			$log.error "Can't set IP address for #{netif}: #{$!}"
		end
	end
	private :wakeupInterface

end


$log.debug "#{File.basename(__FILE__, ".*")} loaded"

