#ifndef VFIELD_ARGPARSER_H__
#define VFIELD_ARGPARSER_H__

#include "exception.h"
#include <string>
#include <vector>
#include <boost/shared_ptr.hpp>
#include <boost/lexical_cast.hpp>

#include <iostream>

namespace VFIELD {


////
// ArgParser
//
class ArgumentKey;
class ArgParser {
public:
	ArgParser();
	~ArgParser();
public:
	void addKey(ArgumentKey* key) { m_keys.push_back(key); }
public:
	void parse(int argc, char* argv[]);
	void showUsage(void) const;
public:
	static const unsigned short USAGE_SPACE = 15;
private:
	typedef std::vector<ArgumentKey*>  keys_type;
	keys_type m_keys;
};


////
// Exception
//
struct ArgumentException : public std::runtime_error {
	ArgumentException(const std::string& message);
};

struct InsufficientArgumentException : public ArgumentException {
	InsufficientArgumentException(const std::string& arg);
};

struct UnknownArgumentException : public ArgumentException {
	UnknownArgumentException(const std::string& arg);
};

struct BadArgumentException : public ArgumentException {
	BadArgumentException(const std::string& message);
};



////
// ArgumentKey (base class)
//
class ArgumentKey {
public:
	ArgumentKey(const std::string& key, const std::string& comment);
	virtual ~ArgumentKey();
public:
	const std::string& getKey(void) const { return m_key; }
public:
	virtual void set(void);
	virtual void set(const std::string& str);
	virtual void eval(void) = 0;
	virtual void showUsage(void) const;
protected:
	const std::string m_key;
	const std::string m_comment;
private:
	ArgumentKey();
};


////
// ExistKey
//
template <typename Evaluator>
class CExistKey : public ArgumentKey {
public:
	CExistKey(const std::string& key, const std::string& comment, Evaluator eval) :
		ArgumentKey(key, comment), m_set(false), m_eval(eval) {}
	~CExistKey() {}
public:
	void set(void)  { m_set = true;  }
	void eval(void) { if(m_set) m_eval(); }
private:
	bool m_set;
	Evaluator m_eval;
private:
	CExistKey();
};
// helper function
template <typename Evaluator>
CExistKey<Evaluator>* ExistKey(const std::string& key, const std::string& comment, Evaluator eval)
{ return new CExistKey<Evaluator>(key, comment, eval); }


////
// StringKey
//
template <typename Evaluator>
class CStringKey : public ArgumentKey {
public:
	CStringKey(const std::string& key, const std::string& comment, Evaluator eval) :
		ArgumentKey(key, comment), m_string(NULL), m_eval(eval) {}
	~CStringKey() { delete m_string; }
public:
	void set(const std::string& str)
	{
		if(m_string) *m_string = str;
		else m_string = new std::string(str);
	}
	void eval(void) { if(m_string) m_eval(*m_string); }
private:
	std::string* m_string;
	Evaluator m_eval;
private:
	CStringKey();
};
// helper function
template <typename Evaluator>
CStringKey<Evaluator>* StringKey(const std::string& key, const std::string& comment, Evaluator eval)
{ return new CStringKey<Evaluator>(key, comment, eval); }



/////
// CopyOperator
//
// generic operator for StringKey
class CopyOperator {
public:
	CopyOperator(std::string& target) : m_target(target) {}
	void operator() (const std::string& source) { m_target = source; }
private:
	std::string& m_target;
	CopyOperator();
};


////
// NumericCopyOperator
//
// generic operator for StringKey
template <typename T>
class CNumericCopyOperator {
public:
	CNumericCopyOperator(T& target) : m_target(target) {}
	void operator() (const std::string& source)
	{
		try {
			m_target = boost::lexical_cast<T>(source);
		} catch(const boost::bad_lexical_cast& ex) {
			// FIXME: メッセージが微妙
			throw BadArgumentException(source + " must be a number");
		}
	}
private:
	T& m_target;
	CNumericCopyOperator();
};
// helper function
template <typename T>
CNumericCopyOperator<T> NumericCopyOperator(T& target) { return CNumericCopyOperator<T>(target); }


////
// LimitedNumericCopyOperator
//
// generic operator for StringKey
template <typename T>
class CLimitedNumericCopyOperator {
public:
	CLimitedNumericCopyOperator(T& target, T min, T max) : m_target(target), m_min(min), m_max(max) {}
	void operator() (const std::string& source)
	{
		try {
			m_target = boost::lexical_cast<T>(source);
		} catch(const boost::bad_lexical_cast& ex) {
			// FIXME: メッセージが微妙
			throw BadArgumentException(source + " must be a number");
		}
		if( m_target < m_min ) {
			std::ostringstream stream;
			stream << source << " must be larger than " << m_min;
			throw BadArgumentException(stream.str());
		} else if( m_target > m_max) {
			std::ostringstream stream;
			stream << source << " must be smaller than " << m_max;
			throw BadArgumentException(stream.str());
		}
	}
private:
	T& m_target;
	T  m_min;
	T  m_max;
	CLimitedNumericCopyOperator();
};
// helper function
template <typename T>
CLimitedNumericCopyOperator<T> LimitedNumericCopyOperator(T& target, T min, T max) { return CLimitedNumericCopyOperator<T>(target, min, max); }



////
// SetOperator
//
// generic operator for ExistKey
class SetOperator {
public:
	SetOperator(bool& target) : m_target(target) {}
	void operator() (void) { m_target = true; }
private:
	bool& m_target;
	SetOperator();
};



/////
// PathOperator
//
// generic operator for StringKey
class PathOperator {
public:
	PathOperator(std::string& target, bool require_absolute = false) : m_target(target), m_req_abs(require_absolute) {}
	void operator() (const std::string& source)
	{
		if( m_req_abs && source[0] != '/' ) {
			throw BadArgumentException(source + " must be absolute path");
		}
		m_target = source;
	}
private:
	std::string& m_target;
	bool m_req_abs;
	PathOperator();
};



}  // namespace VFIELD

#endif /* argparser.h */
