#include "if_nbd.h"
#include "interface/interface.h"
#include "nbd_processor.h"
#include "ip46.h"
#include "threadpool.h"
#include "exception.h"
#include "tools.h"
#include "log.h"
#include <boost/thread.hpp>
#include <boost/ptr_container/ptr_list.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>

namespace VFIELD {


class InterfaceNBDIMPL {
public:
	InterfaceNBDIMPL(Interface& vfif, uint16_t listen_port, ThreadPool& threadpool);
	~InterfaceNBDIMPL();
private:
	Interface& m_vfif;
	ThreadPool& m_threadpool;

	volatile bool m_end_flag;

	void ListenThread(int listen_sock);
	void stopListenThread(void);
	boost::scoped_ptr<boost::thread> m_listen_thread;
};


InterfaceNBD::InterfaceNBD(Interface& vfif, uint16_t listen_port, ThreadPool& threadpool) :
	impl(new InterfaceNBDIMPL(vfif, listen_port, threadpool)) {}
InterfaceNBDIMPL::InterfaceNBDIMPL(Interface& vfif, uint16_t listen_port, ThreadPool& threadpool) :
	m_vfif(vfif),
	m_threadpool(threadpool),
	m_end_flag(false)
{
	NodeIdentity listenon("127.0.0.1", boost::lexical_cast<std::string>(listen_port).c_str());
	int listen_sock = listenTCP4(listenon.getIPv4Address(), listenon.getPortHostByteOrder());
	if( listen_sock < 0 ) {
		throw InterfaceInitializeSystemCallException(errno, "Failed to start NBD interface");
								// XXX: ポート番号を出力
	}
	m_listen_thread.reset( new boost::thread(
				boost::bind(
					&InterfaceNBDIMPL::ListenThread,
					this,
					listen_sock
					)
				) );
	LogInfo( Log::format("NBD is listened on %1%:%2%") % addr46_t(listenon) % listenon.getPortHostByteOrder() );
	LogDebug( Log::format("NBD listen socket: %1%") % listen_sock );
}


InterfaceNBD::~InterfaceNBD() {}
InterfaceNBDIMPL::~InterfaceNBDIMPL()
{
	try {
		stopListenThread();
		LogInfo("NBD listen socket is closed");

	} catch (const std::runtime_error& ex) {
		LogError( Log::format("Can't stop NBD listen thread regularly: %1%") % ex.what() );
	} catch (...) {
		LogError("Can't stop NBD listen thread regularly");
	}
}


void InterfaceNBDIMPL::ListenThread(int listen_sock)
{
	addr46_t from;
	int proc_sock = -1;

	while( !m_end_flag ) {
		proc_sock = accept46(listen_sock, &from);
		if( proc_sock < 0 ) continue;

		boost::mutex proc_sock_mutex;

		// ネゴシエーション
		if( NBD::Negotiate(proc_sock, &proc_sock_mutex, m_vfif.getImageSize()) < 0 ) {
			LogWarn("NBD negotiation failed");
			::close(proc_sock);
			proc_sock = -1;
			continue;
		}
		LogInfo("NBD negotiation successed");
		LogDebug( Log::format("NBD client socket: %1%") % proc_sock );

		while(1) {
			struct nbd_request req;

			ssize_t rl = ::read(proc_sock, &req, sizeof(req));
			if( unlikely(rl < static_cast<ssize_t>(sizeof(req))) ) {
				if( rl < 0 ) {
					LogWarn("Receive broken NBD request: socket error");
					break;
				} else if( rl == 0 ) {
					LogWarn("NBD connection is unexpectedly closed");
					break;
				} else {
					// XXX: 制限回数以上このメッセージを表示したらソケットを閉じる
					LogWarn("Receive broken NBD request: received packet is too small");
					continue;
				}
			}

			uint32_t type = NBD::getType(req);

			if( likely(type == NBD_CMD_READ) ) {
				//NBD::ReadProcessor(proc_sock, &proc_sock_mutex, req, &m_vfif);
				m_threadpool.submitTask(
						boost::bind(
							&NBD::ReadProcessor,
							proc_sock,
							&proc_sock_mutex,
							req,
							&m_vfif
							)
						);

			} else if( unlikely(type == NBD_CMD_WRITE) ) {
				NBD::WriteProcessor(proc_sock, &proc_sock_mutex, req);

			} else if( unlikely(type == NBD_CMD_DISC) ) {
				NBD::DisconnectProcessor(proc_sock, &proc_sock_mutex, req);
				break;	// ソケットを閉じる
			}
		}
		::close(proc_sock);
		proc_sock = -1;

	}

	::close(listen_sock);
}


void InterfaceNBDIMPL::stopListenThread(void)
{
	LogDebug("Stopping NBD listen thread");
	m_end_flag = true;
	// XXX: 未実装
	// m_listen_thread->join();
	LogInfo("NBD interface is stoppped");
}


}  // namespace VFIELD
