/*
    Video maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "command.h"
#include "file.h"
#include "general.h"
#include "jump.h"
#include "prop.h"
#include "sigfile.h"
#include "size.h"
#include "thread.h"
#if GTK_CHECK_VERSION(2,4,0)
# include "avicore/avifileextra.h"
#else /* not GTK_CHECK_VERSION(2,4,0) */
# include "avicore/avifilesel.h"
#endif /* not GTK_CHECK_VERSION(2,4,0) */
#include "misc/fileio.h"
#include "misc/misc.h"
#include "misc/misc_conf.h"
#include "misc/misc_history.h"
#include "misc/misc_mdi.h"
#ifdef G_OS_WIN32
# include <gdk/gdkwin32.h>
#endif /* G_OS_WIN32 */


/******************************************************************************
*                                                                             *
* ja:メニュー関数群                                                           *
*                                                                             *
******************************************************************************/
void
command_new (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
#ifdef USE_THREAD
  g_thread_create ((GThreadFunc)file_open_edit, NULL, TRUE, NULL);
#else /* not USE_THREAD */
  file_open_edit (NULL);
#endif /* not USE_THREAD */
}


typedef struct _CommandOpenOk CommandOpenOk;
struct _CommandOpenOk
{
  gboolean merge;
  gchar *file;
  AviSave *avi_save;
  GtkWidget *dialog;
};


#if ! GTK_CHECK_VERSION(2,4,0)
static void
command_open_ok (GtkWidget     *widget,
                 CommandOpenOk *cmd_ok)
{
  const gchar *file;

  file = gtk_file_selection_get_filename (GTK_FILE_SELECTION (cmd_ok->dialog));
  cmd_ok->file = file && file[0] != '\0'
                && file[g_strlen (file) - 1] != '/' ? g_strdup (file) : NULL;
  cmd_ok->merge = avi_file_selection_get_continual
                                        (AVI_FILE_SELECTION (cmd_ok->dialog));
  cmd_ok->avi_save = avi_file_selection_get_option
                                        (AVI_FILE_SELECTION (cmd_ok->dialog));

  gtk_widget_destroy (cmd_ok->dialog);
}
#endif /* not GTK_CHECK_VERSION(2,4,0) */


void
command_open (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *path;
  CommandOpenOk cmd_ok;
  FileOpen *file_open;
#if GTK_CHECK_VERSION(2,4,0)
  GtkWidget *check;
#endif /* GTK_CHECK_VERSION(2,4,0) */

  cmd_ok.file = NULL;
#if GTK_CHECK_VERSION(2,4,0)
  cmd_ok.dialog = gtk_file_chooser_dialog_new (_("Open"),
                            GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_OPEN,
                            GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            NULL);
  if (open_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (cmd_ok.dialog),
                                                                    open_path);
  check = gtk_check_button_new_with_label (_("Continual"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check), file_merge);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (cmd_ok.dialog), check);
  if (gtk_dialog_run (GTK_DIALOG (cmd_ok.dialog)) == GTK_RESPONSE_ACCEPT)
    {
      cmd_ok.file = gtk_file_chooser_get_filename
                                            (GTK_FILE_CHOOSER (cmd_ok.dialog));
      cmd_ok.merge = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
    }
  gtk_widget_destroy (cmd_ok.dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  cmd_ok.dialog = avi_file_selection_new (_("Open"));
  g_signal_connect (G_OBJECT (cmd_ok.dialog), "destroy", gtk_main_quit, NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), &cmd_ok);
  g_signal_connect_swapped
                (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->cancel_button),
                                    "clicked", G_CALLBACK (gtk_widget_destroy),
                                                    G_OBJECT (cmd_ok.dialog));
  if (open_path)
    {
      path = g_strconcat (open_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename
                                    (GTK_FILE_SELECTION (cmd_ok.dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (cmd_ok.dialog));
  avi_file_selection_set_continual (AVI_FILE_SELECTION (cmd_ok.dialog),
                                    file_merge);
  avi_file_selection_show_continual (AVI_FILE_SELECTION (cmd_ok.dialog));
  gtk_widget_show (cmd_ok.dialog);
  gtk_grab_add (cmd_ok.dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (cmd_ok.file)
    {
      g_free (open_path);
      path = fileio_get_full_path (cmd_ok.file);
      open_path = g_path_get_dirname (path);
      g_free (path);
      file_open = g_malloc (sizeof (FileOpen));
      file_open->file = cmd_ok.file;
      file_open->merge = file_merge = cmd_ok.merge;
#ifdef USE_THREAD
      g_thread_create ((GThreadFunc)file_open_edit, file_open, TRUE, NULL);
#else /* not USE_THREAD */
      file_open_edit (file_open);
#endif /* not USE_THREAD */
    }
}


void
command_close (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
  gint page;

  if ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0
    && prompt_close ((VmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), page)))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
}


void
command_save (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gchar *path;
  gint i = 0;
  AviEdit *avi_edit[3];
  CommandOpenOk cmd_ok;
  FileSave *file_save;
  VmaidWindow *vmaid;
#if GTK_CHECK_VERSION(2,4,0)
  GtkWidget *extra;
#endif /* GTK_CHECK_VERSION(2,4,0) */

  vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  if (vmaid->avi_edit[0])
    avi_edit[i++] = vmaid->avi_edit[0];
  if (vmaid->avi_edit[1])
    avi_edit[i++] = vmaid->avi_edit[1];
  avi_edit[i] = NULL;
  cmd_ok.file = NULL;
  cmd_ok.avi_save = NULL;
#if GTK_CHECK_VERSION(2,4,0)
  cmd_ok.dialog = gtk_file_chooser_dialog_new (_("Save As"),
                            GTK_WINDOW (window), GTK_FILE_CHOOSER_ACTION_SAVE,
                            GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
                            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                            NULL);
  if (save_path)
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (cmd_ok.dialog),
                                                                    save_path);
  extra = avi_file_extra_new ();
  avi_file_extra_set_handle (AVI_FILE_EXTRA (extra), avi_edit);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (cmd_ok.dialog), extra);
  if (gtk_dialog_run (GTK_DIALOG (cmd_ok.dialog)) == GTK_RESPONSE_ACCEPT)
    {
      cmd_ok.file = gtk_file_chooser_get_filename
                                            (GTK_FILE_CHOOSER (cmd_ok.dialog));
      cmd_ok.avi_save = avi_file_extra_get_option (AVI_FILE_EXTRA (extra));
    }
  gtk_widget_destroy (cmd_ok.dialog);
#else /* not GTK_CHECK_VERSION(2,4,0) */
  cmd_ok.dialog = avi_file_selection_new (_("Save As"));
  g_signal_connect (G_OBJECT (cmd_ok.dialog), "destroy", gtk_main_quit, NULL);
  g_signal_connect (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->ok_button),
                            "clicked", G_CALLBACK (command_open_ok), &cmd_ok);
  g_signal_connect_swapped
                (G_OBJECT (GTK_FILE_SELECTION (cmd_ok.dialog)->cancel_button),
                                    "clicked", G_CALLBACK (gtk_widget_destroy),
                                                    G_OBJECT (cmd_ok.dialog));
  if (save_path)
    {
      path = g_strconcat (save_path, G_SEARCHPATH_SEPARATOR_S, NULL);
      gtk_file_selection_set_filename
                                    (GTK_FILE_SELECTION (cmd_ok.dialog), path);
      g_free (path);
    }
  gtk_file_selection_hide_fileop_buttons (GTK_FILE_SELECTION (cmd_ok.dialog));
  avi_file_selection_set_handle (AVI_FILE_SELECTION (cmd_ok.dialog), avi_edit);
  gtk_widget_show (cmd_ok.dialog);
  gtk_grab_add (cmd_ok.dialog);
  gtk_main ();
#endif /* not GTK_CHECK_VERSION(2,4,0) */
  if (cmd_ok.file && cmd_ok.avi_save)
    {
      g_free (save_path);
      path = fileio_get_full_path (cmd_ok.file);
      save_path = g_path_get_dirname (path);
      g_free (path);
      if (avi_save_is_all (avi_edit, cmd_ok.avi_save))
        {
          /* ja:すべて保存されるとき */
          misc_mdi_set_edited (MISC_MDI (mdi),
                misc_mdi_get_page_from_data (MISC_MDI (mdi), vmaid), FALSE);
          delete_list (&vmaid->undo);
          delete_list (&vmaid->redo);
          set_menu_bar (vmaid);
          misc_history_add_file (MISC_HISTORY (history), cmd_ok.file);
        }
      file_save = g_malloc (sizeof (FileSave));
      file_save->file = cmd_ok.file;
      for (i = 0; avi_edit[i]; i++)
        file_save->avi_edit[i] = avi_clone (avi_edit[i]);
      file_save->avi_edit[i] = NULL;
      file_save->avi_save = cmd_ok.avi_save;
      /* ja:保存 */
#ifdef USE_THREAD
      g_thread_create ((GThreadFunc)file_save_edit, file_save, TRUE, NULL);
#else /* not USE_THREAD */
      file_save_edit (file_save);
#endif /* not USE_THREAD */
    }
  else
    {
      g_free (cmd_ok.file);
      avi_save_free (cmd_ok.avi_save);
    }
}


void
command_property (gpointer   callback_data,
                  guint      callback_action,
                  GtkWidget *widget)
{
  VmaidWindow *vmaid;
  PropertyDialog prop;

  vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  g_memset (&prop, 0, sizeof (PropertyDialog));
  prop.rate = vmaid->rate;
  prop.scale = vmaid->scale;
  if (vmaid->avi_edit[0])
    {
      prop.priority0 = avi_get_priority (vmaid->avi_edit[0]);
      prop.language0 = avi_get_language (vmaid->avi_edit[0]);
      if (avi_get_name (vmaid->avi_edit[0]))
        prop.name0 = g_strdup (avi_get_name (vmaid->avi_edit[0]));
      prop.width = avi_get_width (vmaid->avi_edit[0]);
      prop.height = avi_get_height (vmaid->avi_edit[0]);
      prop.bit_count = avi_get_bit_count (vmaid->avi_edit[0]);
      prop.x_pixels_per_meter
                            = avi_get_x_pixels_per_meter (vmaid->avi_edit[0]);
      prop.y_pixels_per_meter
                            = avi_get_y_pixels_per_meter (vmaid->avi_edit[0]);
    }
  else
    {
      prop.x_pixels_per_meter = -1;
      prop.y_pixels_per_meter = -1;
    }
  if (vmaid->avi_edit[1])
    {
      prop.priority1 = avi_get_priority (vmaid->avi_edit[1]);
      prop.language1 = avi_get_language (vmaid->avi_edit[1]);
      if (avi_get_name (vmaid->avi_edit[1]))
        prop.name1 = g_strdup (avi_get_name (vmaid->avi_edit[1]));
      prop.samples_per_sec = avi_get_samples_per_sec (vmaid->avi_edit[1]);
      prop.channels = avi_get_channels (vmaid->avi_edit[1]);
      prop.bits_per_sample = avi_get_bits_per_sample (vmaid->avi_edit[1]);
    }
  if (property_dialog(&prop))
    {
      vmaid->rate = prop.rate;
      vmaid->scale = prop.scale;
      if (vmaid->avi_edit[0])
        {
          avi_set_priority (vmaid->avi_edit[0], prop.priority0);
          avi_set_language (vmaid->avi_edit[0], prop.language0);
          avi_set_name (vmaid->avi_edit[0], prop.name0);
          avi_set_rate (vmaid->avi_edit[0], prop.rate);
          avi_set_scale (vmaid->avi_edit[0], prop.scale);
          avi_set_width (vmaid->avi_edit[0], prop.width);
          avi_set_height (vmaid->avi_edit[0], prop.height);
          avi_set_bit_count (vmaid->avi_edit[0], prop.bit_count);
          avi_set_x_pixels_per_meter (vmaid->avi_edit[0],
                                                    prop.x_pixels_per_meter);
          avi_set_y_pixels_per_meter (vmaid->avi_edit[0],
                                                    prop.y_pixels_per_meter);
          g_free (prop.name0);
        }
      if (vmaid->avi_edit[1])
        {
          avi_set_priority (vmaid->avi_edit[1], prop.priority1);
          avi_set_language (vmaid->avi_edit[1], prop.language1);
          avi_set_name (vmaid->avi_edit[1], prop.name1);
          avi_set_samples_per_sec (vmaid->avi_edit[1], prop.samples_per_sec);
          avi_set_channels (vmaid->avi_edit[1], prop.channels);
          avi_set_bits_per_sample (vmaid->avi_edit[1], prop.bits_per_sample);
          g_free (prop.name1);
        }
      vmaid->cursor.stream = vmaid->avi_edit[0] ? 0 : 1;
      vmaid->cursor.frame = 0;
      vmaid->select.stream = -1;
      vmaid->select.frame = -1;
      vmaid->top = 0;
      delete_list (&vmaid->undo);
      delete_list (&vmaid->redo);
      set_menu_bar (vmaid);
      gtk_widget_draw (vmaid->drawing, NULL);
    }
}


void
command_exit (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GdkEvent event;

  event.any.type = GDK_DELETE;
  event.any.window = window->window;
  event.any.send_event = FALSE;
  gdk_event_put (&event);
}


void
command_undo (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  history_operation ((VmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1),
                                                                        FALSE);
}


void
command_redo (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  history_operation ((VmaidWindow *)misc_mdi_get_data (MISC_MDI (mdi), -1),
                                                                        TRUE);
}


void
command_cut (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  command_copy (callback_data, callback_action, widget);
  command_delete (callback_data, callback_action, widget);
}


void
command_copy (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gint i = 0, start, end, length, max;
  AviEdit *avi_edit[3];
  VmaidWindow *vmaid;

#ifndef G_OS_WIN32
  if (gtk_selection_owner_set (window, GDK_SELECTION_CLIPBOARD,
                                                            GDK_CURRENT_TIME))
    {
#endif /* not G_OS_WIN32 */
      vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
      /* ja:ビデオ */
      if (MIN (vmaid->cursor.stream, vmaid->select.stream) == 0)
        {
          max = avi_length (vmaid->avi_edit[0]);
          start = MAX (MIN (vmaid->cursor.frame, vmaid->select.frame), 0);
          end = MIN (MAX (vmaid->cursor.frame, vmaid->select.frame), max);
          length = end - start;
          if (length > 0 && (avi_edit[i]
                            = avi_copy (vmaid->avi_edit[0], start, length)))
            i++;
        }
      /* ja:オーディオ */
      if (MAX (vmaid->cursor.stream, vmaid->select.stream) == 1)
        {
          max = avi_length (vmaid->avi_edit[1]);
#ifdef G_HAVE_GINT64
          start = MAX (avi_time_to_sample (vmaid->avi_edit[1],
                (gint64)MIN (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                            * vmaid->scale / vmaid->rate), 0);
          end = MIN (avi_time_to_sample (vmaid->avi_edit[1],
                (gint64)MAX (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                        * vmaid->scale / vmaid->rate), max);
#else /* not G_HAVE_GINT64 */
          start = MAX (avi_time_to_sample (vmaid->avi_edit[1],
                        MIN (vmaid->cursor. frame, vmaid->select. frame) * 1000
                                            * vmaid->scale / vmaid->rate), 0);
          end = MIN (avi_time_to_sample (vmaid->avi_edit[1],
                        MAX (vmaid->cursor. frame, vmaid->select. frame) * 1000
                                        * vmaid->scale / vmaid->rate), max);
#endif /* not G_HAVE_GINT64 */
          length = end - start;
          if (length > 0 && (avi_edit[i]
                            = avi_copy (vmaid->avi_edit[1], start, length)))
            i++;
        }
      if (i > 0)
        {
          g_free (clipboard_scenario);
          avi_edit[i] = NULL;
          clipboard_scenario = avi_to_scenario (avi_edit, FALSE);
          for (i = 0; avi_edit[i]; i++)
            avi_release (avi_edit[i]);
#ifdef G_OS_WIN32
          if (clipboard_scenario)
            {
              HGLOBAL hGlobal;
              LPSTR lpScenario;

              hGlobal = GlobalAlloc (GMEM_MOVEABLE,
                        (g_strlen (clipboard_scenario) + 1) * sizeof (gchar));
              if (hGlobal)
                {
                  lpScenario = GlobalLock (hGlobal);
                  if (lpScenario)
                    {
                      g_strcpy (lpScenario, clipboard_scenario);
                      GlobalUnlock (hGlobal);
                      if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
                        {
                          if (!EmptyClipboard ()
                                    || !SetClipboardData (uFormat, hGlobal))
                            GlobalFree (hGlobal);
                          CloseClipboard ();
                        }
                      else
                        {
                          GlobalFree (hGlobal);
                        }
                    }
                  else
                    {
                      GlobalFree (hGlobal);
                    }
                }
            }
#endif /* G_OS_WIN32 */
        }
#ifndef G_OS_WIN32
    }
#endif /* not G_OS_WIN32 */
}


void
command_paste (gpointer   callback_data,
               guint      callback_action,
               GtkWidget *widget)
{
#ifdef G_OS_WIN32
  gchar *scenario = NULL;
  HGLOBAL hGlobal;
  LPSTR lpScenario;

  if (OpenClipboard (GDK_WINDOW_HWND (window->window)))
    {
      if (hGlobal = GetClipboardData (uFormat))
        {
          if (lpScenario = GlobalLock (hGlobal))
            scenario = g_strdup (lpScenario);
          GlobalUnlock (hGlobal);
        }
      CloseClipboard ();
      if (scenario)
        {
          GtkSelectionData data;

          data.selection = GDK_SELECTION_CLIPBOARD;
          data.target = atom_scenario;
          data.type = atom_scenario;
          data.format = 8;
          data.data = scenario;
          data.length = g_strlen (scenario);
          signal_selection_received (window, &data, GDK_CURRENT_TIME, NULL);
          g_free (scenario);
        }
    }
#else /* not G_OS_WIN32 */
  gtk_selection_convert (window, GDK_SELECTION_CLIPBOARD, atom_targets,
                                                            GDK_CURRENT_TIME);
#endif /* not G_OS_WIN32 */
}


void
command_delete (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  gboolean result = TRUE;
  gint i, start, end, length, max, sx;
  VmaidHistory *d;
  VmaidWindow *vmaid;

  vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  /* ja:UNDOのための処理 */
  d = g_malloc (sizeof (VmaidHistory));
  for (i = 0; i < 2; i++)
    {
      d->avi_edit[i] = NULL;
      if (vmaid->avi_edit[i]
                        && !(d->avi_edit[i] = avi_clone (vmaid->avi_edit[i])))
        result = FALSE;
    }
  if (!result)
    {
      misc_message_box ("Video maid", _("AVI Clone Error"), 0, _("OK"), NULL);
      for (i = 0; i < 2; i++)
        if (d->avi_edit[i])
          avi_release (d->avi_edit[i]);
      g_free (d);
      return;
    }
  d->cursor = vmaid->cursor;
  d->select = vmaid->select;
  /* ja:ビデオ */
  if (MIN (vmaid->cursor.stream, vmaid->select.stream) == 0)
    {
      max = avi_length (vmaid->avi_edit[0]);
      start = MAX (MIN (vmaid->cursor.frame, vmaid->select.frame), 0);
      end = MIN (MAX (vmaid->cursor.frame, vmaid->select.frame), max);
      if (start == 0 && end == max)
        {
          if (!avi_release (vmaid->avi_edit[0]))
            result = FALSE;
          vmaid->avi_edit[0] = NULL;
        }
      else
        {
          length = end - start;
          if (length > 0 && !avi_delete(vmaid->avi_edit[0], start, length))
            result=FALSE;
        }
    }
  /* ja:オーディオ */
  if (MAX (vmaid->cursor.stream, vmaid->select.stream) == 1)
    {
      max = avi_length (vmaid->avi_edit[1]);
#ifdef G_HAVE_GINT64
      start = MAX (avi_time_to_sample (vmaid->avi_edit[1],
                (gint64)MIN (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                            * vmaid->scale / vmaid->rate), 0);
      end = MIN (avi_time_to_sample (vmaid->avi_edit[1],
                (gint64)MAX (vmaid->cursor.frame, vmaid->select.frame) * 1000
                                        * vmaid->scale / vmaid->rate), max);
#else /* not G_HAVE_GINT64 */
      start = MAX (avi_time_to_sample (vmaid->avi_edit[1],
                        MIN (vmaid->cursor. frame, vmaid->select. frame) * 1000
                                            * vmaid->scale / vmaid->rate), 0);
      end = MIN (avi_time_to_sample (vmaid->avi_edit[1],
                        MAX (vmaid->cursor. frame, vmaid->select. frame) * 1000
                                        * vmaid->scale / vmaid->rate), max);
#endif /* not G_HAVE_GINT64 */
      if (start == 0 && end == max)
        {
          if (!avi_release (vmaid->avi_edit[1]))
            result=FALSE;
          vmaid->avi_edit[1] = NULL;
        }
      else
        {
          length = end - start;
          if (length > 0 && !avi_delete (vmaid->avi_edit[1], start, length))
            result = FALSE;
        }
    }
  if (!result)
    {
      misc_message_box ("Video maid", _("AVI Delete Error"), 0, _("OK"), NULL);
      for (i = 0; i < 2; i++)
        {
          if (vmaid->avi_edit[i])
            avi_release (vmaid->avi_edit[i]);
          vmaid->avi_edit[i] = d->avi_edit[i];
        }
      g_free (d);
      return;
    }
  d->next = vmaid->undo;
  vmaid->undo = d;
  delete_list (&vmaid->redo);
  max = get_max_frame (vmaid, -1) + 1;
  sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
  vmaid->cursor.frame = (vmaid->avi_edit[0] || vmaid->avi_edit[1])
                        ? MIN(vmaid->cursor.frame, vmaid->select.frame) : 0;
  vmaid->cursor.stream = !vmaid->avi_edit[0] && !vmaid->avi_edit[1]
                                            ? -1 : vmaid->avi_edit[0] ? 0 : 1;
  vmaid->select.stream = -1;
  if (vmaid->cursor.frame > max)
    vmaid->cursor.frame = max;
  if (vmaid->top > vmaid->cursor.frame)
    vmaid->top = vmaid->cursor.frame;
  else if (vmaid->top < vmaid->cursor.frame - sx)
    vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
  /* ja:メニュー,ステートメント,子ウインドウ */
  set_menu_bar (vmaid);
  gtk_widget_set_usize (vmaid->drawing, vmaid->width * 3 + 4,
            system_font_height * 2
            + ((vmaid->avi_edit[0] ? 1 : 0) + (vmaid->avi_edit[1] ? 1 : 0))
                                                            * vmaid->height);
  gtk_widget_draw (vmaid->drawing, NULL);
  draw_caret (vmaid, NULL);
  misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
  misc_mdi_set_edited (MISC_MDI (mdi),
                    misc_mdi_get_page_from_data (MISC_MDI (mdi), vmaid), TRUE);
}


void
command_jump (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  gint sx, top;
  JumpDialog jump;
  VmaidCursor cursor, select;
  VmaidWindow *vmaid;

  vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  cursor = vmaid->cursor;
  select = vmaid->select;
  top = vmaid->top;
  jump.frame = vmaid->cursor.frame;
  jump.max = get_max_frame (vmaid, -1) + 1;
  jump.rate = vmaid->rate;
  jump.scale = vmaid->scale;
  if (jump_dialog (&jump))
    {
      vmaid->cursor.frame = jump.frame;
      sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
      /* ja:キャレットが表示されるように調整 */
      if (vmaid->top > vmaid->cursor.frame)
        vmaid->top = vmaid->cursor.frame;
      else if (vmaid->top < vmaid->cursor.frame - sx)
        vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
      /* ja:ジャンプの例外 ジャンプ先のフレームを表示する */
      if (vmaid->cursor.frame < jump.max
                                    && vmaid->top + sx <= vmaid->cursor.frame)
        vmaid->top = vmaid->cursor.frame - sx + 1;
      if (vmaid->cursor.frame == vmaid->select.frame)
        {
          /* ja:選択を解除する */
          vmaid->select.stream = -1;
          set_menu_bar (vmaid);
        }
      if (vmaid->top != top)
        misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                        vmaid, 0, jump. max, sx, vmaid->top);
      clear_sel (vmaid, &cursor, &select, top);
      draw_caret (vmaid, &cursor);
    }
}


void
command_all (gpointer   callback_data,
             guint      callback_action,
             GtkWidget *widget)
{
  gint max, sx, top;
  VmaidCursor cursor, select;
  VmaidWindow *vmaid;

  vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  if (vmaid->avi_edit[0] || vmaid->avi_edit[1])
    {
      cursor = vmaid->cursor;
      select = vmaid->select;
      top = vmaid->top;
      sx = MAX ((vmaid->drawing->allocation.width - 4) / vmaid->width - 2, 1);
      max = get_max_frame (vmaid, -1) + 1;
      vmaid->select.frame = 0;
      vmaid->cursor.frame = max;
      vmaid->select.stream = vmaid->avi_edit[0] ? 0 : 1;
      vmaid->cursor.stream = vmaid->avi_edit[1] ? 1 : 0;
      /* ja:キャレットが表示されるように調整 */
      if (vmaid->top > vmaid->cursor.frame)
        vmaid->top = vmaid->cursor.frame;
      else if (vmaid->top < vmaid->cursor.frame - sx)
        vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
      if (select.stream< 0 )
        set_menu_bar (vmaid);
      if (vmaid->top != top)
        misc_set_scroll_bar (vmaid->hscroll,
                                    G_CALLBACK (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
      clear_sel (vmaid, &cursor, &select, top);
      draw_caret (vmaid, &cursor);
    }
}


void
command_size (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  SizeDialog size;
  VmaidWindow *vmaid;

  vmaid = misc_mdi_get_data (MISC_MDI (mdi), -1);
  size.view = default_view;
  size.ratio = default_ratio;
  size.width = vmaid->width;
  size.height = vmaid->height;
  if (vmaid->avi_edit[0])
    {
      size.def_width = avi_get_width (vmaid->avi_edit[0]);
      size.def_height  =avi_get_height (vmaid->avi_edit[0]);
    }
  else
    {
      size.def_width = 0;
      size.def_height = 0;
    }
  if (size_dialog (&size))
    {
      default_view = size.view;
      default_ratio = size.ratio;
      vmaid->width = size.width;
      vmaid->height = size.height;
      gtk_widget_set_usize (vmaid->drawing, vmaid->width * 3 + 4,
                        system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height);
      gtk_widget_queue_resize (GTK_WIDGET (window));
      gtk_widget_draw (vmaid->drawing, NULL);
    }
}


static void
command_conf_ok (GtkWidget *widget,
                 GtkWidget *dialog)
{
  misc_history_set_num (MISC_HISTORY (history),
                                misc_conf_get_history (MISC_CONF (dialog)));
  newfile = misc_conf_get_newfile (MISC_CONF (dialog));
  second = misc_conf_get_second (MISC_CONF (dialog));
  misc_conf_get_size (MISC_CONF (dialog), &def_width, &def_height);
  n_pos = misc_conf_get_tab (MISC_CONF (dialog));
  gtk_widget_destroy (dialog);
  /* ja:ノートの位置 */
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (mdi), n_pos);
}


void
command_conf (gpointer   callback_data,
              guint      callback_action,
              GtkWidget *widget)
{
  GtkWidget *dialog;

  dialog = misc_conf_new ();
  g_signal_connect (G_OBJECT (dialog), "destroy",
                                            G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (G_OBJECT (MISC_CONF (dialog)->ok_button),
                            "clicked", G_CALLBACK (command_conf_ok), dialog);
  g_signal_connect_swapped (G_OBJECT (MISC_CONF (dialog)->cancel_button),
            "clicked", G_CALLBACK (gtk_widget_destroy), GTK_OBJECT (dialog));
  misc_conf_set_history (MISC_CONF(dialog),
                                misc_history_get_num (MISC_HISTORY (history)));
  misc_conf_set_newfile (MISC_CONF(dialog), newfile);
  misc_conf_set_second (MISC_CONF(dialog), second);
  misc_conf_set_size (MISC_CONF(dialog), def_width, def_height);
  misc_conf_set_tab (MISC_CONF(dialog), n_pos);
  gtk_widget_show (dialog);
  gtk_grab_add (dialog);
  gtk_main ();
}


void
command_cancel (gpointer   callback_data,
                guint      callback_action,
                GtkWidget *widget)
{
  thread_break (FALSE);
}
