/*
    Video maid
    copyright (c) 1998-2005 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "command.h"
#include "file.h"
#include "general.h"
#include "sigfile.h"
#include "sigmain.h"
#include "thread.h"
#include "misc/argument.h"
#include "misc/misc.h"
#include "misc/misc_history.h"
#include "misc/misc_mdi.h"


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(テキスト)                                        *
*                                                                             *
******************************************************************************/
void
signal_style_set (GtkWidget *widget,
                  GtkStyle  *style,
                  gpointer   user_data)
{
  gint i, ascent, height;

  /* ja:システム色 */
  style = gtk_widget_get_style (widget);
  system_color[0] = style->text[0];
  system_color[1] = style->base[0];
  system_color[2] = style->fg[3];
  system_color[3] = style->bg[3];
  if (system_font)
    gdk_font_unref (system_font);
  /* ja:システムフォント */
  system_font = gdk_font_ref (gtk_style_get_font (style));
  for (i = 0; i < 256; i++)
    {
      gdk_text_extents (system_font, (gchar *)&i, 1,
                                            NULL, NULL, NULL, &ascent, NULL);
      if (system_font_ascent < ascent)
        system_font_ascent = ascent;
      height = gdk_char_height (system_font, (gchar)i);
      if (system_font_height < height)
        system_font_height = height;
    }
  for (i = misc_mdi_get_n_pages (MISC_MDI (mdi)) - 1; i >= 0; i--)
    {
      VmaidWindow *vmaid;

      vmaid = misc_mdi_get_data (MISC_MDI (mdi), i);
      gtk_widget_set_usize (vmaid->drawing, vmaid->width * 3 + 4,
                        system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height);
      gtk_widget_queue_resize (GTK_WIDGET (window));
      gtk_widget_draw (vmaid->drawing, NULL);
    }
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(メニュー)                                        *
*                                                                             *
******************************************************************************/
void
signal_activate_menu_history (GtkWidget   *widget,
                              const gchar *file)
{
  FileOpen *file_open;

  file_open = g_malloc (sizeof (FileOpen));
  file_open->file = g_strdup (file);
  file_open->merge = file_merge;
#ifdef USE_THREAD
  g_thread_create ((GThreadFunc)file_open_edit, file_open, TRUE, NULL);
#else /* not USE_THREAD */
  file_open_edit (file_open);
#endif /* not USE_THREAD */
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(MDI)                                             *
*                                                                             *
******************************************************************************/
void
signal_close_window (GtkWidget   *widget,
                     guint        page_num,
                     VmaidWindow *vmaid)
{
  if (prompt_close (vmaid))
    gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page_num);
}


void
signal_switch_window (GtkWidget   *widget,
                      guint        page_num,
                      VmaidWindow *vmaid)
{
  /* ja:メニュー表示 */
  set_menu_bar (vmaid);
}


void
signal_destroy_mdi (GtkWidget *widget,
                    gpointer   user_data)
{
  gtk_timeout_remove (timer_id);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(リスト)                                          *
*                                                                             *
******************************************************************************/
static void
signal_tree_selection_foreach (GtkTreeModel *model,
                               GtkTreePath  *path,
                               GtkTreeIter  *iter,
                               gpointer      user_data)
{
  (*(gint *)user_data)++;
}


void
signal_changed (GtkTreeSelection *select,
                gpointer          user_data)
{
  gint count;

  gtk_tree_selection_selected_foreach (select, signal_tree_selection_foreach,
                                                                    &count);
  gtk_widget_set_sensitive (gtk_menu_get_attach_widget (GTK_MENU
                (gtk_item_factory_get_widget (ifactory_menu, "<main>/Task"))),
                                                                    count > 0);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(セレクション)                                    *
*                                                                             *
******************************************************************************/
void
signal_drag_data_received (GtkWidget        *widget,
                           GdkDragContext   *context,
                           gint              x,
                           gint              y,
                           GtkSelectionData *selection_data,
                           guint             info,
                           guint             time)
{
  gchar *file, **files;
  gint i;
  FileOpen *file_open;

  files = g_strsplit (selection_data->data, "\r\n", G_MAXINT);
  for (i = 0; files[i]; i++)
    {
      file = g_filename_from_uri (files[i], NULL, NULL);
      if (file)
        {
          file_open = g_malloc (sizeof (FileOpen));
          file_open->file = file;
          file_open->merge = file_merge;
#ifdef USE_THREAD
          g_thread_create ((GThreadFunc)file_open_edit, file_open, TRUE, NULL);
#else /* not USE_THREAD */
          file_open_edit (file_open);
#endif /* not USE_THREAD */
        }
    }
  g_strfreev (files);
}


gboolean
signal_selection_clear (GtkWidget         *widget,
                        GdkEventSelection *event,
                        gpointer           user_data)
{
  if (event->selection == GDK_SELECTION_CLIPBOARD)
    {
      /* ja:クリップボード */
      g_free (clipboard_scenario);
      clipboard_scenario = NULL;
    }
  return TRUE;
}


void
signal_selection_get (GtkWidget        *widget,
                      GtkSelectionData *data,
                      guint             info,
                      guint             time,
                      gpointer          user_data)
{
  /* ja:クリップボード */
  if (data->selection == GDK_SELECTION_CLIPBOARD
                                && clipboard_scenario && info == TARGET_VMAID)
    gtk_selection_data_set (data, atom_scenario, 8,
                (guchar *)clipboard_scenario, g_strlen (clipboard_scenario));
}


void
signal_selection_received (GtkWidget        *widget,
                           GtkSelectionData *data,
                           guint             time,
                           gpointer          user_data)
{

  if (data->selection == GDK_SELECTION_CLIPBOARD && data->length >= 0)
    {
      /* ja:クリップボード */
      gint i;

      if (data->type == GDK_SELECTION_TYPE_ATOM)
        {
          gint count;
          GdkAtom *atoms;

          atoms = (GdkAtom *)data->data;
          count = data->length / sizeof (GdkAtom);
          for (i = 0; i < count; i++)
            if (atoms[i] == atom_scenario)
              break;
          if (i < count)
            gtk_selection_convert (widget, data->selection,atoms[i],
                                                            GDK_CURRENT_TIME);
        }
      else if (data->type == atom_scenario)
        {
          gboolean result = TRUE, paste[2];
          gint max, sx, page_num;
          AviEdit **avi_edit;
          VmaidWindow *vmaid;
          VmaidHistory *d;

          /* ja:貼り付け */
          avi_edit = avi_from_scenario (data->data);
          if (!avi_edit)
            return;
          page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
          vmaid = misc_mdi_get_data (MISC_MDI (mdi), page_num);
          if (vmaid->select.stream < 0)
            {
              /* ja:選択範囲がないときUNDOの準備 */
              d = g_malloc (sizeof (VmaidHistory));
              for (i = 0; i < 2; i++)
                {
                  d->avi_edit[i] = NULL;
                  if (vmaid->avi_edit[i]
                        && !(d->avi_edit[i] = avi_clone (vmaid->avi_edit[i])))
                    result = FALSE;
                }
              if (!result)
                {
                  misc_message_box ("Video maid", _("AVI Clone Error"), 0,
                                                                _("OK"), NULL);
                  for (i = 0; i < 2; i++)
                    if (d->avi_edit[i])
                      avi_release(d->avi_edit[i]);
                  g_free(d);
                  for (i = 0; avi_edit[i]; i++)
                    avi_release (avi_edit[i]);
                  g_free (avi_edit);
                  return;
                }
              d->cursor=vmaid->cursor;
              d->select=vmaid->select;
            }
          else
            {
              /* ja:選択範囲があればそれを削除する */
              d = vmaid->redo;
              vmaid->redo = NULL;
              command_delete (NULL, 0, widget);
              vmaid->redo = d;
            }
          paste[0] = paste[1] = FALSE;
          for (i = 0; avi_edit[i]; i++)
            switch (avi_type(avi_edit[i]))
              {
                case AVI_STREAM_VIDEO:
                  if (paste[0])
                    {
                      avi_release(avi_edit[i]);
                    }
                  else
                    {
                      if (!vmaid->avi_edit[0])
                        {
                          vmaid->avi_edit[0] = avi_edit[i];
                        }
                      else if (!avi_paste (vmaid->avi_edit[0],
                                        MIN (vmaid->cursor.frame,
                                            avi_length (vmaid->avi_edit[0])),
                                                                avi_edit[i]))
                        {
                          avi_release(avi_edit[i]);
                          result = FALSE;
                        }
                      paste[0] = TRUE;
                    }
                  break;
                case AVI_STREAM_AUDIO:
                  if (paste[1])
                    {
                      avi_release(avi_edit[i]);
                    }
                  else
                    {
                      if (!vmaid->avi_edit[1])
                        {
                          vmaid->avi_edit[1] = avi_edit[i];
                        }
                      else if (!avi_paste (vmaid->avi_edit[1],
#ifdef G_HAVE_GINT64
                            MIN (avi_time_to_sample (vmaid->avi_edit[1],
                                            (gint64)vmaid->cursor.frame * 1000
                                                * vmaid->scale / vmaid->rate),
                                        avi_length (vmaid->avi_edit[0])),
#else /* not G_HAVE_GINT64 */
                            MIN (avi_time_to_sample (vmaid->avi_edit[1],
                                                    vmaid->cursor.frame * 1000
                                                * vmaid->scale / vmaid->rate),
                                        avi_length (vmaid->avi_edit[0])),
#endif /* not G_HAVE_GINT64 */
                                                                avi_edit[i]))
                        {
                          avi_release (avi_edit[i]);
                          result = FALSE;
                        }
                      paste[1] = TRUE;
                    }
                  break;
                default:
                  avi_release (avi_edit[i]);
                  result = FALSE;
              }
          g_free (avi_edit);
          if (!result || (!paste[0] && !paste[1]))
            {
              misc_message_box ("Video maid", _("AVI Paste Error"), 0,
                                                                _("OK"), NULL);
              for (i = 0; i < 2; i++)
                if (vmaid->avi_edit[i])
                  avi_release(vmaid->avi_edit[i]);
              d = vmaid->undo;
              vmaid->undo = vmaid->undo->next;
              for (i = 0; i < 2; i++)
                vmaid->avi_edit[i] = d->avi_edit[i];
              g_free (d);
              return;
            }
          if (vmaid->avi_edit[0])
            {
              vmaid->rate = avi_get_rate (vmaid->avi_edit[0]);
              vmaid->scale = avi_get_scale (vmaid->avi_edit[0]);
            }
          max = get_max_frame (vmaid, -1) + 1;
          sx = MAX ((vmaid->drawing->allocation.width - 4)
                                                        / vmaid->width - 2, 1);
          if (vmaid->avi_edit[0] && !vmaid->avi_edit[1])
            vmaid->cursor.stream = 0;
          else if (!vmaid->avi_edit[0] && vmaid->avi_edit[1])
            vmaid->cursor.stream = 1;
          if (vmaid->cursor.frame > max)
            vmaid->cursor.frame = max;
          if (vmaid->top > vmaid->cursor.frame)
            vmaid->top=vmaid->cursor.frame;
          else if (vmaid->top < vmaid->cursor.frame - sx)
            vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
          /* ja:メニュー,ステートメント,子ウインドウ */
          set_menu_bar (vmaid);
          gtk_widget_set_usize (vmaid->drawing, vmaid->width * 3 + 4,
                        system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height);
          gtk_widget_queue_resize (GTK_WIDGET (window));
          gtk_widget_draw (vmaid->drawing, NULL);
          draw_caret (vmaid, NULL);
          misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
          misc_mdi_set_edited (MISC_MDI (mdi), page_num, TRUE);
        }
    }
}


void
signal_selection_recv (GObject   *object,
                       gchar    **argv,
                       gpointer   user_data)
{
  gboolean *arg_files;
  gint i, array[2], def[2];
  ArgumentBool arg_n, arg_x;
  ArgumentValue arg_h;
  ArgumentArray arg_s;
  ArgumentList arg_opts[] = {
{"history", 'h',  "NUM",          N_("History"),     ARGUMENT_TYPE_VALUE, &arg_h},
{"new",     'n',  "ON/OFF",       N_("New File"),    ARGUMENT_TYPE_BOOL,  &arg_n},
{"size",    's',  "WIDTH,HEIGHT", N_("Window Size"), ARGUMENT_TYPE_ARRAY, &arg_s},
{"second",  'x',  "ON/OFF",       N_("Open Window"), ARGUMENT_TYPE_BOOL,  &arg_x},
{NULL,      '\0', NULL,           NULL,              0,                   NULL}};

  /* ja:プロセス間通信 */
  /* ja:ファイルの履歴 */
  arg_h.val = misc_history_get_num (MISC_HISTORY (history));
  arg_h.def = MISC_HISTORY_NUM_DEF;
  /* ja:新規ファイル */
  arg_n.flag = newfile;
  arg_n.def = FALSE;
  /* ja:ウインドウサイズ */
  array[0] = def_width;
  array[1] = def_height;
  def[0] = gdk_screen_width ()  * 2 / 3;
  def[1] = gdk_screen_height () * 2 / 3;
  arg_s.leng = 2;
  arg_s.array = array;
  arg_s.def = def;
  /* ja:新規ウインドウ */
  arg_x.flag = second;
  arg_x.def = FALSE;
  /* ja:コマンドラインの解析 */
  arg_files = arg_analyse (argv, arg_opts);
  misc_history_set_num (MISC_HISTORY (history), arg_h.val);
  newfile = arg_n.flag;
  def_width = array[0];
  def_height = array[1];
  second = arg_x.flag;
  for (i = 1; argv[i] != NULL; i++)
    if (arg_files[i])
      {
        FileOpen *file_open;

        file_open = g_malloc (sizeof (FileOpen));
        file_open->file = g_strdup (argv[i]);
        file_open->merge = file_merge;
#ifdef USE_THREAD
        g_thread_create ((GThreadFunc)file_open_edit, file_open, TRUE, NULL);
#else /* not USE_THREAD */
        file_open_edit (file_open);
#endif /* not USE_THREAD */
      }
  g_free (arg_files);
}


/******************************************************************************
*                                                                             *
* ja:シグナル/イベント関数群(メイン)                                          *
*                                                                             *
******************************************************************************/
gboolean
signal_delete (GtkWidget *widget,
               GdkEvent  *event,
               gpointer   user_data)
{
  gint page;

  while ((page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi))) >= 0)
    {
      if (!prompt_close ((VmaidWindow *)
                                    misc_mdi_get_data (MISC_MDI (mdi), page)))
        return TRUE;
      gtk_notebook_remove_page (GTK_NOTEBOOK (mdi), page);
    }
#ifdef USE_THREAD
  g_thread_create ((GThreadFunc)close_program, NULL, TRUE, NULL);
#else /* not USE_THREAD */
  close_program ();
#endif /* not USE_THREAD */
  return TRUE;
}
