/*
    chunk.c
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include "chunk.h"


/*  4つの文字からIDを作る
     c1,1文字目
     c2,2文字目
     c3,3文字目
     c4,4文字目
    RET,ID                                                                  */
fourcc_t
make4cc (const char c1,
         const char c2,
         const char c3,
         const char c4)
{
  const static long s = 1;

  return *(char *)&s == 0 ? (unsigned long)(unsigned char)c1 << 24
                          | (unsigned long)(unsigned char)c2 << 16
                          | (unsigned long)(unsigned char)c3 << 8
                          | (unsigned long)(unsigned char)c4
                          : (unsigned long)(unsigned char)c1
                          | (unsigned long)(unsigned char)c2 << 8
                          | (unsigned long)(unsigned char)c3 << 16
                          | (unsigned long)(unsigned char)c4 << 24;
}


/*  4つの文字からIDを作る
    fourcc,ID
         n,順番(1..4)
       RET,文字                                                             */
char
getidcc (const fourcc_t fourcc,
         const int      n)
{
  const static long s = 1;

  return (char)(fourcc >> ((*(char *)&s == 0 ? 4 - n : n - 1) * 8));
}


/*  unsigned longデータを読み込む
    cksize,格納するバッファ
        fp,ファイルポインタ
       RET,0:成功,-1:エラー                                                 */
static int
fcksize (cksize_t *cksize,
         FILE     *fp)
{
  int i;
  cksize_t s = 0;

  for (i = 0; i < 4; i++)
    {
      int c;

      c = fgetc (fp);
      if (c == EOF)
        return -1;
      s += (cksize_t)c << (i * 8);
    }
  *cksize = s;
  return 0;
}


/*  ファイルポインタからチャンク構造体を作る
     fp,ファイルポインタ
    RET,チャンク構造体                                                      */
CHUNK *
chunkopen (FILE *fp)
{
  long size;
  CHUNK *ck;

  if (fseek (fp, 0, SEEK_END) != 0 || (size = ftell (fp)) == -1
            || fseek (fp, 0, SEEK_SET) != 0 || !(ck = malloc (sizeof (CHUNK))))
    return NULL;
  ck->num = 1;
  ck->list = calloc (ck->num, sizeof (CHUNKLIST));
  if (!ck->list)
    {
      free (ck);
      return NULL;
    }
  ck->list[ck->num - 1].size = size
                        - (long)sizeof (cksize_t) - (long)sizeof (fourcc_t);
  return ck;
}


/*  チャンク構造体を解放する
    RET,0:成功,-1:エラー                                                    */
int
chunkfree (CHUNK *ck)
{
  free (ck->list);
  free (ck);
  return 0;
}


/*  チャンクの中に入る
     fp,ファイルポインタ
     ck,チャンク構造体
    RET,0:成功,-1:エラー                                                    */
int
chunkin (FILE  *fp,
         CHUNK *ck)
{
  if (ck->num > 1 && ck->list[ck->num - 1].id != make4cc ('R', 'I', 'F', 'F')
                    && ck->list[ck->num - 1].id != make4cc('L', 'I', 'S', 'T'))
    return -1;
  ck->list = realloc (ck->list, (ck->num + 1) * sizeof (CHUNKLIST));
  if (!ck->list)
    {
      ck->num = 0;
      return -1;
    }
  if ((ck->list[ck->num].offset = ftell (fp)) == -1
        || fread (&ck->list[ck->num].id, sizeof (fourcc_t), 1, fp) != 1
        || fcksize (&ck->list[ck->num].size, fp) != 0
        || ((ck->list[ck->num].id == make4cc ('R', 'I', 'F', 'F')
                || ck->list[ck->num].id == make4cc ('L', 'I', 'S', 'T'))
            && fread (&ck->list[ck->num].form, sizeof (fourcc_t), 1, fp) != 1))
    {
      fseek (fp, ck->list[ck->num].offset, SEEK_SET);
      return -1;
    }
  ck->num++;
  return 0;
}


/*  チャンクから出る
     fp,ファイルポインタ
     ck,チャンク構造体
    RET,0:成功,-1:エラー                                                    */
int
chunkout (FILE  *fp,
          CHUNK *ck)
{
  if (ck->num <= 1)
    return -1;
  ck->num--;
  if (fseek (fp, ck->list[ck->num].offset, SEEK_SET) != 0)
    return -1;
  ck->list = realloc (ck->list, ck->num * sizeof (CHUNKLIST));
  if (!ck->list)
    {
      ck->num = 0;
      return -1;
    }
  return 0;
}


/*  次のチャンチャンクに移動する
     fp,ファイルポインタ
     ck,チャンク構造体
    RET,0:成功,-1:エラー                                                    */
int
chunknext (FILE  *fp,
           CHUNK *ck)
{
  cksize_t size;
  long offset;

  offset = ftell (fp);
  if (offset == -1)
    return -1;
  if (fseek (fp, sizeof (fourcc_t), SEEK_CUR) != 0 || fcksize (&size, fp) != 0)
    {
      fseek (fp, offset, SEEK_SET);
      return -1;
    }
  size = (size + 1) & ~1;
  if (ck->list[ck->num - 1].offset + ck->list[ck->num - 1].size
                                                            <= offset + size
                                            || fseek (fp, size, SEEK_CUR) != 0)
    {
      fseek (fp, offset, SEEK_SET);
      return -1;
    }
  return 0;
}


/*  チャンクのIDを取得する
     fp,ファイルポインタ
    RET,ID                                                                  */
fourcc_t
chunkid (FILE *fp)
{
  fourcc_t id;

  return fread (&id, sizeof (fourcc_t), 1, fp) == 1
    && fseek (fp, -(long)sizeof (fourcc_t), SEEK_CUR) == 0 ? id : (fourcc_t)-1;
}


/*  チャンクのサイズを取得する
     fp,ファイルポインタ
    RET,フォーム                                                            */
cksize_t
chunksize (FILE *fp)
{
  cksize_t size;

  return fseek (fp, sizeof (fourcc_t), SEEK_CUR) == 0
            && fcksize (&size, fp) == 0
            && fseek (fp, -(long)sizeof (cksize_t) - (long)sizeof (fourcc_t),
                                                                SEEK_CUR) == 0
                                                        ? size : (cksize_t)-1;
}


/*  チャンクのフォームを取得する
     fp,ファイルポインタ
    RET,フォーム                                                            */
fourcc_t
chunkform (FILE *fp)
{
  fourcc_t id, form;

  return fread (&id, sizeof (fourcc_t), 1, fp) == 1
        && fseek (fp, sizeof (cksize_t), SEEK_CUR) == 0
        && fread (&form, sizeof (fourcc_t), 1, fp) == 1
        && fseek (fp, -(long)sizeof (cksize_t) - (long)sizeof (fourcc_t) * 2,
                                                                SEEK_CUR) == 0
        && (id == make4cc ('R', 'I', 'F', 'F')
                                        || id == make4cc ('L', 'I', 'S', 'T'))
                                                        ? form : (fourcc_t)-1;
}
