/*
    aviplay.c
    copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "chunk.h"


#ifndef min
#define min(a,b) (((a)<(b))?(a):(b))
#endif /* not min */


typedef struct { unsigned char dummy[56]; } AVISTREAMHEADER;
typedef struct { unsigned char dummy[16]; } AVIINDEXENTRY;
typedef struct { unsigned char dummy[40]; } BITMAPINFOHEADER;


#define ash_type(ash) get_value((unsigned char *)(ash),0,4)
#define ash_handler(ash) get_value((unsigned char *)(ash),4,4)
#define ash_scale(ash) get_value((unsigned char *)(ash),20,4)
#define ash_rate(ash) get_value((unsigned char *)(ash),24,4)
#define aie_ckid(aie) get_value((unsigned char *)(aie),0,4)
#define aie_offset(aie) get_value((unsigned char *)(aie),8,4)
#define aie_length(aie) get_value((unsigned char *)(aie),12,4)
#define bmih_width(bmih) get_value((unsigned char *)(bmih),4,4)
#define bmih_height(bmih) get_value((unsigned char *)(bmih),8,4)
#define bmih_bit_count(bmih) get_value((unsigned char *)(bmih),14,2)


static long
get_value (const unsigned char *buf,
           const int            offset,
           const size_t         size)
{
  int i;
  long value = 0;

  for (i = 0; i < size; i++)
    value += (long)buf[i + offset] << (i * 8);
  return value;
}


int
main (int   argc,
      char *argv[])
{
  char *text = " .:-+=ot76x0s&8%";/* 暗い背景に明るい文字 */
  char *q;                  /* 出力する文字 */
  unsigned char *p;         /* ピクセルデータを読み込むバッファ */
  clock_t c;                /* 時間 */
  time_t t;                 /* 時間 */
  cksize_t size;            /* インデックスチャンクのバイト数 */
  fourcc_t type;            /* strhチャンクのチャンクタイプ */
  int i;                    /* ループカウンタ */
  int cx;                   /* 実際の横の文字数 */
  int cy;                   /* 実際の縦の文字数 */
  int s = 0;                /* argvの中のファイル名,インデックスの数 */
  int frame = 0;            /* フレーム */
  int skip = 1;             /* 飛ばすフレーム */
  int width = 0;            /* 横の文字数 */
  int height = 0;           /* 縦の文字数 */
  int esc = 0;              /* エスケープシーケンス */
  int scale, rate;          /* rate÷scale=1秒間のフレーム数 */
  int bwidth, bheight;      /* ビットマップのサイズ */
  int bcount;               /* ビットマップのビット数 */
  long offset;              /* moviフォームのLISTチャンクのオフセット */
  size_t image;             /* ピクセルデータのバイト数 */
  size_t line;              /* 横ラインのバイト数 */
  AVISTREAMHEADER ash;      /* AVISTREAMHEADER構造体 */
  BITMAPINFOHEADER bmih;    /* BITMAPINFOHEADER構造体 */
  CHUNK *ck;                /* チャンクポインタ */
  FILE *fp1;                /* インデックスを読むためのファイルポインタ */
  FILE *fp2;                /* イメージデータを読むためのファイルポインタ */

  for (i = 1; i < argc; i++)
    if ((argv[i])[0] != '-' && (argv[i])[0] != '/')
      /* オプションではなくファイル名 */
      s = i;
    else
      /* オプションの設定 */
      switch (toupper ((argv[i])[1]))
        {
          case 'E':/* エスケープシーケンス */
            esc = 1;
            break;
          case 'L':/* 明るい背景に暗い文字 */
            text = "%8&s0x67to=+-:. ";
            break;
          case 'S':/* 明るい背景に暗い文字 */
            skip = atoi (argv[i] + 2);
            if (skip <= 0)
              skip = 1;
            break;
          case 'W':/* 横の文字数 */
            width = atoi (argv[i] + 2);
            break;
          case 'H':/* 縦の文字数 */
            height = atoi (argv[i] + 2);
        }
  if (s == 0)
    {
      /* 引数にファイル名がないとき */
      fprintf (stderr,
"copyright (c) 1998-2006 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org\n"
"This program comes with NO WARRANTY, to the extent permitted by law.\n"
"You may redistribute it under the terms of the GNU General Public License;\n"
"see the file named COPYING for details.\n"
"\n"
"Usage: %s [-e] [-l] [-w<x>] [-h<x>] <avifile>\n"
" -e    Escape sequence\n"
" -l    Light background\n"
" -s<n> Skip frame\n"
" -w<n> Display width\n"
" -h<n> Display height\n"
"\n", argv[0]);
        return 0;
    }

  /* ファイルを開く */
  fp1 = fopen (argv[s], "rb");
  fp2 = fopen (argv[s], "rb");
  if (!fp1 || !fp2)
    {
      fprintf (stderr, "%s can not open.\n", argv[s]);
      return -1;
    }

  /* はじめのチャンクのBITMAPINFOHEADERを取得する */

  /* ファイルポインタからチャンクポインタを作る */
  ck = chunkopen (fp1);
  if (!ck)
    {
      fprintf (stderr, "chunk open error.\n");
      return -1;
    }
  /* RIFFチャンクに入る */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* フォームがhdrlのLISTチャンクを探す */
  while (chunkform (fp1) != make4cc ('h', 'd', 'r', 'l'))
    if (chunknext (fp1, ck) != 0)
      {
        fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' chunk search error.\n");
        chunkfree (ck);
        return -1;
      }
  /* フォームがhdrlのLISTチャンクに入る(RIFF'AVI ' -> LIST'hdrl') */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* フォームがstrlのLISTチャンクを探す */
  while (chunkform (fp1) != make4cc ('s', 't', 'r', 'l'))
    if (chunknext (fp1, ck) != 0)
      {
        fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' "
                                                    "chunk search error.\n");
        chunkfree (ck);
        return -1;
      }
  /* フォームがstrlのLISTチャンクに入る
                                    (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl') */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' "
                                                        "chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* strhチャンクを探す(AVISTREAMHEADER) */
  while (chunkid (fp1) != make4cc ('s', 't', 'r', 'h'))
    if (chunknext (fp1, ck) != 0)
      {
        fprintf (stderr,
                        "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' -> strh "
                                                    "chunk search error.\n");
        chunkfree (ck);
        return -1;
      }
  /* strhチャンクに入る(AVISTREAMHEADER)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' -> strh) */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' -> strh "
                                                        "chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  if (fread (&ash, sizeof (AVISTREAMHEADER), 1, fp1) != 1)
    {
      fprintf (stderr, "AVISTREAMHEADER read error.\n");
      chunkfree (ck);
      return -1;
    }
  scale = ash_scale (&ash);
  rate = ash_rate (&ash);
  if (ash_type (&ash) != make4cc ('v', 'i', 'd', 's'))
    {
      fprintf (stderr, "first stream is not video.\n");
      chunkfree (ck);
      return -1;
    }
  if (ash_handler (&ash) != make4cc ('D', 'I', 'B', ' ')
                                                    && ash_handler (&ash) != 0)
    {
      fprintf (stderr, "first stream is compressed.\n");
      chunkfree (ck);
      return -1;
    }
  /* strhチャンクから出る(AVISTREAMHEADER)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' <- strh) */
  if (chunkout (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' <- strh "
                                                        "chunk out error.\n");
      chunkfree (ck);
      return -1;
    }
  /* フォームがstrlのLISTチャンクから出る
                                    (RIFF'AVI ' -> LIST'hdrl' <- LIST'strl') */
  if (chunkout (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' <- LIST\'strl\' "
                                                        "chunk out error.\n");
      chunkfree (ck);
      return -1;
    }
  /* フォームがstrlのLISTチャンクに入る
                                    (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl') */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' "
                                                        "chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* strfチャンクを探す(BITMAPINFOHEADER) */
  while (chunkid (fp1) != make4cc ('s', 't', 'r', 'f'))
    if (chunknext (fp1, ck) != 0)
      {
        fprintf (stderr,
                        "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' -> strf "
                                                    "chunk search error.\n");
        chunkfree (ck);
        return -1;
      }
  /* strfチャンクに入る(BITMAPINFOHEADER)
                            (RIFF'AVI ' -> LIST'hdrl' -> LIST'strl' -> strf) */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> LIST\'hdrl\' -> LIST\'strl\' -> strf "
                                                        "chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* BITMAPINFOHEADERを読み込む */
  if (fread (&bmih, sizeof (BITMAPINFOHEADER), 1, fp1) != 1)
    {
      fprintf (stderr, "BITMAPINFOHEADER can not read.\n");
      chunkfree (ck);
      return -1;
    }
  bwidth = bmih_width (&bmih);
  bheight = bmih_height (&bmih);
  bcount = bmih_bit_count (&bmih);
  /* チャンクポインタを解放する */
  if (chunkfree (ck) != 0)
    fprintf (stderr, "chunk free error.\n");

  /* ムービーチャンクへ移動する */

  /* ファイルポインタからチャンクポインタを作る */
  ck = chunkopen (fp1);
  if (!ck)
    {
      fprintf (stderr, "chunk open error.\n");
      return -1;
    }
  /* RIFFチャンクに入る */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* moviチャンクを探す */
  while (chunkform (fp1) != make4cc ('m', 'o', 'v', 'i'))
    if (chunknext (fp1, ck) != 0)
      {
        fprintf (stderr, "RIFF\'AVI \' -> LIST\'movi\' chunk search error.\n");
        chunkfree (ck);
        return -1;
      }
  /* ファイルのオフセットを取得する */
  offset = ftell (fp1);
  if (offset == -1)
    {
      fprintf (stderr, "file pointer error.\n");
      chunkfree (ck);
      return -1;
    }
  offset += sizeof (fourcc_t) + sizeof (cksize_t)
                + sizeof (cksize_t) + sizeof (fourcc_t);/* moviのオフセット */
  /* チャンクポインタを解放する */
  if (chunkfree (ck) != 0)
    fprintf (stderr, "chunk free error.\n");

  /* インデックスチャンクへ移動する */

  /* ファイルポインタからチャンクポインタを作る */
  ck = chunkopen (fp1);
  if (!ck)
    {
      fprintf (stderr, "chunk open error.\n");
      return -1;
    }
  /* RIFFチャンクに入る */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' chunk in error.\n");
      chunkfree (ck);
      return -1;
    }
  /* idx1チャンクを探す */
  while (chunkid (fp1) != make4cc ('i', 'd', 'x', '1'))
    if (chunknext (fp1, ck) != 0)
      {
        fprintf (stderr, "RIFF\'AVI \' -> idx1 chunk search error.\n");
        chunkfree (ck);
        return -1;
      }
  /* idx1チャンクのサイズ */
  size = chunksize (fp1);
  if (size == (cksize_t)-1)
    {
      fprintf (stderr, "idx1 chunk size error.\n");
      chunkfree (ck);
      return -1;
    }
  /* idx1チャンクに入る(RIFF'AVI ' -> idx1) */
  if (chunkin (fp1, ck) != 0)
    {
      fprintf (stderr, "RIFF\'AVI \' -> idx1 chunk in error.\n");
      chunkfree (ck);
      return -1;
    }

  switch (bcount)
    {
      case 16:
        line = (bwidth * 2 + 3) & ~3;
        break;
      case 24:
        line = (bwidth * 3 + 3) & ~3;
        break;
      case 32:
        line = bwidth * 4;
      default:
        /* 16/24/32ビットのビットマップでないとき */
        fprintf (stderr, "%s is not 16/24/32 bit color.\n", argv[s]);
        chunkfree (ck);
        return -1;
    }
  /* フレームのサイズ */
  if (width > 0 && height > 0)
    {
      /* 縦横の比率を保持しない */
      cx = width;
      cx = height;
    }
  else
    {
      if (width <= 0 && height <= 0)
        {
          width = 79;
          height = 23;
          /* 縦横の比率を保持して最大の文字数を求める */
          cx = min (bwidth * height * 2 / bheight, width);
          cy = min (bheight * width / bwidth / 2, height);
        }
      else
        {
          cx = width > 0 ? width : bwidth * height * 2 / bheight;
          cy = height > 0 ? height : bheight * width / bwidth / 2;
        }
    }
  if (cx < 1)
    cx = 1;
  if (cy < 1)
    cy = 1;
  /* ピクセルデータを読みこむメモリを確保する */
  image = line * bheight;
  p = malloc (image);
  q = malloc ((cx + 1) * cy + 1);
  for (i = 0; i < cy; i++)
    q[(cx + 1) * (i + 1) - 1] = '\n';
  q[(cx + 1) * cy] = '\0';
  if (!p)
    {
      fprintf (stderr, "memory alloc error.\n");
      chunkfree (ck);
      return -1;
    }
  /* フレームを表示するループ */
  s = size / sizeof (AVIINDEXENTRY);
  c = clock ();
  t = time (NULL);
  for (i = 0; i < s; i++)
    {
      size_t length;
      AVIINDEXENTRY aie;        /* AVIINDEXENTRY構造体 */

      if (fread (&aie, sizeof (AVIINDEXENTRY), 1, fp1) != 1)
        {
          fprintf (stderr, "AVIINDEXENTRY can not read.\n");
          break;
        }
      length = aie_length (&aie);
      if (aie_ckid (&aie) == make4cc ('0', '0', 'd', 'b')
                                        && (length == 0 || length == image))
        {
          clock_t clk;
          time_t tm;

          /* 0ストリームの非圧縮のフレームのとき */
          frame++;
          clk = clock ();
          tm = time (NULL);
          if (clk - c < (tm - t) * CLOCKS_PER_SEC)
            c = clk - (tm - t) * CLOCKS_PER_SEC;
          if (frame % skip != 0
                || (double)frame * CLOCKS_PER_SEC * scale / rate <= clk - c)
            continue;/* フレーム落ちしているときには表示をスキップ */
          if (length > 0)
            {
              int x, y;

              /* ピクセルデータへシークする */
              if (fseek (fp2, offset + aie_offset (&aie), SEEK_SET) != 0)
                {
                  fprintf (stderr, "file seek error.\n");
                  break;
                }
              /* ピクセルデータを読み込む */
              if (fread (p, sizeof (unsigned char), length, fp2) != length)
                {
                  fprintf (stderr, "pixel data can not read.\n");
                  break;
                }
              switch (bcount)
                {
                  case 16:
                    for (y = 0; y < cy; y++)
                      for (x = 0; x < cx; x++)
                        {
                          int r, g, b;
                          long adr;                 /* ピクセルの位置 */

                          adr = bheight * (cy - y - 1) / cy * line
                                                        + bwidth * x / cx * 2;
                          b = (p[adr] & 0x1f);
                          g = ((p[adr] & 0xe0) >> 5)
                                                + ((p[adr + 1] & 0x03) << 3);
                          r = (p[adr + 1] & 0x7c) >> 2;
                          q[x + y * (cx + 1)] = text[(b * 29 + g * 150
                                                            + r * 77) / 512];
                        }
                    break;
                  case 24:
                    for (y = 0; y < cy; y++)
                      for (x = 0; x < cx; x++)
                        {
                          long adr;                 /* ピクセルの位置 */

                          adr = bheight * (cy - y - 1) / cy * line
                                                        + bwidth * x / cx * 3;
                          q[x + y * (cx + 1)] = text[(p[adr] * 29
                                                    + p[adr + 1] * 150
                                                    + p[adr + 2] * 77) / 4096];
                        }
                    break;
                  case 32:
                    for (y = 0; y < cy; y++)
                      for (x = 0; x < cx; x++)
                        {
                          long adr;                 /* ピクセルの位置 */

                          adr = bheight * (cy - y - 1) / cy * line
                                                        + bwidth * x / cx * 4;
                          q[x + y * (cx + 1)] = text[(p[adr] * 29
                                                    + p[adr + 1] * 150
                                                    + p[adr + 2] * 77) / 4096];
                        }
                }
              if (esc != 0 && i > 0)
                printf ("\x1b[%dA%s", cy, q);/* エスケープシーケンスで移動 */
              else
                fputs (q, stdout);
            }
          /* 表示が早いときにはループする */
          while (clock () - c < (double)frame * CLOCKS_PER_SEC * scale / rate);
        }
    }
  /* チャンクポインタを解放する */
  if (chunkfree (ck) != 0)
    fprintf (stderr, "chunk free error.\n");
  free (p);
  free (q);
  return 0;
}
