/*
    Video maid
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "command.h"
#include "file.h"
#include "general.h"
#include "sigfile.h"
#include "size.h"
#include "thread.h"
#include "orz/orzmdi.h"


/******************************************************************************
*                                                                             *
* ja:一般関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:キャレットを消す
    vmaid,ウインドウ情報                                                    */
void
hide_caret (VmaidWindow *vmaid)
{
  g_source_remove (timer_id);
  timer_id = 0;
  if (caret)
    {
      gint sx;

      caret = FALSE;
      sx = get_sx_frame (vmaid);
      if (vmaid->top <= vmaid->cursor.frame
                                    && vmaid->cursor.frame <= vmaid->top + sx)
        gtk_widget_queue_draw_area (vmaid->drawing,
                        (vmaid->cursor.frame - vmaid->top) * vmaid->width
                                    + (vmaid->selfrm ? vmaid->width + 1 : 0),
                        system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0),
                                                            2, vmaid->height);
    }
}


/*  ja:キャレットの描画
     vmaid,ウインドウ情報
    cursor,古いキャレット位置                                               */
void
draw_caret (VmaidWindow *vmaid,
            VmaidCursor *cursor)
{
  gchar *text;
  gint t, sx;
  guint context_id;

  sx = get_sx_frame (vmaid);
  if (cursor && timer_id != 0 && (vmaid->cursor.frame != cursor->frame
                                    || vmaid->cursor.stream != cursor->stream))
    {
      /* ja:タイマ再設定 */
      g_source_remove (timer_id);
      timer_id = 0;
      /* ja:古いキャレットを消去 */
      if (vmaid->top <= cursor->frame && cursor->frame <= vmaid->top + sx)
        gtk_widget_queue_draw_area (vmaid->drawing,
                    (cursor->frame - vmaid->top) * vmaid->width
                                    + (vmaid->selfrm ? vmaid->width + 1 : 0),
                    system_font_height * 2
                                    + (vmaid->avi_edit[0] && cursor->stream > 0
                                                        ? vmaid->height : 0),
                                                            2, vmaid->height);
    }
  /* ja:新しいキャレットを描画 */
  if (timer_id == 0)
    caret = TRUE;
  if (vmaid->top <= vmaid->cursor.frame
                                    && vmaid->cursor.frame <= vmaid->top + sx)
    gtk_widget_queue_draw_area (vmaid->drawing,
            (vmaid->cursor.frame - vmaid->top) * vmaid->width
                                    + (vmaid->selfrm ? vmaid->width + 1 : 0),
            system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0),
                                                            2, vmaid->height);
  if (timer_id == 0)
    timer_id = g_timeout_add (500, signal_timeout, NULL);
  t = (glonglong)vmaid->cursor.frame * vmaid->scale * 1000 / vmaid->rate;
  text = vmaid->select.stream < 0
        ? g_strdup_printf (_("Frame %d  Time %02d:%02d:%02d"),
                                vmaid->cursor.frame,
                                t / 60000 % 100, t / 1000 % 60, t / 10 % 100)
        : g_strdup_printf (_("Frame %d  Selected %d ... %d"),
            vmaid->cursor.frame,
            vmaid->select.frame
                        - (vmaid->cursor.frame < vmaid->select.frame ? 1 : 0),
            vmaid->cursor.frame
                        - (vmaid->cursor.frame > vmaid->select.frame ? 1 : 0));
  context_id = gtk_statusbar_get_context_id (GTK_STATUSBAR (status), "Status");
  gtk_statusbar_pop (GTK_STATUSBAR (status), context_id);
  gtk_statusbar_push (GTK_STATUSBAR (status), context_id, text);
  g_free (text);
}


/*  ja:メニューを設定する
    vmaid,ウインドウ情報                                                    */
void
set_menu_bar (VmaidWindow *vmaid)
{
  if (vmaid)
    {
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/file/close"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/file/saveas"), vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/file/preview"), vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/file/properties"), vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/edit"), TRUE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/undo"), vmaid->undo != NULL);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/redo"), vmaid->redo != NULL);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/cut"), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/copy"), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/delete"), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/change"), vmaid->avi_edit[0] != NULL);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/reverse"), vmaid->avi_edit[0]
                                && avi_edit_length (vmaid->avi_edit[0]) > 1);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/jump"), vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                "/edit/all"), vmaid->avi_edit[0] || vmaid->avi_edit[1]);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/view"), TRUE);
      gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (misc_find_menu
                                (menu_entries, "/view/frame")), vmaid->selfrm);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "saveas")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                                "properties")),
                    vmaid->avi_edit[0] != NULL || vmaid->avi_edit[1] != NULL);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                                "preview")),
                    vmaid->avi_edit[0] != NULL || vmaid->avi_edit[1] != NULL);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "cut")), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "copy")), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "paste")), TRUE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "delete")), vmaid->select.stream >= 0);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "undo")), vmaid->undo != NULL);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                        "redo")), vmaid->redo != NULL);
    }
  else
    {
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/close"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/saveas"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/preview"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries,
                                                "/file/properties"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/edit"), FALSE);
      gtk_widget_set_sensitive (misc_find_menu (menu_entries, "/view"), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "saveas")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "properties")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "preview")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "cut")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "copy")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "paste")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "delete")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "undo")), FALSE);
      gtk_widget_set_sensitive (GTK_WIDGET (misc_find_toolbar (toolbar_entries,
                                                        "redo")), FALSE);
    }
}


/*  ja:指定範囲を再描画する
     vmaid,ウインドウ情報
    cursor,古いキャレット位置
    select,古い選択開始位置
       top,古い左の座標                                                     */
void
clear_sel (VmaidWindow *vmaid,
           VmaidCursor *cursor,
           VmaidCursor *select,
           const gint   top)
{
  gint sx, frame, width, dx;
  gint left_old = -1, right_old = -1, left_new = -1, right_new = -1;
  GdkGC *gc;
  GdkRectangle rc, rc0, rc1;

  sx = get_sx_frame (vmaid);
  /* ja:スクロール */
  if (vmaid->top != top)
    {
      /* ja:移動量が多すぎるときには全体を再描画 */
      rc.x = vmaid->selfrm ? vmaid->width + 2 : 1;
      rc.y = 0;
      rc.width = vmaid->width * sx;
      rc.height = system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height;
      if (ABS (top - vmaid->top) < sx)
        {
          /* ja:移動量が小さいとき */
          dx = (top - vmaid->top) * vmaid->width;
          gc = gdk_gc_new (vmaid->drawing->window);
          gdk_gc_set_exposures (gc, TRUE);
          gdk_draw_drawable (vmaid->drawing->window, gc,
                        vmaid->drawing->window, dx > 0 ? rc.x : rc.x - dx, 0,
                        dx > 0 ? rc.x + dx : rc.x, 0,
                        rc.width - ABS (dx), rc.height);
          g_object_unref (gc);
          if (dx < 0)
            rc.x = rc.x + rc.width + dx;
          rc.width = ABS (dx);
        }
      if (vmaid->cursor.frame == vmaid->top
                                    || vmaid->cursor.frame == vmaid->top + sx
            || vmaid->cursor.frame == top || vmaid->cursor.frame == top + sx)
        {
          /* ja:端のキャレット */
          rc0.x = (vmaid->cursor.frame == vmaid->top
                                                || vmaid->cursor.frame == top
                                                ? 0 : vmaid->width * sx + 1)
                                    + (vmaid->selfrm ? vmaid->width + 1 : 0);
          rc0.y = system_font_height * 2
                            + (vmaid->avi_edit[0] && vmaid->cursor.stream > 0
                                                        ? vmaid->height : 0);
          rc0.width = 1;
          rc0.height = vmaid->height;
          rc1 = rc;
          gdk_rectangle_union (&rc0, &rc1, &rc);
        }
      if (cursor->frame == vmaid->top || cursor->frame == vmaid->top + sx
                        || cursor->frame == top || cursor->frame == top + sx)
        {
          /* ja:端のキャレット */
          rc0.x = (cursor->frame == vmaid->top || cursor->frame == top
                                                ? 0 : vmaid->width * sx + 1)
                                    + (vmaid->selfrm ? vmaid->width + 1 : 0);
          rc0.y = system_font_height * 2
                                    + (vmaid->avi_edit[0] && cursor->stream > 0
                                                        ? vmaid->height : 0);
          rc0.width = 1;
          rc0.height = vmaid->height;
          rc1 = rc;
          gdk_rectangle_union (&rc0, &rc1, &rc);
        }
    }
  else
    {
      rc.x = -1;
    }
  if (vmaid->selfrm)
    {
      /* ja:以前の左右の選択について */
      if (select->stream >= 0)
        {
          frame = select->frame - (cursor->frame < select->frame ? 1 : 0);
          if (frame < top)
            left_old = frame;
          else if (top + sx <= frame)
            right_old = frame;
        }
      /* ja:現在の左右の選択について */
      if (vmaid->select.stream >= 0)
        {
          frame = vmaid->select.frame
                        - (vmaid->cursor.frame < vmaid->select.frame ? 1 : 0);
          if (frame < vmaid->top)
            left_new = frame;
          else if (vmaid->top + sx <= frame)
            right_new = frame;
        }
      if (left_old != left_new || right_old != right_new)
        {
          rc0.y = 0;
          rc0.width = vmaid->width;
          rc0.height
                = ((vmaid->avi_edit[0] ? 1 : 0) + (vmaid->avi_edit[1] ? 1 : 0))
                                    * vmaid->height + system_font_height * 2;
          /* ja:左側について */
          if (left_old != left_new)
            {
              rc0.x = 0;
              if (rc.x < 0)
                {
                  rc = rc0;
                }
              else
                {
                  rc1 = rc;
                  gdk_rectangle_union (&rc0, &rc1, &rc);
                }
            }
          /* ja:右側について */
          if (right_old != right_new)
            {
              rc0.x = (sx + 1) * vmaid->width + 4;
              if (rc.x < 0)
                {
                  rc = rc0;
                }
              else
                {
                  rc1 = rc;
                  gdk_rectangle_union (&rc0, &rc1, &rc);
                }
            }
        }
    }
  if (vmaid->select.stream < 0 && select->stream >= 0)
    {
      /* ja:選択が解除された */
      width = MIN (MAX (cursor->frame, select->frame), vmaid->top + sx)
                        - MAX (MIN (cursor->frame, select->frame), vmaid->top);
      if (width > 0)
        {
          rc0.x = MAX (MIN (cursor->frame, select->frame) - vmaid->top, 0)
                    * vmaid->width + (vmaid->selfrm ? vmaid->width + 2 : 1);
          rc0.y = MIN (cursor->stream, select->stream) * vmaid->height
                                                    + system_font_height * 2;
          rc0.width = width * vmaid->width;
          rc0.height = (ABS (cursor->stream - select->stream) + 1)
                                                            * vmaid->height;
          if (!vmaid->avi_edit[0])
            rc0.y -= vmaid->height;
          gtk_widget_queue_draw_area (vmaid->drawing,
                                            rc.x, rc.y, rc.width, rc.height);
          if (rc.x < 0)
            {
              rc = rc0;
            }
          else
            {
              rc1 = rc;
              gdk_rectangle_union (&rc0, &rc1, &rc);
            }
        }
    }
  else if (vmaid->select.stream >= 0)
    {
      if (vmaid->cursor.stream != cursor->stream)
        {
          /* ja:ストリームが変化した */
          width = MIN (MAX (cursor->frame, select->frame), vmaid->top + sx)
                        - MAX (MIN (cursor->frame, select->frame), vmaid->top);
          if (width > 0)
            {
              rc0.x = MAX (MIN (cursor->frame, select->frame) - vmaid->top, 0)
                    * vmaid->width + (vmaid->selfrm ? vmaid->width + 2 : 1);
              rc0.y = (vmaid->select.stream == 0 ? vmaid->height : 0)
                                                    + system_font_height * 2;
              rc0.width = width * vmaid->width;
              rc0.height = vmaid->height;
              if (rc.x < 0)
                {
                  rc = rc0;
                }
              else
                {
                  rc1 = rc;
                  gdk_rectangle_union (&rc0, &rc1, &rc);
                }
            }
        }
      if (vmaid->cursor.frame != cursor->frame)
        {
          /* ja:位置が変化した */
          width
            = MIN (MAX (vmaid->cursor.frame, cursor->frame), vmaid->top + sx)
                - MAX (MIN (vmaid->cursor.frame, cursor->frame), vmaid->top);
          if (width > 0)
            {
              rc0.x = MAX (MIN (vmaid->cursor.frame, cursor->frame)
                                                - vmaid->top, 0) * vmaid->width
                                    + (vmaid->selfrm ? vmaid->width + 2 : 1);
              rc0.y = MIN (vmaid->cursor.stream, vmaid->select.stream)
                                    * vmaid->height + system_font_height * 2;
              rc0.width = width * vmaid->width;
              rc0.height = (ABS (vmaid->cursor.stream - vmaid->select.stream)
                                                        + 1) * vmaid->height;
              if (!vmaid->avi_edit[0])
                rc0.y -= vmaid->height;
              if (rc.x < 0)
                {
                  rc = rc0;
                }
              else
                {
                  rc1 = rc;
                  gdk_rectangle_union (&rc0, &rc1, &rc);
                }
            }
        }
    }
  if (rc.x >= 0)
    gtk_widget_queue_draw_area (vmaid->drawing,
                                            rc.x, rc.y, rc.width, rc.height);
}


/*  ja:最小のウインドウサイズを設定する
     vmaid,ウインドウ情報                                                   */
void
set_usize (VmaidWindow *vmaid)
{
  gtk_widget_set_size_request (vmaid->drawing,
                    vmaid->width + (vmaid->selfrm ? vmaid->width * 2 + 4 : 2),
                        system_font_height * 2 + ((vmaid->avi_edit[0] ? 1 : 0)
                            + (vmaid->avi_edit[1] ? 1 : 0)) * vmaid->height);
  gtk_widget_queue_resize (GTK_WIDGET (window));
  gtk_widget_queue_draw (vmaid->drawing);
}


/*  ja:リストから削除する
      d,リストの先頭
    RET,削除したリストの数                                                  */
gint
delete_list (VmaidHistory **d)
{
  gint i, count = 0;
  VmaidHistory *d0, *d1;

  for (d0 = *d; d0; d0 = d1)
    {
      for (i = 0; i < 2; i++)
        if (d0->avi_edit[i])
          avi_edit_close (d0->avi_edit[i]);
      d1 = d0->next;
      g_free (d0);
      count++;
    }
  *d = NULL;
  return count;
}


/*  ja:編集履歴操作
    vmaid,ウインドウ情報
    kind,TRUE:redo,FALSE:undo                                               */
void
history_operation (VmaidWindow    *vmaid,
                   const gboolean  kind)
{
  gint i, sx, max;
  VmaidHistory *d, e;

  if (kind)
    {
      d = vmaid->redo;
      vmaid->redo = d->next;
    }
  else
    {
      d = vmaid->undo;
      vmaid->undo = d->next;
    }
  e.cursor = vmaid->cursor;
  e.select = vmaid->select;
  for (i = 0; i < 2; i++)
    e.avi_edit[i] = vmaid->avi_edit[i];
  vmaid->cursor = d->cursor;
  vmaid->select = d->select;
  for (i = 0; i < 2; i++)
    vmaid->avi_edit[i] = d->avi_edit[i];
  *d = e;
  if (kind)
    {
      d->next = vmaid->undo;
      vmaid->undo = d;
    }
  else
    {
      d->next = vmaid->redo;
      vmaid->redo = d;
    }
  max = get_max_frame (vmaid, -1) + 1;
  sx = get_sx_frame (vmaid);
  if (vmaid->cursor.frame > max)
    vmaid->cursor.frame = max;
   /* ja:キャレットが表示されるように調整 */
  if (vmaid->top > MIN (vmaid->cursor.frame, max - sx))
    vmaid->top = MIN (vmaid->cursor.frame, max - sx);
  if (vmaid->top < MAX (vmaid->cursor.frame - sx, 0))
    vmaid->top = MAX (vmaid->cursor.frame - sx, 0);
  set_menu_bar (vmaid);
  set_usize (vmaid);
  misc_set_scroll_bar (vmaid->hscroll,
                                GTK_SIGNAL_FUNC (signal_value_changed_hscroll),
                                                vmaid, 0, max, sx, vmaid->top);
  draw_caret (vmaid, NULL);
}


/*  ja:ファイルの保存を問い合わせる
    vmaid,ウインドウ情報
      RET,TRUE:ファイルを閉じる,FALSE:ファイルを閉じない                    */
gboolean
prompt_close (VmaidWindow *vmaid)
{
  gint page;

  page = orz_mdi_get_page_from_data (ORZ_MDI (mdi), vmaid);
  if (orz_mdi_get_edited (ORZ_MDI (mdi), page))
    {
      gchar *utf8str;
      gint result;
      GtkWidget *dialog;

#if GLIB_CHECK_VERSION(2,6,0)
      utf8str = g_filename_display_name
                                    (orz_mdi_get_file (ORZ_MDI (mdi), page));
#else /* not GLIB_CHECK_VERSION(2,6,0) */
      utf8str = g_filename_to_utf8
                (orz_mdi_get_file (ORZ_MDI (mdi), page), -1, NULL, NULL, NULL);
#endif /* not GLIB_CHECK_VERSION(2,6,0) */
      dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_INFO, GTK_BUTTONS_NONE,
                            _("File %s was edited.\nSave?"), utf8str);
      g_free (utf8str);
      gtk_dialog_add_buttons (GTK_DIALOG (dialog),
#ifdef G_OS_WIN32
                                        GTK_STOCK_YES, GTK_RESPONSE_YES,
                                        GTK_STOCK_NO, GTK_RESPONSE_NO,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
#else /* not G_OS_WIN32 */
                                        GTK_STOCK_NO, GTK_RESPONSE_NO,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_YES, GTK_RESPONSE_YES,
#endif /* not G_OS_WIN32 */
                                        NULL);
      g_signal_connect (G_OBJECT (dialog), "key-press-event",
                                    G_CALLBACK (misc_dialog_key_press), NULL);
      result = gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      switch (result)
        {
          case GTK_RESPONSE_YES:
            {
              gint page_num;

              page_num = gtk_notebook_get_current_page (GTK_NOTEBOOK (mdi));
              if (page != page_num)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page);
              else
                page_num = -1;
              command_save (NULL, NULL);
              if (page_num >= 0)
                gtk_notebook_set_current_page (GTK_NOTEBOOK (mdi), page_num);
              if (!orz_mdi_get_edited (ORZ_MDI (mdi), page))
                break;
            }
            return FALSE;
          case GTK_RESPONSE_NO:
            break;
          default:
            return FALSE;
        }
    }
  return TRUE;
}


/*  ja:最大フレーム数を求める
     vmaid,ウインドウ情報
    stream,有効なストリーム
       RET,最大フレーム数                                                   */
gint
get_max_frame (VmaidWindow *vmaid,
               const gint   stream)
{
  gint max0, max1;

  max0 = vmaid->avi_edit[0] ? avi_edit_length (vmaid->avi_edit[0]) - 1 : 0;
  max1 = vmaid->avi_edit[1] ? ((glonglong)avi_time_length (vmaid->avi_edit[1])
                                    * vmaid->rate + vmaid->scale * 1000 - 1)
                                    / ((glonglong)vmaid->scale * 1000) - 1 : 0;
  switch (stream)
    {
      case 0: return max0;
      case 1: return max1;
    }
  return MAX (max0, max1);
}


/******************************************************************************
*                                                                             *
* ja:高レベル関数群                                                           *
*                                                                             *
******************************************************************************/
/*  ja:フレームを反転する(独立スレッド)
    avi_edit,AVI編集構造体
         RET,TRUE:正常終了,FALSE:エラー                                     */
gboolean
reverse_frame (AviEdit *avi_edit)
{
  gint i;
  VmaidWindow *vmaid;
#ifdef USE_THREAD
  GThread *id;
#else /* not USE_THREAD */
  gint id = THREAD_ID_STANDARD;
#endif /* not USE_THREAD */

#ifdef USE_THREAD
  id = g_thread_self ();
#endif /* USE_THREAD */
  thread_insert (id, THREAD_MODE_REVERSE, NULL);

  vmaid = g_malloc0 (sizeof (VmaidWindow));
  for (i = 0; i < avi_edit_length (avi_edit); i++)
    {
      AviEdit *avi_edit_tmp;

      if (!thread_idling (id, i * 100 / avi_edit_length (avi_edit)))
        {
          avi_edit_close (vmaid->avi_edit[0]);
          vmaid->avi_edit[0] = NULL;
          break;
        }
      avi_edit_tmp = avi_edit_copy (avi_edit, i, 1);
      if (!avi_edit_tmp)
        {
          avi_edit_close (vmaid->avi_edit[0]);
          vmaid->avi_edit[0] = NULL;
          while (thread_idling (id, -1));
          break;
        }
      if (vmaid->avi_edit[0])
        {
          if (!avi_edit_paste (vmaid->avi_edit[0], 0, avi_edit_tmp))
            {
              avi_edit_close (avi_edit_tmp);
              avi_edit_close (vmaid->avi_edit[0]);
              vmaid->avi_edit[0] = NULL;
              while (thread_idling (id, -1));
              break;
            }
        }
      else
        {
          vmaid->avi_edit[0] = avi_edit_tmp;
        }
    }
  if (!avi_edit_close (avi_edit))
    {
      avi_edit_close (vmaid->avi_edit[0]);
      vmaid->avi_edit[0] = NULL;
      while (thread_idling (id, -1));
    }
  if (vmaid->avi_edit[0])
    {
      vmaid->selfrm = default_selfrm;
      vmaid->select.stream = -1;
      vmaid->select.frame = -1;
      size_set_scale (default_view, &vmaid->width, &vmaid->height,
                                    avi_edit_get_width (vmaid->avi_edit[0]),
                                    avi_edit_get_height (vmaid->avi_edit[0]));
      vmaid->rate = avi_edit_get_rate (vmaid->avi_edit[0]);
      vmaid->scale = avi_edit_get_scale (vmaid->avi_edit[0]);
#ifdef USE_THREAD
      gdk_threads_enter ();
#endif /* USE_THREAD */
      file_open_window (vmaid, NULL);
#ifdef USE_THREAD
      gdk_threads_leave ();
#endif /* USE_THREAD */
    }
  else
    {
      g_free (vmaid);
      vmaid = NULL;
    }
  thread_delete (id);
  return vmaid != NULL;
}
