/*
    w32loader
    copyright (c) 1998-2007 Kazuki IWAMOTO http://www.maid.org/ iwm@maid.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include "w32private.h"
#include "kernel32.h"


static LRESULT WINAPI
w32btn_DefWindowProc (HWND   hWnd,
                      UINT   Msg,
                      WPARAM wParam,
                      LPARAM lParam)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (hWnd), "user_data");
  if (!wd)
    return 0;
  switch (Msg)
    {
      case BM_GETCHECK:
        return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hWnd))
                                                ? BST_CHECKED : BST_UNCHECKED;
      case BM_SETCHECK:
        if (wParam != BST_INDETERMINATE)
          gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (hWnd),
                                                        wParam == BST_CHECKED);
        return 0;
      case BM_GETSTATE:
        return (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hWnd))
                                                ? BST_CHECKED : BST_UNCHECKED)
                            | (GTK_WIDGET_HAS_FOCUS (hWnd) ? BST_FOCUS : 0);
      case BM_SETSTATE:
        return 0;
      case BM_SETSTYLE:
        {
          wd->dwStyle = wParam;
          switch (wd->dwStyle & 0x00000300)
            {
              case BS_LEFT:
                gtk_label_set_justify (GTK_LABEL (gtk_bin_get_child
                                        (GTK_BIN (hWnd))), GTK_JUSTIFY_LEFT);
                break;
              case BS_RIGHT:
                gtk_label_set_justify (GTK_LABEL (gtk_bin_get_child
                                        (GTK_BIN (hWnd))), GTK_JUSTIFY_RIGHT);
                break;
              case BS_CENTER:
                gtk_label_set_justify (GTK_LABEL (gtk_bin_get_child
                                        (GTK_BIN (hWnd))), GTK_JUSTIFY_CENTER);
            }
        }
        return 0;
      case BM_CLICK:
        gtk_button_clicked (GTK_BUTTON (hWnd));
        return 0;
      case BM_GETIMAGE:
#if GTK_CHECK_VERSION(2,6,0)
        if (wd->dwStyle & BS_ICON)
          {
            /* ja:アイコン */
          }
        else if (wd->dwStyle & BS_BITMAP)
          {
            /* ja:ビットマップ */
            GtkWidget *image;

            image = gtk_button_get_image (GTK_BUTTON (hWnd));
            if (image)
              return GPOINTER_TO_INT (gtk_image_get_pixbuf
                                                        (GTK_IMAGE (image)));
          }
#endif /* GTK_CHECK_VERSION(2,6,0) */
        return GPOINTER_TO_INT (NULL);
      case BM_SETIMAGE:
#if GTK_CHECK_VERSION(2,6,0)
        if (wd->dwStyle & BS_ICON)
          {
            /* ja:アイコン */
          }
        else if (wd->dwStyle & BS_BITMAP)
          {
            /* ja:ビットマップ */
            GdkPixbuf *pixbuf;
            GtkWidget *image;

            image = gtk_button_get_image (GTK_BUTTON (hWnd));
            pixbuf = image ? gtk_image_get_pixbuf (GTK_IMAGE (image)) : NULL;
            if (pixbuf)
              g_object_ref (pixbuf);
            gtk_button_set_image (GTK_BUTTON (hWnd), lParam
                        ? gtk_image_new_from_pixbuf (GINT_TO_POINTER (lParam))
                        : NULL);
            return GPOINTER_TO_INT (pixbuf);
          }
#endif /* GTK_CHECK_VERSION(2,6,0) */
        return GPOINTER_TO_INT (NULL);
      case WM_GETTEXT:
        {
          const gchar *utf8str;

          utf8str = gtk_label_get_text
                            (GTK_LABEL (gtk_bin_get_child (GTK_BIN (hWnd))));
          if (wParam > 0 && lParam && utf8str)
            {
              if (wd->widechar)
                {
                  gunichar2 *wc;

                  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                  lstrcpynW (GINT_TO_POINTER (lParam), wc, wParam);
                  g_free (wc);
                  return lstrlenW (GINT_TO_POINTER (lParam));
                }
              else
                {
                  gchar *mb;

                  mb = w32ldr_utf8_to_mb (utf8str);
                  g_strncpy (GINT_TO_POINTER (lParam), mb, wParam);
                  g_free (mb);
                  return g_strlen (GINT_TO_POINTER (lParam));
                }
            }
        }
        return 0;
      case WM_GETTEXTLENGTH:
        {
          const gchar *utf8str;

          utf8str = gtk_label_get_text
                            (GTK_LABEL (gtk_bin_get_child (GTK_BIN (hWnd))));
          if (utf8str)
            {
              LRESULT lResult;

              if (wd->widechar)
                {
                  gunichar2 *wc;

                  wc = g_utf8_to_utf16 (utf8str, -1, NULL, NULL, NULL);
                  lResult = lstrlenW (wc);
                  g_free (wc);
                }
              else
                {
                  gchar *mb;

                  mb = w32ldr_utf8_to_mb (utf8str);
                  lResult = g_strlen (mb);
                  g_free (mb);
                }
              return lResult;
            }
        }
        return 0;
      case WM_SETTEXT:
        {
          gchar *utf8str, *title;
          gint i, j = 0;

          utf8str = wd->widechar ? g_utf16_to_utf8 (GINT_TO_POINTER (lParam),
                                                        -1, NULL, NULL, NULL)
                            : w32ldr_utf8_from_mb (GINT_TO_POINTER (lParam));
          title = g_malloc ((g_strlen (utf8str) * 2 + 1) * sizeof (gchar));
          for (i = 0; utf8str[i] != '\0'; i++)
            switch (utf8str[i])
              {
                case '-': title[j++] = '-';
                case '&': title[j++] = '_'; break;
                default:  title[j++] = utf8str[i];
              }
          title[j] = '\0';
          gtk_label_set_text_with_mnemonic
                    (GTK_LABEL (gtk_bin_get_child (GTK_BIN (hWnd))), title);
          g_free (title);
          g_free (utf8str);
        }
        return TRUE;
    }
  return wd->widechar ? DefWindowProcW (hWnd, Msg, wParam, lParam)
                       : DefWindowProcA (hWnd, Msg, wParam, lParam);
}


/* ja:ボタンが押されたとき */
#define w32btn_toggled w32btn_clicked
static void
w32btn_clicked (GtkWidget *widget,
                gpointer   user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, BN_CLICKED), GPOINTER_TO_INT (widget));
  else
    SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, BN_CLICKED), GPOINTER_TO_INT (widget));
}


/* ja:フォーカス取得 */
static gboolean
w32btn_focus_in (GtkWidget     *widget,
                 GdkEventFocus *event,
                 gpointer       user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, BN_SETFOCUS), GPOINTER_TO_INT (widget));
  else
    SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, BN_SETFOCUS), GPOINTER_TO_INT (widget));
  return FALSE;
}


/* ja:フォーカス喪失 */
static gboolean
w32btn_focus_out (GtkWidget     *widget,
                  GdkEventFocus *event,
                  gpointer       user_data)
{
  W32LdrWindowData *wd;

  wd = g_object_get_data (G_OBJECT (widget), "user_data");
  if (wd->widechar)
    SendMessageW (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, BN_KILLFOCUS), GPOINTER_TO_INT (widget));
  else
    SendMessageA (wd->hWndParent, WM_COMMAND,
                MAKEWPARAM (wd->wID, BN_KILLFOCUS), GPOINTER_TO_INT (widget));
  return FALSE;
}


GtkWidget *
w32ldr_control_create_button (const gchar    *windowname,
                              const guint32   style,
                              const guint32   exstyle,
                              const gint      width,
                              const gint      height,
                              const guint16   id,
                              HWND            hWndParent,
                              HINSTANCE       hInstance,
                              const gboolean  widechar)
{
  GtkWidget *button;
  W32LdrWindowData *wd;

  switch (style & BS_TYPEMASK)
    {
      case BS_PUSHBUTTON:
      case BS_DEFPUSHBUTTON:
        button = gtk_button_new_with_label ("");
        g_signal_connect (G_OBJECT (button), "clicked",
                                            G_CALLBACK (w32btn_clicked), NULL);
        break;
      case BS_AUTOCHECKBOX:
        button = gtk_check_button_new_with_label ("");
        g_signal_connect (G_OBJECT (button), "toggled",
                                            G_CALLBACK (w32btn_toggled), NULL);
        break;
      case BS_AUTORADIOBUTTON:
        button = gtk_radio_button_new_with_label (NULL, "");
        g_signal_connect (G_OBJECT (button), "toggled",
                                            G_CALLBACK (w32btn_toggled), NULL);
        break;
      default:
        g_print ("Unsupported Control Button.\n");
        return NULL;
    }
  if (style & BS_NOTIFY)
    {
      g_signal_connect (G_OBJECT (button), "focus-in-event",
                                        G_CALLBACK (w32btn_focus_in), NULL);
      g_signal_connect (G_OBJECT (button), "focus-out-event",
                                        G_CALLBACK (w32btn_focus_out), NULL);
      gtk_widget_add_events (button, GDK_FOCUS_CHANGE_MASK);
    }
  wd = g_malloc (sizeof (W32LdrWindowData));
  wd->dwStyle = style;
  wd->dwExStyle = exstyle;
  wd->hInstance = hInstance;
  wd->lpfnWndProc = w32btn_DefWindowProc;
  wd->hWndParent = hWndParent;
  wd->wID = id;
  wd->dwUserData = 0;
  wd->lpDialogFunc = NULL;
  wd->lResult = -1;
  wd->dwUser = 0;
  wd->csa = NULL;
  wd->csw = NULL;
  wd->dwInitParam = 0;
  wd->widechar = widechar;
  wd->resizable = FALSE;
  wd->classname = g_strdup ("BUTTON");
  wd->focus = button;
  wd->child = NULL;
  g_object_set_data (G_OBJECT (button), "user_data", wd);
  w32btn_DefWindowProc (button, BM_SETSTYLE, style, 0);
  if (style & BS_ICON)
    {
      /* ja:アイコン */
#if GTK_CHECK_VERSION(2,6,0)
#endif /* GTK_CHECK_VERSION(2,6,0) */
    }
  else if (style & BS_BITMAP)
    {
      /* ja:ビットマップ */
#if GTK_CHECK_VERSION(2,6,0)
      GdkPixbuf *pixbuf;

      if (GPOINTER_TO_UINT (windowname) > G_MAXUINT16)
        if (widechar)
          {
            gunichar2 *wc;

            wc = g_utf8_to_utf16 (windowname, -1, NULL, NULL, NULL);
            pixbuf = LoadBitmapW (hInstance, wc);
            g_free (wc);
          }
        else
          {
            gchar *mb;

            mb = w32ldr_utf8_to_mb (windowname);
            pixbuf = LoadBitmapA (hInstance, mb);
            g_free (mb);
          }
      else
        pixbuf = widechar ? LoadBitmapW (hInstance,
                            MAKEINTRESOURCEW (GPOINTER_TO_INT (windowname)))
                          : LoadBitmapA (hInstance,
                            MAKEINTRESOURCEA (GPOINTER_TO_INT (windowname)));
      if (pixbuf)
        {
          gtk_button_set_image (GTK_BUTTON (button),
                                        gtk_image_new_from_pixbuf (pixbuf));
          g_object_unref (pixbuf);
        }
#endif /* GTK_CHECK_VERSION(2,6,0) */
    }
  else
    {
      /* ja:テキスト */
      LPARAM lParam;

      lParam = widechar
                    ? GPOINTER_TO_INT (w32ldr_atom_get_string_wc (windowname))
                    : GPOINTER_TO_INT (w32ldr_atom_get_string_mb (windowname));
      w32btn_DefWindowProc (button, WM_SETTEXT, 0, lParam);
      g_free (GINT_TO_POINTER (lParam));
      wd->resizable = TRUE;
    }
  return button;
}
